# This file is part of pybliographer
# 
# Copyright (C) 1998 Frederic GOBRY
# Email : gobry@idiap.ch
# 	   
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2 
# of the License, or (at your option) any later version.
#   
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
# 
# $Id: Base.py,v 1.22 1999/08/18 14:05:15 gobry Exp $

from string import *
import re
import Pyblio.Help
from types import *

""" This Module contains the base classes one might want to inherit
from in order to provide a new database format """


		
def format (string, width, first, next):
	"Format a string on a given width"

	out = []
	current = first
	
	while len (string) > width - current:
		
		pos = width - current - 1
		
		while pos > 0 and string [pos] <> ' ':
			pos = pos - 1

		if pos == 0:
			pos = width - current
			taille = len (string)
			while pos < taille and string [pos] <> ' ':
				pos = pos + 1

		out.append (' ' * current + string [0:pos])
		string = string [pos+1:]
		current = next

	out.append (' ' * current + string)

	return strip (join (out, '\n'))


class Key:
	"""
	A special key that embeds both database and database key
	Such a key is expected to be completely unique among the whole
	program and should be the reliable information checked to see
	if two entries are the same.

	The .base field is public and is the database from which the
	actual entry can be recovered, whereas the .key is private to
	the database itself.
	"""
	
	def __init__ (self, base, key):
		if type (key) is InstanceType:
			self.base = key.base
			self.key  = key.key
		else:
			self.base = base.key
			self.key  = key
		return

	def __repr__ (self):
		if self.base:
			return `self.base + ' - ' + self.key`
		else:
			return `self.key`
		
	def __cmp__ (self, other):
		try:
			r = cmp (self.base, other.base)
		except AttributeError:
			return 1
		
		if r: return r
		
		return cmp (self.key, other.key)

	def __hash__ (self):
		return hash (self.key + '\0' + self.base)


class Entry:
	"""
	A database entry. It behaves like a dictionnary, which
	returns an instance of Description for each key. For example,
	entry ['author'] is expected to return a Types.AuthorGroup
	instance.

	Each entry class must define an unique ID, which is used
	during conversions.

	The entry.name is the name of the entry inside its database,
	whereas the entry.key is an instance of Key, and has to be
	unique over the whole application.

	The entry.type is an instance of Types.EntryDescription. It
	links the field names with their type.
	"""
	
	id = 'VirtualEntry'
	
	def __init__ (self, key, name, type = None, dict = None):
		self.name = name
		self.type = type
		self.__dict = dict or {}
		self.key    = key
		return
		
	def keys (self):
		return self.__dict.keys ()

	def has_key (self, key):
		return self.__dict.has_key (key)

	def text (self, key):
		""" return text with indication of convertion loss """
		return self.__dict [key], 0

	def get_native (self, key):
		""" returns the field in its native form """
		return self [key]

	def set_native (self, key, value):
		""" sets the field in its native form """
		self [key] = value
	
	def __getitem__ (self, key):
		""" return text representation of a field """
		return self.text (key) [0]
	
	def foreach (self, function, argument = None):
		""" To provide compatibility with ref """
		function (self, argument)
		return
	
	def __setitem__ (self, name, value):
		self.__dict [name] = value
		return
	
	def __delitem__ (self, name):
		del self.__dict [name]
		return
	
	def __add__ (self, other):
		""" Merges two entries, key by key """

		ret = Entry (self.key, self.name, self.type, {})

		# Prendre ses propres entrees
		for f in self.keys ():
			ret [f] = self [f]

		# et ajouter celles qu'on n'a pas
		for f in other.keys ():
			if not self.has_key (f):
				ret [f] = other [f]

		return ret

	def __repr__ (self):
		return "<entry `" + self.name + "'>"


	def __str__ (self):
		""" Nice standard entry  """
		tp = self.type.name
		fields = self.type.fields
			
		text = '%s [%s]\n' % (tp, self.name)
		text = text + ('-' * 70) + '\n'
		
		dico = self.keys ()
			
		for f in fields:
			name = f.name
			lcname = lower (name)
			
			if not self.has_key (lcname): continue
			
			text = text + '  '
			text = text + format (
				'%-14s %s' % (name, str (self [lcname])),
				75, 0, 17)
			text = text + '\n'
			
			try:
				dico.remove (lcname)
			except ValueError:
				raise ValueError,\
				      ("error: field `%s' appears " +
				       "more than once " +
				       "in the definition of `%s'") % \
				       (name, tp)
			
		for f in dico:
			text = text + '  '
			text = text + format (
				'%-14s %s' % (f, str (self [f])), 75, 0, 17)
			text = text + '\n'
		
		return text
	
		
class DataBase:
	"""
	This class represents a full bibliographic database.
	It also looks like a dictionnary, each key being an instance
	of class Key.

	The foreach() method should provide an efficient way of
	looping over all the elements, if one exists. 
	"""

	# a default database provides no editing facilities (as it cannot be saved)
	properties = {
		'edit'        : 0,
		'change_id'   : 0,
		'change_type' : 0,
		'add'         : 0,
		'remove'      : 0,
		'has_extra'   : 0,
		}

	id    = 'VirtualDB'
	__count = 0
	
	def __init__ (self, basename):
		self.name = basename

		self.key  = 'base-' + str (DataBase.__count)
		DataBase.__count = DataBase.__count + 1

		self.__dict = {}
		return
	
		
	def keys (self):
		""
		return self.__dict.keys ()

	def has_key (self, key):
		""
		return self.__dict.has_key (key)

	def __getitem__ (self, key):
		""
		return self.__dict [key]

	def __setitem__ (self, key, value):
		""
		self.__dict [key] = value
		return

	def __delitem__ (self, key):
		""
		del self.__dict [key]
		return
	
	def __len__ (self):
		""
		return len (self.keys ())

	def __repr__ (self):
		""
		return "<generic bibliographic database (" + `len (self)` + \
		       " entries)>"
		

	def where (self, test):
		""" Returns a subset of the entries according to a
		criterion """
		keys = []
		for e in self.keys ():
			if test.match (self [e]):
				keys.append (e)

		refs = Reference ()
		
		if len (keys):
			refs.add (self, keys)

		refs.properties = self.properties
		return refs
	
	def foreach (self, function, args = None):
		""" Run function on every entry. """
		for e in self.keys ():
			function (self [e], args)
		return

	def update (self):
		""" Remise a jour des donnes """
		return


	def sort (self, sortkey):
		""" sort entries according to a given key """
		# collect key values
		dict = {}
		none = []
		
		def collect (entry, arg):
			sortkey, dict, none = arg
			
			if entry.has_key (sortkey):
				dict [entry.key] = entry [sortkey]
			else:
				none.append (entry.key)
			return

		self.foreach (collect, (sortkey, dict, none))
		
		keys = dict.keys ()
		def sort_method (a,b, dict = dict):
			return cmp (dict [a], dict [b])
		
		keys.sort (sort_method)
		
		r = Reference ()
		r.add (self, none + keys)

		# simple search, copy the properties
		r.properties = self.properties
		return r



# ----- A class that holds references from several databases -----

Pyblio.Help.register ('references', """

References are generic holders of subsets or complete databases. One
can apply several method on a reference :

 - ref.add (base, items) : adds `items' from a given `base' in the
   reference
 
 - ref.foreach (method, argument) : calls `method' on every entry
 - ref.where ( condition ) : returns the reference of entries matching
   the condition

 - ref.bases () : a list of all the bases that compose the reference
 - ref.refs (base) : entries from a given database
 - ref.base_foreach (base) : like `foreach' but only on a given database

In addition, references can be merged with `+'.
""")


class Reference (DataBase):

	__count = 0
	id = 'Reference'

	properties = {
		'edit'        : 0,
		'change_id'   : 0,
		'change_type' : 0,
		'add'         : 0,
		'remove'      : 0,
		'has_extra'   : 0,
        }

	def __init__ (self):
		self.__refs  = []
		self.__bases = {}
		
		self.key  = 'ref-' + str (Reference.__count)
		Reference.__count = Reference.__count + 1

		return
	

	def add (self, base, liste = None):
		""" Ajouter des references liees a une base """

		if base.id == 'Reference':
			for b in base.bases ():
				self.__bases [b.key] = b
		else:
			self.__bases [base.key] = base
		
		if liste is None:
			liste = base.keys ()

		if type (liste) is ListType:
			self.__refs = self.__refs + liste
		else:
			self.__refs.append (liste)
		return


	def __repr__ (self):
		""
		return `self.__refs`

	def bases (self):
		""" Returns the databases available in the Reference
		object """
		
		return self.__bases.values ()
	
	def base (self, entry):
		""" Returns the database the entry actually comes from """
		return self.__bases [entry.key.base]
	
	def refs (self, base):
		""" Returns the references available in a given
		database """
		
		if self.__bases.has_key (base.key):
			return filter (lambda x, base = base: x.base == base.key,
				       self.__refs)
		
		return None

	def base_foreach (self, base, function, arg = None):
		""" Foreach entry in a base """
		
		for k in self.refs (base):
			entry = base [k]
			function (entry, arg)
		return
	
	def foreach (self, function, arg = None):
		"Foreach entry"

		for k in self.keys ():
			function (self [k], arg)
		return
		

	def __len__ (self):
		return len (self.keys ())


	def keys (self):
		return self.__refs
					  
	def has_key (self, key):
		return self.__refs.count (key)
	
	def __getitem__ (self, key):
		return self.__bases [key.base] [key]

	def __delitem__ (self, key):
		del self.__bases [key.base] [key]

	def __setitem__ (self, key, value):
		self.__bases [key.base] [key] = value
		return
	
	def __add__ (self, other):

		r = Reference ()

		for b in self.bases ():
			r.add (b, self.refs (b))

		for b in other.bases ():
			r.add (b, other.refs (b))

		return r


	def where (self, test):
		r = Reference ()

		for k in self.keys ():
			entry = self [k]
			
			if test.match (self [k]):
				r.add (self.base (entry), k)
				
		return r

