# This file is part of pybliographer
# 
# Copyright (C) 1998 Frederic GOBRY
# Email : gobry@idiap.ch
# 	   
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2 
# of the License, or (at your option) any later version.
#   
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
# 
# $Id: Pybliographic.py,v 1.15 1999/08/26 11:16:04 gobry Exp $

import string, os

from gnome.ui import *
from gtk import *
import GtkExtra, GDK, GTK

from Pyblio import Open, Types, Base, Config

from Pyblio.GnomeUI import Utils

from Pyblio.GnomeUI.Database import *
from Pyblio.GnomeUI.Entry  import *
from Pyblio.GnomeUI.Search import *
    

class URLFileSelection (GtkFileSelection):
    """ Extended file selection dialog, with an URL field and a type
    selector. """
    
    def __init__(self, title = "File", url=TRUE, modal=TRUE, has_auto=TRUE):
        
        GtkFileSelection.__init__(self)
        self.set_title (title)
        
        self.connect("destroy", self.quit)
        self.connect("delete_event", self.quit)
        self.cancel_button.connect('clicked', self.quit)
        self.ok_button.connect('clicked', self.ok_cb)

        if modal:
            grab_add (self)

        self.ret = None
        self.url = None
        
        vbox = self.main_vbox
        
        # url handler
        if url:
            hbox = GtkHBox ()
            hbox.set_spacing (5)
            hbox.pack_start (GtkLabel ("URL:"), expand = FALSE, fill = FALSE)
            self.url = GtkEntry ()
            hbox.pack_start (self.url)
            vbox.pack_start (hbox, expand = FALSE, fill = FALSE)

        # type selector
        hbox = GtkHBox ()
        hbox.set_spacing (5)
        hbox.pack_start (GtkLabel ("Bibliography type:"),
                         expand = FALSE, fill = FALSE)
        self.menu = GtkOptionMenu ()
        hbox.pack_start (self.menu)
        vbox.pack_start (hbox, expand = FALSE, fill = FALSE)

        # menu content
        menu = GtkMenu ()
        self.menu.set_menu (menu)
        
        liste = Open.available ()

        if has_auto:
            Utils.popup_add (menu, " - Auto - ", self.menu_select, None)
            self.type = None
        else:
            self.type = liste [0]
            
        for avail in liste:
            Utils.popup_add (menu, avail, self.menu_select, avail)

        self.menu.set_history (0)
        return

    def menu_select (self, widget, selection):
        self.type = selection
        return
        
    def quit(self, *args):
        self.hide()
        self.destroy()
        mainquit()
        return
    
    def ok_cb(self, b):
        self.ret = self.get_filename()
        
        if self.ret [-1] == '/':
            if self.url:
                self.ret = self.url.get_text ()
                if self.ret == "": self.ret = None
            else:
                self.ret = None
        
        self.quit()
        return
    
    def run (self):
        self.show_all ()
        mainloop ()
        
        return self.ret

    
class Pybliographic (GnomeApp):
    """ Main Class defining a Pybliographic window """
    
    def __init__ (self, version = '0.0'):
        GnomeApp.__init__ (self, "Pybliographer", "Pybliographer")

        self.connect ("destroy", self.quit)
        self.connect ("delete_event", self.quit)

        self.set_title("Pybliographic")
        self.version = version
        
        self.main_box = GtkVBox()
        self.set_contents (self.main_box)
        self.main_box.show()
        
        file_menu = [
            UIINFO_ITEM_STOCK ('New', None, self.file_new, STOCK_MENU_NEW),
            UIINFO_ITEM_STOCK ('Open', None, self.file_open, STOCK_MENU_OPEN),
            UIINFO_ITEM_STOCK ('Save', None, self.file_save, STOCK_MENU_SAVE),
            UIINFO_ITEM_STOCK ('Save As...', None, self.file_save_as,
                               STOCK_MENU_SAVE_AS),
            UIINFO_SEPARATOR,
            UIINFO_ITEM_STOCK ('Quit', None, self.file_exit, STOCK_MENU_QUIT),
            ]
        
        help_menu = [
            UIINFO_ITEM_STOCK('About...', None, self.help_about,
                              STOCK_MENU_ABOUT),
            ]

        edit_menu = [
            UIINFO_ITEM_STOCK('Copy', None, self.edit_copy, STOCK_MENU_COPY),
            UIINFO_ITEM_STOCK('Cut', None, self.edit_cut, STOCK_MENU_CUT),
            UIINFO_ITEM_STOCK('Paste', None, self.edit_paste,
                              STOCK_MENU_PASTE),
            UIINFO_SEPARATOR,
            UIINFO_ITEM_STOCK('New...', None, self.edit_new, STOCK_MENU_NEW),
            UIINFO_ITEM('Edit...', None, self.edit_entry, None),
            UIINFO_ITEM_STOCK('Delete', None, self.edit_delete,
                              STOCK_MENU_TRASH),
            UIINFO_SEPARATOR,
            UIINFO_ITEM_STOCK('Search...', None, self.edit_search,
                              STOCK_MENU_SEARCH),
            ]

        config_menu = [
            UIINFO_ITEM_STOCK('Preferences...', None, self.configure_prefs,
                              STOCK_MENU_PREF),
            ]
        
        menus = [
            UIINFO_SUBTREE('File', file_menu),
            UIINFO_SUBTREE('Edit', edit_menu),
            UIINFO_SUBTREE('Configure', config_menu),
            UIINFO_SUBTREE('Help', help_menu)
            ]
        
        toolbar = [
            UIINFO_ITEM_STOCK('Open', None, self.file_open,
                              STOCK_PIXMAP_OPEN),
            UIINFO_ITEM_STOCK('Save', None, self.file_save,
                              STOCK_PIXMAP_SAVE),
            UIINFO_SEPARATOR,
            UIINFO_ITEM_STOCK('Search', None, self.edit_search,
                              STOCK_PIXMAP_SEARCH),
            UIINFO_SEPARATOR,
            UIINFO_ITEM_STOCK('Exit', None, self.file_exit,
                              STOCK_PIXMAP_QUIT),
            ]
        
        self.create_menus (menus)
        self.create_toolbar (toolbar)

        # Put information in Paned windows
        paned = GtkVPaned ()
        paned.show ()

        # The List of data
        self.database = Database ()
        holder = GtkScrolledWindow ()
        holder.set_policy (POLICY_AUTOMATIC, POLICY_AUTOMATIC)
        holder.add (self.database)
        holder.show ()

        self.database.connect ("button_press_event", self.popup_menu)
        self.database.connect ("key_press_event", self.key_handler)
        self.database.connect ("select_row", self.update_display)
        
        paned.add1 (holder)
        self.database.set_usize (600,300)
        self.database.show ()

        # La zone de texte du bas
        self.display = GtkText ()
        self.display.set_word_wrap (1)
        self.display.set_usize (500,200)
        self.display.show ()
        holder = GtkScrolledWindow ()
        holder.set_policy (POLICY_AUTOMATIC, POLICY_AUTOMATIC)
        holder.add (self.display)
        holder.show ()
        
        paned.add2 (holder)

        self.main_box.pack_start (paned, expand=TRUE)

        self.status = GnomeAppBar (FALSE)
        self.set_statusbar (self.status)
        self.status.push ("Welcome to Pybliographic")

        # Le popup menu
        self.menu = GtkMenu ()
        self.menu_item = {}
        self.menu_item ['add']    = \
                       popup_add (self.menu, "New...",  self.edit_new)
        self.menu_item ['edit']   = \
                       popup_add (self.menu, "Edit...", self.edit_entry)
        self.menu_item ['remove'] = \
                       popup_add (self.menu, "Delete",  self.edit_delete)
        self.menu.show ()

        Utils.init_colors (self.get_colormap ())

        self.directory = './'
        self.modified  = 0

        self.search = None
        return
    
    # --------------------------------------------------
        
    def quit (self, *arg):
        mainquit ()
    
    # --------------------------------------------------

    def file_new (self, *arg):
        win = URLFileSelection ("Save", FALSE, TRUE, FALSE)
        win.set_filename (self.directory)

        file = win.run ()
        if file:
            self.directory = os.path.split (file) [0] + '/'
            self.new (file, win.type)
        return


    def new (self, file, type):
        try:
            self.database.set (Open.bibnew (file, type))
            
        except IOError, err:
            Utils.error_dialog ('Creation error', err)
            return

        self.filename = file
        self.type     = type
        
        self.status.pop ()
        self.status.push ("Database `%s' [%s]" % (self.filename, self.type))
        return
    
    # --------------------------------------------------

    def open (self, file, how = None):
        """ open a database """
        
        Utils.set_cursor (self, 'clock')

        try:
            self.database.set (Open.bibopen (file, how))
            Utils.set_cursor (self, 'normal')
            
        except IOError, err:
            Utils.set_cursor (self, 'normal')

            Utils.error_dialog ('Open Error', err)
            return

        self.status.pop ()
        
        self.filename = file
        self.type     = self.database.data.id
        
        self.status.push ("Database `%s' [%s, %d entries]"
                   % (self.filename, self.type, len (self.database.data.keys ())))


    def file_open (self, *arg):
        """ callback for the Open button """
        
        win = URLFileSelection ("Open", TRUE, TRUE)
        win.set_filename (self.directory)

        file = win.run ()
        
        if file:
            self.directory = os.path.split (file) [0] + '/'
            self.open (file, win.type)

        
    # --------------------------------------------------

    def file_exit (self, *arg):
        def callback (arg, self = self):
            if arg == 0: self.modified = 0
            return

        if self.modified:
            d = GnomeQuestionDialog ("Database is modified.\n" +
                                     "Quit without saving ?", callback)
            d.run_and_close ()
            
        if not self.modified: self.quit ()
        return
    

    # --------------------------------------------------

    def file_save (self, *arg):
        if not self.modified: return
        
        try:
            self.database.data.update ()
            self.status.pop ()
            self.status.push ("Database saved")
            self.modified = 0
            
        except IOError, err:
            d = GnomeErrorDialog ("can't save file: %s" % str (err))
            d.run_and_close ()
        return
    
    # --------------------------------------------------

    def file_save_as (self, *arg):
        win = URLFileSelection ("Save As", FALSE, TRUE, FALSE)
        win.set_filename (self.directory)

        file = win.run ()
        if not file: return
        
        if os.path.exists (file):
            var = []
                
            def callback (arg, var = var):
                if arg == 0: var.append (1)
                return

            d = GnomeQuestionDialog ("File `%s' exists.\nOverwrite it ?" % file,
                                     callback)
            d.run_and_close ()

            if not var: return
                
        # write the file
        try:
            fh = open (file, 'w')    
            Open.bibwrite (self.database.data, how = win.type, out = fh)
            fh.close ()
        except IOError, msg:
            d = GnomeErrorDialog ("Can't save file:\n%s" % msg)
            d.run_and_close ()
            return

        self.open (file, win.type)
        return
    
    # --------------------------------------------------
    # --------------------------------------------------

    def edit_copy (self, * arg):
        d = GnomeOkDialog ("Not yet implemented...")
        d.show ()
        return

    # --------------------------------------------------

    def edit_cut (self, * arg):
        d = GnomeOkDialog ("Not yet implemented...")
        d.show ()
        return

    # --------------------------------------------------

    def edit_paste (self, * arg):
        d = GnomeOkDialog ("Not yet implemented...")
        d.show ()
        return
    
    # --------------------------------------------------

    def edit_search (self, * arg):
        if self.search is None:
            self.search = SearchDialog (self.database)
        
        self.search.show_all ()
        return
    
    # --------------------------------------------------

    def edit_new (self, *arg):
        selection = self.database.selection

        if not self.database.data.has_property ('add'):
            d = GnomeErrorDialog ("edition is disabled in this database", self)
            d.run_and_close ()
            return
        
        if not selection: 
            row = 0
        else:
            row = self.database.selected_rows () [0]

        entry = self.database.data.new_entry ()
        
        def callback (entry, self, selection = row):
            self.status.pop ()
            self.status.push ("...modified...")
            self.modified = 1
            
            self.database.data [entry.key] = entry
            self.database.insert (selection, entry)
            return
        
        dialog = EntryDialog (entry, self.database.data, callback, self)
        dialog.show_all ()
        pass

    # --------------------------------------------------

    def edit_entry (self, *arg):
        selection = self.database.selection
        if not selection: return

        # if we have no right to edit
        if not self.database.data.has_property ('edit'): 
            d = GnomeErrorDialog ("edition is disabled in this database")
            d.run_and_close ()
            return

        row = self.database.selected_rows () [0]
        e = selection [0]
        
        def callback (entry, self, selection = row):
            self.status.pop ()
            self.status.push ("...modified...")
            self.modified = 1
            
            self.database.select_row (selection, 0)
            self.database.set_row (selection, entry)
            return
        
        dialog = EntryDialog (e, self.database.data, callback, self)
        dialog.show_all ()

    # --------------------------------------------------

    def edit_delete (self, *arg):
        selection = self.database.selection
        if not selection: return

        if not self.database.data.has_property ('remove'): 
            d = GnomeErrorDialog ("edition is disabled in this database", self)
            d.run_and_close ()
            return

        entry = selection [0]
        self.__ans = 0
        def callback (arg, self = self):
            if arg == 0: self.__ans = 1
            return

        d = GnomeQuestionDialog ("Really remove entry %s ?" %
                                 (entry.name), callback)
        d.run_and_close ()

        if self.__ans:
            self.database.remove (entry.key)
            self.modified = 1
            
            self.status.pop ()
            self.status.push ("...modified...")
        return
    
        
    # --------------------------------------------------

    def configure_prefs (self, * arg):
        d = GnomeOkDialog ("Not yet implemented...")
        d.show ()
        return

    # --------------------------------------------------

    def popup_menu (self, *arg):
        clist, event, = arg
        if self.database.data is None: return
        
        if (event.type == GDK._2BUTTON_PRESS and event.button == 1):
            # Always select the item below the cursor
            couple = self.database.get_selection_info (event.x, event.y)
            if couple:
                self.database.select_row (couple [0], couple [1])
                if not self.database.data.has_property ('edit'): return

                # else, do as if the Edit menu as been activated
                self.menu_item ['edit'].emit ("activate", None)
                return
        
        if (event.type == GDK.BUTTON_PRESS and event.button == 3):
            for f in ('add', 'edit', 'remove'):
                self.menu_item [f].set_sensitive \
                               (self.database.data.has_property (f))
            
            # Always select the item below the cursor
            couple = self.database.get_selection_info (event.x, event.y)
            if couple:
                self.database.select_row (couple [0], couple [1])
                
            self.menu.popup (None, None, None, event.button, event.time)

    # --------------------------------------------------

    def update_display (self, *arg):
    
        e = self.database.selection
        
        if e:
            e = e [0]
            
            # Display this entry
            self.display.freeze ()
            self.display.delete_text (0, -1)

            entry = e.type
                
            self.display.insert (None, Utils.color['blue'], None, entry.name)
            self.display.insert_defaults (" ["+ str (e.name) + "]\n\n")
            
            dico = e.keys ()

            # Search the longest field
            mlen = 0
            for f in dico:
                mlen = max (mlen, len (f))

            for f in entry.fields:
                
		field = string.lower (f.name)
                
		if e.has_key (field):
                    sp = ' ' * (mlen - len (f.name))
                    self.display.insert (None, Utils.color ['red'], None,
                                         f.name + ": " + sp)
                    self.display.insert_defaults (str (e [field]) + "\n")
                    dico.remove (field)


            self.display.insert_defaults ("\n")
            
            for f in dico:
                sp = ' ' * (mlen - len (f))
                self.display.insert (None, Utils.color['red'], None,
                                     f + ": " + sp)
                self.display.insert_defaults (str (e [f]) + "\n")

            self.display.thaw ()

            
    # --------------------------------------------------

    def help_about (self, *arg):
        GnomeAbout('Pybliographic', self.version,
                   'This program is copyrighted under the GNU GPL',
                   ['Frdric Gobry'],
                   'Gnome interface to the Pybliographer system').show()


    def key_handler (self, * arg):
        event = arg [1]
        if event.keyval == GDK.Return:
             self.database.select_row (self.database.focus_row, 0)
        return
    
