/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Quick Lounge Applet
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomevfs/gnome-vfs-directory.h>
#include <libgnome/gnome-desktop-item.h>
#include <libgnomeui/gnome-help.h>
#include <libgnomeui/gnome-window-icon.h>
#include <menu-tree.h>
#include "dlg-pick-applications.h"
#include "dlg-properties.h"
#include "gtk-utils.h"
#include "file-utils.h"


#define GLADE_FILE      "quick-lounge.glade"
#define ICON_SIZE_MENU  22    /* icons size in menus */


enum {
	APP_URI_COLUMN = 0,
	USE_APP_COLUMN,
	IS_LAUNCHER_COLUMN,
	APP_ICON_COLUMN,
	APP_NAME_COLUMN,
	APP_COMMENT_COLUMN,
	NUM_COLUMNS
};
	

typedef struct {
	QuickLounge   *quick_lounge;
	QuickBox      *quick_box;
	int            pos;

	GladeXML      *gui;

	GtkWidget     *dialog;
	GtkWidget     *tree_view;

	GtkTreeModel  *model;

	gboolean something_added;
} DialogData;


/* called when the main dialog is closed. */
static void
destroy_cb (GtkWidget  *widget, 
	    DialogData *data)
{
	g_object_unref (data->gui);
	g_free (data);
}


/* called when the "help" button is pressed. */
static void
help_cb (GtkWidget  *widget, 
	 DialogData *data)
{
	GError *err;

	err = NULL;  
	gnome_help_display_desktop_on_screen (
	      NULL, 
	      "quick-lounge", 
	      "quick-lounge", 
	      "quick-lounge-usage-add-existing", 
	      gtk_widget_get_screen (widget),
	      &err); 
	
	if (err != NULL) {
		GtkWidget *dialog;
		
		dialog = gtk_message_dialog_new (GTK_WINDOW (data->dialog),
						 0,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_CLOSE,
						 _("Could not display help: %s"),
						 err->message);
		gnome_window_icon_set_from_file (GTK_WINDOW (dialog), GNOME_ICONDIR "quick-lounge-applet.png");
		gtk_window_set_screen (GTK_WINDOW (dialog), gtk_widget_get_screen (GTK_WIDGET (data->quick_lounge->applet)));	
	
		g_signal_connect (G_OBJECT (dialog), "response",
				  G_CALLBACK (gtk_widget_destroy),
				  NULL);
		
		gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
		
		gtk_widget_show (dialog);
		
		g_error_free (err);
	}
}


static gboolean 
check_check (GtkTreeModel *model, 
	     GtkTreePath  *path, 
	     GtkTreeIter  *iter, 
	     gpointer      callback_data)
{
	DialogData *data = callback_data;
	gboolean    use_app;

	gtk_tree_model_get (data->model, iter, 
			    USE_APP_COLUMN, &use_app, 
			    -1);

	if (use_app) {
		char *src_uri, *dest_uri;

		gtk_tree_model_get (data->model, iter, 
				    APP_URI_COLUMN, &src_uri, 
				    -1);

		dest_uri = quick_lounge_util__get_unique_uri (data->quick_lounge);

		if (! file_copy (src_uri, dest_uri)) {
			g_free (src_uri);
			return FALSE;
		}

		quick_box_add_button (data->quick_box, dest_uri, ++data->pos);

		g_free (src_uri);
		g_free (dest_uri);

		data->something_added = TRUE;
	}

	return FALSE;
}


/* called when the "help" button is pressed. */
static void
ok_cb (GtkWidget  *widget, 
       DialogData *data)
{
	gtk_tree_model_foreach (data->model, check_check, data);

	if (data->something_added) {
		quick_lounge_save_order (data->quick_lounge);
		dlg_properties_update (data->quick_lounge->prop_dialog);
	}

	gtk_widget_destroy (data->dialog);
}


void
add_applications_from_dir (DialogData         *data,
			   GtkTreeIter        *tree_iter,
			   MenuTreeDirectory  *menu_dir)
{
	QuickLounge *quick_lounge = data->quick_lounge;
	GSList      *entries, *subdirs, *scan;
	GtkTreeIter  iter;

	subdirs = menu_tree_directory_get_subdirs (menu_dir);
	for (scan = subdirs; scan; scan = scan->next) {
		MenuTreeDirectory *dir = scan->data;
		const char        *icon_path = NULL;
		GdkPixbuf         *icon = NULL;

		icon_path = menu_tree_directory_get_icon (dir);
		if (icon_path != NULL) 
			icon = create_pixbuf (quick_lounge->icon_theme, icon_path, ICON_SIZE_MENU);

		gtk_tree_store_append (GTK_TREE_STORE (data->model), &iter, tree_iter);
		gtk_tree_store_set (GTK_TREE_STORE (data->model), &iter,
				    APP_URI_COLUMN, NULL,
				    USE_APP_COLUMN, FALSE,
				    IS_LAUNCHER_COLUMN, FALSE,
				    APP_ICON_COLUMN, icon,
				    APP_NAME_COLUMN, menu_tree_directory_get_name (dir),
				    APP_COMMENT_COLUMN, menu_tree_directory_get_comment (dir),
				    -1);

		if (icon != NULL)
			g_object_unref (icon);

		add_applications_from_dir (data, &iter, dir);

		menu_tree_directory_unref (dir);
	}
	g_slist_free (subdirs);

	entries = menu_tree_directory_get_entries (menu_dir);
	for (scan = entries; scan; scan = scan->next) {
		MenuTreeEntry *entry = scan->data;
		const char    *icon_path = NULL;
		GdkPixbuf     *icon = NULL;

		icon_path = menu_tree_entry_get_icon (entry);
		if (icon_path != NULL)
			icon = create_pixbuf (quick_lounge->icon_theme, icon_path, ICON_SIZE_MENU);

		gtk_tree_store_append (GTK_TREE_STORE (data->model), &iter, tree_iter);
		gtk_tree_store_set (GTK_TREE_STORE (data->model), &iter,
				    APP_URI_COLUMN, menu_tree_entry_get_desktop_file_path (entry),
				    USE_APP_COLUMN, FALSE,
				    IS_LAUNCHER_COLUMN, TRUE,
				    APP_ICON_COLUMN, icon,
				    APP_NAME_COLUMN, menu_tree_entry_get_name (entry),
				    APP_COMMENT_COLUMN, menu_tree_entry_get_comment (entry),
				    -1);

		if (icon != NULL)
			g_object_unref (icon);

		menu_tree_entry_unref (entry);
	}
	g_slist_free (entries);
}


void
add_applications (DialogData  *data,
		  const char  *menu_file)
{
	MenuTree           *menu_tree;
	MenuTreeDirectory  *menu_dir;

	menu_tree = menu_tree_lookup (menu_file);
	if (menu_tree == NULL)
		return;

	menu_dir = menu_tree_get_root_directory (menu_tree);
	add_applications_from_dir (data, NULL, menu_dir);
	menu_tree_directory_unref (menu_dir);

	gtk_tree_view_expand_all (GTK_TREE_VIEW (data->tree_view));

	menu_tree_unref (menu_tree);
}


static int
name_column_sort_func (GtkTreeModel *model, 
                       GtkTreeIter  *a, 
                       GtkTreeIter  *b, 
                       gpointer      user_data)
{
        char     *app1, *app2;
	int       result;
	gboolean  is_launcher1, is_launcher2;

        gtk_tree_model_get (model, a, 
			    APP_NAME_COLUMN, &app1, 
			    IS_LAUNCHER_COLUMN, &is_launcher1,
			    -1);

	gtk_tree_model_get (model, b, 
			    APP_NAME_COLUMN, &app2, 
			    IS_LAUNCHER_COLUMN, &is_launcher2,
			    -1);

	if (is_launcher1 == is_launcher2)
		result = strcasecmp (app1, app2);
	else if (is_launcher1)
		result = 1;
	else
		result = -1;

	g_free (app1);
	g_free (app2);

        return result;
}


static void
use_app_toggled (GtkCellRendererToggle *cell,
		 char                  *path_string,
		 gpointer               callback_data)
{
	DialogData   *data  = callback_data;
	GtkTreePath  *path = gtk_tree_path_new_from_string (path_string);
	GtkTreeIter   iter;
        gboolean      active;

	gtk_tree_model_get_iter (data->model, &iter, path);
	gtk_tree_path_free (path);
	
	gtk_tree_model_get (GTK_TREE_MODEL (data->model), 
			    &iter, 
			    USE_APP_COLUMN, 
			    &active, 
			    -1);

	gtk_tree_store_set (GTK_TREE_STORE (data->model), 
			    &iter, 
			    USE_APP_COLUMN, 
			    !active, 
			    -1);
}


static void
add_columns (DialogData  *data,
	     GtkTreeView *treeview)
{
	GtkCellRenderer   *renderer;
	GtkTreeViewColumn *column;

	column = gtk_tree_view_column_new ();

	renderer = gtk_cell_renderer_toggle_new ();
	g_signal_connect (G_OBJECT (renderer), 
			  "toggled", 
			  G_CALLBACK (use_app_toggled),
			  data);

	gtk_tree_view_column_pack_start (column, renderer, FALSE);
	gtk_tree_view_column_set_attributes (column, renderer,
                                             "active", USE_APP_COLUMN,
					     "visible", IS_LAUNCHER_COLUMN,
                                             NULL);
	gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN (column),
					 GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_fixed_width (GTK_TREE_VIEW_COLUMN (column), 30);
	g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL); 
        gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);	

	/**/

	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Application"));

	renderer = gtk_cell_renderer_pixbuf_new ();
	gtk_tree_view_column_pack_start (column, renderer, FALSE);
	gtk_tree_view_column_set_attributes (column, renderer,
                                             "pixbuf", APP_ICON_COLUMN,
                                             NULL);

	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_attributes (column, renderer,
                                             "text", APP_NAME_COLUMN,
                                             NULL);

	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
	gtk_tree_view_column_set_sort_column_id (column, APP_NAME_COLUMN);

        gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
	gtk_tree_view_set_expander_column (treeview, column);
	gtk_tree_view_column_set_clickable (column, FALSE);

	/**/

	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Description"));

	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_attributes (column, renderer,
                                             "text", APP_COMMENT_COLUMN,
                                             NULL);

	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
        gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
}





void
dlg_pick_applications (QuickLounge  *quick_lounge,
		       int           pos)
{
	DialogData *data;
	GtkWidget  *btn_ok;
	GtkWidget  *btn_cancel;
	GtkWidget  *btn_help;
	GtkWidget  *btn_collapse;
	GtkWidget  *btn_expand;

	data = g_new0 (DialogData, 1);

	data->quick_lounge = quick_lounge;
	data->quick_box = QUICK_BOX (quick_lounge->quick_box);
	data->pos = pos;

	data->gui = glade_xml_new (GLADEDIR "/" GLADE_FILE , NULL, NULL);
        if (! data->gui) {
		g_free (data);
                g_warning ("Could not find " GLADE_FILE "\n");
                return;
        }

	/* Get the widgets. */

	data->dialog = glade_xml_get_widget (data->gui, "from_menu_dialog");
	data->tree_view = glade_xml_get_widget (data->gui, "fm_treeview");

	btn_collapse = glade_xml_get_widget (data->gui, "fm_collapse_button");
	btn_expand = glade_xml_get_widget (data->gui, "fm_expand_button");

	btn_cancel = glade_xml_get_widget (data->gui, "fm_cancel_button");
	btn_ok = glade_xml_get_widget (data->gui, "fm_ok_button");
	btn_help = glade_xml_get_widget (data->gui, "fm_help_button");

	/* Set widgets data. */

	data->model = GTK_TREE_MODEL (gtk_tree_store_new (NUM_COLUMNS, 
							  G_TYPE_STRING,
							  G_TYPE_BOOLEAN,
							  G_TYPE_BOOLEAN,
							  GDK_TYPE_PIXBUF,
							  G_TYPE_STRING,
							  G_TYPE_STRING));
	gtk_tree_view_set_model (GTK_TREE_VIEW (data->tree_view), data->model);
	g_object_unref (data->model);

	add_columns (data, GTK_TREE_VIEW (data->tree_view));

	gtk_tree_sortable_set_default_sort_func (GTK_TREE_SORTABLE (data->model), name_column_sort_func, NULL, NULL);

	gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (data->model),
                                         APP_NAME_COLUMN, name_column_sort_func,
                                         NULL, NULL);

	gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (data->model), 
					      GTK_TREE_SORTABLE_DEFAULT_SORT_COLUMN_ID,
					      GTK_SORT_ASCENDING);

	/* Set the signals handlers. */
	
	g_signal_connect (G_OBJECT (data->dialog), 
			  "destroy",
			  G_CALLBACK (destroy_cb),
			  data);

	g_signal_connect_swapped (G_OBJECT (btn_cancel), 
				  "clicked",
				  G_CALLBACK (gtk_widget_destroy),
				  data->dialog);

	g_signal_connect (G_OBJECT (btn_help), 
			  "clicked",
			  G_CALLBACK (help_cb),
			  data);

	g_signal_connect (G_OBJECT (btn_ok), 
			  "clicked",
			  G_CALLBACK (ok_cb),
			  data);

	g_signal_connect_swapped (G_OBJECT (btn_collapse), 
				  "clicked",
				  G_CALLBACK (gtk_tree_view_collapse_all),
				  data->tree_view);

	g_signal_connect_swapped (G_OBJECT (btn_expand), 
				  "clicked",
				  G_CALLBACK (gtk_tree_view_expand_all),
				  data->tree_view);

	/* run dialog. */

	g_object_set_data (G_OBJECT (data->dialog), "dialog_data", data);

	gtk_widget_show (data->dialog);

	add_applications (data, "applications.menu"); 
	add_applications (data, "settings.menu"); 
}
