// RadioActive Copyright (C) 1999 RDI Gerg <cactus@cactus.rulez.org>
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// (included in the RadioActive distribution in doc/GPL) as published by
// the Free Software Foundation.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

//$Id: driver.cc,v 1.18 2001/02/07 17:05:08 cactus Exp $
#include "driver.h"

#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <math.h>
#include <stdexcept>

Radio::Radio(const string& devicefile):
    vol(RADIO_VOL_DEF),
    filename(devicefile)
{
    open_dev(devicefile);
    mute.connect (SigC::slot (this, &Radio::mute_impl));
}

void Radio::open_dev(const string& devicefile)
{
    int fd_new;
    if (!((fd_new = open(devicefile.c_str(), O_RDONLY)) > 0))
    {
	char* errormsg_c = g_strdup_printf (_("Couldn't open radio device `%s'"),
					    devicefile.c_str ());
	string errormsg = errormsg_c;
	g_free (errormsg_c);
	throw runtime_error(errormsg);
    } else {
	if (fd)
	    close(fd);
	fd = fd_new;
    }

    tuner.tuner = 0;
    ioctl (fd, VIDIOCGTUNER, &tuner);
}

void Radio::set_vol(int lvl)
{
    
    if (lvl > RADIO_VOL_MAX)
        lvl = RADIO_VOL_MAX;
    if (lvl < RADIO_VOL_MIN)
        lvl = RADIO_VOL_MIN;

    va.flags = VIDEO_AUDIO_VOLUME|(muted ? VIDEO_AUDIO_MUTE
				   : 0);
//    va.flags = VIDEO_AUDIO_VOLUME;
    va.audio = 0;
    va.volume = lvl * (65535/(RADIO_VOL_MAX-RADIO_VOL_MIN));
    ioctl (fd, VIDIOCSAUDIO, &va);
    vol = lvl;
}

void Radio::mute_impl (bool mute)
{
    if (mute)
    {
	va.volume = 0;
	va.flags = VIDEO_AUDIO_MUTE;
    } else {
	va.volume = vol * (65535/(RADIO_VOL_MAX - RADIO_VOL_MIN));
	va.flags = VIDEO_AUDIO_VOLUME;
    }
    va.audio = 0;
    ioctl (fd, VIDIOCSAUDIO, &va);
    muted = mute;
}

void Radio::toggle_mute()
{
    mute(!muted);
}

void Radio::set(double freq)
{
    fq = freq;

    struct video_tuner v;
    unsigned long xl_freq;
    
    xl_freq = (unsigned long)(freq * get_freq_fact () + 0.5);
    
    v.tuner = 0;
    
    ioctl (fd, VIDIOCSFREQ, &xl_freq);
}

double Radio::get_freq_fact()
{
    if ((tuner.flags & VIDEO_TUNER_LOW) == 0)
        return .016 * 1000;
    return 16 * 1000;
}

double Radio::get_freq_min()
{
    return ((tuner.rangelow - 0.5) / get_freq_fact ());
}

double Radio::get_freq_max()
{
    return ((tuner.rangehigh - 0.5) / get_freq_fact ());
}
