/*
 * Copyright (C) 2008 Nokia Corporation, all rights reserved.
 * Copyright (C) 2008 Zeeshan Ali (Khattak) <zeeshanak@gnome.org>.
 *
 * Author: Zeeshan Ali (Khattak) <zeeshanak@gnome.org>
 *                               <zeeshan.ali@nokia.com>
 *
 * This file is part of Rygel.
 *
 * Rygel is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Rygel is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "rygel-media-server-factory.h"
#include <gconf/gconf-client.h>
#include <libgupnp/gupnp.h>
#include <stdlib.h>
#include <string.h>
#include <libxml/tree.h>
#include <libxml/parser.h>
#include <gee/collection.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <rygel-icon-info.h>
#include <rygel-resource-info.h>
#include <stdio.h>
#include <gio/gio.h>
#include <gobject/gvaluecollector.h>




struct _RygelMediaServerFactoryPrivate {
	GConfClient* gconf;
	GUPnPContext* context;
};

#define RYGEL_MEDIA_SERVER_FACTORY_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), RYGEL_TYPE_MEDIA_SERVER_FACTORY, RygelMediaServerFactoryPrivate))
enum  {
	RYGEL_MEDIA_SERVER_FACTORY_DUMMY_PROPERTY
};
static xmlDoc* rygel_media_server_factory_create_desc (RygelMediaServerFactory* self, RygelPlugin* plugin, const char* desc_path, const char* gconf_path, GError** error);
static GUPnPContext* rygel_media_server_factory_create_upnp_context (RygelMediaServerFactory* self, GError** error);
static char* rygel_media_server_factory_get_str_from_gconf (RygelMediaServerFactory* self, const char* key, const char* default_value);
static void rygel_media_server_factory_add_xbox_specifics (RygelMediaServerFactory* self, xmlDoc* doc);
static void rygel_media_server_factory_prepare_desc_for_plugin (RygelMediaServerFactory* self, xmlDoc* doc, RygelPlugin* plugin, const char* gconf_path);
static void rygel_media_server_factory_set_friendly_name_and_udn (RygelMediaServerFactory* self, xmlNode* device_element, const char* plugin_name, const char* gconf_path);
static void rygel_media_server_factory_add_services_to_desc (RygelMediaServerFactory* self, xmlNode* device_element, RygelPlugin* plugin);
static void rygel_media_server_factory_add_service_to_desc (RygelMediaServerFactory* self, xmlNode* service_list_node, const char* plugin_name, RygelResourceInfo* resource_info);
static void rygel_media_server_factory_add_icons_to_desc (RygelMediaServerFactory* self, xmlNode* device_element, RygelPlugin* plugin);
static void rygel_media_server_factory_add_icon_to_desc (RygelMediaServerFactory* self, xmlNode* icon_list_node, RygelIconInfo* icon_info, RygelPlugin* plugin);
static void rygel_media_server_factory_save_modified_desc (RygelMediaServerFactory* self, xmlDoc* doc, const char* desc_path, GError** error);
static gpointer rygel_media_server_factory_parent_class = NULL;
static void rygel_media_server_factory_finalize (RygelMediaServerFactory* obj);



GQuark media_server_factory_error_quark (void) {
	return g_quark_from_static_string ("media_server_factory_error-quark");
}


RygelMediaServerFactory* rygel_media_server_factory_construct (GType object_type, GError** error) {
	GError * inner_error;
	RygelMediaServerFactory* self;
	GConfClient* _tmp1;
	GConfClient* _tmp0;
	GUPnPContext* _tmp2;
	GUPnPContext* _tmp3;
	inner_error = NULL;
	self = (RygelMediaServerFactory*) g_type_create_instance (object_type);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->gconf = (_tmp1 = (_tmp0 = gconf_client_get_default (), (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0)), (self->priv->gconf == NULL) ? NULL : (self->priv->gconf = (g_object_unref (self->priv->gconf), NULL)), _tmp1);
	_tmp2 = rygel_media_server_factory_create_upnp_context (self, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	/* Set up GUPnP context */
	_tmp3 = NULL;
	self->priv->context = (_tmp3 = _tmp2, (self->priv->context == NULL) ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL)), _tmp3);
	return self;
}


RygelMediaServerFactory* rygel_media_server_factory_new (GError** error) {
	return rygel_media_server_factory_construct (RYGEL_TYPE_MEDIA_SERVER_FACTORY, error);
}


RygelMediaServer* rygel_media_server_factory_create_media_server (RygelMediaServerFactory* self, RygelPlugin* plugin, GError** error) {
	GError * inner_error;
	char* _tmp0;
	char* _tmp1;
	char* gconf_path;
	char* _tmp2;
	char* _tmp3;
	char* modified_desc;
	char* desc_path;
	xmlDoc* doc;
	char* _tmp4;
	RygelMediaServer* _tmp5;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (plugin != NULL, NULL);
	inner_error = NULL;
	_tmp0 = NULL;
	_tmp1 = NULL;
	gconf_path = (_tmp1 = g_strconcat (_tmp0 = g_strconcat (RYGEL_MEDIA_SERVER_FACTORY_ROOT_GCONF_PATH, plugin->name, NULL), "/", NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
	_tmp2 = NULL;
	_tmp3 = NULL;
	modified_desc = (_tmp3 = g_strconcat (_tmp2 = g_strconcat (RYGEL_MEDIA_SERVER_FACTORY_DESC_PREFIX "-", plugin->name, NULL), ".xml", NULL), _tmp2 = (g_free (_tmp2), NULL), _tmp3);
	/* We store a modified description.xml in the user's config dir */
	desc_path = g_build_filename (g_get_user_config_dir (), modified_desc, NULL);
	/* Create the description xml */
	doc = rygel_media_server_factory_create_desc (self, plugin, desc_path, gconf_path, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		gconf_path = (g_free (gconf_path), NULL);
		modified_desc = (g_free (modified_desc), NULL);
		desc_path = (g_free (desc_path), NULL);
		return NULL;
	}
	/* Host our modified file */
	_tmp4 = NULL;
	gupnp_context_host_path (self->priv->context, desc_path, _tmp4 = g_strconcat ("/", modified_desc, NULL));
	_tmp4 = (g_free (_tmp4), NULL);
	_tmp5 = NULL;
	return (_tmp5 = rygel_media_server_new (self->priv->context, plugin, doc, modified_desc), gconf_path = (g_free (gconf_path), NULL), modified_desc = (g_free (modified_desc), NULL), desc_path = (g_free (desc_path), NULL), _tmp5);
}


static xmlDoc* rygel_media_server_factory_create_desc (RygelMediaServerFactory* self, RygelPlugin* plugin, const char* desc_path, const char* gconf_path, GError** error) {
	GError * inner_error;
	gboolean enable_xbox;
	char* orig_desc_path;
	xmlDoc* doc;
	xmlDoc* _tmp5;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (plugin != NULL, NULL);
	g_return_val_if_fail (desc_path != NULL, NULL);
	g_return_val_if_fail (gconf_path != NULL, NULL);
	inner_error = NULL;
	enable_xbox = FALSE;
	{
		char* _tmp0;
		gboolean _tmp1;
		gboolean _tmp2;
		_tmp0 = NULL;
		_tmp2 = (_tmp1 = gconf_client_get_bool (self->priv->gconf, _tmp0 = g_strconcat (gconf_path, "enable-xbox", NULL), &inner_error), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
		if (inner_error != NULL) {
			goto __catch0_g_error;
			goto __finally0;
		}
		enable_xbox = _tmp2;
	}
	goto __finally0;
	__catch0_g_error:
	{
		GError * error;
		error = inner_error;
		inner_error = NULL;
		{
			g_warning ("rygel-media-server-factory.vala:81: %s", error->message);
			(error == NULL) ? NULL : (error = (g_error_free (error), NULL));
		}
	}
	__finally0:
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return NULL;
	}
	orig_desc_path = NULL;
	if (enable_xbox) {
		char* _tmp3;
		_tmp3 = NULL;
		orig_desc_path = (_tmp3 = g_build_filename (DATA_DIR, RYGEL_MEDIA_SERVER_FACTORY_XBOX_DESC_DOC, NULL), orig_desc_path = (g_free (orig_desc_path), NULL), _tmp3);
	} else {
		char* _tmp4;
		_tmp4 = NULL;
		orig_desc_path = (_tmp4 = g_build_filename (DATA_DIR, RYGEL_MEDIA_SERVER_FACTORY_DESC_DOC, NULL), orig_desc_path = (g_free (orig_desc_path), NULL), _tmp4);
	}
	doc = xmlParseFile (orig_desc_path);
	if (doc == NULL) {
		char* message;
		message = g_strdup_printf ("Failed to parse %s", orig_desc_path);
		inner_error = g_error_new_literal (MEDIA_SERVER_FACTORY_ERROR, MEDIA_SERVER_FACTORY_ERROR_XML_PARSE, message);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			message = (g_free (message), NULL);
			orig_desc_path = (g_free (orig_desc_path), NULL);
			return NULL;
		}
		message = (g_free (message), NULL);
	}
	/* Modify description to include Plugin-specific stuff */
	rygel_media_server_factory_prepare_desc_for_plugin (self, doc, plugin, gconf_path);
	if (enable_xbox) {
		rygel_media_server_factory_add_xbox_specifics (self, doc);
	}
	rygel_media_server_factory_save_modified_desc (self, doc, desc_path, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		orig_desc_path = (g_free (orig_desc_path), NULL);
		return NULL;
	}
	return (_tmp5 = doc, orig_desc_path = (g_free (orig_desc_path), NULL), _tmp5);
}


static GUPnPContext* rygel_media_server_factory_create_upnp_context (RygelMediaServerFactory* self, GError** error) {
	GError * inner_error;
	char* host_ip;
	gint port;
	GUPnPContext* context;
	GUPnPContext* _tmp5;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	host_ip = NULL;
	{
		const char* _tmp0;
		char* _tmp2;
		const char* _tmp1;
		_tmp0 = gconf_client_get_string (self->priv->gconf, RYGEL_MEDIA_SERVER_FACTORY_ROOT_GCONF_PATH "host-ip", &inner_error);
		if (inner_error != NULL) {
			goto __catch1_g_error;
			goto __finally1;
		}
		_tmp2 = NULL;
		_tmp1 = NULL;
		host_ip = (_tmp2 = (_tmp1 = _tmp0, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), host_ip = (g_free (host_ip), NULL), _tmp2);
	}
	goto __finally1;
	__catch1_g_error:
	{
		GError * error;
		error = inner_error;
		inner_error = NULL;
		{
			char* _tmp3;
			g_warning ("rygel-media-server-factory.vala:117: %s", error->message);
			_tmp3 = NULL;
			host_ip = (_tmp3 = NULL, host_ip = (g_free (host_ip), NULL), _tmp3);
			(error == NULL) ? NULL : (error = (g_error_free (error), NULL));
		}
	}
	__finally1:
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		host_ip = (g_free (host_ip), NULL);
		return NULL;
	}
	port = 0;
	{
		gint _tmp4;
		_tmp4 = gconf_client_get_int (self->priv->gconf, RYGEL_MEDIA_SERVER_FACTORY_ROOT_GCONF_PATH "port", &inner_error);
		if (inner_error != NULL) {
			goto __catch2_g_error;
			goto __finally2;
		}
		port = _tmp4;
	}
	goto __finally2;
	__catch2_g_error:
	{
		GError * error;
		error = inner_error;
		inner_error = NULL;
		{
			g_warning ("rygel-media-server-factory.vala:126: %s", error->message);
			port = 0;
			(error == NULL) ? NULL : (error = (g_error_free (error), NULL));
		}
	}
	__finally2:
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		host_ip = (g_free (host_ip), NULL);
		return NULL;
	}
	context = gupnp_context_new (NULL, host_ip, (guint) port, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		host_ip = (g_free (host_ip), NULL);
		return NULL;
	}
	/* Host UPnP dir */
	gupnp_context_host_path (context, DATA_DIR, "");
	_tmp5 = NULL;
	return (_tmp5 = context, host_ip = (g_free (host_ip), NULL), _tmp5);
}


static char* rygel_media_server_factory_get_str_from_gconf (RygelMediaServerFactory* self, const char* key, const char* default_value) {
	GError * inner_error;
	char* str;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (key != NULL, NULL);
	g_return_val_if_fail (default_value != NULL, NULL);
	inner_error = NULL;
	str = NULL;
	{
		const char* _tmp0;
		char* _tmp2;
		const char* _tmp1;
		_tmp0 = gconf_client_get_string (self->priv->gconf, key, &inner_error);
		if (inner_error != NULL) {
			goto __catch3_g_error;
			goto __finally3;
		}
		_tmp2 = NULL;
		_tmp1 = NULL;
		str = (_tmp2 = (_tmp1 = _tmp0, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), str = (g_free (str), NULL), _tmp2);
	}
	goto __finally3;
	__catch3_g_error:
	{
		GError * error;
		error = inner_error;
		inner_error = NULL;
		{
			char* _tmp4;
			const char* _tmp3;
			g_warning ("Error getting gconf key '%s': %s." " Assuming default value '%s'.", key, error->message, default_value);
			_tmp4 = NULL;
			_tmp3 = NULL;
			str = (_tmp4 = (_tmp3 = default_value, (_tmp3 == NULL) ? NULL : g_strdup (_tmp3)), str = (g_free (str), NULL), _tmp4);
			(error == NULL) ? NULL : (error = (g_error_free (error), NULL));
		}
	}
	__finally3:
	if (inner_error != NULL) {
		str = (g_free (str), NULL);
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return NULL;
	}
	if (str == NULL) {
		char* _tmp6;
		const char* _tmp5;
		_tmp6 = NULL;
		_tmp5 = NULL;
		str = (_tmp6 = (_tmp5 = default_value, (_tmp5 == NULL) ? NULL : g_strdup (_tmp5)), str = (g_free (str), NULL), _tmp6);
		{
			gconf_client_set_string (self->priv->gconf, key, default_value, &inner_error);
			if (inner_error != NULL) {
				goto __catch4_g_error;
				goto __finally4;
			}
		}
		goto __finally4;
		__catch4_g_error:
		{
			GError * error;
			error = inner_error;
			inner_error = NULL;
			{
				g_warning ("rygel-media-server-factory.vala:161: Error setting gconf key '%s': %s.", key, error->message);
				(error == NULL) ? NULL : (error = (g_error_free (error), NULL));
			}
		}
		__finally4:
		if (inner_error != NULL) {
			str = (g_free (str), NULL);
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	return str;
}


static void rygel_media_server_factory_add_xbox_specifics (RygelMediaServerFactory* self, xmlDoc* doc) {
	xmlNode* element;
	g_return_if_fail (self != NULL);
	g_return_if_fail (doc != NULL);
	element = NULL;
	element = get_xml_element ((xmlNode*) doc, "root", "device", "friendlyName", NULL);
	/* friendlyName */
	if (element == NULL) {
		g_warning ("rygel-media-server-factory.vala:179: Element /root/device/friendlyName not found.");
		return;
	}
	xmlNodeAddContent (element, ": 1 : Windows Media Connect");
}


static void rygel_media_server_factory_prepare_desc_for_plugin (RygelMediaServerFactory* self, xmlDoc* doc, RygelPlugin* plugin, const char* gconf_path) {
	xmlNode* device_element;
	g_return_if_fail (self != NULL);
	g_return_if_fail (doc != NULL);
	g_return_if_fail (plugin != NULL);
	g_return_if_fail (gconf_path != NULL);
	device_element = NULL;
	device_element = get_xml_element ((xmlNode*) doc, "root", "device", NULL, NULL);
	if (device_element == NULL) {
		g_warning ("rygel-media-server-factory.vala:197: Element /root/device not found.");
		return;
	}
	/* First, set the Friendly name and UDN */
	rygel_media_server_factory_set_friendly_name_and_udn (self, device_element, plugin->name, gconf_path);
	/* Then list each icon */
	rygel_media_server_factory_add_icons_to_desc (self, device_element, plugin);
	/* Then list each service */
	rygel_media_server_factory_add_services_to_desc (self, device_element, plugin);
}


/**
     * Fills the description doc @doc with a friendly name, and UDN from gconf.
     * If these keys are not present in gconf, they are set with default values.
     */
static void rygel_media_server_factory_set_friendly_name_and_udn (RygelMediaServerFactory* self, xmlNode* device_element, const char* plugin_name, const char* gconf_path) {
	xmlNode* element;
	char* _tmp0;
	char* _tmp1;
	char* str;
	char* default_value;
	char* _tmp3;
	char* _tmp2;
	g_return_if_fail (self != NULL);
	g_return_if_fail (plugin_name != NULL);
	g_return_if_fail (gconf_path != NULL);
	/* friendlyName */
	element = get_xml_element (device_element, "friendlyName", NULL, NULL);
	if (element == NULL) {
		g_warning ("rygel-media-server-factory.vala:226: Element /root/device/friendlyName not found.");
		return;
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	str = (_tmp1 = rygel_media_server_factory_get_str_from_gconf (self, _tmp0 = g_strconcat (gconf_path, "friendly-name", NULL), plugin_name), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
	xmlNodeSetContent (element, str);
	/* UDN */
	element = get_xml_element (device_element, "UDN", NULL);
	if (element == NULL) {
		g_warning ("rygel-media-server-factory.vala:238: Element /root/device/UDN not found.");
		str = (g_free (str), NULL);
		return;
	}
	/* Generate new UUID */
	default_value = generate_random_udn ();
	_tmp3 = NULL;
	_tmp2 = NULL;
	str = (_tmp3 = rygel_media_server_factory_get_str_from_gconf (self, _tmp2 = g_strconcat (gconf_path, "UDN", NULL), default_value), str = (g_free (str), NULL), _tmp3);
	_tmp2 = (g_free (_tmp2), NULL);
	xmlNodeSetContent (element, str);
	str = (g_free (str), NULL);
	default_value = (g_free (default_value), NULL);
}


static void rygel_media_server_factory_add_services_to_desc (RygelMediaServerFactory* self, xmlNode* device_element, RygelPlugin* plugin) {
	xmlNode* service_list_node;
	g_return_if_fail (self != NULL);
	g_return_if_fail (plugin != NULL);
	service_list_node = get_xml_element (device_element, "serviceList", NULL, NULL);
	if (service_list_node == NULL) {
		g_warning ("rygel-media-server-factory.vala:255: Element /root/device/serviceList not found.");
		return;
	}
	{
		GeeIterator* _resource_info_it;
		_resource_info_it = gee_iterable_iterator ((GeeIterable*) plugin->resource_infos);
		while (gee_iterator_next (_resource_info_it)) {
			RygelResourceInfo* resource_info;
			resource_info = (RygelResourceInfo*) gee_iterator_get (_resource_info_it);
			/* FIXME: We only support plugable services for now*/
			if (g_type_is_a (resource_info->type, GUPNP_TYPE_SERVICE)) {
				rygel_media_server_factory_add_service_to_desc (self, service_list_node, plugin->name, resource_info);
			}
			(resource_info == NULL) ? NULL : (resource_info = (rygel_resource_info_unref (resource_info), NULL));
		}
		(_resource_info_it == NULL) ? NULL : (_resource_info_it = (g_object_unref (_resource_info_it), NULL));
	}
}


static void rygel_media_server_factory_add_service_to_desc (RygelMediaServerFactory* self, xmlNode* service_list_node, const char* plugin_name, RygelResourceInfo* resource_info) {
	xmlNode* service_node;
	const char* _tmp0;
	char* url;
	char* _tmp3;
	char* _tmp2;
	char* _tmp1;
	char* _tmp6;
	char* _tmp5;
	char* _tmp4;
	g_return_if_fail (self != NULL);
	g_return_if_fail (plugin_name != NULL);
	g_return_if_fail (resource_info != NULL);
	/* Create the service node*/
	service_node = xmlNewChild (service_list_node, NULL, "service", NULL);
	xmlNewChild (service_node, NULL, "serviceType", resource_info->upnp_type);
	xmlNewChild (service_node, NULL, "serviceId", resource_info->upnp_id);
	/* Now the relative (to base URL) URLs*/
	_tmp0 = NULL;
	url = (_tmp0 = resource_info->description_path, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
	xmlNewChild (service_node, NULL, "SCPDURL", url);
	_tmp3 = NULL;
	_tmp2 = NULL;
	_tmp1 = NULL;
	url = (_tmp3 = g_strconcat (_tmp2 = g_strconcat (_tmp1 = g_strconcat (plugin_name, "/", NULL), g_type_name (resource_info->type), NULL), "/Event", NULL), url = (g_free (url), NULL), _tmp3);
	_tmp2 = (g_free (_tmp2), NULL);
	_tmp1 = (g_free (_tmp1), NULL);
	xmlNewChild (service_node, NULL, "eventSubURL", url);
	_tmp6 = NULL;
	_tmp5 = NULL;
	_tmp4 = NULL;
	url = (_tmp6 = g_strconcat (_tmp5 = g_strconcat (_tmp4 = g_strconcat (plugin_name, "/", NULL), g_type_name (resource_info->type), NULL), "/Control", NULL), url = (g_free (url), NULL), _tmp6);
	_tmp5 = (g_free (_tmp5), NULL);
	_tmp4 = (g_free (_tmp4), NULL);
	xmlNewChild (service_node, NULL, "controlURL", url);
	url = (g_free (url), NULL);
}


static void rygel_media_server_factory_add_icons_to_desc (RygelMediaServerFactory* self, xmlNode* device_element, RygelPlugin* plugin) {
	gboolean _tmp0;
	xmlNode* icon_list_node;
	g_return_if_fail (self != NULL);
	g_return_if_fail (plugin != NULL);
	_tmp0 = FALSE;
	if (plugin->icon_infos == NULL) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = gee_collection_get_size ((GeeCollection*) plugin->icon_infos) == 0;
	}
	if (_tmp0) {
		g_debug ("rygel-media-server-factory.vala:293: No icon provided by %s.", plugin->name);
		return;
	}
	icon_list_node = xmlNewChild (device_element, NULL, "iconList", NULL);
	{
		GeeIterator* _icon_info_it;
		_icon_info_it = gee_iterable_iterator ((GeeIterable*) plugin->icon_infos);
		while (gee_iterator_next (_icon_info_it)) {
			RygelIconInfo* icon_info;
			icon_info = (RygelIconInfo*) gee_iterator_get (_icon_info_it);
			rygel_media_server_factory_add_icon_to_desc (self, icon_list_node, icon_info, plugin);
			(icon_info == NULL) ? NULL : (icon_info = (rygel_icon_info_unref (icon_info), NULL));
		}
		(_icon_info_it == NULL) ? NULL : (_icon_info_it = (g_object_unref (_icon_info_it), NULL));
	}
}


static void rygel_media_server_factory_add_icon_to_desc (RygelMediaServerFactory* self, xmlNode* icon_list_node, RygelIconInfo* icon_info, RygelPlugin* plugin) {
	xmlNode* icon_node;
	char* width;
	char* height;
	char* depth;
	char* _tmp5;
	char* _tmp4;
	char* _tmp3;
	char* _tmp2;
	char* _tmp1;
	char* _tmp0;
	char* _tmp6;
	char* url;
	char* _tmp7;
	g_return_if_fail (self != NULL);
	g_return_if_fail (icon_info != NULL);
	g_return_if_fail (plugin != NULL);
	/* Create the service node*/
	icon_node = xmlNewChild (icon_list_node, NULL, "icon", NULL);
	width = g_strdup_printf ("%u", icon_info->width);
	height = g_strdup_printf ("%u", icon_info->height);
	depth = g_strdup_printf ("%u", icon_info->depth);
	xmlNewChild (icon_node, NULL, "mimetype", icon_info->mimetype);
	xmlNewChild (icon_node, NULL, "width", width);
	xmlNewChild (icon_node, NULL, "height", height);
	xmlNewChild (icon_node, NULL, "depth", depth);
	/* PLUGIN_NAME-WIDTHxHEIGHTxDEPTH.png*/
	_tmp5 = NULL;
	_tmp4 = NULL;
	_tmp3 = NULL;
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp6 = NULL;
	url = (_tmp6 = g_strconcat (_tmp5 = g_strconcat (_tmp4 = g_strconcat (_tmp3 = g_strconcat (_tmp2 = g_strconcat (_tmp1 = g_strconcat (_tmp0 = g_strconcat (plugin->name, "-", NULL), width, NULL), "x", NULL), height, NULL), "x", NULL), depth, NULL), ".png", NULL), _tmp5 = (g_free (_tmp5), NULL), _tmp4 = (g_free (_tmp4), NULL), _tmp3 = (g_free (_tmp3), NULL), _tmp2 = (g_free (_tmp2), NULL), _tmp1 = (g_free (_tmp1), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp6);
	_tmp7 = NULL;
	gupnp_context_host_path (self->priv->context, icon_info->path, _tmp7 = g_strconcat ("/", url, NULL));
	_tmp7 = (g_free (_tmp7), NULL);
	xmlNewChild (icon_node, NULL, "url", url);
	width = (g_free (width), NULL);
	height = (g_free (height), NULL);
	depth = (g_free (depth), NULL);
	url = (g_free (url), NULL);
}


static void rygel_media_server_factory_save_modified_desc (RygelMediaServerFactory* self, xmlDoc* doc, const char* desc_path, GError** error) {
	GError * inner_error;
	FILE* f;
	gint res;
	gboolean _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (desc_path != NULL);
	inner_error = NULL;
	f = fopen (desc_path, "w+");
	res = -1;
	if (f != NULL) {
		res = xmlDocDump (f, doc);
	}
	_tmp0 = FALSE;
	if (f == NULL) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = res == (-1);
	}
	if (_tmp0) {
		char* _tmp1;
		char* _tmp2;
		char* message;
		_tmp1 = NULL;
		_tmp2 = NULL;
		message = (_tmp2 = g_strconcat ("Failed to write modified description", _tmp1 = g_strdup_printf (" to %s.\n", desc_path), NULL), _tmp1 = (g_free (_tmp1), NULL), _tmp2);
		xmlFreeDoc (doc);
		inner_error = g_error_new_literal (G_IO_ERROR, G_IO_ERROR_FAILED, message);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			message = (g_free (message), NULL);
			(f == NULL) ? NULL : (f = (fclose (f), NULL));
			return;
		}
		message = (g_free (message), NULL);
	}
	(f == NULL) ? NULL : (f = (fclose (f), NULL));
}


static void rygel_value_media_server_factory_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void rygel_value_media_server_factory_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		rygel_media_server_factory_unref (value->data[0].v_pointer);
	}
}


static void rygel_value_media_server_factory_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = rygel_media_server_factory_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer rygel_value_media_server_factory_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* rygel_value_media_server_factory_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		RygelMediaServerFactory* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = rygel_media_server_factory_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* rygel_value_media_server_factory_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	RygelMediaServerFactory** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = rygel_media_server_factory_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* rygel_param_spec_media_server_factory (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	RygelParamSpecMediaServerFactory* spec;
	g_return_val_if_fail (g_type_is_a (object_type, RYGEL_TYPE_MEDIA_SERVER_FACTORY), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer rygel_value_get_media_server_factory (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, RYGEL_TYPE_MEDIA_SERVER_FACTORY), NULL);
	return value->data[0].v_pointer;
}


void rygel_value_set_media_server_factory (GValue* value, gpointer v_object) {
	RygelMediaServerFactory* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, RYGEL_TYPE_MEDIA_SERVER_FACTORY));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, RYGEL_TYPE_MEDIA_SERVER_FACTORY));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		rygel_media_server_factory_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		rygel_media_server_factory_unref (old);
	}
}


static void rygel_media_server_factory_class_init (RygelMediaServerFactoryClass * klass) {
	rygel_media_server_factory_parent_class = g_type_class_peek_parent (klass);
	RYGEL_MEDIA_SERVER_FACTORY_CLASS (klass)->finalize = rygel_media_server_factory_finalize;
	g_type_class_add_private (klass, sizeof (RygelMediaServerFactoryPrivate));
}


static void rygel_media_server_factory_instance_init (RygelMediaServerFactory * self) {
	self->priv = RYGEL_MEDIA_SERVER_FACTORY_GET_PRIVATE (self);
	self->ref_count = 1;
}


static void rygel_media_server_factory_finalize (RygelMediaServerFactory* obj) {
	RygelMediaServerFactory * self;
	self = RYGEL_MEDIA_SERVER_FACTORY (obj);
	(self->priv->gconf == NULL) ? NULL : (self->priv->gconf = (g_object_unref (self->priv->gconf), NULL));
	(self->priv->context == NULL) ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL));
}


GType rygel_media_server_factory_get_type (void) {
	static GType rygel_media_server_factory_type_id = 0;
	if (rygel_media_server_factory_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { rygel_value_media_server_factory_init, rygel_value_media_server_factory_free_value, rygel_value_media_server_factory_copy_value, rygel_value_media_server_factory_peek_pointer, "p", rygel_value_media_server_factory_collect_value, "p", rygel_value_media_server_factory_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (RygelMediaServerFactoryClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) rygel_media_server_factory_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (RygelMediaServerFactory), 0, (GInstanceInitFunc) rygel_media_server_factory_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		rygel_media_server_factory_type_id = g_type_register_fundamental (g_type_fundamental_next (), "RygelMediaServerFactory", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return rygel_media_server_factory_type_id;
}


gpointer rygel_media_server_factory_ref (gpointer instance) {
	RygelMediaServerFactory* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void rygel_media_server_factory_unref (gpointer instance) {
	RygelMediaServerFactory* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		RYGEL_MEDIA_SERVER_FACTORY_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}




