/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * soup-queue.c: Asyncronous Callback-based SOAP Request Queue.
 *
 * Authors:
 *      Alex Graveley (alex@ximian.com)
 *
 * Copyright (C) 2001, Ximian, Inc.
 */

#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <glib.h>

#include "soup-server.h"
#include "soup-headers.h"
#include "soup-private.h"

static GMainLoop *server_main;
static guint      server_read_source_tag;

static void
soup_cgi_write_header (gchar *key, gchar *val, gpointer not_used)
{
	g_print ("%s: %s\r\n", key, val);
}

static gboolean 
soup_cgi_read_cb (GIOChannel   *iochannel, 
		  GIOCondition  condition, 
		  gpointer      not_used) 
{
	SoupServerHandler *hand;
	SoupContext *ctx;
	SoupMessage *msg;
	GHashTable *req_headers;
	gchar *req_host, *req_path, *req_method, *url;
	gchar *str = NULL;
	gint len = 0, index;
	SoupAction action;

	index = soup_substring_index (str, len, "\r\n\r\n");
	if (!index) goto THROW_MALFORMED_REQUEST;

	req_headers = g_hash_table_new (soup_str_case_hash, 
					soup_str_case_equal);

	if (!soup_headers_parse_request (str, 
					 index, 
					 req_headers, 
					 &req_method, 
					 &req_path))
		goto THROW_MALFORMED_HEADER;

	if ((g_strcasecmp (req_method, "POST") != 0 && 
	     g_strcasecmp (req_method, "M-POST") != 0))
		goto THROW_MALFORMED_HEADER;

	action = g_hash_table_lookup (req_headers, "SOAPAction");
	if (!action) goto THROW_MALFORMED_HEADER;

	hand = soup_server_get_handler (action);
	if (!hand) goto THROW_NO_HANDLER;

	req_host = g_hash_table_lookup (req_headers, "Host");
	if (req_host) 
		url = g_strconcat ("http://", req_host, req_path, NULL);
	else 
		url = g_strdup (req_path);

	ctx = soup_context_get (url);
	g_free (url);

	/* No Host, no AbsoluteUri */
	if (!ctx) {
		url = g_strconcat ("http://localhost/", req_path, NULL);
		ctx = soup_context_get (url);
		g_free (url);
	}

	if (!ctx) goto THROW_MALFORMED_HEADER;

	msg = soup_message_new (ctx, action);

	msg->request_headers = req_headers;

	msg->request.owner = SOUP_BUFFER_SYSTEM_OWNED;
	msg->request.length = len - index - 4;
	msg->request.body = &str [index + 4];

	msg->response_code = 200;
	msg->response_phrase = "OK";
	msg->response_headers = g_hash_table_new (soup_str_case_hash, 
						  soup_str_case_equal);
	g_hash_table_insert (msg->response_headers, 
			     "Content-Type",
			     "text/xml\r\n\tcharset=\"utf-8\"");

	(*hand->cb) (msg, hand->user_data); 

	g_print ("HTTP/1.1 %d %s\r\n", 
		 msg->response_code, 
		 msg->response_phrase);

	g_hash_table_foreach (msg->response_headers, 
			      (GHFunc) soup_cgi_write_header,
			      NULL);

	g_print ("\r\n");

	// write msg->response to STDOUT

	g_free (req_method);
	g_free (req_path);
	soup_message_free (msg);
	
 THROW_NO_HANDLER:
	return TRUE;

 THROW_MALFORMED_HEADER:
	g_hash_table_destroy (req_headers);
	g_free (req_method);
	g_free (req_path);

 THROW_MALFORMED_REQUEST:
	return TRUE;
}

static gboolean 
soup_cgi_error_cb (GIOChannel   *iochannel, 
		      GIOCondition  condition, 
		      gpointer      not_used) 
{
	g_source_remove (server_read_source_tag);
	soup_server_main_quit ();
	return FALSE;
}

void
soup_server_main (void)
{
	GIOChannel *chan = g_io_channel_unix_new (STDIN_FILENO);
	server_main = g_main_new (TRUE);

	server_read_source_tag = g_io_add_watch (chan, 
						 G_IO_IN, 
						 soup_cgi_read_cb, 
						 NULL);
	g_io_add_watch (chan, 
			G_IO_ERR | G_IO_HUP | G_IO_NVAL, 
			soup_cgi_error_cb, 
			NULL);

	g_main_run (server_main);
}

void  
soup_server_main_quit (void)
{
	g_return_if_fail (server_main != NULL);
	g_main_quit (server_main);
	server_main = NULL;
}
