/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * wsdl-common.c: Emit code common to both stubs and skels
 *
 * Authors:
 *	Dick Porter (dick@ximian.com)
 *
 * Copyright (C) 2001, Ximian, Inc.
 */

#include <glib.h>
#include <string.h>
#include <errno.h>

#include <libwsdl/wsdl-typecodes.h>
#include <libwsdl/wsdl-typecodes-c.h>

#include "wsdl-soap-common.h"
#include "wsdl-trace.h"

static void
wsdl_emit_soap_common_typecodes (const wsdl_typecode * const tc,
				 gpointer                    user_data)
{
	FILE *out = (FILE *) user_data;

	wsdl_typecode_write_c_definition (out, tc);
}

static void
wsdl_emit_soap_common_mm (const wsdl_typecode * const tc, gpointer user_data)
{
	FILE *out = (FILE *) user_data;

	wsdl_typecode_write_c_mm (out, tc);
}

/**
 * wsdl_emit_soap_common:
 * @outdir: a string containing the path to a directory.  This
 * function expects the string to have a trailing '/'.
 * @fileroot: a string containing the root of a filename.  "-common.c"
 * will be appended to this name.
 * @definitions: a pointer to a #wsdl_definitions structure,
 * containing a set of WSDL elements.
 *
 * Creates the file @outdir/@fileroot-common.c, and writes C code
 * containing typecode definitions, memory management functions, and
 * any other generated code that is common to both client stubs and
 * server skeletons.
 */
void
wsdl_emit_soap_common (const guchar                  *outdir, 
		       const guchar                  *fileroot,
		       const wsdl_definitions * const definitions)
{
	guchar *filename;
	const guchar *namespace;

	filename = g_strconcat (outdir, fileroot, "-common.c", NULL);
	wsdl_debug (WSDL_LOG_DOMAIN_COMMON, G_LOG_LEVEL_DEBUG, "file: [%s]",
		    filename);

	wsdl_set_output_file (fopen (filename, "w"));
	g_free (filename);

	if (wsdl_get_output_file () == NULL) {
		g_warning ("Couldn't open %s for writing: %s", filename,
			   g_strerror (errno));
		return;
	}

	W ("/*\n");

	if (definitions->name) {
		W (" * %s\n", definitions->name);
		namespace = definitions->name;
	} else {
		namespace = "";
	}

	W (" *\n");
	W (" * Automatically generated by soup-wsdl.\n");
	W (" */\n");
	W ("\n");

	W ("#include <glib.h>\n");
	W ("#include <libsoup/soup.h>\n");
	W ("#include <libwsdl/wsdl.h>\n");
	W ("#include \"%s.h\"\n\n", fileroot);

	if (definitions->documentation) {
		W ("/* %s */\n", definitions->documentation->str);
		W ("\n");
	}

	W ("\n");

	/* 
	 * Call wsdl_typecode_write_c_definition() for each non-simple
	 * typecode known
	 */
	wsdl_typecode_foreach (FALSE, 
			       wsdl_emit_soap_common_typecodes, 
			       wsdl_get_output_file ());
	wsdl_typecode_foreach (FALSE, 
			       wsdl_emit_soap_common_mm, 
			       wsdl_get_output_file());

	fclose (wsdl_get_output_file ());
	wsdl_set_output_file (NULL);
}
