/* valacallback.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valacallback.h"
#include <vala/valatypereference.h>
#include <vala/valasourcereference.h>
#include <vala/valacallback.h>
#include <vala/valatypeparameter.h>
#include <vala/valaformalparameter.h>
#include <vala/valamethod.h>
#include <vala/valacodevisitor.h>
#include <vala/valanamespace.h>
#include <vala/valaattribute.h>

struct _ValaCallbackPrivate {
	ValaTypeReference* _return_type;
	gboolean _instance;
	GList* type_parameters;
	GList* parameters;
	char* cname;
};
#define VALA_CALLBACK_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CALLBACK, ValaCallbackPrivate))
enum  {
	VALA_CALLBACK_DUMMY_PROPERTY,
	VALA_CALLBACK_RETURN_TYPE,
	VALA_CALLBACK_INSTANCE,
};
static void vala_callback_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_callback_real_get_cname (ValaDataType* base, gboolean const_type);
static void vala_callback_process_ccode_attribute (ValaCallback* self, ValaAttribute* a);
static gboolean vala_callback_real_is_reference_type (ValaDataType* base);
static char* vala_callback_real_get_type_id (ValaDataType* base);
static char* vala_callback_real_get_marshaller_type_name (ValaDataType* base);
static char* vala_callback_real_get_get_value_function (ValaDataType* base);
static char* vala_callback_real_get_set_value_function (ValaDataType* base);
static gpointer vala_callback_parent_class = NULL;
static void vala_callback_dispose (GObject * obj);


/**
 * Creates a new callback.
 *
 * @param name        callback type name
 * @param return_type return type
 * @param source      reference to source code
 * @return            newly created callback
 */
ValaCallback* vala_callback_new (const char* _name, ValaTypeReference* _return_type, ValaSourceReference* source)
{
	g_return_val_if_fail (_return_type == NULL || VALA_IS_TYPE_REFERENCE (_return_type), NULL);
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	GParameter * __params = g_new0 (GParameter, 3);
	GParameter * __params_it = __params;
	(__params_it->name = "name", g_value_init (&__params_it->value, G_TYPE_STRING), g_value_set_string (&__params_it->value, _name), __params_it++);
	(__params_it->name = "return-type", g_value_init (&__params_it->value, VALA_TYPE_TYPE_REFERENCE), g_value_set_object (&__params_it->value, _return_type), __params_it++);
	(__params_it->name = "source-reference", g_value_init (&__params_it->value, VALA_TYPE_SOURCE_REFERENCE), g_value_set_object (&__params_it->value, source), __params_it++);
	ValaCallback * self = g_object_newv (VALA_TYPE_CALLBACK, __params_it - __params, __params);
	return self;
}


/**
 * Appends the specified parameter to the list of type parameters.
 *
 * @param p a type parameter
 */
void vala_callback_add_type_parameter (ValaCallback* self, ValaTypeParameter* p)
{
	g_return_if_fail (VALA_IS_CALLBACK (self));
	g_return_if_fail (VALA_IS_TYPE_PARAMETER (p));
	self->priv->type_parameters = g_list_append (self->priv->type_parameters, g_object_ref (p));
	p->type = VALA_DATA_TYPE (self);
}


/**
 * Appends paramater to this callback function.
 *
 * @param param a formal parameter
 */
void vala_callback_add_parameter (ValaCallback* self, ValaFormalParameter* param)
{
	g_return_if_fail (VALA_IS_CALLBACK (self));
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (param));
	self->priv->parameters = g_list_append (self->priv->parameters, g_object_ref (param));
}


/**
 * Return copy of parameter list.
 *
 * @return parameter list
 */
GList* vala_callback_get_parameters (ValaCallback* self)
{
	g_return_val_if_fail (VALA_IS_CALLBACK (self), NULL);
	return g_list_copy (self->priv->parameters);
}


/**
 * Checks whether the arguments and return type of the specified method
 * matches this callback.
 *
 * @param m a method
 * @return  true if the specified method is compatible to this callback
 */
gboolean vala_callback_matches_method (ValaCallback* self, ValaMethod* m)
{
	g_return_val_if_fail (VALA_IS_CALLBACK (self), FALSE);
	g_return_val_if_fail (VALA_IS_METHOD (m), FALSE);
	if (!vala_type_reference_stricter (vala_method_get_return_type (m), vala_callback_get_return_type (self))) {
		return FALSE;
	}
	GList* method_params = vala_invokable_get_parameters (VALA_INVOKABLE (m));
	GList* method_params_it = method_params;
	gboolean first = TRUE;
	{
		GList* __temp4 = NULL;
		__temp4 = self->priv->parameters;
		GList* param_it;
		for (param_it = __temp4; param_it != NULL; param_it = param_it->next) {
			ValaFormalParameter* param = param_it->data;
			{
				/* use first callback parameter as instance parameter if
				 * an instance method is being compared to a static
				 * callback
				 */
				if (first && vala_method_get_instance (m) && !vala_callback_get_instance (self)) {
					first = FALSE;
					continue;
				}
				/* method is allowed to accept less arguments */
				if (method_params_it == NULL) {
					break;
				}
				ValaFormalParameter* __temp2 = NULL;
				ValaFormalParameter* method_param = (__temp2 = VALA_FORMAL_PARAMETER (method_params_it->data), (__temp2 == NULL ? NULL : g_object_ref (__temp2)));
				if (!vala_type_reference_stricter (vala_formal_parameter_get_type_reference (param), vala_formal_parameter_get_type_reference (method_param))) {
					gboolean __temp3;
					return (__temp3 = FALSE, (method_param == NULL ? NULL : (method_param = (g_object_unref (method_param), NULL))), __temp3);
				}
				method_params_it = method_params_it->next;
				(method_param == NULL ? NULL : (method_param = (g_object_unref (method_param), NULL)));
			}
		}
	}
	/* method may not expect more arguments */
	if (method_params_it != NULL) {
		gboolean __temp5;
		return (__temp5 = FALSE, (method_params == NULL ? NULL : (method_params = (g_list_free (method_params), NULL))), __temp5);
	}
	gboolean __temp6;
	return (__temp6 = TRUE, (method_params == NULL ? NULL : (method_params = (g_list_free (method_params), NULL))), __temp6);
	(method_params == NULL ? NULL : (method_params = (g_list_free (method_params), NULL)));
}


static void vala_callback_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaCallback * self = VALA_CALLBACK (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_begin_callback (visitor, self);
	{
		GList* __temp7 = NULL;
		__temp7 = self->priv->type_parameters;
		GList* p_it;
		for (p_it = __temp7; p_it != NULL; p_it = p_it->next) {
			ValaTypeParameter* p = p_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (p), visitor);
			}
		}
	}
	vala_code_node_accept (VALA_CODE_NODE (vala_callback_get_return_type (self)), visitor);
	{
		GList* __temp8 = NULL;
		__temp8 = self->priv->parameters;
		GList* param_it;
		for (param_it = __temp8; param_it != NULL; param_it = param_it->next) {
			ValaFormalParameter* param = param_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (param), visitor);
			}
		}
	}
	vala_code_visitor_visit_end_callback (visitor, self);
}


static char* vala_callback_real_get_cname (ValaDataType* base, gboolean const_type)
{
	ValaCallback * self = VALA_CALLBACK (base);
	if (self->priv->cname == NULL) {
		char* __temp10 = NULL;
		char* __temp9 = NULL;
		self->priv->cname = (__temp10 = g_strdup_printf ("%s%s", (__temp9 = vala_namespace_get_cprefix (VALA_DATA_TYPE (self)->namespace)), vala_data_type_get_name (VALA_DATA_TYPE (self))), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp10);
		(__temp9 = (g_free (__temp9), NULL));
	}
	const char* __temp11 = NULL;
	return (__temp11 = self->priv->cname, (__temp11 == NULL ? NULL : g_strdup (__temp11)));
}


/**
 * Sets the name of this callback as it is used in C code.
 *
 * @param cname the name to be used in C code
 */
void vala_callback_set_cname (ValaCallback* self, const char* cname)
{
	g_return_if_fail (VALA_IS_CALLBACK (self));
	char* __temp14 = NULL;
	const char* __temp13 = NULL;
	self->priv->cname = (__temp14 = (__temp13 = cname, (__temp13 == NULL ? NULL : g_strdup (__temp13))), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp14);
}


static void vala_callback_process_ccode_attribute (ValaCallback* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_CALLBACK (self));
	g_return_if_fail (a == NULL || VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cname")) {
		char* __temp15 = NULL;
		vala_callback_set_cname (self, (__temp15 = vala_attribute_get_string (a, "cname")));
		(__temp15 = (g_free (__temp15), NULL));
	}
}


/**
 * Process all associated attributes.
 */
void vala_callback_process_attributes (ValaCallback* self)
{
	g_return_if_fail (VALA_IS_CALLBACK (self));
	{
		GList* __temp16 = NULL;
		__temp16 = VALA_CODE_NODE (self)->attributes;
		GList* a_it;
		for (a_it = __temp16; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* a = a_it->data;
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "CCode") == 0) {
					vala_callback_process_ccode_attribute (self, a);
				}
			}
		}
	}
}


static gboolean vala_callback_real_is_reference_type (ValaDataType* base)
{
	ValaCallback * self = VALA_CALLBACK (base);
	return FALSE;
}


static char* vala_callback_real_get_type_id (ValaDataType* base)
{
	ValaCallback * self = VALA_CALLBACK (base);
	return g_strdup ("G_TYPE_POINTER");
}


static char* vala_callback_real_get_marshaller_type_name (ValaDataType* base)
{
	ValaCallback * self = VALA_CALLBACK (base);
	return g_strdup ("POINTER");
}


static char* vala_callback_real_get_get_value_function (ValaDataType* base)
{
	ValaCallback * self = VALA_CALLBACK (base);
	return g_strdup ("g_value_get_pointer");
}


static char* vala_callback_real_get_set_value_function (ValaDataType* base)
{
	ValaCallback * self = VALA_CALLBACK (base);
	return g_strdup ("g_value_set_pointer");
}


ValaTypeReference* vala_callback_get_return_type (ValaCallback* self)
{
	g_return_val_if_fail (VALA_IS_CALLBACK (self), NULL);
	return self->priv->_return_type;
}


void vala_callback_set_return_type (ValaCallback* self, ValaTypeReference* value)
{
	g_return_if_fail (VALA_IS_CALLBACK (self));
	ValaTypeReference* __temp24 = NULL;
	ValaTypeReference* __temp23 = NULL;
	self->priv->_return_type = (__temp24 = (__temp23 = value, (__temp23 == NULL ? NULL : g_object_ref (__temp23))), (self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL))), __temp24);
}


gboolean vala_callback_get_instance (ValaCallback* self)
{
	g_return_val_if_fail (VALA_IS_CALLBACK (self), FALSE);
	return self->priv->_instance;
}


void vala_callback_set_instance (ValaCallback* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CALLBACK (self));
	self->priv->_instance = value;
}


static void vala_callback_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaCallback * self = VALA_CALLBACK (object);
	switch (property_id) {
		case VALA_CALLBACK_RETURN_TYPE:
		g_value_set_object (value, vala_callback_get_return_type (self));
		break;
		case VALA_CALLBACK_INSTANCE:
		g_value_set_boolean (value, vala_callback_get_instance (self));
		break;
	}
}


static void vala_callback_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaCallback * self = VALA_CALLBACK (object);
	switch (property_id) {
		case VALA_CALLBACK_RETURN_TYPE:
		vala_callback_set_return_type (self, g_value_get_object (value));
		break;
		case VALA_CALLBACK_INSTANCE:
		vala_callback_set_instance (self, g_value_get_boolean (value));
		break;
	}
}


static void vala_callback_class_init (ValaCallbackClass * klass)
{
	vala_callback_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaCallbackPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_callback_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_callback_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_callback_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_callback_real_accept;
	VALA_DATA_TYPE_CLASS (klass)->get_cname = vala_callback_real_get_cname;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_type = vala_callback_real_is_reference_type;
	VALA_DATA_TYPE_CLASS (klass)->get_type_id = vala_callback_real_get_type_id;
	VALA_DATA_TYPE_CLASS (klass)->get_marshaller_type_name = vala_callback_real_get_marshaller_type_name;
	VALA_DATA_TYPE_CLASS (klass)->get_get_value_function = vala_callback_real_get_get_value_function;
	VALA_DATA_TYPE_CLASS (klass)->get_set_value_function = vala_callback_real_get_set_value_function;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CALLBACK_RETURN_TYPE, g_param_spec_object ("return-type", "foo", "bar", VALA_TYPE_TYPE_REFERENCE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CALLBACK_INSTANCE, g_param_spec_boolean ("instance", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_callback_init (ValaCallback * self)
{
	self->priv = VALA_CALLBACK_GET_PRIVATE (self);
}


static void vala_callback_dispose (GObject * obj)
{
	ValaCallback * self = VALA_CALLBACK (obj);
	(self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL)));
	(self->priv->type_parameters == NULL ? NULL : (self->priv->type_parameters = (g_list_foreach (self->priv->type_parameters, (GFunc) g_object_unref, NULL), g_list_free (self->priv->type_parameters), NULL)));
	(self->priv->parameters == NULL ? NULL : (self->priv->parameters = (g_list_foreach (self->priv->parameters, (GFunc) g_object_unref, NULL), g_list_free (self->priv->parameters), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	ValaCallbackClass * klass;
	GObjectClass * parent_class;
	klass = VALA_CALLBACK_CLASS (g_type_class_peek (VALA_TYPE_CALLBACK));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_callback_get_type ()
{
	static GType vala_callback_type_id = 0;
	if (G_UNLIKELY (vala_callback_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCallbackClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_callback_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCallback), 0, (GInstanceInitFunc) vala_callback_init };
		vala_callback_type_id = g_type_register_static (VALA_TYPE_DATA_TYPE, "ValaCallback", &g_define_type_info, 0);
	}
	return vala_callback_type_id;
}




