/* valaclass.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valaclass.h"
#include <vala/valaclass.h>
#include <vala/valamethod.h>
#include <vala/valaconstructor.h>
#include <vala/valadestructor.h>
#include <vala/valasourcereference.h>
#include <vala/valatypereference.h>
#include <vala/valatypeparameter.h>
#include <vala/valaconstant.h>
#include <vala/valafield.h>
#include <vala/valamemberaccessibility.h>
#include <vala/valaproperty.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valasourcefile.h>
#include <vala/valaexpression.h>
#include <vala/valasignal.h>
#include <vala/valacodevisitor.h>
#include <vala/valanamespace.h>
#include <vala/valaattribute.h>

struct _ValaClassPrivate {
	ValaClass* _base_class;
	gboolean _is_abstract;
	gboolean _is_static;
	char* cname;
	char* lower_case_csuffix;
	char* type_id;
	gboolean _has_private_fields;
	GList* type_parameters;
	GList* base_types;
	GList* constants;
	GList* fields;
	GList* methods;
	GList* properties;
	GList* signals;
	ValaMethod* _default_construction_method;
	ValaConstructor* _constructor;
	ValaDestructor* _destructor;
};
#define VALA_CLASS_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CLASS, ValaClassPrivate))
enum  {
	VALA_CLASS_DUMMY_PROPERTY,
	VALA_CLASS_BASE_CLASS,
	VALA_CLASS_IS_ABSTRACT,
	VALA_CLASS_IS_STATIC,
	VALA_CLASS_HAS_PRIVATE_FIELDS,
	VALA_CLASS_DEFAULT_CONSTRUCTION_METHOD,
	VALA_CLASS_CONSTRUCTOR,
	VALA_CLASS_DESTRUCTOR
};
static void vala_class_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_class_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_class_real_get_cname (ValaDataType* base, gboolean const_type);
static char* vala_class_get_lower_case_csuffix (ValaClass* self);
static char* vala_class_real_get_lower_case_cname (ValaDataType* base, const char* infix);
static char* vala_class_real_get_lower_case_cprefix (ValaDataType* base);
static char* vala_class_real_get_upper_case_cname (ValaDataType* base, const char* infix);
static gboolean vala_class_real_is_reference_type (ValaDataType* base);
static void vala_class_process_ccode_attribute (ValaClass* self, ValaAttribute* a);
static char* vala_class_real_get_type_id (ValaDataType* base);
static char* vala_class_real_get_marshaller_type_name (ValaDataType* base);
static char* vala_class_real_get_get_value_function (ValaDataType* base);
static char* vala_class_real_get_set_value_function (ValaDataType* base);
static gboolean vala_class_real_is_reference_counting (ValaDataType* base);
static char* vala_class_real_get_ref_function (ValaDataType* base);
static char* vala_class_real_get_unref_function (ValaDataType* base);
static gboolean vala_class_real_is_subtype_of (ValaDataType* base, ValaDataType* t);
static gint vala_class_real_get_type_parameter_index (ValaDataType* base, const char* name);
static gpointer vala_class_parent_class = NULL;
static void vala_class_dispose (GObject * obj);


/**
 * Creates a new class.
 *
 * @param name   type name
 * @param source reference to source code
 * @return       newly created class
 */
ValaClass* vala_class_new (const char* _name, ValaSourceReference* source)
{
	GParameter * __params;
	GParameter * __params_it;
	ValaClass * self;
	g_return_val_if_fail (_name != NULL, NULL);
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	__params = g_new0 (GParameter, 2);
	__params_it = __params;
	(__params_it->name = "name", g_value_init (&__params_it->value, G_TYPE_STRING), g_value_set_string (&__params_it->value, _name), __params_it++);
	(__params_it->name = "source-reference", g_value_init (&__params_it->value, VALA_TYPE_SOURCE_REFERENCE), g_value_set_object (&__params_it->value, source), __params_it++);
	self = g_object_newv (VALA_TYPE_CLASS, __params_it - __params, __params);
	return self;
}


/**
 * Adds the specified class or interface to the list of base types of
 * this class.
 *
 * @param type a class or interface reference
 */
void vala_class_add_base_type (ValaClass* self, ValaTypeReference* type)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (type));
	self->priv->base_types = g_list_append (self->priv->base_types, g_object_ref (type));
}


/**
 * Returns a copy of the base type list.
 *
 * @return list of base types
 */
GList* vala_class_get_base_types (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return g_list_copy (self->priv->base_types);
}


/**
 * Appends the specified parameter to the list of type parameters.
 *
 * @param p a type parameter
 */
void vala_class_add_type_parameter (ValaClass* self, ValaTypeParameter* p)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_TYPE_PARAMETER (p));
	self->priv->type_parameters = g_list_append (self->priv->type_parameters, g_object_ref (p));
	p->type = VALA_DATA_TYPE (self);
}


/**
 * Returns a copy of the type parameter list.
 *
 * @return list of type parameters
 */
GList* vala_class_get_type_parameters (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return g_list_copy (self->priv->type_parameters);
}


/**
 * Adds the specified constant as a member to this class.
 *
 * @param c a constant
 */
void vala_class_add_constant (ValaClass* self, ValaConstant* c)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_CONSTANT (c));
	self->priv->constants = g_list_append (self->priv->constants, g_object_ref (c));
}


/**
 * Adds the specified field as a member to this class.
 *
 * @param f a field
 */
void vala_class_add_field (ValaClass* self, ValaField* f)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_FIELD (f));
	/* non_null fields not yet supported due to initialization issues*/
	(vala_type_reference_set_non_null (vala_field_get_type_reference (f), FALSE), vala_type_reference_get_non_null (vala_field_get_type_reference (f)));
	self->priv->fields = g_list_append (self->priv->fields, g_object_ref (f));
	if (f->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE && vala_field_get_instance (f)) {
		self->priv->_has_private_fields = TRUE;
	}
}


/**
 * Returns a copy of the list of fields.
 *
 * @return list of fields
 */
GList* vala_class_get_fields (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return g_list_copy (self->priv->fields);
}


/**
 * Adds the specified method as a member to this class.
 *
 * @param m a method
 */
void vala_class_add_method (ValaClass* self, ValaMethod* m)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	self->priv->methods = g_list_append (self->priv->methods, g_object_ref (m));
}


/**
 * Returns a copy of the list of methods.
 *
 * @return list of methods
 */
GList* vala_class_get_methods (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return g_list_copy (self->priv->methods);
}


/**
 * Adds the specified property as a member to this class.
 *
 * @param prop a property
 */
void vala_class_add_property (ValaClass* self, ValaProperty* prop, gboolean no_field)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	self->priv->properties = g_list_append (self->priv->properties, g_object_ref (prop));
	if (!no_field && vala_property_get_set_accessor (prop) != NULL && vala_property_accessor_get_body (vala_property_get_set_accessor (prop)) == NULL && vala_code_node_get_source_reference (VALA_CODE_NODE (self)) != NULL && !vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (self))))) {
		ValaTypeReference* field_type;
		ValaField* f;
		/* automatic property accessor body generation */
		field_type = vala_type_reference_copy (vala_property_get_type_reference (prop));
		/* non_null fields not yet supported due to initialization issues*/
		(vala_type_reference_set_non_null (field_type, FALSE), vala_type_reference_get_non_null (field_type));
		f = vala_field_new (g_strdup_printf ("_%s", vala_property_get_name (prop)), field_type, NULL, vala_code_node_get_source_reference (VALA_CODE_NODE (prop)));
		f->access = VALA_MEMBER_ACCESSIBILITY_PRIVATE;
		vala_class_add_field (self, f);
		(field_type = (g_object_unref (field_type), NULL));
		(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
	}
}


/**
 * Returns a copy of the list of properties.
 *
 * @return list of properties
 */
GList* vala_class_get_properties (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return g_list_copy (self->priv->properties);
}


/**
 * Adds the specified signal as a member to this class.
 *
 * @param sig a signal
 */
void vala_class_add_signal (ValaClass* self, ValaSignal* sig)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_SIGNAL (sig));
	self->priv->signals = g_list_append (self->priv->signals, g_object_ref (sig));
}


/**
 * Returns a copy of the list of signals.
 *
 * @return list of signals
 */
GList* vala_class_get_signals (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return g_list_copy (self->priv->signals);
}


static void vala_class_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaClass * self;
	self = VALA_CLASS (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_class (visitor, self);
}


static void vala_class_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaClass * self;
	self = VALA_CLASS (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GList* __temp6;
		GList* type_it;
		__temp6 = NULL;
		__temp6 = self->priv->base_types;
		for (type_it = __temp6; type_it != NULL; type_it = type_it->next) {
			ValaTypeReference* type;
			type = type_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (type), visitor);
			}
		}
	}
	{
		GList* __temp7;
		GList* p_it;
		__temp7 = NULL;
		__temp7 = self->priv->type_parameters;
		for (p_it = __temp7; p_it != NULL; p_it = p_it->next) {
			ValaTypeParameter* p;
			p = p_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (p), visitor);
			}
		}
	}
	{
		GList* __temp8;
		GList* f_it;
		__temp8 = NULL;
		__temp8 = self->priv->fields;
		for (f_it = __temp8; f_it != NULL; f_it = f_it->next) {
			ValaField* f;
			f = f_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (f), visitor);
			}
		}
	}
	{
		GList* __temp9;
		GList* c_it;
		__temp9 = NULL;
		__temp9 = self->priv->constants;
		for (c_it = __temp9; c_it != NULL; c_it = c_it->next) {
			ValaConstant* c;
			c = c_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (c), visitor);
			}
		}
	}
	{
		GList* __temp10;
		GList* m_it;
		__temp10 = NULL;
		__temp10 = self->priv->methods;
		for (m_it = __temp10; m_it != NULL; m_it = m_it->next) {
			ValaMethod* m;
			m = m_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (m), visitor);
			}
		}
	}
	{
		GList* __temp11;
		GList* prop_it;
		__temp11 = NULL;
		__temp11 = self->priv->properties;
		for (prop_it = __temp11; prop_it != NULL; prop_it = prop_it->next) {
			ValaProperty* prop;
			prop = prop_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (prop), visitor);
			}
		}
	}
	{
		GList* __temp12;
		GList* sig_it;
		__temp12 = NULL;
		__temp12 = self->priv->signals;
		for (sig_it = __temp12; sig_it != NULL; sig_it = sig_it->next) {
			ValaSignal* sig;
			sig = sig_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (sig), visitor);
			}
		}
	}
	if (vala_class_get_constructor (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_class_get_constructor (self)), visitor);
	}
	if (vala_class_get_destructor (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_class_get_destructor (self)), visitor);
	}
}


static char* vala_class_real_get_cname (ValaDataType* base, gboolean const_type)
{
	ValaClass * self;
	const char* __temp15;
	self = VALA_CLASS (base);
	if (self->priv->cname == NULL) {
		char* __temp14;
		char* __temp13;
		__temp14 = NULL;
		__temp13 = NULL;
		self->priv->cname = (__temp14 = g_strdup_printf ("%s%s", (__temp13 = vala_namespace_get_cprefix (VALA_DATA_TYPE (self)->namespace)), vala_data_type_get_name (VALA_DATA_TYPE (self))), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp14);
		(__temp13 = (g_free (__temp13), NULL));
	}
	__temp15 = NULL;
	return (__temp15 = self->priv->cname, (__temp15 == NULL ? NULL : g_strdup (__temp15)));
}


/**
 * Sets the name of this class as it is used in C code.
 *
 * @param cname the name to be used in C code
 */
void vala_class_set_cname (ValaClass* self, const char* cname)
{
	char* __temp17;
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (cname != NULL);
	__temp17 = NULL;
	self->priv->cname = (__temp17 = g_strdup (cname), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp17);
}


static char* vala_class_get_lower_case_csuffix (ValaClass* self)
{
	const char* __temp19;
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	if (self->priv->lower_case_csuffix == NULL) {
		char* __temp18;
		__temp18 = NULL;
		self->priv->lower_case_csuffix = (__temp18 = vala_namespace_camel_case_to_lower_case (vala_data_type_get_name (VALA_DATA_TYPE (self))), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), __temp18);
	}
	__temp19 = NULL;
	return (__temp19 = self->priv->lower_case_csuffix, (__temp19 == NULL ? NULL : g_strdup (__temp19)));
}


static char* vala_class_real_get_lower_case_cname (ValaDataType* base, const char* infix)
{
	ValaClass * self;
	char* __temp22;
	char* __temp21;
	char* __temp23;
	self = VALA_CLASS (base);
	if (infix == NULL) {
		infix = "";
	}
	__temp22 = NULL;
	__temp21 = NULL;
	__temp23 = NULL;
	return (__temp23 = g_strdup_printf ("%s%s%s", (__temp21 = vala_namespace_get_lower_case_cprefix (VALA_DATA_TYPE (self)->namespace)), infix, (__temp22 = vala_class_get_lower_case_csuffix (self))), (__temp22 = (g_free (__temp22), NULL)), (__temp21 = (g_free (__temp21), NULL)), __temp23);
}


static char* vala_class_real_get_lower_case_cprefix (ValaDataType* base)
{
	ValaClass * self;
	char* __temp25;
	char* __temp26;
	self = VALA_CLASS (base);
	__temp25 = NULL;
	__temp26 = NULL;
	return (__temp26 = g_strdup_printf ("%s_", (__temp25 = vala_data_type_get_lower_case_cname (VALA_DATA_TYPE (self), NULL))), (__temp25 = (g_free (__temp25), NULL)), __temp26);
}


static char* vala_class_real_get_upper_case_cname (ValaDataType* base, const char* infix)
{
	ValaClass * self;
	char* __temp28;
	char* __temp29;
	self = VALA_CLASS (base);
	__temp28 = NULL;
	__temp29 = NULL;
	return (__temp29 = g_utf8_strup ((__temp28 = vala_data_type_get_lower_case_cname (VALA_DATA_TYPE (self), infix)), -1), (__temp28 = (g_free (__temp28), NULL)), __temp29);
}


static gboolean vala_class_real_is_reference_type (ValaDataType* base)
{
	ValaClass * self;
	self = VALA_CLASS (base);
	return TRUE;
}


static void vala_class_process_ccode_attribute (ValaClass* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cname")) {
		char* __temp32;
		__temp32 = NULL;
		vala_class_set_cname (self, (__temp32 = vala_attribute_get_string (a, "cname")));
		(__temp32 = (g_free (__temp32), NULL));
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** __temp33;
			char** filename_it;
			__temp33 = NULL;
			__temp33 = g_strsplit (val, ",", 0);
			for (filename_it = __temp33; *filename_it != NULL; filename_it = filename_it + 1) {
				char* filename;
				filename = *filename_it;
				{
					vala_data_type_add_cheader_filename (VALA_DATA_TYPE (self), filename);
				}
			}
			(__temp33 = (g_free (__temp33), NULL));
		}
		(val = (g_free (val), NULL));
	}
}


/**
 * Process all associated attributes.
 */
void vala_class_process_attributes (ValaClass* self)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	{
		GList* __temp34;
		GList* a_it;
		__temp34 = NULL;
		__temp34 = VALA_CODE_NODE (self)->attributes;
		for (a_it = __temp34; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* a;
			a = a_it->data;
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "CCode") == 0) {
					vala_class_process_ccode_attribute (self, a);
				}
			}
		}
	}
}


static char* vala_class_real_get_type_id (ValaDataType* base)
{
	ValaClass * self;
	const char* __temp36;
	self = VALA_CLASS (base);
	if (self->priv->type_id == NULL) {
		char* __temp35;
		__temp35 = NULL;
		self->priv->type_id = (__temp35 = vala_data_type_get_upper_case_cname (VALA_DATA_TYPE (self), "TYPE_"), (self->priv->type_id = (g_free (self->priv->type_id), NULL)), __temp35);
	}
	__temp36 = NULL;
	return (__temp36 = self->priv->type_id, (__temp36 == NULL ? NULL : g_strdup (__temp36)));
}


static char* vala_class_real_get_marshaller_type_name (ValaDataType* base)
{
	ValaClass * self;
	self = VALA_CLASS (base);
	return g_strdup ("OBJECT");
}


static char* vala_class_real_get_get_value_function (ValaDataType* base)
{
	ValaClass * self;
	self = VALA_CLASS (base);
	return g_strdup ("g_value_get_object");
}


static char* vala_class_real_get_set_value_function (ValaDataType* base)
{
	ValaClass * self;
	self = VALA_CLASS (base);
	return g_strdup ("g_value_set_object");
}


static gboolean vala_class_real_is_reference_counting (ValaDataType* base)
{
	ValaClass * self;
	self = VALA_CLASS (base);
	return TRUE;
}


static char* vala_class_real_get_ref_function (ValaDataType* base)
{
	ValaClass * self;
	self = VALA_CLASS (base);
	return g_strdup ("g_object_ref");
}


static char* vala_class_real_get_unref_function (ValaDataType* base)
{
	ValaClass * self;
	self = VALA_CLASS (base);
	return g_strdup ("g_object_unref");
}


static gboolean vala_class_real_is_subtype_of (ValaDataType* base, ValaDataType* t)
{
	ValaClass * self;
	self = VALA_CLASS (base);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (t), FALSE);
	{
		GList* __temp45;
		GList* base_type_it;
		__temp45 = NULL;
		__temp45 = self->priv->base_types;
		for (base_type_it = __temp45; base_type_it != NULL; base_type_it = base_type_it->next) {
			ValaTypeReference* base_type;
			base_type = base_type_it->data;
			{
				if (vala_type_reference_get_data_type (base_type) == t || vala_data_type_is_subtype_of (vala_type_reference_get_data_type (base_type), t)) {
					return TRUE;
				}
			}
		}
	}
	return FALSE;
}


static gint vala_class_real_get_type_parameter_index (ValaDataType* base, const char* name)
{
	ValaClass * self;
	gint i;
	self = VALA_CLASS (base);
	g_return_val_if_fail (name != NULL, 0);
	i = 0;
	{
		GList* __temp48;
		GList* parameter_it;
		__temp48 = NULL;
		__temp48 = self->priv->type_parameters;
		for (parameter_it = __temp48; parameter_it != NULL; parameter_it = parameter_it->next) {
			ValaTypeParameter* parameter;
			parameter = parameter_it->data;
			{
				if (g_utf8_collate (vala_type_parameter_get_name (parameter), name) == 0) {
					return i;
				}
				i++;
			}
		}
	}
	return -1;
}


ValaClass* vala_class_get_base_class (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return self->priv->_base_class;
}


void vala_class_set_base_class (ValaClass* self, ValaClass* value)
{
	ValaClass* __temp52;
	ValaClass* __temp51;
	g_return_if_fail (VALA_IS_CLASS (self));
	__temp52 = NULL;
	__temp51 = NULL;
	self->priv->_base_class = (__temp52 = (__temp51 = value, (__temp51 == NULL ? NULL : g_object_ref (__temp51))), (self->priv->_base_class == NULL ? NULL : (self->priv->_base_class = (g_object_unref (self->priv->_base_class), NULL))), __temp52);
}


gboolean vala_class_get_is_abstract (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), FALSE);
	return self->priv->_is_abstract;
}


void vala_class_set_is_abstract (ValaClass* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	self->priv->_is_abstract = value;
}


gboolean vala_class_get_is_static (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), FALSE);
	return self->priv->_is_static;
}


void vala_class_set_is_static (ValaClass* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	self->priv->_is_static = value;
}


gboolean vala_class_get_has_private_fields (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), FALSE);
	return self->priv->_has_private_fields;
}


ValaMethod* vala_class_get_default_construction_method (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return self->priv->_default_construction_method;
}


void vala_class_set_default_construction_method (ValaClass* self, ValaMethod* value)
{
	ValaMethod* __temp58;
	ValaMethod* __temp57;
	g_return_if_fail (VALA_IS_CLASS (self));
	__temp58 = NULL;
	__temp57 = NULL;
	self->priv->_default_construction_method = (__temp58 = (__temp57 = value, (__temp57 == NULL ? NULL : g_object_ref (__temp57))), (self->priv->_default_construction_method == NULL ? NULL : (self->priv->_default_construction_method = (g_object_unref (self->priv->_default_construction_method), NULL))), __temp58);
}


ValaConstructor* vala_class_get_constructor (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return self->priv->_constructor;
}


void vala_class_set_constructor (ValaClass* self, ValaConstructor* value)
{
	ValaConstructor* __temp61;
	ValaConstructor* __temp60;
	g_return_if_fail (VALA_IS_CLASS (self));
	__temp61 = NULL;
	__temp60 = NULL;
	self->priv->_constructor = (__temp61 = (__temp60 = value, (__temp60 == NULL ? NULL : g_object_ref (__temp60))), (self->priv->_constructor == NULL ? NULL : (self->priv->_constructor = (g_object_unref (self->priv->_constructor), NULL))), __temp61);
}


ValaDestructor* vala_class_get_destructor (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return self->priv->_destructor;
}


void vala_class_set_destructor (ValaClass* self, ValaDestructor* value)
{
	ValaDestructor* __temp64;
	ValaDestructor* __temp63;
	g_return_if_fail (VALA_IS_CLASS (self));
	__temp64 = NULL;
	__temp63 = NULL;
	self->priv->_destructor = (__temp64 = (__temp63 = value, (__temp63 == NULL ? NULL : g_object_ref (__temp63))), (self->priv->_destructor == NULL ? NULL : (self->priv->_destructor = (g_object_unref (self->priv->_destructor), NULL))), __temp64);
}


static void vala_class_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaClass * self;
	self = VALA_CLASS (object);
	switch (property_id) {
		case VALA_CLASS_BASE_CLASS:
		g_value_set_object (value, vala_class_get_base_class (self));
		break;
		case VALA_CLASS_IS_ABSTRACT:
		g_value_set_boolean (value, vala_class_get_is_abstract (self));
		break;
		case VALA_CLASS_IS_STATIC:
		g_value_set_boolean (value, vala_class_get_is_static (self));
		break;
		case VALA_CLASS_HAS_PRIVATE_FIELDS:
		g_value_set_boolean (value, vala_class_get_has_private_fields (self));
		break;
		case VALA_CLASS_DEFAULT_CONSTRUCTION_METHOD:
		g_value_set_object (value, vala_class_get_default_construction_method (self));
		break;
		case VALA_CLASS_CONSTRUCTOR:
		g_value_set_object (value, vala_class_get_constructor (self));
		break;
		case VALA_CLASS_DESTRUCTOR:
		g_value_set_object (value, vala_class_get_destructor (self));
		break;
	}
}


static void vala_class_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaClass * self;
	self = VALA_CLASS (object);
	switch (property_id) {
		case VALA_CLASS_BASE_CLASS:
		vala_class_set_base_class (self, g_value_get_object (value));
		break;
		case VALA_CLASS_IS_ABSTRACT:
		vala_class_set_is_abstract (self, g_value_get_boolean (value));
		break;
		case VALA_CLASS_IS_STATIC:
		vala_class_set_is_static (self, g_value_get_boolean (value));
		break;
		case VALA_CLASS_DEFAULT_CONSTRUCTION_METHOD:
		vala_class_set_default_construction_method (self, g_value_get_object (value));
		break;
		case VALA_CLASS_CONSTRUCTOR:
		vala_class_set_constructor (self, g_value_get_object (value));
		break;
		case VALA_CLASS_DESTRUCTOR:
		vala_class_set_destructor (self, g_value_get_object (value));
		break;
	}
}


static void vala_class_class_init (ValaClassClass * klass)
{
	vala_class_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaClassPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_class_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_class_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_class_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_class_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_class_real_accept_children;
	VALA_DATA_TYPE_CLASS (klass)->get_cname = vala_class_real_get_cname;
	VALA_DATA_TYPE_CLASS (klass)->get_lower_case_cname = vala_class_real_get_lower_case_cname;
	VALA_DATA_TYPE_CLASS (klass)->get_lower_case_cprefix = vala_class_real_get_lower_case_cprefix;
	VALA_DATA_TYPE_CLASS (klass)->get_upper_case_cname = vala_class_real_get_upper_case_cname;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_type = vala_class_real_is_reference_type;
	VALA_DATA_TYPE_CLASS (klass)->get_type_id = vala_class_real_get_type_id;
	VALA_DATA_TYPE_CLASS (klass)->get_marshaller_type_name = vala_class_real_get_marshaller_type_name;
	VALA_DATA_TYPE_CLASS (klass)->get_get_value_function = vala_class_real_get_get_value_function;
	VALA_DATA_TYPE_CLASS (klass)->get_set_value_function = vala_class_real_get_set_value_function;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_counting = vala_class_real_is_reference_counting;
	VALA_DATA_TYPE_CLASS (klass)->get_ref_function = vala_class_real_get_ref_function;
	VALA_DATA_TYPE_CLASS (klass)->get_unref_function = vala_class_real_get_unref_function;
	VALA_DATA_TYPE_CLASS (klass)->is_subtype_of = vala_class_real_is_subtype_of;
	VALA_DATA_TYPE_CLASS (klass)->get_type_parameter_index = vala_class_real_get_type_parameter_index;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_BASE_CLASS, g_param_spec_object ("base-class", "foo", "bar", VALA_TYPE_CLASS, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_IS_ABSTRACT, g_param_spec_boolean ("is-abstract", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_IS_STATIC, g_param_spec_boolean ("is-static", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_HAS_PRIVATE_FIELDS, g_param_spec_boolean ("has-private-fields", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_DEFAULT_CONSTRUCTION_METHOD, g_param_spec_object ("default-construction-method", "foo", "bar", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_CONSTRUCTOR, g_param_spec_object ("constructor", "foo", "bar", VALA_TYPE_CONSTRUCTOR, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_DESTRUCTOR, g_param_spec_object ("destructor", "foo", "bar", VALA_TYPE_DESTRUCTOR, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_class_init (ValaClass * self)
{
	self->priv = VALA_CLASS_GET_PRIVATE (self);
}


static void vala_class_dispose (GObject * obj)
{
	ValaClass * self;
	ValaClassClass * klass;
	GObjectClass * parent_class;
	self = VALA_CLASS (obj);
	(self->priv->_base_class == NULL ? NULL : (self->priv->_base_class = (g_object_unref (self->priv->_base_class), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	(self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL));
	(self->priv->type_id = (g_free (self->priv->type_id), NULL));
	(self->priv->type_parameters == NULL ? NULL : (self->priv->type_parameters = (g_list_foreach (self->priv->type_parameters, (GFunc) g_object_unref, NULL), g_list_free (self->priv->type_parameters), NULL)));
	(self->priv->base_types == NULL ? NULL : (self->priv->base_types = (g_list_foreach (self->priv->base_types, (GFunc) g_object_unref, NULL), g_list_free (self->priv->base_types), NULL)));
	(self->priv->constants == NULL ? NULL : (self->priv->constants = (g_list_foreach (self->priv->constants, (GFunc) g_object_unref, NULL), g_list_free (self->priv->constants), NULL)));
	(self->priv->fields == NULL ? NULL : (self->priv->fields = (g_list_foreach (self->priv->fields, (GFunc) g_object_unref, NULL), g_list_free (self->priv->fields), NULL)));
	(self->priv->methods == NULL ? NULL : (self->priv->methods = (g_list_foreach (self->priv->methods, (GFunc) g_object_unref, NULL), g_list_free (self->priv->methods), NULL)));
	(self->priv->properties == NULL ? NULL : (self->priv->properties = (g_list_foreach (self->priv->properties, (GFunc) g_object_unref, NULL), g_list_free (self->priv->properties), NULL)));
	(self->priv->signals == NULL ? NULL : (self->priv->signals = (g_list_foreach (self->priv->signals, (GFunc) g_object_unref, NULL), g_list_free (self->priv->signals), NULL)));
	(self->priv->_default_construction_method == NULL ? NULL : (self->priv->_default_construction_method = (g_object_unref (self->priv->_default_construction_method), NULL)));
	(self->priv->_constructor == NULL ? NULL : (self->priv->_constructor = (g_object_unref (self->priv->_constructor), NULL)));
	(self->priv->_destructor == NULL ? NULL : (self->priv->_destructor = (g_object_unref (self->priv->_destructor), NULL)));
	klass = VALA_CLASS_CLASS (g_type_class_peek (VALA_TYPE_CLASS));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_class_get_type ()
{
	static GType vala_class_type_id = 0;
	if (G_UNLIKELY (vala_class_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaClassClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_class_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaClass), 0, (GInstanceInitFunc) vala_class_init };
		vala_class_type_id = g_type_register_static (VALA_TYPE_DATA_TYPE, "ValaClass", &g_define_type_info, 0);
	}
	return vala_class_type_id;
}




