/* valainterface.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valainterface.h"
#include <vala/valasourcereference.h>
#include <vala/valatypeparameter.h>
#include <vala/valainterface.h>
#include <vala/valatypereference.h>
#include <vala/valamethod.h>
#include <vala/valaproperty.h>
#include <vala/valasignal.h>
#include <vala/valanamespace.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>

struct _ValaInterfacePrivate {
	gboolean _is_static;
	GList* type_parameters;
	GList* prerequisites;
	GList* methods;
	GList* properties;
	GList* signals;
	char* cname;
	char* lower_case_csuffix;
	char* type_cname;
	char* type_id;
};
#define VALA_INTERFACE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_INTERFACE, ValaInterfacePrivate))
enum  {
	VALA_INTERFACE_DUMMY_PROPERTY,
	VALA_INTERFACE_IS_STATIC
};
static char* vala_interface_real_get_cname (ValaDataType* base, gboolean const_type);
static char* vala_interface_real_get_lower_case_cname (ValaDataType* base, const char* infix);
static char* vala_interface_real_get_lower_case_cprefix (ValaDataType* base);
static char* vala_interface_real_get_upper_case_cname (ValaDataType* base, const char* infix);
static void vala_interface_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_interface_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_interface_real_is_reference_type (ValaDataType* base);
static gboolean vala_interface_real_is_reference_counting (ValaDataType* base);
static char* vala_interface_real_get_ref_function (ValaDataType* base);
static char* vala_interface_real_get_unref_function (ValaDataType* base);
static gboolean vala_interface_real_is_subtype_of (ValaDataType* base, ValaDataType* t);
static void vala_interface_process_ccode_attribute (ValaInterface* self, ValaAttribute* a);
static char* vala_interface_real_get_marshaller_type_name (ValaDataType* base);
static char* vala_interface_real_get_get_value_function (ValaDataType* base);
static char* vala_interface_real_get_set_value_function (ValaDataType* base);
static char* vala_interface_real_get_type_id (ValaDataType* base);
static gint vala_interface_real_get_type_parameter_index (ValaDataType* base, const char* name);
static gpointer vala_interface_parent_class = NULL;
static void vala_interface_dispose (GObject * obj);


/**
 * Creates a new interface.
 *
 * @param name   type name
 * @param source reference to source code
 * @return       newly created interface
 */
ValaInterface* vala_interface_new (const char* _name, ValaSourceReference* source)
{
	GParameter * __params;
	GParameter * __params_it;
	ValaInterface * self;
	g_return_val_if_fail (_name != NULL, NULL);
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	__params = g_new0 (GParameter, 2);
	__params_it = __params;
	(__params_it->name = "name", g_value_init (&__params_it->value, G_TYPE_STRING), g_value_set_string (&__params_it->value, _name), __params_it++);
	(__params_it->name = "source-reference", g_value_init (&__params_it->value, VALA_TYPE_SOURCE_REFERENCE), g_value_set_object (&__params_it->value, source), __params_it++);
	self = g_object_newv (VALA_TYPE_INTERFACE, __params_it - __params, __params);
	return self;
}


/**
 * Appends the specified parameter to the list of type parameters.
 *
 * @param p a type parameter
 */
void vala_interface_add_type_parameter (ValaInterface* self, ValaTypeParameter* p)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	g_return_if_fail (VALA_IS_TYPE_PARAMETER (p));
	self->priv->type_parameters = g_list_append (self->priv->type_parameters, g_object_ref (p));
	p->type = VALA_DATA_TYPE (self);
}


/**
 * Adds the specified interface or class to the list of prerequisites of
 * this interface.
 *
 * @param type an interface or class reference
 */
void vala_interface_add_prerequisite (ValaInterface* self, ValaTypeReference* type)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (type));
	self->priv->prerequisites = g_list_append (self->priv->prerequisites, g_object_ref (type));
}


/**
 * Returns a copy of the base type list.
 *
 * @return list of base types
 */
GList* vala_interface_get_prerequisites (ValaInterface* self)
{
	g_return_val_if_fail (VALA_IS_INTERFACE (self), NULL);
	return g_list_copy (self->priv->prerequisites);
}


/**
 * Adds the specified method as a member to this interface.
 *
 * @param m a method
 */
void vala_interface_add_method (ValaInterface* self, ValaMethod* m)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	self->priv->methods = g_list_append (self->priv->methods, g_object_ref (m));
}


/**
 * Returns a copy of the list of methods.
 *
 * @return list of methods
 */
GList* vala_interface_get_methods (ValaInterface* self)
{
	g_return_val_if_fail (VALA_IS_INTERFACE (self), NULL);
	return g_list_copy (self->priv->methods);
}


/**
 * Adds the specified property as a member to this interface.
 *
 * @param prop a property
 */
void vala_interface_add_property (ValaInterface* self, ValaProperty* prop)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	self->priv->properties = g_list_append (self->priv->properties, g_object_ref (prop));
}


/**
 * Returns a copy of the list of properties.
 *
 * @return list of properties
 */
GList* vala_interface_get_properties (ValaInterface* self)
{
	g_return_val_if_fail (VALA_IS_INTERFACE (self), NULL);
	return g_list_copy (self->priv->properties);
}


/**
 * Adds the specified signal as a member to this interface.
 *
 * @param sig a signal
 */
void vala_interface_add_signal (ValaInterface* self, ValaSignal* sig)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	g_return_if_fail (VALA_IS_SIGNAL (sig));
	self->priv->signals = g_list_append (self->priv->signals, g_object_ref (sig));
}


/**
 * Returns a copy of the list of signals.
 *
 * @return list of signals
 */
GList* vala_interface_get_signals (ValaInterface* self)
{
	g_return_val_if_fail (VALA_IS_INTERFACE (self), NULL);
	return g_list_copy (self->priv->signals);
}


static char* vala_interface_real_get_cname (ValaDataType* base, gboolean const_type)
{
	ValaInterface * self;
	const char* __temp6;
	self = VALA_INTERFACE (base);
	if (self->priv->cname == NULL) {
		char* __temp5;
		char* __temp4;
		__temp5 = NULL;
		__temp4 = NULL;
		self->priv->cname = (__temp5 = g_strdup_printf ("%s%s", (__temp4 = vala_namespace_get_cprefix (VALA_DATA_TYPE (self)->namespace)), vala_data_type_get_name (VALA_DATA_TYPE (self))), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp5);
		(__temp4 = (g_free (__temp4), NULL));
	}
	__temp6 = NULL;
	return (__temp6 = self->priv->cname, (__temp6 == NULL ? NULL : g_strdup (__temp6)));
}


/**
 * Returns the string to be prepended to the name of members of this
 * interface when used in C code.
 *
 * @return the suffix to be used in C code
 */
char* vala_interface_get_lower_case_csuffix (ValaInterface* self)
{
	const char* __temp9;
	g_return_val_if_fail (VALA_IS_INTERFACE (self), NULL);
	if (self->priv->lower_case_csuffix == NULL) {
		char* __temp8;
		__temp8 = NULL;
		self->priv->lower_case_csuffix = (__temp8 = vala_namespace_camel_case_to_lower_case (vala_data_type_get_name (VALA_DATA_TYPE (self))), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), __temp8);
	}
	__temp9 = NULL;
	return (__temp9 = self->priv->lower_case_csuffix, (__temp9 == NULL ? NULL : g_strdup (__temp9)));
}


/**
 * Sets the string to be prepended to the name of members of this
 * interface when used in C code.
 *
 * @param csuffix the suffix to be used in C code
 */
void vala_interface_set_lower_case_csuffix (ValaInterface* self, const char* csuffix)
{
	char* __temp11;
	g_return_if_fail (VALA_IS_INTERFACE (self));
	g_return_if_fail (csuffix != NULL);
	__temp11 = NULL;
	self->priv->lower_case_csuffix = (__temp11 = g_strdup (csuffix), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), __temp11);
}


static char* vala_interface_real_get_lower_case_cname (ValaDataType* base, const char* infix)
{
	ValaInterface * self;
	char* __temp13;
	char* __temp12;
	char* __temp14;
	self = VALA_INTERFACE (base);
	if (infix == NULL) {
		infix = "";
	}
	__temp13 = NULL;
	__temp12 = NULL;
	__temp14 = NULL;
	return (__temp14 = g_strdup_printf ("%s%s%s", (__temp12 = vala_namespace_get_lower_case_cprefix (VALA_DATA_TYPE (self)->namespace)), infix, (__temp13 = vala_interface_get_lower_case_csuffix (self))), (__temp13 = (g_free (__temp13), NULL)), (__temp12 = (g_free (__temp12), NULL)), __temp14);
}


static char* vala_interface_real_get_lower_case_cprefix (ValaDataType* base)
{
	ValaInterface * self;
	char* __temp16;
	char* __temp17;
	self = VALA_INTERFACE (base);
	__temp16 = NULL;
	__temp17 = NULL;
	return (__temp17 = g_strdup_printf ("%s_", (__temp16 = vala_data_type_get_lower_case_cname (VALA_DATA_TYPE (self), NULL))), (__temp16 = (g_free (__temp16), NULL)), __temp17);
}


static char* vala_interface_real_get_upper_case_cname (ValaDataType* base, const char* infix)
{
	ValaInterface * self;
	char* __temp19;
	char* __temp20;
	self = VALA_INTERFACE (base);
	__temp19 = NULL;
	__temp20 = NULL;
	return (__temp20 = g_utf8_strup ((__temp19 = vala_data_type_get_lower_case_cname (VALA_DATA_TYPE (self), infix)), -1), (__temp19 = (g_free (__temp19), NULL)), __temp20);
}


static void vala_interface_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaInterface * self;
	self = VALA_INTERFACE (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_interface (visitor, self);
}


static void vala_interface_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaInterface * self;
	self = VALA_INTERFACE (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GList* __temp22;
		GList* type_it;
		__temp22 = NULL;
		__temp22 = self->priv->prerequisites;
		for (type_it = __temp22; type_it != NULL; type_it = type_it->next) {
			ValaTypeReference* type;
			type = type_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (type), visitor);
			}
		}
	}
	{
		GList* __temp23;
		GList* p_it;
		__temp23 = NULL;
		__temp23 = self->priv->type_parameters;
		for (p_it = __temp23; p_it != NULL; p_it = p_it->next) {
			ValaTypeParameter* p;
			p = p_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (p), visitor);
			}
		}
	}
	{
		GList* __temp24;
		GList* m_it;
		__temp24 = NULL;
		__temp24 = self->priv->methods;
		for (m_it = __temp24; m_it != NULL; m_it = m_it->next) {
			ValaMethod* m;
			m = m_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (m), visitor);
			}
		}
	}
	{
		GList* __temp25;
		GList* prop_it;
		__temp25 = NULL;
		__temp25 = self->priv->properties;
		for (prop_it = __temp25; prop_it != NULL; prop_it = prop_it->next) {
			ValaProperty* prop;
			prop = prop_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (prop), visitor);
			}
		}
	}
	{
		GList* __temp26;
		GList* sig_it;
		__temp26 = NULL;
		__temp26 = self->priv->signals;
		for (sig_it = __temp26; sig_it != NULL; sig_it = sig_it->next) {
			ValaSignal* sig;
			sig = sig_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (sig), visitor);
			}
		}
	}
}


static gboolean vala_interface_real_is_reference_type (ValaDataType* base)
{
	ValaInterface * self;
	self = VALA_INTERFACE (base);
	return TRUE;
}


static gboolean vala_interface_real_is_reference_counting (ValaDataType* base)
{
	ValaInterface * self;
	self = VALA_INTERFACE (base);
	return TRUE;
}


static char* vala_interface_real_get_ref_function (ValaDataType* base)
{
	ValaInterface * self;
	self = VALA_INTERFACE (base);
	return g_strdup ("g_object_ref");
}


static char* vala_interface_real_get_unref_function (ValaDataType* base)
{
	ValaInterface * self;
	self = VALA_INTERFACE (base);
	return g_strdup ("g_object_unref");
}


static gboolean vala_interface_real_is_subtype_of (ValaDataType* base, ValaDataType* t)
{
	ValaInterface * self;
	self = VALA_INTERFACE (base);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (t), FALSE);
	{
		GList* __temp32;
		GList* prerequisite_it;
		__temp32 = NULL;
		__temp32 = self->priv->prerequisites;
		for (prerequisite_it = __temp32; prerequisite_it != NULL; prerequisite_it = prerequisite_it->next) {
			ValaTypeReference* prerequisite;
			prerequisite = prerequisite_it->data;
			{
				if (vala_type_reference_get_data_type (prerequisite) == t || vala_data_type_is_subtype_of (vala_type_reference_get_data_type (prerequisite), t)) {
					return TRUE;
				}
			}
		}
	}
	return FALSE;
}


static void vala_interface_process_ccode_attribute (ValaInterface* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "type_cname")) {
		char* __temp34;
		__temp34 = NULL;
		vala_interface_set_type_cname (self, (__temp34 = vala_attribute_get_string (a, "type_cname")));
		(__temp34 = (g_free (__temp34), NULL));
	}
}


/**
 * Process all associated attributes.
 */
void vala_interface_process_attributes (ValaInterface* self)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	{
		GList* __temp35;
		GList* a_it;
		__temp35 = NULL;
		__temp35 = VALA_CODE_NODE (self)->attributes;
		for (a_it = __temp35; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* a;
			a = a_it->data;
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "CCode") == 0) {
					vala_interface_process_ccode_attribute (self, a);
				}
			}
		}
	}
}


/**
 * Returns the name of the type struct as it is used in C code.
 *
 * @return the type struct name to be used in C code
 */
char* vala_interface_get_type_cname (ValaInterface* self)
{
	const char* __temp38;
	g_return_val_if_fail (VALA_IS_INTERFACE (self), NULL);
	if (self->priv->type_cname == NULL) {
		char* __temp37;
		char* __temp36;
		__temp37 = NULL;
		__temp36 = NULL;
		self->priv->type_cname = (__temp37 = g_strdup_printf ("%sIface", (__temp36 = vala_data_type_get_cname (VALA_DATA_TYPE (self), FALSE))), (self->priv->type_cname = (g_free (self->priv->type_cname), NULL)), __temp37);
		(__temp36 = (g_free (__temp36), NULL));
	}
	__temp38 = NULL;
	return (__temp38 = self->priv->type_cname, (__temp38 == NULL ? NULL : g_strdup (__temp38)));
}


/**
 * Sets the name of the type struct as it is used in C code.
 *
 * @param type_cname the type struct name to be used in C code
 */
void vala_interface_set_type_cname (ValaInterface* self, const char* type_cname)
{
	char* __temp40;
	g_return_if_fail (VALA_IS_INTERFACE (self));
	g_return_if_fail (type_cname != NULL);
	__temp40 = NULL;
	self->priv->type_cname = (__temp40 = g_strdup (type_cname), (self->priv->type_cname = (g_free (self->priv->type_cname), NULL)), __temp40);
}


static char* vala_interface_real_get_marshaller_type_name (ValaDataType* base)
{
	ValaInterface * self;
	self = VALA_INTERFACE (base);
	return g_strdup ("OBJECT");
}


static char* vala_interface_real_get_get_value_function (ValaDataType* base)
{
	ValaInterface * self;
	self = VALA_INTERFACE (base);
	return g_strdup ("g_value_get_object");
}


static char* vala_interface_real_get_set_value_function (ValaDataType* base)
{
	ValaInterface * self;
	self = VALA_INTERFACE (base);
	return g_strdup ("g_value_set_object");
}


static char* vala_interface_real_get_type_id (ValaDataType* base)
{
	ValaInterface * self;
	const char* __temp45;
	self = VALA_INTERFACE (base);
	if (self->priv->type_id == NULL) {
		char* __temp44;
		__temp44 = NULL;
		self->priv->type_id = (__temp44 = vala_data_type_get_upper_case_cname (VALA_DATA_TYPE (self), "TYPE_"), (self->priv->type_id = (g_free (self->priv->type_id), NULL)), __temp44);
	}
	__temp45 = NULL;
	return (__temp45 = self->priv->type_id, (__temp45 == NULL ? NULL : g_strdup (__temp45)));
}


static gint vala_interface_real_get_type_parameter_index (ValaDataType* base, const char* name)
{
	ValaInterface * self;
	gint i;
	self = VALA_INTERFACE (base);
	g_return_val_if_fail (name != NULL, 0);
	i = 0;
	{
		GList* __temp48;
		GList* parameter_it;
		__temp48 = NULL;
		__temp48 = self->priv->type_parameters;
		for (parameter_it = __temp48; parameter_it != NULL; parameter_it = parameter_it->next) {
			ValaTypeParameter* parameter;
			parameter = parameter_it->data;
			{
				if (g_utf8_collate (vala_type_parameter_get_name (parameter), name) == 0) {
					return i;
				}
				i++;
			}
		}
	}
	return -1;
}


gboolean vala_interface_get_is_static (ValaInterface* self)
{
	g_return_val_if_fail (VALA_IS_INTERFACE (self), FALSE);
	return self->priv->_is_static;
}


void vala_interface_set_is_static (ValaInterface* self, gboolean value)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	self->priv->_is_static = value;
}


static void vala_interface_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaInterface * self;
	self = VALA_INTERFACE (object);
	switch (property_id) {
		case VALA_INTERFACE_IS_STATIC:
		g_value_set_boolean (value, vala_interface_get_is_static (self));
		break;
	}
}


static void vala_interface_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaInterface * self;
	self = VALA_INTERFACE (object);
	switch (property_id) {
		case VALA_INTERFACE_IS_STATIC:
		vala_interface_set_is_static (self, g_value_get_boolean (value));
		break;
	}
}


static void vala_interface_class_init (ValaInterfaceClass * klass)
{
	vala_interface_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaInterfacePrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_interface_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_interface_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_interface_dispose;
	VALA_DATA_TYPE_CLASS (klass)->get_cname = vala_interface_real_get_cname;
	VALA_DATA_TYPE_CLASS (klass)->get_lower_case_cname = vala_interface_real_get_lower_case_cname;
	VALA_DATA_TYPE_CLASS (klass)->get_lower_case_cprefix = vala_interface_real_get_lower_case_cprefix;
	VALA_DATA_TYPE_CLASS (klass)->get_upper_case_cname = vala_interface_real_get_upper_case_cname;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_interface_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_interface_real_accept_children;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_type = vala_interface_real_is_reference_type;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_counting = vala_interface_real_is_reference_counting;
	VALA_DATA_TYPE_CLASS (klass)->get_ref_function = vala_interface_real_get_ref_function;
	VALA_DATA_TYPE_CLASS (klass)->get_unref_function = vala_interface_real_get_unref_function;
	VALA_DATA_TYPE_CLASS (klass)->is_subtype_of = vala_interface_real_is_subtype_of;
	VALA_DATA_TYPE_CLASS (klass)->get_marshaller_type_name = vala_interface_real_get_marshaller_type_name;
	VALA_DATA_TYPE_CLASS (klass)->get_get_value_function = vala_interface_real_get_get_value_function;
	VALA_DATA_TYPE_CLASS (klass)->get_set_value_function = vala_interface_real_get_set_value_function;
	VALA_DATA_TYPE_CLASS (klass)->get_type_id = vala_interface_real_get_type_id;
	VALA_DATA_TYPE_CLASS (klass)->get_type_parameter_index = vala_interface_real_get_type_parameter_index;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_INTERFACE_IS_STATIC, g_param_spec_boolean ("is-static", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_interface_init (ValaInterface * self)
{
	self->priv = VALA_INTERFACE_GET_PRIVATE (self);
}


static void vala_interface_dispose (GObject * obj)
{
	ValaInterface * self;
	ValaInterfaceClass * klass;
	GObjectClass * parent_class;
	self = VALA_INTERFACE (obj);
	(self->priv->type_parameters == NULL ? NULL : (self->priv->type_parameters = (g_list_foreach (self->priv->type_parameters, (GFunc) g_object_unref, NULL), g_list_free (self->priv->type_parameters), NULL)));
	(self->priv->prerequisites == NULL ? NULL : (self->priv->prerequisites = (g_list_foreach (self->priv->prerequisites, (GFunc) g_object_unref, NULL), g_list_free (self->priv->prerequisites), NULL)));
	(self->priv->methods == NULL ? NULL : (self->priv->methods = (g_list_foreach (self->priv->methods, (GFunc) g_object_unref, NULL), g_list_free (self->priv->methods), NULL)));
	(self->priv->properties == NULL ? NULL : (self->priv->properties = (g_list_foreach (self->priv->properties, (GFunc) g_object_unref, NULL), g_list_free (self->priv->properties), NULL)));
	(self->priv->signals == NULL ? NULL : (self->priv->signals = (g_list_foreach (self->priv->signals, (GFunc) g_object_unref, NULL), g_list_free (self->priv->signals), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	(self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL));
	(self->priv->type_cname = (g_free (self->priv->type_cname), NULL));
	(self->priv->type_id = (g_free (self->priv->type_id), NULL));
	klass = VALA_INTERFACE_CLASS (g_type_class_peek (VALA_TYPE_INTERFACE));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_interface_get_type ()
{
	static GType vala_interface_type_id = 0;
	if (G_UNLIKELY (vala_interface_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaInterfaceClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_interface_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaInterface), 0, (GInstanceInitFunc) vala_interface_init };
		vala_interface_type_id = g_type_register_static (VALA_TYPE_DATA_TYPE, "ValaInterface", &g_define_type_info, 0);
	}
	return vala_interface_type_id;
}




