/* valacodecontext.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valacodecontext.h"
#include <vala/valamethod.h>
#include <vala/valasymbol.h>
#include <vala/valacodecontext.h>
#include <vala/valasourcefile.h>
#include <vala/valacodevisitor.h>
#include <vala/valasourcefilecycle.h>

struct _ValaCodeContextPrivate {
	char* _library;
	gboolean _memory_management;
	gboolean _assert;
	gboolean _checking;
	gboolean _ccode_only;
	gboolean _compile_only;
	char* _output;
	gboolean _debug;
	gint _optlevel;
	gboolean _thread;
	ValaMethod* _module_init_method;
	GList* source_files;
	ValaSymbol* root;
	GList* cycles;
	GList* packages;
};
#define VALA_CODE_CONTEXT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CODE_CONTEXT, ValaCodeContextPrivate))
enum  {
	VALA_CODE_CONTEXT_DUMMY_PROPERTY,
	VALA_CODE_CONTEXT_LIBRARY,
	VALA_CODE_CONTEXT_MEMORY_MANAGEMENT,
	VALA_CODE_CONTEXT_ASSERT,
	VALA_CODE_CONTEXT_CHECKING,
	VALA_CODE_CONTEXT_CCODE_ONLY,
	VALA_CODE_CONTEXT_COMPILE_ONLY,
	VALA_CODE_CONTEXT_OUTPUT,
	VALA_CODE_CONTEXT_DEBUG,
	VALA_CODE_CONTEXT_OPTLEVEL,
	VALA_CODE_CONTEXT_THREAD,
	VALA_CODE_CONTEXT_MODULE_INIT_METHOD
};
static ValaSourceFile* vala_code_context_find_cycle_head (ValaCodeContext* self, ValaSourceFile* file);
static void vala_code_context_visit (ValaCodeContext* self, ValaSourceFile* file, GList* chain);
static gpointer vala_code_context_parent_class = NULL;
static void vala_code_context_dispose (GObject * obj);


/**
 * Returns the root symbol of the code tree.
 *
 * @return root symbol
 */
ValaSymbol* vala_code_context_get_root (ValaCodeContext* self)
{
	ValaSymbol* __temp0;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	__temp0 = NULL;
	return (__temp0 = self->priv->root, (__temp0 == NULL ? NULL : g_object_ref (__temp0)));
}


/**
 * Returns a copy of the list of source files.
 *
 * @return list of source files
 */
GList* vala_code_context_get_source_files (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return g_list_copy (self->priv->source_files);
}


/**
 * Adds the specified file to the list of source files.
 *
 * @param file a source file
 */
void vala_code_context_add_source_file (ValaCodeContext* self, ValaSourceFile* file)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	g_return_if_fail (VALA_IS_SOURCE_FILE (file));
	self->priv->source_files = g_list_append (self->priv->source_files, g_object_ref (file));
}


/**
 * Returns a copy of the list of used packages.
 *
 * @return list of used packages
 */
GList* vala_code_context_get_packages (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return g_list_copy (self->priv->packages);
}


/**
 * Returns whether the specified package is being used.
 *
 * @param pkg a package name
 * @return    true if the specified package is being used
 */
gboolean vala_code_context_has_package (ValaCodeContext* self, const char* pkg)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	g_return_val_if_fail (pkg != NULL, FALSE);
	return g_list_find_custom (self->priv->packages, pkg, ((GCompareFunc) strcmp)) != NULL;
}


/**
 * Adds the specified package to the list of used packages.
 *
 * @param pkg a package name
 */
void vala_code_context_add_package (ValaCodeContext* self, const char* pkg)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	g_return_if_fail (pkg != NULL);
	self->priv->packages = g_list_append (self->priv->packages, g_strdup (pkg));
}


/**
 * Visits the complete code tree file by file.
 *
 * @param visitor the visitor to be called when traversing
 */
void vala_code_context_accept (ValaCodeContext* self, ValaCodeVisitor* visitor)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GList* __temp5;
		GList* file_it;
		__temp5 = NULL;
		__temp5 = self->priv->source_files;
		for (file_it = __temp5; file_it != NULL; file_it = file_it->next) {
			ValaSourceFile* file;
			file = file_it->data;
			{
				vala_source_file_accept (file, visitor);
			}
		}
	}
}


/**
 * Find and resolve cycles in source file dependencies.
 */
void vala_code_context_find_header_cycles (ValaCodeContext* self)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	{
		GList* __temp6;
		GList* file_it;
		__temp6 = NULL;
		__temp6 = self->priv->source_files;
		for (file_it = __temp6; file_it != NULL; file_it = file_it->next) {
			ValaSourceFile* file;
			file = file_it->data;
			{
				/* find cycles in dependencies between source files 
				 we're only interested in internal source files */
				if (vala_source_file_get_pkg (file)) {
					continue;
				}
				if (vala_source_file_get_mark (file) == 0) {
					vala_code_context_visit (self, file, NULL);
				}
			}
		}
	}
	{
		GList* __temp7;
		GList* cycle_it;
		__temp7 = NULL;
		__temp7 = self->priv->cycles;
		for (cycle_it = __temp7; cycle_it != NULL; cycle_it = cycle_it->next) {
			ValaSourceFileCycle* cycle;
			cycle = cycle_it->data;
			{
				/* find one head for each cycle, it must not have any
				 * hard dependencies on other files in the cycle
				 */
				cycle->head = vala_code_context_find_cycle_head (self, VALA_SOURCE_FILE (cycle->files->data));
				(vala_source_file_set_is_cycle_head (cycle->head, TRUE), vala_source_file_get_is_cycle_head (cycle->head));
			}
		}
	}
	{
		GList* __temp11;
		GList* file2_it;
		__temp11 = NULL;
		__temp11 = self->priv->source_files;
		for (file2_it = __temp11; file2_it != NULL; file2_it = file2_it->next) {
			ValaSourceFile* file2;
			file2 = file2_it->data;
			{
				/* connect the source files in a non-cyclic way
				 * cycle members connect to the head of their cycle
				 
				 we're only interested in internal source files */
				if (vala_source_file_get_pkg (file2)) {
					continue;
				}
				{
					GList* __temp10;
					GList* dep_it;
					__temp10 = NULL;
					__temp10 = vala_source_file_get_header_internal_dependencies (file2);
					for (dep_it = __temp10; dep_it != NULL; dep_it = dep_it->next) {
						ValaSourceFile* dep;
						dep = dep_it->data;
						{
							if (vala_source_file_get_cycle (file2) != NULL && vala_source_file_get_cycle (dep) == vala_source_file_get_cycle (file2)) {
								/* in the same cycle */
								if (!vala_source_file_get_is_cycle_head (file2)) {
									char* __temp8;
									/* include header of cycle head */
									__temp8 = NULL;
									vala_source_file_add_header_internal_include (file2, (__temp8 = vala_source_file_get_cinclude_filename (vala_source_file_get_cycle (file2)->head)));
									(__temp8 = (g_free (__temp8), NULL));
								}
							} else {
								char* __temp9;
								/* we can just include the headers if they are not in a cycle or not in the same cycle as the current file */
								__temp9 = NULL;
								vala_source_file_add_header_internal_include (file2, (__temp9 = vala_source_file_get_cinclude_filename (dep)));
								(__temp9 = (g_free (__temp9), NULL));
							}
						}
					}
				}
			}
		}
	}
}


static ValaSourceFile* vala_code_context_find_cycle_head (ValaCodeContext* self, ValaSourceFile* file)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (file), NULL);
	{
		GList* __temp14;
		GList* dep_it;
		__temp14 = NULL;
		__temp14 = vala_source_file_get_header_internal_full_dependencies (file);
		for (dep_it = __temp14; dep_it != NULL; dep_it = dep_it->next) {
			ValaSourceFile* dep;
			dep = dep_it->data;
			{
				if (dep == file) {
					continue;
				}
				{
					GList* __temp13;
					GList* cycle_file_it;
					__temp13 = NULL;
					__temp13 = vala_source_file_get_cycle (file)->files;
					for (cycle_file_it = __temp13; cycle_file_it != NULL; cycle_file_it = cycle_file_it->next) {
						ValaSourceFile* cycle_file;
						cycle_file = cycle_file_it->data;
						{
							/* ignore file-internal dependencies */
							if (dep == cycle_file) {
								return vala_code_context_find_cycle_head (self, dep);
							}
						}
					}
				}
			}
		}
	}
	return file;
}


/* no hard dependencies on members of the same cycle found
 * source file suitable as cycle head
 */
static void vala_code_context_visit (ValaCodeContext* self, ValaSourceFile* file, GList* chain)
{
	GList* l;
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	g_return_if_fail (VALA_IS_SOURCE_FILE (file));
	/* no deep copy available yet
	 * var l = chain.copy ();
	 */
	l = NULL;
	{
		GList* __temp16;
		GList* chain_file_it;
		__temp16 = NULL;
		__temp16 = chain;
		for (chain_file_it = __temp16; chain_file_it != NULL; chain_file_it = chain_file_it->next) {
			ValaSourceFile* chain_file;
			chain_file = chain_file_it->data;
			{
				l = g_list_append (l, chain_file);
			}
		}
	}
	l = g_list_append (l, file);
	/* end workaround 
	 mark file as currently being visited */
	(vala_source_file_set_mark (file, 1), vala_source_file_get_mark (file));
	{
		GList* __temp26;
		GList* dep_it;
		__temp26 = NULL;
		__temp26 = vala_source_file_get_header_internal_dependencies (file);
		for (dep_it = __temp26; dep_it != NULL; dep_it = dep_it->next) {
			ValaSourceFile* dep;
			dep = dep_it->data;
			{
				if (file == dep) {
					continue;
				}
				if (vala_source_file_get_mark (dep) == 1) {
					ValaSourceFileCycle* cycle;
					ValaSourceFileCycle* __temp17;
					gboolean cycle_start_found;
					/* found cycle */
					cycle = g_object_new (VALA_TYPE_SOURCE_FILE_CYCLE, NULL);
					__temp17 = NULL;
					self->priv->cycles = g_list_append (self->priv->cycles, (__temp17 = cycle, (__temp17 == NULL ? NULL : g_object_ref (__temp17))));
					cycle_start_found = FALSE;
					{
						GList* __temp25;
						GList* cycle_file_it;
						__temp25 = NULL;
						__temp25 = l;
						for (cycle_file_it = __temp25; cycle_file_it != NULL; cycle_file_it = cycle_file_it->next) {
							ValaSourceFile* cycle_file;
							cycle_file = cycle_file_it->data;
							{
								ValaSourceFileCycle* __temp18;
								ValaSourceFileCycle* ref_cycle_file_cycle;
								__temp18 = NULL;
								ref_cycle_file_cycle = (__temp18 = vala_source_file_get_cycle (cycle_file), (__temp18 == NULL ? NULL : g_object_ref (__temp18)));
								if (!cycle_start_found) {
									if (cycle_file == dep) {
										cycle_start_found = TRUE;
									}
								}
								if (!cycle_start_found) {
									continue;
								}
								if (vala_source_file_get_cycle (cycle_file) != NULL) {
									GList* newlist;
									GList* __temp20;
									GList* __temp23;
									/* file already in a cycle */
									if (vala_source_file_get_cycle (cycle_file) == cycle) {
										continue;
									}
									/* file is in the same cycle, nothing to do 
									 file is in an other cycle, merge the two cycles 
									 broken memory management cycles.remove (cycle_file.cycle); */
									newlist = NULL;
									{
										GList* __temp19;
										GList* oldcycle_it;
										__temp19 = NULL;
										__temp19 = self->priv->cycles;
										for (oldcycle_it = __temp19; oldcycle_it != NULL; oldcycle_it = oldcycle_it->next) {
											ValaSourceFileCycle* oldcycle;
											oldcycle = oldcycle_it->data;
											{
												if (oldcycle != vala_source_file_get_cycle (cycle_file)) {
													newlist = g_list_append (newlist, oldcycle);
												}
											}
										}
									}
									__temp20 = NULL;
									self->priv->cycles = (__temp20 = NULL, (self->priv->cycles == NULL ? NULL : (self->priv->cycles = (g_list_foreach (self->priv->cycles, (GFunc) g_object_unref, NULL), g_list_free (self->priv->cycles), NULL))), __temp20);
									{
										GList* __temp22;
										GList* newcycle_it;
										__temp22 = NULL;
										__temp22 = newlist;
										for (newcycle_it = __temp22; newcycle_it != NULL; newcycle_it = newcycle_it->next) {
											ValaSourceFileCycle* newcycle;
											newcycle = newcycle_it->data;
											{
												ValaSourceFileCycle* __temp21;
												__temp21 = NULL;
												self->priv->cycles = g_list_append (self->priv->cycles, (__temp21 = newcycle, (__temp21 == NULL ? NULL : g_object_ref (__temp21))));
											}
										}
									}
									__temp23 = NULL;
									newlist = (__temp23 = NULL, (newlist == NULL ? NULL : (newlist = (g_list_free (newlist), NULL))), __temp23);
									{
										GList* __temp24;
										GList* inner_cycle_file_it;
										__temp24 = NULL;
										__temp24 = vala_source_file_get_cycle (cycle_file)->files;
										for (inner_cycle_file_it = __temp24; inner_cycle_file_it != NULL; inner_cycle_file_it = inner_cycle_file_it->next) {
											ValaSourceFile* inner_cycle_file;
											inner_cycle_file = inner_cycle_file_it->data;
											{
												/* end workaround for broken memory management */
												if (vala_source_file_get_cycle (inner_cycle_file) != cycle) {
													/* file in inner cycle not yet added to outer cycle */
													cycle->files = g_list_append (cycle->files, inner_cycle_file);
													(vala_source_file_set_cycle (inner_cycle_file, cycle), vala_source_file_get_cycle (inner_cycle_file));
												}
											}
										}
									}
									(newlist == NULL ? NULL : (newlist = (g_list_free (newlist), NULL)));
								} else {
									cycle->files = g_list_append (cycle->files, cycle_file);
									(vala_source_file_set_cycle (cycle_file, cycle), vala_source_file_get_cycle (cycle_file));
								}
								(ref_cycle_file_cycle == NULL ? NULL : (ref_cycle_file_cycle = (g_object_unref (ref_cycle_file_cycle), NULL)));
							}
						}
					}
					(cycle == NULL ? NULL : (cycle = (g_object_unref (cycle), NULL)));
				} else {
					if (vala_source_file_get_mark (dep) == 0) {
						/* found not yet visited file */
						vala_code_context_visit (self, dep, l);
					}
				}
			}
		}
	}
	/* mark file as successfully visited */
	(vala_source_file_set_mark (file, 2), vala_source_file_get_mark (file));
	(l == NULL ? NULL : (l = (g_list_free (l), NULL)));
}


char* vala_code_context_get_library (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return self->priv->_library;
}


void vala_code_context_set_library (ValaCodeContext* self, const char* value)
{
	char* __temp29;
	const char* __temp28;
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	__temp29 = NULL;
	__temp28 = NULL;
	self->priv->_library = (__temp29 = (__temp28 = value, (__temp28 == NULL ? NULL : g_strdup (__temp28))), (self->priv->_library = (g_free (self->priv->_library), NULL)), __temp29);
}


gboolean vala_code_context_get_memory_management (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_memory_management;
}


void vala_code_context_set_memory_management (ValaCodeContext* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_memory_management = value;
}


gboolean vala_code_context_get_assert (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_assert;
}


void vala_code_context_set_assert (ValaCodeContext* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_assert = value;
}


gboolean vala_code_context_get_checking (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_checking;
}


void vala_code_context_set_checking (ValaCodeContext* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_checking = value;
}


gboolean vala_code_context_get_ccode_only (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_ccode_only;
}


void vala_code_context_set_ccode_only (ValaCodeContext* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_ccode_only = value;
}


gboolean vala_code_context_get_compile_only (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_compile_only;
}


void vala_code_context_set_compile_only (ValaCodeContext* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_compile_only = value;
}


char* vala_code_context_get_output (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return self->priv->_output;
}


void vala_code_context_set_output (ValaCodeContext* self, const char* value)
{
	char* __temp37;
	const char* __temp36;
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	__temp37 = NULL;
	__temp36 = NULL;
	self->priv->_output = (__temp37 = (__temp36 = value, (__temp36 == NULL ? NULL : g_strdup (__temp36))), (self->priv->_output = (g_free (self->priv->_output), NULL)), __temp37);
}


gboolean vala_code_context_get_debug (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_debug;
}


void vala_code_context_set_debug (ValaCodeContext* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_debug = value;
}


gint vala_code_context_get_optlevel (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), 0);
	return self->priv->_optlevel;
}


void vala_code_context_set_optlevel (ValaCodeContext* self, gint value)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_optlevel = value;
}


gboolean vala_code_context_get_thread (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), FALSE);
	return self->priv->_thread;
}


void vala_code_context_set_thread (ValaCodeContext* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	self->priv->_thread = value;
}


ValaMethod* vala_code_context_get_module_init_method (ValaCodeContext* self)
{
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (self), NULL);
	return self->priv->_module_init_method;
}


void vala_code_context_set_module_init_method (ValaCodeContext* self, ValaMethod* value)
{
	ValaMethod* __temp43;
	ValaMethod* __temp42;
	g_return_if_fail (VALA_IS_CODE_CONTEXT (self));
	__temp43 = NULL;
	__temp42 = NULL;
	self->priv->_module_init_method = (__temp43 = (__temp42 = value, (__temp42 == NULL ? NULL : g_object_ref (__temp42))), (self->priv->_module_init_method == NULL ? NULL : (self->priv->_module_init_method = (g_object_unref (self->priv->_module_init_method), NULL))), __temp43);
}


static void vala_code_context_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaCodeContext * self;
	self = VALA_CODE_CONTEXT (object);
	switch (property_id) {
		case VALA_CODE_CONTEXT_LIBRARY:
		g_value_set_string (value, vala_code_context_get_library (self));
		break;
		case VALA_CODE_CONTEXT_MEMORY_MANAGEMENT:
		g_value_set_boolean (value, vala_code_context_get_memory_management (self));
		break;
		case VALA_CODE_CONTEXT_ASSERT:
		g_value_set_boolean (value, vala_code_context_get_assert (self));
		break;
		case VALA_CODE_CONTEXT_CHECKING:
		g_value_set_boolean (value, vala_code_context_get_checking (self));
		break;
		case VALA_CODE_CONTEXT_CCODE_ONLY:
		g_value_set_boolean (value, vala_code_context_get_ccode_only (self));
		break;
		case VALA_CODE_CONTEXT_COMPILE_ONLY:
		g_value_set_boolean (value, vala_code_context_get_compile_only (self));
		break;
		case VALA_CODE_CONTEXT_OUTPUT:
		g_value_set_string (value, vala_code_context_get_output (self));
		break;
		case VALA_CODE_CONTEXT_DEBUG:
		g_value_set_boolean (value, vala_code_context_get_debug (self));
		break;
		case VALA_CODE_CONTEXT_OPTLEVEL:
		g_value_set_int (value, vala_code_context_get_optlevel (self));
		break;
		case VALA_CODE_CONTEXT_THREAD:
		g_value_set_boolean (value, vala_code_context_get_thread (self));
		break;
		case VALA_CODE_CONTEXT_MODULE_INIT_METHOD:
		g_value_set_object (value, vala_code_context_get_module_init_method (self));
		break;
	}
}


static void vala_code_context_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaCodeContext * self;
	self = VALA_CODE_CONTEXT (object);
	switch (property_id) {
		case VALA_CODE_CONTEXT_LIBRARY:
		vala_code_context_set_library (self, g_value_get_string (value));
		break;
		case VALA_CODE_CONTEXT_MEMORY_MANAGEMENT:
		vala_code_context_set_memory_management (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_ASSERT:
		vala_code_context_set_assert (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_CHECKING:
		vala_code_context_set_checking (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_CCODE_ONLY:
		vala_code_context_set_ccode_only (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_COMPILE_ONLY:
		vala_code_context_set_compile_only (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_OUTPUT:
		vala_code_context_set_output (self, g_value_get_string (value));
		break;
		case VALA_CODE_CONTEXT_DEBUG:
		vala_code_context_set_debug (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_OPTLEVEL:
		vala_code_context_set_optlevel (self, g_value_get_int (value));
		break;
		case VALA_CODE_CONTEXT_THREAD:
		vala_code_context_set_thread (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_CONTEXT_MODULE_INIT_METHOD:
		vala_code_context_set_module_init_method (self, g_value_get_object (value));
		break;
	}
}


static void vala_code_context_class_init (ValaCodeContextClass * klass)
{
	vala_code_context_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaCodeContextPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_code_context_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_code_context_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_code_context_dispose;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_LIBRARY, g_param_spec_string ("library", "foo", "bar", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_MEMORY_MANAGEMENT, g_param_spec_boolean ("memory-management", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_ASSERT, g_param_spec_boolean ("assert", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_CHECKING, g_param_spec_boolean ("checking", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_CCODE_ONLY, g_param_spec_boolean ("ccode-only", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_COMPILE_ONLY, g_param_spec_boolean ("compile-only", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_OUTPUT, g_param_spec_string ("output", "foo", "bar", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_DEBUG, g_param_spec_boolean ("debug", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_OPTLEVEL, g_param_spec_int ("optlevel", "foo", "bar", G_MININT, G_MAXINT, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_THREAD, g_param_spec_boolean ("thread", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_CONTEXT_MODULE_INIT_METHOD, g_param_spec_object ("module-init-method", "foo", "bar", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_code_context_init (ValaCodeContext * self)
{
	self->priv = VALA_CODE_CONTEXT_GET_PRIVATE (self);
	self->priv->root = vala_symbol_new (NULL);
}


static void vala_code_context_dispose (GObject * obj)
{
	ValaCodeContext * self;
	ValaCodeContextClass * klass;
	GObjectClass * parent_class;
	self = VALA_CODE_CONTEXT (obj);
	(self->priv->_library = (g_free (self->priv->_library), NULL));
	(self->priv->_output = (g_free (self->priv->_output), NULL));
	(self->priv->_module_init_method == NULL ? NULL : (self->priv->_module_init_method = (g_object_unref (self->priv->_module_init_method), NULL)));
	(self->priv->source_files == NULL ? NULL : (self->priv->source_files = (g_list_foreach (self->priv->source_files, (GFunc) g_object_unref, NULL), g_list_free (self->priv->source_files), NULL)));
	(self->priv->root == NULL ? NULL : (self->priv->root = (g_object_unref (self->priv->root), NULL)));
	(self->priv->cycles == NULL ? NULL : (self->priv->cycles = (g_list_foreach (self->priv->cycles, (GFunc) g_object_unref, NULL), g_list_free (self->priv->cycles), NULL)));
	(self->priv->packages == NULL ? NULL : (self->priv->packages = (g_list_foreach (self->priv->packages, (GFunc) g_free, NULL), g_list_free (self->priv->packages), NULL)));
	klass = VALA_CODE_CONTEXT_CLASS (g_type_class_peek (VALA_TYPE_CODE_CONTEXT));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_code_context_get_type ()
{
	static GType vala_code_context_type_id = 0;
	if (G_UNLIKELY (vala_code_context_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCodeContextClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_code_context_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCodeContext), 0, (GInstanceInitFunc) vala_code_context_init };
		vala_code_context_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaCodeContext", &g_define_type_info, 0);
	}
	return vala_code_context_type_id;
}




