/* valamethod.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valamethod.h"
#include <vala/valatypereference.h>
#include <vala/valablock.h>
#include <vala/valamethod.h>
#include <vala/valaformalparameter.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasymbol.h>
#include <vala/valadatatype.h>
#include <vala/valanamespace.h>
#include <vala/valaclass.h>
#include <vala/valaattribute.h>

struct _ValaMethodPrivate {
	char* _name;
	ValaTypeReference* _return_type;
	ValaBlock* _body;
	gboolean _is_abstract;
	gboolean _is_virtual;
	gboolean _overrides;
	gboolean _returns_modified_pointer;
	gboolean _instance_last;
	gboolean _instance_by_reference;
	ValaMethod* _base_method;
	ValaMethod* _base_interface_method;
	ValaFormalParameter* _this_parameter;
	gboolean _printf_format;
	gboolean _instance;
	GList* parameters;
	char* cname;
	gboolean _no_array_length;
	GList* error_domains;
};
#define VALA_METHOD_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_METHOD, ValaMethodPrivate))
enum  {
	VALA_METHOD_DUMMY_PROPERTY,
	VALA_METHOD_NAME,
	VALA_METHOD_RETURN_TYPE,
	VALA_METHOD_BODY,
	VALA_METHOD_INSTANCE,
	VALA_METHOD_IS_ABSTRACT,
	VALA_METHOD_IS_VIRTUAL,
	VALA_METHOD_OVERRIDES,
	VALA_METHOD_RETURNS_MODIFIED_POINTER,
	VALA_METHOD_INSTANCE_LAST,
	VALA_METHOD_INSTANCE_BY_REFERENCE,
	VALA_METHOD_BASE_METHOD,
	VALA_METHOD_BASE_INTERFACE_METHOD,
	VALA_METHOD_THIS_PARAMETER,
	VALA_METHOD_NO_ARRAY_LENGTH,
	VALA_METHOD_PRINTF_FORMAT
};
static GList* vala_method_real_get_parameters (ValaInvokable* base);
static ValaTypeReference* vala_method_real_get_return_type (ValaInvokable* base);
static gboolean vala_method_real_is_invokable (ValaInvokable* base);
static void vala_method_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_method_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_method_real_get_default_cname (ValaMethod* self);
static void vala_method_process_ccode_attribute (ValaMethod* self, ValaAttribute* a);
static gpointer vala_method_parent_class = NULL;
static ValaInvokableIface* vala_method_vala_invokable_parent_iface = NULL;
static void vala_method_dispose (GObject * obj);


/**
 * Creates a new method.
 *
 * @param name        method name
 * @param return_type method return type
 * @param source      reference to source code
 * @return            newly created method
 */
ValaMethod* vala_method_new (const char* _name, ValaTypeReference* _return_type, ValaSourceReference* source)
{
	GParameter * __params;
	GParameter * __params_it;
	ValaMethod * self;
	g_return_val_if_fail (_return_type == NULL || VALA_IS_TYPE_REFERENCE (_return_type), NULL);
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	__params = g_new0 (GParameter, 3);
	__params_it = __params;
	(__params_it->name = "name", g_value_init (&__params_it->value, G_TYPE_STRING), g_value_set_string (&__params_it->value, _name), __params_it++);
	(__params_it->name = "return-type", g_value_init (&__params_it->value, VALA_TYPE_TYPE_REFERENCE), g_value_set_object (&__params_it->value, _return_type), __params_it++);
	(__params_it->name = "source-reference", g_value_init (&__params_it->value, VALA_TYPE_SOURCE_REFERENCE), g_value_set_object (&__params_it->value, source), __params_it++);
	self = g_object_newv (VALA_TYPE_METHOD, __params_it - __params, __params);
	return self;
}


/**
 * Appends parameter to this method.
 *
 * @param param a formal parameter
 */
void vala_method_add_parameter (ValaMethod* self, ValaFormalParameter* param)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (param));
	if (vala_method_get_no_array_length (self)) {
		(vala_formal_parameter_set_no_array_length (param, TRUE), vala_formal_parameter_get_no_array_length (param));
	}
	self->priv->parameters = g_list_append (self->priv->parameters, g_object_ref (param));
}


static GList* vala_method_real_get_parameters (ValaInvokable* base)
{
	ValaMethod * self;
	self = VALA_METHOD (base);
	return g_list_copy (self->priv->parameters);
}


static ValaTypeReference* vala_method_real_get_return_type (ValaInvokable* base)
{
	ValaMethod * self;
	ValaTypeReference* __temp1;
	self = VALA_METHOD (base);
	__temp1 = NULL;
	return (__temp1 = vala_method_get_return_type (self), (__temp1 == NULL ? NULL : g_object_ref (__temp1)));
}


static gboolean vala_method_real_is_invokable (ValaInvokable* base)
{
	ValaMethod * self;
	self = VALA_METHOD (base);
	return TRUE;
}


static void vala_method_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaMethod * self;
	self = VALA_METHOD (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_method (visitor, self);
}


static void vala_method_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaMethod * self;
	self = VALA_METHOD (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	if (vala_method_get_return_type (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_method_get_return_type (self)), visitor);
	}
	{
		GList* __temp4;
		GList* param_it;
		__temp4 = NULL;
		__temp4 = self->priv->parameters;
		for (param_it = __temp4; param_it != NULL; param_it = param_it->next) {
			ValaFormalParameter* param;
			param = param_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (param), visitor);
			}
		}
	}
	{
		GList* __temp5;
		GList* error_domain_it;
		__temp5 = NULL;
		__temp5 = self->priv->error_domains;
		for (error_domain_it = __temp5; error_domain_it != NULL; error_domain_it = error_domain_it->next) {
			ValaTypeReference* error_domain;
			error_domain = error_domain_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (error_domain), visitor);
			}
		}
	}
	if (vala_method_get_body (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_method_get_body (self)), visitor);
	}
}


/**
 * Returns the interface name of this method as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_method_get_cname (ValaMethod* self)
{
	const char* __temp7;
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (self->priv->cname == NULL) {
		char* __temp6;
		__temp6 = NULL;
		self->priv->cname = (__temp6 = vala_method_get_default_cname (self), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp6);
	}
	__temp7 = NULL;
	return (__temp7 = self->priv->cname, (__temp7 == NULL ? NULL : g_strdup (__temp7)));
}


/**
 * Returns the default interface name of this method as it is used in C
 * code.
 *
 * @return the name to be used in C code by default
 */
static char* vala_method_real_get_default_cname (ValaMethod* self)
{
	ValaCodeNode* __temp9;
	ValaCodeNode* parent;
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	__temp9 = NULL;
	parent = (__temp9 = vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (self)))), (__temp9 == NULL ? NULL : g_object_ref (__temp9)));
	if (VALA_IS_DATA_TYPE (parent)) {
		if (g_str_has_prefix (vala_method_get_name (self), "_")) {
			char* __temp10;
			char* __temp11;
			char* __temp12;
			__temp10 = NULL;
			__temp11 = NULL;
			__temp12 = NULL;
			return (__temp12 = (__temp11 = g_strdup_printf ("_%s%s", (__temp10 = vala_data_type_get_lower_case_cprefix ((VALA_DATA_TYPE (parent)))), g_utf8_offset_to_pointer (vala_method_get_name (self), 1)), (__temp10 = (g_free (__temp10), NULL)), __temp11), (parent == NULL ? NULL : (parent = (g_object_unref (parent), NULL))), __temp12);
		} else {
			char* __temp13;
			char* __temp14;
			char* __temp15;
			__temp13 = NULL;
			__temp14 = NULL;
			__temp15 = NULL;
			return (__temp15 = (__temp14 = g_strdup_printf ("%s%s", (__temp13 = vala_data_type_get_lower_case_cprefix ((VALA_DATA_TYPE (parent)))), vala_method_get_name (self)), (__temp13 = (g_free (__temp13), NULL)), __temp14), (parent == NULL ? NULL : (parent = (g_object_unref (parent), NULL))), __temp15);
		}
	} else {
		if (VALA_IS_NAMESPACE (parent)) {
			char* __temp16;
			char* __temp17;
			char* __temp18;
			__temp16 = NULL;
			__temp17 = NULL;
			__temp18 = NULL;
			return (__temp18 = (__temp17 = g_strdup_printf ("%s%s", (__temp16 = vala_namespace_get_lower_case_cprefix ((VALA_NAMESPACE (parent)))), vala_method_get_name (self)), (__temp16 = (g_free (__temp16), NULL)), __temp17), (parent == NULL ? NULL : (parent = (g_object_unref (parent), NULL))), __temp18);
		} else {
			const char* __temp19;
			char* __temp20;
			__temp19 = NULL;
			__temp20 = NULL;
			return (__temp20 = (__temp19 = vala_method_get_name (self), (__temp19 == NULL ? NULL : g_strdup (__temp19))), (parent == NULL ? NULL : (parent = (g_object_unref (parent), NULL))), __temp20);
		}
	}
	(parent == NULL ? NULL : (parent = (g_object_unref (parent), NULL)));
}


char* vala_method_get_default_cname (ValaMethod* self)
{
	return VALA_METHOD_GET_CLASS (self)->get_default_cname (self);
}


/**
 * Returns the implementation name of this data type as it is used in C
 * code.
 *
 * @return the name to be used in C code
 */
char* vala_method_get_real_cname (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (vala_method_get_base_method (self) != NULL || vala_method_get_base_interface_method (self) != NULL) {
		ValaClass* __temp21;
		ValaClass* parent;
		char* __temp22;
		char* __temp23;
		char* __temp24;
		__temp21 = NULL;
		parent = (__temp21 = VALA_CLASS (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (self))))), (__temp21 == NULL ? NULL : g_object_ref (__temp21)));
		__temp22 = NULL;
		__temp23 = NULL;
		__temp24 = NULL;
		return (__temp24 = (__temp23 = g_strdup_printf ("%s_real_%s", (__temp22 = vala_data_type_get_lower_case_cname (VALA_DATA_TYPE (parent), NULL)), vala_method_get_name (self)), (__temp22 = (g_free (__temp22), NULL)), __temp23), (parent == NULL ? NULL : (parent = (g_object_unref (parent), NULL))), __temp24);
		(parent == NULL ? NULL : (parent = (g_object_unref (parent), NULL)));
	} else {
		return vala_method_get_cname (self);
	}
}


/**
 * Sets the name of this method as it is used in C code.
 *
 * @param cname the name to be used in C code
 */
void vala_method_set_cname (ValaMethod* self, const char* cname)
{
	char* __temp27;
	const char* __temp26;
	g_return_if_fail (VALA_IS_METHOD (self));
	__temp27 = NULL;
	__temp26 = NULL;
	self->priv->cname = (__temp27 = (__temp26 = cname, (__temp26 == NULL ? NULL : g_strdup (__temp26))), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp27);
}


static void vala_method_process_ccode_attribute (ValaMethod* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (a == NULL || VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cname")) {
		char* __temp28;
		__temp28 = NULL;
		vala_method_set_cname (self, (__temp28 = vala_attribute_get_string (a, "cname")));
		(__temp28 = (g_free (__temp28), NULL));
	}
}


/**
 * Process all associated attributes.
 */
void vala_method_process_attributes (ValaMethod* self)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	{
		GList* __temp29;
		GList* a_it;
		__temp29 = NULL;
		__temp29 = VALA_CODE_NODE (self)->attributes;
		for (a_it = __temp29; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* a;
			a = a_it->data;
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "CCode") == 0) {
					vala_method_process_ccode_attribute (self, a);
				} else {
					if (g_utf8_collate (vala_attribute_get_name (a), "ReturnsModifiedPointer") == 0) {
						(vala_method_set_returns_modified_pointer (self, TRUE), vala_method_get_returns_modified_pointer (self));
					} else {
						if (g_utf8_collate (vala_attribute_get_name (a), "InstanceLast") == 0) {
							(vala_method_set_instance_last (self, TRUE), vala_method_get_instance_last (self));
						} else {
							if (g_utf8_collate (vala_attribute_get_name (a), "InstanceByReference") == 0) {
								(vala_method_set_instance_by_reference (self, TRUE), vala_method_get_instance_by_reference (self));
							} else {
								if (g_utf8_collate (vala_attribute_get_name (a), "FloatingReference") == 0) {
									(vala_type_reference_set_floating_reference (vala_method_get_return_type (self), TRUE), vala_type_reference_get_floating_reference (vala_method_get_return_type (self)));
								} else {
									if (g_utf8_collate (vala_attribute_get_name (a), "NoArrayLength") == 0) {
										(vala_method_set_no_array_length (self, TRUE), vala_method_get_no_array_length (self));
									} else {
										if (g_utf8_collate (vala_attribute_get_name (a), "PrintfFormat") == 0) {
											(vala_method_set_printf_format (self, TRUE), vala_method_get_printf_format (self));
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
}


/**
 * Checks whether the arguments and return type of the specified method
 * matches this method.
 *
 * @param m a method
 * @return  true if the specified method is compatible to this method
 */
gboolean vala_method_equals (ValaMethod* self, ValaMethod* m2)
{
	GList* method_params;
	GList* method_params_it;
	gboolean __temp36;
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	g_return_val_if_fail (VALA_IS_METHOD (m2), FALSE);
	if (!vala_type_reference_equals (vala_method_get_return_type (m2), vala_method_get_return_type (self))) {
		return FALSE;
	}
	method_params = vala_invokable_get_parameters (VALA_INVOKABLE (m2));
	method_params_it = method_params;
	{
		GList* __temp34;
		GList* param_it;
		__temp34 = NULL;
		__temp34 = self->priv->parameters;
		for (param_it = __temp34; param_it != NULL; param_it = param_it->next) {
			ValaFormalParameter* param;
			param = param_it->data;
			{
				ValaFormalParameter* __temp32;
				ValaFormalParameter* method_param;
				/* method may not expect less arguments */
				if (method_params_it == NULL) {
					gboolean __temp31;
					return (__temp31 = FALSE, (method_params == NULL ? NULL : (method_params = (g_list_free (method_params), NULL))), __temp31);
				}
				__temp32 = NULL;
				method_param = (__temp32 = VALA_FORMAL_PARAMETER (method_params_it->data), (__temp32 == NULL ? NULL : g_object_ref (__temp32)));
				if (!vala_type_reference_equals (vala_formal_parameter_get_type_reference (method_param), vala_formal_parameter_get_type_reference (param))) {
					gboolean __temp33;
					return (__temp33 = FALSE, (method_param == NULL ? NULL : (method_param = (g_object_unref (method_param), NULL))), __temp33);
				}
				method_params_it = method_params_it->next;
				(method_param == NULL ? NULL : (method_param = (g_object_unref (method_param), NULL)));
			}
		}
	}
	/* method may not expect more arguments */
	if (method_params_it != NULL) {
		gboolean __temp35;
		return (__temp35 = FALSE, (method_params == NULL ? NULL : (method_params = (g_list_free (method_params), NULL))), __temp35);
	}
	return (__temp36 = TRUE, (method_params == NULL ? NULL : (method_params = (g_list_free (method_params), NULL))), __temp36);
	(method_params == NULL ? NULL : (method_params = (g_list_free (method_params), NULL)));
}


/**
 * Adds an error domain to this method.
 *
 * @param error_domain an error domain
 */
void vala_method_add_error_domain (ValaMethod* self, ValaTypeReference* error_domain)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (error_domain));
	self->priv->error_domains = g_list_append (self->priv->error_domains, g_object_ref (error_domain));
}


/**
 * Returns a copy of the list of error domains of this method.
 *
 * @return list of error domains
 */
GList* vala_method_get_error_domains (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return g_list_copy (self->priv->error_domains);
}


char* vala_method_get_name (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_name;
}


void vala_method_set_name (ValaMethod* self, const char* value)
{
	char* __temp40;
	const char* __temp39;
	g_return_if_fail (VALA_IS_METHOD (self));
	__temp40 = NULL;
	__temp39 = NULL;
	self->priv->_name = (__temp40 = (__temp39 = value, (__temp39 == NULL ? NULL : g_strdup (__temp39))), (self->priv->_name = (g_free (self->priv->_name), NULL)), __temp40);
}


ValaTypeReference* vala_method_get_return_type (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_return_type;
}


void vala_method_set_return_type (ValaMethod* self, ValaTypeReference* value)
{
	ValaTypeReference* __temp43;
	ValaTypeReference* __temp42;
	g_return_if_fail (VALA_IS_METHOD (self));
	__temp43 = NULL;
	__temp42 = NULL;
	self->priv->_return_type = (__temp43 = (__temp42 = value, (__temp42 == NULL ? NULL : g_object_ref (__temp42))), (self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL))), __temp43);
}


ValaBlock* vala_method_get_body (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_body;
}


void vala_method_set_body (ValaMethod* self, ValaBlock* value)
{
	ValaBlock* __temp46;
	ValaBlock* __temp45;
	g_return_if_fail (VALA_IS_METHOD (self));
	__temp46 = NULL;
	__temp45 = NULL;
	self->priv->_body = (__temp46 = (__temp45 = value, (__temp45 == NULL ? NULL : g_object_ref (__temp45))), (self->priv->_body == NULL ? NULL : (self->priv->_body = (g_object_unref (self->priv->_body), NULL))), __temp46);
}


gboolean vala_method_get_instance (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_instance;
}


void vala_method_set_instance (ValaMethod* self, gboolean value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_instance = value;
}


gboolean vala_method_get_is_abstract (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_is_abstract;
}


void vala_method_set_is_abstract (ValaMethod* self, gboolean value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_is_abstract = value;
}


gboolean vala_method_get_is_virtual (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_is_virtual;
}


void vala_method_set_is_virtual (ValaMethod* self, gboolean value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_is_virtual = value;
}


gboolean vala_method_get_overrides (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_overrides;
}


void vala_method_set_overrides (ValaMethod* self, gboolean value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_overrides = value;
}


gboolean vala_method_get_returns_modified_pointer (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_returns_modified_pointer;
}


void vala_method_set_returns_modified_pointer (ValaMethod* self, gboolean value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_returns_modified_pointer = value;
}


gboolean vala_method_get_instance_last (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_instance_last;
}


void vala_method_set_instance_last (ValaMethod* self, gboolean value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_instance_last = value;
}


gboolean vala_method_get_instance_by_reference (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_instance_by_reference;
}


void vala_method_set_instance_by_reference (ValaMethod* self, gboolean value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_instance_by_reference = value;
}


ValaMethod* vala_method_get_base_method (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_base_method;
}


void vala_method_set_base_method (ValaMethod* self, ValaMethod* value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_base_method = value;
}


ValaMethod* vala_method_get_base_interface_method (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_base_interface_method;
}


void vala_method_set_base_interface_method (ValaMethod* self, ValaMethod* value)
{
	ValaMethod* __temp57;
	ValaMethod* __temp56;
	g_return_if_fail (VALA_IS_METHOD (self));
	__temp57 = NULL;
	__temp56 = NULL;
	self->priv->_base_interface_method = (__temp57 = (__temp56 = value, (__temp56 == NULL ? NULL : g_object_ref (__temp56))), (self->priv->_base_interface_method == NULL ? NULL : (self->priv->_base_interface_method = (g_object_unref (self->priv->_base_interface_method), NULL))), __temp57);
}


ValaFormalParameter* vala_method_get_this_parameter (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_this_parameter;
}


void vala_method_set_this_parameter (ValaMethod* self, ValaFormalParameter* value)
{
	ValaFormalParameter* __temp60;
	ValaFormalParameter* __temp59;
	g_return_if_fail (VALA_IS_METHOD (self));
	__temp60 = NULL;
	__temp59 = NULL;
	self->priv->_this_parameter = (__temp60 = (__temp59 = value, (__temp59 == NULL ? NULL : g_object_ref (__temp59))), (self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (g_object_unref (self->priv->_this_parameter), NULL))), __temp60);
}


gboolean vala_method_get_no_array_length (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_no_array_length;
}


void vala_method_set_no_array_length (ValaMethod* self, gboolean value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_no_array_length = value;
	{
		GList* __temp62;
		GList* param_it;
		__temp62 = NULL;
		__temp62 = self->priv->parameters;
		for (param_it = __temp62; param_it != NULL; param_it = param_it->next) {
			ValaFormalParameter* param;
			param = param_it->data;
			{
				(vala_formal_parameter_set_no_array_length (param, value), vala_formal_parameter_get_no_array_length (param));
			}
		}
	}
}


gboolean vala_method_get_printf_format (ValaMethod* self)
{
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_printf_format;
}


void vala_method_set_printf_format (ValaMethod* self, gboolean value)
{
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_printf_format = value;
}


static void vala_method_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaMethod * self;
	self = VALA_METHOD (object);
	switch (property_id) {
		case VALA_METHOD_NAME:
		g_value_set_string (value, vala_method_get_name (self));
		break;
		case VALA_METHOD_RETURN_TYPE:
		g_value_set_object (value, vala_method_get_return_type (self));
		break;
		case VALA_METHOD_BODY:
		g_value_set_object (value, vala_method_get_body (self));
		break;
		case VALA_METHOD_INSTANCE:
		g_value_set_boolean (value, vala_method_get_instance (self));
		break;
		case VALA_METHOD_IS_ABSTRACT:
		g_value_set_boolean (value, vala_method_get_is_abstract (self));
		break;
		case VALA_METHOD_IS_VIRTUAL:
		g_value_set_boolean (value, vala_method_get_is_virtual (self));
		break;
		case VALA_METHOD_OVERRIDES:
		g_value_set_boolean (value, vala_method_get_overrides (self));
		break;
		case VALA_METHOD_RETURNS_MODIFIED_POINTER:
		g_value_set_boolean (value, vala_method_get_returns_modified_pointer (self));
		break;
		case VALA_METHOD_INSTANCE_LAST:
		g_value_set_boolean (value, vala_method_get_instance_last (self));
		break;
		case VALA_METHOD_INSTANCE_BY_REFERENCE:
		g_value_set_boolean (value, vala_method_get_instance_by_reference (self));
		break;
		case VALA_METHOD_BASE_METHOD:
		g_value_set_object (value, vala_method_get_base_method (self));
		break;
		case VALA_METHOD_BASE_INTERFACE_METHOD:
		g_value_set_object (value, vala_method_get_base_interface_method (self));
		break;
		case VALA_METHOD_THIS_PARAMETER:
		g_value_set_object (value, vala_method_get_this_parameter (self));
		break;
		case VALA_METHOD_NO_ARRAY_LENGTH:
		g_value_set_boolean (value, vala_method_get_no_array_length (self));
		break;
		case VALA_METHOD_PRINTF_FORMAT:
		g_value_set_boolean (value, vala_method_get_printf_format (self));
		break;
	}
}


static void vala_method_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaMethod * self;
	self = VALA_METHOD (object);
	switch (property_id) {
		case VALA_METHOD_NAME:
		vala_method_set_name (self, g_value_get_string (value));
		break;
		case VALA_METHOD_RETURN_TYPE:
		vala_method_set_return_type (self, g_value_get_object (value));
		break;
		case VALA_METHOD_BODY:
		vala_method_set_body (self, g_value_get_object (value));
		break;
		case VALA_METHOD_INSTANCE:
		vala_method_set_instance (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_IS_ABSTRACT:
		vala_method_set_is_abstract (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_IS_VIRTUAL:
		vala_method_set_is_virtual (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_OVERRIDES:
		vala_method_set_overrides (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_RETURNS_MODIFIED_POINTER:
		vala_method_set_returns_modified_pointer (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_INSTANCE_LAST:
		vala_method_set_instance_last (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_INSTANCE_BY_REFERENCE:
		vala_method_set_instance_by_reference (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_BASE_METHOD:
		vala_method_set_base_method (self, g_value_get_object (value));
		break;
		case VALA_METHOD_BASE_INTERFACE_METHOD:
		vala_method_set_base_interface_method (self, g_value_get_object (value));
		break;
		case VALA_METHOD_THIS_PARAMETER:
		vala_method_set_this_parameter (self, g_value_get_object (value));
		break;
		case VALA_METHOD_NO_ARRAY_LENGTH:
		vala_method_set_no_array_length (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_PRINTF_FORMAT:
		vala_method_set_printf_format (self, g_value_get_boolean (value));
		break;
	}
}


static void vala_method_class_init (ValaMethodClass * klass)
{
	vala_method_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaMethodPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_method_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_method_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_method_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_method_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_method_real_accept_children;
	VALA_METHOD_CLASS (klass)->get_default_cname = vala_method_real_get_default_cname;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_NAME, g_param_spec_string ("name", "foo", "bar", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_RETURN_TYPE, g_param_spec_object ("return-type", "foo", "bar", VALA_TYPE_TYPE_REFERENCE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_BODY, g_param_spec_object ("body", "foo", "bar", VALA_TYPE_BLOCK, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_INSTANCE, g_param_spec_boolean ("instance", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_IS_ABSTRACT, g_param_spec_boolean ("is-abstract", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_IS_VIRTUAL, g_param_spec_boolean ("is-virtual", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_OVERRIDES, g_param_spec_boolean ("overrides", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_RETURNS_MODIFIED_POINTER, g_param_spec_boolean ("returns-modified-pointer", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_INSTANCE_LAST, g_param_spec_boolean ("instance-last", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_INSTANCE_BY_REFERENCE, g_param_spec_boolean ("instance-by-reference", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_BASE_METHOD, g_param_spec_object ("base-method", "foo", "bar", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_BASE_INTERFACE_METHOD, g_param_spec_object ("base-interface-method", "foo", "bar", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_THIS_PARAMETER, g_param_spec_object ("this-parameter", "foo", "bar", VALA_TYPE_FORMAL_PARAMETER, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_NO_ARRAY_LENGTH, g_param_spec_boolean ("no-array-length", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_PRINTF_FORMAT, g_param_spec_boolean ("printf-format", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_method_vala_invokable_interface_init (ValaInvokableIface * iface)
{
	vala_method_vala_invokable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_parameters = vala_method_real_get_parameters;
	iface->get_return_type = vala_method_real_get_return_type;
	iface->is_invokable = vala_method_real_is_invokable;
}


static void vala_method_init (ValaMethod * self)
{
	self->priv = VALA_METHOD_GET_PRIVATE (self);
	self->priv->_instance = TRUE;
}


static void vala_method_dispose (GObject * obj)
{
	ValaMethod * self;
	ValaMethodClass * klass;
	GObjectClass * parent_class;
	self = VALA_METHOD (obj);
	(self->priv->_name = (g_free (self->priv->_name), NULL));
	(self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL)));
	(self->priv->_body == NULL ? NULL : (self->priv->_body = (g_object_unref (self->priv->_body), NULL)));
	(self->priv->_base_interface_method == NULL ? NULL : (self->priv->_base_interface_method = (g_object_unref (self->priv->_base_interface_method), NULL)));
	(self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (g_object_unref (self->priv->_this_parameter), NULL)));
	(self->priv->parameters == NULL ? NULL : (self->priv->parameters = (g_list_foreach (self->priv->parameters, (GFunc) g_object_unref, NULL), g_list_free (self->priv->parameters), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	(self->priv->error_domains == NULL ? NULL : (self->priv->error_domains = (g_list_foreach (self->priv->error_domains, (GFunc) g_object_unref, NULL), g_list_free (self->priv->error_domains), NULL)));
	klass = VALA_METHOD_CLASS (g_type_class_peek (VALA_TYPE_METHOD));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_method_get_type ()
{
	static GType vala_method_type_id = 0;
	if (G_UNLIKELY (vala_method_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaMethodClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_method_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaMethod), 0, (GInstanceInitFunc) vala_method_init };
		static const GInterfaceInfo vala_invokable_info = { (GInterfaceInitFunc) vala_method_vala_invokable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_method_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaMethod", &g_define_type_info, 0);
		g_type_add_interface_static (vala_method_type_id, VALA_TYPE_INVOKABLE, &vala_invokable_info);
	}
	return vala_method_type_id;
}




