/* valaenum.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valaenum.h"
#include <vala/valasourcereference.h>
#include <vala/valaenumvalue.h>
#include <vala/valaenum.h>
#include <vala/valamethod.h>
#include <vala/valacodevisitor.h>
#include <vala/valanamespace.h>
#include <vala/valaattribute.h>

struct _ValaEnumPrivate {
	gboolean _error_domain;
	GList* values;
	GList* methods;
	char* cname;
	char* cprefix;
	char* lower_case_cprefix;
	char* lower_case_csuffix;
};
#define VALA_ENUM_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_ENUM, ValaEnumPrivate))
enum  {
	VALA_ENUM_DUMMY_PROPERTY,
	VALA_ENUM_ERROR_DOMAIN
};
static void vala_enum_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_enum_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_enum_real_get_cname (ValaDataType* base, gboolean const_type);
static char* vala_enum_real_get_lower_case_cprefix (ValaDataType* base);
static char* vala_enum_get_lower_case_csuffix (ValaEnum* self);
static char* vala_enum_real_get_lower_case_cname (ValaDataType* base, const char* infix);
static char* vala_enum_real_get_upper_case_cname (ValaDataType* base, const char* infix);
static gboolean vala_enum_real_is_reference_type (ValaDataType* base);
static void vala_enum_set_cname (ValaEnum* self, const char* cname);
static void vala_enum_process_ccode_attribute (ValaEnum* self, ValaAttribute* a);
static char* vala_enum_real_get_type_id (ValaDataType* base);
static char* vala_enum_real_get_marshaller_type_name (ValaDataType* base);
static char* vala_enum_real_get_get_value_function (ValaDataType* base);
static char* vala_enum_real_get_set_value_function (ValaDataType* base);
static char* vala_enum_real_get_default_value (ValaDataType* base);
static gpointer vala_enum_parent_class = NULL;
static void vala_enum_dispose (GObject * obj);


/**
 * Creates a new enum.
 *
 * @param name             type name
 * @param source_reference reference to source code
 * @return                 newly created enum
 */
ValaEnum* vala_enum_new (const char* name, ValaSourceReference* source_reference)
{
	GParameter * __params;
	GParameter * __params_it;
	ValaEnum * self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	__params = g_new0 (GParameter, 2);
	__params_it = __params;
	(__params_it->name = "name", g_value_init (&__params_it->value, G_TYPE_STRING), g_value_set_string (&__params_it->value, name), __params_it++);
	(__params_it->name = "source-reference", g_value_init (&__params_it->value, VALA_TYPE_SOURCE_REFERENCE), g_value_set_object (&__params_it->value, source_reference), __params_it++);
	self = g_object_newv (VALA_TYPE_ENUM, __params_it - __params, __params);
	return self;
}


/**
 * Appends the specified enum value to the list of values.
 *
 * @param value an enum value
 */
void vala_enum_add_value (ValaEnum* self, ValaEnumValue* value)
{
	g_return_if_fail (VALA_IS_ENUM (self));
	g_return_if_fail (VALA_IS_ENUM_VALUE (value));
	self->priv->values = g_list_append (self->priv->values, g_object_ref (value));
}


/**
 * Adds the specified method as a member to this enum.
 *
 * @param m a method
 */
void vala_enum_add_method (ValaEnum* self, ValaMethod* m)
{
	g_return_if_fail (VALA_IS_ENUM (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	self->priv->methods = g_list_append (self->priv->methods, g_object_ref (m));
}


/**
 * Returns a copy of the list of methods.
 *
 * @return list of methods
 */
GList* vala_enum_get_methods (ValaEnum* self)
{
	g_return_val_if_fail (VALA_IS_ENUM (self), NULL);
	return g_list_copy (self->priv->methods);
}


static void vala_enum_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaEnum * self;
	self = VALA_ENUM (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_enum (visitor, self);
}


static void vala_enum_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaEnum * self;
	self = VALA_ENUM (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GList* __temp1;
		GList* value_it;
		__temp1 = NULL;
		__temp1 = self->priv->values;
		for (value_it = __temp1; value_it != NULL; value_it = value_it->next) {
			ValaEnumValue* value;
			value = value_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (value), visitor);
			}
		}
	}
	{
		GList* __temp2;
		GList* m_it;
		__temp2 = NULL;
		__temp2 = self->priv->methods;
		for (m_it = __temp2; m_it != NULL; m_it = m_it->next) {
			ValaMethod* m;
			m = m_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (m), visitor);
			}
		}
	}
}


static char* vala_enum_real_get_cname (ValaDataType* base, gboolean const_type)
{
	ValaEnum * self;
	const char* __temp5;
	self = VALA_ENUM (base);
	if (self->priv->cname == NULL) {
		char* __temp4;
		char* __temp3;
		__temp4 = NULL;
		__temp3 = NULL;
		self->priv->cname = (__temp4 = g_strdup_printf ("%s%s", (__temp3 = vala_namespace_get_cprefix (VALA_DATA_TYPE (self)->namespace)), vala_data_type_get_name (VALA_DATA_TYPE (self))), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp4);
		(__temp3 = (g_free (__temp3), NULL));
	}
	__temp5 = NULL;
	return (__temp5 = self->priv->cname, (__temp5 == NULL ? NULL : g_strdup (__temp5)));
}


static char* vala_enum_real_get_lower_case_cprefix (ValaDataType* base)
{
	ValaEnum * self;
	const char* __temp9;
	self = VALA_ENUM (base);
	if (self->priv->lower_case_cprefix == NULL) {
		char* __temp8;
		char* __temp7;
		__temp8 = NULL;
		__temp7 = NULL;
		self->priv->lower_case_cprefix = (__temp8 = g_strdup_printf ("%s_", (__temp7 = vala_data_type_get_lower_case_cname (VALA_DATA_TYPE (self), NULL))), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), __temp8);
		(__temp7 = (g_free (__temp7), NULL));
	}
	__temp9 = NULL;
	return (__temp9 = self->priv->lower_case_cprefix, (__temp9 == NULL ? NULL : g_strdup (__temp9)));
}


static char* vala_enum_get_lower_case_csuffix (ValaEnum* self)
{
	const char* __temp12;
	g_return_val_if_fail (VALA_IS_ENUM (self), NULL);
	if (self->priv->lower_case_csuffix == NULL) {
		char* __temp11;
		__temp11 = NULL;
		self->priv->lower_case_csuffix = (__temp11 = vala_namespace_camel_case_to_lower_case (vala_data_type_get_name (VALA_DATA_TYPE (self))), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), __temp11);
	}
	__temp12 = NULL;
	return (__temp12 = self->priv->lower_case_csuffix, (__temp12 == NULL ? NULL : g_strdup (__temp12)));
}


static char* vala_enum_real_get_lower_case_cname (ValaDataType* base, const char* infix)
{
	ValaEnum * self;
	char* __temp15;
	char* __temp14;
	char* __temp16;
	self = VALA_ENUM (base);
	if (infix == NULL) {
		infix = "";
	}
	__temp15 = NULL;
	__temp14 = NULL;
	__temp16 = NULL;
	return (__temp16 = g_strdup_printf ("%s%s%s", (__temp14 = vala_namespace_get_lower_case_cprefix (VALA_DATA_TYPE (self)->namespace)), infix, (__temp15 = vala_enum_get_lower_case_csuffix (self))), (__temp15 = (g_free (__temp15), NULL)), (__temp14 = (g_free (__temp14), NULL)), __temp16);
}


static char* vala_enum_real_get_upper_case_cname (ValaDataType* base, const char* infix)
{
	ValaEnum * self;
	char* __temp20;
	char* __temp19;
	char* __temp18;
	char* __temp21;
	self = VALA_ENUM (base);
	__temp20 = NULL;
	__temp19 = NULL;
	__temp18 = NULL;
	__temp21 = NULL;
	return (__temp21 = g_utf8_strup ((__temp20 = g_strdup_printf ("%s%s", (__temp18 = vala_namespace_get_lower_case_cprefix (VALA_DATA_TYPE (self)->namespace)), (__temp19 = vala_namespace_camel_case_to_lower_case (vala_data_type_get_name (VALA_DATA_TYPE (self)))))), -1), (__temp20 = (g_free (__temp20), NULL)), (__temp19 = (g_free (__temp19), NULL)), (__temp18 = (g_free (__temp18), NULL)), __temp21);
}


static gboolean vala_enum_real_is_reference_type (ValaDataType* base)
{
	ValaEnum * self;
	self = VALA_ENUM (base);
	return FALSE;
}


static void vala_enum_set_cname (ValaEnum* self, const char* cname)
{
	char* __temp24;
	g_return_if_fail (VALA_IS_ENUM (self));
	g_return_if_fail (cname != NULL);
	__temp24 = NULL;
	self->priv->cname = (__temp24 = g_strdup (cname), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp24);
}


/**
 * Returns the string to be prepended to the name of members of this
 * enum when used in C code.
 *
 * @return the prefix to be used in C code
 */
char* vala_enum_get_cprefix (ValaEnum* self)
{
	const char* __temp27;
	g_return_val_if_fail (VALA_IS_ENUM (self), NULL);
	if (self->priv->cprefix == NULL) {
		char* __temp26;
		char* __temp25;
		__temp26 = NULL;
		__temp25 = NULL;
		self->priv->cprefix = (__temp26 = g_strdup_printf ("%s_", (__temp25 = vala_data_type_get_upper_case_cname (VALA_DATA_TYPE (self), NULL))), (self->priv->cprefix = (g_free (self->priv->cprefix), NULL)), __temp26);
		(__temp25 = (g_free (__temp25), NULL));
	}
	__temp27 = NULL;
	return (__temp27 = self->priv->cprefix, (__temp27 == NULL ? NULL : g_strdup (__temp27)));
}


/**
 * Sets the string to be prepended to the name of members of this enum
 * when used in C code.
 *
 * @param cprefix the prefix to be used in C code
 */
void vala_enum_set_cprefix (ValaEnum* self, const char* cprefix)
{
	char* __temp29;
	g_return_if_fail (VALA_IS_ENUM (self));
	g_return_if_fail (cprefix != NULL);
	__temp29 = NULL;
	self->priv->cprefix = (__temp29 = g_strdup (cprefix), (self->priv->cprefix = (g_free (self->priv->cprefix), NULL)), __temp29);
}


static void vala_enum_process_ccode_attribute (ValaEnum* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_ENUM (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cname")) {
		char* __temp30;
		__temp30 = NULL;
		vala_enum_set_cname (self, (__temp30 = vala_attribute_get_string (a, "cname")));
		(__temp30 = (g_free (__temp30), NULL));
	}
	if (vala_attribute_has_argument (a, "cprefix")) {
		char* __temp31;
		__temp31 = NULL;
		vala_enum_set_cprefix (self, (__temp31 = vala_attribute_get_string (a, "cprefix")));
		(__temp31 = (g_free (__temp31), NULL));
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** __temp32;
			char** filename_it;
			__temp32 = NULL;
			__temp32 = g_strsplit (val, ",", 0);
			for (filename_it = __temp32; *filename_it != NULL; filename_it = filename_it + 1) {
				char* filename;
				filename = *filename_it;
				{
					vala_data_type_add_cheader_filename (VALA_DATA_TYPE (self), filename);
				}
			}
			(__temp32 = (g_free (__temp32), NULL));
		}
		(val = (g_free (val), NULL));
	}
}


/**
 * Process all associated attributes.
 */
void vala_enum_process_attributes (ValaEnum* self)
{
	g_return_if_fail (VALA_IS_ENUM (self));
	{
		GList* __temp33;
		GList* a_it;
		__temp33 = NULL;
		__temp33 = VALA_CODE_NODE (self)->attributes;
		for (a_it = __temp33; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* a;
			a = a_it->data;
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "CCode") == 0) {
					vala_enum_process_ccode_attribute (self, a);
				} else {
					if (g_utf8_collate (vala_attribute_get_name (a), "ErrorDomain") == 0) {
						(vala_enum_set_error_domain (self, TRUE), vala_enum_get_error_domain (self));
					}
				}
			}
		}
	}
}


static char* vala_enum_real_get_type_id (ValaDataType* base)
{
	ValaEnum * self;
	self = VALA_ENUM (base);
	return g_strdup ("G_TYPE_INT");
}


/* FIXME: use GType-registered enums*/
static char* vala_enum_real_get_marshaller_type_name (ValaDataType* base)
{
	ValaEnum * self;
	self = VALA_ENUM (base);
	return g_strdup ("ENUM");
}


static char* vala_enum_real_get_get_value_function (ValaDataType* base)
{
	ValaEnum * self;
	self = VALA_ENUM (base);
	return g_strdup ("g_value_get_enum");
}


static char* vala_enum_real_get_set_value_function (ValaDataType* base)
{
	ValaEnum * self;
	self = VALA_ENUM (base);
	return g_strdup ("g_value_set_enum");
}


static char* vala_enum_real_get_default_value (ValaDataType* base)
{
	ValaEnum * self;
	self = VALA_ENUM (base);
	return g_strdup ("0");
}


gboolean vala_enum_get_error_domain (ValaEnum* self)
{
	g_return_val_if_fail (VALA_IS_ENUM (self), FALSE);
	return self->priv->_error_domain;
}


void vala_enum_set_error_domain (ValaEnum* self, gboolean value)
{
	g_return_if_fail (VALA_IS_ENUM (self));
	self->priv->_error_domain = value;
}


static void vala_enum_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaEnum * self;
	self = VALA_ENUM (object);
	switch (property_id) {
		case VALA_ENUM_ERROR_DOMAIN:
		g_value_set_boolean (value, vala_enum_get_error_domain (self));
		break;
	}
}


static void vala_enum_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaEnum * self;
	self = VALA_ENUM (object);
	switch (property_id) {
		case VALA_ENUM_ERROR_DOMAIN:
		vala_enum_set_error_domain (self, g_value_get_boolean (value));
		break;
	}
}


static void vala_enum_class_init (ValaEnumClass * klass)
{
	vala_enum_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaEnumPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_enum_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_enum_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_enum_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_enum_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_enum_real_accept_children;
	VALA_DATA_TYPE_CLASS (klass)->get_cname = vala_enum_real_get_cname;
	VALA_DATA_TYPE_CLASS (klass)->get_lower_case_cprefix = vala_enum_real_get_lower_case_cprefix;
	VALA_DATA_TYPE_CLASS (klass)->get_lower_case_cname = vala_enum_real_get_lower_case_cname;
	VALA_DATA_TYPE_CLASS (klass)->get_upper_case_cname = vala_enum_real_get_upper_case_cname;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_type = vala_enum_real_is_reference_type;
	VALA_DATA_TYPE_CLASS (klass)->get_type_id = vala_enum_real_get_type_id;
	VALA_DATA_TYPE_CLASS (klass)->get_marshaller_type_name = vala_enum_real_get_marshaller_type_name;
	VALA_DATA_TYPE_CLASS (klass)->get_get_value_function = vala_enum_real_get_get_value_function;
	VALA_DATA_TYPE_CLASS (klass)->get_set_value_function = vala_enum_real_get_set_value_function;
	VALA_DATA_TYPE_CLASS (klass)->get_default_value = vala_enum_real_get_default_value;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_ENUM_ERROR_DOMAIN, g_param_spec_boolean ("error-domain", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_enum_init (ValaEnum * self)
{
	self->priv = VALA_ENUM_GET_PRIVATE (self);
}


static void vala_enum_dispose (GObject * obj)
{
	ValaEnum * self;
	ValaEnumClass * klass;
	GObjectClass * parent_class;
	self = VALA_ENUM (obj);
	(self->priv->values == NULL ? NULL : (self->priv->values = (g_list_foreach (self->priv->values, (GFunc) g_object_unref, NULL), g_list_free (self->priv->values), NULL)));
	(self->priv->methods == NULL ? NULL : (self->priv->methods = (g_list_foreach (self->priv->methods, (GFunc) g_object_unref, NULL), g_list_free (self->priv->methods), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	(self->priv->cprefix = (g_free (self->priv->cprefix), NULL));
	(self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL));
	(self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL));
	klass = VALA_ENUM_CLASS (g_type_class_peek (VALA_TYPE_ENUM));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_enum_get_type ()
{
	static GType vala_enum_type_id = 0;
	if (G_UNLIKELY (vala_enum_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaEnumClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_enum_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaEnum), 0, (GInstanceInitFunc) vala_enum_init };
		vala_enum_type_id = g_type_register_static (VALA_TYPE_DATA_TYPE, "ValaEnum", &g_define_type_info, 0);
	}
	return vala_enum_type_id;
}




