/* valacallback.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include "valacallback.h"
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valatypereference.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valacallback.h>
#include <vala/valatypeparameter.h>
#include <vala/valascope.h>
#include <vala/valaformalparameter.h>
#include <vala/valamethod.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>

struct _ValaCallbackPrivate {
	ValaTypeReference* _return_type;
	gboolean _instance;
	GeeList* type_parameters;
	GeeList* parameters;
	char* cname;
};
#define VALA_CALLBACK_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CALLBACK, ValaCallbackPrivate))
enum  {
	VALA_CALLBACK_DUMMY_PROPERTY,
	VALA_CALLBACK_RETURN_TYPE,
	VALA_CALLBACK_INSTANCE
};
static void vala_callback_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_callback_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_callback_real_get_cname (ValaDataType* base, gboolean const_type);
static void vala_callback_process_ccode_attribute (ValaCallback* self, ValaAttribute* a);
static gboolean vala_callback_real_is_reference_type (ValaDataType* base);
static char* vala_callback_real_get_type_id (ValaDataType* base);
static char* vala_callback_real_get_marshaller_type_name (ValaDataType* base);
static char* vala_callback_real_get_get_value_function (ValaDataType* base);
static char* vala_callback_real_get_set_value_function (ValaDataType* base);
static gpointer vala_callback_parent_class = NULL;
static void vala_callback_dispose (GObject * obj);


/**
 * Creates a new callback.
 *
 * @param name        callback type name
 * @param return_type return type
 * @param source      reference to source code
 * @return            newly created callback
 */
ValaCallback* vala_callback_new (const char* name, ValaTypeReference* return_type, ValaSourceReference* source_reference)
{
	ValaCallback * self;
	g_return_val_if_fail (return_type == NULL || VALA_IS_TYPE_REFERENCE (return_type), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_CALLBACK, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_callback_set_return_type (self, return_type);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Appends the specified parameter to the list of type parameters.
 *
 * @param p a type parameter
 */
void vala_callback_add_type_parameter (ValaCallback* self, ValaTypeParameter* p)
{
	g_return_if_fail (VALA_IS_CALLBACK (self));
	g_return_if_fail (VALA_IS_TYPE_PARAMETER (p));
	gee_collection_add (GEE_COLLECTION (self->priv->type_parameters), p);
	p->type = VALA_DATA_TYPE (self);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) p)), VALA_SYMBOL (p));
}


/**
 * Appends paramater to this callback function.
 *
 * @param param a formal parameter
 */
void vala_callback_add_parameter (ValaCallback* self, ValaFormalParameter* param)
{
	g_return_if_fail (VALA_IS_CALLBACK (self));
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (param));
	gee_collection_add (GEE_COLLECTION (self->priv->parameters), param);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) param)), VALA_SYMBOL (param));
}


/**
 * Return copy of parameter list.
 *
 * @return parameter list
 */
GeeCollection* vala_callback_get_parameters (ValaCallback* self)
{
	g_return_val_if_fail (VALA_IS_CALLBACK (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->parameters)));
}


/**
 * Checks whether the arguments and return type of the specified method
 * matches this callback.
 *
 * @param m a method
 * @return  true if the specified method is compatible to this callback
 */
gboolean vala_callback_matches_method (ValaCallback* self, ValaMethod* m)
{
	GeeCollection* method_params;
	GeeIterator* method_params_it;
	gboolean first;
	gboolean __temp4;
	g_return_val_if_fail (VALA_IS_CALLBACK (self), FALSE);
	g_return_val_if_fail (VALA_IS_METHOD (m), FALSE);
	if (!vala_type_reference_stricter (vala_method_get_return_type (m), vala_callback_get_return_type (self))) {
		return FALSE;
	}
	method_params = vala_invokable_get_parameters (VALA_INVOKABLE (m));
	method_params_it = gee_iterable_iterator (GEE_ITERABLE (method_params));
	first = TRUE;
	{
		GeeList* param_collection;
		GeeIterator* param_it;
		param_collection = self->priv->parameters;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = gee_iterator_get (param_it);
			{
				ValaFormalParameter* method_param;
				/* use first callback parameter as instance parameter if
				 * an instance method is being compared to a static
				 * callback
				 */
				if (first && vala_method_get_instance (m) && !vala_callback_get_instance (self)) {
					first = FALSE;
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
					continue;
				}
				/* method is allowed to accept less arguments */
				if (!gee_iterator_next (method_params_it)) {
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
					break;
				}
				method_param = gee_iterator_get (method_params_it);
				if (!vala_type_reference_stricter (vala_formal_parameter_get_type_reference (param), vala_formal_parameter_get_type_reference (method_param))) {
					gboolean __temp2;
					return (__temp2 = FALSE, (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (method_param == NULL ? NULL : (method_param = (g_object_unref (method_param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (method_params == NULL ? NULL : (method_params = (g_object_unref (method_params), NULL))), (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), __temp2);
				}
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
				(method_param == NULL ? NULL : (method_param = (g_object_unref (method_param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
	/* method may not expect more arguments */
	if (gee_iterator_next (method_params_it)) {
		gboolean __temp3;
		return (__temp3 = FALSE, (method_params == NULL ? NULL : (method_params = (g_object_unref (method_params), NULL))), (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), __temp3);
	}
	return (__temp4 = TRUE, (method_params == NULL ? NULL : (method_params = (g_object_unref (method_params), NULL))), (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), __temp4);
	(method_params == NULL ? NULL : (method_params = (g_object_unref (method_params), NULL)));
	(method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL)));
}


static void vala_callback_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaCallback * self;
	self = ((ValaCallback*) base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_callback (visitor, self);
}


static void vala_callback_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaCallback * self;
	self = ((ValaCallback*) base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GeeList* p_collection;
		GeeIterator* p_it;
		p_collection = self->priv->type_parameters;
		p_it = gee_iterable_iterator (GEE_ITERABLE (p_collection));
		while (gee_iterator_next (p_it)) {
			ValaTypeParameter* p;
			p = gee_iterator_get (p_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (p), visitor);
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
		(p_it == NULL ? NULL : (p_it = (g_object_unref (p_it), NULL)));
	}
	vala_code_node_accept (VALA_CODE_NODE (vala_callback_get_return_type (self)), visitor);
	{
		GeeList* param_collection;
		GeeIterator* param_it;
		param_collection = self->priv->parameters;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = gee_iterator_get (param_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (param), visitor);
				(param = (g_object_unref (param), NULL));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
}


static char* vala_callback_real_get_cname (ValaDataType* base, gboolean const_type)
{
	ValaCallback * self;
	const char* __temp7;
	self = ((ValaCallback*) base);
	if (self->priv->cname == NULL) {
		char* __temp6;
		char* __temp5;
		__temp6 = NULL;
		__temp5 = NULL;
		self->priv->cname = (__temp6 = g_strdup_printf ("%s%s", (__temp5 = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol (((ValaSymbol*) self)))), vala_symbol_get_name (((ValaSymbol*) self))), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp6);
		(__temp5 = (g_free (__temp5), NULL));
	}
	__temp7 = NULL;
	return (__temp7 = self->priv->cname, (__temp7 == NULL ? NULL : g_strdup (__temp7)));
}


/**
 * Sets the name of this callback as it is used in C code.
 *
 * @param cname the name to be used in C code
 */
void vala_callback_set_cname (ValaCallback* self, const char* cname)
{
	char* __temp10;
	const char* __temp9;
	g_return_if_fail (VALA_IS_CALLBACK (self));
	__temp10 = NULL;
	__temp9 = NULL;
	self->priv->cname = (__temp10 = (__temp9 = cname, (__temp9 == NULL ? NULL : g_strdup (__temp9))), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp10);
}


static void vala_callback_process_ccode_attribute (ValaCallback* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_CALLBACK (self));
	g_return_if_fail (a == NULL || VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cname")) {
		char* __temp11;
		__temp11 = NULL;
		vala_callback_set_cname (self, (__temp11 = vala_attribute_get_string (a, "cname")));
		(__temp11 = (g_free (__temp11), NULL));
	}
}


/**
 * Process all associated attributes.
 */
void vala_callback_process_attributes (ValaCallback* self)
{
	g_return_if_fail (VALA_IS_CALLBACK (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* __temp12;
			ValaAttribute* a;
			__temp12 = NULL;
			a = (__temp12 = a_it->data, (__temp12 == NULL ? NULL : g_object_ref (__temp12)));
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "CCode") == 0) {
					vala_callback_process_ccode_attribute (self, a);
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


static gboolean vala_callback_real_is_reference_type (ValaDataType* base)
{
	ValaCallback * self;
	self = ((ValaCallback*) base);
	return FALSE;
}


static char* vala_callback_real_get_type_id (ValaDataType* base)
{
	ValaCallback * self;
	self = ((ValaCallback*) base);
	return g_strdup ("G_TYPE_POINTER");
}


static char* vala_callback_real_get_marshaller_type_name (ValaDataType* base)
{
	ValaCallback * self;
	self = ((ValaCallback*) base);
	return g_strdup ("POINTER");
}


static char* vala_callback_real_get_get_value_function (ValaDataType* base)
{
	ValaCallback * self;
	self = ((ValaCallback*) base);
	return g_strdup ("g_value_get_pointer");
}


static char* vala_callback_real_get_set_value_function (ValaDataType* base)
{
	ValaCallback * self;
	self = ((ValaCallback*) base);
	return g_strdup ("g_value_set_pointer");
}


ValaTypeReference* vala_callback_get_return_type (ValaCallback* self)
{
	g_return_val_if_fail (VALA_IS_CALLBACK (self), NULL);
	return self->priv->_return_type;
}


void vala_callback_set_return_type (ValaCallback* self, ValaTypeReference* value)
{
	ValaTypeReference* __temp20;
	ValaTypeReference* __temp19;
	g_return_if_fail (VALA_IS_CALLBACK (self));
	__temp20 = NULL;
	__temp19 = NULL;
	self->priv->_return_type = (__temp20 = (__temp19 = value, (__temp19 == NULL ? NULL : g_object_ref (__temp19))), (self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL))), __temp20);
}


gboolean vala_callback_get_instance (ValaCallback* self)
{
	g_return_val_if_fail (VALA_IS_CALLBACK (self), FALSE);
	return self->priv->_instance;
}


void vala_callback_set_instance (ValaCallback* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CALLBACK (self));
	self->priv->_instance = value;
}


static void vala_callback_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaCallback * self;
	self = VALA_CALLBACK (object);
	switch (property_id) {
		case VALA_CALLBACK_RETURN_TYPE:
		g_value_set_object (value, vala_callback_get_return_type (self));
		break;
		case VALA_CALLBACK_INSTANCE:
		g_value_set_boolean (value, vala_callback_get_instance (self));
		break;
	}
}


static void vala_callback_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaCallback * self;
	self = VALA_CALLBACK (object);
	switch (property_id) {
		case VALA_CALLBACK_RETURN_TYPE:
		vala_callback_set_return_type (self, g_value_get_object (value));
		break;
		case VALA_CALLBACK_INSTANCE:
		vala_callback_set_instance (self, g_value_get_boolean (value));
		break;
	}
}


static void vala_callback_class_init (ValaCallbackClass * klass)
{
	vala_callback_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaCallbackPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_callback_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_callback_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_callback_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_callback_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_callback_real_accept_children;
	VALA_DATA_TYPE_CLASS (klass)->get_cname = vala_callback_real_get_cname;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_type = vala_callback_real_is_reference_type;
	VALA_DATA_TYPE_CLASS (klass)->get_type_id = vala_callback_real_get_type_id;
	VALA_DATA_TYPE_CLASS (klass)->get_marshaller_type_name = vala_callback_real_get_marshaller_type_name;
	VALA_DATA_TYPE_CLASS (klass)->get_get_value_function = vala_callback_real_get_get_value_function;
	VALA_DATA_TYPE_CLASS (klass)->get_set_value_function = vala_callback_real_get_set_value_function;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CALLBACK_RETURN_TYPE, g_param_spec_object ("return-type", "foo", "bar", VALA_TYPE_TYPE_REFERENCE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CALLBACK_INSTANCE, g_param_spec_boolean ("instance", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_callback_init (ValaCallback * self)
{
	self->priv = VALA_CALLBACK_GET_PRIVATE (self);
	self->priv->type_parameters = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->parameters = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
}


static void vala_callback_dispose (GObject * obj)
{
	ValaCallback * self;
	ValaCallbackClass * klass;
	GObjectClass * parent_class;
	self = VALA_CALLBACK (obj);
	(self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL)));
	(self->priv->type_parameters == NULL ? NULL : (self->priv->type_parameters = (g_object_unref (self->priv->type_parameters), NULL)));
	(self->priv->parameters == NULL ? NULL : (self->priv->parameters = (g_object_unref (self->priv->parameters), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	klass = VALA_CALLBACK_CLASS (g_type_class_peek (VALA_TYPE_CALLBACK));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_callback_get_type (void)
{
	static GType vala_callback_type_id = 0;
	if (G_UNLIKELY (vala_callback_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCallbackClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_callback_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCallback), 0, (GInstanceInitFunc) vala_callback_init };
		vala_callback_type_id = g_type_register_static (VALA_TYPE_DATA_TYPE, "ValaCallback", &g_define_type_info, 0);
	}
	return vala_callback_type_id;
}




