/* valamemberaccess.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include "valamemberaccess.h"
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <vala/valasourcereference.h>
#include <vala/valamemberaccess.h>
#include <vala/valatypereference.h>
#include <vala/valacodevisitor.h>

struct _ValaMemberAccessPrivate {
	char* _member_name;
	ValaExpression* _inner;
	GeeList* type_argument_list;
};
#define VALA_MEMBER_ACCESS_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_MEMBER_ACCESS, ValaMemberAccessPrivate))
enum  {
	VALA_MEMBER_ACCESS_DUMMY_PROPERTY,
	VALA_MEMBER_ACCESS_INNER,
	VALA_MEMBER_ACCESS_MEMBER_NAME
};
static void vala_member_access_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_member_access_real_to_string (ValaCodeNode* base);
static void vala_member_access_real_replace (ValaCodeNode* base, ValaCodeNode* old_node, ValaCodeNode* new_node);
static gpointer vala_member_access_parent_class = NULL;
static void vala_member_access_dispose (GObject * obj);


/**
 * Creates a new member access expression.
 *
 * @param inner            parent of the member
 * @param member_name      member name
 * @param source_reference reference to source code
 * @return                 newly created member access expression
 */
ValaMemberAccess* vala_member_access_new (ValaExpression* inner, const char* member_name, ValaSourceReference* source_reference)
{
	ValaMemberAccess * self;
	g_return_val_if_fail (inner == NULL || VALA_IS_EXPRESSION (inner), NULL);
	g_return_val_if_fail (member_name != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_MEMBER_ACCESS, 0, NULL);
	vala_member_access_set_inner (self, inner);
	vala_member_access_set_member_name (self, member_name);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


ValaMemberAccess* vala_member_access_new_simple (const char* member_name, ValaSourceReference* source_reference)
{
	ValaMemberAccess * self;
	g_return_val_if_fail (member_name != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_MEMBER_ACCESS, 0, NULL);
	vala_member_access_set_member_name (self, member_name);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Appends the specified type as generic type argument.
 *
 * @param arg a type reference
 */
void vala_member_access_add_type_argument (ValaMemberAccess* self, ValaTypeReference* arg)
{
	g_return_if_fail (VALA_IS_MEMBER_ACCESS (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (arg));
	gee_collection_add (GEE_COLLECTION (self->priv->type_argument_list), arg);
}


/**
 * Returns a copy of the list of generic type arguments.
 *
 * @return type argument list
 */
GeeCollection* vala_member_access_get_type_arguments (ValaMemberAccess* self)
{
	g_return_val_if_fail (VALA_IS_MEMBER_ACCESS (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->type_argument_list)));
}


static void vala_member_access_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaMemberAccess * self;
	self = ((ValaMemberAccess*) base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	if (vala_member_access_get_inner (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_member_access_get_inner (self)), visitor);
	}
	{
		GeeList* type_arg_collection;
		GeeIterator* type_arg_it;
		type_arg_collection = self->priv->type_argument_list;
		type_arg_it = gee_iterable_iterator (GEE_ITERABLE (type_arg_collection));
		while (gee_iterator_next (type_arg_it)) {
			ValaTypeReference* type_arg;
			type_arg = gee_iterator_get (type_arg_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (type_arg), visitor);
				(type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL)));
			}
		}
		(type_arg_it == NULL ? NULL : (type_arg_it = (g_object_unref (type_arg_it), NULL)));
	}
	vala_code_visitor_visit_member_access (visitor, self);
}


static char* vala_member_access_real_to_string (ValaCodeNode* base)
{
	ValaMemberAccess * self;
	self = ((ValaMemberAccess*) base);
	if (vala_member_access_get_inner (self) == NULL) {
		return g_strdup (vala_member_access_get_member_name (self));
	} else {
		char* __temp2;
		char* __temp3;
		__temp2 = NULL;
		__temp3 = NULL;
		return (__temp3 = g_strdup_printf ("%s.%s", (__temp2 = vala_code_node_to_string (VALA_CODE_NODE (vala_member_access_get_inner (self)))), vala_member_access_get_member_name (self)), (__temp2 = (g_free (__temp2), NULL)), __temp3);
	}
}


static void vala_member_access_real_replace (ValaCodeNode* base, ValaCodeNode* old_node, ValaCodeNode* new_node)
{
	ValaMemberAccess * self;
	self = ((ValaMemberAccess*) base);
	g_return_if_fail (VALA_IS_CODE_NODE (old_node));
	g_return_if_fail (VALA_IS_CODE_NODE (new_node));
	if (VALA_CODE_NODE (vala_member_access_get_inner (self)) == old_node) {
		vala_member_access_set_inner (self, VALA_EXPRESSION (new_node));
	}
}


ValaExpression* vala_member_access_get_inner (ValaMemberAccess* self)
{
	g_return_val_if_fail (VALA_IS_MEMBER_ACCESS (self), NULL);
	return self->priv->_inner;
}


void vala_member_access_set_inner (ValaMemberAccess* self, ValaExpression* value)
{
	ValaExpression* __temp7;
	ValaExpression* __temp6;
	g_return_if_fail (VALA_IS_MEMBER_ACCESS (self));
	__temp7 = NULL;
	__temp6 = NULL;
	self->priv->_inner = (__temp7 = (__temp6 = value, (__temp6 == NULL ? NULL : g_object_ref (__temp6))), (self->priv->_inner == NULL ? NULL : (self->priv->_inner = (g_object_unref (self->priv->_inner), NULL))), __temp7);
	if (self->priv->_inner != NULL) {
		vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_inner), VALA_CODE_NODE (self));
	}
}


char* vala_member_access_get_member_name (ValaMemberAccess* self)
{
	g_return_val_if_fail (VALA_IS_MEMBER_ACCESS (self), NULL);
	return self->priv->_member_name;
}


void vala_member_access_set_member_name (ValaMemberAccess* self, const char* value)
{
	char* __temp9;
	g_return_if_fail (VALA_IS_MEMBER_ACCESS (self));
	__temp9 = NULL;
	self->priv->_member_name = (__temp9 = g_strdup (value), (self->priv->_member_name = (g_free (self->priv->_member_name), NULL)), __temp9);
}


static void vala_member_access_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaMemberAccess * self;
	self = VALA_MEMBER_ACCESS (object);
	switch (property_id) {
		case VALA_MEMBER_ACCESS_INNER:
		g_value_set_object (value, vala_member_access_get_inner (self));
		break;
		case VALA_MEMBER_ACCESS_MEMBER_NAME:
		g_value_set_string (value, vala_member_access_get_member_name (self));
		break;
	}
}


static void vala_member_access_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaMemberAccess * self;
	self = VALA_MEMBER_ACCESS (object);
	switch (property_id) {
		case VALA_MEMBER_ACCESS_INNER:
		vala_member_access_set_inner (self, g_value_get_object (value));
		break;
		case VALA_MEMBER_ACCESS_MEMBER_NAME:
		vala_member_access_set_member_name (self, g_value_get_string (value));
		break;
	}
}


static void vala_member_access_class_init (ValaMemberAccessClass * klass)
{
	vala_member_access_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaMemberAccessPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_member_access_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_member_access_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_member_access_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_member_access_real_accept;
	VALA_CODE_NODE_CLASS (klass)->to_string = vala_member_access_real_to_string;
	VALA_CODE_NODE_CLASS (klass)->replace = vala_member_access_real_replace;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_MEMBER_ACCESS_INNER, g_param_spec_object ("inner", "foo", "bar", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_MEMBER_ACCESS_MEMBER_NAME, g_param_spec_string ("member-name", "foo", "bar", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_member_access_init (ValaMemberAccess * self)
{
	self->priv = VALA_MEMBER_ACCESS_GET_PRIVATE (self);
	self->priv->type_argument_list = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
}


static void vala_member_access_dispose (GObject * obj)
{
	ValaMemberAccess * self;
	ValaMemberAccessClass * klass;
	GObjectClass * parent_class;
	self = VALA_MEMBER_ACCESS (obj);
	(self->priv->_member_name = (g_free (self->priv->_member_name), NULL));
	(self->priv->_inner == NULL ? NULL : (self->priv->_inner = (g_object_unref (self->priv->_inner), NULL)));
	(self->priv->type_argument_list == NULL ? NULL : (self->priv->type_argument_list = (g_object_unref (self->priv->type_argument_list), NULL)));
	klass = VALA_MEMBER_ACCESS_CLASS (g_type_class_peek (VALA_TYPE_MEMBER_ACCESS));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_member_access_get_type (void)
{
	static GType vala_member_access_type_id = 0;
	if (G_UNLIKELY (vala_member_access_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaMemberAccessClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_member_access_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaMemberAccess), 0, (GInstanceInitFunc) vala_member_access_init };
		vala_member_access_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaMemberAccess", &g_define_type_info, 0);
	}
	return vala_member_access_type_id;
}




