/* valaclass.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include "valaclass.h"
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <vala/valaclass.h>
#include <vala/valamethod.h>
#include <vala/valaconstructor.h>
#include <vala/valadestructor.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valatypereference.h>
#include <vala/valatypeparameter.h>
#include <vala/valascope.h>
#include <vala/valaconstant.h>
#include <vala/valafield.h>
#include <vala/valamemberaccessibility.h>
#include <vala/valaformalparameter.h>
#include <vala/valacreationmethod.h>
#include <vala/valaproperty.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valasourcefile.h>
#include <vala/valaexpression.h>
#include <vala/valasignal.h>
#include <vala/valastruct.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>

struct _ValaClassPrivate {
	ValaClass* _base_class;
	gboolean _is_abstract;
	gboolean _is_static;
	char* cname;
	char* lower_case_csuffix;
	char* type_id;
	gboolean _has_private_fields;
	GeeList* type_parameters;
	GeeList* base_types;
	GeeList* constants;
	GeeList* fields;
	GeeList* methods;
	GeeList* properties;
	GeeList* signals;
	GeeList* classes;
	GeeList* structs;
	ValaMethod* _default_construction_method;
	ValaConstructor* _constructor;
	ValaDestructor* _destructor;
};
#define VALA_CLASS_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CLASS, ValaClassPrivate))
enum  {
	VALA_CLASS_DUMMY_PROPERTY,
	VALA_CLASS_BASE_CLASS,
	VALA_CLASS_IS_ABSTRACT,
	VALA_CLASS_IS_STATIC,
	VALA_CLASS_HAS_PRIVATE_FIELDS,
	VALA_CLASS_DEFAULT_CONSTRUCTION_METHOD,
	VALA_CLASS_CONSTRUCTOR,
	VALA_CLASS_DESTRUCTOR
};
static void vala_class_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_class_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_class_real_get_cprefix (ValaSymbol* base);
static char* vala_class_real_get_cname (ValaDataType* base, gboolean const_type);
static char* vala_class_get_lower_case_csuffix (ValaClass* self);
static char* vala_class_real_get_lower_case_cname (ValaSymbol* base, const char* infix);
static char* vala_class_real_get_lower_case_cprefix (ValaSymbol* base);
static char* vala_class_real_get_upper_case_cname (ValaDataType* base, const char* infix);
static gboolean vala_class_real_is_reference_type (ValaDataType* base);
static void vala_class_process_ccode_attribute (ValaClass* self, ValaAttribute* a);
static char* vala_class_real_get_type_id (ValaDataType* base);
static char* vala_class_real_get_marshaller_type_name (ValaDataType* base);
static char* vala_class_real_get_get_value_function (ValaDataType* base);
static char* vala_class_real_get_set_value_function (ValaDataType* base);
static gboolean vala_class_real_is_reference_counting (ValaDataType* base);
static char* vala_class_real_get_ref_function (ValaDataType* base);
static char* vala_class_real_get_unref_function (ValaDataType* base);
static gboolean vala_class_real_is_subtype_of (ValaDataType* base, ValaDataType* t);
static gint vala_class_real_get_type_parameter_index (ValaDataType* base, const char* name);
static gpointer vala_class_parent_class = NULL;
static void vala_class_dispose (GObject * obj);


/**
 * Creates a new class.
 *
 * @param name   type name
 * @param source reference to source code
 * @return       newly created class
 */
ValaClass* vala_class_new (const char* name, ValaSourceReference* source_reference)
{
	ValaClass * self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_CLASS, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Adds the specified class or interface to the list of base types of
 * this class.
 *
 * @param type a class or interface reference
 */
void vala_class_add_base_type (ValaClass* self, ValaTypeReference* type)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (type));
	gee_collection_add (GEE_COLLECTION (self->priv->base_types), type);
}


/**
 * Returns a copy of the base type list.
 *
 * @return list of base types
 */
GeeCollection* vala_class_get_base_types (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->base_types)));
}


/**
 * Appends the specified parameter to the list of type parameters.
 *
 * @param p a type parameter
 */
void vala_class_add_type_parameter (ValaClass* self, ValaTypeParameter* p)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_TYPE_PARAMETER (p));
	gee_collection_add (GEE_COLLECTION (self->priv->type_parameters), p);
	p->type = VALA_DATA_TYPE (self);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) p)), VALA_SYMBOL (p));
}


/**
 * Returns a copy of the type parameter list.
 *
 * @return list of type parameters
 */
GeeCollection* vala_class_get_type_parameters (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->type_parameters)));
}


/**
 * Adds the specified constant as a member to this class.
 *
 * @param c a constant
 */
void vala_class_add_constant (ValaClass* self, ValaConstant* c)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_CONSTANT (c));
	gee_collection_add (GEE_COLLECTION (self->priv->constants), c);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) c)), VALA_SYMBOL (c));
}


/**
 * Adds the specified field as a member to this class.
 *
 * @param f a field
 */
void vala_class_add_field (ValaClass* self, ValaField* f)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_FIELD (f));
	/* non_null fields not yet supported due to initialization issues*/
	vala_type_reference_set_non_null (vala_field_get_type_reference (f), FALSE);
	gee_collection_add (GEE_COLLECTION (self->priv->fields), f);
	if (f->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE && vala_field_get_instance (f)) {
		self->priv->_has_private_fields = TRUE;
	}
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) f)), VALA_SYMBOL (f));
}


/**
 * Returns a copy of the list of fields.
 *
 * @return list of fields
 */
GeeCollection* vala_class_get_fields (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->fields)));
}


/**
 * Adds the specified method as a member to this class.
 *
 * @param m a method
 */
void vala_class_add_method (ValaClass* self, ValaMethod* m)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	if (vala_method_get_instance (m)) {
		ValaFormalParameter* __temp4;
		ValaTypeReference* __temp3;
		__temp4 = NULL;
		__temp3 = NULL;
		vala_method_set_this_parameter (m, (__temp4 = vala_formal_parameter_new ("this", (__temp3 = vala_type_reference_new ()), NULL)));
		(__temp4 == NULL ? NULL : (__temp4 = (g_object_unref (__temp4), NULL)));
		(__temp3 == NULL ? NULL : (__temp3 = (g_object_unref (__temp3), NULL)));
		vala_type_reference_set_data_type (vala_formal_parameter_get_type_reference (vala_method_get_this_parameter (m)), VALA_DATA_TYPE (self));
		vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) m)), vala_symbol_get_name (((ValaSymbol*) vala_method_get_this_parameter (m))), VALA_SYMBOL (vala_method_get_this_parameter (m)));
	}
	if (VALA_IS_CREATION_METHOD (m) && vala_symbol_get_name (((ValaSymbol*) m)) == NULL) {
		vala_class_set_default_construction_method (self, m);
	}
	gee_collection_add (GEE_COLLECTION (self->priv->methods), m);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) m)), VALA_SYMBOL (m));
}


/**
 * Returns a copy of the list of methods.
 *
 * @return list of methods
 */
GeeCollection* vala_class_get_methods (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->methods)));
}


/**
 * Adds the specified property as a member to this class.
 *
 * @param prop a property
 */
void vala_class_add_property (ValaClass* self, ValaProperty* prop, gboolean no_field)
{
	ValaFormalParameter* __temp7;
	ValaTypeReference* __temp6;
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	gee_collection_add (GEE_COLLECTION (self->priv->properties), prop);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) prop)), VALA_SYMBOL (prop));
	__temp7 = NULL;
	__temp6 = NULL;
	vala_property_set_this_parameter (prop, (__temp7 = vala_formal_parameter_new ("this", (__temp6 = vala_type_reference_new ()), NULL)));
	(__temp7 == NULL ? NULL : (__temp7 = (g_object_unref (__temp7), NULL)));
	(__temp6 == NULL ? NULL : (__temp6 = (g_object_unref (__temp6), NULL)));
	vala_type_reference_set_data_type (vala_formal_parameter_get_type_reference (vala_property_get_this_parameter (prop)), VALA_DATA_TYPE (self));
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) prop)), vala_symbol_get_name (((ValaSymbol*) vala_property_get_this_parameter (prop))), VALA_SYMBOL (vala_property_get_this_parameter (prop)));
	if (!no_field && vala_property_get_set_accessor (prop) != NULL && vala_property_accessor_get_body (vala_property_get_set_accessor (prop)) == NULL && vala_code_node_get_source_reference (((ValaCodeNode*) self)) != NULL && !vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (((ValaCodeNode*) self))))) {
		ValaTypeReference* field_type;
		char* __temp8;
		ValaField* __temp9;
		ValaField* f;
		/* automatic property accessor body generation */
		field_type = vala_type_reference_copy (vala_property_get_type_reference (prop));
		/* non_null fields not yet supported due to initialization issues*/
		vala_type_reference_set_non_null (field_type, FALSE);
		__temp8 = NULL;
		__temp9 = NULL;
		f = (__temp9 = vala_field_new ((__temp8 = g_strdup_printf ("_%s", vala_symbol_get_name (((ValaSymbol*) prop)))), field_type, NULL, vala_code_node_get_source_reference (((ValaCodeNode*) prop))), (__temp8 = (g_free (__temp8), NULL)), __temp9);
		f->access = VALA_MEMBER_ACCESSIBILITY_PRIVATE;
		vala_class_add_field (self, f);
		(field_type = (g_object_unref (field_type), NULL));
		(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
	}
}


/**
 * Returns a copy of the list of properties.
 *
 * @return list of properties
 */
GeeCollection* vala_class_get_properties (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->properties)));
}


/**
 * Adds the specified signal as a member to this class.
 *
 * @param sig a signal
 */
void vala_class_add_signal (ValaClass* self, ValaSignal* sig)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_SIGNAL (sig));
	gee_collection_add (GEE_COLLECTION (self->priv->signals), sig);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) sig)), VALA_SYMBOL (sig));
}


/**
 * Returns a copy of the list of signals.
 *
 * @return list of signals
 */
GeeCollection* vala_class_get_signals (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->signals)));
}


/**
 * Adds the specified class as an inner class.
 *
 * @param cl a class
 */
void vala_class_add_class (ValaClass* self, ValaClass* cl)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_CLASS (cl));
	gee_collection_add (GEE_COLLECTION (self->priv->classes), cl);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) cl)), VALA_SYMBOL (cl));
}


/**
 * Adds the specified struct as an inner struct.
 *
 * @param st a struct
 */
void vala_class_add_struct (ValaClass* self, ValaStruct* st)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_STRUCT (st));
	gee_collection_add (GEE_COLLECTION (self->priv->structs), st);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) st)), VALA_SYMBOL (st));
}


static void vala_class_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaClass * self;
	self = ((ValaClass*) base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_class (visitor, self);
}


static void vala_class_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaClass * self;
	self = ((ValaClass*) base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GeeList* type_collection;
		GeeIterator* type_it;
		type_collection = self->priv->base_types;
		type_it = gee_iterable_iterator (GEE_ITERABLE (type_collection));
		while (gee_iterator_next (type_it)) {
			ValaTypeReference* type;
			type = gee_iterator_get (type_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (type), visitor);
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
	}
	{
		GeeList* p_collection;
		GeeIterator* p_it;
		p_collection = self->priv->type_parameters;
		p_it = gee_iterable_iterator (GEE_ITERABLE (p_collection));
		while (gee_iterator_next (p_it)) {
			ValaTypeParameter* p;
			p = gee_iterator_get (p_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (p), visitor);
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
		(p_it == NULL ? NULL : (p_it = (g_object_unref (p_it), NULL)));
	}
	{
		GeeList* f_collection;
		GeeIterator* f_it;
		f_collection = self->priv->fields;
		f_it = gee_iterable_iterator (GEE_ITERABLE (f_collection));
		while (gee_iterator_next (f_it)) {
			ValaField* f;
			f = gee_iterator_get (f_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (f), visitor);
				(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
			}
		}
		(f_it == NULL ? NULL : (f_it = (g_object_unref (f_it), NULL)));
	}
	{
		GeeList* c_collection;
		GeeIterator* c_it;
		c_collection = self->priv->constants;
		c_it = gee_iterable_iterator (GEE_ITERABLE (c_collection));
		while (gee_iterator_next (c_it)) {
			ValaConstant* c;
			c = gee_iterator_get (c_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (c), visitor);
				(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
			}
		}
		(c_it == NULL ? NULL : (c_it = (g_object_unref (c_it), NULL)));
	}
	{
		GeeList* m_collection;
		GeeIterator* m_it;
		m_collection = self->priv->methods;
		m_it = gee_iterable_iterator (GEE_ITERABLE (m_collection));
		while (gee_iterator_next (m_it)) {
			ValaMethod* m;
			m = gee_iterator_get (m_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (m), visitor);
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
			}
		}
		(m_it == NULL ? NULL : (m_it = (g_object_unref (m_it), NULL)));
	}
	{
		GeeList* prop_collection;
		GeeIterator* prop_it;
		prop_collection = self->priv->properties;
		prop_it = gee_iterable_iterator (GEE_ITERABLE (prop_collection));
		while (gee_iterator_next (prop_it)) {
			ValaProperty* prop;
			prop = gee_iterator_get (prop_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (prop), visitor);
				(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
			}
		}
		(prop_it == NULL ? NULL : (prop_it = (g_object_unref (prop_it), NULL)));
	}
	{
		GeeList* sig_collection;
		GeeIterator* sig_it;
		sig_collection = self->priv->signals;
		sig_it = gee_iterable_iterator (GEE_ITERABLE (sig_collection));
		while (gee_iterator_next (sig_it)) {
			ValaSignal* sig;
			sig = gee_iterator_get (sig_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (sig), visitor);
				(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
			}
		}
		(sig_it == NULL ? NULL : (sig_it = (g_object_unref (sig_it), NULL)));
	}
	if (vala_class_get_constructor (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_class_get_constructor (self)), visitor);
	}
	if (vala_class_get_destructor (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_class_get_destructor (self)), visitor);
	}
	{
		GeeList* cl_collection;
		GeeIterator* cl_it;
		cl_collection = self->priv->classes;
		cl_it = gee_iterable_iterator (GEE_ITERABLE (cl_collection));
		while (gee_iterator_next (cl_it)) {
			ValaClass* cl;
			cl = gee_iterator_get (cl_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (cl), visitor);
				(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
			}
		}
		(cl_it == NULL ? NULL : (cl_it = (g_object_unref (cl_it), NULL)));
	}
	{
		GeeList* st_collection;
		GeeIterator* st_it;
		st_collection = self->priv->structs;
		st_it = gee_iterable_iterator (GEE_ITERABLE (st_collection));
		while (gee_iterator_next (st_it)) {
			ValaStruct* st;
			st = gee_iterator_get (st_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (st), visitor);
				(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
			}
		}
		(st_it == NULL ? NULL : (st_it = (g_object_unref (st_it), NULL)));
	}
}


static char* vala_class_real_get_cprefix (ValaSymbol* base)
{
	ValaClass * self;
	self = ((ValaClass*) base);
	return vala_data_type_get_cname (VALA_DATA_TYPE (self), FALSE);
}


static char* vala_class_real_get_cname (ValaDataType* base, gboolean const_type)
{
	ValaClass * self;
	const char* __temp15;
	self = ((ValaClass*) base);
	if (self->priv->cname == NULL) {
		char* __temp14;
		char* __temp13;
		__temp14 = NULL;
		__temp13 = NULL;
		self->priv->cname = (__temp14 = g_strdup_printf ("%s%s", (__temp13 = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol (((ValaSymbol*) self)))), vala_symbol_get_name (((ValaSymbol*) self))), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp14);
		(__temp13 = (g_free (__temp13), NULL));
	}
	__temp15 = NULL;
	return (__temp15 = self->priv->cname, (__temp15 == NULL ? NULL : g_strdup (__temp15)));
}


/**
 * Sets the name of this class as it is used in C code.
 *
 * @param cname the name to be used in C code
 */
void vala_class_set_cname (ValaClass* self, const char* cname)
{
	char* __temp17;
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (cname != NULL);
	__temp17 = NULL;
	self->priv->cname = (__temp17 = g_strdup (cname), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp17);
}


static char* vala_class_get_lower_case_csuffix (ValaClass* self)
{
	const char* __temp19;
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	if (self->priv->lower_case_csuffix == NULL) {
		char* __temp18;
		__temp18 = NULL;
		self->priv->lower_case_csuffix = (__temp18 = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name (((ValaSymbol*) self))), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), __temp18);
	}
	__temp19 = NULL;
	return (__temp19 = self->priv->lower_case_csuffix, (__temp19 == NULL ? NULL : g_strdup (__temp19)));
}


static char* vala_class_real_get_lower_case_cname (ValaSymbol* base, const char* infix)
{
	ValaClass * self;
	char* __temp22;
	char* __temp21;
	char* __temp23;
	self = ((ValaClass*) base);
	if (infix == NULL) {
		infix = "";
	}
	__temp22 = NULL;
	__temp21 = NULL;
	__temp23 = NULL;
	return (__temp23 = g_strdup_printf ("%s%s%s", (__temp21 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (((ValaSymbol*) self)))), infix, (__temp22 = vala_class_get_lower_case_csuffix (self))), (__temp22 = (g_free (__temp22), NULL)), (__temp21 = (g_free (__temp21), NULL)), __temp23);
}


static char* vala_class_real_get_lower_case_cprefix (ValaSymbol* base)
{
	ValaClass * self;
	char* __temp25;
	char* __temp26;
	self = ((ValaClass*) base);
	__temp25 = NULL;
	__temp26 = NULL;
	return (__temp26 = g_strdup_printf ("%s_", (__temp25 = vala_symbol_get_lower_case_cname (VALA_SYMBOL (self), NULL))), (__temp25 = (g_free (__temp25), NULL)), __temp26);
}


static char* vala_class_real_get_upper_case_cname (ValaDataType* base, const char* infix)
{
	ValaClass * self;
	char* __temp28;
	char* __temp29;
	self = ((ValaClass*) base);
	__temp28 = NULL;
	__temp29 = NULL;
	return (__temp29 = g_utf8_strup ((__temp28 = vala_symbol_get_lower_case_cname (VALA_SYMBOL (self), infix)), -1), (__temp28 = (g_free (__temp28), NULL)), __temp29);
}


static gboolean vala_class_real_is_reference_type (ValaDataType* base)
{
	ValaClass * self;
	self = ((ValaClass*) base);
	return TRUE;
}


static void vala_class_process_ccode_attribute (ValaClass* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cname")) {
		char* __temp32;
		__temp32 = NULL;
		vala_class_set_cname (self, (__temp32 = vala_attribute_get_string (a, "cname")));
		(__temp32 = (g_free (__temp32), NULL));
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			char** filename_it;
			filename_collection = g_strsplit (val, ",", 0);
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* __temp33;
				char* filename;
				__temp33 = NULL;
				filename = (__temp33 = *filename_it, (__temp33 == NULL ? NULL : g_strdup (__temp33)));
				{
					vala_data_type_add_cheader_filename (VALA_DATA_TYPE (self), filename);
					(filename = (g_free (filename), NULL));
				}
			}
			(filename_collection = (g_strfreev (filename_collection), NULL));
		}
		(val = (g_free (val), NULL));
	}
}


/**
 * Process all associated attributes.
 */
void vala_class_process_attributes (ValaClass* self)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* __temp34;
			ValaAttribute* a;
			__temp34 = NULL;
			a = (__temp34 = a_it->data, (__temp34 == NULL ? NULL : g_object_ref (__temp34)));
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "CCode") == 0) {
					vala_class_process_ccode_attribute (self, a);
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


static char* vala_class_real_get_type_id (ValaDataType* base)
{
	ValaClass * self;
	const char* __temp36;
	self = ((ValaClass*) base);
	if (self->priv->type_id == NULL) {
		char* __temp35;
		__temp35 = NULL;
		self->priv->type_id = (__temp35 = vala_data_type_get_upper_case_cname (VALA_DATA_TYPE (self), "TYPE_"), (self->priv->type_id = (g_free (self->priv->type_id), NULL)), __temp35);
	}
	__temp36 = NULL;
	return (__temp36 = self->priv->type_id, (__temp36 == NULL ? NULL : g_strdup (__temp36)));
}


static char* vala_class_real_get_marshaller_type_name (ValaDataType* base)
{
	ValaClass * self;
	self = ((ValaClass*) base);
	return g_strdup ("OBJECT");
}


static char* vala_class_real_get_get_value_function (ValaDataType* base)
{
	ValaClass * self;
	self = ((ValaClass*) base);
	return g_strdup ("g_value_get_object");
}


static char* vala_class_real_get_set_value_function (ValaDataType* base)
{
	ValaClass * self;
	self = ((ValaClass*) base);
	return g_strdup ("g_value_set_object");
}


static gboolean vala_class_real_is_reference_counting (ValaDataType* base)
{
	ValaClass * self;
	self = ((ValaClass*) base);
	return TRUE;
}


static char* vala_class_real_get_ref_function (ValaDataType* base)
{
	ValaClass * self;
	self = ((ValaClass*) base);
	return g_strdup ("g_object_ref");
}


static char* vala_class_real_get_unref_function (ValaDataType* base)
{
	ValaClass * self;
	self = ((ValaClass*) base);
	return g_strdup ("g_object_unref");
}


static gboolean vala_class_real_is_subtype_of (ValaDataType* base, ValaDataType* t)
{
	ValaClass * self;
	self = ((ValaClass*) base);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (t), FALSE);
	{
		GeeList* base_type_collection;
		GeeIterator* base_type_it;
		base_type_collection = self->priv->base_types;
		base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
		while (gee_iterator_next (base_type_it)) {
			ValaTypeReference* base_type;
			base_type = gee_iterator_get (base_type_it);
			{
				if (vala_type_reference_get_data_type (base_type) == t || vala_data_type_is_subtype_of (vala_type_reference_get_data_type (base_type), t)) {
					gboolean __temp44;
					return (__temp44 = TRUE, (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), (base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL))), __temp44);
				}
				(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
			}
		}
		(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
	}
	return FALSE;
}


static gint vala_class_real_get_type_parameter_index (ValaDataType* base, const char* name)
{
	ValaClass * self;
	gint i;
	self = ((ValaClass*) base);
	g_return_val_if_fail (name != NULL, 0);
	i = 0;
	{
		GeeList* parameter_collection;
		GeeIterator* parameter_it;
		parameter_collection = self->priv->type_parameters;
		parameter_it = gee_iterable_iterator (GEE_ITERABLE (parameter_collection));
		while (gee_iterator_next (parameter_it)) {
			ValaTypeParameter* parameter;
			parameter = gee_iterator_get (parameter_it);
			{
				if (g_utf8_collate (vala_symbol_get_name (((ValaSymbol*) parameter)), name) == 0) {
					gint __temp46;
					return (__temp46 = i, (parameter == NULL ? NULL : (parameter = (g_object_unref (parameter), NULL))), (parameter_it == NULL ? NULL : (parameter_it = (g_object_unref (parameter_it), NULL))), __temp46);
				}
				i++;
				(parameter == NULL ? NULL : (parameter = (g_object_unref (parameter), NULL)));
			}
		}
		(parameter_it == NULL ? NULL : (parameter_it = (g_object_unref (parameter_it), NULL)));
	}
	return -1;
}


ValaClass* vala_class_get_base_class (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return self->priv->_base_class;
}


void vala_class_set_base_class (ValaClass* self, ValaClass* value)
{
	ValaClass* __temp50;
	ValaClass* __temp49;
	g_return_if_fail (VALA_IS_CLASS (self));
	__temp50 = NULL;
	__temp49 = NULL;
	self->priv->_base_class = (__temp50 = (__temp49 = value, (__temp49 == NULL ? NULL : g_object_ref (__temp49))), (self->priv->_base_class == NULL ? NULL : (self->priv->_base_class = (g_object_unref (self->priv->_base_class), NULL))), __temp50);
}


gboolean vala_class_get_is_abstract (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), FALSE);
	return self->priv->_is_abstract;
}


void vala_class_set_is_abstract (ValaClass* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	self->priv->_is_abstract = value;
}


gboolean vala_class_get_is_static (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), FALSE);
	return self->priv->_is_static;
}


void vala_class_set_is_static (ValaClass* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CLASS (self));
	self->priv->_is_static = value;
}


gboolean vala_class_get_has_private_fields (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), FALSE);
	return self->priv->_has_private_fields;
}


ValaMethod* vala_class_get_default_construction_method (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return self->priv->_default_construction_method;
}


void vala_class_set_default_construction_method (ValaClass* self, ValaMethod* value)
{
	ValaMethod* __temp56;
	ValaMethod* __temp55;
	g_return_if_fail (VALA_IS_CLASS (self));
	__temp56 = NULL;
	__temp55 = NULL;
	self->priv->_default_construction_method = (__temp56 = (__temp55 = value, (__temp55 == NULL ? NULL : g_object_ref (__temp55))), (self->priv->_default_construction_method == NULL ? NULL : (self->priv->_default_construction_method = (g_object_unref (self->priv->_default_construction_method), NULL))), __temp56);
}


ValaConstructor* vala_class_get_constructor (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return self->priv->_constructor;
}


void vala_class_set_constructor (ValaClass* self, ValaConstructor* value)
{
	ValaConstructor* __temp59;
	ValaConstructor* __temp58;
	g_return_if_fail (VALA_IS_CLASS (self));
	__temp59 = NULL;
	__temp58 = NULL;
	self->priv->_constructor = (__temp59 = (__temp58 = value, (__temp58 == NULL ? NULL : g_object_ref (__temp58))), (self->priv->_constructor == NULL ? NULL : (self->priv->_constructor = (g_object_unref (self->priv->_constructor), NULL))), __temp59);
}


ValaDestructor* vala_class_get_destructor (ValaClass* self)
{
	g_return_val_if_fail (VALA_IS_CLASS (self), NULL);
	return self->priv->_destructor;
}


void vala_class_set_destructor (ValaClass* self, ValaDestructor* value)
{
	ValaDestructor* __temp62;
	ValaDestructor* __temp61;
	g_return_if_fail (VALA_IS_CLASS (self));
	__temp62 = NULL;
	__temp61 = NULL;
	self->priv->_destructor = (__temp62 = (__temp61 = value, (__temp61 == NULL ? NULL : g_object_ref (__temp61))), (self->priv->_destructor == NULL ? NULL : (self->priv->_destructor = (g_object_unref (self->priv->_destructor), NULL))), __temp62);
}


static void vala_class_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaClass * self;
	self = VALA_CLASS (object);
	switch (property_id) {
		case VALA_CLASS_BASE_CLASS:
		g_value_set_object (value, vala_class_get_base_class (self));
		break;
		case VALA_CLASS_IS_ABSTRACT:
		g_value_set_boolean (value, vala_class_get_is_abstract (self));
		break;
		case VALA_CLASS_IS_STATIC:
		g_value_set_boolean (value, vala_class_get_is_static (self));
		break;
		case VALA_CLASS_HAS_PRIVATE_FIELDS:
		g_value_set_boolean (value, vala_class_get_has_private_fields (self));
		break;
		case VALA_CLASS_DEFAULT_CONSTRUCTION_METHOD:
		g_value_set_object (value, vala_class_get_default_construction_method (self));
		break;
		case VALA_CLASS_CONSTRUCTOR:
		g_value_set_object (value, vala_class_get_constructor (self));
		break;
		case VALA_CLASS_DESTRUCTOR:
		g_value_set_object (value, vala_class_get_destructor (self));
		break;
	}
}


static void vala_class_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaClass * self;
	self = VALA_CLASS (object);
	switch (property_id) {
		case VALA_CLASS_BASE_CLASS:
		vala_class_set_base_class (self, g_value_get_object (value));
		break;
		case VALA_CLASS_IS_ABSTRACT:
		vala_class_set_is_abstract (self, g_value_get_boolean (value));
		break;
		case VALA_CLASS_IS_STATIC:
		vala_class_set_is_static (self, g_value_get_boolean (value));
		break;
		case VALA_CLASS_DEFAULT_CONSTRUCTION_METHOD:
		vala_class_set_default_construction_method (self, g_value_get_object (value));
		break;
		case VALA_CLASS_CONSTRUCTOR:
		vala_class_set_constructor (self, g_value_get_object (value));
		break;
		case VALA_CLASS_DESTRUCTOR:
		vala_class_set_destructor (self, g_value_get_object (value));
		break;
	}
}


static void vala_class_class_init (ValaClassClass * klass)
{
	vala_class_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaClassPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_class_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_class_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_class_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_class_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_class_real_accept_children;
	VALA_SYMBOL_CLASS (klass)->get_cprefix = vala_class_real_get_cprefix;
	VALA_DATA_TYPE_CLASS (klass)->get_cname = vala_class_real_get_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cname = vala_class_real_get_lower_case_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_class_real_get_lower_case_cprefix;
	VALA_DATA_TYPE_CLASS (klass)->get_upper_case_cname = vala_class_real_get_upper_case_cname;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_type = vala_class_real_is_reference_type;
	VALA_DATA_TYPE_CLASS (klass)->get_type_id = vala_class_real_get_type_id;
	VALA_DATA_TYPE_CLASS (klass)->get_marshaller_type_name = vala_class_real_get_marshaller_type_name;
	VALA_DATA_TYPE_CLASS (klass)->get_get_value_function = vala_class_real_get_get_value_function;
	VALA_DATA_TYPE_CLASS (klass)->get_set_value_function = vala_class_real_get_set_value_function;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_counting = vala_class_real_is_reference_counting;
	VALA_DATA_TYPE_CLASS (klass)->get_ref_function = vala_class_real_get_ref_function;
	VALA_DATA_TYPE_CLASS (klass)->get_unref_function = vala_class_real_get_unref_function;
	VALA_DATA_TYPE_CLASS (klass)->is_subtype_of = vala_class_real_is_subtype_of;
	VALA_DATA_TYPE_CLASS (klass)->get_type_parameter_index = vala_class_real_get_type_parameter_index;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_BASE_CLASS, g_param_spec_object ("base-class", "foo", "bar", VALA_TYPE_CLASS, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_IS_ABSTRACT, g_param_spec_boolean ("is-abstract", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_IS_STATIC, g_param_spec_boolean ("is-static", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_HAS_PRIVATE_FIELDS, g_param_spec_boolean ("has-private-fields", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_DEFAULT_CONSTRUCTION_METHOD, g_param_spec_object ("default-construction-method", "foo", "bar", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_CONSTRUCTOR, g_param_spec_object ("constructor", "foo", "bar", VALA_TYPE_CONSTRUCTOR, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CLASS_DESTRUCTOR, g_param_spec_object ("destructor", "foo", "bar", VALA_TYPE_DESTRUCTOR, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_class_init (ValaClass * self)
{
	self->priv = VALA_CLASS_GET_PRIVATE (self);
	self->priv->type_parameters = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->base_types = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->constants = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->fields = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->methods = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->properties = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->signals = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->classes = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->structs = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
}


static void vala_class_dispose (GObject * obj)
{
	ValaClass * self;
	ValaClassClass * klass;
	GObjectClass * parent_class;
	self = VALA_CLASS (obj);
	(self->priv->_base_class == NULL ? NULL : (self->priv->_base_class = (g_object_unref (self->priv->_base_class), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	(self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL));
	(self->priv->type_id = (g_free (self->priv->type_id), NULL));
	(self->priv->type_parameters == NULL ? NULL : (self->priv->type_parameters = (g_object_unref (self->priv->type_parameters), NULL)));
	(self->priv->base_types == NULL ? NULL : (self->priv->base_types = (g_object_unref (self->priv->base_types), NULL)));
	(self->priv->constants == NULL ? NULL : (self->priv->constants = (g_object_unref (self->priv->constants), NULL)));
	(self->priv->fields == NULL ? NULL : (self->priv->fields = (g_object_unref (self->priv->fields), NULL)));
	(self->priv->methods == NULL ? NULL : (self->priv->methods = (g_object_unref (self->priv->methods), NULL)));
	(self->priv->properties == NULL ? NULL : (self->priv->properties = (g_object_unref (self->priv->properties), NULL)));
	(self->priv->signals == NULL ? NULL : (self->priv->signals = (g_object_unref (self->priv->signals), NULL)));
	(self->priv->classes == NULL ? NULL : (self->priv->classes = (g_object_unref (self->priv->classes), NULL)));
	(self->priv->structs == NULL ? NULL : (self->priv->structs = (g_object_unref (self->priv->structs), NULL)));
	(self->priv->_default_construction_method == NULL ? NULL : (self->priv->_default_construction_method = (g_object_unref (self->priv->_default_construction_method), NULL)));
	(self->priv->_constructor == NULL ? NULL : (self->priv->_constructor = (g_object_unref (self->priv->_constructor), NULL)));
	(self->priv->_destructor == NULL ? NULL : (self->priv->_destructor = (g_object_unref (self->priv->_destructor), NULL)));
	klass = VALA_CLASS_CLASS (g_type_class_peek (VALA_TYPE_CLASS));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_class_get_type (void)
{
	static GType vala_class_type_id = 0;
	if (G_UNLIKELY (vala_class_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaClassClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_class_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaClass), 0, (GInstanceInitFunc) vala_class_init };
		vala_class_type_id = g_type_register_static (VALA_TYPE_DATA_TYPE, "ValaClass", &g_define_type_info, 0);
	}
	return vala_class_type_id;
}




