/* valacodevisitor.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */

#include "valacodevisitor.h"
#include <vala/valasourcefile.h>
#include <vala/valanamespace.h>
#include <vala/valaclass.h>
#include <vala/valastruct.h>
#include <vala/valainterface.h>
#include <vala/valaenum.h>
#include <vala/valaenumvalue.h>
#include <vala/valacallback.h>
#include <vala/valamember.h>
#include <vala/valaconstant.h>
#include <vala/valafield.h>
#include <vala/valamethod.h>
#include <vala/valacreationmethod.h>
#include <vala/valaformalparameter.h>
#include <vala/valaproperty.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valasignal.h>
#include <vala/valaconstructor.h>
#include <vala/valadestructor.h>
#include <vala/valanamedargument.h>
#include <vala/valatypeparameter.h>
#include <vala/valanamespacereference.h>
#include <vala/valatypereference.h>
#include <vala/valablock.h>
#include <vala/valaemptystatement.h>
#include <vala/valadeclarationstatement.h>
#include <vala/valalocalvariabledeclaration.h>
#include <vala/valavariabledeclarator.h>
#include <vala/valainitializerlist.h>
#include <vala/valaexpressionstatement.h>
#include <vala/valaifstatement.h>
#include <vala/valaswitchstatement.h>
#include <vala/valaswitchsection.h>
#include <vala/valaswitchlabel.h>
#include <vala/valawhilestatement.h>
#include <vala/valadostatement.h>
#include <vala/valaforstatement.h>
#include <vala/valaforeachstatement.h>
#include <vala/valabreakstatement.h>
#include <vala/valacontinuestatement.h>
#include <vala/valareturnstatement.h>
#include <vala/valathrowstatement.h>
#include <vala/valatrystatement.h>
#include <vala/valacatchclause.h>
#include <vala/valalockstatement.h>
#include <vala/valaarraycreationexpression.h>
#include <vala/valabooleanliteral.h>
#include <vala/valacharacterliteral.h>
#include <vala/valaintegerliteral.h>
#include <vala/valarealliteral.h>
#include <vala/valastringliteral.h>
#include <vala/valanullliteral.h>
#include <vala/valaliteralexpression.h>
#include <vala/valaparenthesizedexpression.h>
#include <vala/valamemberaccess.h>
#include <vala/valainvocationexpression.h>
#include <vala/valaelementaccess.h>
#include <vala/valabaseaccess.h>
#include <vala/valapostfixexpression.h>
#include <vala/valaobjectcreationexpression.h>
#include <vala/valasizeofexpression.h>
#include <vala/valatypeofexpression.h>
#include <vala/valaunaryexpression.h>
#include <vala/valacastexpression.h>
#include <vala/valapointerindirection.h>
#include <vala/valaaddressofexpression.h>
#include <vala/valareferencetransferexpression.h>
#include <vala/valabinaryexpression.h>
#include <vala/valatypecheck.h>
#include <vala/valaconditionalexpression.h>
#include <vala/valalambdaexpression.h>
#include <vala/valaassignment.h>
#include <vala/valaexpression.h>

enum  {
	VALA_CODE_VISITOR_DUMMY_PROPERTY
};
static void vala_code_visitor_real_visit_source_file (ValaCodeVisitor* self, ValaSourceFile* source_file);
static void vala_code_visitor_real_visit_namespace (ValaCodeVisitor* self, ValaNamespace* ns);
static void vala_code_visitor_real_visit_class (ValaCodeVisitor* self, ValaClass* cl);
static void vala_code_visitor_real_visit_struct (ValaCodeVisitor* self, ValaStruct* st);
static void vala_code_visitor_real_visit_interface (ValaCodeVisitor* self, ValaInterface* iface);
static void vala_code_visitor_real_visit_enum (ValaCodeVisitor* self, ValaEnum* en);
static void vala_code_visitor_real_visit_enum_value (ValaCodeVisitor* self, ValaEnumValue* ev);
static void vala_code_visitor_real_visit_callback (ValaCodeVisitor* self, ValaCallback* cb);
static void vala_code_visitor_real_visit_member (ValaCodeVisitor* self, ValaMember* m);
static void vala_code_visitor_real_visit_constant (ValaCodeVisitor* self, ValaConstant* c);
static void vala_code_visitor_real_visit_field (ValaCodeVisitor* self, ValaField* f);
static void vala_code_visitor_real_visit_method (ValaCodeVisitor* self, ValaMethod* m);
static void vala_code_visitor_real_visit_creation_method (ValaCodeVisitor* self, ValaCreationMethod* m);
static void vala_code_visitor_real_visit_formal_parameter (ValaCodeVisitor* self, ValaFormalParameter* p);
static void vala_code_visitor_real_visit_property (ValaCodeVisitor* self, ValaProperty* prop);
static void vala_code_visitor_real_visit_property_accessor (ValaCodeVisitor* self, ValaPropertyAccessor* acc);
static void vala_code_visitor_real_visit_signal (ValaCodeVisitor* self, ValaSignal* sig);
static void vala_code_visitor_real_visit_constructor (ValaCodeVisitor* self, ValaConstructor* c);
static void vala_code_visitor_real_visit_destructor (ValaCodeVisitor* self, ValaDestructor* d);
static void vala_code_visitor_real_visit_named_argument (ValaCodeVisitor* self, ValaNamedArgument* n);
static void vala_code_visitor_real_visit_type_parameter (ValaCodeVisitor* self, ValaTypeParameter* p);
static void vala_code_visitor_real_visit_namespace_reference (ValaCodeVisitor* self, ValaNamespaceReference* ns);
static void vala_code_visitor_real_visit_type_reference (ValaCodeVisitor* self, ValaTypeReference* type);
static void vala_code_visitor_real_visit_begin_block (ValaCodeVisitor* self, ValaBlock* b);
static void vala_code_visitor_real_visit_end_block (ValaCodeVisitor* self, ValaBlock* b);
static void vala_code_visitor_real_visit_empty_statement (ValaCodeVisitor* self, ValaEmptyStatement* stmt);
static void vala_code_visitor_real_visit_declaration_statement (ValaCodeVisitor* self, ValaDeclarationStatement* stmt);
static void vala_code_visitor_real_visit_local_variable_declaration (ValaCodeVisitor* self, ValaLocalVariableDeclaration* decl);
static void vala_code_visitor_real_visit_variable_declarator (ValaCodeVisitor* self, ValaVariableDeclarator* decl);
static void vala_code_visitor_real_visit_begin_initializer_list (ValaCodeVisitor* self, ValaInitializerList* list);
static void vala_code_visitor_real_visit_end_initializer_list (ValaCodeVisitor* self, ValaInitializerList* list);
static void vala_code_visitor_real_visit_expression_statement (ValaCodeVisitor* self, ValaExpressionStatement* stmt);
static void vala_code_visitor_real_visit_if_statement (ValaCodeVisitor* self, ValaIfStatement* stmt);
static void vala_code_visitor_real_visit_switch_statement (ValaCodeVisitor* self, ValaSwitchStatement* stmt);
static void vala_code_visitor_real_visit_switch_section (ValaCodeVisitor* self, ValaSwitchSection* section);
static void vala_code_visitor_real_visit_switch_label (ValaCodeVisitor* self, ValaSwitchLabel* label);
static void vala_code_visitor_real_visit_while_statement (ValaCodeVisitor* self, ValaWhileStatement* stmt);
static void vala_code_visitor_real_visit_do_statement (ValaCodeVisitor* self, ValaDoStatement* stmt);
static void vala_code_visitor_real_visit_for_statement (ValaCodeVisitor* self, ValaForStatement* stmt);
static void vala_code_visitor_real_visit_begin_foreach_statement (ValaCodeVisitor* self, ValaForeachStatement* stmt);
static void vala_code_visitor_real_visit_end_foreach_statement (ValaCodeVisitor* self, ValaForeachStatement* stmt);
static void vala_code_visitor_real_visit_break_statement (ValaCodeVisitor* self, ValaBreakStatement* stmt);
static void vala_code_visitor_real_visit_continue_statement (ValaCodeVisitor* self, ValaContinueStatement* stmt);
static void vala_code_visitor_real_visit_begin_return_statement (ValaCodeVisitor* self, ValaReturnStatement* stmt);
static void vala_code_visitor_real_visit_end_return_statement (ValaCodeVisitor* self, ValaReturnStatement* stmt);
static void vala_code_visitor_real_visit_throw_statement (ValaCodeVisitor* self, ValaThrowStatement* stmt);
static void vala_code_visitor_real_visit_try_statement (ValaCodeVisitor* self, ValaTryStatement* stmt);
static void vala_code_visitor_real_visit_catch_clause (ValaCodeVisitor* self, ValaCatchClause* clause);
static void vala_code_visitor_real_visit_lock_statement (ValaCodeVisitor* self, ValaLockStatement* stmt);
static void vala_code_visitor_real_visit_begin_array_creation_expression (ValaCodeVisitor* self, ValaArrayCreationExpression* expr);
static void vala_code_visitor_real_visit_end_array_creation_expression (ValaCodeVisitor* self, ValaArrayCreationExpression* expr);
static void vala_code_visitor_real_visit_boolean_literal (ValaCodeVisitor* self, ValaBooleanLiteral* lit);
static void vala_code_visitor_real_visit_character_literal (ValaCodeVisitor* self, ValaCharacterLiteral* lit);
static void vala_code_visitor_real_visit_integer_literal (ValaCodeVisitor* self, ValaIntegerLiteral* lit);
static void vala_code_visitor_real_visit_real_literal (ValaCodeVisitor* self, ValaRealLiteral* lit);
static void vala_code_visitor_real_visit_string_literal (ValaCodeVisitor* self, ValaStringLiteral* lit);
static void vala_code_visitor_real_visit_null_literal (ValaCodeVisitor* self, ValaNullLiteral* lit);
static void vala_code_visitor_real_visit_literal_expression (ValaCodeVisitor* self, ValaLiteralExpression* expr);
static void vala_code_visitor_real_visit_parenthesized_expression (ValaCodeVisitor* self, ValaParenthesizedExpression* expr);
static void vala_code_visitor_real_visit_member_access (ValaCodeVisitor* self, ValaMemberAccess* expr);
static void vala_code_visitor_real_visit_begin_invocation_expression (ValaCodeVisitor* self, ValaInvocationExpression* expr);
static void vala_code_visitor_real_visit_end_invocation_expression (ValaCodeVisitor* self, ValaInvocationExpression* expr);
static void vala_code_visitor_real_visit_element_access (ValaCodeVisitor* self, ValaElementAccess* expr);
static void vala_code_visitor_real_visit_base_access (ValaCodeVisitor* self, ValaBaseAccess* expr);
static void vala_code_visitor_real_visit_postfix_expression (ValaCodeVisitor* self, ValaPostfixExpression* expr);
static void vala_code_visitor_real_visit_begin_object_creation_expression (ValaCodeVisitor* self, ValaObjectCreationExpression* expr);
static void vala_code_visitor_real_visit_end_object_creation_expression (ValaCodeVisitor* self, ValaObjectCreationExpression* expr);
static void vala_code_visitor_real_visit_sizeof_expression (ValaCodeVisitor* self, ValaSizeofExpression* expr);
static void vala_code_visitor_real_visit_typeof_expression (ValaCodeVisitor* self, ValaTypeofExpression* expr);
static void vala_code_visitor_real_visit_unary_expression (ValaCodeVisitor* self, ValaUnaryExpression* expr);
static void vala_code_visitor_real_visit_cast_expression (ValaCodeVisitor* self, ValaCastExpression* expr);
static void vala_code_visitor_real_visit_pointer_indirection (ValaCodeVisitor* self, ValaPointerIndirection* expr);
static void vala_code_visitor_real_visit_addressof_expression (ValaCodeVisitor* self, ValaAddressofExpression* expr);
static void vala_code_visitor_real_visit_reference_transfer_expression (ValaCodeVisitor* self, ValaReferenceTransferExpression* expr);
static void vala_code_visitor_real_visit_binary_expression (ValaCodeVisitor* self, ValaBinaryExpression* expr);
static void vala_code_visitor_real_visit_type_check (ValaCodeVisitor* self, ValaTypeCheck* expr);
static void vala_code_visitor_real_visit_conditional_expression (ValaCodeVisitor* self, ValaConditionalExpression* expr);
static void vala_code_visitor_real_visit_begin_lambda_expression (ValaCodeVisitor* self, ValaLambdaExpression* expr);
static void vala_code_visitor_real_visit_end_lambda_expression (ValaCodeVisitor* self, ValaLambdaExpression* expr);
static void vala_code_visitor_real_visit_begin_assignment (ValaCodeVisitor* self, ValaAssignment* a);
static void vala_code_visitor_real_visit_end_assignment (ValaCodeVisitor* self, ValaAssignment* a);
static void vala_code_visitor_real_visit_end_full_expression (ValaCodeVisitor* self, ValaExpression* expr);
static gpointer vala_code_visitor_parent_class = NULL;


/**
 * Visit operation called for source files.
 *
 * @param source_file a source file
 */
static void vala_code_visitor_real_visit_source_file (ValaCodeVisitor* self, ValaSourceFile* source_file)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_SOURCE_FILE (source_file));
}


void vala_code_visitor_visit_source_file (ValaCodeVisitor* self, ValaSourceFile* source_file)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_source_file (self, source_file);
}


/**
 * Visit operation called for namespaces.
 *
 * @param ns a namespace
 */
static void vala_code_visitor_real_visit_namespace (ValaCodeVisitor* self, ValaNamespace* ns)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
}


void vala_code_visitor_visit_namespace (ValaCodeVisitor* self, ValaNamespace* ns)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_namespace (self, ns);
}


/**
 * Visit operation called for classes.
 *
 * @param cl a class
 */
static void vala_code_visitor_real_visit_class (ValaCodeVisitor* self, ValaClass* cl)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_CLASS (cl));
}


void vala_code_visitor_visit_class (ValaCodeVisitor* self, ValaClass* cl)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_class (self, cl);
}


/**
 * Visit operation called for structs.
 *
 * @param st a struct
 */
static void vala_code_visitor_real_visit_struct (ValaCodeVisitor* self, ValaStruct* st)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_STRUCT (st));
}


void vala_code_visitor_visit_struct (ValaCodeVisitor* self, ValaStruct* st)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_struct (self, st);
}


/**
 * Visit operation called for interfaces.
 *
 * @param iface an interface
 */
static void vala_code_visitor_real_visit_interface (ValaCodeVisitor* self, ValaInterface* iface)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_INTERFACE (iface));
}


void vala_code_visitor_visit_interface (ValaCodeVisitor* self, ValaInterface* iface)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_interface (self, iface);
}


/**
 * Visit operation called for enums.
 *
 * @param en an enum
 */
static void vala_code_visitor_real_visit_enum (ValaCodeVisitor* self, ValaEnum* en)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_ENUM (en));
}


void vala_code_visitor_visit_enum (ValaCodeVisitor* self, ValaEnum* en)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_enum (self, en);
}


/**
 * Visit operation called for enum values.
 *
 * @param ev an enum value
 */
static void vala_code_visitor_real_visit_enum_value (ValaCodeVisitor* self, ValaEnumValue* ev)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_ENUM_VALUE (ev));
}


void vala_code_visitor_visit_enum_value (ValaCodeVisitor* self, ValaEnumValue* ev)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_enum_value (self, ev);
}


/**
 * Visit operation called for callbacks.
 *
 * @param cb a callback
 */
static void vala_code_visitor_real_visit_callback (ValaCodeVisitor* self, ValaCallback* cb)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_CALLBACK (cb));
}


void vala_code_visitor_visit_callback (ValaCodeVisitor* self, ValaCallback* cb)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_callback (self, cb);
}


/**
 * Visit operation called for members.
 *
 * @param m a member
 */
static void vala_code_visitor_real_visit_member (ValaCodeVisitor* self, ValaMember* m)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_MEMBER (m));
}


void vala_code_visitor_visit_member (ValaCodeVisitor* self, ValaMember* m)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_member (self, m);
}


/**
 * Visit operation called for constants.
 *
 * @param c a constant
 */
static void vala_code_visitor_real_visit_constant (ValaCodeVisitor* self, ValaConstant* c)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_CONSTANT (c));
}


void vala_code_visitor_visit_constant (ValaCodeVisitor* self, ValaConstant* c)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_constant (self, c);
}


/**
 * Visit operation called for fields.
 *
 * @param f a field
 */
static void vala_code_visitor_real_visit_field (ValaCodeVisitor* self, ValaField* f)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_FIELD (f));
}


void vala_code_visitor_visit_field (ValaCodeVisitor* self, ValaField* f)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_field (self, f);
}


/**
 * Visit operation called for methods.
 *
 * @param m a method
 */
static void vala_code_visitor_real_visit_method (ValaCodeVisitor* self, ValaMethod* m)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_METHOD (m));
}


void vala_code_visitor_visit_method (ValaCodeVisitor* self, ValaMethod* m)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_method (self, m);
}


/**
 * Visit operation called for creation methods.
 *
 * @param m a method
 */
static void vala_code_visitor_real_visit_creation_method (ValaCodeVisitor* self, ValaCreationMethod* m)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_CREATION_METHOD (m));
}


void vala_code_visitor_visit_creation_method (ValaCodeVisitor* self, ValaCreationMethod* m)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_creation_method (self, m);
}


/**
 * Visit operation called for formal parameters.
 *
 * @param p a formal parameter
 */
static void vala_code_visitor_real_visit_formal_parameter (ValaCodeVisitor* self, ValaFormalParameter* p)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (p));
}


void vala_code_visitor_visit_formal_parameter (ValaCodeVisitor* self, ValaFormalParameter* p)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_formal_parameter (self, p);
}


/**
 * Visit operation called for properties.
 *
 * @param prop a property
 */
static void vala_code_visitor_real_visit_property (ValaCodeVisitor* self, ValaProperty* prop)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_PROPERTY (prop));
}


void vala_code_visitor_visit_property (ValaCodeVisitor* self, ValaProperty* prop)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_property (self, prop);
}


/**
 * Visit operation called for property accessors.
 *
 * @param acc a property accessor
 */
static void vala_code_visitor_real_visit_property_accessor (ValaCodeVisitor* self, ValaPropertyAccessor* acc)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (acc));
}


void vala_code_visitor_visit_property_accessor (ValaCodeVisitor* self, ValaPropertyAccessor* acc)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_property_accessor (self, acc);
}


/**
 * Visit operation called for signals.
 *
 * @param sig a signal
 */
static void vala_code_visitor_real_visit_signal (ValaCodeVisitor* self, ValaSignal* sig)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_SIGNAL (sig));
}


void vala_code_visitor_visit_signal (ValaCodeVisitor* self, ValaSignal* sig)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_signal (self, sig);
}


/**
 * Visit operation called for constructors.
 *
 * @param c a constructor
 */
static void vala_code_visitor_real_visit_constructor (ValaCodeVisitor* self, ValaConstructor* c)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_CONSTRUCTOR (c));
}


void vala_code_visitor_visit_constructor (ValaCodeVisitor* self, ValaConstructor* c)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_constructor (self, c);
}


/**
 * Visit operation called for destructors.
 *
 * @param d a destructor
 */
static void vala_code_visitor_real_visit_destructor (ValaCodeVisitor* self, ValaDestructor* d)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_DESTRUCTOR (d));
}


void vala_code_visitor_visit_destructor (ValaCodeVisitor* self, ValaDestructor* d)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_destructor (self, d);
}


/**
 * Visit operation called for named arguments.
 *
 * @param n a named argument
 */
static void vala_code_visitor_real_visit_named_argument (ValaCodeVisitor* self, ValaNamedArgument* n)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_NAMED_ARGUMENT (n));
}


void vala_code_visitor_visit_named_argument (ValaCodeVisitor* self, ValaNamedArgument* n)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_named_argument (self, n);
}


/**
 * Visit operation called for type parameters.
 *
 * @param p a type parameter
 */
static void vala_code_visitor_real_visit_type_parameter (ValaCodeVisitor* self, ValaTypeParameter* p)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_TYPE_PARAMETER (p));
}


void vala_code_visitor_visit_type_parameter (ValaCodeVisitor* self, ValaTypeParameter* p)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_type_parameter (self, p);
}


/**
 * Visit operation called for namespace references.
 *
 * @param ns a namespace reference
 */
static void vala_code_visitor_real_visit_namespace_reference (ValaCodeVisitor* self, ValaNamespaceReference* ns)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_NAMESPACE_REFERENCE (ns));
}


void vala_code_visitor_visit_namespace_reference (ValaCodeVisitor* self, ValaNamespaceReference* ns)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_namespace_reference (self, ns);
}


/**
 * Visit operation called for type references.
 *
 * @param type a type reference
 */
static void vala_code_visitor_real_visit_type_reference (ValaCodeVisitor* self, ValaTypeReference* type)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (type));
}


void vala_code_visitor_visit_type_reference (ValaCodeVisitor* self, ValaTypeReference* type)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_type_reference (self, type);
}


/**
 * Visit operation called at beginning of blocks.
 *
 * @param b a block
 */
static void vala_code_visitor_real_visit_begin_block (ValaCodeVisitor* self, ValaBlock* b)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_BLOCK (b));
}


void vala_code_visitor_visit_begin_block (ValaCodeVisitor* self, ValaBlock* b)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_begin_block (self, b);
}


/**
 * Visit operation called at end of blocks.
 *
 * @param b a block
 */
static void vala_code_visitor_real_visit_end_block (ValaCodeVisitor* self, ValaBlock* b)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_BLOCK (b));
}


void vala_code_visitor_visit_end_block (ValaCodeVisitor* self, ValaBlock* b)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_end_block (self, b);
}


/**
 * Visit operation called for empty statements.
 *
 * @param stmt an empty statement
 */
static void vala_code_visitor_real_visit_empty_statement (ValaCodeVisitor* self, ValaEmptyStatement* stmt)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_EMPTY_STATEMENT (stmt));
}


void vala_code_visitor_visit_empty_statement (ValaCodeVisitor* self, ValaEmptyStatement* stmt)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_empty_statement (self, stmt);
}


/**
 * Visit operation called for declaration statements.
 *
 * @param stmt a declaration statement
 */
static void vala_code_visitor_real_visit_declaration_statement (ValaCodeVisitor* self, ValaDeclarationStatement* stmt)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_DECLARATION_STATEMENT (stmt));
}


void vala_code_visitor_visit_declaration_statement (ValaCodeVisitor* self, ValaDeclarationStatement* stmt)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_declaration_statement (self, stmt);
}


/**
 * Visit operation called for local variable declarations.
 *
 * @param decl a local variable declaration
 */
static void vala_code_visitor_real_visit_local_variable_declaration (ValaCodeVisitor* self, ValaLocalVariableDeclaration* decl)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_LOCAL_VARIABLE_DECLARATION (decl));
}


void vala_code_visitor_visit_local_variable_declaration (ValaCodeVisitor* self, ValaLocalVariableDeclaration* decl)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_local_variable_declaration (self, decl);
}


/**
 * Visit operation called for variable declarators.
 *
 * @param decl a variable declarator
 */
static void vala_code_visitor_real_visit_variable_declarator (ValaCodeVisitor* self, ValaVariableDeclarator* decl)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_VARIABLE_DECLARATOR (decl));
}


void vala_code_visitor_visit_variable_declarator (ValaCodeVisitor* self, ValaVariableDeclarator* decl)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_variable_declarator (self, decl);
}


/**
 * Visit operation called for initializer lists
 *
 * @param list an initializer list
 */
static void vala_code_visitor_real_visit_begin_initializer_list (ValaCodeVisitor* self, ValaInitializerList* list)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_INITIALIZER_LIST (list));
}


void vala_code_visitor_visit_begin_initializer_list (ValaCodeVisitor* self, ValaInitializerList* list)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_begin_initializer_list (self, list);
}


/**
 * Visit operation called for initializer lists
 *
 * @param list an initializer list
 */
static void vala_code_visitor_real_visit_end_initializer_list (ValaCodeVisitor* self, ValaInitializerList* list)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_INITIALIZER_LIST (list));
}


void vala_code_visitor_visit_end_initializer_list (ValaCodeVisitor* self, ValaInitializerList* list)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_end_initializer_list (self, list);
}


/**
 * Visit operation called for expression statements.
 *
 * @param stmt an expression statement
 */
static void vala_code_visitor_real_visit_expression_statement (ValaCodeVisitor* self, ValaExpressionStatement* stmt)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_EXPRESSION_STATEMENT (stmt));
}


void vala_code_visitor_visit_expression_statement (ValaCodeVisitor* self, ValaExpressionStatement* stmt)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_expression_statement (self, stmt);
}


/**
 * Visit operation called for if statements.
 *
 * @param stmt an if statement
 */
static void vala_code_visitor_real_visit_if_statement (ValaCodeVisitor* self, ValaIfStatement* stmt)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_IF_STATEMENT (stmt));
}


void vala_code_visitor_visit_if_statement (ValaCodeVisitor* self, ValaIfStatement* stmt)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_if_statement (self, stmt);
}


/**
 * Visit operation called for switch statements.
 *
 * @param stmt a switch statement
 */
static void vala_code_visitor_real_visit_switch_statement (ValaCodeVisitor* self, ValaSwitchStatement* stmt)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_SWITCH_STATEMENT (stmt));
}


void vala_code_visitor_visit_switch_statement (ValaCodeVisitor* self, ValaSwitchStatement* stmt)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_switch_statement (self, stmt);
}


/**
 * Visit operation called for switch sections.
 *
 * @param section a switch section
 */
static void vala_code_visitor_real_visit_switch_section (ValaCodeVisitor* self, ValaSwitchSection* section)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_SWITCH_SECTION (section));
}


void vala_code_visitor_visit_switch_section (ValaCodeVisitor* self, ValaSwitchSection* section)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_switch_section (self, section);
}


/**
 * Visit operation called for switch label.
 *
 * @param label a switch label
 */
static void vala_code_visitor_real_visit_switch_label (ValaCodeVisitor* self, ValaSwitchLabel* label)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_SWITCH_LABEL (label));
}


void vala_code_visitor_visit_switch_label (ValaCodeVisitor* self, ValaSwitchLabel* label)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_switch_label (self, label);
}


/**
 * Visit operation called for while statements.
 *
 * @param stmt an while statement
 */
static void vala_code_visitor_real_visit_while_statement (ValaCodeVisitor* self, ValaWhileStatement* stmt)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_WHILE_STATEMENT (stmt));
}


void vala_code_visitor_visit_while_statement (ValaCodeVisitor* self, ValaWhileStatement* stmt)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_while_statement (self, stmt);
}


/**
 * Visit operation called for do statements.
 *
 * @param stmt a do statement
 */
static void vala_code_visitor_real_visit_do_statement (ValaCodeVisitor* self, ValaDoStatement* stmt)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_DO_STATEMENT (stmt));
}


void vala_code_visitor_visit_do_statement (ValaCodeVisitor* self, ValaDoStatement* stmt)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_do_statement (self, stmt);
}


/**
 * Visit operation called for for statements.
 *
 * @param stmt a for statement
 */
static void vala_code_visitor_real_visit_for_statement (ValaCodeVisitor* self, ValaForStatement* stmt)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_FOR_STATEMENT (stmt));
}


void vala_code_visitor_visit_for_statement (ValaCodeVisitor* self, ValaForStatement* stmt)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_for_statement (self, stmt);
}


/**
 * Visit operation called at beginning of foreach statements.
 *
 * @param stmt a foreach statement
 */
static void vala_code_visitor_real_visit_begin_foreach_statement (ValaCodeVisitor* self, ValaForeachStatement* stmt)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_FOREACH_STATEMENT (stmt));
}


void vala_code_visitor_visit_begin_foreach_statement (ValaCodeVisitor* self, ValaForeachStatement* stmt)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_begin_foreach_statement (self, stmt);
}


/**
 * Visit operation called at end of foreach statements.
 *
 * @param stmt a foreach statement
 */
static void vala_code_visitor_real_visit_end_foreach_statement (ValaCodeVisitor* self, ValaForeachStatement* stmt)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_FOREACH_STATEMENT (stmt));
}


void vala_code_visitor_visit_end_foreach_statement (ValaCodeVisitor* self, ValaForeachStatement* stmt)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_end_foreach_statement (self, stmt);
}


/**
 * Visit operation called for break statements.
 *
 * @param stmt a break statement
 */
static void vala_code_visitor_real_visit_break_statement (ValaCodeVisitor* self, ValaBreakStatement* stmt)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_BREAK_STATEMENT (stmt));
}


void vala_code_visitor_visit_break_statement (ValaCodeVisitor* self, ValaBreakStatement* stmt)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_break_statement (self, stmt);
}


/**
 * Visit operation called for continue statements.
 *
 * @param stmt a continue statement
 */
static void vala_code_visitor_real_visit_continue_statement (ValaCodeVisitor* self, ValaContinueStatement* stmt)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_CONTINUE_STATEMENT (stmt));
}


void vala_code_visitor_visit_continue_statement (ValaCodeVisitor* self, ValaContinueStatement* stmt)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_continue_statement (self, stmt);
}


/**
 * Visit operation called at beginning of return statements.
 *
 * @param stmt a return statement
 */
static void vala_code_visitor_real_visit_begin_return_statement (ValaCodeVisitor* self, ValaReturnStatement* stmt)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_RETURN_STATEMENT (stmt));
}


void vala_code_visitor_visit_begin_return_statement (ValaCodeVisitor* self, ValaReturnStatement* stmt)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_begin_return_statement (self, stmt);
}


/**
 * Visit operation called at end of return statements.
 *
 * @param stmt a return statement
 */
static void vala_code_visitor_real_visit_end_return_statement (ValaCodeVisitor* self, ValaReturnStatement* stmt)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_RETURN_STATEMENT (stmt));
}


void vala_code_visitor_visit_end_return_statement (ValaCodeVisitor* self, ValaReturnStatement* stmt)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_end_return_statement (self, stmt);
}


/**
 * Visit operation called for throw statements.
 *
 * @param stmt a throw statement
 */
static void vala_code_visitor_real_visit_throw_statement (ValaCodeVisitor* self, ValaThrowStatement* stmt)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_THROW_STATEMENT (stmt));
}


void vala_code_visitor_visit_throw_statement (ValaCodeVisitor* self, ValaThrowStatement* stmt)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_throw_statement (self, stmt);
}


/**
 * Visit operation called for try statements.
 *
 * @param stmt a try statement
 */
static void vala_code_visitor_real_visit_try_statement (ValaCodeVisitor* self, ValaTryStatement* stmt)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_TRY_STATEMENT (stmt));
}


void vala_code_visitor_visit_try_statement (ValaCodeVisitor* self, ValaTryStatement* stmt)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_try_statement (self, stmt);
}


/**
 * Visit operation called for catch clauses.
 *
 * @param clause a catch cluase
 */
static void vala_code_visitor_real_visit_catch_clause (ValaCodeVisitor* self, ValaCatchClause* clause)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_CATCH_CLAUSE (clause));
}


void vala_code_visitor_visit_catch_clause (ValaCodeVisitor* self, ValaCatchClause* clause)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_catch_clause (self, clause);
}


/**
 * Visit operation called for lock statements before the body has been visited.
 *
 * @param stmt a lock statement
 */
static void vala_code_visitor_real_visit_lock_statement (ValaCodeVisitor* self, ValaLockStatement* stmt)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_LOCK_STATEMENT (stmt));
}


void vala_code_visitor_visit_lock_statement (ValaCodeVisitor* self, ValaLockStatement* stmt)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_lock_statement (self, stmt);
}


/**
 * Visit operations called for array creation expresions.
 *
 * @param expr an array creation expression
 */
static void vala_code_visitor_real_visit_begin_array_creation_expression (ValaCodeVisitor* self, ValaArrayCreationExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_ARRAY_CREATION_EXPRESSION (expr));
}


void vala_code_visitor_visit_begin_array_creation_expression (ValaCodeVisitor* self, ValaArrayCreationExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_begin_array_creation_expression (self, expr);
}


/**
 * Visit operations called for array creation expresions.
 *
 * @param expr an array creation expression
 */
static void vala_code_visitor_real_visit_end_array_creation_expression (ValaCodeVisitor* self, ValaArrayCreationExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_ARRAY_CREATION_EXPRESSION (expr));
}


void vala_code_visitor_visit_end_array_creation_expression (ValaCodeVisitor* self, ValaArrayCreationExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_end_array_creation_expression (self, expr);
}


/**
 * Visit operation called for boolean literals.
 *
 * @param lit a boolean literal
 */
static void vala_code_visitor_real_visit_boolean_literal (ValaCodeVisitor* self, ValaBooleanLiteral* lit)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_BOOLEAN_LITERAL (lit));
}


void vala_code_visitor_visit_boolean_literal (ValaCodeVisitor* self, ValaBooleanLiteral* lit)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_boolean_literal (self, lit);
}


/**
 * Visit operation called for character literals.
 *
 * @param lit a character literal
 */
static void vala_code_visitor_real_visit_character_literal (ValaCodeVisitor* self, ValaCharacterLiteral* lit)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_CHARACTER_LITERAL (lit));
}


void vala_code_visitor_visit_character_literal (ValaCodeVisitor* self, ValaCharacterLiteral* lit)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_character_literal (self, lit);
}


/**
 * Visit operation called for integer literals.
 *
 * @param lit an integer literal
 */
static void vala_code_visitor_real_visit_integer_literal (ValaCodeVisitor* self, ValaIntegerLiteral* lit)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_INTEGER_LITERAL (lit));
}


void vala_code_visitor_visit_integer_literal (ValaCodeVisitor* self, ValaIntegerLiteral* lit)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_integer_literal (self, lit);
}


/**
 * Visit operation called for real literals.
 *
 * @param lit a real literal
 */
static void vala_code_visitor_real_visit_real_literal (ValaCodeVisitor* self, ValaRealLiteral* lit)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_REAL_LITERAL (lit));
}


void vala_code_visitor_visit_real_literal (ValaCodeVisitor* self, ValaRealLiteral* lit)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_real_literal (self, lit);
}


/**
 * Visit operation called for string literals.
 *
 * @param lit a string literal
 */
static void vala_code_visitor_real_visit_string_literal (ValaCodeVisitor* self, ValaStringLiteral* lit)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_STRING_LITERAL (lit));
}


void vala_code_visitor_visit_string_literal (ValaCodeVisitor* self, ValaStringLiteral* lit)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_string_literal (self, lit);
}


/**
 * Visit operation called for null literals.
 *
 * @param lit a null literal
 */
static void vala_code_visitor_real_visit_null_literal (ValaCodeVisitor* self, ValaNullLiteral* lit)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_NULL_LITERAL (lit));
}


void vala_code_visitor_visit_null_literal (ValaCodeVisitor* self, ValaNullLiteral* lit)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_null_literal (self, lit);
}


/**
 * Visit operation called for literal expressions.
 *
 * @param expr a literal expression
 */
static void vala_code_visitor_real_visit_literal_expression (ValaCodeVisitor* self, ValaLiteralExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_LITERAL_EXPRESSION (expr));
}


void vala_code_visitor_visit_literal_expression (ValaCodeVisitor* self, ValaLiteralExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_literal_expression (self, expr);
}


/**
 * Visit operation called for parenthesized expressions.
 *
 * @param expr a parenthesized expression
 */
static void vala_code_visitor_real_visit_parenthesized_expression (ValaCodeVisitor* self, ValaParenthesizedExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_PARENTHESIZED_EXPRESSION (expr));
}


void vala_code_visitor_visit_parenthesized_expression (ValaCodeVisitor* self, ValaParenthesizedExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_parenthesized_expression (self, expr);
}


/**
 * Visit operation called for member access expressions.
 *
 * @param expr a member access expression
 */
static void vala_code_visitor_real_visit_member_access (ValaCodeVisitor* self, ValaMemberAccess* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_MEMBER_ACCESS (expr));
}


void vala_code_visitor_visit_member_access (ValaCodeVisitor* self, ValaMemberAccess* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_member_access (self, expr);
}


/**
 * Visit operation called at beginning of invocation expressions.
 *
 * @param expr an invocation expression
 */
static void vala_code_visitor_real_visit_begin_invocation_expression (ValaCodeVisitor* self, ValaInvocationExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_INVOCATION_EXPRESSION (expr));
}


void vala_code_visitor_visit_begin_invocation_expression (ValaCodeVisitor* self, ValaInvocationExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_begin_invocation_expression (self, expr);
}


/**
 * Visit operation called at end of invocation expressions.
 *
 * @param expr an invocation expression
 */
static void vala_code_visitor_real_visit_end_invocation_expression (ValaCodeVisitor* self, ValaInvocationExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_INVOCATION_EXPRESSION (expr));
}


void vala_code_visitor_visit_end_invocation_expression (ValaCodeVisitor* self, ValaInvocationExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_end_invocation_expression (self, expr);
}


/**
 * Visit operation called for element access expressions.
 *
 * @param expr an element access expression
 */
static void vala_code_visitor_real_visit_element_access (ValaCodeVisitor* self, ValaElementAccess* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_ELEMENT_ACCESS (expr));
}


void vala_code_visitor_visit_element_access (ValaCodeVisitor* self, ValaElementAccess* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_element_access (self, expr);
}


/**
 * Visit operation called for base access expressions.
 *
 * @param expr a base access expression
 */
static void vala_code_visitor_real_visit_base_access (ValaCodeVisitor* self, ValaBaseAccess* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_BASE_ACCESS (expr));
}


void vala_code_visitor_visit_base_access (ValaCodeVisitor* self, ValaBaseAccess* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_base_access (self, expr);
}


/**
 * Visit operation called for postfix expressions.
 *
 * @param expr a postfix expression
 */
static void vala_code_visitor_real_visit_postfix_expression (ValaCodeVisitor* self, ValaPostfixExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_POSTFIX_EXPRESSION (expr));
}


void vala_code_visitor_visit_postfix_expression (ValaCodeVisitor* self, ValaPostfixExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_postfix_expression (self, expr);
}


/**
 * Visit operation called at beginning of object creation
 * expressions.
 *
 * @param expr an object creation expression
 */
static void vala_code_visitor_real_visit_begin_object_creation_expression (ValaCodeVisitor* self, ValaObjectCreationExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (expr));
}


void vala_code_visitor_visit_begin_object_creation_expression (ValaCodeVisitor* self, ValaObjectCreationExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_begin_object_creation_expression (self, expr);
}


/**
 * Visit operation called at end of object creation expressions.
 *
 * @param expr an object creation expression
 */
static void vala_code_visitor_real_visit_end_object_creation_expression (ValaCodeVisitor* self, ValaObjectCreationExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (expr));
}


void vala_code_visitor_visit_end_object_creation_expression (ValaCodeVisitor* self, ValaObjectCreationExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_end_object_creation_expression (self, expr);
}


/**
 * Visit operation called for sizeof expressions.
 *
 * @param expr a sizeof expression
 */
static void vala_code_visitor_real_visit_sizeof_expression (ValaCodeVisitor* self, ValaSizeofExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_SIZEOF_EXPRESSION (expr));
}


void vala_code_visitor_visit_sizeof_expression (ValaCodeVisitor* self, ValaSizeofExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_sizeof_expression (self, expr);
}


/**
 * Visit operation called for typeof expressions.
 *
 * @param expr a typeof expression
 */
static void vala_code_visitor_real_visit_typeof_expression (ValaCodeVisitor* self, ValaTypeofExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_TYPEOF_EXPRESSION (expr));
}


void vala_code_visitor_visit_typeof_expression (ValaCodeVisitor* self, ValaTypeofExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_typeof_expression (self, expr);
}


/**
 * Visit operation called for unary expressions.
 *
 * @param expr an unary expression
 */
static void vala_code_visitor_real_visit_unary_expression (ValaCodeVisitor* self, ValaUnaryExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_UNARY_EXPRESSION (expr));
}


void vala_code_visitor_visit_unary_expression (ValaCodeVisitor* self, ValaUnaryExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_unary_expression (self, expr);
}


/**
 * Visit operation called for call expressions.
 *
 * @param expr a call expression
 */
static void vala_code_visitor_real_visit_cast_expression (ValaCodeVisitor* self, ValaCastExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_CAST_EXPRESSION (expr));
}


void vala_code_visitor_visit_cast_expression (ValaCodeVisitor* self, ValaCastExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_cast_expression (self, expr);
}


/**
 * Visit operation called for pointer indirections.
 *
 * @param expr a pointer indirection
 */
static void vala_code_visitor_real_visit_pointer_indirection (ValaCodeVisitor* self, ValaPointerIndirection* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_POINTER_INDIRECTION (expr));
}


void vala_code_visitor_visit_pointer_indirection (ValaCodeVisitor* self, ValaPointerIndirection* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_pointer_indirection (self, expr);
}


/**
 * Visit operation called for address-of expressions.
 *
 * @param expr an address-of expression
 */
static void vala_code_visitor_real_visit_addressof_expression (ValaCodeVisitor* self, ValaAddressofExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_ADDRESSOF_EXPRESSION (expr));
}


void vala_code_visitor_visit_addressof_expression (ValaCodeVisitor* self, ValaAddressofExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_addressof_expression (self, expr);
}


/**
 * Visit operation called for reference transfer expressions.
 *
 * @param expr a reference transfer expression
 */
static void vala_code_visitor_real_visit_reference_transfer_expression (ValaCodeVisitor* self, ValaReferenceTransferExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_REFERENCE_TRANSFER_EXPRESSION (expr));
}


void vala_code_visitor_visit_reference_transfer_expression (ValaCodeVisitor* self, ValaReferenceTransferExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_reference_transfer_expression (self, expr);
}


/**
 * Visit operation called for binary expressions.
 *
 * @param expr a binary expression
 */
static void vala_code_visitor_real_visit_binary_expression (ValaCodeVisitor* self, ValaBinaryExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_BINARY_EXPRESSION (expr));
}


void vala_code_visitor_visit_binary_expression (ValaCodeVisitor* self, ValaBinaryExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_binary_expression (self, expr);
}


/**
 * Visit operation called for type checks.
 *
 * @param expr a type check expression
 */
static void vala_code_visitor_real_visit_type_check (ValaCodeVisitor* self, ValaTypeCheck* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_TYPE_CHECK (expr));
}


void vala_code_visitor_visit_type_check (ValaCodeVisitor* self, ValaTypeCheck* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_type_check (self, expr);
}


/**
 * Visit operation called for conditional expressions.
 *
 * @param expr a conditional expression
 */
static void vala_code_visitor_real_visit_conditional_expression (ValaCodeVisitor* self, ValaConditionalExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_CONDITIONAL_EXPRESSION (expr));
}


void vala_code_visitor_visit_conditional_expression (ValaCodeVisitor* self, ValaConditionalExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_conditional_expression (self, expr);
}


/**
 * Visit operation called at beginning of lambda expressions.
 *
 * @param expr a lambda expression
 */
static void vala_code_visitor_real_visit_begin_lambda_expression (ValaCodeVisitor* self, ValaLambdaExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_LAMBDA_EXPRESSION (expr));
}


void vala_code_visitor_visit_begin_lambda_expression (ValaCodeVisitor* self, ValaLambdaExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_begin_lambda_expression (self, expr);
}


/**
 * Visit operation called at end of lambda expressions.
 *
 * @param expr a lambda expression
 */
static void vala_code_visitor_real_visit_end_lambda_expression (ValaCodeVisitor* self, ValaLambdaExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_LAMBDA_EXPRESSION (expr));
}


void vala_code_visitor_visit_end_lambda_expression (ValaCodeVisitor* self, ValaLambdaExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_end_lambda_expression (self, expr);
}


/**
 * Visit operation called at beginning of assignments.
 *
 * @param a an assignment
 */
static void vala_code_visitor_real_visit_begin_assignment (ValaCodeVisitor* self, ValaAssignment* a)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_ASSIGNMENT (a));
}


void vala_code_visitor_visit_begin_assignment (ValaCodeVisitor* self, ValaAssignment* a)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_begin_assignment (self, a);
}


/**
 * Visit operation called at end of assignments.
 *
 * @param a an assignment
 */
static void vala_code_visitor_real_visit_end_assignment (ValaCodeVisitor* self, ValaAssignment* a)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_ASSIGNMENT (a));
}


void vala_code_visitor_visit_end_assignment (ValaCodeVisitor* self, ValaAssignment* a)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_end_assignment (self, a);
}


/**
 * Visit operation called at end of full expressions.
 *
 * @param expr a full expression
 */
static void vala_code_visitor_real_visit_end_full_expression (ValaCodeVisitor* self, ValaExpression* expr)
{
	g_return_if_fail (VALA_IS_CODE_VISITOR (self));
	g_return_if_fail (VALA_IS_EXPRESSION (expr));
}


void vala_code_visitor_visit_end_full_expression (ValaCodeVisitor* self, ValaExpression* expr)
{
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_end_full_expression (self, expr);
}


static void vala_code_visitor_class_init (ValaCodeVisitorClass * klass)
{
	vala_code_visitor_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_code_visitor_real_visit_source_file;
	VALA_CODE_VISITOR_CLASS (klass)->visit_namespace = vala_code_visitor_real_visit_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_code_visitor_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_code_visitor_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_code_visitor_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum = vala_code_visitor_real_visit_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum_value = vala_code_visitor_real_visit_enum_value;
	VALA_CODE_VISITOR_CLASS (klass)->visit_callback = vala_code_visitor_real_visit_callback;
	VALA_CODE_VISITOR_CLASS (klass)->visit_member = vala_code_visitor_real_visit_member;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constant = vala_code_visitor_real_visit_constant;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_code_visitor_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_code_visitor_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_creation_method = vala_code_visitor_real_visit_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_formal_parameter = vala_code_visitor_real_visit_formal_parameter;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_code_visitor_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property_accessor = vala_code_visitor_real_visit_property_accessor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_signal = vala_code_visitor_real_visit_signal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constructor = vala_code_visitor_real_visit_constructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_destructor = vala_code_visitor_real_visit_destructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_named_argument = vala_code_visitor_real_visit_named_argument;
	VALA_CODE_VISITOR_CLASS (klass)->visit_type_parameter = vala_code_visitor_real_visit_type_parameter;
	VALA_CODE_VISITOR_CLASS (klass)->visit_namespace_reference = vala_code_visitor_real_visit_namespace_reference;
	VALA_CODE_VISITOR_CLASS (klass)->visit_type_reference = vala_code_visitor_real_visit_type_reference;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_block = vala_code_visitor_real_visit_begin_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_block = vala_code_visitor_real_visit_end_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_empty_statement = vala_code_visitor_real_visit_empty_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_declaration_statement = vala_code_visitor_real_visit_declaration_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_local_variable_declaration = vala_code_visitor_real_visit_local_variable_declaration;
	VALA_CODE_VISITOR_CLASS (klass)->visit_variable_declarator = vala_code_visitor_real_visit_variable_declarator;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_initializer_list = vala_code_visitor_real_visit_begin_initializer_list;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_initializer_list = vala_code_visitor_real_visit_end_initializer_list;
	VALA_CODE_VISITOR_CLASS (klass)->visit_expression_statement = vala_code_visitor_real_visit_expression_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_if_statement = vala_code_visitor_real_visit_if_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_statement = vala_code_visitor_real_visit_switch_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_section = vala_code_visitor_real_visit_switch_section;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_label = vala_code_visitor_real_visit_switch_label;
	VALA_CODE_VISITOR_CLASS (klass)->visit_while_statement = vala_code_visitor_real_visit_while_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_do_statement = vala_code_visitor_real_visit_do_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_for_statement = vala_code_visitor_real_visit_for_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_foreach_statement = vala_code_visitor_real_visit_begin_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_foreach_statement = vala_code_visitor_real_visit_end_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_break_statement = vala_code_visitor_real_visit_break_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_continue_statement = vala_code_visitor_real_visit_continue_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_return_statement = vala_code_visitor_real_visit_begin_return_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_return_statement = vala_code_visitor_real_visit_end_return_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_throw_statement = vala_code_visitor_real_visit_throw_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_try_statement = vala_code_visitor_real_visit_try_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_catch_clause = vala_code_visitor_real_visit_catch_clause;
	VALA_CODE_VISITOR_CLASS (klass)->visit_lock_statement = vala_code_visitor_real_visit_lock_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_array_creation_expression = vala_code_visitor_real_visit_begin_array_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_array_creation_expression = vala_code_visitor_real_visit_end_array_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_boolean_literal = vala_code_visitor_real_visit_boolean_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_character_literal = vala_code_visitor_real_visit_character_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_integer_literal = vala_code_visitor_real_visit_integer_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_real_literal = vala_code_visitor_real_visit_real_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_string_literal = vala_code_visitor_real_visit_string_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_null_literal = vala_code_visitor_real_visit_null_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_literal_expression = vala_code_visitor_real_visit_literal_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_parenthesized_expression = vala_code_visitor_real_visit_parenthesized_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_member_access = vala_code_visitor_real_visit_member_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_invocation_expression = vala_code_visitor_real_visit_begin_invocation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_invocation_expression = vala_code_visitor_real_visit_end_invocation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_element_access = vala_code_visitor_real_visit_element_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_base_access = vala_code_visitor_real_visit_base_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_postfix_expression = vala_code_visitor_real_visit_postfix_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_object_creation_expression = vala_code_visitor_real_visit_begin_object_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_object_creation_expression = vala_code_visitor_real_visit_end_object_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_sizeof_expression = vala_code_visitor_real_visit_sizeof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_typeof_expression = vala_code_visitor_real_visit_typeof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_unary_expression = vala_code_visitor_real_visit_unary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_cast_expression = vala_code_visitor_real_visit_cast_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_pointer_indirection = vala_code_visitor_real_visit_pointer_indirection;
	VALA_CODE_VISITOR_CLASS (klass)->visit_addressof_expression = vala_code_visitor_real_visit_addressof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_reference_transfer_expression = vala_code_visitor_real_visit_reference_transfer_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_binary_expression = vala_code_visitor_real_visit_binary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_type_check = vala_code_visitor_real_visit_type_check;
	VALA_CODE_VISITOR_CLASS (klass)->visit_conditional_expression = vala_code_visitor_real_visit_conditional_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_lambda_expression = vala_code_visitor_real_visit_begin_lambda_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_lambda_expression = vala_code_visitor_real_visit_end_lambda_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_assignment = vala_code_visitor_real_visit_begin_assignment;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_assignment = vala_code_visitor_real_visit_end_assignment;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_full_expression = vala_code_visitor_real_visit_end_full_expression;
}


static void vala_code_visitor_init (ValaCodeVisitor * self)
{
}


GType vala_code_visitor_get_type (void)
{
	static GType vala_code_visitor_type_id = 0;
	if (G_UNLIKELY (vala_code_visitor_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCodeVisitorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_code_visitor_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCodeVisitor), 0, (GInstanceInitFunc) vala_code_visitor_init };
		vala_code_visitor_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaCodeVisitor", &g_define_type_info, G_TYPE_FLAG_ABSTRACT);
	}
	return vala_code_visitor_type_id;
}




