/* valaccodeblock.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include "valaccodeblock.h"
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/collection.h>
#include <ccode/valaccodeblock.h>

struct _ValaCCodeBlockPrivate {
	gboolean _suppress_newline;
	GeeList* statements;
};
#define VALA_CCODE_BLOCK_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CCODE_BLOCK, ValaCCodeBlockPrivate))
enum  {
	VALA_CCODE_BLOCK_DUMMY_PROPERTY,
	VALA_CCODE_BLOCK_SUPPRESS_NEWLINE
};
static void vala_ccode_block_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer);
static gpointer vala_ccode_block_parent_class = NULL;
static void vala_ccode_block_dispose (GObject * obj);


/**
 * Prepend the specified statement to the list of statements.
 */
void vala_ccode_block_prepend_statement (ValaCCodeBlock* self, ValaCCodeNode* statement)
{
	g_return_if_fail (VALA_IS_CCODE_BLOCK (self));
	g_return_if_fail (VALA_IS_CCODE_NODE (statement));
	gee_list_insert (self->priv->statements, 0, statement);
}


/**
 * Append the specified statement to the list of statements.
 */
void vala_ccode_block_add_statement (ValaCCodeBlock* self, ValaCCodeNode* statement)
{
	g_return_if_fail (VALA_IS_CCODE_BLOCK (self));
	g_return_if_fail (VALA_IS_CCODE_NODE (statement));
	/* allow generic nodes to include comments */
	gee_collection_add (GEE_COLLECTION (self->priv->statements), statement);
}


static void vala_ccode_block_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer)
{
	ValaCCodeBlock * self;
	self = ((ValaCCodeBlock*) base);
	g_return_if_fail (VALA_IS_CCODE_WRITER (writer));
	vala_ccode_writer_write_begin_block (writer);
	{
		GeeList* statement_collection;
		GeeIterator* statement_it;
		statement_collection = self->priv->statements;
		statement_it = gee_iterable_iterator (GEE_ITERABLE (statement_collection));
		while (gee_iterator_next (statement_it)) {
			ValaCCodeNode* statement;
			statement = gee_iterator_get (statement_it);
			{
				vala_ccode_node_write_declaration (statement, writer);
				(statement == NULL ? NULL : (statement = (g_object_unref (statement), NULL)));
			}
		}
		(statement_it == NULL ? NULL : (statement_it = (g_object_unref (statement_it), NULL)));
	}
	{
		GeeList* statement_collection;
		GeeIterator* statement_it;
		statement_collection = self->priv->statements;
		statement_it = gee_iterable_iterator (GEE_ITERABLE (statement_collection));
		while (gee_iterator_next (statement_it)) {
			ValaCCodeNode* statement;
			statement = gee_iterator_get (statement_it);
			{
				vala_ccode_node_write (statement, writer);
				(statement == NULL ? NULL : (statement = (g_object_unref (statement), NULL)));
			}
		}
		(statement_it == NULL ? NULL : (statement_it = (g_object_unref (statement_it), NULL)));
	}
	vala_ccode_writer_write_end_block (writer);
	if (!vala_ccode_block_get_suppress_newline (self)) {
		vala_ccode_writer_write_newline (writer);
	}
}


gboolean vala_ccode_block_get_suppress_newline (ValaCCodeBlock* self)
{
	g_return_val_if_fail (VALA_IS_CCODE_BLOCK (self), FALSE);
	return self->priv->_suppress_newline;
}


void vala_ccode_block_set_suppress_newline (ValaCCodeBlock* self, gboolean value)
{
	g_return_if_fail (VALA_IS_CCODE_BLOCK (self));
	self->priv->_suppress_newline = value;
}


static void vala_ccode_block_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaCCodeBlock * self;
	self = VALA_CCODE_BLOCK (object);
	switch (property_id) {
		case VALA_CCODE_BLOCK_SUPPRESS_NEWLINE:
		g_value_set_boolean (value, vala_ccode_block_get_suppress_newline (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_ccode_block_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaCCodeBlock * self;
	self = VALA_CCODE_BLOCK (object);
	switch (property_id) {
		case VALA_CCODE_BLOCK_SUPPRESS_NEWLINE:
		vala_ccode_block_set_suppress_newline (self, g_value_get_boolean (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_ccode_block_class_init (ValaCCodeBlockClass * klass)
{
	vala_ccode_block_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaCCodeBlockPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_ccode_block_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_ccode_block_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_ccode_block_dispose;
	VALA_CCODE_NODE_CLASS (klass)->write = vala_ccode_block_real_write;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CCODE_BLOCK_SUPPRESS_NEWLINE, g_param_spec_boolean ("suppress-newline", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_ccode_block_init (ValaCCodeBlock * self)
{
	self->priv = VALA_CCODE_BLOCK_GET_PRIVATE (self);
	self->priv->statements = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
}


static void vala_ccode_block_dispose (GObject * obj)
{
	ValaCCodeBlock * self;
	ValaCCodeBlockClass * klass;
	GObjectClass * parent_class;
	self = VALA_CCODE_BLOCK (obj);
	(self->priv->statements == NULL ? NULL : (self->priv->statements = (g_object_unref (self->priv->statements), NULL)));
	klass = VALA_CCODE_BLOCK_CLASS (g_type_class_peek (VALA_TYPE_CCODE_BLOCK));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_ccode_block_get_type (void)
{
	static GType vala_ccode_block_type_id = 0;
	if (G_UNLIKELY (vala_ccode_block_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeBlockClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_block_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeBlock), 0, (GInstanceInitFunc) vala_ccode_block_init };
		vala_ccode_block_type_id = g_type_register_static (VALA_TYPE_CCODE_STATEMENT, "ValaCCodeBlock", &g_define_type_info, 0);
	}
	return vala_ccode_block_type_id;
}




