/* valanamespace.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include "valanamespace.h"
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <vala/valasourcereference.h>
#include <vala/valanamespace.h>
#include <vala/valascope.h>
#include <vala/valaclass.h>
#include <vala/valainterface.h>
#include <vala/valastruct.h>
#include <vala/valaenum.h>
#include <vala/valacallback.h>
#include <vala/valaconstant.h>
#include <vala/valafield.h>
#include <vala/valamethod.h>
#include <vala/valacreationmethod.h>
#include <vala/valareport.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>

struct _ValaNamespacePrivate {
	gboolean _pkg;
	GeeList* classes;
	GeeList* interfaces;
	GeeList* structs;
	GeeList* enums;
	GeeList* callbacks;
	GeeList* constants;
	GeeList* fields;
	GeeList* methods;
	char* cprefix;
	char* lower_case_cprefix;
	GeeList* cheader_filenames;
	GeeList* namespaces;
};
#define VALA_NAMESPACE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_NAMESPACE, ValaNamespacePrivate))
enum  {
	VALA_NAMESPACE_DUMMY_PROPERTY,
	VALA_NAMESPACE_PKG
};
static void vala_namespace_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_namespace_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_namespace_real_get_cprefix (ValaSymbol* base);
static char* vala_namespace_real_get_lower_case_cprefix (ValaSymbol* base);
static GeeCollection* vala_namespace_real_get_cheader_filenames (ValaSymbol* base);
static void vala_namespace_process_ccode_attribute (ValaNamespace* self, ValaAttribute* a);
static gpointer vala_namespace_parent_class = NULL;
static void vala_namespace_dispose (GObject * obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);


/**
 * Creates a new namespace.
 *
 * @param name             namespace name
 * @param source_reference reference to source code
 * @return                 newly created namespace
 */
ValaNamespace* vala_namespace_new (const char* name, ValaSourceReference* source_reference)
{
	ValaNamespace * self;
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_NAMESPACE, 0, NULL);
	vala_symbol_set_access (VALA_SYMBOL (self), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Adds the specified namespace to this source file.
 *
 * @param ns a namespace
 */
void vala_namespace_add_namespace (ValaNamespace* self, ValaNamespace* ns)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
	gee_collection_add (GEE_COLLECTION (self->priv->namespaces), ns);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) ns)), VALA_SYMBOL (ns));
}


/**
 * Returns a copy of the list of namespaces.
 *
 * @return namespace list
 */
GeeCollection* vala_namespace_get_namespaces (ValaNamespace* self)
{
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->namespaces)));
}


/**
 * Adds the specified class to this namespace.
 *
 * @param cl a class
 */
void vala_namespace_add_class (ValaNamespace* self, ValaClass* cl)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_CLASS (cl));
	gee_collection_add (GEE_COLLECTION (self->priv->classes), cl);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) cl)), VALA_SYMBOL (cl));
}


/**
 * Adds the specified interface to this namespace.
 *
 * @param iface an interface
 */
void vala_namespace_add_interface (ValaNamespace* self, ValaInterface* iface)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	gee_collection_add (GEE_COLLECTION (self->priv->interfaces), iface);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) iface)), VALA_SYMBOL (iface));
}


/**
 * Adds the specified struct to this namespace.
 *
 * @param st a struct
 */
void vala_namespace_add_struct (ValaNamespace* self, ValaStruct* st)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_STRUCT (st));
	gee_collection_add (GEE_COLLECTION (self->priv->structs), st);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) st)), VALA_SYMBOL (st));
}


/**
 * Adds the specified enum to this namespace.
 *
 * @param en an enum
 */
void vala_namespace_add_enum (ValaNamespace* self, ValaEnum* en)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_ENUM (en));
	gee_collection_add (GEE_COLLECTION (self->priv->enums), en);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) en)), VALA_SYMBOL (en));
}


/**
 * Adds the specified callback to this namespace.
 *
 * @param cb a callback
 */
void vala_namespace_add_callback (ValaNamespace* self, ValaCallback* cb)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_CALLBACK (cb));
	gee_collection_add (GEE_COLLECTION (self->priv->callbacks), cb);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) cb)), VALA_SYMBOL (cb));
}


/**
 * Returns a copy of the list of structs.
 *
 * @return struct list
 */
GeeCollection* vala_namespace_get_structs (ValaNamespace* self)
{
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->structs)));
}


/**
 * Returns a copy of the list of classes.
 *
 * @return class list
 */
GeeCollection* vala_namespace_get_classes (ValaNamespace* self)
{
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->classes)));
}


/**
 * Returns a copy of the list of interfaces.
 *
 * @return interface list
 */
GeeCollection* vala_namespace_get_interfaces (ValaNamespace* self)
{
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->interfaces)));
}


/**
 * Adds the specified constant to this namespace.
 *
 * @param constant a constant
 */
void vala_namespace_add_constant (ValaNamespace* self, ValaConstant* constant)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_CONSTANT (constant));
	gee_collection_add (GEE_COLLECTION (self->priv->constants), constant);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) constant)), VALA_SYMBOL (constant));
}


/**
 * Adds the specified field to this namespace.
 *
 * @param f a field
 */
void vala_namespace_add_field (ValaNamespace* self, ValaField* f)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_FIELD (f));
	gee_collection_add (GEE_COLLECTION (self->priv->fields), f);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) f)), VALA_SYMBOL (f));
}


/**
 * Adds the specified method to this namespace.
 *
 * @param m a method
 */
void vala_namespace_add_method (ValaNamespace* self, ValaMethod* m)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	if (VALA_IS_CREATION_METHOD (m)) {
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) m)), "construction methods may only be declared within classes and structs");
		vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
		return;
	}
	if (vala_method_get_instance (m)) {
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) m)), "instance methods not allowed outside of data types");
		vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
		return;
	}
	gee_collection_add (GEE_COLLECTION (self->priv->methods), m);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) m)), VALA_SYMBOL (m));
}


static void vala_namespace_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaNamespace * self;
	self = ((ValaNamespace*) base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_namespace (visitor, self);
}


static void vala_namespace_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaNamespace * self;
	self = ((ValaNamespace*) base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GeeList* ns_collection;
		GeeIterator* ns_it;
		ns_collection = self->priv->namespaces;
		ns_it = gee_iterable_iterator (GEE_ITERABLE (ns_collection));
		while (gee_iterator_next (ns_it)) {
			ValaNamespace* ns;
			ns = gee_iterator_get (ns_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (ns), visitor);
				(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
			}
		}
		(ns_it == NULL ? NULL : (ns_it = (g_object_unref (ns_it), NULL)));
	}
	{
		GeeList* en_collection;
		GeeIterator* en_it;
		en_collection = self->priv->enums;
		en_it = gee_iterable_iterator (GEE_ITERABLE (en_collection));
		while (gee_iterator_next (en_it)) {
			ValaEnum* en;
			en = gee_iterator_get (en_it);
			{
				/* process enums first to avoid order problems in C code */
				vala_code_node_accept (VALA_CODE_NODE (en), visitor);
				(en == NULL ? NULL : (en = (g_object_unref (en), NULL)));
			}
		}
		(en_it == NULL ? NULL : (en_it = (g_object_unref (en_it), NULL)));
	}
	{
		GeeList* cl_collection;
		GeeIterator* cl_it;
		cl_collection = self->priv->classes;
		cl_it = gee_iterable_iterator (GEE_ITERABLE (cl_collection));
		while (gee_iterator_next (cl_it)) {
			ValaClass* cl;
			cl = gee_iterator_get (cl_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (cl), visitor);
				(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
			}
		}
		(cl_it == NULL ? NULL : (cl_it = (g_object_unref (cl_it), NULL)));
	}
	{
		GeeList* iface_collection;
		GeeIterator* iface_it;
		iface_collection = self->priv->interfaces;
		iface_it = gee_iterable_iterator (GEE_ITERABLE (iface_collection));
		while (gee_iterator_next (iface_it)) {
			ValaInterface* iface;
			iface = gee_iterator_get (iface_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (iface), visitor);
				(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
			}
		}
		(iface_it == NULL ? NULL : (iface_it = (g_object_unref (iface_it), NULL)));
	}
	{
		GeeList* st_collection;
		GeeIterator* st_it;
		st_collection = self->priv->structs;
		st_it = gee_iterable_iterator (GEE_ITERABLE (st_collection));
		while (gee_iterator_next (st_it)) {
			ValaStruct* st;
			st = gee_iterator_get (st_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (st), visitor);
				(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
			}
		}
		(st_it == NULL ? NULL : (st_it = (g_object_unref (st_it), NULL)));
	}
	{
		GeeList* cb_collection;
		GeeIterator* cb_it;
		cb_collection = self->priv->callbacks;
		cb_it = gee_iterable_iterator (GEE_ITERABLE (cb_collection));
		while (gee_iterator_next (cb_it)) {
			ValaCallback* cb;
			cb = gee_iterator_get (cb_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (cb), visitor);
				(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
			}
		}
		(cb_it == NULL ? NULL : (cb_it = (g_object_unref (cb_it), NULL)));
	}
	{
		GeeList* c_collection;
		GeeIterator* c_it;
		c_collection = self->priv->constants;
		c_it = gee_iterable_iterator (GEE_ITERABLE (c_collection));
		while (gee_iterator_next (c_it)) {
			ValaConstant* c;
			c = gee_iterator_get (c_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (c), visitor);
				(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
			}
		}
		(c_it == NULL ? NULL : (c_it = (g_object_unref (c_it), NULL)));
	}
	{
		GeeList* f_collection;
		GeeIterator* f_it;
		f_collection = self->priv->fields;
		f_it = gee_iterable_iterator (GEE_ITERABLE (f_collection));
		while (gee_iterator_next (f_it)) {
			ValaField* f;
			f = gee_iterator_get (f_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (f), visitor);
				(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
			}
		}
		(f_it == NULL ? NULL : (f_it = (g_object_unref (f_it), NULL)));
	}
	{
		GeeList* m_collection;
		GeeIterator* m_it;
		m_collection = self->priv->methods;
		m_it = gee_iterable_iterator (GEE_ITERABLE (m_collection));
		while (gee_iterator_next (m_it)) {
			ValaMethod* m;
			m = gee_iterator_get (m_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (m), visitor);
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
			}
		}
		(m_it == NULL ? NULL : (m_it = (g_object_unref (m_it), NULL)));
	}
}


static char* vala_namespace_real_get_cprefix (ValaSymbol* base)
{
	ValaNamespace * self;
	const char* _tmp3;
	self = ((ValaNamespace*) base);
	if (self->priv->cprefix == NULL) {
		if (vala_symbol_get_name (((ValaSymbol*) self)) == NULL) {
			char* _tmp0;
			_tmp0 = NULL;
			self->priv->cprefix = (_tmp0 = g_strdup (""), (self->priv->cprefix = (g_free (self->priv->cprefix), NULL)), _tmp0);
		} else {
			char* _tmp2;
			const char* _tmp1;
			_tmp2 = NULL;
			_tmp1 = NULL;
			self->priv->cprefix = (_tmp2 = (_tmp1 = vala_symbol_get_name (((ValaSymbol*) self)), (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->cprefix = (g_free (self->priv->cprefix), NULL)), _tmp2);
		}
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->cprefix, (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
}


/**
 * Sets the camel case string to be prepended to the name of members of
 * this namespace when used in C code.
 *
 * @param cprefix the camel case prefix to be used in C code
 */
void vala_namespace_set_cprefix (ValaNamespace* self, const char* cprefix)
{
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cprefix = (_tmp1 = (_tmp0 = cprefix, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->cprefix = (g_free (self->priv->cprefix), NULL)), _tmp1);
}


/**
 * Returns the lower case string to be prepended to the name of members
 * of this namespace when used in C code.
 *
 * @return the lower case prefix to be used in C code
 */
static char* vala_namespace_real_get_lower_case_cprefix (ValaSymbol* base)
{
	ValaNamespace * self;
	const char* _tmp3;
	self = ((ValaNamespace*) base);
	if (self->priv->lower_case_cprefix == NULL) {
		if (vala_symbol_get_name (((ValaSymbol*) self)) == NULL) {
			char* _tmp0;
			_tmp0 = NULL;
			self->priv->lower_case_cprefix = (_tmp0 = g_strdup (""), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), _tmp0);
		} else {
			char* _tmp2;
			char* _tmp1;
			_tmp2 = NULL;
			_tmp1 = NULL;
			self->priv->lower_case_cprefix = (_tmp2 = g_strdup_printf ("%s_", (_tmp1 = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name (((ValaSymbol*) self))))), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), _tmp2);
			(_tmp1 = (g_free (_tmp1), NULL));
		}
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->lower_case_cprefix, (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
}


/**
 * Sets the lower case string to be prepended to the name of members of
 * this namespace when used in C code.
 *
 * @param cprefix the lower case prefix to be used in C code
 */
void vala_namespace_set_lower_case_cprefix (ValaNamespace* self, const char* cprefix)
{
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->lower_case_cprefix = (_tmp1 = (_tmp0 = cprefix, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), _tmp1);
}


static GeeCollection* vala_namespace_real_get_cheader_filenames (ValaSymbol* base)
{
	ValaNamespace * self;
	self = ((ValaNamespace*) base);
	return GEE_COLLECTION (gee_read_only_collection_new (g_strdup, g_free, GEE_COLLECTION (self->priv->cheader_filenames)));
}


/**
 * Returns the C header filename of this namespace.
 *
 * @return header filename
 */
char* vala_namespace_get_cheader_filename (ValaNamespace* self)
{
	GString* s;
	gboolean first;
	const char* _tmp0;
	char* _tmp1;
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	s = g_string_new ("");
	first = TRUE;
	{
		GeeCollection* cheader_filename_collection;
		GeeIterator* cheader_filename_it;
		cheader_filename_collection = vala_symbol_get_cheader_filenames (VALA_SYMBOL (self));
		cheader_filename_it = gee_iterable_iterator (GEE_ITERABLE (cheader_filename_collection));
		while (gee_iterator_next (cheader_filename_it)) {
			char* cheader_filename;
			cheader_filename = gee_iterator_get (cheader_filename_it);
			{
				if (first) {
					first = FALSE;
				} else {
					g_string_append_c (s, ',');
				}
				g_string_append (s, cheader_filename);
				(cheader_filename = (g_free (cheader_filename), NULL));
			}
		}
		(cheader_filename_collection == NULL ? NULL : (cheader_filename_collection = (g_object_unref (cheader_filename_collection), NULL)));
		(cheader_filename_it == NULL ? NULL : (cheader_filename_it = (g_object_unref (cheader_filename_it), NULL)));
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = (_tmp0 = s->str, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (s == NULL ? NULL : (s = (g_string_free (s, TRUE), NULL))), _tmp1);
	(s == NULL ? NULL : (s = (g_string_free (s, TRUE), NULL)));
}


/**
 * Sets the C header filename of this namespace to the specified
 * filename.
 *
 * @param cheader_filename header filename
 */
void vala_namespace_set_cheader_filename (ValaNamespace* self, const char* cheader_filename)
{
	GeeList* _tmp0;
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (cheader_filename != NULL);
	_tmp0 = NULL;
	self->priv->cheader_filenames = (_tmp0 = GEE_LIST (gee_array_list_new (g_strdup, g_free, g_direct_equal)), (self->priv->cheader_filenames == NULL ? NULL : (self->priv->cheader_filenames = (g_object_unref (self->priv->cheader_filenames), NULL))), _tmp0);
	gee_collection_add (GEE_COLLECTION (self->priv->cheader_filenames), cheader_filename);
}


static void vala_namespace_process_ccode_attribute (ValaNamespace* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cprefix")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_namespace_set_cprefix (self, (_tmp0 = vala_attribute_get_string (a, "cprefix")));
		(_tmp0 = (g_free (_tmp0), NULL));
	}
	if (vala_attribute_has_argument (a, "lower_case_cprefix")) {
		char* _tmp1;
		_tmp1 = NULL;
		vala_namespace_set_lower_case_cprefix (self, (_tmp1 = vala_attribute_get_string (a, "lower_case_cprefix")));
		(_tmp1 = (g_free (_tmp1), NULL));
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			char** filename_it;
			int filename_collection_length1;
			filename_collection = g_strsplit (val, ",", 0);
			filename_collection_length1 = -1;
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* _tmp2;
				char* filename;
				_tmp2 = NULL;
				filename = (_tmp2 = *filename_it, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
				{
					gee_collection_add (GEE_COLLECTION (self->priv->cheader_filenames), filename);
					(filename = (g_free (filename), NULL));
				}
			}
			(filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, ((GDestroyNotify) g_free)), NULL));
		}
		(val = (g_free (val), NULL));
	}
}


/**
 * Process all associated attributes.
 */
void vala_namespace_process_attributes (ValaNamespace* self)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = a_it->data, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "CCode") == 0) {
					vala_namespace_process_ccode_attribute (self, a);
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


gboolean vala_namespace_get_pkg (ValaNamespace* self)
{
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), FALSE);
	return self->priv->_pkg;
}


void vala_namespace_set_pkg (ValaNamespace* self, gboolean value)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	self->priv->_pkg = value;
}


static void vala_namespace_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaNamespace * self;
	self = VALA_NAMESPACE (object);
	switch (property_id) {
		case VALA_NAMESPACE_PKG:
		g_value_set_boolean (value, vala_namespace_get_pkg (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_namespace_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaNamespace * self;
	self = VALA_NAMESPACE (object);
	switch (property_id) {
		case VALA_NAMESPACE_PKG:
		vala_namespace_set_pkg (self, g_value_get_boolean (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_namespace_class_init (ValaNamespaceClass * klass)
{
	vala_namespace_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaNamespacePrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_namespace_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_namespace_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_namespace_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_namespace_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_namespace_real_accept_children;
	VALA_SYMBOL_CLASS (klass)->get_cprefix = vala_namespace_real_get_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_namespace_real_get_lower_case_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_cheader_filenames = vala_namespace_real_get_cheader_filenames;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_NAMESPACE_PKG, g_param_spec_boolean ("pkg", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_namespace_init (ValaNamespace * self)
{
	self->priv = VALA_NAMESPACE_GET_PRIVATE (self);
	self->priv->classes = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->interfaces = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->structs = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->enums = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->callbacks = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->constants = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->fields = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->methods = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->cheader_filenames = GEE_LIST (gee_array_list_new (g_strdup, g_free, g_direct_equal));
	self->priv->namespaces = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
}


static void vala_namespace_dispose (GObject * obj)
{
	ValaNamespace * self;
	ValaNamespaceClass * klass;
	GObjectClass * parent_class;
	self = VALA_NAMESPACE (obj);
	(self->priv->classes == NULL ? NULL : (self->priv->classes = (g_object_unref (self->priv->classes), NULL)));
	(self->priv->interfaces == NULL ? NULL : (self->priv->interfaces = (g_object_unref (self->priv->interfaces), NULL)));
	(self->priv->structs == NULL ? NULL : (self->priv->structs = (g_object_unref (self->priv->structs), NULL)));
	(self->priv->enums == NULL ? NULL : (self->priv->enums = (g_object_unref (self->priv->enums), NULL)));
	(self->priv->callbacks == NULL ? NULL : (self->priv->callbacks = (g_object_unref (self->priv->callbacks), NULL)));
	(self->priv->constants == NULL ? NULL : (self->priv->constants = (g_object_unref (self->priv->constants), NULL)));
	(self->priv->fields == NULL ? NULL : (self->priv->fields = (g_object_unref (self->priv->fields), NULL)));
	(self->priv->methods == NULL ? NULL : (self->priv->methods = (g_object_unref (self->priv->methods), NULL)));
	(self->priv->cprefix = (g_free (self->priv->cprefix), NULL));
	(self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL));
	(self->priv->cheader_filenames == NULL ? NULL : (self->priv->cheader_filenames = (g_object_unref (self->priv->cheader_filenames), NULL)));
	(self->priv->namespaces == NULL ? NULL : (self->priv->namespaces = (g_object_unref (self->priv->namespaces), NULL)));
	klass = VALA_NAMESPACE_CLASS (g_type_class_peek (VALA_TYPE_NAMESPACE));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_namespace_get_type (void)
{
	static GType vala_namespace_type_id = 0;
	if (G_UNLIKELY (vala_namespace_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaNamespaceClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_namespace_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaNamespace), 0, (GInstanceInitFunc) vala_namespace_init };
		vala_namespace_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaNamespace", &g_define_type_info, 0);
	}
	return vala_namespace_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func)
{
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}




