/* valacompiler.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include "valacompiler.h"
#include <stdlib.h>
#include <string.h>
#include <vala/valacodecontext.h>
#include <vala/valareport.h>
#include <stdio.h>
#include <glib/gstdio.h>
#include <config.h>
#include <vala/valasourcefile.h>
#include <vala/valasourcereference.h>
#include <vala/valaparser.h>
#include <vala/valaattributeprocessor.h>
#include <vala/valasymbolresolver.h>
#include <gobject/valadbusbindingprovider.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valabindingprovider.h>
#include <vala/valamemorymanager.h>
#include <gobject/valacodegenerator.h>
#include <vala/valainterfacewriter.h>
#include <gobject/valaccodecompiler.h>

typedef struct _ValaCompiler ValaCompiler;
typedef struct _ValaCompilerClass ValaCompilerClass;
typedef struct _ValaCompilerPrivate ValaCompilerPrivate;

#define VALA_TYPE_COMPILER (vala_compiler_get_type ())
#define VALA_COMPILER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_COMPILER, ValaCompiler))
#define VALA_COMPILER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_COMPILER, ValaCompilerClass))
#define VALA_IS_COMPILER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_COMPILER))
#define VALA_IS_COMPILER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_COMPILER))
#define VALA_COMPILER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_COMPILER, ValaCompilerClass))

struct _ValaCompiler {
	GObject parent;
	ValaCompilerPrivate * priv;
};
struct _ValaCompilerClass {
	GObjectClass parent;
};
struct _ValaCompilerPrivate {
	ValaCodeContext* context;
};
#define VALA_COMPILER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_COMPILER, ValaCompilerPrivate))
enum  {
	VALA_COMPILER_DUMMY_PROPERTY
};
static char* vala_compiler_directory;
static gboolean vala_compiler_version;
static char** vala_compiler_sources;
static char** vala_compiler_vapi_directories;
static char* vala_compiler_library;
static char** vala_compiler_packages;
static gboolean vala_compiler_disable_memory_management;
static gboolean vala_compiler_ccode_only;
static gboolean vala_compiler_compile_only;
static char* vala_compiler_output;
static gboolean vala_compiler_debug;
static gboolean vala_compiler_thread;
static gint vala_compiler_optlevel;
static gboolean vala_compiler_disable_assert;
static gboolean vala_compiler_enable_checking;
static char* vala_compiler_cc_command;
static char** vala_compiler_cc_options;
static gboolean vala_compiler_save_temps;
static const GOptionEntry VALA_COMPILER_options[] = {{"vapidir", 0, 0, G_OPTION_ARG_FILENAME_ARRAY, &vala_compiler_vapi_directories, "Look for package bindings in DIRECTORY", "DIRECTORY..."}, {"pkg", 0, 0, G_OPTION_ARG_STRING_ARRAY, &vala_compiler_packages, "Include binding for PACKAGE", "PACKAGE..."}, {"library", 0, 0, G_OPTION_ARG_STRING, &vala_compiler_library, "Library name", "NAME"}, {"directory", 'd', 0, G_OPTION_ARG_FILENAME, &vala_compiler_directory, "Output directory", "DIRECTORY"}, {"version", 0, 0, G_OPTION_ARG_NONE, &vala_compiler_version, "Display version number", NULL}, {"disable-memory-management", 0, 0, G_OPTION_ARG_NONE, &vala_compiler_disable_memory_management, "Disable memory management", NULL}, {"ccode", 'C', 0, G_OPTION_ARG_NONE, &vala_compiler_ccode_only, "Output C code", NULL}, {"compile", 'c', 0, G_OPTION_ARG_NONE, &vala_compiler_compile_only, "Compile but do not link", NULL}, {"output", 'o', 0, G_OPTION_ARG_FILENAME, &vala_compiler_output, "Place output in file FILE", "FILE"}, {"debug", 'g', 0, G_OPTION_ARG_NONE, &vala_compiler_debug, "Produce debug information", NULL}, {"thread", 0, 0, G_OPTION_ARG_NONE, &vala_compiler_thread, "Enable multithreading support", NULL}, {"optimize", 'O', 0, G_OPTION_ARG_INT, &vala_compiler_optlevel, "Optimization level", "OPTLEVEL"}, {"disable-assert", 0, 0, G_OPTION_ARG_NONE, &vala_compiler_disable_assert, "Disable assertions", NULL}, {"enable-checking", 0, 0, G_OPTION_ARG_NONE, &vala_compiler_enable_checking, "Enable run-time checks", NULL}, {"cc", 0, 0, G_OPTION_ARG_STRING, &vala_compiler_cc_command, "Use COMMAND as C compiler command", "COMMAND"}, {"Xcc", 'X', 0, G_OPTION_ARG_STRING_ARRAY, &vala_compiler_cc_options, "Pass OPTION to the C compiler", "OPTION..."}, {"save-temps", 0, 0, G_OPTION_ARG_NONE, &vala_compiler_save_temps, "Keep temporary files", NULL}, {"", 0, 0, G_OPTION_ARG_FILENAME_ARRAY, &vala_compiler_sources, NULL, "FILE..."}, {NULL}};
static gint vala_compiler_quit (ValaCompiler* self);
static char* vala_compiler_get_package_path (ValaCompiler* self, const char* pkg);
static gboolean vala_compiler_add_package (ValaCompiler* self, ValaCodeContext* context, const char* pkg);
static gint vala_compiler_run (ValaCompiler* self);
static gint vala_compiler_main (int args_length1, char** args);
static gpointer vala_compiler_parent_class = NULL;
static void vala_compiler_dispose (GObject * obj);
static GType vala_compiler_get_type (void);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);


static gint vala_compiler_quit (ValaCompiler* self)
{
	g_return_val_if_fail (VALA_IS_COMPILER (self), 0);
	if (vala_report_get_errors () == 0) {
		fprintf (stdout, "Compilation succeeded - %d warning(s)\n", vala_report_get_warnings ());
		return 0;
	} else {
		fprintf (stdout, "Compilation failed: %d error(s), %d warning(s)\n", vala_report_get_errors (), vala_report_get_warnings ());
		return 1;
	}
}


static char* vala_compiler_get_package_path (ValaCompiler* self, const char* pkg)
{
	char* basename;
	char* filename;
	char* _tmp3;
	char* _tmp5;
	gpointer _tmp7;
	g_return_val_if_fail (VALA_IS_COMPILER (self), NULL);
	g_return_val_if_fail (pkg != NULL, NULL);
	basename = g_strdup_printf ("%s.vala", pkg);
	if (vala_compiler_vapi_directories != NULL) {
		{
			char** vapidir_collection;
			char** vapidir_it;
			int vapidir_collection_length1;
			vapidir_collection = vala_compiler_vapi_directories;
			vapidir_collection_length1 = -1;
			for (vapidir_it = vapidir_collection; *vapidir_it != NULL; vapidir_it = vapidir_it + 1) {
				const char* _tmp1;
				char* vapidir;
				_tmp1 = NULL;
				vapidir = (_tmp1 = *vapidir_it, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
				{
					char* filename;
					filename = g_build_filename (vapidir, basename, NULL);
					if (g_file_test (filename, G_FILE_TEST_EXISTS)) {
						char* _tmp0;
						_tmp0 = NULL;
						return (_tmp0 = filename, (vapidir = (g_free (vapidir), NULL)), (basename = (g_free (basename), NULL)), _tmp0);
					}
					(vapidir = (g_free (vapidir), NULL));
					(filename = (g_free (filename), NULL));
				}
			}
		}
	}
	filename = g_build_filename (PACKAGE_DATADIR, "vapi", basename, NULL);
	if (g_file_test (filename, G_FILE_TEST_EXISTS)) {
		char* _tmp2;
		_tmp2 = NULL;
		return (_tmp2 = filename, (basename = (g_free (basename), NULL)), _tmp2);
	}
	_tmp3 = NULL;
	filename = (_tmp3 = g_build_filename ("/usr/local/share/vala/vapi", basename, NULL), (filename = (g_free (filename), NULL)), _tmp3);
	if (g_file_test (filename, G_FILE_TEST_EXISTS)) {
		char* _tmp4;
		_tmp4 = NULL;
		return (_tmp4 = filename, (basename = (g_free (basename), NULL)), _tmp4);
	}
	_tmp5 = NULL;
	filename = (_tmp5 = g_build_filename ("/usr/share/vala/vapi", basename, NULL), (filename = (g_free (filename), NULL)), _tmp5);
	if (g_file_test (filename, G_FILE_TEST_EXISTS)) {
		char* _tmp6;
		_tmp6 = NULL;
		return (_tmp6 = filename, (basename = (g_free (basename), NULL)), _tmp6);
	}
	return (_tmp7 = NULL, (basename = (g_free (basename), NULL)), (filename = (g_free (filename), NULL)), _tmp7);
	(basename = (g_free (basename), NULL));
	(filename = (g_free (filename), NULL));
}


static gboolean vala_compiler_add_package (ValaCompiler* self, ValaCodeContext* context, const char* pkg)
{
	GError * inner_error;
	char* package_path;
	ValaSourceFile* _tmp2;
	char* _tmp4;
	char* _tmp3;
	char* _tmp5;
	char* deps_filename;
	gboolean _tmp9;
	g_return_val_if_fail (VALA_IS_COMPILER (self), FALSE);
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (context), FALSE);
	g_return_val_if_fail (pkg != NULL, FALSE);
	inner_error = NULL;
	if (vala_code_context_has_package (context, pkg)) {
		return TRUE;
	}
	/* ignore multiple occurences of the same package*/
	package_path = vala_compiler_get_package_path (self, pkg);
	if (package_path == NULL) {
		gboolean _tmp1;
		return (_tmp1 = FALSE, (package_path = (g_free (package_path), NULL)), _tmp1);
	}
	vala_code_context_add_package (context, pkg);
	_tmp2 = NULL;
	vala_code_context_add_source_file (context, (_tmp2 = vala_source_file_new (context, package_path, TRUE)));
	(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
	_tmp4 = NULL;
	_tmp3 = NULL;
	_tmp5 = NULL;
	deps_filename = (_tmp5 = g_build_filename ((_tmp3 = g_path_get_dirname (package_path)), (_tmp4 = g_strdup_printf ("%s.deps", pkg)), NULL), (_tmp4 = (g_free (_tmp4), NULL)), (_tmp3 = (g_free (_tmp3), NULL)), _tmp5);
	if (g_file_test (deps_filename, G_FILE_TEST_EXISTS)) {
		{
			char* deps_content;
			glong deps_len;
			deps_content = NULL;
			deps_len = 0L;
			g_file_get_contents (deps_filename, &deps_content, &deps_len, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == G_FILE_ERROR) {
					goto __catch0_g_file_error;
				}
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
			}
			{
				char** dep_collection;
				char** dep_it;
				int dep_collection_length1;
				dep_collection = g_strsplit (deps_content, "\n", 0);
				dep_collection_length1 = -1;
				for (dep_it = dep_collection; *dep_it != NULL; dep_it = dep_it + 1) {
					const char* _tmp7;
					char* dep;
					_tmp7 = NULL;
					dep = (_tmp7 = *dep_it, (_tmp7 == NULL ? NULL : g_strdup (_tmp7)));
					{
						if (g_utf8_collate (dep, "") != 0) {
							if (!vala_compiler_add_package (self, context, dep)) {
								char* _tmp6;
								_tmp6 = NULL;
								vala_report_error (NULL, (_tmp6 = g_strdup_printf ("%s, dependency of %s, not found in specified Vala API directories", dep, pkg)));
								(_tmp6 = (g_free (_tmp6), NULL));
							}
						}
						(dep = (g_free (dep), NULL));
					}
				}
				(dep_collection = (_vala_array_free (dep_collection, dep_collection_length1, ((GDestroyNotify) g_free)), NULL));
			}
			(deps_content = (g_free (deps_content), NULL));
		}
		goto __finally0;
		__catch0_g_file_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				char* _tmp8;
				_tmp8 = NULL;
				vala_report_error (NULL, (_tmp8 = g_strdup_printf ("Unable to read dependency file: %s", e->message)));
				(_tmp8 = (g_free (_tmp8), NULL));
			}
		}
		__finally0:
		;
	}
	return (_tmp9 = TRUE, (package_path = (g_free (package_path), NULL)), (deps_filename = (g_free (deps_filename), NULL)), _tmp9);
	(package_path = (g_free (package_path), NULL));
	(deps_filename = (g_free (deps_filename), NULL));
}


static gint vala_compiler_run (ValaCompiler* self)
{
	ValaCodeContext* _tmp0;
	char** _tmp9;
	ValaParser* parser;
	ValaAttributeProcessor* attributeprocessor;
	ValaSymbolResolver* resolver;
	ValaDBusBindingProvider* dbus_binding_provider;
	ValaSemanticAnalyzer* analyzer;
	ValaCodeGenerator* code_generator;
	gint _tmp19;
	g_return_val_if_fail (VALA_IS_COMPILER (self), 0);
	_tmp0 = NULL;
	self->priv->context = (_tmp0 = g_object_new (VALA_TYPE_CODE_CONTEXT, NULL), (self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL))), _tmp0);
	/* support old command line interface */
	if (!vala_compiler_ccode_only && !vala_compiler_compile_only && vala_compiler_output == NULL) {
		vala_compiler_ccode_only = TRUE;
	}
	vala_code_context_set_library (self->priv->context, vala_compiler_library);
	vala_code_context_set_memory_management (self->priv->context, !vala_compiler_disable_memory_management);
	vala_code_context_set_assert (self->priv->context, !vala_compiler_disable_assert);
	vala_code_context_set_checking (self->priv->context, vala_compiler_enable_checking);
	vala_code_context_set_ccode_only (self->priv->context, vala_compiler_ccode_only);
	vala_code_context_set_compile_only (self->priv->context, vala_compiler_compile_only);
	vala_code_context_set_output (self->priv->context, vala_compiler_output);
	vala_code_context_set_directory (self->priv->context, vala_compiler_directory);
	vala_code_context_set_debug (self->priv->context, vala_compiler_debug);
	vala_code_context_set_thread (self->priv->context, vala_compiler_thread);
	vala_code_context_set_optlevel (self->priv->context, vala_compiler_optlevel);
	vala_code_context_set_save_temps (self->priv->context, vala_compiler_save_temps);
	/* default package */
	if (!vala_compiler_add_package (self, self->priv->context, "glib-2.0")) {
		vala_report_error (NULL, "glib-2.0 not found in specified Vala API directories");
	}
	if (vala_compiler_packages != NULL) {
		char** _tmp3;
		{
			char** package_collection;
			char** package_it;
			int package_collection_length1;
			package_collection = vala_compiler_packages;
			package_collection_length1 = -1;
			for (package_it = package_collection; *package_it != NULL; package_it = package_it + 1) {
				const char* _tmp2;
				char* package;
				_tmp2 = NULL;
				package = (_tmp2 = *package_it, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
				{
					if (!vala_compiler_add_package (self, self->priv->context, package)) {
						char* _tmp1;
						_tmp1 = NULL;
						vala_report_error (NULL, (_tmp1 = g_strdup_printf ("%s not found in specified Vala API directories", package)));
						(_tmp1 = (g_free (_tmp1), NULL));
					}
					(package = (g_free (package), NULL));
				}
			}
		}
		_tmp3 = NULL;
		vala_compiler_packages = (_tmp3 = NULL, (vala_compiler_packages = (_vala_array_free (vala_compiler_packages, -1, ((GDestroyNotify) g_free)), NULL)), _tmp3);
	}
	if (vala_report_get_errors () > 0) {
		return vala_compiler_quit (self);
	}
	{
		char** source_collection;
		char** source_it;
		int source_collection_length1;
		source_collection = vala_compiler_sources;
		source_collection_length1 = -1;
		for (source_it = source_collection; *source_it != NULL; source_it = source_it + 1) {
			const char* _tmp8;
			char* source;
			_tmp8 = NULL;
			source = (_tmp8 = *source_it, (_tmp8 == NULL ? NULL : g_strdup (_tmp8)));
			{
				if (g_file_test (source, G_FILE_TEST_EXISTS)) {
					if (g_str_has_suffix (source, ".vala")) {
						ValaSourceFile* _tmp5;
						_tmp5 = NULL;
						vala_code_context_add_source_file (self->priv->context, (_tmp5 = vala_source_file_new (self->priv->context, source, FALSE)));
						(_tmp5 == NULL ? NULL : (_tmp5 = (g_object_unref (_tmp5), NULL)));
					} else {
						if (g_str_has_suffix (source, ".c")) {
							vala_code_context_add_c_source_file (self->priv->context, source);
						} else {
							char* _tmp6;
							_tmp6 = NULL;
							vala_report_error (NULL, (_tmp6 = g_strdup_printf ("%s is not a supported source file type. Only .vala and .c files are supported.", source)));
							(_tmp6 = (g_free (_tmp6), NULL));
						}
					}
				} else {
					char* _tmp7;
					_tmp7 = NULL;
					vala_report_error (NULL, (_tmp7 = g_strdup_printf ("%s not found", source)));
					(_tmp7 = (g_free (_tmp7), NULL));
				}
				(source = (g_free (source), NULL));
			}
		}
	}
	_tmp9 = NULL;
	vala_compiler_sources = (_tmp9 = NULL, (vala_compiler_sources = (_vala_array_free (vala_compiler_sources, -1, ((GDestroyNotify) g_free)), NULL)), _tmp9);
	if (vala_report_get_errors () > 0) {
		return vala_compiler_quit (self);
	}
	parser = g_object_new (VALA_TYPE_PARSER, NULL);
	vala_parser_parse (parser, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp11;
		return (_tmp11 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), _tmp11);
	}
	attributeprocessor = g_object_new (VALA_TYPE_ATTRIBUTE_PROCESSOR, NULL);
	vala_attribute_processor_process (attributeprocessor, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp12;
		return (_tmp12 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), _tmp12);
	}
	resolver = g_object_new (VALA_TYPE_SYMBOL_RESOLVER, NULL);
	vala_symbol_resolver_resolve (resolver, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp13;
		return (_tmp13 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), _tmp13);
	}
	dbus_binding_provider = vala_dbus_binding_provider_new ();
	vala_dbus_binding_provider_set_context (dbus_binding_provider, self->priv->context);
	analyzer = vala_semantic_analyzer_new (!vala_compiler_disable_memory_management);
	vala_semantic_analyzer_add_binding_provider (analyzer, VALA_BINDING_PROVIDER (dbus_binding_provider));
	vala_semantic_analyzer_analyze (analyzer, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp14;
		return (_tmp14 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), (dbus_binding_provider == NULL ? NULL : (dbus_binding_provider = (g_object_unref (dbus_binding_provider), NULL))), (analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL))), _tmp14);
	}
	if (!vala_compiler_disable_memory_management) {
		ValaMemoryManager* memory_manager;
		memory_manager = g_object_new (VALA_TYPE_MEMORY_MANAGER, NULL);
		vala_memory_manager_analyze (memory_manager, self->priv->context);
		if (vala_report_get_errors () > 0) {
			gint _tmp15;
			return (_tmp15 = vala_compiler_quit (self), (memory_manager == NULL ? NULL : (memory_manager = (g_object_unref (memory_manager), NULL))), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), (dbus_binding_provider == NULL ? NULL : (dbus_binding_provider = (g_object_unref (dbus_binding_provider), NULL))), (analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL))), _tmp15);
		}
		(memory_manager == NULL ? NULL : (memory_manager = (g_object_unref (memory_manager), NULL)));
	}
	code_generator = vala_code_generator_new (!vala_compiler_disable_memory_management);
	vala_code_generator_emit (code_generator, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp16;
		return (_tmp16 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), (dbus_binding_provider == NULL ? NULL : (dbus_binding_provider = (g_object_unref (dbus_binding_provider), NULL))), (analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL))), (code_generator == NULL ? NULL : (code_generator = (g_object_unref (code_generator), NULL))), _tmp16);
	}
	if (vala_compiler_library != NULL) {
		ValaInterfaceWriter* interface_writer;
		char* _tmp17;
		char* _tmp18;
		interface_writer = g_object_new (VALA_TYPE_INTERFACE_WRITER, NULL);
		_tmp17 = NULL;
		vala_interface_writer_write_file (interface_writer, self->priv->context, (_tmp17 = g_strdup_printf ("%s.vala", vala_compiler_library)));
		(_tmp17 = (g_free (_tmp17), NULL));
		_tmp18 = NULL;
		vala_compiler_library = (_tmp18 = NULL, (vala_compiler_library = (g_free (vala_compiler_library), NULL)), _tmp18);
		(interface_writer == NULL ? NULL : (interface_writer = (g_object_unref (interface_writer), NULL)));
	}
	if (!vala_compiler_ccode_only) {
		ValaCCodeCompiler* ccompiler;
		ccompiler = vala_ccode_compiler_new ();
		vala_ccode_compiler_compile (ccompiler, self->priv->context, vala_compiler_cc_command, vala_compiler_cc_options);
		(ccompiler == NULL ? NULL : (ccompiler = (g_object_unref (ccompiler), NULL)));
	}
	return (_tmp19 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), (dbus_binding_provider == NULL ? NULL : (dbus_binding_provider = (g_object_unref (dbus_binding_provider), NULL))), (analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL))), (code_generator == NULL ? NULL : (code_generator = (g_object_unref (code_generator), NULL))), _tmp19);
	(parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL)));
	(attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL)));
	(resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL)));
	(dbus_binding_provider == NULL ? NULL : (dbus_binding_provider = (g_object_unref (dbus_binding_provider), NULL)));
	(analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL)));
	(code_generator == NULL ? NULL : (code_generator = (g_object_unref (code_generator), NULL)));
}


static gint vala_compiler_main (int args_length1, char** args)
{
	GError * inner_error;
	ValaCompiler* compiler;
	gint _tmp3;
	inner_error = NULL;
	{
		GOptionContext* opt_context;
		opt_context = g_option_context_new ("- Vala Compiler");
		g_option_context_set_help_enabled (opt_context, TRUE);
		g_option_context_add_main_entries (opt_context, VALA_COMPILER_options, NULL);
		g_option_context_parse (opt_context, &args_length1, &args, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == G_OPTION_ERROR) {
				goto __catch1_g_option_error;
			}
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
		}
		(opt_context == NULL ? NULL : (opt_context = (g_option_context_free (opt_context), NULL)));
	}
	goto __finally1;
	__catch1_g_option_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			fprintf (stdout, "%s\n", e->message);
			fprintf (stdout, "Run '%s --help' to see a full list of available command line options.\n", args[0]);
			return 1;
		}
	}
	__finally1:
	;
	if (vala_compiler_version) {
		fprintf (stdout, "Vala %s\n", PACKAGE_VERSION);
		return 0;
	}
	if (vala_compiler_sources == NULL) {
		fprintf (stderr, "No source file specified.\n");
		return 1;
	}
	compiler = g_object_new (VALA_TYPE_COMPILER, NULL);
	return (_tmp3 = vala_compiler_run (compiler), (compiler == NULL ? NULL : (compiler = (g_object_unref (compiler), NULL))), _tmp3);
	(compiler == NULL ? NULL : (compiler = (g_object_unref (compiler), NULL)));
}


int main (int argc, char ** argv)
{
	g_type_init ();
	return vala_compiler_main (argc, argv);
}


static void vala_compiler_class_init (ValaCompilerClass * klass)
{
	vala_compiler_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaCompilerPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_compiler_dispose;
}


static void vala_compiler_init (ValaCompiler * self)
{
	self->priv = VALA_COMPILER_GET_PRIVATE (self);
}


static void vala_compiler_dispose (GObject * obj)
{
	ValaCompiler * self;
	ValaCompilerClass * klass;
	GObjectClass * parent_class;
	self = VALA_COMPILER (obj);
	(self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL)));
	klass = VALA_COMPILER_CLASS (g_type_class_peek (VALA_TYPE_COMPILER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


static GType vala_compiler_get_type (void)
{
	static GType vala_compiler_type_id = 0;
	if (G_UNLIKELY (vala_compiler_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCompilerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_compiler_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCompiler), 0, (GInstanceInitFunc) vala_compiler_init };
		vala_compiler_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaCompiler", &g_define_type_info, 0);
	}
	return vala_compiler_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func)
{
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}




