/* valainterface.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include "valainterface.h"
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <vala/valasourcereference.h>
#include <vala/valainterface.h>
#include <vala/valasymbol.h>
#include <vala/valatypeparameter.h>
#include <vala/valascope.h>
#include <vala/valatypereference.h>
#include <vala/valamethod.h>
#include <vala/valacreationmethod.h>
#include <vala/valareport.h>
#include <vala/valaformalparameter.h>
#include <vala/valafield.h>
#include <vala/valaproperty.h>
#include <vala/valasignal.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>

struct _ValaInterfacePrivate {
	gboolean _is_static;
	gboolean _declaration_only;
	GeeList* type_parameters;
	GeeList* prerequisites;
	GeeList* methods;
	GeeList* fields;
	GeeList* properties;
	GeeList* signals;
	char* cname;
	char* lower_case_csuffix;
	char* type_cname;
	char* type_id;
};
#define VALA_INTERFACE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_INTERFACE, ValaInterfacePrivate))
enum  {
	VALA_INTERFACE_DUMMY_PROPERTY,
	VALA_INTERFACE_IS_STATIC,
	VALA_INTERFACE_DECLARATION_ONLY
};
static char* vala_interface_real_get_cname (ValaDataType* base, gboolean const_type);
static char* vala_interface_real_get_lower_case_cname (ValaSymbol* base, const char* infix);
static char* vala_interface_real_get_lower_case_cprefix (ValaSymbol* base);
static char* vala_interface_real_get_upper_case_cname (ValaDataType* base, const char* infix);
static void vala_interface_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_interface_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_interface_real_is_reference_type (ValaDataType* base);
static gboolean vala_interface_real_is_reference_counting (ValaDataType* base);
static char* vala_interface_real_get_ref_function (ValaDataType* base);
static char* vala_interface_real_get_unref_function (ValaDataType* base);
static gboolean vala_interface_real_is_subtype_of (ValaDataType* base, ValaDataType* t);
static void vala_interface_process_ccode_attribute (ValaInterface* self, ValaAttribute* a);
static void vala_interface_process_dbus_interface_attribute (ValaInterface* self, ValaAttribute* a);
static char* vala_interface_real_get_marshaller_type_name (ValaDataType* base);
static char* vala_interface_real_get_get_value_function (ValaDataType* base);
static char* vala_interface_real_get_set_value_function (ValaDataType* base);
static char* vala_interface_real_get_type_id (ValaDataType* base);
static gint vala_interface_real_get_type_parameter_index (ValaDataType* base, const char* name);
static gpointer vala_interface_parent_class = NULL;
static void vala_interface_dispose (GObject * obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);


/**
 * Creates a new interface.
 *
 * @param name   type name
 * @param source reference to source code
 * @return       newly created interface
 */
ValaInterface* vala_interface_new (const char* name, ValaSourceReference* source_reference)
{
	ValaInterface * self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_INTERFACE, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Appends the specified parameter to the list of type parameters.
 *
 * @param p a type parameter
 */
void vala_interface_add_type_parameter (ValaInterface* self, ValaTypeParameter* p)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	g_return_if_fail (VALA_IS_TYPE_PARAMETER (p));
	gee_collection_add (GEE_COLLECTION (self->priv->type_parameters), p);
	p->type = VALA_DATA_TYPE (self);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) p)), VALA_SYMBOL (p));
}


/**
 * Returns a copy of the type parameter list.
 *
 * @return list of type parameters
 */
GeeCollection* vala_interface_get_type_parameters (ValaInterface* self)
{
	g_return_val_if_fail (VALA_IS_INTERFACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->type_parameters)));
}


/**
 * Adds the specified interface or class to the list of prerequisites of
 * this interface.
 *
 * @param type an interface or class reference
 */
void vala_interface_add_prerequisite (ValaInterface* self, ValaTypeReference* type)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (type));
	gee_collection_add (GEE_COLLECTION (self->priv->prerequisites), type);
}


/**
 * Prepends the specified interface or class to the list of
 * prerequisites of this interface.
 *
 * @param type an interface or class reference
 */
void vala_interface_prepend_prerequisite (ValaInterface* self, ValaTypeReference* type)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (type));
	gee_list_insert (self->priv->prerequisites, 0, type);
}


/**
 * Returns a copy of the base type list.
 *
 * @return list of base types
 */
GeeCollection* vala_interface_get_prerequisites (ValaInterface* self)
{
	g_return_val_if_fail (VALA_IS_INTERFACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->prerequisites)));
}


/**
 * Adds the specified method as a member to this interface.
 *
 * @param m a method
 */
void vala_interface_add_method (ValaInterface* self, ValaMethod* m)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	if (VALA_IS_CREATION_METHOD (m)) {
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) m)), "construction methods may only be declared within classes and structs");
		vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
		return;
	}
	if (vala_method_get_instance (m)) {
		ValaFormalParameter* _tmp1;
		ValaTypeReference* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		vala_method_set_this_parameter (m, (_tmp1 = vala_formal_parameter_new ("this", (_tmp0 = vala_type_reference_new ()), NULL)));
		(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
		(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
		vala_type_reference_set_data_type (vala_formal_parameter_get_type_reference (vala_method_get_this_parameter (m)), VALA_DATA_TYPE (self));
		vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) m)), vala_symbol_get_name (((ValaSymbol*) vala_method_get_this_parameter (m))), VALA_SYMBOL (vala_method_get_this_parameter (m)));
	}
	gee_collection_add (GEE_COLLECTION (self->priv->methods), m);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) m)), VALA_SYMBOL (m));
}


/**
 * Returns a copy of the list of methods.
 *
 * @return list of methods
 */
GeeCollection* vala_interface_get_methods (ValaInterface* self)
{
	g_return_val_if_fail (VALA_IS_INTERFACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->methods)));
}


/**
 * Adds the specified field as a member to this interface. The field
 * must be private and static.
 *
 * @param f a field
 */
void vala_interface_add_field (ValaInterface* self, ValaField* f)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	g_return_if_fail (VALA_IS_FIELD (f));
	/* non_null fields not yet supported due to initialization issues*/
	vala_type_reference_set_non_null (vala_field_get_type_reference (f), FALSE);
	gee_collection_add (GEE_COLLECTION (self->priv->fields), f);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) f)), VALA_SYMBOL (f));
}


/**
 * Returns a copy of the list of fields.
 *
 * @return list of fields
 */
GeeCollection* vala_interface_get_fields (ValaInterface* self)
{
	g_return_val_if_fail (VALA_IS_INTERFACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->fields)));
}


/**
 * Adds the specified property as a member to this interface.
 *
 * @param prop a property
 */
void vala_interface_add_property (ValaInterface* self, ValaProperty* prop)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	gee_collection_add (GEE_COLLECTION (self->priv->properties), prop);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) prop)), VALA_SYMBOL (prop));
}


/**
 * Returns a copy of the list of properties.
 *
 * @return list of properties
 */
GeeCollection* vala_interface_get_properties (ValaInterface* self)
{
	g_return_val_if_fail (VALA_IS_INTERFACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->properties)));
}


/**
 * Adds the specified signal as a member to this interface.
 *
 * @param sig a signal
 */
void vala_interface_add_signal (ValaInterface* self, ValaSignal* sig)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	g_return_if_fail (VALA_IS_SIGNAL (sig));
	gee_collection_add (GEE_COLLECTION (self->priv->signals), sig);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) sig)), VALA_SYMBOL (sig));
}


/**
 * Returns a copy of the list of signals.
 *
 * @return list of signals
 */
GeeCollection* vala_interface_get_signals (ValaInterface* self)
{
	g_return_val_if_fail (VALA_IS_INTERFACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->signals)));
}


static char* vala_interface_real_get_cname (ValaDataType* base, gboolean const_type)
{
	ValaInterface * self;
	const char* _tmp2;
	self = ((ValaInterface*) base);
	if (self->priv->cname == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->cname = (_tmp1 = g_strdup_printf ("%s%s", (_tmp0 = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol (((ValaSymbol*) self)))), vala_symbol_get_name (((ValaSymbol*) self))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp1);
		(_tmp0 = (g_free (_tmp0), NULL));
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->cname, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


/**
 * Returns the string to be prepended to the name of members of this
 * interface when used in C code.
 *
 * @return the suffix to be used in C code
 */
char* vala_interface_get_lower_case_csuffix (ValaInterface* self)
{
	const char* _tmp1;
	g_return_val_if_fail (VALA_IS_INTERFACE (self), NULL);
	if (self->priv->lower_case_csuffix == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->lower_case_csuffix = (_tmp0 = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name (((ValaSymbol*) self))), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->lower_case_csuffix, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
}


/**
 * Sets the string to be prepended to the name of members of this
 * interface when used in C code.
 *
 * @param csuffix the suffix to be used in C code
 */
void vala_interface_set_lower_case_csuffix (ValaInterface* self, const char* csuffix)
{
	char* _tmp0;
	g_return_if_fail (VALA_IS_INTERFACE (self));
	g_return_if_fail (csuffix != NULL);
	_tmp0 = NULL;
	self->priv->lower_case_csuffix = (_tmp0 = g_strdup (csuffix), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), _tmp0);
}


static char* vala_interface_real_get_lower_case_cname (ValaSymbol* base, const char* infix)
{
	ValaInterface * self;
	char* _tmp1;
	char* _tmp0;
	char* _tmp2;
	self = ((ValaInterface*) base);
	if (infix == NULL) {
		infix = "";
	}
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = g_strdup_printf ("%s%s%s", (_tmp0 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (((ValaSymbol*) self)))), infix, (_tmp1 = vala_interface_get_lower_case_csuffix (self))), (_tmp1 = (g_free (_tmp1), NULL)), (_tmp0 = (g_free (_tmp0), NULL)), _tmp2);
}


static char* vala_interface_real_get_lower_case_cprefix (ValaSymbol* base)
{
	ValaInterface * self;
	char* _tmp0;
	char* _tmp1;
	self = ((ValaInterface*) base);
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_strdup_printf ("%s_", (_tmp0 = vala_symbol_get_lower_case_cname (VALA_SYMBOL (self), NULL))), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
}


static char* vala_interface_real_get_upper_case_cname (ValaDataType* base, const char* infix)
{
	ValaInterface * self;
	char* _tmp0;
	char* _tmp1;
	self = ((ValaInterface*) base);
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_utf8_strup ((_tmp0 = vala_symbol_get_lower_case_cname (VALA_SYMBOL (self), infix)), -1), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
}


static void vala_interface_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaInterface * self;
	self = ((ValaInterface*) base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_interface (visitor, self);
}


static void vala_interface_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaInterface * self;
	self = ((ValaInterface*) base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GeeList* type_collection;
		GeeIterator* type_it;
		type_collection = self->priv->prerequisites;
		type_it = gee_iterable_iterator (GEE_ITERABLE (type_collection));
		while (gee_iterator_next (type_it)) {
			ValaTypeReference* type;
			type = gee_iterator_get (type_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (type), visitor);
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
	}
	{
		GeeList* p_collection;
		GeeIterator* p_it;
		p_collection = self->priv->type_parameters;
		p_it = gee_iterable_iterator (GEE_ITERABLE (p_collection));
		while (gee_iterator_next (p_it)) {
			ValaTypeParameter* p;
			p = gee_iterator_get (p_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (p), visitor);
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
		(p_it == NULL ? NULL : (p_it = (g_object_unref (p_it), NULL)));
	}
	{
		GeeList* m_collection;
		GeeIterator* m_it;
		m_collection = self->priv->methods;
		m_it = gee_iterable_iterator (GEE_ITERABLE (m_collection));
		while (gee_iterator_next (m_it)) {
			ValaMethod* m;
			m = gee_iterator_get (m_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (m), visitor);
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
			}
		}
		(m_it == NULL ? NULL : (m_it = (g_object_unref (m_it), NULL)));
	}
	{
		GeeList* f_collection;
		GeeIterator* f_it;
		f_collection = self->priv->fields;
		f_it = gee_iterable_iterator (GEE_ITERABLE (f_collection));
		while (gee_iterator_next (f_it)) {
			ValaField* f;
			f = gee_iterator_get (f_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (f), visitor);
				(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
			}
		}
		(f_it == NULL ? NULL : (f_it = (g_object_unref (f_it), NULL)));
	}
	{
		GeeList* prop_collection;
		GeeIterator* prop_it;
		prop_collection = self->priv->properties;
		prop_it = gee_iterable_iterator (GEE_ITERABLE (prop_collection));
		while (gee_iterator_next (prop_it)) {
			ValaProperty* prop;
			prop = gee_iterator_get (prop_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (prop), visitor);
				(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
			}
		}
		(prop_it == NULL ? NULL : (prop_it = (g_object_unref (prop_it), NULL)));
	}
	{
		GeeList* sig_collection;
		GeeIterator* sig_it;
		sig_collection = self->priv->signals;
		sig_it = gee_iterable_iterator (GEE_ITERABLE (sig_collection));
		while (gee_iterator_next (sig_it)) {
			ValaSignal* sig;
			sig = gee_iterator_get (sig_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (sig), visitor);
				(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
			}
		}
		(sig_it == NULL ? NULL : (sig_it = (g_object_unref (sig_it), NULL)));
	}
}


static gboolean vala_interface_real_is_reference_type (ValaDataType* base)
{
	ValaInterface * self;
	self = ((ValaInterface*) base);
	return TRUE;
}


static gboolean vala_interface_real_is_reference_counting (ValaDataType* base)
{
	ValaInterface * self;
	self = ((ValaInterface*) base);
	return TRUE;
}


static char* vala_interface_real_get_ref_function (ValaDataType* base)
{
	ValaInterface * self;
	self = ((ValaInterface*) base);
	return g_strdup ("g_object_ref");
}


static char* vala_interface_real_get_unref_function (ValaDataType* base)
{
	ValaInterface * self;
	self = ((ValaInterface*) base);
	return g_strdup ("g_object_unref");
}


static gboolean vala_interface_real_is_subtype_of (ValaDataType* base, ValaDataType* t)
{
	ValaInterface * self;
	self = ((ValaInterface*) base);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (t), FALSE);
	{
		GeeList* prerequisite_collection;
		GeeIterator* prerequisite_it;
		prerequisite_collection = self->priv->prerequisites;
		prerequisite_it = gee_iterable_iterator (GEE_ITERABLE (prerequisite_collection));
		while (gee_iterator_next (prerequisite_it)) {
			ValaTypeReference* prerequisite;
			prerequisite = gee_iterator_get (prerequisite_it);
			{
				if (vala_type_reference_get_data_type (prerequisite) == t || vala_data_type_is_subtype_of (vala_type_reference_get_data_type (prerequisite), t)) {
					gboolean _tmp0;
					return (_tmp0 = TRUE, (prerequisite == NULL ? NULL : (prerequisite = (g_object_unref (prerequisite), NULL))), (prerequisite_it == NULL ? NULL : (prerequisite_it = (g_object_unref (prerequisite_it), NULL))), _tmp0);
				}
				(prerequisite == NULL ? NULL : (prerequisite = (g_object_unref (prerequisite), NULL)));
			}
		}
		(prerequisite_it == NULL ? NULL : (prerequisite_it = (g_object_unref (prerequisite_it), NULL)));
	}
	return FALSE;
}


static void vala_interface_process_ccode_attribute (ValaInterface* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "type_cname")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_interface_set_type_cname (self, (_tmp0 = vala_attribute_get_string (a, "type_cname")));
		(_tmp0 = (g_free (_tmp0), NULL));
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			char** filename_it;
			int filename_collection_length1;
			filename_collection = g_strsplit (val, ",", 0);
			filename_collection_length1 = -1;
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* _tmp1;
				char* filename;
				_tmp1 = NULL;
				filename = (_tmp1 = *filename_it, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
				{
					vala_data_type_add_cheader_filename (VALA_DATA_TYPE (self), filename);
					(filename = (g_free (filename), NULL));
				}
			}
			(filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, ((GDestroyNotify) g_free)), NULL));
		}
		(val = (g_free (val), NULL));
	}
}


static void vala_interface_process_dbus_interface_attribute (ValaInterface* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_interface_get_declaration_only (self)) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->cname = (_tmp0 = g_strdup ("DBusGProxy"), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp0);
	}
}


/**
 * Process all associated attributes.
 */
void vala_interface_process_attributes (ValaInterface* self)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = a_it->data, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "CCode") == 0) {
					vala_interface_process_ccode_attribute (self, a);
				} else {
					if (g_utf8_collate (vala_attribute_get_name (a), "DBusInterface") == 0) {
						vala_interface_process_dbus_interface_attribute (self, a);
					}
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


/**
 * Returns the name of the type struct as it is used in C code.
 *
 * @return the type struct name to be used in C code
 */
char* vala_interface_get_type_cname (ValaInterface* self)
{
	const char* _tmp2;
	g_return_val_if_fail (VALA_IS_INTERFACE (self), NULL);
	if (self->priv->type_cname == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->type_cname = (_tmp1 = g_strdup_printf ("%sIface", (_tmp0 = vala_data_type_get_cname (VALA_DATA_TYPE (self), FALSE))), (self->priv->type_cname = (g_free (self->priv->type_cname), NULL)), _tmp1);
		(_tmp0 = (g_free (_tmp0), NULL));
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->type_cname, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


/**
 * Sets the name of the type struct as it is used in C code.
 *
 * @param type_cname the type struct name to be used in C code
 */
void vala_interface_set_type_cname (ValaInterface* self, const char* type_cname)
{
	char* _tmp0;
	g_return_if_fail (VALA_IS_INTERFACE (self));
	g_return_if_fail (type_cname != NULL);
	_tmp0 = NULL;
	self->priv->type_cname = (_tmp0 = g_strdup (type_cname), (self->priv->type_cname = (g_free (self->priv->type_cname), NULL)), _tmp0);
}


static char* vala_interface_real_get_marshaller_type_name (ValaDataType* base)
{
	ValaInterface * self;
	self = ((ValaInterface*) base);
	return g_strdup ("OBJECT");
}


static char* vala_interface_real_get_get_value_function (ValaDataType* base)
{
	ValaInterface * self;
	self = ((ValaInterface*) base);
	return g_strdup ("g_value_get_object");
}


static char* vala_interface_real_get_set_value_function (ValaDataType* base)
{
	ValaInterface * self;
	self = ((ValaInterface*) base);
	return g_strdup ("g_value_set_object");
}


static char* vala_interface_real_get_type_id (ValaDataType* base)
{
	ValaInterface * self;
	const char* _tmp1;
	self = ((ValaInterface*) base);
	if (self->priv->type_id == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->type_id = (_tmp0 = vala_data_type_get_upper_case_cname (VALA_DATA_TYPE (self), "TYPE_"), (self->priv->type_id = (g_free (self->priv->type_id), NULL)), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->type_id, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
}


static gint vala_interface_real_get_type_parameter_index (ValaDataType* base, const char* name)
{
	ValaInterface * self;
	gint i;
	self = ((ValaInterface*) base);
	g_return_val_if_fail (name != NULL, 0);
	i = 0;
	{
		GeeList* parameter_collection;
		GeeIterator* parameter_it;
		parameter_collection = self->priv->type_parameters;
		parameter_it = gee_iterable_iterator (GEE_ITERABLE (parameter_collection));
		while (gee_iterator_next (parameter_it)) {
			ValaTypeParameter* parameter;
			parameter = gee_iterator_get (parameter_it);
			{
				if (g_utf8_collate (vala_symbol_get_name (((ValaSymbol*) parameter)), name) == 0) {
					gint _tmp0;
					return (_tmp0 = i, (parameter == NULL ? NULL : (parameter = (g_object_unref (parameter), NULL))), (parameter_it == NULL ? NULL : (parameter_it = (g_object_unref (parameter_it), NULL))), _tmp0);
				}
				i++;
				(parameter == NULL ? NULL : (parameter = (g_object_unref (parameter), NULL)));
			}
		}
		(parameter_it == NULL ? NULL : (parameter_it = (g_object_unref (parameter_it), NULL)));
	}
	return -1;
}


gboolean vala_interface_get_is_static (ValaInterface* self)
{
	g_return_val_if_fail (VALA_IS_INTERFACE (self), FALSE);
	return self->priv->_is_static;
}


void vala_interface_set_is_static (ValaInterface* self, gboolean value)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	self->priv->_is_static = value;
}


gboolean vala_interface_get_declaration_only (ValaInterface* self)
{
	g_return_val_if_fail (VALA_IS_INTERFACE (self), FALSE);
	return self->priv->_declaration_only;
}


void vala_interface_set_declaration_only (ValaInterface* self, gboolean value)
{
	g_return_if_fail (VALA_IS_INTERFACE (self));
	self->priv->_declaration_only = value;
}


static void vala_interface_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaInterface * self;
	self = VALA_INTERFACE (object);
	switch (property_id) {
		case VALA_INTERFACE_IS_STATIC:
		g_value_set_boolean (value, vala_interface_get_is_static (self));
		break;
		case VALA_INTERFACE_DECLARATION_ONLY:
		g_value_set_boolean (value, vala_interface_get_declaration_only (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_interface_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaInterface * self;
	self = VALA_INTERFACE (object);
	switch (property_id) {
		case VALA_INTERFACE_IS_STATIC:
		vala_interface_set_is_static (self, g_value_get_boolean (value));
		break;
		case VALA_INTERFACE_DECLARATION_ONLY:
		vala_interface_set_declaration_only (self, g_value_get_boolean (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_interface_class_init (ValaInterfaceClass * klass)
{
	vala_interface_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaInterfacePrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_interface_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_interface_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_interface_dispose;
	VALA_DATA_TYPE_CLASS (klass)->get_cname = vala_interface_real_get_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cname = vala_interface_real_get_lower_case_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_interface_real_get_lower_case_cprefix;
	VALA_DATA_TYPE_CLASS (klass)->get_upper_case_cname = vala_interface_real_get_upper_case_cname;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_interface_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_interface_real_accept_children;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_type = vala_interface_real_is_reference_type;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_counting = vala_interface_real_is_reference_counting;
	VALA_DATA_TYPE_CLASS (klass)->get_ref_function = vala_interface_real_get_ref_function;
	VALA_DATA_TYPE_CLASS (klass)->get_unref_function = vala_interface_real_get_unref_function;
	VALA_DATA_TYPE_CLASS (klass)->is_subtype_of = vala_interface_real_is_subtype_of;
	VALA_DATA_TYPE_CLASS (klass)->get_marshaller_type_name = vala_interface_real_get_marshaller_type_name;
	VALA_DATA_TYPE_CLASS (klass)->get_get_value_function = vala_interface_real_get_get_value_function;
	VALA_DATA_TYPE_CLASS (klass)->get_set_value_function = vala_interface_real_get_set_value_function;
	VALA_DATA_TYPE_CLASS (klass)->get_type_id = vala_interface_real_get_type_id;
	VALA_DATA_TYPE_CLASS (klass)->get_type_parameter_index = vala_interface_real_get_type_parameter_index;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_INTERFACE_IS_STATIC, g_param_spec_boolean ("is-static", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_INTERFACE_DECLARATION_ONLY, g_param_spec_boolean ("declaration-only", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_interface_init (ValaInterface * self)
{
	self->priv = VALA_INTERFACE_GET_PRIVATE (self);
	self->priv->type_parameters = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->prerequisites = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->methods = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->fields = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->properties = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->signals = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
}


static void vala_interface_dispose (GObject * obj)
{
	ValaInterface * self;
	ValaInterfaceClass * klass;
	GObjectClass * parent_class;
	self = VALA_INTERFACE (obj);
	(self->priv->type_parameters == NULL ? NULL : (self->priv->type_parameters = (g_object_unref (self->priv->type_parameters), NULL)));
	(self->priv->prerequisites == NULL ? NULL : (self->priv->prerequisites = (g_object_unref (self->priv->prerequisites), NULL)));
	(self->priv->methods == NULL ? NULL : (self->priv->methods = (g_object_unref (self->priv->methods), NULL)));
	(self->priv->fields == NULL ? NULL : (self->priv->fields = (g_object_unref (self->priv->fields), NULL)));
	(self->priv->properties == NULL ? NULL : (self->priv->properties = (g_object_unref (self->priv->properties), NULL)));
	(self->priv->signals == NULL ? NULL : (self->priv->signals = (g_object_unref (self->priv->signals), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	(self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL));
	(self->priv->type_cname = (g_free (self->priv->type_cname), NULL));
	(self->priv->type_id = (g_free (self->priv->type_id), NULL));
	klass = VALA_INTERFACE_CLASS (g_type_class_peek (VALA_TYPE_INTERFACE));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_interface_get_type (void)
{
	static GType vala_interface_type_id = 0;
	if (G_UNLIKELY (vala_interface_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaInterfaceClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_interface_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaInterface), 0, (GInstanceInitFunc) vala_interface_init };
		vala_interface_type_id = g_type_register_static (VALA_TYPE_DATA_TYPE, "ValaInterface", &g_define_type_info, 0);
	}
	return vala_interface_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func)
{
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}




