/* valasourcefile.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include "valasourcefile.h"
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <vala/valasourcefilecycle.h>
#include <vala/valacodecontext.h>
#include <vala/valanamespacereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasymbol.h>
#include <vala/valadatatype.h>
#include <vala/valamethod.h>
#include <vala/valafield.h>
#include <vala/valaproperty.h>
#include <vala/valaconstant.h>
#include <vala/valaformalparameter.h>
#include <vala/valatypereference.h>
#include <vala/valasourcereference.h>

struct _ValaSourceFilePrivate {
	char* _filename;
	char* _comment;
	gboolean _pkg;
	ValaSourceFileCycle* _cycle;
	gboolean _is_cycle_head;
	gint _mark;
	ValaCodeContext* _context;
	GeeList* using_directives;
	GeeList* nodes;
	char* cheader_filename;
	char* csource_filename;
	char* cinclude_filename;
	GeeList* header_external_includes;
	GeeList* header_internal_includes;
	GeeList* source_external_includes;
	GeeList* source_internal_includes;
	GeeList* header_internal_full_dependencies;
	GeeList* header_internal_dependencies;
};
#define VALA_SOURCE_FILE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SOURCE_FILE, ValaSourceFilePrivate))
enum  {
	VALA_SOURCE_FILE_DUMMY_PROPERTY,
	VALA_SOURCE_FILE_FILENAME,
	VALA_SOURCE_FILE_COMMENT,
	VALA_SOURCE_FILE_PKG,
	VALA_SOURCE_FILE_CYCLE,
	VALA_SOURCE_FILE_IS_CYCLE_HEAD,
	VALA_SOURCE_FILE_MARK,
	VALA_SOURCE_FILE_CONTEXT
};
static gpointer vala_source_file_parent_class = NULL;
static void vala_source_file_dispose (GObject * obj);


/**
 * Creates a new source file.
 *
 * @param filename source file name
 * @param pkg      true if this is a VAPI package file
 * @return         newly created source file
 */
ValaSourceFile* vala_source_file_new (ValaCodeContext* context, const char* filename, gboolean pkg)
{
	ValaSourceFile * self;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (context), NULL);
	g_return_val_if_fail (filename != NULL, NULL);
	self = g_object_newv (VALA_TYPE_SOURCE_FILE, 0, NULL);
	vala_source_file_set_context (self, context);
	vala_source_file_set_filename (self, filename);
	vala_source_file_set_pkg (self, pkg);
	return self;
}


/**
 * Adds a new using directive with the specified namespace.
 *
 * @param ns reference to namespace
 */
void vala_source_file_add_using_directive (ValaSourceFile* self, ValaNamespaceReference* ns)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	g_return_if_fail (VALA_IS_NAMESPACE_REFERENCE (ns));
	gee_collection_add (GEE_COLLECTION (self->priv->using_directives), ns);
}


/**
 * Returns a copy of the list of using directives.
 *
 * @return using directive list
 */
GeeCollection* vala_source_file_get_using_directives (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->using_directives)));
}


/**
 * Adds the specified code node to this source file.
 *
 * @param node a code node
 */
void vala_source_file_add_node (ValaSourceFile* self, ValaCodeNode* node)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	g_return_if_fail (VALA_IS_CODE_NODE (node));
	gee_collection_add (GEE_COLLECTION (self->priv->nodes), node);
}


/**
 * Returns a copy of the list of code nodes.
 *
 * @return code node list
 */
GeeCollection* vala_source_file_get_nodes (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->nodes)));
}


void vala_source_file_accept (ValaSourceFile* self, ValaCodeVisitor* visitor)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_source_file (visitor, self);
}


void vala_source_file_accept_children (ValaSourceFile* self, ValaCodeVisitor* visitor)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GeeList* ns_ref_collection;
		GeeIterator* ns_ref_it;
		ns_ref_collection = self->priv->using_directives;
		ns_ref_it = gee_iterable_iterator (GEE_ITERABLE (ns_ref_collection));
		while (gee_iterator_next (ns_ref_it)) {
			ValaNamespaceReference* ns_ref;
			ns_ref = gee_iterator_get (ns_ref_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (ns_ref), visitor);
				(ns_ref == NULL ? NULL : (ns_ref = (g_object_unref (ns_ref), NULL)));
			}
		}
		(ns_ref_it == NULL ? NULL : (ns_ref_it = (g_object_unref (ns_ref_it), NULL)));
	}
	{
		GeeList* node_collection;
		GeeIterator* node_it;
		node_collection = self->priv->nodes;
		node_it = gee_iterable_iterator (GEE_ITERABLE (node_collection));
		while (gee_iterator_next (node_it)) {
			ValaCodeNode* node;
			node = gee_iterator_get (node_it);
			{
				vala_code_node_accept (node, visitor);
				(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
			}
		}
		(node_it == NULL ? NULL : (node_it = (g_object_unref (node_it), NULL)));
	}
}


/**
 * Returns the filename to use when generating C header files.
 *
 * @return generated C header filename
 */
char* vala_source_file_get_cheader_filename (ValaSourceFile* self)
{
	const char* _tmp3;
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	if (self->priv->cheader_filename == NULL) {
		char* basename;
		char* _tmp0;
		basename = g_strndup (vala_source_file_get_filename (self), ((guint) (g_utf8_strlen (vala_source_file_get_filename (self), -1) - g_utf8_strlen (".vala", -1))));
		_tmp0 = NULL;
		basename = (_tmp0 = g_path_get_basename (basename), (basename = (g_free (basename), NULL)), _tmp0);
		if (vala_code_context_get_directory (vala_source_file_get_context (self)) != NULL && g_utf8_collate (vala_code_context_get_directory (vala_source_file_get_context (self)), "") != 0) {
			char* _tmp1;
			_tmp1 = NULL;
			self->priv->cheader_filename = (_tmp1 = g_strdup_printf ("%s/%s.h", vala_code_context_get_directory (vala_source_file_get_context (self)), basename), (self->priv->cheader_filename = (g_free (self->priv->cheader_filename), NULL)), _tmp1);
		} else {
			char* _tmp2;
			_tmp2 = NULL;
			self->priv->cheader_filename = (_tmp2 = g_strdup_printf ("%s.h", basename), (self->priv->cheader_filename = (g_free (self->priv->cheader_filename), NULL)), _tmp2);
		}
		(basename = (g_free (basename), NULL));
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->cheader_filename, (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
}


/**
 * Returns the filename to use when generating C source files.
 *
 * @return generated C source filename
 */
char* vala_source_file_get_csource_filename (ValaSourceFile* self)
{
	const char* _tmp3;
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	if (self->priv->csource_filename == NULL) {
		char* basename;
		char* _tmp0;
		basename = g_strndup (vala_source_file_get_filename (self), ((guint) (g_utf8_strlen (vala_source_file_get_filename (self), -1) - g_utf8_strlen (".vala", -1))));
		_tmp0 = NULL;
		basename = (_tmp0 = g_path_get_basename (basename), (basename = (g_free (basename), NULL)), _tmp0);
		if (vala_code_context_get_directory (vala_source_file_get_context (self)) != NULL && g_utf8_collate (vala_code_context_get_directory (vala_source_file_get_context (self)), "") != 0) {
			char* _tmp1;
			_tmp1 = NULL;
			self->priv->csource_filename = (_tmp1 = g_strdup_printf ("%s/%s.c", vala_code_context_get_directory (vala_source_file_get_context (self)), basename), (self->priv->csource_filename = (g_free (self->priv->csource_filename), NULL)), _tmp1);
		} else {
			char* _tmp2;
			_tmp2 = NULL;
			self->priv->csource_filename = (_tmp2 = g_strdup_printf ("%s.c", basename), (self->priv->csource_filename = (g_free (self->priv->csource_filename), NULL)), _tmp2);
		}
		(basename = (g_free (basename), NULL));
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->csource_filename, (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
}


/**
 * Returns the filename to use when including the generated C header
 * file.
 *
 * @return C header filename to include
 */
char* vala_source_file_get_cinclude_filename (ValaSourceFile* self)
{
	const char* _tmp3;
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	if (self->priv->cinclude_filename == NULL) {
		char* basename;
		char* _tmp0;
		basename = g_strndup (vala_source_file_get_filename (self), ((guint) (g_utf8_strlen (vala_source_file_get_filename (self), -1) - g_utf8_strlen (".vala", -1))));
		_tmp0 = NULL;
		basename = (_tmp0 = g_path_get_basename (basename), (basename = (g_free (basename), NULL)), _tmp0);
		if (vala_code_context_get_library (vala_source_file_get_context (self)) != NULL) {
			char* _tmp1;
			_tmp1 = NULL;
			self->priv->cinclude_filename = (_tmp1 = g_strdup_printf ("%s/%s.h", vala_code_context_get_library (vala_source_file_get_context (self)), basename), (self->priv->cinclude_filename = (g_free (self->priv->cinclude_filename), NULL)), _tmp1);
		} else {
			char* _tmp2;
			_tmp2 = NULL;
			self->priv->cinclude_filename = (_tmp2 = g_strdup_printf ("%s.h", basename), (self->priv->cinclude_filename = (g_free (self->priv->cinclude_filename), NULL)), _tmp2);
		}
		(basename = (g_free (basename), NULL));
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->cinclude_filename, (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
}


/**
 * Adds the specified symbol to the list of symbols code in this source
 * file depends on.
 *
 * @param sym      a symbol
 * @param dep_type type of dependency
 */
void vala_source_file_add_symbol_dependency (ValaSourceFile* self, ValaSymbol* sym, ValaSourceFileDependencyType dep_type)
{
	ValaSymbol* s;
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	g_return_if_fail (VALA_IS_SYMBOL (sym));
	if (vala_source_file_get_pkg (self)) {
		return;
	}
	s = NULL;
	if (VALA_IS_DATA_TYPE (sym) || VALA_IS_METHOD (sym) || VALA_IS_FIELD (sym) || VALA_IS_PROPERTY (sym) || VALA_IS_CONSTANT (sym)) {
		ValaSymbol* _tmp0;
		_tmp0 = NULL;
		s = (_tmp0 = g_object_ref (sym), (s == NULL ? NULL : (s = (g_object_unref (s), NULL))), _tmp0);
	} else {
		if (VALA_IS_FORMAL_PARAMETER (sym)) {
			ValaFormalParameter* _tmp1;
			ValaFormalParameter* fp;
			ValaSymbol* _tmp3;
			ValaDataType* _tmp2;
			_tmp1 = NULL;
			fp = (_tmp1 = VALA_FORMAL_PARAMETER (sym), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
			_tmp3 = NULL;
			_tmp2 = NULL;
			s = (_tmp3 = VALA_SYMBOL ((_tmp2 = vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (fp)), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)))), (s == NULL ? NULL : (s = (g_object_unref (s), NULL))), _tmp3);
			if (s == NULL) {
				(fp == NULL ? NULL : (fp = (g_object_unref (fp), NULL)));
				(s == NULL ? NULL : (s = (g_object_unref (s), NULL)));
				return;
			}
			(fp == NULL ? NULL : (fp = (g_object_unref (fp), NULL)));
		} else {
			(s == NULL ? NULL : (s = (g_object_unref (s), NULL)));
			return;
		}
	}
	/* generic type parameter */
	if (dep_type == VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE) {
		if (vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (((ValaCodeNode*) s))))) {
			{
				GeeCollection* fn_collection;
				GeeIterator* fn_it;
				fn_collection = vala_symbol_get_cheader_filenames (s);
				fn_it = gee_iterable_iterator (GEE_ITERABLE (fn_collection));
				while (gee_iterator_next (fn_it)) {
					char* fn;
					fn = gee_iterator_get (fn_it);
					{
						gee_collection_add (GEE_COLLECTION (self->priv->source_external_includes), fn);
						(fn = (g_free (fn), NULL));
					}
				}
				(fn_collection == NULL ? NULL : (fn_collection = (g_object_unref (fn_collection), NULL)));
				(fn_it == NULL ? NULL : (fn_it = (g_object_unref (fn_it), NULL)));
			}
		} else {
			{
				GeeCollection* fn_collection;
				GeeIterator* fn_it;
				fn_collection = vala_symbol_get_cheader_filenames (s);
				fn_it = gee_iterable_iterator (GEE_ITERABLE (fn_collection));
				while (gee_iterator_next (fn_it)) {
					char* fn;
					fn = gee_iterator_get (fn_it);
					{
						gee_collection_add (GEE_COLLECTION (self->priv->source_internal_includes), fn);
						(fn = (g_free (fn), NULL));
					}
				}
				(fn_collection == NULL ? NULL : (fn_collection = (g_object_unref (fn_collection), NULL)));
				(fn_it == NULL ? NULL : (fn_it = (g_object_unref (fn_it), NULL)));
			}
		}
		(s == NULL ? NULL : (s = (g_object_unref (s), NULL)));
		return;
	}
	if (vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (((ValaCodeNode*) s))))) {
		{
			GeeCollection* fn_collection;
			GeeIterator* fn_it;
			fn_collection = vala_symbol_get_cheader_filenames (s);
			fn_it = gee_iterable_iterator (GEE_ITERABLE (fn_collection));
			while (gee_iterator_next (fn_it)) {
				char* fn;
				fn = gee_iterator_get (fn_it);
				{
					/* external package */
					gee_collection_add (GEE_COLLECTION (self->priv->header_external_includes), fn);
					(fn = (g_free (fn), NULL));
				}
			}
			(fn_collection == NULL ? NULL : (fn_collection = (g_object_unref (fn_collection), NULL)));
			(fn_it == NULL ? NULL : (fn_it = (g_object_unref (fn_it), NULL)));
		}
		(s == NULL ? NULL : (s = (g_object_unref (s), NULL)));
		return;
	}
	if (dep_type == VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL || (VALA_IS_DATA_TYPE (s) && !vala_data_type_is_reference_type ((VALA_DATA_TYPE (s))))) {
		{
			GeeCollection* fn_collection;
			GeeIterator* fn_it;
			fn_collection = vala_symbol_get_cheader_filenames (s);
			fn_it = gee_iterable_iterator (GEE_ITERABLE (fn_collection));
			while (gee_iterator_next (fn_it)) {
				char* fn;
				fn = gee_iterator_get (fn_it);
				{
					gee_collection_add (GEE_COLLECTION (self->priv->header_internal_includes), fn);
					(fn = (g_free (fn), NULL));
				}
			}
			(fn_collection == NULL ? NULL : (fn_collection = (g_object_unref (fn_collection), NULL)));
			(fn_it == NULL ? NULL : (fn_it = (g_object_unref (fn_it), NULL)));
		}
		gee_collection_add (GEE_COLLECTION (self->priv->header_internal_full_dependencies), vala_source_reference_get_file (vala_code_node_get_source_reference (((ValaCodeNode*) s))));
	}
	gee_collection_add (GEE_COLLECTION (self->priv->header_internal_dependencies), vala_source_reference_get_file (vala_code_node_get_source_reference (((ValaCodeNode*) s))));
	(s == NULL ? NULL : (s = (g_object_unref (s), NULL)));
}


/**
 * Returns the list of external includes the generated C header file
 * requires.
 *
 * @return external include list for C header file
 */
GeeCollection* vala_source_file_get_header_external_includes (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_strdup, g_free, GEE_COLLECTION (self->priv->header_external_includes)));
}


/**
 * Adds the specified filename to the list of package-internal includes
 * the generated C header file requires.
 *
 * @param include internal include for C header file
 */
void vala_source_file_add_header_internal_include (ValaSourceFile* self, const char* include)
{
	char* _tmp0;
	gboolean _tmp1;
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	g_return_if_fail (include != NULL);
	/* skip includes to self */
	_tmp0 = NULL;
	if ((_tmp1 = g_utf8_collate (include, (_tmp0 = vala_source_file_get_cinclude_filename (self))) != 0, (_tmp0 = (g_free (_tmp0), NULL)), _tmp1)) {
		gee_collection_add (GEE_COLLECTION (self->priv->header_internal_includes), include);
	}
}


/**
 * Returns the list of package-internal includes the generated C header
 * file requires.
 *
 * @return internal include list for C header file
 */
GeeCollection* vala_source_file_get_header_internal_includes (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_strdup, g_free, GEE_COLLECTION (self->priv->header_internal_includes)));
}


/**
 * Returns the list of external includes the generated C source file
 * requires.
 *
 * @return include list for C source file
 */
GeeCollection* vala_source_file_get_source_external_includes (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_strdup, g_free, GEE_COLLECTION (self->priv->source_external_includes)));
}


/**
 * Returns the list of package-internal includes the generated C source
 * file requires.
 *
 * @return include list for C source file
 */
GeeCollection* vala_source_file_get_source_internal_includes (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_strdup, g_free, GEE_COLLECTION (self->priv->source_internal_includes)));
}


/**
 * Returns the list of source files the generated C header file requires
 * definitely.
 *
 * @return definite source file dependencies
 */
GeeCollection* vala_source_file_get_header_internal_full_dependencies (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (NULL, NULL, GEE_COLLECTION (self->priv->header_internal_full_dependencies)));
}


/**
 * Returns the list of source files the generated C header file loosely
 * depends on.
 *
 * @return loose source file dependencies
 */
GeeCollection* vala_source_file_get_header_internal_dependencies (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (NULL, NULL, GEE_COLLECTION (self->priv->header_internal_dependencies)));
}


char* vala_source_file_get_filename (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return self->priv->_filename;
}


void vala_source_file_set_filename (ValaSourceFile* self, const char* value)
{
	char* _tmp1;
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	_tmp1 = NULL;
	self->priv->_filename = (_tmp1 = g_strdup (value), (self->priv->_filename = (g_free (self->priv->_filename), NULL)), _tmp1);
}


char* vala_source_file_get_comment (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return self->priv->_comment;
}


void vala_source_file_set_comment (ValaSourceFile* self, const char* value)
{
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_comment = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_comment = (g_free (self->priv->_comment), NULL)), _tmp2);
}


gboolean vala_source_file_get_pkg (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), FALSE);
	return self->priv->_pkg;
}


void vala_source_file_set_pkg (ValaSourceFile* self, gboolean value)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	self->priv->_pkg = value;
}


ValaSourceFileCycle* vala_source_file_get_cycle (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return self->priv->_cycle;
}


void vala_source_file_set_cycle (ValaSourceFile* self, ValaSourceFileCycle* value)
{
	ValaSourceFileCycle* _tmp2;
	ValaSourceFileCycle* _tmp1;
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_cycle = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_cycle == NULL ? NULL : (self->priv->_cycle = (g_object_unref (self->priv->_cycle), NULL))), _tmp2);
}


gboolean vala_source_file_get_is_cycle_head (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), FALSE);
	return self->priv->_is_cycle_head;
}


void vala_source_file_set_is_cycle_head (ValaSourceFile* self, gboolean value)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	self->priv->_is_cycle_head = value;
}


gint vala_source_file_get_mark (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), 0);
	return self->priv->_mark;
}


void vala_source_file_set_mark (ValaSourceFile* self, gint value)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	self->priv->_mark = value;
}


ValaCodeContext* vala_source_file_get_context (ValaSourceFile* self)
{
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return self->priv->_context;
}


void vala_source_file_set_context (ValaSourceFile* self, ValaCodeContext* value)
{
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	self->priv->_context = value;
}


static void vala_source_file_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaSourceFile * self;
	self = VALA_SOURCE_FILE (object);
	switch (property_id) {
		case VALA_SOURCE_FILE_FILENAME:
		g_value_set_string (value, vala_source_file_get_filename (self));
		break;
		case VALA_SOURCE_FILE_COMMENT:
		g_value_set_string (value, vala_source_file_get_comment (self));
		break;
		case VALA_SOURCE_FILE_PKG:
		g_value_set_boolean (value, vala_source_file_get_pkg (self));
		break;
		case VALA_SOURCE_FILE_CYCLE:
		g_value_set_object (value, vala_source_file_get_cycle (self));
		break;
		case VALA_SOURCE_FILE_IS_CYCLE_HEAD:
		g_value_set_boolean (value, vala_source_file_get_is_cycle_head (self));
		break;
		case VALA_SOURCE_FILE_MARK:
		g_value_set_int (value, vala_source_file_get_mark (self));
		break;
		case VALA_SOURCE_FILE_CONTEXT:
		g_value_set_object (value, vala_source_file_get_context (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_source_file_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaSourceFile * self;
	self = VALA_SOURCE_FILE (object);
	switch (property_id) {
		case VALA_SOURCE_FILE_FILENAME:
		vala_source_file_set_filename (self, g_value_get_string (value));
		break;
		case VALA_SOURCE_FILE_COMMENT:
		vala_source_file_set_comment (self, g_value_get_string (value));
		break;
		case VALA_SOURCE_FILE_PKG:
		vala_source_file_set_pkg (self, g_value_get_boolean (value));
		break;
		case VALA_SOURCE_FILE_CYCLE:
		vala_source_file_set_cycle (self, g_value_get_object (value));
		break;
		case VALA_SOURCE_FILE_IS_CYCLE_HEAD:
		vala_source_file_set_is_cycle_head (self, g_value_get_boolean (value));
		break;
		case VALA_SOURCE_FILE_MARK:
		vala_source_file_set_mark (self, g_value_get_int (value));
		break;
		case VALA_SOURCE_FILE_CONTEXT:
		vala_source_file_set_context (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_source_file_class_init (ValaSourceFileClass * klass)
{
	vala_source_file_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaSourceFilePrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_source_file_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_source_file_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_source_file_dispose;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_FILENAME, g_param_spec_string ("filename", "foo", "bar", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_COMMENT, g_param_spec_string ("comment", "foo", "bar", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_PKG, g_param_spec_boolean ("pkg", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_CYCLE, g_param_spec_object ("cycle", "foo", "bar", VALA_TYPE_SOURCE_FILE_CYCLE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_IS_CYCLE_HEAD, g_param_spec_boolean ("is-cycle-head", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_MARK, g_param_spec_int ("mark", "foo", "bar", G_MININT, G_MAXINT, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_CONTEXT, g_param_spec_object ("context", "foo", "bar", VALA_TYPE_CODE_CONTEXT, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_source_file_init (ValaSourceFile * self)
{
	self->priv = VALA_SOURCE_FILE_GET_PRIVATE (self);
	self->priv->using_directives = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->nodes = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->cheader_filename = NULL;
	self->priv->csource_filename = NULL;
	self->priv->cinclude_filename = NULL;
	self->priv->header_external_includes = GEE_LIST (gee_array_list_new (g_strdup, g_free, g_direct_equal));
	self->priv->header_internal_includes = GEE_LIST (gee_array_list_new (g_strdup, g_free, g_direct_equal));
	self->priv->source_external_includes = GEE_LIST (gee_array_list_new (g_strdup, g_free, g_direct_equal));
	self->priv->source_internal_includes = GEE_LIST (gee_array_list_new (g_strdup, g_free, g_direct_equal));
	self->priv->header_internal_full_dependencies = GEE_LIST (gee_array_list_new (NULL, NULL, g_direct_equal));
	self->priv->header_internal_dependencies = GEE_LIST (gee_array_list_new (NULL, NULL, g_direct_equal));
}


static void vala_source_file_dispose (GObject * obj)
{
	ValaSourceFile * self;
	ValaSourceFileClass * klass;
	GObjectClass * parent_class;
	self = VALA_SOURCE_FILE (obj);
	(self->priv->_filename = (g_free (self->priv->_filename), NULL));
	(self->priv->_comment = (g_free (self->priv->_comment), NULL));
	(self->priv->_cycle == NULL ? NULL : (self->priv->_cycle = (g_object_unref (self->priv->_cycle), NULL)));
	(self->priv->using_directives == NULL ? NULL : (self->priv->using_directives = (g_object_unref (self->priv->using_directives), NULL)));
	(self->priv->nodes == NULL ? NULL : (self->priv->nodes = (g_object_unref (self->priv->nodes), NULL)));
	(self->priv->cheader_filename = (g_free (self->priv->cheader_filename), NULL));
	(self->priv->csource_filename = (g_free (self->priv->csource_filename), NULL));
	(self->priv->cinclude_filename = (g_free (self->priv->cinclude_filename), NULL));
	(self->priv->header_external_includes == NULL ? NULL : (self->priv->header_external_includes = (g_object_unref (self->priv->header_external_includes), NULL)));
	(self->priv->header_internal_includes == NULL ? NULL : (self->priv->header_internal_includes = (g_object_unref (self->priv->header_internal_includes), NULL)));
	(self->priv->source_external_includes == NULL ? NULL : (self->priv->source_external_includes = (g_object_unref (self->priv->source_external_includes), NULL)));
	(self->priv->source_internal_includes == NULL ? NULL : (self->priv->source_internal_includes = (g_object_unref (self->priv->source_internal_includes), NULL)));
	(self->priv->header_internal_full_dependencies == NULL ? NULL : (self->priv->header_internal_full_dependencies = (g_object_unref (self->priv->header_internal_full_dependencies), NULL)));
	(self->priv->header_internal_dependencies == NULL ? NULL : (self->priv->header_internal_dependencies = (g_object_unref (self->priv->header_internal_dependencies), NULL)));
	klass = VALA_SOURCE_FILE_CLASS (g_type_class_peek (VALA_TYPE_SOURCE_FILE));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_source_file_get_type (void)
{
	static GType vala_source_file_type_id = 0;
	if (G_UNLIKELY (vala_source_file_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSourceFileClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_source_file_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSourceFile), 0, (GInstanceInitFunc) vala_source_file_init };
		vala_source_file_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaSourceFile", &g_define_type_info, 0);
	}
	return vala_source_file_type_id;
}




