/* hashset.vala
 *
 * Copyright (C) 1995-1997  Peter Mattis, Spencer Kimball and Josh MacDonald
 * Copyright (C) 1997-2000  GLib Team and others
 * Copyright (C) 2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <gee/hashset.h>

typedef struct _GeeHashSetNode GeeHashSetNode;
typedef struct _GeeHashSetIterator GeeHashSetIterator;
typedef struct _GeeHashSetIteratorClass GeeHashSetIteratorClass;
typedef struct _GeeHashSetIteratorPrivate GeeHashSetIteratorPrivate;
struct _GeeHashSetPrivate {
	gint _array_size;
	gint _nnodes;
	GeeHashSetNode** _nodes;
	gint _nodes_length1;
	gint _stamp;
	GHashFunc _hash_func;
	GEqualFunc _equal_func;
	GBoxedCopyFunc g_dup_func;
	GDestroyNotify g_destroy_func;
};
#define GEE_HASH_SET_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_TYPE_HASH_SET, GeeHashSetPrivate))
enum  {
	GEE_HASH_SET_DUMMY_PROPERTY,
	GEE_HASH_SET_SIZE,
	GEE_HASH_SET_HASH_FUNC,
	GEE_HASH_SET_EQUAL_FUNC,
	GEE_HASH_SET_G_DUP_FUNC,
	GEE_HASH_SET_G_DESTROY_FUNC
};
static const gint GEE_HASH_SET_MIN_SIZE = 11;
static const gint GEE_HASH_SET_MAX_SIZE = 13845163;
static GeeHashSetNode** gee_hash_set_lookup_node (GeeHashSet* self, gpointer key);
static gboolean gee_hash_set_real_contains (GeeCollection* base, gpointer key);
static GeeIterator* gee_hash_set_real_iterator (GeeIterable* base);
static gboolean gee_hash_set_real_add (GeeCollection* base, gpointer key);
static gboolean gee_hash_set_real_remove (GeeCollection* base, gpointer key);
static void gee_hash_set_real_clear (GeeCollection* base);
static void gee_hash_set_resize (GeeHashSet* self);
static GObject * gee_hash_set_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
struct _GeeHashSetNode {
	gpointer key;
	GeeHashSetNode* next;
	guint key_hash;
};
static GeeHashSetNode* gee_hash_set_node_new (gpointer k, guint hash);
static void gee_hash_set_node_free (GeeHashSetNode* self);

#define GEE_HASH_SET_TYPE_ITERATOR (gee_hash_set_iterator_get_type ())
#define GEE_HASH_SET_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_HASH_SET_TYPE_ITERATOR, GeeHashSetIterator))
#define GEE_HASH_SET_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_HASH_SET_TYPE_ITERATOR, GeeHashSetIteratorClass))
#define GEE_HASH_SET_IS_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_HASH_SET_TYPE_ITERATOR))
#define GEE_HASH_SET_IS_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_HASH_SET_TYPE_ITERATOR))
#define GEE_HASH_SET_ITERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_HASH_SET_TYPE_ITERATOR, GeeHashSetIteratorClass))

struct _GeeHashSetIterator {
	GObject parent;
	GeeHashSetIteratorPrivate * priv;
};
struct _GeeHashSetIteratorClass {
	GObjectClass parent;
};
struct _GeeHashSetIteratorPrivate {
	GeeHashSet* _set;
	gint _index;
	GeeHashSetNode* _node;
	gint _stamp;
	GBoxedCopyFunc g_dup_func;
	GDestroyNotify g_destroy_func;
};
#define GEE_HASH_SET_ITERATOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_HASH_SET_TYPE_ITERATOR, GeeHashSetIteratorPrivate))
enum  {
	GEE_HASH_SET_ITERATOR_DUMMY_PROPERTY,
	GEE_HASH_SET_ITERATOR_SET,
	GEE_HASH_SET_ITERATOR_G_DUP_FUNC,
	GEE_HASH_SET_ITERATOR_G_DESTROY_FUNC
};
static GeeHashSetIterator* gee_hash_set_iterator_new (GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeHashSet* set);
static gboolean gee_hash_set_iterator_real_next (GeeIterator* base);
static gpointer gee_hash_set_iterator_real_get (GeeIterator* base);
static void gee_hash_set_iterator_set_set (GeeHashSetIterator* self, GeeHashSet* value);
static gpointer gee_hash_set_iterator_parent_class = NULL;
static GeeIteratorIface* gee_hash_set_iterator_gee_iterator_parent_iface = NULL;
static void gee_hash_set_iterator_dispose (GObject * obj);
static GType gee_hash_set_iterator_get_type (void);
static gpointer gee_hash_set_parent_class = NULL;
static GeeIterableIface* gee_hash_set_gee_iterable_parent_iface = NULL;
static GeeCollectionIface* gee_hash_set_gee_collection_parent_iface = NULL;
static GeeSetIface* gee_hash_set_gee_set_parent_iface = NULL;
static void gee_hash_set_dispose (GObject * obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);


GeeHashSet* gee_hash_set_new (GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GHashFunc hash_func, GEqualFunc equal_func) {
	GeeHashSet * self;
	self = g_object_newv (GEE_TYPE_HASH_SET, 0, NULL);
	gee_hash_set_set_hash_func (self, hash_func);
	gee_hash_set_set_equal_func (self, equal_func);
	self->priv->g_dup_func = g_dup_func;
	self->priv->g_destroy_func = g_destroy_func;
	return self;
}


static GeeHashSetNode** gee_hash_set_lookup_node (GeeHashSet* self, gpointer key) {
	guint hash_value;
	GeeHashSetNode** node;
	g_return_val_if_fail (GEE_IS_HASH_SET (self), NULL);
	hash_value = self->priv->_hash_func (key);
	node = &self->priv->_nodes[hash_value % self->priv->_array_size];
	while ((*node) != NULL && (hash_value != (*node)->key_hash || !self->priv->_equal_func ((*node)->key, key))) {
		node = &((*node)->next);
	}
	return node;
}


static gboolean gee_hash_set_real_contains (GeeCollection* base, gpointer key) {
	GeeHashSet * self;
	GeeHashSetNode** node;
	self = GEE_HASH_SET (base);
	node = gee_hash_set_lookup_node (self, key);
	return (*node != NULL);
}


static GeeIterator* gee_hash_set_real_iterator (GeeIterable* base) {
	GeeHashSet * self;
	self = GEE_HASH_SET (base);
	return GEE_ITERATOR (gee_hash_set_iterator_new (self->priv->g_dup_func, self->priv->g_destroy_func, self));
}


static gboolean gee_hash_set_real_add (GeeCollection* base, gpointer key) {
	GeeHashSet * self;
	GeeHashSetNode** node;
	self = GEE_HASH_SET (base);
	node = gee_hash_set_lookup_node (self, key);
	if (*node != NULL) {
		return FALSE;
	} else {
		guint hash_value;
		gpointer _tmp1;
		hash_value = self->priv->_hash_func (key);
		*node = gee_hash_set_node_new ((_tmp1 = key, (_tmp1 == NULL || self->priv->g_dup_func == NULL ? _tmp1 : self->priv->g_dup_func (_tmp1))), hash_value);
		self->priv->_nnodes++;
		gee_hash_set_resize (self);
		self->priv->_stamp++;
		return TRUE;
	}
}


static gboolean gee_hash_set_real_remove (GeeCollection* base, gpointer key) {
	GeeHashSet * self;
	GeeHashSetNode** node;
	self = GEE_HASH_SET (base);
	node = gee_hash_set_lookup_node (self, key);
	if (*node != NULL) {
		gpointer _tmp0;
		(*node)->key = (_tmp0 = NULL, ((*node)->key == NULL || self->priv->g_destroy_func == NULL ? NULL : ((*node)->key = (self->priv->g_destroy_func ((*node)->key), NULL))), _tmp0);
		*node = (*node)->next;
		self->priv->_nnodes--;
		gee_hash_set_resize (self);
		self->priv->_stamp++;
		return TRUE;
	}
	return FALSE;
}


static void gee_hash_set_real_clear (GeeCollection* base) {
	GeeHashSet * self;
	self = GEE_HASH_SET (base);
	{
		gint i;
		i = 0;
		for (i = 0; i < self->priv->_array_size; i++) {
			GeeHashSetNode* _tmp0;
			GeeHashSetNode* node;
			_tmp0 = NULL;
			node = (_tmp0 = self->priv->_nodes[i], self->priv->_nodes[i] = NULL, _tmp0);
			while (node != NULL) {
				GeeHashSetNode* _tmp1;
				GeeHashSetNode* next;
				gpointer _tmp2;
				GeeHashSetNode* _tmp4;
				GeeHashSetNode* _tmp3;
				_tmp1 = NULL;
				next = (_tmp1 = node->next, node->next = NULL, _tmp1);
				node->key = (_tmp2 = NULL, (node->key == NULL || self->priv->g_destroy_func == NULL ? NULL : (node->key = (self->priv->g_destroy_func (node->key), NULL))), _tmp2);
				_tmp4 = NULL;
				_tmp3 = NULL;
				node = (_tmp4 = (_tmp3 = next, next = NULL, _tmp3), (node == NULL ? NULL : (node = (gee_hash_set_node_free (node), NULL))), _tmp4);
				(next == NULL ? NULL : (next = (gee_hash_set_node_free (next), NULL)));
			}
			(node == NULL ? NULL : (node = (gee_hash_set_node_free (node), NULL)));
		}
	}
	self->priv->_nnodes = 0;
	gee_hash_set_resize (self);
}


static void gee_hash_set_resize (GeeHashSet* self) {
	g_return_if_fail (GEE_IS_HASH_SET (self));
	if ((self->priv->_array_size >= 3 * self->priv->_nnodes && self->priv->_array_size >= GEE_HASH_SET_MIN_SIZE) || (3 * self->priv->_array_size <= self->priv->_nnodes && self->priv->_array_size < GEE_HASH_SET_MAX_SIZE)) {
		gint new_array_size;
		GeeHashSetNode** _tmp0;
		gint new_nodes_length1;
		GeeHashSetNode** new_nodes;
		GeeHashSetNode** _tmp12;
		GeeHashSetNode** _tmp11;
		new_array_size = g_spaced_primes_closest (self->priv->_nnodes);
		new_array_size = CLAMP (new_array_size, GEE_HASH_SET_MIN_SIZE, GEE_HASH_SET_MAX_SIZE);
		_tmp0 = NULL;
		new_nodes = (_tmp0 = g_new0 (GeeHashSetNode*, new_array_size + 1), new_nodes_length1 = new_array_size, _tmp0);
		{
			gint i;
			i = 0;
			for (i = 0; i < self->priv->_array_size; i++) {
				GeeHashSetNode* node;
				GeeHashSetNode* next;
				GeeHashSetNode* _tmp4;
				GeeHashSetNode* _tmp3;
				GeeHashSetNode* _tmp2;
				GeeHashSetNode* _tmp1;
				node = NULL;
				next = NULL;
				_tmp4 = NULL;
				_tmp3 = NULL;
				_tmp2 = NULL;
				_tmp1 = NULL;
				for (node = (_tmp2 = (_tmp1 = self->priv->_nodes[i], self->priv->_nodes[i] = NULL, _tmp1), (node == NULL ? NULL : (node = (gee_hash_set_node_free (node), NULL))), _tmp2); node != NULL; node = (_tmp4 = (_tmp3 = next, next = NULL, _tmp3), (node == NULL ? NULL : (node = (gee_hash_set_node_free (node), NULL))), _tmp4)) {
					GeeHashSetNode* _tmp6;
					GeeHashSetNode* _tmp5;
					guint hash_val;
					GeeHashSetNode* _tmp8;
					GeeHashSetNode* _tmp7;
					GeeHashSetNode* _tmp10;
					GeeHashSetNode* _tmp9;
					_tmp6 = NULL;
					_tmp5 = NULL;
					next = (_tmp6 = (_tmp5 = node->next, node->next = NULL, _tmp5), (next == NULL ? NULL : (next = (gee_hash_set_node_free (next), NULL))), _tmp6);
					hash_val = node->key_hash % new_array_size;
					_tmp8 = NULL;
					_tmp7 = NULL;
					node->next = (_tmp8 = (_tmp7 = new_nodes[hash_val], new_nodes[hash_val] = NULL, _tmp7), (node->next == NULL ? NULL : (node->next = (gee_hash_set_node_free (node->next), NULL))), _tmp8);
					_tmp10 = NULL;
					_tmp9 = NULL;
					new_nodes[hash_val] = (_tmp10 = (_tmp9 = node, node = NULL, _tmp9), (new_nodes[hash_val] == NULL ? NULL : (new_nodes[hash_val] = (gee_hash_set_node_free (new_nodes[hash_val]), NULL))), _tmp10);
				}
				(node == NULL ? NULL : (node = (gee_hash_set_node_free (node), NULL)));
				(next == NULL ? NULL : (next = (gee_hash_set_node_free (next), NULL)));
			}
		}
		_tmp12 = NULL;
		_tmp11 = NULL;
		self->priv->_nodes = (_tmp12 = (_tmp11 = new_nodes, new_nodes = NULL, _tmp11), (self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, ((GDestroyNotify) gee_hash_set_node_free)), NULL)), self->priv->_nodes_length1 = new_nodes_length1, _tmp12);
		self->priv->_array_size = new_array_size;
		(new_nodes = (_vala_array_free (new_nodes, new_nodes_length1, ((GDestroyNotify) gee_hash_set_node_free)), NULL));
	}
}


static gint gee_hash_set_real_get_size (GeeHashSet* self) {
	g_return_val_if_fail (GEE_IS_HASH_SET (self), 0);
	return self->priv->_nnodes;
}


void gee_hash_set_set_hash_func (GeeHashSet* self, GHashFunc value) {
	g_return_if_fail (GEE_IS_HASH_SET (self));
	self->priv->_hash_func = value;
}


void gee_hash_set_set_equal_func (GeeHashSet* self, GEqualFunc value) {
	g_return_if_fail (GEE_IS_HASH_SET (self));
	self->priv->_equal_func = value;
}


static GObject * gee_hash_set_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	GeeHashSetClass * klass;
	GObjectClass * parent_class;
	GeeHashSet * self;
	klass = GEE_HASH_SET_CLASS (g_type_class_peek (GEE_TYPE_HASH_SET));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = GEE_HASH_SET (obj);
	{
		GeeHashSetNode** _tmp0;
		self->priv->_array_size = GEE_HASH_SET_MIN_SIZE;
		_tmp0 = NULL;
		self->priv->_nodes = (_tmp0 = g_new0 (GeeHashSetNode*, self->priv->_array_size + 1), (self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, ((GDestroyNotify) gee_hash_set_node_free)), NULL)), self->priv->_nodes_length1 = self->priv->_array_size, _tmp0);
	}
	return obj;
}


static GeeHashSetNode* gee_hash_set_node_new (gpointer k, guint hash) {
	GeeHashSetNode* self;
	gpointer _tmp1;
	gpointer _tmp0;
	self = g_slice_new0 (GeeHashSetNode);
	self->key = (_tmp1 = (_tmp0 = k, k = NULL, _tmp0), NULL, _tmp1);
	self->key_hash = hash;
	return self;
}


static void gee_hash_set_node_free (GeeHashSetNode* self) {
	NULL;
	(self->next == NULL ? NULL : (self->next = (gee_hash_set_node_free (self->next), NULL)));
	g_slice_free (GeeHashSetNode, self);
}


static GeeHashSetIterator* gee_hash_set_iterator_new (GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeHashSet* set) {
	GeeHashSetIterator * self;
	g_return_val_if_fail (GEE_IS_HASH_SET (set), NULL);
	self = g_object_newv (GEE_HASH_SET_TYPE_ITERATOR, 0, NULL);
	gee_hash_set_iterator_set_set (self, set);
	self->priv->g_dup_func = g_dup_func;
	self->priv->g_destroy_func = g_destroy_func;
	return self;
}


static gboolean gee_hash_set_iterator_real_next (GeeIterator* base) {
	GeeHashSetIterator * self;
	self = GEE_HASH_SET_ITERATOR (base);
	if (self->priv->_node != NULL) {
		self->priv->_node = self->priv->_node->next;
	}
	while (self->priv->_node == NULL && self->priv->_index + 1 < self->priv->_set->priv->_array_size) {
		self->priv->_index++;
		self->priv->_node = self->priv->_set->priv->_nodes[self->priv->_index];
	}
	return (self->priv->_node != NULL);
}


static gpointer gee_hash_set_iterator_real_get (GeeIterator* base) {
	GeeHashSetIterator * self;
	gpointer _tmp0;
	self = GEE_HASH_SET_ITERATOR (base);
	g_assert (self->priv->_stamp == self->priv->_set->priv->_stamp);
	g_assert (self->priv->_node != NULL);
	return (_tmp0 = self->priv->_node->key, (_tmp0 == NULL || self->priv->g_dup_func == NULL ? _tmp0 : self->priv->g_dup_func (_tmp0)));
}


static void gee_hash_set_iterator_set_set (GeeHashSetIterator* self, GeeHashSet* value) {
	GeeHashSet* _tmp1;
	GeeHashSet* _tmp0;
	g_return_if_fail (GEE_HASH_SET_IS_ITERATOR (self));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->_set = (_tmp1 = (_tmp0 = value, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (self->priv->_set == NULL ? NULL : (self->priv->_set = (g_object_unref (self->priv->_set), NULL))), _tmp1);
	self->priv->_stamp = self->priv->_set->priv->_stamp;
	while (self->priv->_node == NULL && self->priv->_index + 1 < self->priv->_set->priv->_array_size) {
		/* find first node*/
		self->priv->_index++;
		self->priv->_node = self->priv->_set->priv->_nodes[self->priv->_index];
	}
}


static void gee_hash_set_iterator_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	GeeHashSetIterator * self;
	self = GEE_HASH_SET_ITERATOR (object);
	switch (property_id) {
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_hash_set_iterator_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	GeeHashSetIterator * self;
	self = GEE_HASH_SET_ITERATOR (object);
	switch (property_id) {
		case GEE_HASH_SET_ITERATOR_SET:
		gee_hash_set_iterator_set_set (self, g_value_get_object (value));
		break;
		case GEE_HASH_SET_ITERATOR_G_DESTROY_FUNC:
		self->priv->g_destroy_func = g_value_get_pointer (value);
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_hash_set_iterator_class_init (GeeHashSetIteratorClass * klass) {
	gee_hash_set_iterator_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeHashSetIteratorPrivate));
	G_OBJECT_CLASS (klass)->get_property = gee_hash_set_iterator_get_property;
	G_OBJECT_CLASS (klass)->set_property = gee_hash_set_iterator_set_property;
	G_OBJECT_CLASS (klass)->dispose = gee_hash_set_iterator_dispose;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_SET_ITERATOR_G_DUP_FUNC, g_param_spec_pointer ("g-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_SET_ITERATOR_G_DESTROY_FUNC, g_param_spec_pointer ("g-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_SET_ITERATOR_SET, g_param_spec_object ("set", "set", "set", GEE_TYPE_HASH_SET, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
}


static void gee_hash_set_iterator_gee_iterator_interface_init (GeeIteratorIface * iface) {
	gee_hash_set_iterator_gee_iterator_parent_iface = g_type_interface_peek_parent (iface);
	iface->next = gee_hash_set_iterator_real_next;
	iface->get = gee_hash_set_iterator_real_get;
}


static void gee_hash_set_iterator_init (GeeHashSetIterator * self) {
	self->priv = GEE_HASH_SET_ITERATOR_GET_PRIVATE (self);
	self->priv->_index = -1;
	self->priv->_stamp = 0;
}


static void gee_hash_set_iterator_dispose (GObject * obj) {
	GeeHashSetIterator * self;
	self = GEE_HASH_SET_ITERATOR (obj);
	(self->priv->_set == NULL ? NULL : (self->priv->_set = (g_object_unref (self->priv->_set), NULL)));
	G_OBJECT_CLASS (gee_hash_set_iterator_parent_class)->dispose (obj);
}


static GType gee_hash_set_iterator_get_type (void) {
	static GType gee_hash_set_iterator_type_id = 0;
	if (G_UNLIKELY (gee_hash_set_iterator_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashSetIteratorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_set_iterator_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashSetIterator), 0, (GInstanceInitFunc) gee_hash_set_iterator_init };
		static const GInterfaceInfo gee_iterator_info = { (GInterfaceInitFunc) gee_hash_set_iterator_gee_iterator_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_set_iterator_type_id = g_type_register_static (G_TYPE_OBJECT, "GeeHashSetIterator", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_set_iterator_type_id, GEE_TYPE_ITERATOR, &gee_iterator_info);
	}
	return gee_hash_set_iterator_type_id;
}


static void gee_hash_set_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	GeeHashSet * self;
	self = GEE_HASH_SET (object);
	switch (property_id) {
		case GEE_HASH_SET_SIZE:
		g_value_set_int (value, gee_hash_set_real_get_size (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_hash_set_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	GeeHashSet * self;
	self = GEE_HASH_SET (object);
	switch (property_id) {
		case GEE_HASH_SET_HASH_FUNC:
		gee_hash_set_set_hash_func (self, g_value_get_pointer (value));
		break;
		case GEE_HASH_SET_EQUAL_FUNC:
		gee_hash_set_set_equal_func (self, g_value_get_pointer (value));
		break;
		case GEE_HASH_SET_G_DESTROY_FUNC:
		self->priv->g_destroy_func = g_value_get_pointer (value);
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_hash_set_class_init (GeeHashSetClass * klass) {
	gee_hash_set_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeHashSetPrivate));
	G_OBJECT_CLASS (klass)->get_property = gee_hash_set_get_property;
	G_OBJECT_CLASS (klass)->set_property = gee_hash_set_set_property;
	G_OBJECT_CLASS (klass)->constructor = gee_hash_set_constructor;
	G_OBJECT_CLASS (klass)->dispose = gee_hash_set_dispose;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_SET_G_DUP_FUNC, g_param_spec_pointer ("g-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_SET_G_DESTROY_FUNC, g_param_spec_pointer ("g-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_override_property (G_OBJECT_CLASS (klass), GEE_HASH_SET_SIZE, "size");
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_SET_HASH_FUNC, g_param_spec_pointer ("hash-func", "hash-func", "hash-func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_SET_EQUAL_FUNC, g_param_spec_pointer ("equal-func", "equal-func", "equal-func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
}


static void gee_hash_set_gee_iterable_interface_init (GeeIterableIface * iface) {
	gee_hash_set_gee_iterable_parent_iface = g_type_interface_peek_parent (iface);
	iface->iterator = gee_hash_set_real_iterator;
}


static void gee_hash_set_gee_collection_interface_init (GeeCollectionIface * iface) {
	gee_hash_set_gee_collection_parent_iface = g_type_interface_peek_parent (iface);
	iface->contains = gee_hash_set_real_contains;
	iface->add = gee_hash_set_real_add;
	iface->remove = gee_hash_set_real_remove;
	iface->clear = gee_hash_set_real_clear;
}


static void gee_hash_set_gee_set_interface_init (GeeSetIface * iface) {
	gee_hash_set_gee_set_parent_iface = g_type_interface_peek_parent (iface);
}


static void gee_hash_set_init (GeeHashSet * self) {
	self->priv = GEE_HASH_SET_GET_PRIVATE (self);
	self->priv->_stamp = 0;
}


static void gee_hash_set_dispose (GObject * obj) {
	GeeHashSet * self;
	self = GEE_HASH_SET (obj);
	{
		gee_collection_clear (GEE_COLLECTION (self));
	}
	(self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, ((GDestroyNotify) gee_hash_set_node_free)), NULL));
	G_OBJECT_CLASS (gee_hash_set_parent_class)->dispose (obj);
}


GType gee_hash_set_get_type (void) {
	static GType gee_hash_set_type_id = 0;
	if (G_UNLIKELY (gee_hash_set_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashSetClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_set_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashSet), 0, (GInstanceInitFunc) gee_hash_set_init };
		static const GInterfaceInfo gee_iterable_info = { (GInterfaceInitFunc) gee_hash_set_gee_iterable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_collection_info = { (GInterfaceInitFunc) gee_hash_set_gee_collection_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_set_info = { (GInterfaceInitFunc) gee_hash_set_gee_set_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_set_type_id = g_type_register_static (G_TYPE_OBJECT, "GeeHashSet", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_set_type_id, GEE_TYPE_ITERABLE, &gee_iterable_info);
		g_type_add_interface_static (gee_hash_set_type_id, GEE_TYPE_COLLECTION, &gee_collection_info);
		g_type_add_interface_static (gee_hash_set_type_id, GEE_TYPE_SET, &gee_set_info);
	}
	return gee_hash_set_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}




