/* valamemberinitializer.vala
 *
 * Copyright (C) 2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valamemberinitializer.h>
#include <vala/valaexpression.h>
#include <vala/valasymbol.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>

struct _ValaMemberInitializerPrivate {
	char* _name;
	ValaExpression* _initializer;
	ValaSymbol* _symbol_reference;
};
#define VALA_MEMBER_INITIALIZER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_MEMBER_INITIALIZER, ValaMemberInitializerPrivate))
enum  {
	VALA_MEMBER_INITIALIZER_DUMMY_PROPERTY,
	VALA_MEMBER_INITIALIZER_NAME,
	VALA_MEMBER_INITIALIZER_INITIALIZER,
	VALA_MEMBER_INITIALIZER_SYMBOL_REFERENCE
};
static void vala_member_initializer_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gpointer vala_member_initializer_parent_class = NULL;
static void vala_member_initializer_dispose (GObject * obj);


/**
 * Creates a new member initializer.
 *
 * @param name             member name
 * @param initializer      initializer expression
 * @param source_reference reference to source code
 * @return                 newly created member initializer
 */
ValaMemberInitializer* vala_member_initializer_new (const char* name, ValaExpression* initializer, ValaSourceReference* source_reference) {
	ValaMemberInitializer * self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (initializer), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_MEMBER_INITIALIZER, 0, NULL);
	vala_member_initializer_set_name (self, name);
	vala_member_initializer_set_initializer (self, initializer);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


static void vala_member_initializer_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaMemberInitializer * self;
	self = VALA_MEMBER_INITIALIZER (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (vala_member_initializer_get_initializer (self)), visitor);
}


char* vala_member_initializer_get_name (ValaMemberInitializer* self) {
	g_return_val_if_fail (VALA_IS_MEMBER_INITIALIZER (self), NULL);
	return self->priv->_name;
}


void vala_member_initializer_set_name (ValaMemberInitializer* self, const char* value) {
	char* _tmp1;
	g_return_if_fail (VALA_IS_MEMBER_INITIALIZER (self));
	_tmp1 = NULL;
	self->priv->_name = (_tmp1 = g_strdup (value), (self->priv->_name = (g_free (self->priv->_name), NULL)), _tmp1);
}


ValaExpression* vala_member_initializer_get_initializer (ValaMemberInitializer* self) {
	g_return_val_if_fail (VALA_IS_MEMBER_INITIALIZER (self), NULL);
	return self->priv->_initializer;
}


void vala_member_initializer_set_initializer (ValaMemberInitializer* self, ValaExpression* value) {
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_MEMBER_INITIALIZER (self));
	_tmp1 = NULL;
	self->priv->_initializer = (_tmp1 = g_object_ref (value), (self->priv->_initializer == NULL ? NULL : (self->priv->_initializer = (g_object_unref (self->priv->_initializer), NULL))), _tmp1);
}


ValaSymbol* vala_member_initializer_get_symbol_reference (ValaMemberInitializer* self) {
	g_return_val_if_fail (VALA_IS_MEMBER_INITIALIZER (self), NULL);
	return self->priv->_symbol_reference;
}


void vala_member_initializer_set_symbol_reference (ValaMemberInitializer* self, ValaSymbol* value) {
	g_return_if_fail (VALA_IS_MEMBER_INITIALIZER (self));
	self->priv->_symbol_reference = value;
}


static void vala_member_initializer_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaMemberInitializer * self;
	self = VALA_MEMBER_INITIALIZER (object);
	switch (property_id) {
		case VALA_MEMBER_INITIALIZER_NAME:
		g_value_set_string (value, vala_member_initializer_get_name (self));
		break;
		case VALA_MEMBER_INITIALIZER_INITIALIZER:
		g_value_set_object (value, vala_member_initializer_get_initializer (self));
		break;
		case VALA_MEMBER_INITIALIZER_SYMBOL_REFERENCE:
		g_value_set_object (value, vala_member_initializer_get_symbol_reference (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_member_initializer_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaMemberInitializer * self;
	self = VALA_MEMBER_INITIALIZER (object);
	switch (property_id) {
		case VALA_MEMBER_INITIALIZER_NAME:
		vala_member_initializer_set_name (self, g_value_get_string (value));
		break;
		case VALA_MEMBER_INITIALIZER_INITIALIZER:
		vala_member_initializer_set_initializer (self, g_value_get_object (value));
		break;
		case VALA_MEMBER_INITIALIZER_SYMBOL_REFERENCE:
		vala_member_initializer_set_symbol_reference (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_member_initializer_class_init (ValaMemberInitializerClass * klass) {
	vala_member_initializer_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaMemberInitializerPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_member_initializer_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_member_initializer_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_member_initializer_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_member_initializer_real_accept;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_MEMBER_INITIALIZER_NAME, g_param_spec_string ("name", "name", "name", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_MEMBER_INITIALIZER_INITIALIZER, g_param_spec_object ("initializer", "initializer", "initializer", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_MEMBER_INITIALIZER_SYMBOL_REFERENCE, g_param_spec_object ("symbol-reference", "symbol-reference", "symbol-reference", VALA_TYPE_SYMBOL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_member_initializer_init (ValaMemberInitializer * self) {
	self->priv = VALA_MEMBER_INITIALIZER_GET_PRIVATE (self);
}


static void vala_member_initializer_dispose (GObject * obj) {
	ValaMemberInitializer * self;
	self = VALA_MEMBER_INITIALIZER (obj);
	(self->priv->_name = (g_free (self->priv->_name), NULL));
	(self->priv->_initializer == NULL ? NULL : (self->priv->_initializer = (g_object_unref (self->priv->_initializer), NULL)));
	G_OBJECT_CLASS (vala_member_initializer_parent_class)->dispose (obj);
}


GType vala_member_initializer_get_type (void) {
	static GType vala_member_initializer_type_id = 0;
	if (G_UNLIKELY (vala_member_initializer_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaMemberInitializerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_member_initializer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaMemberInitializer), 0, (GInstanceInitFunc) vala_member_initializer_init };
		vala_member_initializer_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaMemberInitializer", &g_define_type_info, 0);
	}
	return vala_member_initializer_type_id;
}




