/* valamethod.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */

#include <vala/valamethod.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valatypereference.h>
#include <vala/valablock.h>
#include <vala/valaformalparameter.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>

struct _ValaMethodPrivate {
	ValaTypeReference* _return_type;
	ValaBlock* _body;
	gboolean _is_abstract;
	gboolean _is_virtual;
	gboolean _overrides;
	gboolean _returns_modified_pointer;
	gboolean _instance_last;
	ValaMethod* _base_method;
	ValaMethod* _base_interface_method;
	ValaFormalParameter* _this_parameter;
	gboolean _printf_format;
	gboolean _instance;
	GeeList* parameters;
	char* cname;
	char* _vfunc_name;
	char* _sentinel;
	gboolean _no_array_length;
	GeeList* error_domains;
};
#define VALA_METHOD_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_METHOD, ValaMethodPrivate))
enum  {
	VALA_METHOD_DUMMY_PROPERTY,
	VALA_METHOD_RETURN_TYPE,
	VALA_METHOD_BODY,
	VALA_METHOD_INSTANCE,
	VALA_METHOD_VFUNC_NAME,
	VALA_METHOD_SENTINEL,
	VALA_METHOD_IS_ABSTRACT,
	VALA_METHOD_IS_VIRTUAL,
	VALA_METHOD_OVERRIDES,
	VALA_METHOD_RETURNS_MODIFIED_POINTER,
	VALA_METHOD_INSTANCE_LAST,
	VALA_METHOD_BASE_METHOD,
	VALA_METHOD_BASE_INTERFACE_METHOD,
	VALA_METHOD_THIS_PARAMETER,
	VALA_METHOD_NO_ARRAY_LENGTH,
	VALA_METHOD_PRINTF_FORMAT
};
static GeeCollection* vala_method_real_get_parameters (ValaInvokable* base);
static ValaTypeReference* vala_method_real_get_return_type (ValaInvokable* base);
static gboolean vala_method_real_is_invokable (ValaInvokable* base);
static void vala_method_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_method_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_method_real_get_default_cname (ValaMethod* self);
static void vala_method_process_ccode_attribute (ValaMethod* self, ValaAttribute* a);
static gpointer vala_method_parent_class = NULL;
static ValaInvokableIface* vala_method_vala_invokable_parent_iface = NULL;
static void vala_method_dispose (GObject * obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);


/**
 * Creates a new method.
 *
 * @param name        method name
 * @param return_type method return type
 * @param source      reference to source code
 * @return            newly created method
 */
ValaMethod* vala_method_new (const char* name, ValaTypeReference* return_type, ValaSourceReference* source_reference) {
	ValaMethod * self;
	g_return_val_if_fail (return_type == NULL || VALA_IS_TYPE_REFERENCE (return_type), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_METHOD, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_method_set_return_type (self, return_type);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Appends parameter to this method.
 *
 * @param param a formal parameter
 */
void vala_method_add_parameter (ValaMethod* self, ValaFormalParameter* param) {
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (param));
	if (vala_method_get_no_array_length (self)) {
		vala_formal_parameter_set_no_array_length (param, TRUE);
	}
	gee_collection_add (GEE_COLLECTION (self->priv->parameters), param);
	if (!vala_formal_parameter_get_ellipsis (param)) {
		vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (param)), VALA_SYMBOL (param));
	}
}


static GeeCollection* vala_method_real_get_parameters (ValaInvokable* base) {
	ValaMethod * self;
	self = VALA_METHOD (base);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->parameters)));
}


static ValaTypeReference* vala_method_real_get_return_type (ValaInvokable* base) {
	ValaMethod * self;
	ValaTypeReference* _tmp0;
	self = VALA_METHOD (base);
	_tmp0 = NULL;
	return (_tmp0 = vala_method_get_return_type (self), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
}


static gboolean vala_method_real_is_invokable (ValaInvokable* base) {
	ValaMethod * self;
	self = VALA_METHOD (base);
	return TRUE;
}


static void vala_method_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaMethod * self;
	self = VALA_METHOD (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_method (visitor, self);
}


static void vala_method_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaMethod * self;
	self = VALA_METHOD (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	if (vala_method_get_return_type (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_method_get_return_type (self)), visitor);
	}
	{
		GeeList* param_collection;
		GeeIterator* param_it;
		param_collection = self->priv->parameters;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = gee_iterator_get (param_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (param), visitor);
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
	{
		GeeList* error_domain_collection;
		GeeIterator* error_domain_it;
		error_domain_collection = self->priv->error_domains;
		error_domain_it = gee_iterable_iterator (GEE_ITERABLE (error_domain_collection));
		while (gee_iterator_next (error_domain_it)) {
			ValaTypeReference* error_domain;
			error_domain = gee_iterator_get (error_domain_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (error_domain), visitor);
				(error_domain == NULL ? NULL : (error_domain = (g_object_unref (error_domain), NULL)));
			}
		}
		(error_domain_it == NULL ? NULL : (error_domain_it = (g_object_unref (error_domain_it), NULL)));
	}
	if (vala_method_get_body (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_method_get_body (self)), visitor);
	}
}


/**
 * Returns the interface name of this method as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_method_get_cname (ValaMethod* self) {
	const char* _tmp1;
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (self->priv->cname == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->cname = (_tmp0 = vala_method_get_default_cname (self), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->cname, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
}


/**
 * Returns the default interface name of this method as it is used in C
 * code.
 *
 * @return the name to be used in C code by default
 */
static char* vala_method_real_get_default_cname (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (g_str_has_prefix (vala_symbol_get_name (VALA_SYMBOL (self)), "_")) {
		char* _tmp0;
		char* _tmp1;
		_tmp0 = NULL;
		_tmp1 = NULL;
		return (_tmp1 = g_strdup_printf ("_%s%s", (_tmp0 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))), g_utf8_offset_to_pointer (vala_symbol_get_name (VALA_SYMBOL (self)), 1)), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
	} else {
		char* _tmp3;
		char* _tmp4;
		_tmp3 = NULL;
		_tmp4 = NULL;
		return (_tmp4 = g_strdup_printf ("%s%s", (_tmp3 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))), vala_symbol_get_name (VALA_SYMBOL (self))), (_tmp3 = (g_free (_tmp3), NULL)), _tmp4);
	}
}


char* vala_method_get_default_cname (ValaMethod* self) {
	return VALA_METHOD_GET_CLASS (self)->get_default_cname (self);
}


/**
 * Returns the implementation name of this data type as it is used in C
 * code.
 *
 * @return the name to be used in C code
 */
char* vala_method_get_real_cname (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (vala_method_get_base_method (self) != NULL || vala_method_get_base_interface_method (self) != NULL) {
		char* _tmp0;
		char* _tmp1;
		_tmp0 = NULL;
		_tmp1 = NULL;
		return (_tmp1 = g_strdup_printf ("%s_real_%s", (_tmp0 = vala_symbol_get_lower_case_cname (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)), NULL)), vala_symbol_get_name (VALA_SYMBOL (self))), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
	} else {
		return vala_method_get_cname (self);
	}
}


/**
 * Sets the name of this method as it is used in C code.
 *
 * @param cname the name to be used in C code
 */
void vala_method_set_cname (ValaMethod* self, const char* cname) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cname = (_tmp1 = (_tmp0 = cname, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp1);
}


static void vala_method_process_ccode_attribute (ValaMethod* self, ValaAttribute* a) {
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (a == NULL || VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cname")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_method_set_cname (self, (_tmp0 = vala_attribute_get_string (a, "cname")));
		(_tmp0 = (g_free (_tmp0), NULL));
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			char** filename_it;
			int filename_collection_length1;
			filename_collection = g_strsplit (val, ",", 0);
			filename_collection_length1 = -1;
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* _tmp1;
				char* filename;
				_tmp1 = NULL;
				filename = (_tmp1 = *filename_it, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
				{
					vala_member_add_cheader_filename (VALA_MEMBER (self), filename);
					(filename = (g_free (filename), NULL));
				}
			}
			(filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, ((GDestroyNotify) g_free)), NULL));
		}
		(val = (g_free (val), NULL));
	}
	if (vala_attribute_has_argument (a, "vfunc_name")) {
		char* _tmp2;
		_tmp2 = NULL;
		vala_method_set_vfunc_name (self, (_tmp2 = vala_attribute_get_string (a, "vfunc_name")));
		(_tmp2 = (g_free (_tmp2), NULL));
	}
	if (vala_attribute_has_argument (a, "sentinel")) {
		char* _tmp3;
		_tmp3 = NULL;
		vala_method_set_sentinel (self, (_tmp3 = vala_attribute_get_string (a, "sentinel")));
		(_tmp3 = (g_free (_tmp3), NULL));
	}
}


/**
 * Process all associated attributes.
 */
void vala_method_process_attributes (ValaMethod* self) {
	g_return_if_fail (VALA_IS_METHOD (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = VALA_CODE_NODE (self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = a_it->data, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "CCode") == 0) {
					vala_method_process_ccode_attribute (self, a);
				} else {
					if (g_utf8_collate (vala_attribute_get_name (a), "ReturnsModifiedPointer") == 0) {
						vala_method_set_returns_modified_pointer (self, TRUE);
					} else {
						if (g_utf8_collate (vala_attribute_get_name (a), "InstanceLast") == 0) {
							vala_method_set_instance_last (self, TRUE);
						} else {
							if (g_utf8_collate (vala_attribute_get_name (a), "FloatingReference") == 0) {
								vala_type_reference_set_floating_reference (vala_method_get_return_type (self), TRUE);
							} else {
								if (g_utf8_collate (vala_attribute_get_name (a), "NoArrayLength") == 0) {
									vala_method_set_no_array_length (self, TRUE);
								} else {
									if (g_utf8_collate (vala_attribute_get_name (a), "PrintfFormat") == 0) {
										vala_method_set_printf_format (self, TRUE);
									} else {
										if (g_utf8_collate (vala_attribute_get_name (a), "Import") == 0) {
											vala_symbol_set_is_imported (VALA_SYMBOL (self), TRUE);
										}
									}
								}
							}
						}
					}
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


/**
 * Checks whether the arguments and return type of the specified method
 * matches this method.
 *
 * @param m a method
 * @return  true if the specified method is compatible to this method
 */
gboolean vala_method_equals (ValaMethod* self, ValaMethod* m2) {
	GeeCollection* _tmp1;
	GeeIterator* _tmp2;
	GeeIterator* method_params_it;
	GeeCollection* _tmp8;
	GeeIterator* _tmp9;
	GeeIterator* method_error_domains_it;
	gboolean _tmp14;
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	g_return_val_if_fail (VALA_IS_METHOD (m2), FALSE);
	if (!vala_type_reference_equals (vala_method_get_return_type (m2), vala_method_get_return_type (self))) {
		return FALSE;
	}
	_tmp1 = NULL;
	_tmp2 = NULL;
	method_params_it = (_tmp2 = gee_iterable_iterator (GEE_ITERABLE ((_tmp1 = vala_invokable_get_parameters (VALA_INVOKABLE (m2))))), (_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL))), _tmp2);
	{
		GeeList* param_collection;
		GeeIterator* param_it;
		param_collection = self->priv->parameters;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = gee_iterator_get (param_it);
			{
				ValaFormalParameter* _tmp4;
				gboolean _tmp5;
				/* method may not expect less arguments */
				if (!gee_iterator_next (method_params_it)) {
					gboolean _tmp3;
					return (_tmp3 = FALSE, (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), _tmp3);
				}
				_tmp4 = NULL;
				if ((_tmp5 = !vala_type_reference_equals (vala_formal_parameter_get_type_reference ((_tmp4 = gee_iterator_get (method_params_it))), vala_formal_parameter_get_type_reference (param)), (_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL))), _tmp5)) {
					gboolean _tmp6;
					return (_tmp6 = FALSE, (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), _tmp6);
				}
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
	/* method may not expect more arguments */
	if (gee_iterator_next (method_params_it)) {
		gboolean _tmp7;
		return (_tmp7 = FALSE, (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), _tmp7);
	}
	_tmp8 = NULL;
	_tmp9 = NULL;
	method_error_domains_it = (_tmp9 = gee_iterable_iterator (GEE_ITERABLE ((_tmp8 = vala_method_get_error_domains (m2)))), (_tmp8 == NULL ? NULL : (_tmp8 = (g_object_unref (_tmp8), NULL))), _tmp9);
	{
		GeeList* error_domain_collection;
		GeeIterator* error_domain_it;
		error_domain_collection = self->priv->error_domains;
		error_domain_it = gee_iterable_iterator (GEE_ITERABLE (error_domain_collection));
		while (gee_iterator_next (error_domain_it)) {
			ValaTypeReference* error_domain;
			error_domain = gee_iterator_get (error_domain_it);
			{
				ValaTypeReference* _tmp11;
				gboolean _tmp12;
				/* method may not have less error domains */
				if (!gee_iterator_next (method_error_domains_it)) {
					gboolean _tmp10;
					return (_tmp10 = FALSE, (error_domain == NULL ? NULL : (error_domain = (g_object_unref (error_domain), NULL))), (error_domain_it == NULL ? NULL : (error_domain_it = (g_object_unref (error_domain_it), NULL))), (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), (method_error_domains_it == NULL ? NULL : (method_error_domains_it = (g_object_unref (method_error_domains_it), NULL))), _tmp10);
				}
				_tmp11 = NULL;
				if ((_tmp12 = !vala_type_reference_equals ((_tmp11 = gee_iterator_get (method_error_domains_it)), error_domain), (_tmp11 == NULL ? NULL : (_tmp11 = (g_object_unref (_tmp11), NULL))), _tmp12)) {
					gboolean _tmp13;
					return (_tmp13 = FALSE, (error_domain == NULL ? NULL : (error_domain = (g_object_unref (error_domain), NULL))), (error_domain_it == NULL ? NULL : (error_domain_it = (g_object_unref (error_domain_it), NULL))), (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), (method_error_domains_it == NULL ? NULL : (method_error_domains_it = (g_object_unref (method_error_domains_it), NULL))), _tmp13);
				}
				(error_domain == NULL ? NULL : (error_domain = (g_object_unref (error_domain), NULL)));
			}
		}
		(error_domain_it == NULL ? NULL : (error_domain_it = (g_object_unref (error_domain_it), NULL)));
	}
	return (_tmp14 = TRUE, (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), (method_error_domains_it == NULL ? NULL : (method_error_domains_it = (g_object_unref (method_error_domains_it), NULL))), _tmp14);
	(method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL)));
	(method_error_domains_it == NULL ? NULL : (method_error_domains_it = (g_object_unref (method_error_domains_it), NULL)));
}


/**
 * Adds an error domain to this method.
 *
 * @param error_domain an error domain
 */
void vala_method_add_error_domain (ValaMethod* self, ValaTypeReference* error_domain) {
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (error_domain));
	gee_collection_add (GEE_COLLECTION (self->priv->error_domains), error_domain);
}


/**
 * Returns a copy of the list of error domains of this method.
 *
 * @return list of error domains
 */
GeeCollection* vala_method_get_error_domains (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->error_domains)));
}


ValaTypeReference* vala_method_get_return_type (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_return_type;
}


void vala_method_set_return_type (ValaMethod* self, ValaTypeReference* value) {
	ValaTypeReference* _tmp2;
	ValaTypeReference* _tmp1;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_return_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL))), _tmp2);
}


ValaBlock* vala_method_get_body (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_body;
}


void vala_method_set_body (ValaMethod* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_body == NULL ? NULL : (self->priv->_body = (g_object_unref (self->priv->_body), NULL))), _tmp2);
}


gboolean vala_method_get_instance (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_instance;
}


void vala_method_set_instance (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_instance = value;
}


char* vala_method_get_vfunc_name (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (self->priv->_vfunc_name == NULL) {
		char* _tmp1;
		const char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->_vfunc_name = (_tmp1 = (_tmp0 = vala_symbol_get_name (VALA_SYMBOL (self)), (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->_vfunc_name = (g_free (self->priv->_vfunc_name), NULL)), _tmp1);
	}
	return self->priv->_vfunc_name;
}


void vala_method_set_vfunc_name (ValaMethod* self, const char* value) {
	char* _tmp3;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp3 = NULL;
	self->priv->_vfunc_name = (_tmp3 = g_strdup (value), (self->priv->_vfunc_name = (g_free (self->priv->_vfunc_name), NULL)), _tmp3);
}


char* vala_method_get_sentinel (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (self->priv->_sentinel == NULL) {
		return VALA_METHOD_DEFAULT_SENTINEL;
	}
	return self->priv->_sentinel;
}


void vala_method_set_sentinel (ValaMethod* self, const char* value) {
	char* _tmp2;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp2 = NULL;
	self->priv->_sentinel = (_tmp2 = g_strdup (value), (self->priv->_sentinel = (g_free (self->priv->_sentinel), NULL)), _tmp2);
}


gboolean vala_method_get_is_abstract (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_is_abstract;
}


void vala_method_set_is_abstract (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_is_abstract = value;
}


gboolean vala_method_get_is_virtual (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_is_virtual;
}


void vala_method_set_is_virtual (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_is_virtual = value;
}


gboolean vala_method_get_overrides (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_overrides;
}


void vala_method_set_overrides (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_overrides = value;
}


gboolean vala_method_get_returns_modified_pointer (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_returns_modified_pointer;
}


void vala_method_set_returns_modified_pointer (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_returns_modified_pointer = value;
}


gboolean vala_method_get_instance_last (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_instance_last;
}


void vala_method_set_instance_last (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_instance_last = value;
}


ValaMethod* vala_method_get_base_method (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_base_method;
}


void vala_method_set_base_method (ValaMethod* self, ValaMethod* value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_base_method = value;
}


ValaMethod* vala_method_get_base_interface_method (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_base_interface_method;
}


void vala_method_set_base_interface_method (ValaMethod* self, ValaMethod* value) {
	ValaMethod* _tmp2;
	ValaMethod* _tmp1;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_base_interface_method = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_base_interface_method == NULL ? NULL : (self->priv->_base_interface_method = (g_object_unref (self->priv->_base_interface_method), NULL))), _tmp2);
}


ValaFormalParameter* vala_method_get_this_parameter (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_this_parameter;
}


void vala_method_set_this_parameter (ValaMethod* self, ValaFormalParameter* value) {
	ValaFormalParameter* _tmp2;
	ValaFormalParameter* _tmp1;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_this_parameter = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (g_object_unref (self->priv->_this_parameter), NULL))), _tmp2);
}


gboolean vala_method_get_no_array_length (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_no_array_length;
}


void vala_method_set_no_array_length (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_no_array_length = value;
	{
		GeeList* param_collection;
		GeeIterator* param_it;
		param_collection = self->priv->parameters;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = gee_iterator_get (param_it);
			{
				vala_formal_parameter_set_no_array_length (param, value);
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
}


gboolean vala_method_get_printf_format (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_printf_format;
}


void vala_method_set_printf_format (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_printf_format = value;
}


static void vala_method_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaMethod * self;
	self = VALA_METHOD (object);
	switch (property_id) {
		case VALA_METHOD_RETURN_TYPE:
		g_value_set_object (value, vala_method_get_return_type (self));
		break;
		case VALA_METHOD_BODY:
		g_value_set_object (value, vala_method_get_body (self));
		break;
		case VALA_METHOD_INSTANCE:
		g_value_set_boolean (value, vala_method_get_instance (self));
		break;
		case VALA_METHOD_VFUNC_NAME:
		g_value_set_string (value, vala_method_get_vfunc_name (self));
		break;
		case VALA_METHOD_SENTINEL:
		g_value_set_string (value, vala_method_get_sentinel (self));
		break;
		case VALA_METHOD_IS_ABSTRACT:
		g_value_set_boolean (value, vala_method_get_is_abstract (self));
		break;
		case VALA_METHOD_IS_VIRTUAL:
		g_value_set_boolean (value, vala_method_get_is_virtual (self));
		break;
		case VALA_METHOD_OVERRIDES:
		g_value_set_boolean (value, vala_method_get_overrides (self));
		break;
		case VALA_METHOD_RETURNS_MODIFIED_POINTER:
		g_value_set_boolean (value, vala_method_get_returns_modified_pointer (self));
		break;
		case VALA_METHOD_INSTANCE_LAST:
		g_value_set_boolean (value, vala_method_get_instance_last (self));
		break;
		case VALA_METHOD_BASE_METHOD:
		g_value_set_object (value, vala_method_get_base_method (self));
		break;
		case VALA_METHOD_BASE_INTERFACE_METHOD:
		g_value_set_object (value, vala_method_get_base_interface_method (self));
		break;
		case VALA_METHOD_THIS_PARAMETER:
		g_value_set_object (value, vala_method_get_this_parameter (self));
		break;
		case VALA_METHOD_NO_ARRAY_LENGTH:
		g_value_set_boolean (value, vala_method_get_no_array_length (self));
		break;
		case VALA_METHOD_PRINTF_FORMAT:
		g_value_set_boolean (value, vala_method_get_printf_format (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_method_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaMethod * self;
	self = VALA_METHOD (object);
	switch (property_id) {
		case VALA_METHOD_RETURN_TYPE:
		vala_method_set_return_type (self, g_value_get_object (value));
		break;
		case VALA_METHOD_BODY:
		vala_method_set_body (self, g_value_get_object (value));
		break;
		case VALA_METHOD_INSTANCE:
		vala_method_set_instance (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_VFUNC_NAME:
		vala_method_set_vfunc_name (self, g_value_get_string (value));
		break;
		case VALA_METHOD_SENTINEL:
		vala_method_set_sentinel (self, g_value_get_string (value));
		break;
		case VALA_METHOD_IS_ABSTRACT:
		vala_method_set_is_abstract (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_IS_VIRTUAL:
		vala_method_set_is_virtual (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_OVERRIDES:
		vala_method_set_overrides (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_RETURNS_MODIFIED_POINTER:
		vala_method_set_returns_modified_pointer (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_INSTANCE_LAST:
		vala_method_set_instance_last (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_BASE_METHOD:
		vala_method_set_base_method (self, g_value_get_object (value));
		break;
		case VALA_METHOD_BASE_INTERFACE_METHOD:
		vala_method_set_base_interface_method (self, g_value_get_object (value));
		break;
		case VALA_METHOD_THIS_PARAMETER:
		vala_method_set_this_parameter (self, g_value_get_object (value));
		break;
		case VALA_METHOD_NO_ARRAY_LENGTH:
		vala_method_set_no_array_length (self, g_value_get_boolean (value));
		break;
		case VALA_METHOD_PRINTF_FORMAT:
		vala_method_set_printf_format (self, g_value_get_boolean (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_method_class_init (ValaMethodClass * klass) {
	vala_method_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaMethodPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_method_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_method_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_method_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_method_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_method_real_accept_children;
	VALA_METHOD_CLASS (klass)->get_default_cname = vala_method_real_get_default_cname;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_RETURN_TYPE, g_param_spec_object ("return-type", "return-type", "return-type", VALA_TYPE_TYPE_REFERENCE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_BODY, g_param_spec_object ("body", "body", "body", VALA_TYPE_BLOCK, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_INSTANCE, g_param_spec_boolean ("instance", "instance", "instance", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_VFUNC_NAME, g_param_spec_string ("vfunc-name", "vfunc-name", "vfunc-name", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_SENTINEL, g_param_spec_string ("sentinel", "sentinel", "sentinel", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_IS_ABSTRACT, g_param_spec_boolean ("is-abstract", "is-abstract", "is-abstract", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_IS_VIRTUAL, g_param_spec_boolean ("is-virtual", "is-virtual", "is-virtual", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_OVERRIDES, g_param_spec_boolean ("overrides", "overrides", "overrides", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_RETURNS_MODIFIED_POINTER, g_param_spec_boolean ("returns-modified-pointer", "returns-modified-pointer", "returns-modified-pointer", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_INSTANCE_LAST, g_param_spec_boolean ("instance-last", "instance-last", "instance-last", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_BASE_METHOD, g_param_spec_object ("base-method", "base-method", "base-method", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_BASE_INTERFACE_METHOD, g_param_spec_object ("base-interface-method", "base-interface-method", "base-interface-method", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_THIS_PARAMETER, g_param_spec_object ("this-parameter", "this-parameter", "this-parameter", VALA_TYPE_FORMAL_PARAMETER, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_NO_ARRAY_LENGTH, g_param_spec_boolean ("no-array-length", "no-array-length", "no-array-length", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_METHOD_PRINTF_FORMAT, g_param_spec_boolean ("printf-format", "printf-format", "printf-format", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_method_vala_invokable_interface_init (ValaInvokableIface * iface) {
	vala_method_vala_invokable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_parameters = vala_method_real_get_parameters;
	iface->get_return_type = vala_method_real_get_return_type;
	iface->is_invokable = vala_method_real_is_invokable;
}


static void vala_method_init (ValaMethod * self) {
	self->priv = VALA_METHOD_GET_PRIVATE (self);
	self->priv->_instance = TRUE;
	self->priv->parameters = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->error_domains = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
}


static void vala_method_dispose (GObject * obj) {
	ValaMethod * self;
	self = VALA_METHOD (obj);
	(self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL)));
	(self->priv->_body == NULL ? NULL : (self->priv->_body = (g_object_unref (self->priv->_body), NULL)));
	(self->priv->_base_interface_method == NULL ? NULL : (self->priv->_base_interface_method = (g_object_unref (self->priv->_base_interface_method), NULL)));
	(self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (g_object_unref (self->priv->_this_parameter), NULL)));
	(self->priv->parameters == NULL ? NULL : (self->priv->parameters = (g_object_unref (self->priv->parameters), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	(self->priv->_vfunc_name = (g_free (self->priv->_vfunc_name), NULL));
	(self->priv->_sentinel = (g_free (self->priv->_sentinel), NULL));
	(self->priv->error_domains == NULL ? NULL : (self->priv->error_domains = (g_object_unref (self->priv->error_domains), NULL)));
	G_OBJECT_CLASS (vala_method_parent_class)->dispose (obj);
}


GType vala_method_get_type (void) {
	static GType vala_method_type_id = 0;
	if (G_UNLIKELY (vala_method_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaMethodClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_method_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaMethod), 0, (GInstanceInitFunc) vala_method_init };
		static const GInterfaceInfo vala_invokable_info = { (GInterfaceInitFunc) vala_method_vala_invokable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_method_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaMethod", &g_define_type_info, 0);
		g_type_add_interface_static (vala_method_type_id, VALA_TYPE_INVOKABLE, &vala_invokable_info);
	}
	return vala_method_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}




