/* valaobjectcreationexpression.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaobjectcreationexpression.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <vala/valatypereference.h>
#include <vala/valamethod.h>
#include <vala/valamemberaccess.h>
#include <vala/valasourcereference.h>
#include <vala/valamemberinitializer.h>
#include <vala/valacodevisitor.h>

struct _ValaObjectCreationExpressionPrivate {
	ValaTypeReference* _type_reference;
	ValaMethod* _constructor;
	ValaMemberAccess* _member_name;
	GeeList* argument_list;
	GeeList* object_initializer;
};
#define VALA_OBJECT_CREATION_EXPRESSION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_OBJECT_CREATION_EXPRESSION, ValaObjectCreationExpressionPrivate))
enum  {
	VALA_OBJECT_CREATION_EXPRESSION_DUMMY_PROPERTY,
	VALA_OBJECT_CREATION_EXPRESSION_TYPE_REFERENCE,
	VALA_OBJECT_CREATION_EXPRESSION_CONSTRUCTOR,
	VALA_OBJECT_CREATION_EXPRESSION_MEMBER_NAME
};
static void vala_object_creation_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_object_creation_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_object_creation_expression_real_replace (ValaCodeNode* base, ValaCodeNode* old_node, ValaCodeNode* new_node);
static gpointer vala_object_creation_expression_parent_class = NULL;
static void vala_object_creation_expression_dispose (GObject * obj);


/**
 * Creates a new object creation expression.
 *
 * @param member_name      object type to create
 * @param source_reference reference to source code
 * @return                 newly created object creation expression
 */
ValaObjectCreationExpression* vala_object_creation_expression_new (ValaMemberAccess* member_name, ValaSourceReference* source_reference) {
	ValaObjectCreationExpression * self;
	g_return_val_if_fail (VALA_IS_MEMBER_ACCESS (member_name), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_OBJECT_CREATION_EXPRESSION, 0, NULL);
	vala_object_creation_expression_set_member_name (self, member_name);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Appends the specified expression to the list of arguments.
 *
 * @param arg an argument
 */
void vala_object_creation_expression_add_argument (ValaObjectCreationExpression* self, ValaExpression* arg) {
	g_return_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self));
	g_return_if_fail (VALA_IS_EXPRESSION (arg));
	gee_collection_add (GEE_COLLECTION (self->priv->argument_list), arg);
	vala_code_node_set_parent_node (VALA_CODE_NODE (arg), VALA_CODE_NODE (self));
}


/**
 * Returns a copy of the argument list.
 *
 * @return argument list
 */
GeeCollection* vala_object_creation_expression_get_argument_list (ValaObjectCreationExpression* self) {
	g_return_val_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->argument_list)));
}


/**
 * Appends the specified member initializer to the object initializer.
 *
 * @param init a member initializer
 */
void vala_object_creation_expression_add_member_initializer (ValaObjectCreationExpression* self, ValaMemberInitializer* init) {
	g_return_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self));
	g_return_if_fail (VALA_IS_MEMBER_INITIALIZER (init));
	gee_collection_add (GEE_COLLECTION (self->priv->object_initializer), init);
	vala_code_node_set_parent_node (VALA_CODE_NODE (init), VALA_CODE_NODE (self));
}


/**
 * Returns the object initializer.
 *
 * @return member initializer list
 */
GeeCollection* vala_object_creation_expression_get_object_initializer (ValaObjectCreationExpression* self) {
	g_return_val_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->object_initializer)));
}


static void vala_object_creation_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaObjectCreationExpression * self;
	self = VALA_OBJECT_CREATION_EXPRESSION (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_object_creation_expression (visitor, self);
}


static void vala_object_creation_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaObjectCreationExpression * self;
	self = VALA_OBJECT_CREATION_EXPRESSION (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	if (vala_object_creation_expression_get_type_reference (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_object_creation_expression_get_type_reference (self)), visitor);
	}
	if (vala_object_creation_expression_get_member_name (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_object_creation_expression_get_member_name (self)), visitor);
	}
	{
		GeeList* arg_collection;
		GeeIterator* arg_it;
		arg_collection = self->priv->argument_list;
		arg_it = gee_iterable_iterator (GEE_ITERABLE (arg_collection));
		while (gee_iterator_next (arg_it)) {
			ValaExpression* arg;
			arg = gee_iterator_get (arg_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (arg), visitor);
				(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
			}
		}
		(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
	}
	{
		GeeList* init_collection;
		GeeIterator* init_it;
		init_collection = self->priv->object_initializer;
		init_it = gee_iterable_iterator (GEE_ITERABLE (init_collection));
		while (gee_iterator_next (init_it)) {
			ValaMemberInitializer* init;
			init = gee_iterator_get (init_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (init), visitor);
				(init == NULL ? NULL : (init = (g_object_unref (init), NULL)));
			}
		}
		(init_it == NULL ? NULL : (init_it = (g_object_unref (init_it), NULL)));
	}
}


static void vala_object_creation_expression_real_replace (ValaCodeNode* base, ValaCodeNode* old_node, ValaCodeNode* new_node) {
	ValaObjectCreationExpression * self;
	gint index;
	self = VALA_OBJECT_CREATION_EXPRESSION (base);
	g_return_if_fail (VALA_IS_CODE_NODE (old_node));
	g_return_if_fail (VALA_IS_CODE_NODE (new_node));
	index = gee_list_index_of (self->priv->argument_list, old_node);
	if (index >= 0 && vala_code_node_get_parent_node (new_node) == NULL) {
		gee_list_set (((GeeList*) self->priv->argument_list), index, VALA_EXPRESSION (new_node));
		vala_code_node_set_parent_node (new_node, VALA_CODE_NODE (self));
	}
}


ValaTypeReference* vala_object_creation_expression_get_type_reference (ValaObjectCreationExpression* self) {
	g_return_val_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self), NULL);
	return self->priv->_type_reference;
}


void vala_object_creation_expression_set_type_reference (ValaObjectCreationExpression* self, ValaTypeReference* value) {
	ValaTypeReference* _tmp2;
	ValaTypeReference* _tmp1;
	g_return_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_type_reference = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_type_reference == NULL ? NULL : (self->priv->_type_reference = (g_object_unref (self->priv->_type_reference), NULL))), _tmp2);
}


ValaMethod* vala_object_creation_expression_get_constructor (ValaObjectCreationExpression* self) {
	g_return_val_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self), NULL);
	return self->priv->_constructor;
}


void vala_object_creation_expression_set_constructor (ValaObjectCreationExpression* self, ValaMethod* value) {
	ValaMethod* _tmp2;
	ValaMethod* _tmp1;
	g_return_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_constructor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_constructor == NULL ? NULL : (self->priv->_constructor = (g_object_unref (self->priv->_constructor), NULL))), _tmp2);
}


ValaMemberAccess* vala_object_creation_expression_get_member_name (ValaObjectCreationExpression* self) {
	g_return_val_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self), NULL);
	return self->priv->_member_name;
}


void vala_object_creation_expression_set_member_name (ValaObjectCreationExpression* self, ValaMemberAccess* value) {
	ValaMemberAccess* _tmp2;
	ValaMemberAccess* _tmp1;
	g_return_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_member_name = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_member_name == NULL ? NULL : (self->priv->_member_name = (g_object_unref (self->priv->_member_name), NULL))), _tmp2);
}


static void vala_object_creation_expression_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaObjectCreationExpression * self;
	self = VALA_OBJECT_CREATION_EXPRESSION (object);
	switch (property_id) {
		case VALA_OBJECT_CREATION_EXPRESSION_TYPE_REFERENCE:
		g_value_set_object (value, vala_object_creation_expression_get_type_reference (self));
		break;
		case VALA_OBJECT_CREATION_EXPRESSION_CONSTRUCTOR:
		g_value_set_object (value, vala_object_creation_expression_get_constructor (self));
		break;
		case VALA_OBJECT_CREATION_EXPRESSION_MEMBER_NAME:
		g_value_set_object (value, vala_object_creation_expression_get_member_name (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_object_creation_expression_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaObjectCreationExpression * self;
	self = VALA_OBJECT_CREATION_EXPRESSION (object);
	switch (property_id) {
		case VALA_OBJECT_CREATION_EXPRESSION_TYPE_REFERENCE:
		vala_object_creation_expression_set_type_reference (self, g_value_get_object (value));
		break;
		case VALA_OBJECT_CREATION_EXPRESSION_CONSTRUCTOR:
		vala_object_creation_expression_set_constructor (self, g_value_get_object (value));
		break;
		case VALA_OBJECT_CREATION_EXPRESSION_MEMBER_NAME:
		vala_object_creation_expression_set_member_name (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_object_creation_expression_class_init (ValaObjectCreationExpressionClass * klass) {
	vala_object_creation_expression_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaObjectCreationExpressionPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_object_creation_expression_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_object_creation_expression_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_object_creation_expression_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_object_creation_expression_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_object_creation_expression_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace = vala_object_creation_expression_real_replace;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_OBJECT_CREATION_EXPRESSION_TYPE_REFERENCE, g_param_spec_object ("type-reference", "type-reference", "type-reference", VALA_TYPE_TYPE_REFERENCE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_OBJECT_CREATION_EXPRESSION_CONSTRUCTOR, g_param_spec_object ("constructor", "constructor", "constructor", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_OBJECT_CREATION_EXPRESSION_MEMBER_NAME, g_param_spec_object ("member-name", "member-name", "member-name", VALA_TYPE_MEMBER_ACCESS, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_object_creation_expression_init (ValaObjectCreationExpression * self) {
	self->priv = VALA_OBJECT_CREATION_EXPRESSION_GET_PRIVATE (self);
	self->priv->argument_list = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->object_initializer = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
}


static void vala_object_creation_expression_dispose (GObject * obj) {
	ValaObjectCreationExpression * self;
	self = VALA_OBJECT_CREATION_EXPRESSION (obj);
	(self->priv->_type_reference == NULL ? NULL : (self->priv->_type_reference = (g_object_unref (self->priv->_type_reference), NULL)));
	(self->priv->_constructor == NULL ? NULL : (self->priv->_constructor = (g_object_unref (self->priv->_constructor), NULL)));
	(self->priv->_member_name == NULL ? NULL : (self->priv->_member_name = (g_object_unref (self->priv->_member_name), NULL)));
	(self->priv->argument_list == NULL ? NULL : (self->priv->argument_list = (g_object_unref (self->priv->argument_list), NULL)));
	(self->priv->object_initializer == NULL ? NULL : (self->priv->object_initializer = (g_object_unref (self->priv->object_initializer), NULL)));
	G_OBJECT_CLASS (vala_object_creation_expression_parent_class)->dispose (obj);
}


GType vala_object_creation_expression_get_type (void) {
	static GType vala_object_creation_expression_type_id = 0;
	if (G_UNLIKELY (vala_object_creation_expression_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaObjectCreationExpressionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_object_creation_expression_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaObjectCreationExpression), 0, (GInstanceInitFunc) vala_object_creation_expression_init };
		vala_object_creation_expression_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaObjectCreationExpression", &g_define_type_info, 0);
	}
	return vala_object_creation_expression_type_id;
}




