/* valagidlparser.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */

#include <vapigen/valagidlparser.h>
#include <vala/valasourcereference.h>
#include <vala/valadatatype.h>
#include <gee/map.h>
#include <gee/set.h>
#include <gee/hashmap.h>
#include <vala/valacodenode.h>
#include <stdlib.h>
#include <string.h>
#include <glib/gstdio.h>
#include <vala/valareport.h>
#include <gidlparser.h>
#include <gidlmodule.h>
#include <stdio.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <gidlnode.h>
#include <vala/valaconstant.h>
#include <vala/valamethod.h>
#include <vala/valatypereference.h>
#include <vala/valaformalparameter.h>
#include <vala/valafield.h>
#include <vala/valaenumvalue.h>
#include <gee/hashset.h>
#include <vala/valaproperty.h>
#include <vala/valasignal.h>
#include <gee/collection.h>
#include <gmetadata.h>
#include <vala/valacreationmethod.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valablock.h>
#include <vala/valaexpression.h>

struct _ValaGIdlParserPrivate {
	ValaCodeContext* context;
	ValaSourceFile* current_source_file;
	ValaSourceReference* current_source_reference;
	ValaNamespace* current_namespace;
	ValaDataType* current_data_type;
	GeeMap* codenode_attributes_map;
	GeeMap* codenode_attributes_patterns;
	GeeSet* current_type_symbol_set;
	GeeMap* cname_type_map;
};
#define VALA_GIDL_PARSER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_GIDL_PARSER, ValaGIdlParserPrivate))
enum  {
	VALA_GIDL_PARSER_DUMMY_PROPERTY
};
static void vala_gidl_parser_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns);
static void vala_gidl_parser_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_gidl_parser_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_gidl_parser_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_gidl_parser_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en);
static void vala_gidl_parser_real_visit_callback (ValaCodeVisitor* base, ValaCallback* cb);
static void vala_gidl_parser_visit_type (ValaGIdlParser* self, ValaDataType* t);
static void vala_gidl_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file);
static void vala_gidl_parser_parse_file (ValaGIdlParser* self, ValaSourceFile* source_file);
static char* vala_gidl_parser_fix_type_name (ValaGIdlParser* self, const char* type_name, ValaNamespace* ns);
static char* vala_gidl_parser_fix_const_name (ValaGIdlParser* self, const char* const_name, ValaNamespace* ns);
static ValaNamespace* vala_gidl_parser_parse_module (ValaGIdlParser* self, GIdlModule* module);
static ValaCallback* vala_gidl_parser_parse_callback (ValaGIdlParser* self, GIdlNodeFunction* f_node);
static gboolean vala_gidl_parser_is_reference_type (ValaGIdlParser* self, const char* cname);
static void vala_gidl_parser_parse_struct (ValaGIdlParser* self, GIdlNodeStruct* st_node, ValaNamespace* ns, GIdlModule* module);
static void vala_gidl_parser_parse_boxed (ValaGIdlParser* self, GIdlNodeBoxed* boxed_node, ValaNamespace* ns, GIdlModule* module);
static ValaEnum* vala_gidl_parser_parse_enum (ValaGIdlParser* self, GIdlNodeEnum* en_node);
static void vala_gidl_parser_parse_object (ValaGIdlParser* self, GIdlNodeInterface* node, ValaNamespace* ns, GIdlModule* module);
static void vala_gidl_parser_parse_interface (ValaGIdlParser* self, GIdlNodeInterface* node, ValaNamespace* ns, GIdlModule* module);
static ValaTypeReference* vala_gidl_parser_parse_type (ValaGIdlParser* self, GIdlNodeType* type_node);
static gboolean vala_gidl_parser_is_value_type (ValaGIdlParser* self, const char* type_name);
static void vala_gidl_parser_parse_type_string (ValaGIdlParser* self, ValaTypeReference* type, const char* n);
static ValaTypeReference* vala_gidl_parser_parse_param (ValaGIdlParser* self, GIdlNodeParam* param);
static ValaMethod* vala_gidl_parser_parse_function (ValaGIdlParser* self, GIdlNodeFunction* f, gboolean is_virtual, gboolean is_interface);
static char* vala_gidl_parser_fix_prop_name (ValaGIdlParser* self, const char* name);
static ValaProperty* vala_gidl_parser_parse_property (ValaGIdlParser* self, GIdlNodeProperty* prop_node);
static ValaConstant* vala_gidl_parser_parse_constant (ValaGIdlParser* self, GIdlNodeConstant* const_node);
static ValaField* vala_gidl_parser_parse_field (ValaGIdlParser* self, GIdlNodeField* field_node);
static char** vala_gidl_parser_get_attributes (ValaGIdlParser* self, const char* codenode);
static char* vala_gidl_parser_eval (ValaGIdlParser* self, const char* s);
static ValaSignal* vala_gidl_parser_parse_signal (ValaGIdlParser* self, GIdlNodeSignal* sig_node);
static gpointer vala_gidl_parser_parent_class = NULL;
static void vala_gidl_parser_dispose (GObject * obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);


/**
 * Parse all source files in the specified code context and build a
 * code tree.
 *
 * @param context a code context
 */
void vala_gidl_parser_parse (ValaGIdlParser* self, ValaCodeContext* context) {
	GeeMap* _tmp0;
	ValaCodeContext* _tmp1;
	GeeMap* _tmp2;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (VALA_IS_CODE_CONTEXT (context));
	_tmp0 = NULL;
	self->priv->cname_type_map = (_tmp0 = GEE_MAP (gee_hash_map_new (g_strdup, g_free, g_object_ref, g_object_unref, g_str_hash, g_str_equal, g_direct_equal)), (self->priv->cname_type_map == NULL ? NULL : (self->priv->cname_type_map = (g_object_unref (self->priv->cname_type_map), NULL))), _tmp0);
	_tmp1 = NULL;
	self->priv->context = (_tmp1 = g_object_ref (context), (self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL))), _tmp1);
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
	_tmp2 = NULL;
	self->priv->cname_type_map = (_tmp2 = NULL, (self->priv->cname_type_map == NULL ? NULL : (self->priv->cname_type_map = (g_object_unref (self->priv->cname_type_map), NULL))), _tmp2);
}


static void vala_gidl_parser_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns) {
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (base);
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
	vala_code_node_accept_children (VALA_CODE_NODE (ns), VALA_CODE_VISITOR (self));
}


static void vala_gidl_parser_real_visit_class (ValaCodeVisitor* base, ValaClass* cl) {
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (base);
	g_return_if_fail (VALA_IS_CLASS (cl));
	vala_gidl_parser_visit_type (self, VALA_DATA_TYPE (cl));
}


static void vala_gidl_parser_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st) {
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (base);
	g_return_if_fail (VALA_IS_STRUCT (st));
	vala_gidl_parser_visit_type (self, VALA_DATA_TYPE (st));
}


static void vala_gidl_parser_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface) {
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (base);
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	vala_gidl_parser_visit_type (self, VALA_DATA_TYPE (iface));
}


static void vala_gidl_parser_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en) {
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (base);
	g_return_if_fail (VALA_IS_ENUM (en));
	vala_gidl_parser_visit_type (self, VALA_DATA_TYPE (en));
}


static void vala_gidl_parser_real_visit_callback (ValaCodeVisitor* base, ValaCallback* cb) {
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (base);
	g_return_if_fail (VALA_IS_CALLBACK (cb));
	vala_gidl_parser_visit_type (self, VALA_DATA_TYPE (cb));
}


static void vala_gidl_parser_visit_type (ValaGIdlParser* self, ValaDataType* t) {
	char* _tmp0;
	gboolean _tmp1;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (VALA_IS_DATA_TYPE (t));
	_tmp0 = NULL;
	if ((_tmp1 = !gee_map_contains (self->priv->cname_type_map, (_tmp0 = vala_data_type_get_cname (t, FALSE))), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1)) {
		gee_map_set (((GeeMap*) self->priv->cname_type_map), vala_data_type_get_cname (t, FALSE), t);
	}
}


static void vala_gidl_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file) {
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (base);
	g_return_if_fail (VALA_IS_SOURCE_FILE (source_file));
	if (g_str_has_suffix (vala_source_file_get_filename (source_file), ".gidl")) {
		vala_gidl_parser_parse_file (self, source_file);
	}
}


static void vala_gidl_parser_parse_file (ValaGIdlParser* self, ValaSourceFile* source_file) {
	GError * inner_error;
	char* _tmp0;
	char* _tmp1;
	char* metadata_filename;
	ValaSourceFile* _tmp2;
	GeeMap* _tmp3;
	GeeMap* _tmp4;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (VALA_IS_SOURCE_FILE (source_file));
	inner_error = NULL;
	_tmp0 = NULL;
	_tmp1 = NULL;
	metadata_filename = (_tmp1 = g_strdup_printf ("%s.metadata", (_tmp0 = g_strndup (vala_source_file_get_filename (source_file), strlen (vala_source_file_get_filename (source_file)) - strlen (".gidl")))), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
	_tmp2 = NULL;
	self->priv->current_source_file = (_tmp2 = g_object_ref (source_file), (self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (g_object_unref (self->priv->current_source_file), NULL))), _tmp2);
	_tmp3 = NULL;
	self->priv->codenode_attributes_map = (_tmp3 = GEE_MAP (gee_hash_map_new (g_strdup, g_free, g_strdup, g_free, g_str_hash, g_str_equal, g_direct_equal)), (self->priv->codenode_attributes_map == NULL ? NULL : (self->priv->codenode_attributes_map = (g_object_unref (self->priv->codenode_attributes_map), NULL))), _tmp3);
	_tmp4 = NULL;
	self->priv->codenode_attributes_patterns = (_tmp4 = GEE_MAP (gee_hash_map_new (NULL, NULL, g_strdup, g_free, g_direct_hash, g_pattern_spec_equal, g_direct_equal)), (self->priv->codenode_attributes_patterns == NULL ? NULL : (self->priv->codenode_attributes_patterns = (g_object_unref (self->priv->codenode_attributes_patterns), NULL))), _tmp4);
	if (g_file_test (metadata_filename, G_FILE_TEST_EXISTS)) {
		{
			char* metadata;
			gulong metadata_len;
			metadata = NULL;
			metadata_len = 0UL;
			g_file_get_contents (metadata_filename, &metadata, &metadata_len, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == G_FILE_ERROR) {
					goto __catch0_g_file_error;
				}
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
			}
			{
				char** line_collection;
				char** line_it;
				int line_collection_length1;
				line_collection = g_strsplit (metadata, "\n", 0);
				line_collection_length1 = -1;
				for (line_it = line_collection; *line_it != NULL; line_it = line_it + 1) {
					const char* _tmp7;
					char* line;
					_tmp7 = NULL;
					line = (_tmp7 = *line_it, (_tmp7 == NULL ? NULL : g_strdup (_tmp7)));
					{
						char** _tmp5;
						gint tokens_length1;
						char** tokens;
						if (g_str_has_prefix (line, "#")) {
							(line = (g_free (line), NULL));
							continue;
						}
						/* ignore comment lines*/
						_tmp5 = NULL;
						tokens = (_tmp5 = g_strsplit (line, " ", 2), tokens_length1 = -1, _tmp5);
						if (NULL == tokens[0]) {
							(line = (g_free (line), NULL));
							(tokens = (_vala_array_free (tokens, tokens_length1, ((GDestroyNotify) g_free)), NULL));
							continue;
						}
						if (NULL != g_utf8_strchr (tokens[0], -1, '*')) {
							GPatternSpec* pattern;
							GPatternSpec* _tmp6;
							pattern = g_pattern_spec_new (tokens[0]);
							_tmp6 = NULL;
							gee_map_set (((GeeMap*) self->priv->codenode_attributes_patterns), (_tmp6 = pattern, pattern = NULL, _tmp6), tokens[0]);
							(pattern == NULL ? NULL : (pattern = (g_pattern_spec_free (pattern), NULL)));
						}
						gee_map_set (((GeeMap*) self->priv->codenode_attributes_map), tokens[0], tokens[1]);
						(line = (g_free (line), NULL));
						(tokens = (_vala_array_free (tokens, tokens_length1, ((GDestroyNotify) g_free)), NULL));
					}
				}
				(line_collection = (_vala_array_free (line_collection, line_collection_length1, ((GDestroyNotify) g_free)), NULL));
			}
			(metadata = (g_free (metadata), NULL));
		}
		goto __finally0;
		__catch0_g_file_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				char* _tmp8;
				_tmp8 = NULL;
				vala_report_error (NULL, (_tmp8 = g_strdup_printf ("Unable to read metadata file: %s", e->message)));
				(_tmp8 = (g_free (_tmp8), NULL));
			}
		}
		__finally0:
		;
	}
	{
		GList* modules;
		ValaSourceReference* _tmp9;
		modules = g_idl_parse_file (vala_source_file_get_filename (source_file), &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == G_MARKUP_ERROR) {
				goto __catch1_g_markup_error;
			}
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
		}
		_tmp9 = NULL;
		self->priv->current_source_reference = (_tmp9 = vala_source_reference_new (source_file, 0, 0, 0, 0), (self->priv->current_source_reference == NULL ? NULL : (self->priv->current_source_reference = (g_object_unref (self->priv->current_source_reference), NULL))), _tmp9);
		{
			GList* module_collection;
			GList* module_it;
			module_collection = modules;
			for (module_it = module_collection; module_it != NULL; module_it = module_it->next) {
				GIdlModule* module;
				module = module_it->data;
				{
					ValaNamespace* ns;
					ns = vala_gidl_parser_parse_module (self, module);
					if (ns != NULL) {
						vala_namespace_add_namespace (vala_code_context_get_root (self->priv->context), ns);
					}
					(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
				}
			}
		}
		(modules == NULL ? NULL : (modules = (g_list_foreach (modules, ((GFunc) g_idl_module_free), NULL), g_list_free (modules), NULL)));
	}
	goto __finally1;
	__catch1_g_markup_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			fprintf (stdout, "error parsing GIDL file: %s\n", e->message);
		}
	}
	__finally1:
	;
	(metadata_filename = (g_free (metadata_filename), NULL));
}


static char* vala_gidl_parser_fix_type_name (ValaGIdlParser* self, const char* type_name, ValaNamespace* ns) {
	char** _tmp0;
	gint attributes_length1;
	char** attributes;
	char* _tmp12;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (type_name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_NAMESPACE (ns), NULL);
	_tmp0 = NULL;
	attributes = (_tmp0 = vala_gidl_parser_get_attributes (self, type_name), attributes_length1 = -1, _tmp0);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp3;
				char* attr;
				_tmp3 = NULL;
				attr = (_tmp3 = attr_collection[attr_it], (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
				{
					char** _tmp1;
					gint nv_length1;
					char** nv;
					_tmp1 = NULL;
					nv = (_tmp1 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp1);
					if (g_utf8_collate (nv[0], "name") == 0) {
						char* _tmp2;
						_tmp2 = NULL;
						return (_tmp2 = vala_gidl_parser_eval (self, nv[1]), (attr = (g_free (attr), NULL)), (nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL)), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp2);
					}
					(attr = (g_free (attr), NULL));
					(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
				}
			}
		}
	}
	if (g_str_has_prefix (type_name, vala_symbol_get_name (VALA_SYMBOL (ns)))) {
		const char* _tmp4;
		char* _tmp5;
		_tmp4 = NULL;
		_tmp5 = NULL;
		return (_tmp5 = (_tmp4 = g_utf8_offset_to_pointer (type_name, g_utf8_strlen (vala_symbol_get_name (VALA_SYMBOL (ns)), -1)), (_tmp4 == NULL ? NULL : g_strdup (_tmp4))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp5);
	} else {
		if (g_utf8_collate (vala_symbol_get_name (VALA_SYMBOL (ns)), "GLib") == 0 && g_str_has_prefix (type_name, "G")) {
			const char* _tmp6;
			char* _tmp7;
			_tmp6 = NULL;
			_tmp7 = NULL;
			return (_tmp7 = (_tmp6 = g_utf8_offset_to_pointer (type_name, 1), (_tmp6 == NULL ? NULL : g_strdup (_tmp6))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp7);
		} else {
			char* best_match;
			best_match = NULL;
			{
				GeeList* cprefix_collection;
				GeeIterator* cprefix_it;
				cprefix_collection = vala_namespace_get_cprefixes (ns);
				cprefix_it = gee_iterable_iterator (GEE_ITERABLE (cprefix_collection));
				while (gee_iterator_next (cprefix_it)) {
					char* cprefix;
					cprefix = gee_iterator_get (cprefix_it);
					{
						if (g_str_has_prefix (type_name, cprefix)) {
							if (best_match == NULL || g_utf8_strlen (cprefix, -1) > g_utf8_strlen (best_match, -1)) {
								char* _tmp9;
								const char* _tmp8;
								_tmp9 = NULL;
								_tmp8 = NULL;
								best_match = (_tmp9 = (_tmp8 = cprefix, (_tmp8 == NULL ? NULL : g_strdup (_tmp8))), (best_match = (g_free (best_match), NULL)), _tmp9);
							}
						}
						(cprefix = (g_free (cprefix), NULL));
					}
				}
				(cprefix_collection == NULL ? NULL : (cprefix_collection = (g_object_unref (cprefix_collection), NULL)));
				(cprefix_it == NULL ? NULL : (cprefix_it = (g_object_unref (cprefix_it), NULL)));
			}
			if (best_match != NULL) {
				const char* _tmp10;
				char* _tmp11;
				_tmp10 = NULL;
				_tmp11 = NULL;
				return (_tmp11 = (_tmp10 = g_utf8_offset_to_pointer (type_name, g_utf8_strlen (best_match, -1)), (_tmp10 == NULL ? NULL : g_strdup (_tmp10))), (best_match = (g_free (best_match), NULL)), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp11);
				;
			}
			(best_match = (g_free (best_match), NULL));
		}
	}
	_tmp12 = NULL;
	return (_tmp12 = g_strdup (type_name), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp12);
	(attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL));
}


static char* vala_gidl_parser_fix_const_name (ValaGIdlParser* self, const char* const_name, ValaNamespace* ns) {
	char* _tmp1;
	char* _tmp0;
	gboolean _tmp2;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (const_name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_NAMESPACE (ns), NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	if ((_tmp2 = g_str_has_prefix (const_name, (_tmp1 = g_strconcat ((_tmp0 = g_utf8_strup (vala_symbol_get_name (VALA_SYMBOL (ns)), -1)), "_", NULL))), (_tmp1 = (g_free (_tmp1), NULL)), (_tmp0 = (g_free (_tmp0), NULL)), _tmp2)) {
		const char* _tmp3;
		_tmp3 = NULL;
		return (_tmp3 = g_utf8_offset_to_pointer (const_name, g_utf8_strlen (vala_symbol_get_name (VALA_SYMBOL (ns)), -1) + 1), (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
	} else {
		if (g_utf8_collate (vala_symbol_get_name (VALA_SYMBOL (ns)), "GLib") == 0 && g_str_has_prefix (const_name, "G_")) {
			const char* _tmp5;
			_tmp5 = NULL;
			return (_tmp5 = g_utf8_offset_to_pointer (const_name, 2), (_tmp5 == NULL ? NULL : g_strdup (_tmp5)));
		}
	}
	return g_strdup (const_name);
}


static ValaNamespace* vala_gidl_parser_parse_module (ValaGIdlParser* self, GIdlModule* module) {
	ValaSymbol* sym;
	ValaNamespace* ns;
	ValaNamespace* _tmp4;
	ValaNamespace* _tmp3;
	char** _tmp5;
	gint attributes_length1;
	char** attributes;
	ValaNamespace* _tmp17;
	ValaNamespace* _tmp19;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (module != NULL, NULL);
	sym = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (vala_code_context_get_root (self->priv->context))), module->name);
	ns = NULL;
	if (VALA_IS_NAMESPACE (sym)) {
		ValaNamespace* _tmp1;
		ValaNamespace* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		ns = (_tmp1 = (_tmp0 = VALA_NAMESPACE (sym), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL))), _tmp1);
		vala_namespace_set_pkg (ns, FALSE);
	} else {
		ValaNamespace* _tmp2;
		_tmp2 = NULL;
		ns = (_tmp2 = vala_namespace_new (module->name, self->priv->current_source_reference), (ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL))), _tmp2);
	}
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->priv->current_namespace = (_tmp4 = (_tmp3 = ns, (_tmp3 == NULL ? NULL : g_object_ref (_tmp3))), (self->priv->current_namespace == NULL ? NULL : (self->priv->current_namespace = (g_object_unref (self->priv->current_namespace), NULL))), _tmp4);
	_tmp5 = NULL;
	attributes = (_tmp5 = vala_gidl_parser_get_attributes (self, vala_symbol_get_name (VALA_SYMBOL (ns))), attributes_length1 = -1, _tmp5);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp13;
				char* attr;
				_tmp13 = NULL;
				attr = (_tmp13 = attr_collection[attr_it], (_tmp13 == NULL ? NULL : g_strdup (_tmp13)));
				{
					char** _tmp6;
					gint nv_length1;
					char** nv;
					_tmp6 = NULL;
					nv = (_tmp6 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp6);
					if (g_utf8_collate (nv[0], "cheader_filename") == 0) {
						char* _tmp7;
						_tmp7 = NULL;
						vala_namespace_set_cheader_filename (ns, (_tmp7 = vala_gidl_parser_eval (self, nv[1])));
						(_tmp7 = (g_free (_tmp7), NULL));
					} else {
						if (g_utf8_collate (nv[0], "cprefix") == 0) {
							char** _tmp10;
							gint cprefixes_length1;
							char* _tmp8;
							char** _tmp9;
							char** cprefixes;
							_tmp10 = NULL;
							_tmp8 = NULL;
							_tmp9 = NULL;
							cprefixes = (_tmp10 = (_tmp9 = g_strsplit ((_tmp8 = vala_gidl_parser_eval (self, nv[1])), ",", 0), (_tmp8 = (g_free (_tmp8), NULL)), _tmp9), cprefixes_length1 = -1, _tmp10);
							{
								char** name_collection;
								int name_it;
								name_collection = cprefixes;
								for (name_it = 0; (cprefixes_length1 != -1 && name_it < cprefixes_length1) || (cprefixes_length1 == -1 && name_collection[name_it] != NULL); name_it = name_it + 1) {
									const char* _tmp11;
									char* name;
									_tmp11 = NULL;
									name = (_tmp11 = name_collection[name_it], (_tmp11 == NULL ? NULL : g_strdup (_tmp11)));
									{
										vala_namespace_add_cprefix (ns, name);
										(name = (g_free (name), NULL));
									}
								}
							}
							(cprefixes = (_vala_array_free (cprefixes, cprefixes_length1, ((GDestroyNotify) g_free)), NULL));
						} else {
							if (g_utf8_collate (nv[0], "lower_case_cprefix") == 0) {
								char* _tmp12;
								_tmp12 = NULL;
								vala_namespace_set_lower_case_cprefix (ns, (_tmp12 = vala_gidl_parser_eval (self, nv[1])));
								(_tmp12 = (g_free (_tmp12), NULL));
							}
						}
					}
					(attr = (g_free (attr), NULL));
					(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
				}
			}
		}
	}
	{
		GList* node_collection;
		GList* node_it;
		node_collection = module->entries;
		for (node_it = node_collection; node_it != NULL; node_it = node_it->next) {
			GIdlNode* node;
			node = node_it->data;
			{
				if (node->type == G_IDL_NODE_CALLBACK) {
					ValaCallback* cb;
					char* _tmp14;
					cb = vala_gidl_parser_parse_callback (self, ((GIdlNodeFunction*) node));
					if (cb == NULL) {
						(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
						continue;
					}
					_tmp14 = NULL;
					vala_symbol_set_name (VALA_SYMBOL (cb), (_tmp14 = vala_gidl_parser_fix_type_name (self, vala_symbol_get_name (VALA_SYMBOL (cb)), ns)));
					(_tmp14 = (g_free (_tmp14), NULL));
					vala_namespace_add_callback (ns, cb);
					vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (cb));
					(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
				} else {
					if (node->type == G_IDL_NODE_STRUCT) {
						vala_gidl_parser_parse_struct (self, ((GIdlNodeStruct*) node), ns, module);
					} else {
						if (node->type == G_IDL_NODE_BOXED) {
							vala_gidl_parser_parse_boxed (self, ((GIdlNodeBoxed*) node), ns, module);
						} else {
							if (node->type == G_IDL_NODE_ENUM) {
								ValaEnum* en;
								char* _tmp15;
								en = vala_gidl_parser_parse_enum (self, ((GIdlNodeEnum*) node));
								_tmp15 = NULL;
								vala_symbol_set_name (VALA_SYMBOL (en), (_tmp15 = vala_gidl_parser_fix_type_name (self, vala_symbol_get_name (VALA_SYMBOL (en)), ns)));
								(_tmp15 = (g_free (_tmp15), NULL));
								vala_namespace_add_enum (ns, en);
								vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (en));
								(en == NULL ? NULL : (en = (g_object_unref (en), NULL)));
							} else {
								if (node->type == G_IDL_NODE_OBJECT) {
									vala_gidl_parser_parse_object (self, ((GIdlNodeInterface*) node), ns, module);
								} else {
									if (node->type == G_IDL_NODE_INTERFACE) {
										vala_gidl_parser_parse_interface (self, ((GIdlNodeInterface*) node), ns, module);
									} else {
										if (node->type == G_IDL_NODE_CONSTANT) {
											ValaConstant* c;
											char* _tmp16;
											c = vala_gidl_parser_parse_constant (self, ((GIdlNodeConstant*) node));
											_tmp16 = NULL;
											vala_symbol_set_name (VALA_SYMBOL (c), (_tmp16 = vala_gidl_parser_fix_const_name (self, vala_symbol_get_name (VALA_SYMBOL (c)), ns)));
											(_tmp16 = (g_free (_tmp16), NULL));
											vala_namespace_add_constant (ns, c);
											vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (c));
											(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
										} else {
											if (node->type == G_IDL_NODE_FUNCTION) {
												ValaMethod* m;
												m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) node), FALSE, FALSE);
												if (m != NULL) {
													vala_method_set_instance (m, FALSE);
													vala_namespace_add_method (ns, m);
													vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (m));
												}
												(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	_tmp17 = NULL;
	self->priv->current_namespace = (_tmp17 = NULL, (self->priv->current_namespace == NULL ? NULL : (self->priv->current_namespace = (g_object_unref (self->priv->current_namespace), NULL))), _tmp17);
	if (VALA_IS_NAMESPACE (sym)) {
		gpointer _tmp18;
		return (_tmp18 = NULL, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), (ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp18);
	}
	_tmp19 = NULL;
	return (_tmp19 = ns, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp19);
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
	(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
	(attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL));
}


static ValaCallback* vala_gidl_parser_parse_callback (ValaGIdlParser* self, GIdlNodeFunction* f_node) {
	GIdlNode* node;
	char** _tmp0;
	gint attributes_length1;
	char** attributes;
	ValaTypeReference* _tmp6;
	ValaCallback* _tmp7;
	ValaCallback* cb;
	ValaCallback* _tmp10;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (f_node != NULL, NULL);
	node = ((GIdlNode*) f_node);
	_tmp0 = NULL;
	attributes = (_tmp0 = vala_gidl_parser_get_attributes (self, node->name), attributes_length1 = -1, _tmp0);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp5;
				char* attr;
				_tmp5 = NULL;
				attr = (_tmp5 = attr_collection[attr_it], (_tmp5 == NULL ? NULL : g_strdup (_tmp5)));
				{
					char** _tmp1;
					gint nv_length1;
					char** nv;
					_tmp1 = NULL;
					nv = (_tmp1 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp1);
					if (g_utf8_collate (nv[0], "hidden") == 0) {
						char* _tmp2;
						gboolean _tmp3;
						_tmp2 = NULL;
						if ((_tmp3 = g_utf8_collate ((_tmp2 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp2 = (g_free (_tmp2), NULL)), _tmp3)) {
							gpointer _tmp4;
							return (_tmp4 = NULL, (attr = (g_free (attr), NULL)), (nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL)), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp4);
						}
					}
					(attr = (g_free (attr), NULL));
					(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
				}
			}
		}
	}
	_tmp6 = NULL;
	_tmp7 = NULL;
	cb = (_tmp7 = vala_callback_new (node->name, (_tmp6 = vala_gidl_parser_parse_param (self, f_node->result)), self->priv->current_source_reference), (_tmp6 == NULL ? NULL : (_tmp6 = (g_object_unref (_tmp6), NULL))), _tmp7);
	vala_symbol_set_access (VALA_SYMBOL (cb), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	{
		GList* param_collection;
		GList* param_it;
		param_collection = f_node->parameters;
		for (param_it = param_collection; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = param_it->data;
			{
				GIdlNode* param_node;
				ValaTypeReference* _tmp8;
				ValaFormalParameter* _tmp9;
				ValaFormalParameter* p;
				param_node = ((GIdlNode*) param);
				_tmp8 = NULL;
				_tmp9 = NULL;
				p = (_tmp9 = vala_formal_parameter_new (param_node->name, (_tmp8 = vala_gidl_parser_parse_param (self, param)), NULL), (_tmp8 == NULL ? NULL : (_tmp8 = (g_object_unref (_tmp8), NULL))), _tmp9);
				vala_callback_add_parameter (cb, p);
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
	}
	_tmp10 = NULL;
	return (_tmp10 = cb, (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp10);
	(attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL));
	(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
}


static gboolean vala_gidl_parser_is_reference_type (ValaGIdlParser* self, const char* cname) {
	char** _tmp0;
	gint st_attributes_length1;
	char** st_attributes;
	gboolean _tmp6;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), FALSE);
	g_return_val_if_fail (cname != NULL, FALSE);
	_tmp0 = NULL;
	st_attributes = (_tmp0 = vala_gidl_parser_get_attributes (self, cname), st_attributes_length1 = -1, _tmp0);
	if (st_attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = st_attributes;
			for (attr_it = 0; (st_attributes_length1 != -1 && attr_it < st_attributes_length1) || (st_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp5;
				char* attr;
				_tmp5 = NULL;
				attr = (_tmp5 = attr_collection[attr_it], (_tmp5 == NULL ? NULL : g_strdup (_tmp5)));
				{
					char** _tmp1;
					gint nv_length1;
					char** nv;
					char* _tmp2;
					gboolean _tmp3;
					_tmp1 = NULL;
					nv = (_tmp1 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp1);
					_tmp2 = NULL;
					if ((_tmp3 = g_utf8_collate (nv[0], "is_value_type") == 0 && g_utf8_collate ((_tmp2 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp2 = (g_free (_tmp2), NULL)), _tmp3)) {
						gboolean _tmp4;
						return (_tmp4 = FALSE, (attr = (g_free (attr), NULL)), (nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL)), (st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp4);
					}
					(attr = (g_free (attr), NULL));
					(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
				}
			}
		}
	}
	return (_tmp6 = TRUE, (st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp6);
	(st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, ((GDestroyNotify) g_free)), NULL));
}


static void vala_gidl_parser_parse_struct (ValaGIdlParser* self, GIdlNodeStruct* st_node, ValaNamespace* ns, GIdlModule* module) {
	GIdlNode* node;
	char* name;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (st_node != NULL);
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
	g_return_if_fail (module != NULL);
	node = ((GIdlNode*) st_node);
	if (st_node->deprecated) {
		return;
	}
	name = vala_gidl_parser_fix_type_name (self, node->name, ns);
	if (!vala_gidl_parser_is_reference_type (self, node->name)) {
		ValaSymbol* _tmp0;
		ValaStruct* st;
		ValaDataType* _tmp9;
		ValaStruct* _tmp8;
		ValaDataType* _tmp10;
		_tmp0 = NULL;
		st = (_tmp0 = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (ns)), name), (VALA_IS_STRUCT (_tmp0) ? ((ValaStruct*) _tmp0) : NULL));
		if (st == NULL) {
			ValaStruct* _tmp1;
			char** _tmp2;
			gint st_attributes_length1;
			char** st_attributes;
			_tmp1 = NULL;
			st = (_tmp1 = vala_struct_new (name, self->priv->current_source_reference), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), _tmp1);
			vala_symbol_set_access (VALA_SYMBOL (st), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			_tmp2 = NULL;
			st_attributes = (_tmp2 = vala_gidl_parser_get_attributes (self, node->name), st_attributes_length1 = -1, _tmp2);
			if (st_attributes != NULL) {
				{
					char** attr_collection;
					int attr_it;
					attr_collection = st_attributes;
					for (attr_it = 0; (st_attributes_length1 != -1 && attr_it < st_attributes_length1) || (st_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
						const char* _tmp7;
						char* attr;
						_tmp7 = NULL;
						attr = (_tmp7 = attr_collection[attr_it], (_tmp7 == NULL ? NULL : g_strdup (_tmp7)));
						{
							char** _tmp3;
							gint nv_length1;
							char** nv;
							_tmp3 = NULL;
							nv = (_tmp3 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp3);
							if (g_utf8_collate (nv[0], "cheader_filename") == 0) {
								char* _tmp4;
								_tmp4 = NULL;
								vala_data_type_add_cheader_filename (VALA_DATA_TYPE (st), (_tmp4 = vala_gidl_parser_eval (self, nv[1])));
								(_tmp4 = (g_free (_tmp4), NULL));
							} else {
								if (g_utf8_collate (nv[0], "hidden") == 0) {
									char* _tmp5;
									gboolean _tmp6;
									_tmp5 = NULL;
									if ((_tmp6 = g_utf8_collate ((_tmp5 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp5 = (g_free (_tmp5), NULL)), _tmp6)) {
										(attr = (g_free (attr), NULL));
										(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
										(st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, ((GDestroyNotify) g_free)), NULL));
										(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
										(name = (g_free (name), NULL));
										return;
									}
								}
							}
							(attr = (g_free (attr), NULL));
							(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
						}
					}
				}
			}
			vala_namespace_add_struct (ns, st);
			vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (st));
			(st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, ((GDestroyNotify) g_free)), NULL));
		}
		_tmp9 = NULL;
		_tmp8 = NULL;
		self->priv->current_data_type = (_tmp9 = VALA_DATA_TYPE ((_tmp8 = st, (_tmp8 == NULL ? NULL : g_object_ref (_tmp8)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp9);
		{
			GList* member_collection;
			GList* member_it;
			member_collection = st_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = member_it->data;
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), FALSE, FALSE);
						if (m != NULL) {
							vala_struct_add_method (st, m);
						}
						(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
							if (f != NULL) {
								vala_struct_add_field (st, f);
							}
							(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
						}
					}
				}
			}
		}
		_tmp10 = NULL;
		self->priv->current_data_type = (_tmp10 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp10);
		(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
	} else {
		ValaSymbol* _tmp11;
		ValaClass* cl;
		ValaDataType* _tmp20;
		ValaClass* _tmp19;
		char* ref_function;
		char* unref_function;
		char* copy_function;
		char* free_function;
		ValaDataType* _tmp29;
		_tmp11 = NULL;
		cl = (_tmp11 = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (ns)), name), (VALA_IS_CLASS (_tmp11) ? ((ValaClass*) _tmp11) : NULL));
		if (cl == NULL) {
			ValaClass* _tmp12;
			char** _tmp13;
			gint cl_attributes_length1;
			char** cl_attributes;
			_tmp12 = NULL;
			cl = (_tmp12 = vala_class_new (name, self->priv->current_source_reference), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), _tmp12);
			vala_symbol_set_access (VALA_SYMBOL (cl), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			_tmp13 = NULL;
			cl_attributes = (_tmp13 = vala_gidl_parser_get_attributes (self, node->name), cl_attributes_length1 = -1, _tmp13);
			if (cl_attributes != NULL) {
				{
					char** attr_collection;
					int attr_it;
					attr_collection = cl_attributes;
					for (attr_it = 0; (cl_attributes_length1 != -1 && attr_it < cl_attributes_length1) || (cl_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
						const char* _tmp18;
						char* attr;
						_tmp18 = NULL;
						attr = (_tmp18 = attr_collection[attr_it], (_tmp18 == NULL ? NULL : g_strdup (_tmp18)));
						{
							char** _tmp14;
							gint nv_length1;
							char** nv;
							_tmp14 = NULL;
							nv = (_tmp14 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp14);
							if (g_utf8_collate (nv[0], "cheader_filename") == 0) {
								char* _tmp15;
								_tmp15 = NULL;
								vala_data_type_add_cheader_filename (VALA_DATA_TYPE (cl), (_tmp15 = vala_gidl_parser_eval (self, nv[1])));
								(_tmp15 = (g_free (_tmp15), NULL));
							} else {
								if (g_utf8_collate (nv[0], "hidden") == 0) {
									char* _tmp16;
									gboolean _tmp17;
									_tmp16 = NULL;
									if ((_tmp17 = g_utf8_collate ((_tmp16 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp16 = (g_free (_tmp16), NULL)), _tmp17)) {
										(attr = (g_free (attr), NULL));
										(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
										(cl_attributes = (_vala_array_free (cl_attributes, cl_attributes_length1, ((GDestroyNotify) g_free)), NULL));
										(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
										(name = (g_free (name), NULL));
										return;
									}
								}
							}
							(attr = (g_free (attr), NULL));
							(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
						}
					}
				}
			}
			vala_namespace_add_class (ns, cl);
			vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (cl));
			(cl_attributes = (_vala_array_free (cl_attributes, cl_attributes_length1, ((GDestroyNotify) g_free)), NULL));
		}
		_tmp20 = NULL;
		_tmp19 = NULL;
		self->priv->current_data_type = (_tmp20 = VALA_DATA_TYPE ((_tmp19 = cl, (_tmp19 == NULL ? NULL : g_object_ref (_tmp19)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp20);
		ref_function = NULL;
		unref_function = NULL;
		copy_function = NULL;
		free_function = NULL;
		{
			GList* member_collection;
			GList* member_it;
			member_collection = st_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = member_it->data;
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						if (g_utf8_collate (member->name, "ref") == 0) {
							char* _tmp22;
							const char* _tmp21;
							_tmp22 = NULL;
							_tmp21 = NULL;
							ref_function = (_tmp22 = (_tmp21 = (((GIdlNodeFunction*) member))->symbol, (_tmp21 == NULL ? NULL : g_strdup (_tmp21))), (ref_function = (g_free (ref_function), NULL)), _tmp22);
						} else {
							if (g_utf8_collate (member->name, "unref") == 0) {
								char* _tmp24;
								const char* _tmp23;
								_tmp24 = NULL;
								_tmp23 = NULL;
								unref_function = (_tmp24 = (_tmp23 = (((GIdlNodeFunction*) member))->symbol, (_tmp23 == NULL ? NULL : g_strdup (_tmp23))), (unref_function = (g_free (unref_function), NULL)), _tmp24);
							} else {
								if (g_utf8_collate (member->name, "free") == 0 || g_utf8_collate (member->name, "destroy") == 0) {
									char* _tmp26;
									const char* _tmp25;
									_tmp26 = NULL;
									_tmp25 = NULL;
									free_function = (_tmp26 = (_tmp25 = (((GIdlNodeFunction*) member))->symbol, (_tmp25 == NULL ? NULL : g_strdup (_tmp25))), (free_function = (g_free (free_function), NULL)), _tmp26);
								} else {
									ValaMethod* m;
									if (g_utf8_collate (member->name, "copy") == 0) {
										char* _tmp28;
										const char* _tmp27;
										_tmp28 = NULL;
										_tmp27 = NULL;
										copy_function = (_tmp28 = (_tmp27 = (((GIdlNodeFunction*) member))->symbol, (_tmp27 == NULL ? NULL : g_strdup (_tmp27))), (copy_function = (g_free (copy_function), NULL)), _tmp28);
									}
									m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), FALSE, FALSE);
									if (m != NULL) {
										vala_class_add_method (cl, m);
									}
									(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
								}
							}
						}
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
							if (f != NULL) {
								vala_class_add_field (cl, f);
							}
							(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
						}
					}
				}
			}
		}
		if (ref_function != NULL) {
			vala_class_set_ref_function (cl, ref_function);
		}
		if (copy_function != NULL) {
			vala_class_set_dup_function (cl, copy_function);
		}
		if (unref_function != NULL) {
			vala_class_set_unref_function (cl, unref_function);
		} else {
			if (free_function != NULL) {
				vala_class_set_free_function (cl, free_function);
			}
		}
		_tmp29 = NULL;
		self->priv->current_data_type = (_tmp29 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp29);
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
		(ref_function = (g_free (ref_function), NULL));
		(unref_function = (g_free (unref_function), NULL));
		(copy_function = (g_free (copy_function), NULL));
		(free_function = (g_free (free_function), NULL));
	}
	(name = (g_free (name), NULL));
}


static void vala_gidl_parser_parse_boxed (ValaGIdlParser* self, GIdlNodeBoxed* boxed_node, ValaNamespace* ns, GIdlModule* module) {
	GIdlNode* node;
	char* name;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (boxed_node != NULL);
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
	g_return_if_fail (module != NULL);
	node = ((GIdlNode*) boxed_node);
	name = vala_gidl_parser_fix_type_name (self, node->name, ns);
	if (!vala_gidl_parser_is_reference_type (self, node->name)) {
		ValaSymbol* _tmp0;
		ValaStruct* st;
		ValaDataType* _tmp8;
		ValaStruct* _tmp7;
		ValaDataType* _tmp9;
		_tmp0 = NULL;
		st = (_tmp0 = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (ns)), name), (VALA_IS_STRUCT (_tmp0) ? ((ValaStruct*) _tmp0) : NULL));
		if (st == NULL) {
			ValaStruct* _tmp1;
			char** _tmp2;
			gint st_attributes_length1;
			char** st_attributes;
			char* _tmp6;
			_tmp1 = NULL;
			st = (_tmp1 = vala_struct_new (name, self->priv->current_source_reference), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), _tmp1);
			vala_symbol_set_access (VALA_SYMBOL (st), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			_tmp2 = NULL;
			st_attributes = (_tmp2 = vala_gidl_parser_get_attributes (self, node->name), st_attributes_length1 = -1, _tmp2);
			if (st_attributes != NULL) {
				{
					char** attr_collection;
					int attr_it;
					attr_collection = st_attributes;
					for (attr_it = 0; (st_attributes_length1 != -1 && attr_it < st_attributes_length1) || (st_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
						const char* _tmp5;
						char* attr;
						_tmp5 = NULL;
						attr = (_tmp5 = attr_collection[attr_it], (_tmp5 == NULL ? NULL : g_strdup (_tmp5)));
						{
							char** _tmp3;
							gint nv_length1;
							char** nv;
							_tmp3 = NULL;
							nv = (_tmp3 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp3);
							if (g_utf8_collate (nv[0], "cheader_filename") == 0) {
								char* _tmp4;
								_tmp4 = NULL;
								vala_data_type_add_cheader_filename (VALA_DATA_TYPE (st), (_tmp4 = vala_gidl_parser_eval (self, nv[1])));
								(_tmp4 = (g_free (_tmp4), NULL));
							}
							(attr = (g_free (attr), NULL));
							(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
						}
					}
				}
			}
			vala_namespace_add_struct (ns, st);
			_tmp6 = NULL;
			vala_struct_set_type_id (st, (_tmp6 = vala_data_type_get_upper_case_cname (VALA_DATA_TYPE (st), "TYPE_")));
			(_tmp6 = (g_free (_tmp6), NULL));
			vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (st));
			(st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, ((GDestroyNotify) g_free)), NULL));
		}
		_tmp8 = NULL;
		_tmp7 = NULL;
		self->priv->current_data_type = (_tmp8 = VALA_DATA_TYPE ((_tmp7 = st, (_tmp7 == NULL ? NULL : g_object_ref (_tmp7)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp8);
		{
			GList* member_collection;
			GList* member_it;
			member_collection = boxed_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = member_it->data;
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), FALSE, FALSE);
						if (m != NULL) {
							vala_struct_add_method (st, m);
						}
						(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
							if (f != NULL) {
								vala_struct_add_field (st, f);
							}
							(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
						}
					}
				}
			}
		}
		_tmp9 = NULL;
		self->priv->current_data_type = (_tmp9 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp9);
		(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
	} else {
		ValaSymbol* _tmp10;
		ValaClass* cl;
		ValaTypeReference* parent;
		ValaDataType* _tmp18;
		ValaClass* _tmp17;
		char* ref_function;
		char* unref_function;
		char* copy_function;
		char* free_function;
		ValaDataType* _tmp27;
		_tmp10 = NULL;
		cl = (_tmp10 = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (ns)), name), (VALA_IS_CLASS (_tmp10) ? ((ValaClass*) _tmp10) : NULL));
		if (cl == NULL) {
			ValaClass* _tmp11;
			char** _tmp12;
			gint cl_attributes_length1;
			char** cl_attributes;
			char* _tmp16;
			_tmp11 = NULL;
			cl = (_tmp11 = vala_class_new (name, self->priv->current_source_reference), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), _tmp11);
			vala_symbol_set_access (VALA_SYMBOL (cl), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			_tmp12 = NULL;
			cl_attributes = (_tmp12 = vala_gidl_parser_get_attributes (self, node->name), cl_attributes_length1 = -1, _tmp12);
			if (cl_attributes != NULL) {
				{
					char** attr_collection;
					int attr_it;
					attr_collection = cl_attributes;
					for (attr_it = 0; (cl_attributes_length1 != -1 && attr_it < cl_attributes_length1) || (cl_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
						const char* _tmp15;
						char* attr;
						_tmp15 = NULL;
						attr = (_tmp15 = attr_collection[attr_it], (_tmp15 == NULL ? NULL : g_strdup (_tmp15)));
						{
							char** _tmp13;
							gint nv_length1;
							char** nv;
							_tmp13 = NULL;
							nv = (_tmp13 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp13);
							if (g_utf8_collate (nv[0], "cheader_filename") == 0) {
								char* _tmp14;
								_tmp14 = NULL;
								vala_data_type_add_cheader_filename (VALA_DATA_TYPE (cl), (_tmp14 = vala_gidl_parser_eval (self, nv[1])));
								(_tmp14 = (g_free (_tmp14), NULL));
							}
							(attr = (g_free (attr), NULL));
							(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
						}
					}
				}
			}
			vala_namespace_add_class (ns, cl);
			_tmp16 = NULL;
			vala_class_set_type_id (cl, (_tmp16 = vala_data_type_get_upper_case_cname (VALA_DATA_TYPE (cl), "TYPE_")));
			(_tmp16 = (g_free (_tmp16), NULL));
			vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (cl));
			(cl_attributes = (_vala_array_free (cl_attributes, cl_attributes_length1, ((GDestroyNotify) g_free)), NULL));
		}
		parent = vala_type_reference_new ();
		vala_type_reference_set_namespace_name (parent, "GLib");
		vala_type_reference_set_type_name (parent, "Boxed");
		vala_class_add_base_type (cl, parent);
		_tmp18 = NULL;
		_tmp17 = NULL;
		self->priv->current_data_type = (_tmp18 = VALA_DATA_TYPE ((_tmp17 = cl, (_tmp17 == NULL ? NULL : g_object_ref (_tmp17)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp18);
		ref_function = NULL;
		unref_function = NULL;
		copy_function = NULL;
		free_function = NULL;
		{
			GList* member_collection;
			GList* member_it;
			member_collection = boxed_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = member_it->data;
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						if (g_utf8_collate (member->name, "ref") == 0) {
							char* _tmp20;
							const char* _tmp19;
							_tmp20 = NULL;
							_tmp19 = NULL;
							ref_function = (_tmp20 = (_tmp19 = (((GIdlNodeFunction*) member))->symbol, (_tmp19 == NULL ? NULL : g_strdup (_tmp19))), (ref_function = (g_free (ref_function), NULL)), _tmp20);
						} else {
							if (g_utf8_collate (member->name, "unref") == 0) {
								char* _tmp22;
								const char* _tmp21;
								_tmp22 = NULL;
								_tmp21 = NULL;
								unref_function = (_tmp22 = (_tmp21 = (((GIdlNodeFunction*) member))->symbol, (_tmp21 == NULL ? NULL : g_strdup (_tmp21))), (unref_function = (g_free (unref_function), NULL)), _tmp22);
							} else {
								if (g_utf8_collate (member->name, "free") == 0 || g_utf8_collate (member->name, "destroy") == 0) {
									char* _tmp24;
									const char* _tmp23;
									_tmp24 = NULL;
									_tmp23 = NULL;
									free_function = (_tmp24 = (_tmp23 = (((GIdlNodeFunction*) member))->symbol, (_tmp23 == NULL ? NULL : g_strdup (_tmp23))), (free_function = (g_free (free_function), NULL)), _tmp24);
								} else {
									ValaMethod* m;
									if (g_utf8_collate (member->name, "copy") == 0) {
										char* _tmp26;
										const char* _tmp25;
										_tmp26 = NULL;
										_tmp25 = NULL;
										copy_function = (_tmp26 = (_tmp25 = (((GIdlNodeFunction*) member))->symbol, (_tmp25 == NULL ? NULL : g_strdup (_tmp25))), (copy_function = (g_free (copy_function), NULL)), _tmp26);
									}
									m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), FALSE, FALSE);
									if (m != NULL) {
										vala_class_add_method (cl, m);
									}
									(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
								}
							}
						}
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
							if (f != NULL) {
								vala_class_add_field (cl, f);
							}
							(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
						}
					}
				}
			}
		}
		if (ref_function != NULL) {
			vala_class_set_ref_function (cl, ref_function);
		}
		if (copy_function != NULL) {
			vala_class_set_dup_function (cl, copy_function);
		}
		if (unref_function != NULL) {
			vala_class_set_unref_function (cl, unref_function);
		} else {
			if (free_function != NULL) {
				vala_class_set_free_function (cl, free_function);
			}
		}
		_tmp27 = NULL;
		self->priv->current_data_type = (_tmp27 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp27);
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
		(parent == NULL ? NULL : (parent = (g_object_unref (parent), NULL)));
		(ref_function = (g_free (ref_function), NULL));
		(unref_function = (g_free (unref_function), NULL));
		(copy_function = (g_free (copy_function), NULL));
		(free_function = (g_free (free_function), NULL));
	}
	(name = (g_free (name), NULL));
}


static ValaEnum* vala_gidl_parser_parse_enum (ValaGIdlParser* self, GIdlNodeEnum* en_node) {
	GIdlNode* node;
	ValaEnum* en;
	char* common_prefix;
	ValaEnum* _tmp5;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (en_node != NULL, NULL);
	node = ((GIdlNode*) en_node);
	en = vala_enum_new (node->name, self->priv->current_source_reference);
	vala_symbol_set_access (VALA_SYMBOL (en), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	common_prefix = NULL;
	{
		GList* value_collection;
		GList* value_it;
		value_collection = en_node->values;
		for (value_it = value_collection; value_it != NULL; value_it = value_it->next) {
			GIdlNode* value;
			value = value_it->data;
			{
				if (common_prefix == NULL) {
					char* _tmp1;
					const char* _tmp0;
					_tmp1 = NULL;
					_tmp0 = NULL;
					common_prefix = (_tmp1 = (_tmp0 = value->name, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (common_prefix = (g_free (common_prefix), NULL)), _tmp1);
					while (g_utf8_strlen (common_prefix, -1) > 0 && !g_str_has_suffix (common_prefix, "_")) {
						char* _tmp2;
						/* FIXME: could easily be made faster*/
						_tmp2 = NULL;
						common_prefix = (_tmp2 = g_strndup (common_prefix, strlen (common_prefix) - 1), (common_prefix = (g_free (common_prefix), NULL)), _tmp2);
					}
				} else {
					while (!g_str_has_prefix (value->name, common_prefix)) {
						char* _tmp3;
						_tmp3 = NULL;
						common_prefix = (_tmp3 = g_strndup (common_prefix, strlen (common_prefix) - 1), (common_prefix = (g_free (common_prefix), NULL)), _tmp3);
					}
				}
				while (g_utf8_strlen (common_prefix, -1) > 0 && (!g_str_has_suffix (common_prefix, "_") || (g_unichar_isdigit (g_utf8_get_char (g_utf8_offset_to_pointer (value->name, strlen (common_prefix))))) && (g_utf8_strlen (value->name, -1) - g_utf8_strlen (common_prefix, -1)) <= 1)) {
					char* _tmp4;
					/* enum values may not consist solely of digits*/
					_tmp4 = NULL;
					common_prefix = (_tmp4 = g_strndup (common_prefix, strlen (common_prefix) - 1), (common_prefix = (g_free (common_prefix), NULL)), _tmp4);
				}
			}
		}
	}
	vala_enum_set_cprefix (en, common_prefix);
	{
		GList* value2_collection;
		GList* value2_it;
		value2_collection = en_node->values;
		for (value2_it = value2_collection; value2_it != NULL; value2_it = value2_it->next) {
			GIdlNode* value2;
			value2 = value2_it->data;
			{
				ValaEnumValue* ev;
				ev = vala_enum_value_new (g_utf8_offset_to_pointer (value2->name, g_utf8_strlen (common_prefix, -1)));
				vala_enum_add_value (en, ev);
				(ev == NULL ? NULL : (ev = (g_object_unref (ev), NULL)));
			}
		}
	}
	_tmp5 = NULL;
	return (_tmp5 = en, (common_prefix = (g_free (common_prefix), NULL)), _tmp5);
	(en == NULL ? NULL : (en = (g_object_unref (en), NULL)));
	(common_prefix = (g_free (common_prefix), NULL));
}


static void vala_gidl_parser_parse_object (ValaGIdlParser* self, GIdlNodeInterface* node, ValaNamespace* ns, GIdlModule* module) {
	char* name;
	char* base_class;
	ValaSymbol* _tmp0;
	ValaClass* cl;
	ValaDataType* _tmp11;
	ValaClass* _tmp10;
	GeeSet* _tmp12;
	GeeHashMap* current_type_vfunc_map;
	ValaDataType* _tmp15;
	GeeSet* _tmp16;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (node != NULL);
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
	g_return_if_fail (module != NULL);
	name = vala_gidl_parser_fix_type_name (self, node->gtype_name, ns);
	base_class = NULL;
	_tmp0 = NULL;
	cl = (_tmp0 = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (ns)), name), (VALA_IS_CLASS (_tmp0) ? ((ValaClass*) _tmp0) : NULL));
	if (cl == NULL) {
		ValaClass* _tmp1;
		char** _tmp2;
		gint attributes_length1;
		char** attributes;
		_tmp1 = NULL;
		cl = (_tmp1 = vala_class_new (name, self->priv->current_source_reference), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), _tmp1);
		vala_symbol_set_access (VALA_SYMBOL (cl), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
		_tmp2 = NULL;
		attributes = (_tmp2 = vala_gidl_parser_get_attributes (self, node->gtype_name), attributes_length1 = -1, _tmp2);
		if (attributes != NULL) {
			{
				char** attr_collection;
				int attr_it;
				attr_collection = attributes;
				for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
					const char* _tmp8;
					char* attr;
					_tmp8 = NULL;
					attr = (_tmp8 = attr_collection[attr_it], (_tmp8 == NULL ? NULL : g_strdup (_tmp8)));
					{
						char** _tmp3;
						gint nv_length1;
						char** nv;
						_tmp3 = NULL;
						nv = (_tmp3 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp3);
						if (g_utf8_collate (nv[0], "cheader_filename") == 0) {
							char* _tmp4;
							_tmp4 = NULL;
							vala_data_type_add_cheader_filename (VALA_DATA_TYPE (cl), (_tmp4 = vala_gidl_parser_eval (self, nv[1])));
							(_tmp4 = (g_free (_tmp4), NULL));
						} else {
							if (g_utf8_collate (nv[0], "base_class") == 0) {
								char* _tmp5;
								_tmp5 = NULL;
								base_class = (_tmp5 = vala_gidl_parser_eval (self, nv[1]), (base_class = (g_free (base_class), NULL)), _tmp5);
							} else {
								if (g_utf8_collate (nv[0], "hidden") == 0) {
									char* _tmp6;
									gboolean _tmp7;
									_tmp6 = NULL;
									if ((_tmp7 = g_utf8_collate ((_tmp6 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp6 = (g_free (_tmp6), NULL)), _tmp7)) {
										(attr = (g_free (attr), NULL));
										(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
										(attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL));
										(name = (g_free (name), NULL));
										(base_class = (g_free (base_class), NULL));
										(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
										return;
									}
								}
							}
						}
						(attr = (g_free (attr), NULL));
						(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
					}
				}
			}
		}
		vala_namespace_add_class (ns, cl);
		vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (cl));
		(attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL));
	}
	if (base_class != NULL) {
		ValaTypeReference* parent;
		parent = vala_type_reference_new ();
		vala_gidl_parser_parse_type_string (self, parent, base_class);
		vala_class_add_base_type (cl, parent);
		(parent == NULL ? NULL : (parent = (g_object_unref (parent), NULL)));
	} else {
		if (node->parent != NULL) {
			ValaTypeReference* parent;
			parent = vala_type_reference_new ();
			vala_gidl_parser_parse_type_string (self, parent, node->parent);
			vala_class_add_base_type (cl, parent);
			(parent == NULL ? NULL : (parent = (g_object_unref (parent), NULL)));
		} else {
			ValaTypeReference* parent;
			parent = vala_type_reference_new ();
			vala_type_reference_set_namespace_name (parent, "GLib");
			vala_type_reference_set_type_name (parent, "Object");
			vala_class_add_base_type (cl, parent);
			(parent == NULL ? NULL : (parent = (g_object_unref (parent), NULL)));
		}
	}
	{
		GList* iface_name_collection;
		GList* iface_name_it;
		iface_name_collection = node->interfaces;
		for (iface_name_it = iface_name_collection; iface_name_it != NULL; iface_name_it = iface_name_it->next) {
			const char* _tmp9;
			char* iface_name;
			_tmp9 = NULL;
			iface_name = (_tmp9 = iface_name_it->data, (_tmp9 == NULL ? NULL : g_strdup (_tmp9)));
			{
				ValaTypeReference* iface;
				iface = vala_type_reference_new ();
				vala_gidl_parser_parse_type_string (self, iface, iface_name);
				vala_class_add_base_type (cl, iface);
				(iface_name = (g_free (iface_name), NULL));
				(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
			}
		}
	}
	_tmp11 = NULL;
	_tmp10 = NULL;
	self->priv->current_data_type = (_tmp11 = VALA_DATA_TYPE ((_tmp10 = cl, (_tmp10 == NULL ? NULL : g_object_ref (_tmp10)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp11);
	_tmp12 = NULL;
	self->priv->current_type_symbol_set = (_tmp12 = GEE_SET (gee_hash_set_new (g_strdup, g_free, g_str_hash, g_str_equal)), (self->priv->current_type_symbol_set == NULL ? NULL : (self->priv->current_type_symbol_set = (g_object_unref (self->priv->current_type_symbol_set), NULL))), _tmp12);
	current_type_vfunc_map = gee_hash_map_new (g_strdup, g_free, g_strdup, g_free, g_str_hash, g_str_equal, g_direct_equal);
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_VFUNC) {
					gee_map_set (GEE_MAP (current_type_vfunc_map), member->name, "1");
				}
			}
		}
	}
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					char* _tmp13;
					gboolean _tmp14;
					gboolean is_virtual;
					ValaMethod* m;
					_tmp13 = NULL;
					is_virtual = (_tmp14 = (_tmp13 = gee_map_get (GEE_MAP (current_type_vfunc_map), member->name)) != NULL, (_tmp13 = (g_free (_tmp13), NULL)), _tmp14);
					m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), is_virtual, FALSE);
					if (m != NULL) {
						vala_class_add_method (cl, m);
					}
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				} else {
					if (member->type == G_IDL_NODE_PROPERTY) {
						ValaProperty* prop;
						prop = vala_gidl_parser_parse_property (self, ((GIdlNodeProperty*) member));
						if (prop != NULL) {
							vala_class_add_property (cl, prop, FALSE);
						}
						(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
					} else {
						if (member->type == G_IDL_NODE_SIGNAL) {
							ValaSignal* sig;
							sig = vala_gidl_parser_parse_signal (self, ((GIdlNodeSignal*) member));
							if (sig != NULL) {
								vala_class_add_signal (cl, sig);
							}
							(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
						}
					}
				}
			}
		}
	}
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_FIELD) {
					if (!gee_collection_contains (GEE_COLLECTION (self->priv->current_type_symbol_set), member->name)) {
						ValaField* f;
						f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
						if (f != NULL) {
							vala_class_add_field (cl, f);
						}
						(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
					}
				}
			}
		}
	}
	{
		GeeCollection* prop_collection;
		GeeIterator* prop_it;
		prop_collection = vala_class_get_properties (cl);
		prop_it = gee_iterable_iterator (GEE_ITERABLE (prop_collection));
		while (gee_iterator_next (prop_it)) {
			ValaProperty* prop;
			prop = gee_iterator_get (prop_it);
			{
				char* getter;
				char* setter;
				getter = g_strdup_printf ("get_%s", vala_symbol_get_name (VALA_SYMBOL (prop)));
				if (vala_property_get_get_accessor (prop) != NULL && !gee_collection_contains (GEE_COLLECTION (self->priv->current_type_symbol_set), getter)) {
					vala_property_set_no_accessor_method (prop, TRUE);
				}
				setter = g_strdup_printf ("set_%s", vala_symbol_get_name (VALA_SYMBOL (prop)));
				if (vala_property_get_set_accessor (prop) != NULL && !gee_collection_contains (GEE_COLLECTION (self->priv->current_type_symbol_set), setter)) {
					vala_property_set_no_accessor_method (prop, TRUE);
				}
				(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
				(getter = (g_free (getter), NULL));
				(setter = (g_free (setter), NULL));
			}
		}
		(prop_collection == NULL ? NULL : (prop_collection = (g_object_unref (prop_collection), NULL)));
		(prop_it == NULL ? NULL : (prop_it = (g_object_unref (prop_it), NULL)));
	}
	_tmp15 = NULL;
	self->priv->current_data_type = (_tmp15 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp15);
	_tmp16 = NULL;
	self->priv->current_type_symbol_set = (_tmp16 = NULL, (self->priv->current_type_symbol_set == NULL ? NULL : (self->priv->current_type_symbol_set = (g_object_unref (self->priv->current_type_symbol_set), NULL))), _tmp16);
	(name = (g_free (name), NULL));
	(base_class = (g_free (base_class), NULL));
	(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	(current_type_vfunc_map == NULL ? NULL : (current_type_vfunc_map = (g_object_unref (current_type_vfunc_map), NULL)));
}


static void vala_gidl_parser_parse_interface (ValaGIdlParser* self, GIdlNodeInterface* node, ValaNamespace* ns, GIdlModule* module) {
	char* name;
	ValaSymbol* _tmp0;
	ValaInterface* iface;
	ValaDataType* _tmp8;
	ValaInterface* _tmp7;
	GeeHashMap* current_type_vfunc_map;
	ValaDataType* _tmp11;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (node != NULL);
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
	g_return_if_fail (module != NULL);
	name = vala_gidl_parser_fix_type_name (self, node->gtype_name, ns);
	_tmp0 = NULL;
	iface = (_tmp0 = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (ns)), name), (VALA_IS_INTERFACE (_tmp0) ? ((ValaInterface*) _tmp0) : NULL));
	if (iface == NULL) {
		ValaInterface* _tmp1;
		char** _tmp2;
		gint attributes_length1;
		char** attributes;
		_tmp1 = NULL;
		iface = (_tmp1 = vala_interface_new (name, self->priv->current_source_reference), (iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL))), _tmp1);
		vala_symbol_set_access (VALA_SYMBOL (iface), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
		_tmp2 = NULL;
		attributes = (_tmp2 = vala_gidl_parser_get_attributes (self, node->gtype_name), attributes_length1 = -1, _tmp2);
		if (attributes != NULL) {
			{
				char** attr_collection;
				int attr_it;
				attr_collection = attributes;
				for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
					const char* _tmp5;
					char* attr;
					_tmp5 = NULL;
					attr = (_tmp5 = attr_collection[attr_it], (_tmp5 == NULL ? NULL : g_strdup (_tmp5)));
					{
						char** _tmp3;
						gint nv_length1;
						char** nv;
						_tmp3 = NULL;
						nv = (_tmp3 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp3);
						if (g_utf8_collate (nv[0], "cheader_filename") == 0) {
							char* _tmp4;
							_tmp4 = NULL;
							vala_data_type_add_cheader_filename (VALA_DATA_TYPE (iface), (_tmp4 = vala_gidl_parser_eval (self, nv[1])));
							(_tmp4 = (g_free (_tmp4), NULL));
						}
						(attr = (g_free (attr), NULL));
						(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
					}
				}
			}
		}
		{
			GList* prereq_name_collection;
			GList* prereq_name_it;
			prereq_name_collection = node->prerequisites;
			for (prereq_name_it = prereq_name_collection; prereq_name_it != NULL; prereq_name_it = prereq_name_it->next) {
				const char* _tmp6;
				char* prereq_name;
				_tmp6 = NULL;
				prereq_name = (_tmp6 = prereq_name_it->data, (_tmp6 == NULL ? NULL : g_strdup (_tmp6)));
				{
					ValaTypeReference* prereq;
					prereq = vala_type_reference_new ();
					vala_gidl_parser_parse_type_string (self, prereq, prereq_name);
					vala_interface_add_prerequisite (iface, prereq);
					(prereq_name = (g_free (prereq_name), NULL));
					(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
				}
			}
		}
		vala_namespace_add_interface (ns, iface);
		vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (iface));
		(attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL));
	}
	_tmp8 = NULL;
	_tmp7 = NULL;
	self->priv->current_data_type = (_tmp8 = VALA_DATA_TYPE ((_tmp7 = iface, (_tmp7 == NULL ? NULL : g_object_ref (_tmp7)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp8);
	current_type_vfunc_map = gee_hash_map_new (g_strdup, g_free, g_strdup, g_free, g_str_hash, g_str_equal, g_direct_equal);
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_VFUNC) {
					gee_map_set (GEE_MAP (current_type_vfunc_map), member->name, "1");
				}
			}
		}
	}
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					char* _tmp9;
					gboolean _tmp10;
					gboolean is_virtual;
					ValaMethod* m;
					_tmp9 = NULL;
					is_virtual = (_tmp10 = (_tmp9 = gee_map_get (GEE_MAP (current_type_vfunc_map), member->name)) != NULL, (_tmp9 = (g_free (_tmp9), NULL)), _tmp10);
					m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), is_virtual, TRUE);
					if (m != NULL) {
						vala_interface_add_method (iface, m);
					}
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				} else {
					if (member->type == G_IDL_NODE_SIGNAL) {
						ValaSignal* sig;
						sig = vala_gidl_parser_parse_signal (self, ((GIdlNodeSignal*) member));
						if (sig != NULL) {
							vala_interface_add_signal (iface, sig);
						}
						(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
					}
				}
			}
		}
	}
	_tmp11 = NULL;
	self->priv->current_data_type = (_tmp11 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp11);
	(name = (g_free (name), NULL));
	(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
	(current_type_vfunc_map == NULL ? NULL : (current_type_vfunc_map = (g_object_unref (current_type_vfunc_map), NULL)));
}


static ValaTypeReference* vala_gidl_parser_parse_type (ValaGIdlParser* self, GIdlNodeType* type_node) {
	ValaTypeReference* type;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (type_node != NULL, NULL);
	type = vala_type_reference_new ();
	if (type_node->tag == TYPE_TAG_VOID) {
		if (type_node->is_pointer) {
			vala_type_reference_set_type_name (type, "pointer");
		} else {
			vala_type_reference_set_type_name (type, "void");
		}
	} else {
		if (type_node->tag == TYPE_TAG_BOOLEAN) {
			vala_type_reference_set_type_name (type, "bool");
		} else {
			if (type_node->tag == TYPE_TAG_INT8) {
				vala_type_reference_set_type_name (type, "char");
			} else {
				if (type_node->tag == TYPE_TAG_UINT8) {
					vala_type_reference_set_type_name (type, "uchar");
				} else {
					if (type_node->tag == TYPE_TAG_INT16) {
						vala_type_reference_set_type_name (type, "short");
					} else {
						if (type_node->tag == TYPE_TAG_UINT16) {
							vala_type_reference_set_type_name (type, "ushort");
						} else {
							if (type_node->tag == TYPE_TAG_INT32) {
								vala_type_reference_set_type_name (type, "int");
							} else {
								if (type_node->tag == TYPE_TAG_UINT32) {
									vala_type_reference_set_type_name (type, "uint");
								} else {
									if (type_node->tag == TYPE_TAG_INT64) {
										vala_type_reference_set_type_name (type, "int64");
									} else {
										if (type_node->tag == TYPE_TAG_UINT64) {
											vala_type_reference_set_type_name (type, "uint64");
										} else {
											if (type_node->tag == TYPE_TAG_INT) {
												vala_type_reference_set_type_name (type, "int");
											} else {
												if (type_node->tag == TYPE_TAG_UINT) {
													vala_type_reference_set_type_name (type, "uint");
												} else {
													if (type_node->tag == TYPE_TAG_LONG) {
														vala_type_reference_set_type_name (type, "long");
													} else {
														if (type_node->tag == TYPE_TAG_ULONG) {
															vala_type_reference_set_type_name (type, "ulong");
														} else {
															if (type_node->tag == TYPE_TAG_SSIZE) {
																vala_type_reference_set_type_name (type, "long");
															} else {
																if (type_node->tag == TYPE_TAG_SIZE) {
																	vala_type_reference_set_type_name (type, "ulong");
																} else {
																	if (type_node->tag == TYPE_TAG_FLOAT) {
																		vala_type_reference_set_type_name (type, "float");
																	} else {
																		if (type_node->tag == TYPE_TAG_DOUBLE) {
																			vala_type_reference_set_type_name (type, "double");
																		} else {
																			if (type_node->tag == TYPE_TAG_UTF8) {
																				vala_type_reference_set_type_name (type, "string");
																			} else {
																				if (type_node->tag == TYPE_TAG_FILENAME) {
																					vala_type_reference_set_type_name (type, "string");
																				} else {
																					if (type_node->tag == TYPE_TAG_ARRAY) {
																						ValaTypeReference* _tmp0;
																						_tmp0 = NULL;
																						type = (_tmp0 = vala_gidl_parser_parse_type (self, type_node->parameter_type1), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp0);
																						vala_type_reference_set_array_rank (type, 1);
																					} else {
																						if (type_node->tag == TYPE_TAG_LIST) {
																							vala_type_reference_set_namespace_name (type, "GLib");
																							vala_type_reference_set_type_name (type, "List");
																						} else {
																							if (type_node->tag == TYPE_TAG_SLIST) {
																								vala_type_reference_set_namespace_name (type, "GLib");
																								vala_type_reference_set_type_name (type, "SList");
																							} else {
																								if (type_node->tag == TYPE_TAG_HASH) {
																									vala_type_reference_set_namespace_name (type, "GLib");
																									vala_type_reference_set_type_name (type, "HashTable");
																								} else {
																									if (type_node->tag == TYPE_TAG_ERROR) {
																										vala_type_reference_set_namespace_name (type, "GLib");
																										vala_type_reference_set_type_name (type, "Error");
																									} else {
																										if (type_node->is_interface) {
																											const char* _tmp1;
																											char* n;
																											_tmp1 = NULL;
																											n = (_tmp1 = type_node->interface, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
																											if (g_utf8_collate (n, "") == 0) {
																												gpointer _tmp2;
																												return (_tmp2 = NULL, (n = (g_free (n), NULL)), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp2);
																											}
																											if (g_str_has_prefix (n, "const-")) {
																												char* _tmp4;
																												const char* _tmp3;
																												_tmp4 = NULL;
																												_tmp3 = NULL;
																												n = (_tmp4 = (_tmp3 = g_utf8_offset_to_pointer (n, g_utf8_strlen ("const-", -1)), (_tmp3 == NULL ? NULL : g_strdup (_tmp3))), (n = (g_free (n), NULL)), _tmp4);
																											}
																											if (type_node->is_pointer && (g_utf8_collate (n, "gchar") == 0 || g_utf8_collate (n, "char") == 0)) {
																												vala_type_reference_set_type_name (type, "string");
																												if (g_str_has_suffix (type_node->unparsed, "**")) {
																													vala_type_reference_set_is_out (type, TRUE);
																												}
																											} else {
																												if (g_utf8_collate (n, "gunichar") == 0) {
																													vala_type_reference_set_type_name (type, "unichar");
																												} else {
																													if (g_utf8_collate (n, "gchar") == 0) {
																														vala_type_reference_set_type_name (type, "char");
																													} else {
																														if (g_utf8_collate (n, "guchar") == 0 || g_utf8_collate (n, "guint8") == 0) {
																															vala_type_reference_set_type_name (type, "uchar");
																															if (type_node->is_pointer) {
																																vala_type_reference_set_array_rank (type, 1);
																															}
																														} else {
																															if (g_utf8_collate (n, "gushort") == 0) {
																																vala_type_reference_set_type_name (type, "ushort");
																															} else {
																																if (g_utf8_collate (n, "gshort") == 0) {
																																	vala_type_reference_set_type_name (type, "short");
																																} else {
																																	if (g_utf8_collate (n, "gconstpointer") == 0 || g_utf8_collate (n, "void") == 0) {
																																		vala_type_reference_set_type_name (type, "pointer");
																																	} else {
																																		if (g_utf8_collate (n, "goffset") == 0) {
																																			vala_type_reference_set_type_name (type, "int64");
																																		} else {
																																			if (g_utf8_collate (n, "value_array") == 0) {
																																				vala_type_reference_set_namespace_name (type, "GLib");
																																				vala_type_reference_set_type_name (type, "ValueArray");
																																			} else {
																																				if (g_utf8_collate (n, "time_t") == 0) {
																																					vala_type_reference_set_type_name (type, "ulong");
																																				} else {
																																					if (g_utf8_collate (n, "mode_t") == 0) {
																																						vala_type_reference_set_type_name (type, "uint");
																																					} else {
																																						if (g_utf8_collate (n, "gint") == 0 || g_utf8_collate (n, "pid_t") == 0) {
																																							vala_type_reference_set_type_name (type, "int");
																																						} else {
																																							if (g_utf8_collate (n, "unsigned") == 0 || g_utf8_collate (n, "unsigned-int") == 0) {
																																								vala_type_reference_set_type_name (type, "uint");
																																							} else {
																																								if (g_utf8_collate (n, "FILE") == 0) {
																																									vala_type_reference_set_namespace_name (type, "GLib");
																																									vala_type_reference_set_type_name (type, "FileStream");
																																								} else {
																																									if (g_utf8_collate (n, "GType") == 0) {
																																										vala_type_reference_set_namespace_name (type, "GLib");
																																										vala_type_reference_set_type_name (type, "Type");
																																										if (type_node->is_pointer) {
																																											vala_type_reference_set_array_rank (type, 1);
																																										}
																																									} else {
																																										vala_gidl_parser_parse_type_string (self, type, n);
																																										if (type_node->is_pointer && vala_gidl_parser_is_value_type (self, n)) {
																																											vala_type_reference_set_is_out (type, TRUE);
																																										}
																																									}
																																								}
																																							}
																																						}
																																					}
																																				}
																																			}
																																		}
																																	}
																																}
																															}
																														}
																													}
																												}
																											}
																											(n = (g_free (n), NULL));
																										} else {
																											fprintf (stdout, "%d\n", type_node->tag);
																										}
																									}
																								}
																							}
																						}
																					}
																				}
																			}
																		}
																	}
																}
															}
														}
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	return type;
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


static gboolean vala_gidl_parser_is_value_type (ValaGIdlParser* self, const char* type_name) {
	char** _tmp0;
	gint type_attributes_length1;
	char** type_attributes;
	gboolean _tmp7;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), FALSE);
	g_return_val_if_fail (type_name != NULL, FALSE);
	_tmp0 = NULL;
	type_attributes = (_tmp0 = vala_gidl_parser_get_attributes (self, type_name), type_attributes_length1 = -1, _tmp0);
	if (type_attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = type_attributes;
			for (attr_it = 0; (type_attributes_length1 != -1 && attr_it < type_attributes_length1) || (type_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp5;
				char* attr;
				_tmp5 = NULL;
				attr = (_tmp5 = attr_collection[attr_it], (_tmp5 == NULL ? NULL : g_strdup (_tmp5)));
				{
					char** _tmp1;
					gint nv_length1;
					char** nv;
					char* _tmp2;
					gboolean _tmp3;
					/* type in the same package*/
					_tmp1 = NULL;
					nv = (_tmp1 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp1);
					_tmp2 = NULL;
					if ((_tmp3 = g_utf8_collate (nv[0], "is_value_type") == 0 && g_utf8_collate ((_tmp2 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp2 = (g_free (_tmp2), NULL)), _tmp3)) {
						gboolean _tmp4;
						return (_tmp4 = TRUE, (attr = (g_free (attr), NULL)), (nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL)), (type_attributes = (_vala_array_free (type_attributes, type_attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp4);
					}
					(attr = (g_free (attr), NULL));
					(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
				}
			}
		}
	} else {
		ValaDataType* dt;
		/* type in a dependency package*/
		dt = gee_map_get (((GeeMap*) self->priv->cname_type_map), type_name);
		if (dt != NULL) {
			gboolean _tmp6;
			return (_tmp6 = !vala_data_type_is_reference_type (dt), (dt == NULL ? NULL : (dt = (g_object_unref (dt), NULL))), (type_attributes = (_vala_array_free (type_attributes, type_attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp6);
		}
		(dt == NULL ? NULL : (dt = (g_object_unref (dt), NULL)));
	}
	return (_tmp7 = FALSE, (type_attributes = (_vala_array_free (type_attributes, type_attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp7);
	(type_attributes = (_vala_array_free (type_attributes, type_attributes_length1, ((GDestroyNotify) g_free)), NULL));
}


static void vala_gidl_parser_parse_type_string (ValaGIdlParser* self, ValaTypeReference* type, const char* n) {
	ValaDataType* dt;
	char** _tmp0;
	gint type_attributes_length1;
	char** type_attributes;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (type));
	g_return_if_fail (n != NULL);
	dt = gee_map_get (((GeeMap*) self->priv->cname_type_map), n);
	if (dt != NULL) {
		vala_type_reference_set_namespace_name (type, vala_symbol_get_name (vala_symbol_get_parent_symbol (VALA_SYMBOL (dt))));
		vala_type_reference_set_type_name (type, vala_symbol_get_name (VALA_SYMBOL (dt)));
		(dt == NULL ? NULL : (dt = (g_object_unref (dt), NULL)));
		return;
	}
	_tmp0 = NULL;
	type_attributes = (_tmp0 = vala_gidl_parser_get_attributes (self, n), type_attributes_length1 = -1, _tmp0);
	if (NULL != type_attributes) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = type_attributes;
			for (attr_it = 0; (type_attributes_length1 != -1 && attr_it < type_attributes_length1) || (type_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp3;
				char* attr;
				_tmp3 = NULL;
				attr = (_tmp3 = attr_collection[attr_it], (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
				{
					char** _tmp1;
					gint nv_length1;
					char** nv;
					_tmp1 = NULL;
					nv = (_tmp1 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp1);
					if (g_utf8_collate (nv[0], "cprefix") == 0) {
						char* _tmp2;
						_tmp2 = NULL;
						vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ((_tmp2 = vala_gidl_parser_eval (self, nv[1])), -1)));
						(_tmp2 = (g_free (_tmp2), NULL));
						(attr = (g_free (attr), NULL));
						(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
						(dt == NULL ? NULL : (dt = (g_object_unref (dt), NULL)));
						(type_attributes = (_vala_array_free (type_attributes, type_attributes_length1, ((GDestroyNotify) g_free)), NULL));
						return;
					}
					(attr = (g_free (attr), NULL));
					(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
				}
			}
		}
	}
	if (g_utf8_collate (n, "HFONT") == 0 || g_utf8_collate (n, "HGLOBAL") == 0 || g_utf8_collate (n, "GStaticRecMutex") == 0 || g_str_has_suffix (n, "Class") || g_utf8_collate (n, "va_list") == 0 || g_str_has_prefix (n, "LOGFONT") || g_str_has_prefix (n, "xml") || g_utf8_collate (n, "GdkNativeWindow") == 0 || g_utf8_collate (n, "GdkXEvent") == 0 || g_utf8_collate (n, "GtkTextLayout") == 0 || g_utf8_collate (n, "GstClockID") == 0 || g_str_has_prefix (n, "GstXml")) {
		/* unsupported*/
		vala_type_reference_set_type_name (type, "pointer");
	} else {
		if (g_str_has_prefix (n, "cairo")) {
			vala_type_reference_set_namespace_name (type, "Cairo");
			if (g_utf8_collate (n, "cairo_t") == 0) {
				vala_type_reference_set_type_name (type, "Context");
			} else {
				if (g_utf8_collate (n, "cairo_surface_t") == 0) {
					vala_type_reference_set_type_name (type, "Surface");
				} else {
					vala_type_reference_set_namespace_name (type, NULL);
					vala_type_reference_set_type_name (type, "pointer");
				}
			}
		} else {
			if (g_utf8_collate (n, "AtkAttributeSet") == 0) {
				vala_type_reference_set_namespace_name (type, "GLib");
				vala_type_reference_set_type_name (type, "SList");
			} else {
				if (g_utf8_collate (n, "GstClockTime") == 0) {
					vala_type_reference_set_type_name (type, "uint64");
				} else {
					if (g_utf8_collate (n, "GstClockTimeDiff") == 0) {
						vala_type_reference_set_type_name (type, "int64");
					} else {
						if (g_utf8_collate (n, "PangoGlyph") == 0) {
							vala_type_reference_set_type_name (type, "uint");
						} else {
							if (g_utf8_collate (n, "PangoGlyphUnit") == 0) {
								vala_type_reference_set_type_name (type, "int");
							} else {
								if (g_utf8_collate (n, "ClutterFixed") == 0 || g_utf8_collate (n, "ClutterUnit") == 0 || g_utf8_collate (n, "ClutterAngle") == 0) {
									vala_type_reference_set_type_name (type, "int32");
								} else {
									if (g_utf8_collate (n, "SoupProtocol") == 0) {
										vala_type_reference_set_namespace_name (type, "GLib");
										vala_type_reference_set_type_name (type, "Quark");
									} else {
										if (g_utf8_collate (n, "GStrv") == 0) {
											vala_type_reference_set_type_name (type, "string");
											vala_type_reference_set_array_rank (type, 1);
										} else {
											if (g_str_has_prefix (n, vala_symbol_get_name (VALA_SYMBOL (self->priv->current_namespace)))) {
												vala_type_reference_set_namespace_name (type, vala_symbol_get_name (VALA_SYMBOL (self->priv->current_namespace)));
												vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen (vala_symbol_get_name (VALA_SYMBOL (self->priv->current_namespace)), -1)));
											} else {
												if (g_str_has_prefix (n, "G")) {
													vala_type_reference_set_namespace_name (type, "GLib");
													vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, 1));
												} else {
													char** _tmp4;
													gint name_parts_length1;
													char** name_parts;
													_tmp4 = NULL;
													name_parts = (_tmp4 = g_strsplit (n, ".", 2), name_parts_length1 = -1, _tmp4);
													if (name_parts[1] == NULL) {
														vala_type_reference_set_type_name (type, name_parts[0]);
													} else {
														vala_type_reference_set_namespace_name (type, name_parts[0]);
														vala_type_reference_set_type_name (type, name_parts[1]);
													}
													(name_parts = (_vala_array_free (name_parts, name_parts_length1, ((GDestroyNotify) g_free)), NULL));
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	(dt == NULL ? NULL : (dt = (g_object_unref (dt), NULL)));
	(type_attributes = (_vala_array_free (type_attributes, type_attributes_length1, ((GDestroyNotify) g_free)), NULL));
}


static ValaTypeReference* vala_gidl_parser_parse_param (ValaGIdlParser* self, GIdlNodeParam* param) {
	ValaTypeReference* type;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (param != NULL, NULL);
	type = vala_gidl_parser_parse_type (self, param->type);
	return type;
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


/* disable for now as null_ok not yet correctly set
 type.non_null = !param.null_ok;*/
static ValaMethod* vala_gidl_parser_parse_function (ValaGIdlParser* self, GIdlNodeFunction* f, gboolean is_virtual, gboolean is_interface) {
	GIdlNode* node;
	ValaTypeReference* return_type;
	ValaMethod* m;
	gboolean add_ellipsis;
	char** _tmp13;
	gint attributes_length1;
	char** attributes;
	gboolean first;
	ValaFormalParameter* last_param;
	ValaMethod* _tmp40;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (f != NULL, NULL);
	node = ((GIdlNode*) f);
	if (f->deprecated) {
		return NULL;
	}
	return_type = NULL;
	if (f->result != NULL) {
		ValaTypeReference* _tmp1;
		_tmp1 = NULL;
		return_type = (_tmp1 = vala_gidl_parser_parse_param (self, f->result), (return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL))), _tmp1);
	}
	m = NULL;
	if (!is_interface && (f->is_constructor || g_str_has_prefix (node->name, "new"))) {
		ValaMethod* _tmp2;
		_tmp2 = NULL;
		m = (_tmp2 = VALA_METHOD (vala_creation_method_new (node->name, self->priv->current_source_reference)), (m == NULL ? NULL : (m = (g_object_unref (m), NULL))), _tmp2);
		if (g_utf8_collate (vala_symbol_get_name (VALA_SYMBOL (m)), "new") == 0) {
			vala_symbol_set_name (VALA_SYMBOL (m), NULL);
		} else {
			if (g_str_has_prefix (vala_symbol_get_name (VALA_SYMBOL (m)), "new_")) {
				vala_symbol_set_name (VALA_SYMBOL (m), g_utf8_offset_to_pointer (vala_symbol_get_name (VALA_SYMBOL (m)), g_utf8_strlen ("new_", -1)));
			}
		}
	} else {
		ValaMethod* _tmp3;
		_tmp3 = NULL;
		m = (_tmp3 = vala_method_new (node->name, return_type, self->priv->current_source_reference), (m == NULL ? NULL : (m = (g_object_unref (m), NULL))), _tmp3);
	}
	vala_symbol_set_access (VALA_SYMBOL (m), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_method_set_is_virtual (m, is_virtual && !is_interface);
	vala_method_set_is_abstract (m, is_virtual && is_interface);
	/* GIDL generator can't provide array parameter information yet*/
	vala_method_set_no_array_length (m, TRUE);
	if (self->priv->current_type_symbol_set != NULL) {
		gee_collection_add (GEE_COLLECTION (self->priv->current_type_symbol_set), node->name);
	}
	if (self->priv->current_data_type != NULL) {
		char** _tmp7;
		gint sig_attributes_length1;
		char* _tmp5;
		char* _tmp4;
		char** _tmp6;
		char** sig_attributes;
		_tmp7 = NULL;
		_tmp5 = NULL;
		_tmp4 = NULL;
		_tmp6 = NULL;
		sig_attributes = (_tmp7 = (_tmp6 = vala_gidl_parser_get_attributes (self, (_tmp5 = g_strdup_printf ("%s::%s", (_tmp4 = vala_data_type_get_cname (self->priv->current_data_type, FALSE)), node->name))), (_tmp5 = (g_free (_tmp5), NULL)), (_tmp4 = (g_free (_tmp4), NULL)), _tmp6), sig_attributes_length1 = -1, _tmp7);
		if (sig_attributes != NULL) {
			{
				char** attr_collection;
				int attr_it;
				attr_collection = sig_attributes;
				for (attr_it = 0; (sig_attributes_length1 != -1 && attr_it < sig_attributes_length1) || (sig_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
					const char* _tmp12;
					char* attr;
					_tmp12 = NULL;
					attr = (_tmp12 = attr_collection[attr_it], (_tmp12 == NULL ? NULL : g_strdup (_tmp12)));
					{
						char** _tmp8;
						gint nv_length1;
						char** nv;
						char* _tmp9;
						gboolean _tmp10;
						_tmp8 = NULL;
						nv = (_tmp8 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp8);
						_tmp9 = NULL;
						if ((_tmp10 = g_utf8_collate (nv[0], "has_emitter") == 0 && g_utf8_collate ((_tmp9 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp9 = (g_free (_tmp9), NULL)), _tmp10)) {
							gpointer _tmp11;
							return (_tmp11 = NULL, (attr = (g_free (attr), NULL)), (nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL)), (sig_attributes = (_vala_array_free (sig_attributes, sig_attributes_length1, ((GDestroyNotify) g_free)), NULL)), (return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL))), (m == NULL ? NULL : (m = (g_object_unref (m), NULL))), _tmp11);
						}
						(attr = (g_free (attr), NULL));
						(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
					}
				}
			}
		}
		(sig_attributes = (_vala_array_free (sig_attributes, sig_attributes_length1, ((GDestroyNotify) g_free)), NULL));
	}
	add_ellipsis = FALSE;
	_tmp13 = NULL;
	attributes = (_tmp13 = vala_gidl_parser_get_attributes (self, f->symbol), attributes_length1 = -1, _tmp13);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp24;
				char* attr;
				_tmp24 = NULL;
				attr = (_tmp24 = attr_collection[attr_it], (_tmp24 == NULL ? NULL : g_strdup (_tmp24)));
				{
					char** _tmp14;
					gint nv_length1;
					char** nv;
					_tmp14 = NULL;
					nv = (_tmp14 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp14);
					if (g_utf8_collate (nv[0], "name") == 0) {
						char* _tmp15;
						vala_method_set_cname (m, vala_symbol_get_name (VALA_SYMBOL (m)));
						_tmp15 = NULL;
						vala_symbol_set_name (VALA_SYMBOL (m), (_tmp15 = vala_gidl_parser_eval (self, nv[1])));
						(_tmp15 = (g_free (_tmp15), NULL));
					} else {
						if (g_utf8_collate (nv[0], "hidden") == 0) {
							char* _tmp16;
							gboolean _tmp17;
							_tmp16 = NULL;
							if ((_tmp17 = g_utf8_collate ((_tmp16 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp16 = (g_free (_tmp16), NULL)), _tmp17)) {
								gpointer _tmp18;
								return (_tmp18 = NULL, (attr = (g_free (attr), NULL)), (nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL)), (return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL))), (m == NULL ? NULL : (m = (g_object_unref (m), NULL))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp18);
							}
						} else {
							if (g_utf8_collate (nv[0], "ellipsis") == 0) {
								char* _tmp19;
								gboolean _tmp20;
								_tmp19 = NULL;
								if ((_tmp20 = g_utf8_collate ((_tmp19 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp19 = (g_free (_tmp19), NULL)), _tmp20)) {
									add_ellipsis = TRUE;
								}
							} else {
								if (g_utf8_collate (nv[0], "transfer_ownership") == 0) {
									char* _tmp21;
									gboolean _tmp22;
									_tmp21 = NULL;
									if ((_tmp22 = g_utf8_collate ((_tmp21 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp21 = (g_free (_tmp21), NULL)), _tmp22)) {
										vala_type_reference_set_transfers_ownership (return_type, TRUE);
									}
								} else {
									if (g_utf8_collate (nv[0], "sentinel") == 0) {
										char* _tmp23;
										_tmp23 = NULL;
										vala_method_set_sentinel (m, (_tmp23 = vala_gidl_parser_eval (self, nv[1])));
										(_tmp23 = (g_free (_tmp23), NULL));
									}
								}
							}
						}
					}
					(attr = (g_free (attr), NULL));
					(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
				}
			}
		}
	}
	vala_method_set_cname (m, f->symbol);
	first = TRUE;
	last_param = NULL;
	{
		GList* param_collection;
		GList* param_it;
		param_collection = f->parameters;
		for (param_it = param_collection; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = param_it->data;
			{
				GIdlNode* param_node;
				ValaTypeReference* _tmp28;
				ValaFormalParameter* _tmp29;
				ValaFormalParameter* p;
				char** _tmp32;
				gint attributes_length1;
				char* _tmp30;
				char** _tmp31;
				char** attributes;
				char* _tmp35;
				gboolean _tmp36;
				ValaFormalParameter* _tmp38;
				ValaFormalParameter* _tmp37;
				param_node = ((GIdlNode*) param);
				if (first) {
					char* _tmp25;
					gboolean _tmp26;
					first = FALSE;
					_tmp25 = NULL;
					if ((_tmp26 = !(VALA_IS_CREATION_METHOD (m)) && self->priv->current_data_type != NULL && param->type->is_interface && (g_utf8_collate (param_node->name, "self") == 0 || g_str_has_suffix (param->type->interface, (_tmp25 = vala_data_type_get_cname (self->priv->current_data_type, FALSE)))), (_tmp25 = (g_free (_tmp25), NULL)), _tmp26)) {
						continue;
					} else {
						/* instance method
						 static method*/
						vala_method_set_instance (m, FALSE);
					}
				}
				if (param->type->is_error) {
					ValaTypeReference* _tmp27;
					_tmp27 = NULL;
					vala_method_add_error_domain (m, (_tmp27 = vala_gidl_parser_parse_type (self, param->type)));
					(_tmp27 == NULL ? NULL : (_tmp27 = (g_object_unref (_tmp27), NULL)));
					continue;
				}
				_tmp28 = NULL;
				_tmp29 = NULL;
				p = (_tmp29 = vala_formal_parameter_new (param_node->name, (_tmp28 = vala_gidl_parser_parse_param (self, param)), NULL), (_tmp28 == NULL ? NULL : (_tmp28 = (g_object_unref (_tmp28), NULL))), _tmp29);
				vala_method_add_parameter (m, p);
				_tmp32 = NULL;
				_tmp30 = NULL;
				_tmp31 = NULL;
				attributes = (_tmp32 = (_tmp31 = vala_gidl_parser_get_attributes (self, (_tmp30 = g_strdup_printf ("%s.%s", f->symbol, param_node->name))), (_tmp30 = (g_free (_tmp30), NULL)), _tmp31), attributes_length1 = -1, _tmp32);
				if (attributes != NULL) {
					{
						char** attr_collection;
						int attr_it;
						attr_collection = attributes;
						for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
							const char* _tmp34;
							char* attr;
							_tmp34 = NULL;
							attr = (_tmp34 = attr_collection[attr_it], (_tmp34 == NULL ? NULL : g_strdup (_tmp34)));
							{
								char** _tmp33;
								gint nv_length1;
								char** nv;
								_tmp33 = NULL;
								nv = (_tmp33 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp33);
								if (g_utf8_collate (nv[0], "is_array") == 0) {
									vala_type_reference_set_array_rank (vala_formal_parameter_get_type_reference (p), 1);
									vala_type_reference_set_is_out (vala_formal_parameter_get_type_reference (p), FALSE);
								}
								(attr = (g_free (attr), NULL));
								(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
							}
						}
					}
				}
				_tmp35 = NULL;
				if ((_tmp36 = last_param != NULL && g_utf8_collate (vala_symbol_get_name (VALA_SYMBOL (p)), (_tmp35 = g_strconcat ("n_", vala_symbol_get_name (VALA_SYMBOL (last_param)), NULL))) == 0, (_tmp35 = (g_free (_tmp35), NULL)), _tmp36)) {
					/* last_param is array, p is array length*/
					vala_type_reference_set_array_rank (vala_formal_parameter_get_type_reference (last_param), 1);
					vala_type_reference_set_is_out (vala_formal_parameter_get_type_reference (last_param), FALSE);
				}
				_tmp38 = NULL;
				_tmp37 = NULL;
				last_param = (_tmp38 = (_tmp37 = p, (_tmp37 == NULL ? NULL : g_object_ref (_tmp37))), (last_param == NULL ? NULL : (last_param = (g_object_unref (last_param), NULL))), _tmp38);
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
				(attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL));
			}
		}
	}
	if (first) {
		/* no parameters => static method*/
		vala_method_set_instance (m, FALSE);
	}
	if (last_param != NULL && g_str_has_prefix (vala_symbol_get_name (VALA_SYMBOL (last_param)), "first_")) {
		vala_formal_parameter_set_ellipsis (last_param, TRUE);
	} else {
		if (add_ellipsis) {
			ValaFormalParameter* _tmp39;
			_tmp39 = NULL;
			vala_method_add_parameter (m, (_tmp39 = vala_formal_parameter_new_with_ellipsis (NULL)));
			(_tmp39 == NULL ? NULL : (_tmp39 = (g_object_unref (_tmp39), NULL)));
		}
	}
	_tmp40 = NULL;
	return (_tmp40 = m, (return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), (last_param == NULL ? NULL : (last_param = (g_object_unref (last_param), NULL))), _tmp40);
	(return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL)));
	(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	(attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL));
	(last_param == NULL ? NULL : (last_param = (g_object_unref (last_param), NULL)));
}


static char* vala_gidl_parser_fix_prop_name (ValaGIdlParser* self, const char* name) {
	GString* str;
	const char* _tmp0;
	char* i;
	const char* _tmp3;
	char* _tmp4;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	str = g_string_new ("");
	_tmp0 = NULL;
	i = (_tmp0 = name, (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		char* _tmp2;
		const char* _tmp1;
		c = g_utf8_get_char (i);
		if (c == '-') {
			g_string_append_c (str, '_');
		} else {
			g_string_append_unichar (str, c);
		}
		_tmp2 = NULL;
		_tmp1 = NULL;
		i = (_tmp2 = (_tmp1 = g_utf8_next_char (i), (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (i = (g_free (i), NULL)), _tmp2);
	}
	_tmp3 = NULL;
	_tmp4 = NULL;
	return (_tmp4 = (_tmp3 = str->str, (_tmp3 == NULL ? NULL : g_strdup (_tmp3))), (str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL))), (i = (g_free (i), NULL)), _tmp4);
	(str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL)));
	(i = (g_free (i), NULL));
}


static ValaProperty* vala_gidl_parser_parse_property (ValaGIdlParser* self, GIdlNodeProperty* prop_node) {
	GIdlNode* node;
	ValaPropertyAccessor* get_acc;
	ValaPropertyAccessor* set_acc;
	ValaTypeReference* _tmp4;
	char* _tmp3;
	ValaProperty* _tmp5;
	ValaProperty* prop;
	char** _tmp9;
	gint attributes_length1;
	char* _tmp7;
	char* _tmp6;
	char** _tmp8;
	char** attributes;
	ValaProperty* _tmp15;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (prop_node != NULL, NULL);
	node = ((GIdlNode*) prop_node);
	if (prop_node->deprecated) {
		return NULL;
	}
	if (!prop_node->readable && !prop_node->writable) {
		/* buggy GIDL definition*/
		prop_node->readable = TRUE;
		prop_node->writable = TRUE;
	}
	get_acc = NULL;
	set_acc = NULL;
	if (prop_node->readable) {
		ValaPropertyAccessor* _tmp1;
		_tmp1 = NULL;
		get_acc = (_tmp1 = vala_property_accessor_new (TRUE, FALSE, FALSE, NULL, NULL), (get_acc == NULL ? NULL : (get_acc = (g_object_unref (get_acc), NULL))), _tmp1);
	}
	if (prop_node->writable) {
		ValaPropertyAccessor* _tmp2;
		_tmp2 = NULL;
		set_acc = (_tmp2 = vala_property_accessor_new (FALSE, FALSE, FALSE, NULL, NULL), (set_acc == NULL ? NULL : (set_acc = (g_object_unref (set_acc), NULL))), _tmp2);
		if (prop_node->construct_only) {
			vala_property_accessor_set_construction (set_acc, TRUE);
		} else {
			vala_property_accessor_set_writable (set_acc, TRUE);
			vala_property_accessor_set_construction (set_acc, prop_node->construct);
		}
	}
	_tmp4 = NULL;
	_tmp3 = NULL;
	_tmp5 = NULL;
	prop = (_tmp5 = vala_property_new ((_tmp3 = vala_gidl_parser_fix_prop_name (self, node->name)), (_tmp4 = vala_gidl_parser_parse_type (self, prop_node->type)), get_acc, set_acc, self->priv->current_source_reference), (_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL))), (_tmp3 = (g_free (_tmp3), NULL)), _tmp5);
	vala_symbol_set_access (VALA_SYMBOL (prop), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_property_set_interface_only (prop, TRUE);
	_tmp9 = NULL;
	_tmp7 = NULL;
	_tmp6 = NULL;
	_tmp8 = NULL;
	attributes = (_tmp9 = (_tmp8 = vala_gidl_parser_get_attributes (self, (_tmp7 = g_strdup_printf ("%s:%s", (_tmp6 = vala_data_type_get_cname (self->priv->current_data_type, FALSE)), node->name))), (_tmp7 = (g_free (_tmp7), NULL)), (_tmp6 = (g_free (_tmp6), NULL)), _tmp8), attributes_length1 = -1, _tmp9);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp14;
				char* attr;
				_tmp14 = NULL;
				attr = (_tmp14 = attr_collection[attr_it], (_tmp14 == NULL ? NULL : g_strdup (_tmp14)));
				{
					char** _tmp10;
					gint nv_length1;
					char** nv;
					_tmp10 = NULL;
					nv = (_tmp10 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp10);
					if (g_utf8_collate (nv[0], "hidden") == 0) {
						char* _tmp11;
						gboolean _tmp12;
						_tmp11 = NULL;
						if ((_tmp12 = g_utf8_collate ((_tmp11 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp11 = (g_free (_tmp11), NULL)), _tmp12)) {
							gpointer _tmp13;
							return (_tmp13 = NULL, (attr = (g_free (attr), NULL)), (nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL)), (get_acc == NULL ? NULL : (get_acc = (g_object_unref (get_acc), NULL))), (set_acc == NULL ? NULL : (set_acc = (g_object_unref (set_acc), NULL))), (prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp13);
						}
					}
					(attr = (g_free (attr), NULL));
					(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
				}
			}
		}
	}
	if (self->priv->current_type_symbol_set != NULL) {
		gee_collection_add (GEE_COLLECTION (self->priv->current_type_symbol_set), vala_symbol_get_name (VALA_SYMBOL (prop)));
	}
	_tmp15 = NULL;
	return (_tmp15 = prop, (get_acc == NULL ? NULL : (get_acc = (g_object_unref (get_acc), NULL))), (set_acc == NULL ? NULL : (set_acc = (g_object_unref (set_acc), NULL))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp15);
	(get_acc == NULL ? NULL : (get_acc = (g_object_unref (get_acc), NULL)));
	(set_acc == NULL ? NULL : (set_acc = (g_object_unref (set_acc), NULL)));
	(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
	(attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL));
}


static ValaConstant* vala_gidl_parser_parse_constant (ValaGIdlParser* self, GIdlNodeConstant* const_node) {
	GIdlNode* node;
	ValaTypeReference* type;
	ValaConstant* c;
	ValaConstant* _tmp1;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (const_node != NULL, NULL);
	node = ((GIdlNode*) const_node);
	type = vala_gidl_parser_parse_type (self, const_node->type);
	if (type == NULL) {
		gpointer _tmp0;
		return (_tmp0 = NULL, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp0);
	}
	c = vala_constant_new (node->name, type, NULL, self->priv->current_source_reference);
	_tmp1 = NULL;
	return (_tmp1 = c, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp1);
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
	(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
}


static ValaField* vala_gidl_parser_parse_field (ValaGIdlParser* self, GIdlNodeField* field_node) {
	GIdlNode* node;
	ValaTypeReference* type;
	char** _tmp5;
	gint attributes_length1;
	char* _tmp3;
	char* _tmp2;
	char** _tmp4;
	char** attributes;
	ValaField* field;
	ValaField* _tmp15;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (field_node != NULL, NULL);
	node = ((GIdlNode*) field_node);
	type = vala_gidl_parser_parse_type (self, field_node->type);
	if (type == NULL) {
		gpointer _tmp0;
		return (_tmp0 = NULL, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp0);
	}
	if (!field_node->readable) {
		gpointer _tmp1;
		return (_tmp1 = NULL, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp1);
	}
	_tmp5 = NULL;
	_tmp3 = NULL;
	_tmp2 = NULL;
	_tmp4 = NULL;
	attributes = (_tmp5 = (_tmp4 = vala_gidl_parser_get_attributes (self, (_tmp3 = g_strdup_printf ("%s.%s", (_tmp2 = vala_data_type_get_cname (self->priv->current_data_type, FALSE)), node->name))), (_tmp3 = (g_free (_tmp3), NULL)), (_tmp2 = (g_free (_tmp2), NULL)), _tmp4), attributes_length1 = -1, _tmp5);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp14;
				char* attr;
				_tmp14 = NULL;
				attr = (_tmp14 = attr_collection[attr_it], (_tmp14 == NULL ? NULL : g_strdup (_tmp14)));
				{
					char** _tmp6;
					gint nv_length1;
					char** nv;
					_tmp6 = NULL;
					nv = (_tmp6 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp6);
					if (g_utf8_collate (nv[0], "hidden") == 0) {
						char* _tmp7;
						gboolean _tmp8;
						_tmp7 = NULL;
						if ((_tmp8 = g_utf8_collate ((_tmp7 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp7 = (g_free (_tmp7), NULL)), _tmp8)) {
							gpointer _tmp9;
							return (_tmp9 = NULL, (attr = (g_free (attr), NULL)), (nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL)), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp9);
						}
					} else {
						if (g_utf8_collate (nv[0], "is_array") == 0) {
							char* _tmp10;
							gboolean _tmp11;
							_tmp10 = NULL;
							if ((_tmp11 = g_utf8_collate ((_tmp10 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp10 = (g_free (_tmp10), NULL)), _tmp11)) {
								vala_type_reference_set_array_rank (type, 1);
							}
						} else {
							if (g_utf8_collate (nv[0], "weak") == 0) {
								char* _tmp12;
								gboolean _tmp13;
								_tmp12 = NULL;
								if ((_tmp13 = g_utf8_collate ((_tmp12 = vala_gidl_parser_eval (self, nv[1])), "0") == 0, (_tmp12 = (g_free (_tmp12), NULL)), _tmp13)) {
									vala_type_reference_set_takes_ownership (type, TRUE);
								}
							}
						}
					}
					(attr = (g_free (attr), NULL));
					(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
				}
			}
		}
	}
	if (self->priv->current_type_symbol_set != NULL) {
		gee_collection_add (GEE_COLLECTION (self->priv->current_type_symbol_set), node->name);
	}
	field = vala_field_new (node->name, type, NULL, self->priv->current_source_reference);
	vala_symbol_set_access (VALA_SYMBOL (field), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp15 = NULL;
	return (_tmp15 = field, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp15);
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
	(attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL));
	(field == NULL ? NULL : (field = (g_object_unref (field), NULL)));
}


static char** vala_gidl_parser_get_attributes (ValaGIdlParser* self, const char* codenode) {
	char* attributes;
	char** _tmp2;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (codenode != NULL, NULL);
	attributes = gee_map_get (self->priv->codenode_attributes_map, codenode);
	if (attributes == NULL) {
		gboolean dot_required;
		gboolean colon_required;
		GeeSet* pattern_specs;
		dot_required = (NULL != g_utf8_strchr (codenode, -1, '.'));
		colon_required = (NULL != g_utf8_strchr (codenode, -1, ':'));
		pattern_specs = gee_map_get_keys (self->priv->codenode_attributes_patterns);
		{
			GeeSet* pattern_collection;
			GeeIterator* pattern_it;
			pattern_collection = pattern_specs;
			pattern_it = gee_iterable_iterator (GEE_ITERABLE (pattern_collection));
			while (gee_iterator_next (pattern_it)) {
				GPatternSpec* pattern;
				pattern = gee_iterator_get (pattern_it);
				{
					char* pspec;
					pspec = gee_map_get (((GeeMap*) self->priv->codenode_attributes_patterns), pattern);
					if ((dot_required && NULL == g_utf8_strchr (pspec, -1, '.')) || (colon_required && NULL == g_utf8_strchr (pspec, -1, ':'))) {
						(pspec = (g_free (pspec), NULL));
						continue;
					}
					if (g_pattern_match_string (pattern, codenode)) {
						char** _tmp0;
						_tmp0 = NULL;
						return (_tmp0 = vala_gidl_parser_get_attributes (self, pspec), (pspec = (g_free (pspec), NULL)), (pattern_it == NULL ? NULL : (pattern_it = (g_object_unref (pattern_it), NULL))), (pattern_specs == NULL ? NULL : (pattern_specs = (g_object_unref (pattern_specs), NULL))), (attributes = (g_free (attributes), NULL)), _tmp0);
					}
					(pspec = (g_free (pspec), NULL));
				}
			}
			(pattern_it == NULL ? NULL : (pattern_it = (g_object_unref (pattern_it), NULL)));
		}
		(pattern_specs == NULL ? NULL : (pattern_specs = (g_object_unref (pattern_specs), NULL)));
	}
	if (attributes == NULL) {
		gpointer _tmp1;
		return (_tmp1 = NULL, (attributes = (g_free (attributes), NULL)), _tmp1);
	}
	_tmp2 = NULL;
	return (_tmp2 = g_strsplit (attributes, " ", 0), (attributes = (g_free (attributes), NULL)), _tmp2);
	(attributes = (g_free (attributes), NULL));
}


static char* vala_gidl_parser_eval (ValaGIdlParser* self, const char* s) {
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (s != NULL, NULL);
	return g_strndup (g_utf8_offset_to_pointer (s, 1), strlen (s) - 2);
}


static ValaSignal* vala_gidl_parser_parse_signal (ValaGIdlParser* self, GIdlNodeSignal* sig_node) {
	GIdlNode* node;
	ValaTypeReference* _tmp2;
	char* _tmp1;
	ValaSignal* _tmp3;
	ValaSignal* sig;
	char** _tmp7;
	gint attributes_length1;
	char* _tmp5;
	char* _tmp4;
	char** _tmp6;
	char** attributes;
	gboolean first;
	ValaSignal* _tmp18;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (sig_node != NULL, NULL);
	node = ((GIdlNode*) sig_node);
	if (sig_node->deprecated || sig_node->result == NULL) {
		return NULL;
	}
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp3 = NULL;
	sig = (_tmp3 = vala_signal_new ((_tmp1 = vala_gidl_parser_fix_prop_name (self, node->name)), (_tmp2 = vala_gidl_parser_parse_param (self, sig_node->result)), self->priv->current_source_reference), (_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL))), (_tmp1 = (g_free (_tmp1), NULL)), _tmp3);
	vala_symbol_set_access (VALA_SYMBOL (sig), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp7 = NULL;
	_tmp5 = NULL;
	_tmp4 = NULL;
	_tmp6 = NULL;
	attributes = (_tmp7 = (_tmp6 = vala_gidl_parser_get_attributes (self, (_tmp5 = g_strdup_printf ("%s::%s", (_tmp4 = vala_data_type_get_cname (self->priv->current_data_type, FALSE)), vala_symbol_get_name (VALA_SYMBOL (sig))))), (_tmp5 = (g_free (_tmp5), NULL)), (_tmp4 = (g_free (_tmp4), NULL)), _tmp6), attributes_length1 = -1, _tmp7);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp15;
				char* attr;
				_tmp15 = NULL;
				attr = (_tmp15 = attr_collection[attr_it], (_tmp15 == NULL ? NULL : g_strdup (_tmp15)));
				{
					char** _tmp8;
					gint nv_length1;
					char** nv;
					_tmp8 = NULL;
					nv = (_tmp8 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp8);
					if (g_utf8_collate (nv[0], "name") == 0) {
						char* _tmp9;
						vala_signal_set_cname (sig, vala_symbol_get_name (VALA_SYMBOL (sig)));
						_tmp9 = NULL;
						vala_symbol_set_name (VALA_SYMBOL (sig), (_tmp9 = vala_gidl_parser_eval (self, nv[1])));
						(_tmp9 = (g_free (_tmp9), NULL));
					} else {
						char* _tmp10;
						gboolean _tmp11;
						_tmp10 = NULL;
						if ((_tmp11 = g_utf8_collate (nv[0], "has_emitter") == 0 && g_utf8_collate ((_tmp10 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp10 = (g_free (_tmp10), NULL)), _tmp11)) {
							vala_signal_set_has_emitter (sig, TRUE);
						} else {
							if (g_utf8_collate (nv[0], "hidden") == 0) {
								char* _tmp12;
								gboolean _tmp13;
								_tmp12 = NULL;
								if ((_tmp13 = g_utf8_collate ((_tmp12 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp12 = (g_free (_tmp12), NULL)), _tmp13)) {
									gpointer _tmp14;
									return (_tmp14 = NULL, (attr = (g_free (attr), NULL)), (nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL)), (sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp14);
								}
							}
						}
					}
					(attr = (g_free (attr), NULL));
					(nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL));
				}
			}
		}
	}
	first = TRUE;
	{
		GList* param_collection;
		GList* param_it;
		param_collection = sig_node->parameters;
		for (param_it = param_collection; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = param_it->data;
			{
				GIdlNode* param_node;
				ValaTypeReference* _tmp16;
				ValaFormalParameter* _tmp17;
				ValaFormalParameter* p;
				if (first) {
					/* ignore implicit first signal parameter (sender)*/
					first = FALSE;
					continue;
				}
				param_node = ((GIdlNode*) param);
				_tmp16 = NULL;
				_tmp17 = NULL;
				p = (_tmp17 = vala_formal_parameter_new (param_node->name, (_tmp16 = vala_gidl_parser_parse_param (self, param)), NULL), (_tmp16 == NULL ? NULL : (_tmp16 = (g_object_unref (_tmp16), NULL))), _tmp17);
				vala_signal_add_parameter (sig, p);
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
	}
	_tmp18 = NULL;
	return (_tmp18 = sig, (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp18);
	(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
	(attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL));
}


/**
 * Code visitor parsing all GIDL files.
 */
ValaGIdlParser* vala_gidl_parser_new (void) {
	ValaGIdlParser * self;
	self = g_object_newv (VALA_TYPE_GIDL_PARSER, 0, NULL);
	return self;
}


static void vala_gidl_parser_class_init (ValaGIdlParserClass * klass) {
	vala_gidl_parser_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaGIdlParserPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_gidl_parser_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_namespace = vala_gidl_parser_real_visit_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_gidl_parser_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_gidl_parser_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_gidl_parser_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum = vala_gidl_parser_real_visit_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_callback = vala_gidl_parser_real_visit_callback;
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_gidl_parser_real_visit_source_file;
}


static void vala_gidl_parser_init (ValaGIdlParser * self) {
	self->priv = VALA_GIDL_PARSER_GET_PRIVATE (self);
}


static void vala_gidl_parser_dispose (GObject * obj) {
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (obj);
	(self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL)));
	(self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (g_object_unref (self->priv->current_source_file), NULL)));
	(self->priv->current_source_reference == NULL ? NULL : (self->priv->current_source_reference = (g_object_unref (self->priv->current_source_reference), NULL)));
	(self->priv->current_namespace == NULL ? NULL : (self->priv->current_namespace = (g_object_unref (self->priv->current_namespace), NULL)));
	(self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL)));
	(self->priv->codenode_attributes_map == NULL ? NULL : (self->priv->codenode_attributes_map = (g_object_unref (self->priv->codenode_attributes_map), NULL)));
	(self->priv->codenode_attributes_patterns == NULL ? NULL : (self->priv->codenode_attributes_patterns = (g_object_unref (self->priv->codenode_attributes_patterns), NULL)));
	(self->priv->current_type_symbol_set == NULL ? NULL : (self->priv->current_type_symbol_set = (g_object_unref (self->priv->current_type_symbol_set), NULL)));
	(self->priv->cname_type_map == NULL ? NULL : (self->priv->cname_type_map = (g_object_unref (self->priv->cname_type_map), NULL)));
	G_OBJECT_CLASS (vala_gidl_parser_parent_class)->dispose (obj);
}


GType vala_gidl_parser_get_type (void) {
	static GType vala_gidl_parser_type_id = 0;
	if (G_UNLIKELY (vala_gidl_parser_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaGIdlParserClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_gidl_parser_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaGIdlParser), 0, (GInstanceInitFunc) vala_gidl_parser_init };
		vala_gidl_parser_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaGIdlParser", &g_define_type_info, 0);
	}
	return vala_gidl_parser_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}




