/* valabinaryexpression.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valabinaryexpression.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>

struct _ValaBinaryExpressionPrivate {
	ValaBinaryOperator _operator;
	ValaExpression* _left;
	ValaExpression* _right;
};
#define VALA_BINARY_EXPRESSION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_BINARY_EXPRESSION, ValaBinaryExpressionPrivate))
enum  {
	VALA_BINARY_EXPRESSION_DUMMY_PROPERTY,
	VALA_BINARY_EXPRESSION_OPERATOR,
	VALA_BINARY_EXPRESSION_LEFT,
	VALA_BINARY_EXPRESSION_RIGHT
};
static void vala_binary_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_binary_expression_real_replace (ValaCodeNode* base, ValaCodeNode* old_node, ValaCodeNode* new_node);
static gpointer vala_binary_expression_parent_class = NULL;
static void vala_binary_expression_dispose (GObject * obj);


/**
 * Creates a new binary expression.
 *
 * @param op     binary operator
 * @param left   left operand
 * @param right  right operand
 * @param source reference to source code
 * @return       newly created binary expression
 */
ValaBinaryExpression* vala_binary_expression_new (ValaBinaryOperator op, ValaExpression* _left, ValaExpression* _right, ValaSourceReference* source) {
	GParameter * __params;
	GParameter * __params_it;
	ValaBinaryExpression * self;
	g_return_val_if_fail (VALA_IS_EXPRESSION (_left), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (_right), NULL);
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	__params = g_new0 (GParameter, 2);
	__params_it = __params;
	(__params_it->name = "left", g_value_init (&__params_it->value, VALA_TYPE_EXPRESSION), g_value_set_object (&__params_it->value, _left), __params_it++);
	(__params_it->name = "right", g_value_init (&__params_it->value, VALA_TYPE_EXPRESSION), g_value_set_object (&__params_it->value, _right), __params_it++);
	self = g_object_newv (VALA_TYPE_BINARY_EXPRESSION, __params_it - __params, __params);
	vala_binary_expression_set_operator (self, op);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


static void vala_binary_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaBinaryExpression * self;
	self = VALA_BINARY_EXPRESSION (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (vala_binary_expression_get_left (self)), visitor);
	vala_code_node_accept (VALA_CODE_NODE (vala_binary_expression_get_right (self)), visitor);
	vala_code_visitor_visit_binary_expression (visitor, self);
}


static void vala_binary_expression_real_replace (ValaCodeNode* base, ValaCodeNode* old_node, ValaCodeNode* new_node) {
	ValaBinaryExpression * self;
	self = VALA_BINARY_EXPRESSION (base);
	g_return_if_fail (VALA_IS_CODE_NODE (old_node));
	g_return_if_fail (VALA_IS_CODE_NODE (new_node));
	if (VALA_CODE_NODE (vala_binary_expression_get_left (self)) == old_node) {
		vala_binary_expression_set_left (self, VALA_EXPRESSION (new_node));
	}
	if (VALA_CODE_NODE (vala_binary_expression_get_right (self)) == old_node) {
		vala_binary_expression_set_right (self, VALA_EXPRESSION (new_node));
	}
}


ValaBinaryOperator vala_binary_expression_get_operator (ValaBinaryExpression* self) {
	g_return_val_if_fail (VALA_IS_BINARY_EXPRESSION (self), 0);
	return self->priv->_operator;
}


void vala_binary_expression_set_operator (ValaBinaryExpression* self, ValaBinaryOperator value) {
	g_return_if_fail (VALA_IS_BINARY_EXPRESSION (self));
	self->priv->_operator = value;
}


ValaExpression* vala_binary_expression_get_left (ValaBinaryExpression* self) {
	g_return_val_if_fail (VALA_IS_BINARY_EXPRESSION (self), NULL);
	return self->priv->_left;
}


void vala_binary_expression_set_left (ValaBinaryExpression* self, ValaExpression* value) {
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_BINARY_EXPRESSION (self));
	_tmp1 = NULL;
	self->priv->_left = (_tmp1 = g_object_ref (value), (self->priv->_left == NULL ? NULL : (self->priv->_left = (g_object_unref (self->priv->_left), NULL))), _tmp1);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_left), VALA_CODE_NODE (self));
}


ValaExpression* vala_binary_expression_get_right (ValaBinaryExpression* self) {
	g_return_val_if_fail (VALA_IS_BINARY_EXPRESSION (self), NULL);
	return self->priv->_right;
}


void vala_binary_expression_set_right (ValaBinaryExpression* self, ValaExpression* value) {
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_BINARY_EXPRESSION (self));
	_tmp1 = NULL;
	self->priv->_right = (_tmp1 = g_object_ref (value), (self->priv->_right == NULL ? NULL : (self->priv->_right = (g_object_unref (self->priv->_right), NULL))), _tmp1);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_right), VALA_CODE_NODE (self));
}


static void vala_binary_expression_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaBinaryExpression * self;
	self = VALA_BINARY_EXPRESSION (object);
	switch (property_id) {
		case VALA_BINARY_EXPRESSION_OPERATOR:
		g_value_set_int (value, vala_binary_expression_get_operator (self));
		break;
		case VALA_BINARY_EXPRESSION_LEFT:
		g_value_set_object (value, vala_binary_expression_get_left (self));
		break;
		case VALA_BINARY_EXPRESSION_RIGHT:
		g_value_set_object (value, vala_binary_expression_get_right (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_binary_expression_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaBinaryExpression * self;
	self = VALA_BINARY_EXPRESSION (object);
	switch (property_id) {
		case VALA_BINARY_EXPRESSION_OPERATOR:
		vala_binary_expression_set_operator (self, g_value_get_int (value));
		break;
		case VALA_BINARY_EXPRESSION_LEFT:
		vala_binary_expression_set_left (self, g_value_get_object (value));
		break;
		case VALA_BINARY_EXPRESSION_RIGHT:
		vala_binary_expression_set_right (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_binary_expression_class_init (ValaBinaryExpressionClass * klass) {
	vala_binary_expression_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaBinaryExpressionPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_binary_expression_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_binary_expression_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_binary_expression_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_binary_expression_real_accept;
	VALA_CODE_NODE_CLASS (klass)->replace = vala_binary_expression_real_replace;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_BINARY_EXPRESSION_OPERATOR, g_param_spec_int ("operator", "operator", "operator", G_MININT, G_MAXINT, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_BINARY_EXPRESSION_LEFT, g_param_spec_object ("left", "left", "left", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_BINARY_EXPRESSION_RIGHT, g_param_spec_object ("right", "right", "right", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
}


static void vala_binary_expression_init (ValaBinaryExpression * self) {
	self->priv = VALA_BINARY_EXPRESSION_GET_PRIVATE (self);
}


static void vala_binary_expression_dispose (GObject * obj) {
	ValaBinaryExpression * self;
	self = VALA_BINARY_EXPRESSION (obj);
	(self->priv->_left == NULL ? NULL : (self->priv->_left = (g_object_unref (self->priv->_left), NULL)));
	(self->priv->_right == NULL ? NULL : (self->priv->_right = (g_object_unref (self->priv->_right), NULL)));
	G_OBJECT_CLASS (vala_binary_expression_parent_class)->dispose (obj);
}


GType vala_binary_expression_get_type (void) {
	static GType vala_binary_expression_type_id = 0;
	if (G_UNLIKELY (vala_binary_expression_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaBinaryExpressionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_binary_expression_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaBinaryExpression), 0, (GInstanceInitFunc) vala_binary_expression_init };
		vala_binary_expression_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaBinaryExpression", &g_define_type_info, 0);
	}
	return vala_binary_expression_type_id;
}




