/* valaconstant.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaconstant.h>
#include <vala/valatypereference.h>
#include <vala/valaexpression.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valacodevisitor.h>

struct _ValaConstantPrivate {
	ValaTypeReference* _type_reference;
	ValaExpression* _initializer;
	char* cname;
	gboolean lock_used;
};
#define VALA_CONSTANT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CONSTANT, ValaConstantPrivate))
enum  {
	VALA_CONSTANT_DUMMY_PROPERTY,
	VALA_CONSTANT_TYPE_REFERENCE,
	VALA_CONSTANT_INITIALIZER
};
static void vala_constant_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_constant_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_constant_real_get_lock_used (ValaLockable* base);
static void vala_constant_real_set_lock_used (ValaLockable* base, gboolean used);
static gpointer vala_constant_parent_class = NULL;
static ValaLockableIface* vala_constant_vala_lockable_parent_iface = NULL;
static void vala_constant_dispose (GObject * obj);


/**
 * Creates a new constant.
 *
 * @param name             constant name
 * @param type_reference   constant type
 * @param initializer      constant value
 * @param source_reference reference to source code
 * @return                 newly created constant
 */
ValaConstant* vala_constant_new (const char* name, ValaTypeReference* type_reference, ValaExpression* initializer, ValaSourceReference* source_reference) {
	ValaConstant * self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type_reference), NULL);
	g_return_val_if_fail (initializer == NULL || VALA_IS_EXPRESSION (initializer), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_CONSTANT, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_constant_set_type_reference (self, type_reference);
	vala_constant_set_initializer (self, initializer);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


static void vala_constant_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaConstant * self;
	self = VALA_CONSTANT (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_member (visitor, VALA_MEMBER (self));
	vala_code_visitor_visit_constant (visitor, self);
}


static void vala_constant_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaConstant * self;
	self = VALA_CONSTANT (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (vala_constant_get_type_reference (self)), visitor);
	if (vala_constant_get_initializer (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_constant_get_initializer (self)), visitor);
	}
}


/**
 * Returns the name of this constant as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_constant_get_cname (ValaConstant* self) {
	const char* _tmp5;
	g_return_val_if_fail (VALA_IS_CONSTANT (self), NULL);
	if (self->priv->cname == NULL) {
		if (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)) == NULL) {
			char* _tmp1;
			const char* _tmp0;
			/* global constant*/
			_tmp1 = NULL;
			_tmp0 = NULL;
			self->priv->cname = (_tmp1 = (_tmp0 = vala_symbol_get_name (VALA_SYMBOL (self)), (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp1);
		} else {
			char* _tmp4;
			char* _tmp3;
			char* _tmp2;
			_tmp4 = NULL;
			_tmp3 = NULL;
			_tmp2 = NULL;
			self->priv->cname = (_tmp4 = g_strdup_printf ("%s%s", (_tmp3 = g_utf8_strup ((_tmp2 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))), -1)), vala_symbol_get_name (VALA_SYMBOL (self))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp4);
			(_tmp3 = (g_free (_tmp3), NULL));
			(_tmp2 = (g_free (_tmp2), NULL));
		}
	}
	_tmp5 = NULL;
	return (_tmp5 = self->priv->cname, (_tmp5 == NULL ? NULL : g_strdup (_tmp5)));
}


static gboolean vala_constant_real_get_lock_used (ValaLockable* base) {
	ValaConstant * self;
	self = VALA_CONSTANT (base);
	return self->priv->lock_used;
}


static void vala_constant_real_set_lock_used (ValaLockable* base, gboolean used) {
	ValaConstant * self;
	self = VALA_CONSTANT (base);
	self->priv->lock_used = used;
}


ValaTypeReference* vala_constant_get_type_reference (ValaConstant* self) {
	g_return_val_if_fail (VALA_IS_CONSTANT (self), NULL);
	return self->priv->_type_reference;
}


void vala_constant_set_type_reference (ValaConstant* self, ValaTypeReference* value) {
	ValaTypeReference* _tmp1;
	g_return_if_fail (VALA_IS_CONSTANT (self));
	_tmp1 = NULL;
	self->priv->_type_reference = (_tmp1 = g_object_ref (value), (self->priv->_type_reference == NULL ? NULL : (self->priv->_type_reference = (g_object_unref (self->priv->_type_reference), NULL))), _tmp1);
}


ValaExpression* vala_constant_get_initializer (ValaConstant* self) {
	g_return_val_if_fail (VALA_IS_CONSTANT (self), NULL);
	return self->priv->_initializer;
}


void vala_constant_set_initializer (ValaConstant* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_CONSTANT (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_initializer = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_initializer == NULL ? NULL : (self->priv->_initializer = (g_object_unref (self->priv->_initializer), NULL))), _tmp2);
}


static void vala_constant_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaConstant * self;
	self = VALA_CONSTANT (object);
	switch (property_id) {
		case VALA_CONSTANT_TYPE_REFERENCE:
		g_value_set_object (value, vala_constant_get_type_reference (self));
		break;
		case VALA_CONSTANT_INITIALIZER:
		g_value_set_object (value, vala_constant_get_initializer (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_constant_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaConstant * self;
	self = VALA_CONSTANT (object);
	switch (property_id) {
		case VALA_CONSTANT_TYPE_REFERENCE:
		vala_constant_set_type_reference (self, g_value_get_object (value));
		break;
		case VALA_CONSTANT_INITIALIZER:
		vala_constant_set_initializer (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_constant_class_init (ValaConstantClass * klass) {
	vala_constant_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaConstantPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_constant_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_constant_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_constant_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_constant_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_constant_real_accept_children;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CONSTANT_TYPE_REFERENCE, g_param_spec_object ("type-reference", "type-reference", "type-reference", VALA_TYPE_TYPE_REFERENCE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CONSTANT_INITIALIZER, g_param_spec_object ("initializer", "initializer", "initializer", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_constant_vala_lockable_interface_init (ValaLockableIface * iface) {
	vala_constant_vala_lockable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_lock_used = vala_constant_real_get_lock_used;
	iface->set_lock_used = vala_constant_real_set_lock_used;
}


static void vala_constant_init (ValaConstant * self) {
	self->priv = VALA_CONSTANT_GET_PRIVATE (self);
	self->priv->lock_used = FALSE;
}


static void vala_constant_dispose (GObject * obj) {
	ValaConstant * self;
	self = VALA_CONSTANT (obj);
	(self->priv->_type_reference == NULL ? NULL : (self->priv->_type_reference = (g_object_unref (self->priv->_type_reference), NULL)));
	(self->priv->_initializer == NULL ? NULL : (self->priv->_initializer = (g_object_unref (self->priv->_initializer), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	G_OBJECT_CLASS (vala_constant_parent_class)->dispose (obj);
}


GType vala_constant_get_type (void) {
	static GType vala_constant_type_id = 0;
	if (G_UNLIKELY (vala_constant_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaConstantClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_constant_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaConstant), 0, (GInstanceInitFunc) vala_constant_init };
		static const GInterfaceInfo vala_lockable_info = { (GInterfaceInitFunc) vala_constant_vala_lockable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_constant_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaConstant", &g_define_type_info, 0);
		g_type_add_interface_static (vala_constant_type_id, VALA_TYPE_LOCKABLE, &vala_lockable_info);
	}
	return vala_constant_type_id;
}




