/* valastruct.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valastruct.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <vala/valatypeparameter.h>
#include <vala/valaconstant.h>
#include <vala/valafield.h>
#include <vala/valamethod.h>
#include <vala/valatypereference.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valaformalparameter.h>
#include <vala/valacreationmethod.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>
#include <vala/valareport.h>

struct _ValaStructPrivate {
	GeeList* type_parameters;
	GeeList* constants;
	GeeList* fields;
	GeeList* methods;
	GeeList* base_types;
	char* cname;
	char* const_cname;
	char* dup_function;
	char* free_function;
	char* type_id;
	char* lower_case_cprefix;
	char* lower_case_csuffix;
	gboolean integer_type;
	gboolean floating_type;
	gint rank;
	char* marshaller_type_name;
	char* get_value_function;
	char* set_value_function;
	char* default_value;
	gboolean simple_type;
	ValaMethod* _default_construction_method;
};
#define VALA_STRUCT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_STRUCT, ValaStructPrivate))
enum  {
	VALA_STRUCT_DUMMY_PROPERTY,
	VALA_STRUCT_DEFAULT_CONSTRUCTION_METHOD
};
static void vala_struct_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_struct_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_struct_real_get_cname (ValaDataType* base, gboolean const_type);
static void vala_struct_set_cname (ValaStruct* self, const char* cname);
static void vala_struct_set_const_cname (ValaStruct* self, const char* cname);
static char* vala_struct_real_get_lower_case_cprefix (ValaSymbol* base);
static char* vala_struct_get_lower_case_csuffix (ValaStruct* self);
static char* vala_struct_real_get_lower_case_cname (ValaSymbol* base, const char* infix);
static char* vala_struct_real_get_upper_case_cname (ValaDataType* base, const char* infix);
static void vala_struct_process_ccode_attribute (ValaStruct* self, ValaAttribute* a);
static void vala_struct_process_integer_type_attribute (ValaStruct* self, ValaAttribute* a);
static void vala_struct_process_floating_type_attribute (ValaStruct* self, ValaAttribute* a);
static char* vala_struct_real_get_type_id (ValaDataType* base);
static char* vala_struct_real_get_marshaller_type_name (ValaDataType* base);
static void vala_struct_set_marshaller_type_name (ValaStruct* self, const char* name);
static char* vala_struct_real_get_get_value_function (ValaDataType* base);
static char* vala_struct_real_get_set_value_function (ValaDataType* base);
static void vala_struct_set_get_value_function (ValaStruct* self, const char* function);
static void vala_struct_set_set_value_function (ValaStruct* self, const char* function);
static char* vala_struct_real_get_default_value (ValaDataType* base);
static void vala_struct_set_default_value (ValaStruct* self, const char* value);
static gint vala_struct_real_get_type_parameter_index (ValaDataType* base, const char* name);
static gpointer vala_struct_parent_class = NULL;
static void vala_struct_dispose (GObject * obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);


/**
 * Creates a new struct.
 *
 * @param name             type name
 * @param source_reference reference to source code
 * @return                 newly created struct
 */
ValaStruct* vala_struct_new (const char* name, ValaSourceReference* source_reference) {
	ValaStruct * self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_STRUCT, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Appends the specified parameter to the list of type parameters.
 *
 * @param p a type parameter
 */
void vala_struct_add_type_parameter (ValaStruct* self, ValaTypeParameter* p) {
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_TYPE_PARAMETER (p));
	gee_collection_add (GEE_COLLECTION (self->priv->type_parameters), p);
	p->type = VALA_DATA_TYPE (self);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (p)), VALA_SYMBOL (p));
}


/**
 * Adds the specified constant as a member to this struct.
 *
 * @param c a constant
 */
void vala_struct_add_constant (ValaStruct* self, ValaConstant* c) {
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_CONSTANT (c));
	gee_collection_add (GEE_COLLECTION (self->priv->constants), c);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (c)), VALA_SYMBOL (c));
}


/**
 * Adds the specified field as a member to this struct.
 *
 * @param f a field
 */
void vala_struct_add_field (ValaStruct* self, ValaField* f) {
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_FIELD (f));
	gee_collection_add (GEE_COLLECTION (self->priv->fields), f);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (f)), VALA_SYMBOL (f));
}


/**
 * Returns a copy of the list of fields.
 *
 * @return list of fields
 */
GeeCollection* vala_struct_get_fields (ValaStruct* self) {
	g_return_val_if_fail (VALA_IS_STRUCT (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_FIELD, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->fields)));
}


/**
 * Adds the specified method as a member to this struct.
 *
 * @param m a method
 */
void vala_struct_add_method (ValaStruct* self, ValaMethod* m) {
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	g_return_if_fail (m != NULL);
	if (vala_method_get_instance (m)) {
		ValaFormalParameter* _tmp1;
		ValaTypeReference* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		vala_method_set_this_parameter (m, (_tmp1 = vala_formal_parameter_new ("this", (_tmp0 = vala_type_reference_new ()), NULL)));
		(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
		(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
		vala_type_reference_set_data_type (vala_formal_parameter_get_type_reference (vala_method_get_this_parameter (m)), VALA_DATA_TYPE (self));
		vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (m)), vala_symbol_get_name (VALA_SYMBOL (vala_method_get_this_parameter (m))), VALA_SYMBOL (vala_method_get_this_parameter (m)));
	}
	if (VALA_IS_CREATION_METHOD (m) && vala_symbol_get_name (VALA_SYMBOL (m)) == NULL) {
		vala_struct_set_default_construction_method (self, m);
	}
	gee_collection_add (GEE_COLLECTION (self->priv->methods), m);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (m)), VALA_SYMBOL (m));
}


/**
 * Returns a copy of the list of methods.
 *
 * @return list of methods
 */
GeeCollection* vala_struct_get_methods (ValaStruct* self) {
	g_return_val_if_fail (VALA_IS_STRUCT (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_METHOD, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->methods)));
}


static void vala_struct_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaStruct * self;
	self = VALA_STRUCT (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_struct (visitor, self);
}


static void vala_struct_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaStruct * self;
	self = VALA_STRUCT (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GeeList* type_collection;
		GeeIterator* type_it;
		type_collection = self->priv->base_types;
		type_it = gee_iterable_iterator (GEE_ITERABLE (type_collection));
		while (gee_iterator_next (type_it)) {
			ValaTypeReference* type;
			type = gee_iterator_get (type_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (type), visitor);
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
	}
	{
		GeeList* p_collection;
		GeeIterator* p_it;
		p_collection = self->priv->type_parameters;
		p_it = gee_iterable_iterator (GEE_ITERABLE (p_collection));
		while (gee_iterator_next (p_it)) {
			ValaTypeParameter* p;
			p = gee_iterator_get (p_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (p), visitor);
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
		(p_it == NULL ? NULL : (p_it = (g_object_unref (p_it), NULL)));
	}
	{
		GeeList* f_collection;
		GeeIterator* f_it;
		f_collection = self->priv->fields;
		f_it = gee_iterable_iterator (GEE_ITERABLE (f_collection));
		while (gee_iterator_next (f_it)) {
			ValaField* f;
			f = gee_iterator_get (f_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (f), visitor);
				(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
			}
		}
		(f_it == NULL ? NULL : (f_it = (g_object_unref (f_it), NULL)));
	}
	{
		GeeList* c_collection;
		GeeIterator* c_it;
		c_collection = self->priv->constants;
		c_it = gee_iterable_iterator (GEE_ITERABLE (c_collection));
		while (gee_iterator_next (c_it)) {
			ValaConstant* c;
			c = gee_iterator_get (c_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (c), visitor);
				(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
			}
		}
		(c_it == NULL ? NULL : (c_it = (g_object_unref (c_it), NULL)));
	}
	{
		GeeList* m_collection;
		GeeIterator* m_it;
		m_collection = self->priv->methods;
		m_it = gee_iterable_iterator (GEE_ITERABLE (m_collection));
		while (gee_iterator_next (m_it)) {
			ValaMethod* m;
			m = gee_iterator_get (m_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (m), visitor);
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
			}
		}
		(m_it == NULL ? NULL : (m_it = (g_object_unref (m_it), NULL)));
	}
}


static char* vala_struct_real_get_cname (ValaDataType* base, gboolean const_type) {
	ValaStruct * self;
	const char* _tmp4;
	self = VALA_STRUCT (base);
	if (const_type && self->priv->const_cname != NULL) {
		const char* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = self->priv->const_cname, (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
	}
	if (self->priv->cname == NULL) {
		char* _tmp3;
		char* _tmp2;
		_tmp3 = NULL;
		_tmp2 = NULL;
		self->priv->cname = (_tmp3 = g_strdup_printf ("%s%s", (_tmp2 = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))), vala_symbol_get_name (VALA_SYMBOL (self))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp3);
		(_tmp2 = (g_free (_tmp2), NULL));
	}
	_tmp4 = NULL;
	return (_tmp4 = self->priv->cname, (_tmp4 == NULL ? NULL : g_strdup (_tmp4)));
}


static void vala_struct_set_cname (ValaStruct* self, const char* cname) {
	char* _tmp0;
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (cname != NULL);
	_tmp0 = NULL;
	self->priv->cname = (_tmp0 = g_strdup (cname), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp0);
}


static void vala_struct_set_const_cname (ValaStruct* self, const char* cname) {
	char* _tmp0;
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (cname != NULL);
	_tmp0 = NULL;
	self->priv->const_cname = (_tmp0 = g_strdup (cname), (self->priv->const_cname = (g_free (self->priv->const_cname), NULL)), _tmp0);
}


static char* vala_struct_real_get_lower_case_cprefix (ValaSymbol* base) {
	ValaStruct * self;
	const char* _tmp2;
	self = VALA_STRUCT (base);
	if (self->priv->lower_case_cprefix == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->lower_case_cprefix = (_tmp1 = g_strdup_printf ("%s_", (_tmp0 = vala_symbol_get_lower_case_cname (VALA_SYMBOL (self), NULL))), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), _tmp1);
		(_tmp0 = (g_free (_tmp0), NULL));
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->lower_case_cprefix, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


static char* vala_struct_get_lower_case_csuffix (ValaStruct* self) {
	const char* _tmp1;
	g_return_val_if_fail (VALA_IS_STRUCT (self), NULL);
	if (self->priv->lower_case_csuffix == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->lower_case_csuffix = (_tmp0 = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name (VALA_SYMBOL (self))), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->lower_case_csuffix, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
}


static char* vala_struct_real_get_lower_case_cname (ValaSymbol* base, const char* infix) {
	ValaStruct * self;
	char* _tmp1;
	char* _tmp0;
	char* _tmp2;
	self = VALA_STRUCT (base);
	if (infix == NULL) {
		infix = "";
	}
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = g_strdup_printf ("%s%s%s", (_tmp0 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))), infix, (_tmp1 = vala_struct_get_lower_case_csuffix (self))), (_tmp1 = (g_free (_tmp1), NULL)), (_tmp0 = (g_free (_tmp0), NULL)), _tmp2);
}


static char* vala_struct_real_get_upper_case_cname (ValaDataType* base, const char* infix) {
	ValaStruct * self;
	char* _tmp0;
	char* _tmp1;
	self = VALA_STRUCT (base);
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_utf8_strup ((_tmp0 = vala_symbol_get_lower_case_cname (VALA_SYMBOL (self), infix)), -1), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
}


/**
 * Returns whether this is an integer type.
 *
 * @return true if this is an integer type, false otherwise
 */
gboolean vala_struct_is_integer_type (ValaStruct* self) {
	g_return_val_if_fail (VALA_IS_STRUCT (self), FALSE);
	return self->priv->integer_type;
}


/**
 * Returns whether this is a floating point type.
 *
 * @return true if this is a floating point type, false otherwise
 */
gboolean vala_struct_is_floating_type (ValaStruct* self) {
	g_return_val_if_fail (VALA_IS_STRUCT (self), FALSE);
	return self->priv->floating_type;
}


/**
 * Returns the rank of this integer or floating point type.
 *
 * @return the rank if this is an integer or floating point type
 */
gint vala_struct_get_rank (ValaStruct* self) {
	g_return_val_if_fail (VALA_IS_STRUCT (self), 0);
	return self->priv->rank;
}


static void vala_struct_process_ccode_attribute (ValaStruct* self, ValaAttribute* a) {
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cname")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_struct_set_cname (self, (_tmp0 = vala_attribute_get_string (a, "cname")));
		(_tmp0 = (g_free (_tmp0), NULL));
	}
	if (vala_attribute_has_argument (a, "const_cname")) {
		char* _tmp1;
		_tmp1 = NULL;
		vala_struct_set_const_cname (self, (_tmp1 = vala_attribute_get_string (a, "const_cname")));
		(_tmp1 = (g_free (_tmp1), NULL));
	}
	if (vala_attribute_has_argument (a, "cprefix")) {
		char* _tmp2;
		_tmp2 = NULL;
		self->priv->lower_case_cprefix = (_tmp2 = vala_attribute_get_string (a, "cprefix"), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), _tmp2);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			char** filename_it;
			int filename_collection_length1;
			filename_collection = g_strsplit (val, ",", 0);
			filename_collection_length1 = -1;
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* _tmp3;
				char* filename;
				_tmp3 = NULL;
				filename = (_tmp3 = *filename_it, (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
				{
					vala_data_type_add_cheader_filename (VALA_DATA_TYPE (self), filename);
					(filename = (g_free (filename), NULL));
				}
			}
			(filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, ((GDestroyNotify) g_free)), NULL));
		}
		(val = (g_free (val), NULL));
	}
	if (vala_attribute_has_argument (a, "type_id")) {
		char* _tmp4;
		_tmp4 = NULL;
		vala_struct_set_type_id (self, (_tmp4 = vala_attribute_get_string (a, "type_id")));
		(_tmp4 = (g_free (_tmp4), NULL));
	}
	if (vala_attribute_has_argument (a, "marshaller_type_name")) {
		char* _tmp5;
		_tmp5 = NULL;
		vala_struct_set_marshaller_type_name (self, (_tmp5 = vala_attribute_get_string (a, "marshaller_type_name")));
		(_tmp5 = (g_free (_tmp5), NULL));
	}
	if (vala_attribute_has_argument (a, "get_value_function")) {
		char* _tmp6;
		_tmp6 = NULL;
		vala_struct_set_get_value_function (self, (_tmp6 = vala_attribute_get_string (a, "get_value_function")));
		(_tmp6 = (g_free (_tmp6), NULL));
	}
	if (vala_attribute_has_argument (a, "set_value_function")) {
		char* _tmp7;
		_tmp7 = NULL;
		vala_struct_set_set_value_function (self, (_tmp7 = vala_attribute_get_string (a, "set_value_function")));
		(_tmp7 = (g_free (_tmp7), NULL));
	}
	if (vala_attribute_has_argument (a, "default_value")) {
		char* _tmp8;
		_tmp8 = NULL;
		vala_struct_set_default_value (self, (_tmp8 = vala_attribute_get_string (a, "default_value")));
		(_tmp8 = (g_free (_tmp8), NULL));
	}
}


static void vala_struct_process_integer_type_attribute (ValaStruct* self, ValaAttribute* a) {
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	self->priv->integer_type = TRUE;
	if (vala_attribute_has_argument (a, "rank")) {
		self->priv->rank = vala_attribute_get_integer (a, "rank");
	}
}


static void vala_struct_process_floating_type_attribute (ValaStruct* self, ValaAttribute* a) {
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	self->priv->floating_type = TRUE;
	if (vala_attribute_has_argument (a, "rank")) {
		self->priv->rank = vala_attribute_get_integer (a, "rank");
	}
}


/**
 * Process all associated attributes.
 */
void vala_struct_process_attributes (ValaStruct* self) {
	g_return_if_fail (VALA_IS_STRUCT (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = VALA_CODE_NODE (self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = a_it->data, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "CCode") == 0) {
					vala_struct_process_ccode_attribute (self, a);
				} else {
					if (g_utf8_collate (vala_attribute_get_name (a), "SimpleType") == 0) {
						self->priv->simple_type = TRUE;
					} else {
						if (g_utf8_collate (vala_attribute_get_name (a), "IntegerType") == 0) {
							vala_struct_process_integer_type_attribute (self, a);
						} else {
							if (g_utf8_collate (vala_attribute_get_name (a), "FloatingType") == 0) {
								vala_struct_process_floating_type_attribute (self, a);
							}
						}
					}
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


static char* vala_struct_real_get_type_id (ValaDataType* base) {
	ValaStruct * self;
	const char* _tmp2;
	self = VALA_STRUCT (base);
	if (self->priv->type_id == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (self)), (_tmp1 = g_strdup_printf ("The type `%s` doesn't declare a type id", (_tmp0 = vala_symbol_get_full_name (VALA_SYMBOL (self))))));
		(_tmp1 = (g_free (_tmp1), NULL));
		(_tmp0 = (g_free (_tmp0), NULL));
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->type_id, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


void vala_struct_set_type_id (ValaStruct* self, const char* name) {
	char* _tmp0;
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (name != NULL);
	_tmp0 = NULL;
	self->priv->type_id = (_tmp0 = g_strdup (name), (self->priv->type_id = (g_free (self->priv->type_id), NULL)), _tmp0);
}


static char* vala_struct_real_get_marshaller_type_name (ValaDataType* base) {
	ValaStruct * self;
	const char* _tmp2;
	self = VALA_STRUCT (base);
	if (self->priv->marshaller_type_name == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (self)), (_tmp1 = g_strdup_printf ("The type `%s` doesn't declare a marshaller type name", (_tmp0 = vala_symbol_get_full_name (VALA_SYMBOL (self))))));
		(_tmp1 = (g_free (_tmp1), NULL));
		(_tmp0 = (g_free (_tmp0), NULL));
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->marshaller_type_name, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


static void vala_struct_set_marshaller_type_name (ValaStruct* self, const char* name) {
	char* _tmp0;
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (name != NULL);
	_tmp0 = NULL;
	self->priv->marshaller_type_name = (_tmp0 = g_strdup (name), (self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL)), _tmp0);
}


static char* vala_struct_real_get_get_value_function (ValaDataType* base) {
	ValaStruct * self;
	self = VALA_STRUCT (base);
	if (self->priv->get_value_function == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (self)), (_tmp1 = g_strdup_printf ("The value type `%s` doesn't declare a GValue get function", (_tmp0 = vala_symbol_get_full_name (VALA_SYMBOL (self))))));
		(_tmp1 = (g_free (_tmp1), NULL));
		(_tmp0 = (g_free (_tmp0), NULL));
		return NULL;
	} else {
		const char* _tmp3;
		_tmp3 = NULL;
		return (_tmp3 = self->priv->get_value_function, (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
	}
}


static char* vala_struct_real_get_set_value_function (ValaDataType* base) {
	ValaStruct * self;
	self = VALA_STRUCT (base);
	if (self->priv->set_value_function == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (self)), (_tmp1 = g_strdup_printf ("The value type `%s` doesn't declare a GValue set function", (_tmp0 = vala_symbol_get_full_name (VALA_SYMBOL (self))))));
		(_tmp1 = (g_free (_tmp1), NULL));
		(_tmp0 = (g_free (_tmp0), NULL));
		return NULL;
	} else {
		const char* _tmp3;
		_tmp3 = NULL;
		return (_tmp3 = self->priv->set_value_function, (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
	}
}


static void vala_struct_set_get_value_function (ValaStruct* self, const char* function) {
	char* _tmp0;
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (function != NULL);
	_tmp0 = NULL;
	self->priv->get_value_function = (_tmp0 = g_strdup (function), (self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL)), _tmp0);
}


static void vala_struct_set_set_value_function (ValaStruct* self, const char* function) {
	char* _tmp0;
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (function != NULL);
	_tmp0 = NULL;
	self->priv->set_value_function = (_tmp0 = g_strdup (function), (self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL)), _tmp0);
}


static char* vala_struct_real_get_default_value (ValaDataType* base) {
	ValaStruct * self;
	const char* _tmp0;
	self = VALA_STRUCT (base);
	_tmp0 = NULL;
	return (_tmp0 = self->priv->default_value, (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
}


static void vala_struct_set_default_value (ValaStruct* self, const char* value) {
	char* _tmp0;
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (value != NULL);
	_tmp0 = NULL;
	self->priv->default_value = (_tmp0 = g_strdup (value), (self->priv->default_value = (g_free (self->priv->default_value), NULL)), _tmp0);
}


/**
 * Adds the specified struct to the list of base types of this struct.
 *
 * @param type a class or interface reference
 */
void vala_struct_add_base_type (ValaStruct* self, ValaTypeReference* type) {
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (type));
	gee_collection_add (GEE_COLLECTION (self->priv->base_types), type);
}


/**
 * Returns a copy of the base type list.
 *
 * @return list of base types
 */
GeeCollection* vala_struct_get_base_types (ValaStruct* self) {
	g_return_val_if_fail (VALA_IS_STRUCT (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_TYPE_REFERENCE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->base_types)));
}


static gint vala_struct_real_get_type_parameter_index (ValaDataType* base, const char* name) {
	ValaStruct * self;
	gint i;
	self = VALA_STRUCT (base);
	g_return_val_if_fail (name != NULL, 0);
	i = 0;
	{
		GeeList* p_collection;
		GeeIterator* p_it;
		p_collection = self->priv->type_parameters;
		p_it = gee_iterable_iterator (GEE_ITERABLE (p_collection));
		while (gee_iterator_next (p_it)) {
			ValaTypeParameter* p;
			p = gee_iterator_get (p_it);
			{
				if (g_utf8_collate (vala_symbol_get_name (VALA_SYMBOL (p)), name) == 0) {
					gint _tmp0;
					return (_tmp0 = (i), (p == NULL ? NULL : (p = (g_object_unref (p), NULL))), (p_it == NULL ? NULL : (p_it = (g_object_unref (p_it), NULL))), _tmp0);
				}
				i++;
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
		(p_it == NULL ? NULL : (p_it = (g_object_unref (p_it), NULL)));
	}
	return -1;
}


/**
 * Returns whether this struct is a simple type, i.e. whether
 * instances are passed by value.
 */
gboolean vala_struct_is_simple_type (ValaStruct* self) {
	g_return_val_if_fail (VALA_IS_STRUCT (self), FALSE);
	{
		GeeList* type_collection;
		GeeIterator* type_it;
		type_collection = self->priv->base_types;
		type_it = gee_iterable_iterator (GEE_ITERABLE (type_collection));
		while (gee_iterator_next (type_it)) {
			ValaTypeReference* type;
			type = gee_iterator_get (type_it);
			{
				ValaStruct* _tmp1;
				ValaDataType* _tmp0;
				ValaStruct* st;
				_tmp1 = NULL;
				_tmp0 = NULL;
				st = (_tmp1 = (_tmp0 = vala_type_reference_get_data_type (type), (VALA_IS_STRUCT (_tmp0) ? ((ValaStruct*) _tmp0) : NULL)), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
				if (st != NULL && vala_struct_is_simple_type (st)) {
					gboolean _tmp2;
					return (_tmp2 = TRUE, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), (type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL))), _tmp2);
				}
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
				(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
			}
		}
		(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
	}
	return self->priv->simple_type;
}


ValaMethod* vala_struct_get_default_construction_method (ValaStruct* self) {
	g_return_val_if_fail (VALA_IS_STRUCT (self), NULL);
	return self->priv->_default_construction_method;
}


void vala_struct_set_default_construction_method (ValaStruct* self, ValaMethod* value) {
	ValaMethod* _tmp2;
	ValaMethod* _tmp1;
	g_return_if_fail (VALA_IS_STRUCT (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_default_construction_method = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_default_construction_method == NULL ? NULL : (self->priv->_default_construction_method = (g_object_unref (self->priv->_default_construction_method), NULL))), _tmp2);
}


static void vala_struct_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaStruct * self;
	self = VALA_STRUCT (object);
	switch (property_id) {
		case VALA_STRUCT_DEFAULT_CONSTRUCTION_METHOD:
		g_value_set_object (value, vala_struct_get_default_construction_method (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_struct_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaStruct * self;
	self = VALA_STRUCT (object);
	switch (property_id) {
		case VALA_STRUCT_DEFAULT_CONSTRUCTION_METHOD:
		vala_struct_set_default_construction_method (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_struct_class_init (ValaStructClass * klass) {
	vala_struct_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaStructPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_struct_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_struct_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_struct_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_struct_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_struct_real_accept_children;
	VALA_DATA_TYPE_CLASS (klass)->get_cname = vala_struct_real_get_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_struct_real_get_lower_case_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cname = vala_struct_real_get_lower_case_cname;
	VALA_DATA_TYPE_CLASS (klass)->get_upper_case_cname = vala_struct_real_get_upper_case_cname;
	VALA_DATA_TYPE_CLASS (klass)->get_type_id = vala_struct_real_get_type_id;
	VALA_DATA_TYPE_CLASS (klass)->get_marshaller_type_name = vala_struct_real_get_marshaller_type_name;
	VALA_DATA_TYPE_CLASS (klass)->get_get_value_function = vala_struct_real_get_get_value_function;
	VALA_DATA_TYPE_CLASS (klass)->get_set_value_function = vala_struct_real_get_set_value_function;
	VALA_DATA_TYPE_CLASS (klass)->get_default_value = vala_struct_real_get_default_value;
	VALA_DATA_TYPE_CLASS (klass)->get_type_parameter_index = vala_struct_real_get_type_parameter_index;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_STRUCT_DEFAULT_CONSTRUCTION_METHOD, g_param_spec_object ("default-construction-method", "default-construction-method", "default-construction-method", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_struct_init (ValaStruct * self) {
	self->priv = VALA_STRUCT_GET_PRIVATE (self);
	self->priv->type_parameters = GEE_LIST (gee_array_list_new (VALA_TYPE_TYPE_PARAMETER, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->constants = GEE_LIST (gee_array_list_new (VALA_TYPE_CONSTANT, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->fields = GEE_LIST (gee_array_list_new (VALA_TYPE_FIELD, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->methods = GEE_LIST (gee_array_list_new (VALA_TYPE_METHOD, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->base_types = GEE_LIST (gee_array_list_new (VALA_TYPE_TYPE_REFERENCE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->default_value = NULL;
}


static void vala_struct_dispose (GObject * obj) {
	ValaStruct * self;
	self = VALA_STRUCT (obj);
	(self->priv->type_parameters == NULL ? NULL : (self->priv->type_parameters = (g_object_unref (self->priv->type_parameters), NULL)));
	(self->priv->constants == NULL ? NULL : (self->priv->constants = (g_object_unref (self->priv->constants), NULL)));
	(self->priv->fields == NULL ? NULL : (self->priv->fields = (g_object_unref (self->priv->fields), NULL)));
	(self->priv->methods == NULL ? NULL : (self->priv->methods = (g_object_unref (self->priv->methods), NULL)));
	(self->priv->base_types == NULL ? NULL : (self->priv->base_types = (g_object_unref (self->priv->base_types), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	(self->priv->const_cname = (g_free (self->priv->const_cname), NULL));
	(self->priv->dup_function = (g_free (self->priv->dup_function), NULL));
	(self->priv->free_function = (g_free (self->priv->free_function), NULL));
	(self->priv->type_id = (g_free (self->priv->type_id), NULL));
	(self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL));
	(self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL));
	(self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL));
	(self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL));
	(self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL));
	(self->priv->default_value = (g_free (self->priv->default_value), NULL));
	(self->priv->_default_construction_method == NULL ? NULL : (self->priv->_default_construction_method = (g_object_unref (self->priv->_default_construction_method), NULL)));
	G_OBJECT_CLASS (vala_struct_parent_class)->dispose (obj);
}


GType vala_struct_get_type (void) {
	static GType vala_struct_type_id = 0;
	if (G_UNLIKELY (vala_struct_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaStructClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_struct_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaStruct), 0, (GInstanceInitFunc) vala_struct_init };
		vala_struct_type_id = g_type_register_static (VALA_TYPE_DATA_TYPE, "ValaStruct", &g_define_type_info, 0);
	}
	return vala_struct_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}




