/* valaenum.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaenum.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <vala/valaenumvalue.h>
#include <vala/valamethod.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valacreationmethod.h>
#include <vala/valareport.h>
#include <vala/valaformalparameter.h>
#include <vala/valavaluetype.h>
#include <vala/valadatatype.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>

struct _ValaEnumPrivate {
	gboolean _is_flags;
	gboolean _error_domain;
	GeeList* values;
	GeeList* methods;
	char* cname;
	char* cprefix;
	char* lower_case_cprefix;
	char* lower_case_csuffix;
};
#define VALA_ENUM_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_ENUM, ValaEnumPrivate))
enum  {
	VALA_ENUM_DUMMY_PROPERTY,
	VALA_ENUM_IS_FLAGS,
	VALA_ENUM_ERROR_DOMAIN
};
static void vala_enum_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_enum_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_enum_real_get_cname (ValaTypesymbol* base, gboolean const_type);
static char* vala_enum_real_get_lower_case_cprefix (ValaSymbol* base);
static char* vala_enum_get_lower_case_csuffix (ValaEnum* self);
static char* vala_enum_real_get_lower_case_cname (ValaSymbol* base, const char* infix);
static char* vala_enum_real_get_upper_case_cname (ValaTypesymbol* base, const char* infix);
static gboolean vala_enum_real_is_reference_type (ValaTypesymbol* base);
static void vala_enum_set_cname (ValaEnum* self, const char* cname);
static void vala_enum_process_ccode_attribute (ValaEnum* self, ValaAttribute* a);
static char* vala_enum_real_get_type_id (ValaTypesymbol* base);
static char* vala_enum_real_get_marshaller_type_name (ValaTypesymbol* base);
static char* vala_enum_real_get_get_value_function (ValaTypesymbol* base);
static char* vala_enum_real_get_set_value_function (ValaTypesymbol* base);
static char* vala_enum_real_get_default_value (ValaTypesymbol* base);
static gpointer vala_enum_parent_class = NULL;
static void vala_enum_dispose (GObject * obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);


/**
 * Creates a new enum.
 *
 * @param name             type name
 * @param source_reference reference to source code
 * @return                 newly created enum
 */
ValaEnum* vala_enum_new (const char* name, ValaSourceReference* source_reference) {
	ValaEnum * self;
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_ENUM, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Appends the specified enum value to the list of values.
 *
 * @param value an enum value
 */
void vala_enum_add_value (ValaEnum* self, ValaEnumValue* value) {
	g_return_if_fail (VALA_IS_ENUM (self));
	g_return_if_fail (value == NULL || VALA_IS_ENUM_VALUE (value));
	gee_collection_add (GEE_COLLECTION (self->priv->values), value);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (value)), VALA_SYMBOL (value));
}


/**
 * Adds the specified method as a member to this enum.
 *
 * @param m a method
 */
void vala_enum_add_method (ValaEnum* self, ValaMethod* m) {
	g_return_if_fail (VALA_IS_ENUM (self));
	g_return_if_fail (m == NULL || VALA_IS_METHOD (m));
	if (VALA_IS_CREATION_METHOD (m)) {
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), "construction methods may only be declared within classes and structs");
		vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
		return;
	}
	if (vala_method_get_instance (m)) {
		ValaFormalParameter* _tmp1;
		ValaValueType* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		vala_method_set_this_parameter (m, (_tmp1 = vala_formal_parameter_new ("this", VALA_DATA_TYPE ((_tmp0 = vala_value_type_new (VALA_TYPESYMBOL (self)))), NULL)));
		(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
		(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
		vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (m)), vala_symbol_get_name (VALA_SYMBOL (vala_method_get_this_parameter (m))), VALA_SYMBOL (vala_method_get_this_parameter (m)));
	}
	gee_collection_add (GEE_COLLECTION (self->priv->methods), m);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (m)), VALA_SYMBOL (m));
}


/**
 * Returns a copy of the list of enum values.
 *
 * @return list of enum values
 */
GeeCollection* vala_enum_get_values (ValaEnum* self) {
	g_return_val_if_fail (VALA_IS_ENUM (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_ENUM_VALUE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->values)));
}


/**
 * Returns a copy of the list of methods.
 *
 * @return list of methods
 */
GeeCollection* vala_enum_get_methods (ValaEnum* self) {
	g_return_val_if_fail (VALA_IS_ENUM (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_METHOD, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->methods)));
}


static void vala_enum_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaEnum * self;
	self = VALA_ENUM (base);
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_enum (visitor, self);
}


static void vala_enum_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaEnum * self;
	self = VALA_ENUM (base);
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
	{
		GeeList* value_collection;
		GeeIterator* value_it;
		value_collection = self->priv->values;
		value_it = gee_iterable_iterator (GEE_ITERABLE (value_collection));
		while (gee_iterator_next (value_it)) {
			ValaEnumValue* value;
			value = gee_iterator_get (value_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (value), visitor);
				(value == NULL ? NULL : (value = (g_object_unref (value), NULL)));
			}
		}
		(value_it == NULL ? NULL : (value_it = (g_object_unref (value_it), NULL)));
	}
	{
		GeeList* m_collection;
		GeeIterator* m_it;
		m_collection = self->priv->methods;
		m_it = gee_iterable_iterator (GEE_ITERABLE (m_collection));
		while (gee_iterator_next (m_it)) {
			ValaMethod* m;
			m = gee_iterator_get (m_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (m), visitor);
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
			}
		}
		(m_it == NULL ? NULL : (m_it = (g_object_unref (m_it), NULL)));
	}
}


static char* vala_enum_real_get_cname (ValaTypesymbol* base, gboolean const_type) {
	ValaEnum * self;
	const char* _tmp2;
	self = VALA_ENUM (base);
	if (self->priv->cname == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->cname = (_tmp1 = g_strdup_printf ("%s%s", (_tmp0 = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))), vala_symbol_get_name (VALA_SYMBOL (self))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp1);
		(_tmp0 = (g_free (_tmp0), NULL));
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->cname, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


static char* vala_enum_real_get_lower_case_cprefix (ValaSymbol* base) {
	ValaEnum * self;
	const char* _tmp2;
	self = VALA_ENUM (base);
	if (self->priv->lower_case_cprefix == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->lower_case_cprefix = (_tmp1 = g_strdup_printf ("%s_", (_tmp0 = vala_symbol_get_lower_case_cname (VALA_SYMBOL (self), NULL))), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), _tmp1);
		(_tmp0 = (g_free (_tmp0), NULL));
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->lower_case_cprefix, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


static char* vala_enum_get_lower_case_csuffix (ValaEnum* self) {
	const char* _tmp1;
	g_return_val_if_fail (VALA_IS_ENUM (self), NULL);
	if (self->priv->lower_case_csuffix == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->lower_case_csuffix = (_tmp0 = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name (VALA_SYMBOL (self))), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->lower_case_csuffix, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
}


static char* vala_enum_real_get_lower_case_cname (ValaSymbol* base, const char* infix) {
	ValaEnum * self;
	char* _tmp1;
	char* _tmp0;
	char* _tmp2;
	self = VALA_ENUM (base);
	if (infix == NULL) {
		infix = "";
	}
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = g_strdup_printf ("%s%s%s", (_tmp0 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))), infix, (_tmp1 = vala_enum_get_lower_case_csuffix (self))), (_tmp1 = (g_free (_tmp1), NULL)), (_tmp0 = (g_free (_tmp0), NULL)), _tmp2);
}


static char* vala_enum_real_get_upper_case_cname (ValaTypesymbol* base, const char* infix) {
	ValaEnum * self;
	char* _tmp0;
	char* _tmp1;
	self = VALA_ENUM (base);
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_utf8_strup ((_tmp0 = vala_symbol_get_lower_case_cname (VALA_SYMBOL (self), NULL)), -1), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
}


static gboolean vala_enum_real_is_reference_type (ValaTypesymbol* base) {
	ValaEnum * self;
	self = VALA_ENUM (base);
	return FALSE;
}


static void vala_enum_set_cname (ValaEnum* self, const char* cname) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (VALA_IS_ENUM (self));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cname = (_tmp1 = (_tmp0 = cname, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp1);
}


/**
 * Returns the string to be prepended to the name of members of this
 * enum when used in C code.
 *
 * @return the prefix to be used in C code
 */
char* vala_enum_get_cprefix (ValaEnum* self) {
	const char* _tmp2;
	g_return_val_if_fail (VALA_IS_ENUM (self), NULL);
	if (self->priv->cprefix == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->cprefix = (_tmp1 = g_strdup_printf ("%s_", (_tmp0 = vala_typesymbol_get_upper_case_cname (VALA_TYPESYMBOL (self), NULL))), (self->priv->cprefix = (g_free (self->priv->cprefix), NULL)), _tmp1);
		(_tmp0 = (g_free (_tmp0), NULL));
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->cprefix, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


/**
 * Sets the string to be prepended to the name of members of this enum
 * when used in C code.
 *
 * @param cprefix the prefix to be used in C code
 */
void vala_enum_set_cprefix (ValaEnum* self, const char* cprefix) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (VALA_IS_ENUM (self));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cprefix = (_tmp1 = (_tmp0 = cprefix, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->cprefix = (g_free (self->priv->cprefix), NULL)), _tmp1);
}


static void vala_enum_process_ccode_attribute (ValaEnum* self, ValaAttribute* a) {
	g_return_if_fail (VALA_IS_ENUM (self));
	g_return_if_fail (a == NULL || VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cname")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_enum_set_cname (self, (_tmp0 = vala_attribute_get_string (a, "cname")));
		(_tmp0 = (g_free (_tmp0), NULL));
	}
	if (vala_attribute_has_argument (a, "cprefix")) {
		char* _tmp1;
		_tmp1 = NULL;
		vala_enum_set_cprefix (self, (_tmp1 = vala_attribute_get_string (a, "cprefix")));
		(_tmp1 = (g_free (_tmp1), NULL));
	}
	if (vala_attribute_has_argument (a, "lower_case_csuffix")) {
		char* _tmp2;
		_tmp2 = NULL;
		self->priv->lower_case_csuffix = (_tmp2 = vala_attribute_get_string (a, "lower_case_csuffix"), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), _tmp2);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			char** filename_it;
			int filename_collection_length1;
			filename_collection = g_strsplit (val, ",", 0);
			filename_collection_length1 = -1;
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* _tmp3;
				char* filename;
				_tmp3 = NULL;
				filename = (_tmp3 = *filename_it, (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
				{
					vala_typesymbol_add_cheader_filename (VALA_TYPESYMBOL (self), filename);
					(filename = (g_free (filename), NULL));
				}
			}
			(filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, ((GDestroyNotify) g_free)), NULL));
		}
		(val = (g_free (val), NULL));
	}
}


/**
 * Process all associated attributes.
 */
void vala_enum_process_attributes (ValaEnum* self) {
	g_return_if_fail (VALA_IS_ENUM (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = VALA_CODE_NODE (self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = a_it->data, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "CCode") == 0) {
					vala_enum_process_ccode_attribute (self, a);
				} else {
					if (g_utf8_collate (vala_attribute_get_name (a), "Flags") == 0) {
						vala_enum_set_is_flags (self, TRUE);
					} else {
						if (g_utf8_collate (vala_attribute_get_name (a), "ErrorDomain") == 0) {
							vala_enum_set_error_domain (self, TRUE);
						}
					}
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


static char* vala_enum_real_get_type_id (ValaTypesymbol* base) {
	ValaEnum * self;
	self = VALA_ENUM (base);
	return g_strdup ("G_TYPE_INT");
}


/* FIXME: use GType-registered enums*/
static char* vala_enum_real_get_marshaller_type_name (ValaTypesymbol* base) {
	ValaEnum * self;
	self = VALA_ENUM (base);
	return g_strdup ("INT");
}


/* FIXME: use GType-registered enums*/
static char* vala_enum_real_get_get_value_function (ValaTypesymbol* base) {
	ValaEnum * self;
	self = VALA_ENUM (base);
	return g_strdup ("g_value_get_int");
}


/* FIXME: use GType-registered enums*/
static char* vala_enum_real_get_set_value_function (ValaTypesymbol* base) {
	ValaEnum * self;
	self = VALA_ENUM (base);
	return g_strdup ("g_value_set_int");
}


/* FIXME: use GType-registered enums*/
static char* vala_enum_real_get_default_value (ValaTypesymbol* base) {
	ValaEnum * self;
	self = VALA_ENUM (base);
	return g_strdup ("0");
}


gboolean vala_enum_get_is_flags (ValaEnum* self) {
	g_return_val_if_fail (VALA_IS_ENUM (self), FALSE);
	return self->priv->_is_flags;
}


void vala_enum_set_is_flags (ValaEnum* self, gboolean value) {
	g_return_if_fail (VALA_IS_ENUM (self));
	self->priv->_is_flags = value;
}


gboolean vala_enum_get_error_domain (ValaEnum* self) {
	g_return_val_if_fail (VALA_IS_ENUM (self), FALSE);
	return self->priv->_error_domain;
}


void vala_enum_set_error_domain (ValaEnum* self, gboolean value) {
	g_return_if_fail (VALA_IS_ENUM (self));
	self->priv->_error_domain = value;
}


static void vala_enum_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaEnum * self;
	self = VALA_ENUM (object);
	switch (property_id) {
		case VALA_ENUM_IS_FLAGS:
		g_value_set_boolean (value, vala_enum_get_is_flags (self));
		break;
		case VALA_ENUM_ERROR_DOMAIN:
		g_value_set_boolean (value, vala_enum_get_error_domain (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_enum_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaEnum * self;
	self = VALA_ENUM (object);
	switch (property_id) {
		case VALA_ENUM_IS_FLAGS:
		vala_enum_set_is_flags (self, g_value_get_boolean (value));
		break;
		case VALA_ENUM_ERROR_DOMAIN:
		vala_enum_set_error_domain (self, g_value_get_boolean (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_enum_class_init (ValaEnumClass * klass) {
	vala_enum_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaEnumPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_enum_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_enum_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_enum_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_enum_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_enum_real_accept_children;
	VALA_TYPESYMBOL_CLASS (klass)->get_cname = vala_enum_real_get_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_enum_real_get_lower_case_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cname = vala_enum_real_get_lower_case_cname;
	VALA_TYPESYMBOL_CLASS (klass)->get_upper_case_cname = vala_enum_real_get_upper_case_cname;
	VALA_TYPESYMBOL_CLASS (klass)->is_reference_type = vala_enum_real_is_reference_type;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_id = vala_enum_real_get_type_id;
	VALA_TYPESYMBOL_CLASS (klass)->get_marshaller_type_name = vala_enum_real_get_marshaller_type_name;
	VALA_TYPESYMBOL_CLASS (klass)->get_get_value_function = vala_enum_real_get_get_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_set_value_function = vala_enum_real_get_set_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_default_value = vala_enum_real_get_default_value;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_ENUM_IS_FLAGS, g_param_spec_boolean ("is-flags", "is-flags", "is-flags", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_ENUM_ERROR_DOMAIN, g_param_spec_boolean ("error-domain", "error-domain", "error-domain", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_enum_init (ValaEnum * self) {
	self->priv = VALA_ENUM_GET_PRIVATE (self);
	self->priv->values = GEE_LIST (gee_array_list_new (VALA_TYPE_ENUM_VALUE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->methods = GEE_LIST (gee_array_list_new (VALA_TYPE_METHOD, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
}


static void vala_enum_dispose (GObject * obj) {
	ValaEnum * self;
	self = VALA_ENUM (obj);
	(self->priv->values == NULL ? NULL : (self->priv->values = (g_object_unref (self->priv->values), NULL)));
	(self->priv->methods == NULL ? NULL : (self->priv->methods = (g_object_unref (self->priv->methods), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	(self->priv->cprefix = (g_free (self->priv->cprefix), NULL));
	(self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL));
	(self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL));
	G_OBJECT_CLASS (vala_enum_parent_class)->dispose (obj);
}


GType vala_enum_get_type (void) {
	static GType vala_enum_type_id = 0;
	if (G_UNLIKELY (vala_enum_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaEnumClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_enum_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaEnum), 0, (GInstanceInitFunc) vala_enum_init };
		vala_enum_type_id = g_type_register_static (VALA_TYPE_TYPESYMBOL, "ValaEnum", &g_define_type_info, 0);
	}
	return vala_enum_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}




