/* valaparser.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaparser.h>

struct _ValaParserPrivate {
	char* comment;
	char* _file_comment;
};
#define VALA_PARSER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_PARSER, ValaParserPrivate))
enum  {
	VALA_PARSER_DUMMY_PROPERTY
};
static void vala_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file);
static gpointer vala_parser_parent_class = NULL;
static void vala_parser_dispose (GObject * obj);


/**
 * Parse all source files in the specified code context and build a
 * code tree.
 *
 * @param context a code context
 */
void vala_parser_parse (ValaParser* self, ValaCodeContext* context) {
	g_return_if_fail (VALA_IS_PARSER (self));
	g_return_if_fail (context == NULL || VALA_IS_CODE_CONTEXT (context));
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
}


static void vala_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file) {
	ValaParser * self;
	char* _tmp0;
	self = VALA_PARSER (base);
	g_return_if_fail (source_file == NULL || VALA_IS_SOURCE_FILE (source_file));
	if (g_str_has_suffix (vala_source_file_get_filename (source_file), ".vala") || g_str_has_suffix (vala_source_file_get_filename (source_file), ".vapi")) {
		vala_parser_parse_file (self, source_file);
		vala_source_file_set_comment (source_file, self->priv->_file_comment);
	}
	_tmp0 = NULL;
	self->priv->_file_comment = (_tmp0 = NULL, (self->priv->_file_comment = (g_free (self->priv->_file_comment), NULL)), _tmp0);
}


/**
 * Adds the specified comment to the comment stack.
 *
 * @param comment_item a comment string
 * @param file_comment true if file header comment, false otherwise
 */
void vala_parser_push_comment (ValaParser* self, const char* comment_item, gboolean file_comment) {
	g_return_if_fail (VALA_IS_PARSER (self));
	if (self->priv->comment == NULL) {
		char* _tmp1;
		const char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->comment = (_tmp1 = (_tmp0 = comment_item, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->comment = (g_free (self->priv->comment), NULL)), _tmp1);
	} else {
		char* _tmp2;
		_tmp2 = NULL;
		self->priv->comment = (_tmp2 = g_strdup_printf ("%s\n%s", self->priv->comment, comment_item), (self->priv->comment = (g_free (self->priv->comment), NULL)), _tmp2);
	}
	if (file_comment) {
		char* _tmp4;
		const char* _tmp3;
		char* _tmp5;
		_tmp4 = NULL;
		_tmp3 = NULL;
		self->priv->_file_comment = (_tmp4 = (_tmp3 = self->priv->comment, (_tmp3 == NULL ? NULL : g_strdup (_tmp3))), (self->priv->_file_comment = (g_free (self->priv->_file_comment), NULL)), _tmp4);
		_tmp5 = NULL;
		self->priv->comment = (_tmp5 = NULL, (self->priv->comment = (g_free (self->priv->comment), NULL)), _tmp5);
	}
}


/**
 * Clears and returns the content of the comment stack.
 *
 * @return saved comment
 */
char* vala_parser_pop_comment (ValaParser* self) {
	GString* result;
	char* _tmp1;
	const char* index;
	const char* _tmp2;
	char* _tmp3;
	g_return_val_if_fail (VALA_IS_PARSER (self), NULL);
	if (self->priv->comment == NULL) {
		return NULL;
	}
	result = g_string_new (self->priv->comment);
	_tmp1 = NULL;
	self->priv->comment = (_tmp1 = NULL, (self->priv->comment = (g_free (self->priv->comment), NULL)), _tmp1);
	index = NULL;
	while ((index = g_utf8_strchr (result->str, -1, '\t')) != NULL) {
		g_string_erase (result, g_utf8_pointer_to_offset (result->str, index), 1);
	}
	_tmp2 = NULL;
	_tmp3 = NULL;
	return (_tmp3 = (_tmp2 = result->str, (_tmp2 == NULL ? NULL : g_strdup (_tmp2))), (result == NULL ? NULL : (result = (g_string_free (result, TRUE), NULL))), _tmp3);
}


/**
 * Code visitor parsing all Vala source files.
 */
ValaParser* vala_parser_new (void) {
	ValaParser * self;
	self = g_object_newv (VALA_TYPE_PARSER, 0, NULL);
	return self;
}


static void vala_parser_class_init (ValaParserClass * klass) {
	vala_parser_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaParserPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_parser_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_parser_real_visit_source_file;
}


static void vala_parser_init (ValaParser * self) {
	self->priv = VALA_PARSER_GET_PRIVATE (self);
}


static void vala_parser_dispose (GObject * obj) {
	ValaParser * self;
	self = VALA_PARSER (obj);
	(self->priv->comment = (g_free (self->priv->comment), NULL));
	(self->priv->_file_comment = (g_free (self->priv->_file_comment), NULL));
	G_OBJECT_CLASS (vala_parser_parent_class)->dispose (obj);
}


GType vala_parser_get_type (void) {
	static GType vala_parser_type_id = 0;
	if (G_UNLIKELY (vala_parser_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaParserClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_parser_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaParser), 0, (GInstanceInitFunc) vala_parser_init };
		vala_parser_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaParser", &g_define_type_info, 0);
	}
	return vala_parser_type_id;
}




