/* valaccodecompiler.vala
 *
 * Copyright (C) 2007-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <gobject/valaccodecompiler.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <gee/collection.h>
#include <vala/valasourcefile.h>
#include <glib/gstdio.h>

enum  {
	VALA_CCODE_COMPILER_DUMMY_PROPERTY
};
static gboolean vala_ccode_compiler_package_exists (const char* package_name);
static gpointer vala_ccode_compiler_parent_class = NULL;
static int _vala_strcmp0 (const char * str1, const char * str2);


ValaCCodeCompiler* vala_ccode_compiler_new (void) {
	ValaCCodeCompiler * self;
	self = g_object_newv (VALA_TYPE_CCODE_COMPILER, 0, NULL);
	return self;
}


static gboolean vala_ccode_compiler_package_exists (const char* package_name) {
	GError * inner_error;
	char* pc;
	gint exit_status;
	inner_error = NULL;
	pc = g_strconcat ("pkg-config --exists ", package_name, NULL);
	exit_status = 0;
	{
		gboolean _tmp0;
		g_spawn_command_line_sync (pc, NULL, NULL, &exit_status, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == G_SPAWN_ERROR) {
				goto __catch0_g_spawn_error;
			}
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
		}
		return (_tmp0 = (0 == exit_status), (pc = (g_free (pc), NULL)), _tmp0);
	}
	goto __finally0;
	__catch0_g_spawn_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			gboolean _tmp1;
			vala_report_error (NULL, e->message);
			return (_tmp1 = FALSE, (pc = (g_free (pc), NULL)), _tmp1);
		}
	}
	__finally0:
	;
	pc = (g_free (pc), NULL);
}


/**
 * Compile generated C code to object code and optionally link object
 * files.
 *
 * @param context a code context
 */
void vala_ccode_compiler_compile (ValaCCodeCompiler* self, ValaCodeContext* context, const char* cc_command, char** cc_options) {
	GError * inner_error;
	char* pc;
	char* _tmp1;
	char* pkgflags;
	gint exit_status;
	const char* _tmp8;
	char* cmdline;
	char* _tmp17;
	char* _tmp16;
	GeeCollection* source_files;
	GeeCollection* c_source_files;
	g_return_if_fail (VALA_IS_CCODE_COMPILER (self));
	g_return_if_fail (context == NULL || VALA_IS_CODE_CONTEXT (context));
	inner_error = NULL;
	pc = g_strdup ("pkg-config --cflags");
	if (!vala_code_context_get_compile_only (context)) {
		char* _tmp0;
		_tmp0 = NULL;
		pc = (_tmp0 = g_strconcat (pc, (" --libs"), NULL), (pc = (g_free (pc), NULL)), _tmp0);
	}
	_tmp1 = NULL;
	pc = (_tmp1 = g_strconcat (pc, (" gobject-2.0"), NULL), (pc = (g_free (pc), NULL)), _tmp1);
	if (vala_code_context_get_thread (context)) {
		char* _tmp2;
		_tmp2 = NULL;
		pc = (_tmp2 = g_strconcat (pc, (" gthread-2.0"), NULL), (pc = (g_free (pc), NULL)), _tmp2);
	}
	{
		GeeCollection* pkg_collection;
		GeeIterator* pkg_it;
		pkg_collection = vala_code_context_get_packages (context);
		pkg_it = gee_iterable_iterator (GEE_ITERABLE (pkg_collection));
		while (gee_iterator_next (pkg_it)) {
			char* pkg;
			pkg = ((char*) gee_iterator_get (pkg_it));
			{
				if (vala_ccode_compiler_package_exists (pkg)) {
					char* _tmp4;
					char* _tmp3;
					_tmp4 = NULL;
					_tmp3 = NULL;
					pc = (_tmp4 = g_strconcat (pc, (_tmp3 = (g_strconcat (" ", pkg, NULL))), NULL), (pc = (g_free (pc), NULL)), _tmp4);
					_tmp3 = (g_free (_tmp3), NULL);
				}
				pkg = (g_free (pkg), NULL);
			}
		}
		(pkg_collection == NULL ? NULL : (pkg_collection = (g_object_unref (pkg_collection), NULL)));
		(pkg_it == NULL ? NULL : (pkg_it = (g_object_unref (pkg_it), NULL)));
	}
	pkgflags = NULL;
	exit_status = 0;
	{
		gboolean _tmp6;
		char* _tmp5;
		_tmp5 = NULL;
		_tmp6 = g_spawn_command_line_sync (pc, &_tmp5, NULL, &exit_status, &inner_error);
		(pkgflags = (g_free (pkgflags), NULL));
		pkgflags = _tmp5;
		_tmp6;
		if (inner_error != NULL) {
			if (inner_error->domain == G_SPAWN_ERROR) {
				goto __catch1_g_spawn_error;
			}
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
		}
		if (exit_status != 0) {
			char* _tmp7;
			_tmp7 = NULL;
			vala_report_error (NULL, (_tmp7 = g_strdup_printf ("pkg-config exited with status %d", exit_status)));
			_tmp7 = (g_free (_tmp7), NULL);
			pc = (g_free (pc), NULL);
			pkgflags = (g_free (pkgflags), NULL);
			return;
		}
	}
	goto __finally1;
	__catch1_g_spawn_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			vala_report_error (NULL, e->message);
			pc = (g_free (pc), NULL);
			pkgflags = (g_free (pkgflags), NULL);
			return;
		}
	}
	__finally1:
	;
	/* TODO compile the C code files in parallel*/
	if (cc_command == NULL) {
		cc_command = "cc";
	}
	_tmp8 = NULL;
	cmdline = (_tmp8 = cc_command, (_tmp8 == NULL ? NULL : g_strdup (_tmp8)));
	if (vala_code_context_get_debug (context)) {
		char* _tmp9;
		_tmp9 = NULL;
		cmdline = (_tmp9 = g_strconcat (cmdline, (" -g"), NULL), (cmdline = (g_free (cmdline), NULL)), _tmp9);
	}
	if (vala_code_context_get_compile_only (context)) {
		char* _tmp10;
		_tmp10 = NULL;
		cmdline = (_tmp10 = g_strconcat (cmdline, (" -c"), NULL), (cmdline = (g_free (cmdline), NULL)), _tmp10);
	} else {
		if (vala_code_context_get_output (context) != NULL) {
			const char* _tmp11;
			char* output;
			char* _tmp15;
			char* _tmp14;
			char* _tmp13;
			_tmp11 = NULL;
			output = (_tmp11 = vala_code_context_get_output (context), (_tmp11 == NULL ? NULL : g_strdup (_tmp11)));
			if (vala_code_context_get_directory (context) != NULL && _vala_strcmp0 (vala_code_context_get_directory (context), "") != 0) {
				char* _tmp12;
				_tmp12 = NULL;
				output = (_tmp12 = g_strdup_printf ("%s/%s", vala_code_context_get_directory (context), vala_code_context_get_output (context)), (output = (g_free (output), NULL)), _tmp12);
			}
			_tmp15 = NULL;
			_tmp14 = NULL;
			_tmp13 = NULL;
			cmdline = (_tmp15 = g_strconcat (cmdline, (_tmp14 = (g_strconcat (" -o ", (_tmp13 = g_shell_quote (output)), NULL))), NULL), (cmdline = (g_free (cmdline), NULL)), _tmp15);
			_tmp14 = (g_free (_tmp14), NULL);
			_tmp13 = (g_free (_tmp13), NULL);
			output = (g_free (output), NULL);
		}
	}
	_tmp17 = NULL;
	_tmp16 = NULL;
	cmdline = (_tmp17 = g_strconcat (cmdline, (_tmp16 = (g_strconcat (" ", pkgflags, NULL))), NULL), (cmdline = (g_free (cmdline), NULL)), _tmp17);
	_tmp16 = (g_free (_tmp16), NULL);
	if (cc_options != NULL) {
		{
			char** cc_option_collection;
			char** cc_option_it;
			int cc_option_collection_length1;
			cc_option_collection = cc_options;
			cc_option_collection_length1 = -1;
			for (cc_option_it = cc_option_collection; *cc_option_it != NULL; cc_option_it = cc_option_it + 1) {
				const char* _tmp20;
				char* cc_option;
				_tmp20 = NULL;
				cc_option = (_tmp20 = *cc_option_it, (_tmp20 == NULL ? NULL : g_strdup (_tmp20)));
				{
					char* _tmp19;
					char* _tmp18;
					_tmp19 = NULL;
					_tmp18 = NULL;
					cmdline = (_tmp19 = g_strconcat (cmdline, (_tmp18 = (g_strconcat (" ", cc_option, NULL))), NULL), (cmdline = (g_free (cmdline), NULL)), _tmp19);
					_tmp18 = (g_free (_tmp18), NULL);
					cc_option = (g_free (cc_option), NULL);
				}
			}
		}
	}
	/* make sure include files can be found if -d is used */
	if (vala_code_context_get_directory (context) != NULL && _vala_strcmp0 (vala_code_context_get_directory (context), "") != 0) {
		char* _tmp23;
		char* _tmp22;
		char* _tmp21;
		_tmp23 = NULL;
		_tmp22 = NULL;
		_tmp21 = NULL;
		cmdline = (_tmp23 = g_strconcat (cmdline, (_tmp22 = (g_strconcat (" -I", (_tmp21 = g_shell_quote (vala_code_context_get_directory (context))), NULL))), NULL), (cmdline = (g_free (cmdline), NULL)), _tmp23);
		_tmp22 = (g_free (_tmp22), NULL);
		_tmp21 = (g_free (_tmp21), NULL);
	}
	/* we're only interested in non-pkg source files */
	source_files = vala_code_context_get_source_files (context);
	{
		GeeCollection* file_collection;
		GeeIterator* file_it;
		file_collection = source_files;
		file_it = gee_iterable_iterator (GEE_ITERABLE (file_collection));
		while (gee_iterator_next (file_it)) {
			ValaSourceFile* file;
			file = ((ValaSourceFile*) gee_iterator_get (file_it));
			{
				if (!vala_source_file_get_pkg (file)) {
					char* _tmp27;
					char* _tmp26;
					char* _tmp25;
					char* _tmp24;
					_tmp27 = NULL;
					_tmp26 = NULL;
					_tmp25 = NULL;
					_tmp24 = NULL;
					cmdline = (_tmp27 = g_strconcat (cmdline, (_tmp26 = (g_strconcat (" ", (_tmp25 = g_shell_quote ((_tmp24 = vala_source_file_get_csource_filename (file)))), NULL))), NULL), (cmdline = (g_free (cmdline), NULL)), _tmp27);
					_tmp26 = (g_free (_tmp26), NULL);
					_tmp25 = (g_free (_tmp25), NULL);
					_tmp24 = (g_free (_tmp24), NULL);
				}
				(file == NULL ? NULL : (file = (g_object_unref (file), NULL)));
			}
		}
		(file_it == NULL ? NULL : (file_it = (g_object_unref (file_it), NULL)));
	}
	c_source_files = vala_code_context_get_c_source_files (context);
	{
		GeeCollection* file_collection;
		GeeIterator* file_it;
		file_collection = c_source_files;
		file_it = gee_iterable_iterator (GEE_ITERABLE (file_collection));
		while (gee_iterator_next (file_it)) {
			char* file;
			file = ((char*) gee_iterator_get (file_it));
			{
				char* _tmp30;
				char* _tmp29;
				char* _tmp28;
				_tmp30 = NULL;
				_tmp29 = NULL;
				_tmp28 = NULL;
				cmdline = (_tmp30 = g_strconcat (cmdline, (_tmp29 = (g_strconcat (" ", (_tmp28 = g_shell_quote (file)), NULL))), NULL), (cmdline = (g_free (cmdline), NULL)), _tmp30);
				_tmp29 = (g_free (_tmp29), NULL);
				_tmp28 = (g_free (_tmp28), NULL);
				file = (g_free (file), NULL);
			}
		}
		(file_it == NULL ? NULL : (file_it = (g_object_unref (file_it), NULL)));
	}
	{
		g_spawn_command_line_sync (cmdline, NULL, NULL, &exit_status, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == G_SPAWN_ERROR) {
				goto __catch2_g_spawn_error;
			}
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
		}
		if (exit_status != 0) {
			char* _tmp31;
			_tmp31 = NULL;
			vala_report_error (NULL, (_tmp31 = g_strdup_printf ("cc exited with status %d", exit_status)));
			_tmp31 = (g_free (_tmp31), NULL);
		}
	}
	goto __finally2;
	__catch2_g_spawn_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			vala_report_error (NULL, e->message);
		}
	}
	__finally2:
	;
	{
		GeeCollection* file_collection;
		GeeIterator* file_it;
		file_collection = source_files;
		file_it = gee_iterable_iterator (GEE_ITERABLE (file_collection));
		while (gee_iterator_next (file_it)) {
			ValaSourceFile* file;
			file = ((ValaSourceFile*) gee_iterator_get (file_it));
			{
				/* remove generated C source and header files */
				if (!vala_source_file_get_pkg (file)) {
					if (!vala_code_context_get_save_csources (context)) {
						char* _tmp32;
						_tmp32 = NULL;
						g_unlink ((_tmp32 = vala_source_file_get_csource_filename (file)));
						_tmp32 = (g_free (_tmp32), NULL);
					}
					if (!vala_code_context_get_save_cheaders (context)) {
						char* _tmp33;
						_tmp33 = NULL;
						g_unlink ((_tmp33 = vala_source_file_get_cheader_filename (file)));
						_tmp33 = (g_free (_tmp33), NULL);
					}
				}
				(file == NULL ? NULL : (file = (g_object_unref (file), NULL)));
			}
		}
		(file_it == NULL ? NULL : (file_it = (g_object_unref (file_it), NULL)));
	}
	pc = (g_free (pc), NULL);
	pkgflags = (g_free (pkgflags), NULL);
	cmdline = (g_free (cmdline), NULL);
	(source_files == NULL ? NULL : (source_files = (g_object_unref (source_files), NULL)));
	(c_source_files == NULL ? NULL : (c_source_files = (g_object_unref (c_source_files), NULL)));
}


static void vala_ccode_compiler_class_init (ValaCCodeCompilerClass * klass) {
	vala_ccode_compiler_parent_class = g_type_class_peek_parent (klass);
}


static void vala_ccode_compiler_init (ValaCCodeCompiler * self) {
}


GType vala_ccode_compiler_get_type (void) {
	static GType vala_ccode_compiler_type_id = 0;
	if (G_UNLIKELY (vala_ccode_compiler_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeCompilerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_compiler_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeCompiler), 0, (GInstanceInitFunc) vala_ccode_compiler_init };
		vala_ccode_compiler_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaCCodeCompiler", &g_define_type_info, 0);
	}
	return vala_ccode_compiler_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




