/* valaproperty.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaproperty.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valaformalparameter.h>
#include <vala/valaexpression.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>

struct _ValaPropertyPrivate {
	ValaPropertyAccessor* _get_accessor;
	ValaPropertyAccessor* _set_accessor;
	ValaFormalParameter* _this_parameter;
	gboolean _notify;
	gboolean _no_accessor_method;
	gboolean _interface_only;
	gboolean _is_abstract;
	gboolean _is_virtual;
	gboolean _overrides;
	ValaProperty* _base_property;
	ValaProperty* _base_interface_property;
	ValaExpression* _default_expression;
	gboolean lock_used;
	ValaDataType* _data_type;
	gboolean _instance;
	char* _nick;
	char* _blurb;
};
#define VALA_PROPERTY_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_PROPERTY, ValaPropertyPrivate))
enum  {
	VALA_PROPERTY_DUMMY_PROPERTY,
	VALA_PROPERTY_TYPE_REFERENCE,
	VALA_PROPERTY_GET_ACCESSOR,
	VALA_PROPERTY_SET_ACCESSOR,
	VALA_PROPERTY_THIS_PARAMETER,
	VALA_PROPERTY_NOTIFY,
	VALA_PROPERTY_NO_ACCESSOR_METHOD,
	VALA_PROPERTY_INTERFACE_ONLY,
	VALA_PROPERTY_IS_ABSTRACT,
	VALA_PROPERTY_IS_VIRTUAL,
	VALA_PROPERTY_OVERRIDES,
	VALA_PROPERTY_INSTANCE,
	VALA_PROPERTY_BASE_PROPERTY,
	VALA_PROPERTY_BASE_INTERFACE_PROPERTY,
	VALA_PROPERTY_DEFAULT_EXPRESSION,
	VALA_PROPERTY_NICK,
	VALA_PROPERTY_BLURB
};
static void vala_property_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_property_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_property_get_canonical_name (ValaProperty* self);
static gboolean vala_property_real_get_lock_used (ValaLockable* base);
static void vala_property_real_set_lock_used (ValaLockable* base, gboolean used);
static void vala_property_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gpointer vala_property_parent_class = NULL;
static ValaLockableIface* vala_property_vala_lockable_parent_iface = NULL;
static void vala_property_dispose (GObject * obj);
static int _vala_strcmp0 (const char * str1, const char * str2);


/**
 * Creates a new property.
 *
 * @param name         property name
 * @param type         property type
 * @param get_accessor get accessor
 * @param set_accessor set/construct accessor
 * @param source       reference to source code
 * @return             newly created property
 */
ValaProperty* vala_property_new (const char* _name, ValaDataType* type, ValaPropertyAccessor* _get_accessor, ValaPropertyAccessor* _set_accessor, ValaSourceReference* source) {
	ValaProperty * self;
	g_return_val_if_fail (type == NULL || VALA_IS_DATA_TYPE (type), NULL);
	g_return_val_if_fail (_get_accessor == NULL || VALA_IS_PROPERTY_ACCESSOR (_get_accessor), NULL);
	g_return_val_if_fail (_set_accessor == NULL || VALA_IS_PROPERTY_ACCESSOR (_set_accessor), NULL);
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	self = g_object_newv (VALA_TYPE_PROPERTY, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), _name);
	vala_property_set_type_reference (self, type);
	vala_property_set_get_accessor (self, _get_accessor);
	vala_property_set_set_accessor (self, _set_accessor);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source);
	return self;
}


static void vala_property_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaProperty * self;
	self = VALA_PROPERTY (base);
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_member (visitor, VALA_MEMBER (self));
	vala_code_visitor_visit_property (visitor, self);
}


static void vala_property_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaProperty * self;
	self = VALA_PROPERTY (base);
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (vala_property_get_type_reference (self)), visitor);
	if (vala_property_get_get_accessor (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_property_get_get_accessor (self)), visitor);
	}
	if (vala_property_get_set_accessor (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_property_get_set_accessor (self)), visitor);
	}
	if (vala_property_get_default_expression (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_property_get_default_expression (self)), visitor);
	}
}


/**
 * Returns the C name of this property in upper case. Words are
 * separated by underscores. The upper case C name of the class is
 * prefix of the result.
 *
 * @return the upper case name to be used in C code
 */
char* vala_property_get_upper_case_cname (ValaProperty* self) {
	char* _tmp2;
	char* _tmp1;
	char* _tmp0;
	char* _tmp3;
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp3 = NULL;
	return (_tmp3 = g_utf8_strup ((_tmp2 = g_strdup_printf ("%s_%s", (_tmp0 = vala_symbol_get_lower_case_cname (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)), NULL)), (_tmp1 = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name (VALA_SYMBOL (self)))))), -1), (_tmp2 = (g_free (_tmp2), NULL)), (_tmp1 = (g_free (_tmp1), NULL)), (_tmp0 = (g_free (_tmp0), NULL)), _tmp3);
}


/**
 * Returns the string literal of this property to be used in C code.
 *
 * @return string literal to be used in C code
 */
ValaCCodeConstant* vala_property_get_canonical_cconstant (ValaProperty* self) {
	char* _tmp1;
	char* _tmp0;
	ValaCCodeConstant* _tmp2;
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = vala_ccode_constant_new ((_tmp1 = g_strdup_printf ("\"%s\"", (_tmp0 = vala_property_get_canonical_name (self))))), (_tmp1 = (g_free (_tmp1), NULL)), (_tmp0 = (g_free (_tmp0), NULL)), _tmp2);
}


static char* vala_property_get_canonical_name (ValaProperty* self) {
	GString* str;
	const char* _tmp0;
	char* i;
	const char* _tmp3;
	char* _tmp4;
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	str = g_string_new ("");
	_tmp0 = NULL;
	i = (_tmp0 = vala_symbol_get_name (VALA_SYMBOL (self)), (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		char* _tmp2;
		const char* _tmp1;
		c = g_utf8_get_char (i);
		if (c == '_') {
			g_string_append_c (str, '-');
		} else {
			g_string_append_unichar (str, c);
		}
		_tmp2 = NULL;
		_tmp1 = NULL;
		i = (_tmp2 = (_tmp1 = g_utf8_next_char (i), (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (i = (g_free (i), NULL)), _tmp2);
	}
	_tmp3 = NULL;
	_tmp4 = NULL;
	return (_tmp4 = (_tmp3 = str->str, (_tmp3 == NULL ? NULL : g_strdup (_tmp3))), (str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL))), (i = (g_free (i), NULL)), _tmp4);
}


/**
 * Process all associated attributes.
 */
void vala_property_process_attributes (ValaProperty* self) {
	g_return_if_fail (VALA_IS_PROPERTY (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = VALA_CODE_NODE (self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp2;
			ValaAttribute* a;
			_tmp2 = NULL;
			a = (_tmp2 = ((ValaAttribute*) a_it->data), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "Notify") == 0) {
					vala_property_set_notify (self, TRUE);
				} else {
					if (_vala_strcmp0 (vala_attribute_get_name (a), "NoAccessorMethod") == 0) {
						vala_property_set_no_accessor_method (self, TRUE);
					} else {
						if (_vala_strcmp0 (vala_attribute_get_name (a), "Description") == 0) {
							if (vala_attribute_has_argument (a, "nick")) {
								char* _tmp0;
								_tmp0 = NULL;
								vala_property_set_nick (self, (_tmp0 = vala_attribute_get_string (a, "nick")));
								_tmp0 = (g_free (_tmp0), NULL);
							}
							if (vala_attribute_has_argument (a, "blurb")) {
								char* _tmp1;
								_tmp1 = NULL;
								vala_property_set_blurb (self, (_tmp1 = vala_attribute_get_string (a, "blurb")));
								_tmp1 = (g_free (_tmp1), NULL);
							}
						}
					}
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


static gboolean vala_property_real_get_lock_used (ValaLockable* base) {
	ValaProperty * self;
	self = VALA_PROPERTY (base);
	return self->priv->lock_used;
}


static void vala_property_real_set_lock_used (ValaLockable* base, gboolean used) {
	ValaProperty * self;
	self = VALA_PROPERTY (base);
	self->priv->lock_used = used;
}


/**
 * Checks whether the accessors and type of the specified property
 * matches this property.
 *
 * @param prop a property
 * @return     true if the specified property is compatible to this
 *             property
 */
gboolean vala_property_equals (ValaProperty* self, ValaProperty* prop2) {
	g_return_val_if_fail (VALA_IS_PROPERTY (self), FALSE);
	g_return_val_if_fail (prop2 == NULL || VALA_IS_PROPERTY (prop2), FALSE);
	if (!vala_data_type_equals (vala_property_get_type_reference (prop2), vala_property_get_type_reference (self))) {
		return FALSE;
	}
	if ((vala_property_get_get_accessor (self) == NULL && vala_property_get_get_accessor (prop2) != NULL) || (vala_property_get_get_accessor (self) != NULL && vala_property_get_get_accessor (prop2) == NULL)) {
		return FALSE;
	}
	if ((vala_property_get_set_accessor (self) == NULL && vala_property_get_set_accessor (prop2) != NULL) || (vala_property_get_set_accessor (self) != NULL && vala_property_get_set_accessor (prop2) == NULL)) {
		return FALSE;
	}
	if (vala_property_get_set_accessor (self) != NULL) {
		if (vala_property_accessor_get_writable (vala_property_get_set_accessor (self)) != vala_property_accessor_get_writable (vala_property_get_set_accessor (prop2))) {
			return FALSE;
		}
		if (vala_property_accessor_get_construction (vala_property_get_set_accessor (self)) != vala_property_accessor_get_construction (vala_property_get_set_accessor (prop2))) {
			return FALSE;
		}
	}
	return TRUE;
}


static void vala_property_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaProperty * self;
	self = VALA_PROPERTY (base);
	g_return_if_fail (old_type == NULL || VALA_IS_DATA_TYPE (old_type));
	g_return_if_fail (new_type == NULL || VALA_IS_DATA_TYPE (new_type));
	if (vala_property_get_type_reference (self) == old_type) {
		vala_property_set_type_reference (self, new_type);
	}
}


ValaDataType* vala_property_get_type_reference (ValaProperty* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	return self->priv->_data_type;
}


void vala_property_set_type_reference (ValaProperty* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (VALA_IS_PROPERTY (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_data_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (g_object_unref (self->priv->_data_type), NULL))), _tmp2);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_data_type), VALA_CODE_NODE (self));
}


ValaPropertyAccessor* vala_property_get_get_accessor (ValaProperty* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	return self->priv->_get_accessor;
}


void vala_property_set_get_accessor (ValaProperty* self, ValaPropertyAccessor* value) {
	ValaPropertyAccessor* _tmp2;
	ValaPropertyAccessor* _tmp1;
	g_return_if_fail (VALA_IS_PROPERTY (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_get_accessor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_get_accessor == NULL ? NULL : (self->priv->_get_accessor = (g_object_unref (self->priv->_get_accessor), NULL))), _tmp2);
}


ValaPropertyAccessor* vala_property_get_set_accessor (ValaProperty* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	return self->priv->_set_accessor;
}


void vala_property_set_set_accessor (ValaProperty* self, ValaPropertyAccessor* value) {
	ValaPropertyAccessor* _tmp2;
	ValaPropertyAccessor* _tmp1;
	g_return_if_fail (VALA_IS_PROPERTY (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_set_accessor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_set_accessor == NULL ? NULL : (self->priv->_set_accessor = (g_object_unref (self->priv->_set_accessor), NULL))), _tmp2);
}


ValaFormalParameter* vala_property_get_this_parameter (ValaProperty* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	return self->priv->_this_parameter;
}


void vala_property_set_this_parameter (ValaProperty* self, ValaFormalParameter* value) {
	ValaFormalParameter* _tmp2;
	ValaFormalParameter* _tmp1;
	g_return_if_fail (VALA_IS_PROPERTY (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_this_parameter = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (g_object_unref (self->priv->_this_parameter), NULL))), _tmp2);
}


gboolean vala_property_get_notify (ValaProperty* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY (self), FALSE);
	return self->priv->_notify;
}


void vala_property_set_notify (ValaProperty* self, gboolean value) {
	g_return_if_fail (VALA_IS_PROPERTY (self));
	self->priv->_notify = value;
}


gboolean vala_property_get_no_accessor_method (ValaProperty* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY (self), FALSE);
	return self->priv->_no_accessor_method;
}


void vala_property_set_no_accessor_method (ValaProperty* self, gboolean value) {
	g_return_if_fail (VALA_IS_PROPERTY (self));
	self->priv->_no_accessor_method = value;
}


gboolean vala_property_get_interface_only (ValaProperty* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY (self), FALSE);
	return self->priv->_interface_only;
}


void vala_property_set_interface_only (ValaProperty* self, gboolean value) {
	g_return_if_fail (VALA_IS_PROPERTY (self));
	self->priv->_interface_only = value;
}


gboolean vala_property_get_is_abstract (ValaProperty* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY (self), FALSE);
	return self->priv->_is_abstract;
}


void vala_property_set_is_abstract (ValaProperty* self, gboolean value) {
	g_return_if_fail (VALA_IS_PROPERTY (self));
	self->priv->_is_abstract = value;
}


gboolean vala_property_get_is_virtual (ValaProperty* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY (self), FALSE);
	return self->priv->_is_virtual;
}


void vala_property_set_is_virtual (ValaProperty* self, gboolean value) {
	g_return_if_fail (VALA_IS_PROPERTY (self));
	self->priv->_is_virtual = value;
}


gboolean vala_property_get_overrides (ValaProperty* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY (self), FALSE);
	return self->priv->_overrides;
}


void vala_property_set_overrides (ValaProperty* self, gboolean value) {
	g_return_if_fail (VALA_IS_PROPERTY (self));
	self->priv->_overrides = value;
}


gboolean vala_property_get_instance (ValaProperty* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY (self), FALSE);
	return self->priv->_instance;
}


void vala_property_set_instance (ValaProperty* self, gboolean value) {
	g_return_if_fail (VALA_IS_PROPERTY (self));
	self->priv->_instance = value;
}


ValaProperty* vala_property_get_base_property (ValaProperty* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	return self->priv->_base_property;
}


void vala_property_set_base_property (ValaProperty* self, ValaProperty* value) {
	g_return_if_fail (VALA_IS_PROPERTY (self));
	self->priv->_base_property = value;
}


ValaProperty* vala_property_get_base_interface_property (ValaProperty* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	return self->priv->_base_interface_property;
}


void vala_property_set_base_interface_property (ValaProperty* self, ValaProperty* value) {
	ValaProperty* _tmp2;
	ValaProperty* _tmp1;
	g_return_if_fail (VALA_IS_PROPERTY (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_base_interface_property = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_base_interface_property == NULL ? NULL : (self->priv->_base_interface_property = (g_object_unref (self->priv->_base_interface_property), NULL))), _tmp2);
}


ValaExpression* vala_property_get_default_expression (ValaProperty* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	return self->priv->_default_expression;
}


void vala_property_set_default_expression (ValaProperty* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_PROPERTY (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_default_expression = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_default_expression == NULL ? NULL : (self->priv->_default_expression = (g_object_unref (self->priv->_default_expression), NULL))), _tmp2);
}


char* vala_property_get_nick (ValaProperty* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	if (self->priv->_nick == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->_nick = (_tmp0 = vala_property_get_canonical_name (self), (self->priv->_nick = (g_free (self->priv->_nick), NULL)), _tmp0);
	}
	return self->priv->_nick;
}


void vala_property_set_nick (ValaProperty* self, const char* value) {
	char* _tmp3;
	const char* _tmp2;
	g_return_if_fail (VALA_IS_PROPERTY (self));
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->_nick = (_tmp3 = (_tmp2 = value, (_tmp2 == NULL ? NULL : g_strdup (_tmp2))), (self->priv->_nick = (g_free (self->priv->_nick), NULL)), _tmp3);
}


char* vala_property_get_blurb (ValaProperty* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY (self), NULL);
	if (self->priv->_blurb == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->_blurb = (_tmp0 = vala_property_get_canonical_name (self), (self->priv->_blurb = (g_free (self->priv->_blurb), NULL)), _tmp0);
	}
	return self->priv->_blurb;
}


void vala_property_set_blurb (ValaProperty* self, const char* value) {
	char* _tmp3;
	const char* _tmp2;
	g_return_if_fail (VALA_IS_PROPERTY (self));
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->_blurb = (_tmp3 = (_tmp2 = value, (_tmp2 == NULL ? NULL : g_strdup (_tmp2))), (self->priv->_blurb = (g_free (self->priv->_blurb), NULL)), _tmp3);
}


static void vala_property_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaProperty * self;
	self = VALA_PROPERTY (object);
	switch (property_id) {
		case VALA_PROPERTY_TYPE_REFERENCE:
		g_value_set_object (value, vala_property_get_type_reference (self));
		break;
		case VALA_PROPERTY_GET_ACCESSOR:
		g_value_set_object (value, vala_property_get_get_accessor (self));
		break;
		case VALA_PROPERTY_SET_ACCESSOR:
		g_value_set_object (value, vala_property_get_set_accessor (self));
		break;
		case VALA_PROPERTY_THIS_PARAMETER:
		g_value_set_object (value, vala_property_get_this_parameter (self));
		break;
		case VALA_PROPERTY_NOTIFY:
		g_value_set_boolean (value, vala_property_get_notify (self));
		break;
		case VALA_PROPERTY_NO_ACCESSOR_METHOD:
		g_value_set_boolean (value, vala_property_get_no_accessor_method (self));
		break;
		case VALA_PROPERTY_INTERFACE_ONLY:
		g_value_set_boolean (value, vala_property_get_interface_only (self));
		break;
		case VALA_PROPERTY_IS_ABSTRACT:
		g_value_set_boolean (value, vala_property_get_is_abstract (self));
		break;
		case VALA_PROPERTY_IS_VIRTUAL:
		g_value_set_boolean (value, vala_property_get_is_virtual (self));
		break;
		case VALA_PROPERTY_OVERRIDES:
		g_value_set_boolean (value, vala_property_get_overrides (self));
		break;
		case VALA_PROPERTY_INSTANCE:
		g_value_set_boolean (value, vala_property_get_instance (self));
		break;
		case VALA_PROPERTY_BASE_PROPERTY:
		g_value_set_object (value, vala_property_get_base_property (self));
		break;
		case VALA_PROPERTY_BASE_INTERFACE_PROPERTY:
		g_value_set_object (value, vala_property_get_base_interface_property (self));
		break;
		case VALA_PROPERTY_DEFAULT_EXPRESSION:
		g_value_set_object (value, vala_property_get_default_expression (self));
		break;
		case VALA_PROPERTY_NICK:
		g_value_set_string (value, vala_property_get_nick (self));
		break;
		case VALA_PROPERTY_BLURB:
		g_value_set_string (value, vala_property_get_blurb (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_property_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaProperty * self;
	self = VALA_PROPERTY (object);
	switch (property_id) {
		case VALA_PROPERTY_TYPE_REFERENCE:
		vala_property_set_type_reference (self, g_value_get_object (value));
		break;
		case VALA_PROPERTY_GET_ACCESSOR:
		vala_property_set_get_accessor (self, g_value_get_object (value));
		break;
		case VALA_PROPERTY_SET_ACCESSOR:
		vala_property_set_set_accessor (self, g_value_get_object (value));
		break;
		case VALA_PROPERTY_THIS_PARAMETER:
		vala_property_set_this_parameter (self, g_value_get_object (value));
		break;
		case VALA_PROPERTY_NOTIFY:
		vala_property_set_notify (self, g_value_get_boolean (value));
		break;
		case VALA_PROPERTY_NO_ACCESSOR_METHOD:
		vala_property_set_no_accessor_method (self, g_value_get_boolean (value));
		break;
		case VALA_PROPERTY_INTERFACE_ONLY:
		vala_property_set_interface_only (self, g_value_get_boolean (value));
		break;
		case VALA_PROPERTY_IS_ABSTRACT:
		vala_property_set_is_abstract (self, g_value_get_boolean (value));
		break;
		case VALA_PROPERTY_IS_VIRTUAL:
		vala_property_set_is_virtual (self, g_value_get_boolean (value));
		break;
		case VALA_PROPERTY_OVERRIDES:
		vala_property_set_overrides (self, g_value_get_boolean (value));
		break;
		case VALA_PROPERTY_INSTANCE:
		vala_property_set_instance (self, g_value_get_boolean (value));
		break;
		case VALA_PROPERTY_BASE_PROPERTY:
		vala_property_set_base_property (self, g_value_get_object (value));
		break;
		case VALA_PROPERTY_BASE_INTERFACE_PROPERTY:
		vala_property_set_base_interface_property (self, g_value_get_object (value));
		break;
		case VALA_PROPERTY_DEFAULT_EXPRESSION:
		vala_property_set_default_expression (self, g_value_get_object (value));
		break;
		case VALA_PROPERTY_NICK:
		vala_property_set_nick (self, g_value_get_string (value));
		break;
		case VALA_PROPERTY_BLURB:
		vala_property_set_blurb (self, g_value_get_string (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_property_class_init (ValaPropertyClass * klass) {
	vala_property_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaPropertyPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_property_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_property_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_property_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_property_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_property_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_property_real_replace_type;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_TYPE_REFERENCE, g_param_spec_object ("type-reference", "type-reference", "type-reference", VALA_TYPE_DATA_TYPE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_GET_ACCESSOR, g_param_spec_object ("get-accessor", "get-accessor", "get-accessor", VALA_TYPE_PROPERTY_ACCESSOR, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_SET_ACCESSOR, g_param_spec_object ("set-accessor", "set-accessor", "set-accessor", VALA_TYPE_PROPERTY_ACCESSOR, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_THIS_PARAMETER, g_param_spec_object ("this-parameter", "this-parameter", "this-parameter", VALA_TYPE_FORMAL_PARAMETER, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_NOTIFY, g_param_spec_boolean ("notify", "notify", "notify", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_NO_ACCESSOR_METHOD, g_param_spec_boolean ("no-accessor-method", "no-accessor-method", "no-accessor-method", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_INTERFACE_ONLY, g_param_spec_boolean ("interface-only", "interface-only", "interface-only", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_IS_ABSTRACT, g_param_spec_boolean ("is-abstract", "is-abstract", "is-abstract", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_IS_VIRTUAL, g_param_spec_boolean ("is-virtual", "is-virtual", "is-virtual", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_OVERRIDES, g_param_spec_boolean ("overrides", "overrides", "overrides", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_INSTANCE, g_param_spec_boolean ("instance", "instance", "instance", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_BASE_PROPERTY, g_param_spec_object ("base-property", "base-property", "base-property", VALA_TYPE_PROPERTY, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_BASE_INTERFACE_PROPERTY, g_param_spec_object ("base-interface-property", "base-interface-property", "base-interface-property", VALA_TYPE_PROPERTY, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_DEFAULT_EXPRESSION, g_param_spec_object ("default-expression", "default-expression", "default-expression", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_NICK, g_param_spec_string ("nick", "nick", "nick", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_PROPERTY_BLURB, g_param_spec_string ("blurb", "blurb", "blurb", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_property_vala_lockable_interface_init (ValaLockableIface * iface) {
	vala_property_vala_lockable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_lock_used = vala_property_real_get_lock_used;
	iface->set_lock_used = vala_property_real_set_lock_used;
}


static void vala_property_init (ValaProperty * self) {
	self->priv = VALA_PROPERTY_GET_PRIVATE (self);
	self->priv->lock_used = FALSE;
	self->priv->_instance = TRUE;
}


static void vala_property_dispose (GObject * obj) {
	ValaProperty * self;
	self = VALA_PROPERTY (obj);
	(self->priv->_get_accessor == NULL ? NULL : (self->priv->_get_accessor = (g_object_unref (self->priv->_get_accessor), NULL)));
	(self->priv->_set_accessor == NULL ? NULL : (self->priv->_set_accessor = (g_object_unref (self->priv->_set_accessor), NULL)));
	(self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (g_object_unref (self->priv->_this_parameter), NULL)));
	(self->priv->_base_interface_property == NULL ? NULL : (self->priv->_base_interface_property = (g_object_unref (self->priv->_base_interface_property), NULL)));
	(self->priv->_default_expression == NULL ? NULL : (self->priv->_default_expression = (g_object_unref (self->priv->_default_expression), NULL)));
	(self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (g_object_unref (self->priv->_data_type), NULL)));
	self->priv->_nick = (g_free (self->priv->_nick), NULL);
	self->priv->_blurb = (g_free (self->priv->_blurb), NULL);
	G_OBJECT_CLASS (vala_property_parent_class)->dispose (obj);
}


GType vala_property_get_type (void) {
	static GType vala_property_type_id = 0;
	if (G_UNLIKELY (vala_property_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaPropertyClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_property_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaProperty), 0, (GInstanceInitFunc) vala_property_init };
		static const GInterfaceInfo vala_lockable_info = { (GInterfaceInitFunc) vala_property_vala_lockable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_property_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaProperty", &g_define_type_info, 0);
		g_type_add_interface_static (vala_property_type_id, VALA_TYPE_LOCKABLE, &vala_lockable_info);
	}
	return vala_property_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




