/* valacatchclause.vala
 *
 * Copyright (C) 2007-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valacatchclause.h>
#include <vala/valablock.h>
#include <vala/valavariabledeclarator.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>

struct _ValaCatchClausePrivate {
	char* _variable_name;
	ValaBlock* _body;
	ValaVariableDeclarator* _variable_declarator;
	ValaDataType* _data_type;
};
#define VALA_CATCH_CLAUSE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CATCH_CLAUSE, ValaCatchClausePrivate))
enum  {
	VALA_CATCH_CLAUSE_DUMMY_PROPERTY,
	VALA_CATCH_CLAUSE_TYPE_REFERENCE,
	VALA_CATCH_CLAUSE_VARIABLE_NAME,
	VALA_CATCH_CLAUSE_BODY,
	VALA_CATCH_CLAUSE_VARIABLE_DECLARATOR
};
static void vala_catch_clause_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_catch_clause_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_catch_clause_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gpointer vala_catch_clause_parent_class = NULL;
static void vala_catch_clause_dispose (GObject * obj);


/**
 * Creates a new catch clause.
 *
 * @param type_reference   error type
 * @param variable_name    error variable name
 * @param body             error handler body
 * @param source_reference reference to source code
 * @return                 newly created catch clause
 */
ValaCatchClause* vala_catch_clause_new (ValaDataType* type_reference, const char* variable_name, ValaBlock* body, ValaSourceReference* source_reference) {
	ValaCatchClause * self;
	g_return_val_if_fail (type_reference == NULL || VALA_IS_DATA_TYPE (type_reference), NULL);
	g_return_val_if_fail (body == NULL || VALA_IS_BLOCK (body), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_CATCH_CLAUSE, 0, NULL);
	vala_catch_clause_set_type_reference (self, type_reference);
	vala_catch_clause_set_variable_name (self, variable_name);
	vala_catch_clause_set_body (self, body);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


static void vala_catch_clause_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaCatchClause * self;
	self = VALA_CATCH_CLAUSE (base);
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_catch_clause (visitor, self);
}


static void vala_catch_clause_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaCatchClause * self;
	self = VALA_CATCH_CLAUSE (base);
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
	if (vala_catch_clause_get_type_reference (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_catch_clause_get_type_reference (self)), visitor);
	}
	vala_code_node_accept (VALA_CODE_NODE (vala_catch_clause_get_body (self)), visitor);
}


static void vala_catch_clause_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaCatchClause * self;
	self = VALA_CATCH_CLAUSE (base);
	g_return_if_fail (old_type == NULL || VALA_IS_DATA_TYPE (old_type));
	g_return_if_fail (new_type == NULL || VALA_IS_DATA_TYPE (new_type));
	if (vala_catch_clause_get_type_reference (self) == old_type) {
		vala_catch_clause_set_type_reference (self, new_type);
	}
}


ValaDataType* vala_catch_clause_get_type_reference (ValaCatchClause* self) {
	g_return_val_if_fail (VALA_IS_CATCH_CLAUSE (self), NULL);
	return self->priv->_data_type;
}


void vala_catch_clause_set_type_reference (ValaCatchClause* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (VALA_IS_CATCH_CLAUSE (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_data_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (g_object_unref (self->priv->_data_type), NULL))), _tmp2);
	if (self->priv->_data_type != NULL) {
		vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_data_type), VALA_CODE_NODE (self));
	}
}


char* vala_catch_clause_get_variable_name (ValaCatchClause* self) {
	g_return_val_if_fail (VALA_IS_CATCH_CLAUSE (self), NULL);
	return self->priv->_variable_name;
}


void vala_catch_clause_set_variable_name (ValaCatchClause* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (VALA_IS_CATCH_CLAUSE (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_variable_name = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_variable_name = (g_free (self->priv->_variable_name), NULL)), _tmp2);
}


ValaBlock* vala_catch_clause_get_body (ValaCatchClause* self) {
	g_return_val_if_fail (VALA_IS_CATCH_CLAUSE (self), NULL);
	return self->priv->_body;
}


void vala_catch_clause_set_body (ValaCatchClause* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (VALA_IS_CATCH_CLAUSE (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_body == NULL ? NULL : (self->priv->_body = (g_object_unref (self->priv->_body), NULL))), _tmp2);
}


ValaVariableDeclarator* vala_catch_clause_get_variable_declarator (ValaCatchClause* self) {
	g_return_val_if_fail (VALA_IS_CATCH_CLAUSE (self), NULL);
	return self->priv->_variable_declarator;
}


void vala_catch_clause_set_variable_declarator (ValaCatchClause* self, ValaVariableDeclarator* value) {
	ValaVariableDeclarator* _tmp2;
	ValaVariableDeclarator* _tmp1;
	g_return_if_fail (VALA_IS_CATCH_CLAUSE (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_variable_declarator = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_variable_declarator == NULL ? NULL : (self->priv->_variable_declarator = (g_object_unref (self->priv->_variable_declarator), NULL))), _tmp2);
}


static void vala_catch_clause_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaCatchClause * self;
	self = VALA_CATCH_CLAUSE (object);
	switch (property_id) {
		case VALA_CATCH_CLAUSE_TYPE_REFERENCE:
		g_value_set_object (value, vala_catch_clause_get_type_reference (self));
		break;
		case VALA_CATCH_CLAUSE_VARIABLE_NAME:
		g_value_set_string (value, vala_catch_clause_get_variable_name (self));
		break;
		case VALA_CATCH_CLAUSE_BODY:
		g_value_set_object (value, vala_catch_clause_get_body (self));
		break;
		case VALA_CATCH_CLAUSE_VARIABLE_DECLARATOR:
		g_value_set_object (value, vala_catch_clause_get_variable_declarator (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_catch_clause_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaCatchClause * self;
	self = VALA_CATCH_CLAUSE (object);
	switch (property_id) {
		case VALA_CATCH_CLAUSE_TYPE_REFERENCE:
		vala_catch_clause_set_type_reference (self, g_value_get_object (value));
		break;
		case VALA_CATCH_CLAUSE_VARIABLE_NAME:
		vala_catch_clause_set_variable_name (self, g_value_get_string (value));
		break;
		case VALA_CATCH_CLAUSE_BODY:
		vala_catch_clause_set_body (self, g_value_get_object (value));
		break;
		case VALA_CATCH_CLAUSE_VARIABLE_DECLARATOR:
		vala_catch_clause_set_variable_declarator (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_catch_clause_class_init (ValaCatchClauseClass * klass) {
	vala_catch_clause_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaCatchClausePrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_catch_clause_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_catch_clause_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_catch_clause_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_catch_clause_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_catch_clause_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_catch_clause_real_replace_type;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CATCH_CLAUSE_TYPE_REFERENCE, g_param_spec_object ("type-reference", "type-reference", "type-reference", VALA_TYPE_DATA_TYPE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CATCH_CLAUSE_VARIABLE_NAME, g_param_spec_string ("variable-name", "variable-name", "variable-name", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CATCH_CLAUSE_BODY, g_param_spec_object ("body", "body", "body", VALA_TYPE_BLOCK, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CATCH_CLAUSE_VARIABLE_DECLARATOR, g_param_spec_object ("variable-declarator", "variable-declarator", "variable-declarator", VALA_TYPE_VARIABLE_DECLARATOR, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_catch_clause_init (ValaCatchClause * self) {
	self->priv = VALA_CATCH_CLAUSE_GET_PRIVATE (self);
}


static void vala_catch_clause_dispose (GObject * obj) {
	ValaCatchClause * self;
	self = VALA_CATCH_CLAUSE (obj);
	self->priv->_variable_name = (g_free (self->priv->_variable_name), NULL);
	(self->priv->_body == NULL ? NULL : (self->priv->_body = (g_object_unref (self->priv->_body), NULL)));
	(self->priv->_variable_declarator == NULL ? NULL : (self->priv->_variable_declarator = (g_object_unref (self->priv->_variable_declarator), NULL)));
	(self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (g_object_unref (self->priv->_data_type), NULL)));
	G_OBJECT_CLASS (vala_catch_clause_parent_class)->dispose (obj);
}


GType vala_catch_clause_get_type (void) {
	static GType vala_catch_clause_type_id = 0;
	if (G_UNLIKELY (vala_catch_clause_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCatchClauseClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_catch_clause_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCatchClause), 0, (GInstanceInitFunc) vala_catch_clause_init };
		vala_catch_clause_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaCatchClause", &g_define_type_info, 0);
	}
	return vala_catch_clause_type_id;
}




