/* valacodenode.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valacodenode.h>
#include <ccode/valaccodelinedirective.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valadatatype.h>
#include <vala/valaexpression.h>
#include <vala/valaattribute.h>
#include <vala/valasourcefile.h>

struct _ValaCodeNodePrivate {
	ValaCodeNode* _parent_node;
	ValaSourceReference* _source_reference;
	ValaCodeBinding* _code_binding;
	gboolean _error;
	gboolean _tree_can_fail;
	ValaCCodeNode* _ccodenode;
};
#define VALA_CODE_NODE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CODE_NODE, ValaCodeNodePrivate))
enum  {
	VALA_CODE_NODE_DUMMY_PROPERTY,
	VALA_CODE_NODE_PARENT_NODE,
	VALA_CODE_NODE_SOURCE_REFERENCE,
	VALA_CODE_NODE_CCODENODE,
	VALA_CODE_NODE_CODE_BINDING,
	VALA_CODE_NODE_ERROR,
	VALA_CODE_NODE_TREE_CAN_FAIL
};
static void vala_code_node_real_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
static void vala_code_node_real_accept_children (ValaCodeNode* self, ValaCodeVisitor* visitor);
static void vala_code_node_real_replace_type (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
static void vala_code_node_real_replace_expression (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
static char* vala_code_node_real_to_string (ValaCodeNode* self);
static gpointer vala_code_node_parent_class = NULL;
static void vala_code_node_dispose (GObject * obj);
static int _vala_strcmp0 (const char * str1, const char * str2);


/**
 * Visits this code node with the specified CodeVisitor.
 *
 * @param visitor the visitor to be called while traversing
 */
static void vala_code_node_real_accept (ValaCodeNode* self, ValaCodeVisitor* visitor) {
	g_return_if_fail (VALA_IS_CODE_NODE (self));
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
}


void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor) {
	VALA_CODE_NODE_GET_CLASS (self)->accept (self, visitor);
}


/**
 * Visits all children of this code node with the specified CodeVisitor.
 *
 * @param visitor the visitor to be called while traversing
 */
static void vala_code_node_real_accept_children (ValaCodeNode* self, ValaCodeVisitor* visitor) {
	g_return_if_fail (VALA_IS_CODE_NODE (self));
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
}


void vala_code_node_accept_children (ValaCodeNode* self, ValaCodeVisitor* visitor) {
	VALA_CODE_NODE_GET_CLASS (self)->accept_children (self, visitor);
}


static void vala_code_node_real_replace_type (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type) {
	g_return_if_fail (VALA_IS_CODE_NODE (self));
	g_return_if_fail (old_type == NULL || VALA_IS_DATA_TYPE (old_type));
	g_return_if_fail (new_type == NULL || VALA_IS_DATA_TYPE (new_type));
}


void vala_code_node_replace_type (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type) {
	VALA_CODE_NODE_GET_CLASS (self)->replace_type (self, old_type, new_type);
}


static void vala_code_node_real_replace_expression (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node) {
	g_return_if_fail (VALA_IS_CODE_NODE (self));
	g_return_if_fail (old_node == NULL || VALA_IS_EXPRESSION (old_node));
	g_return_if_fail (new_node == NULL || VALA_IS_EXPRESSION (new_node));
}


void vala_code_node_replace_expression (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node) {
	VALA_CODE_NODE_GET_CLASS (self)->replace_expression (self, old_node, new_node);
}


/**
 * Returns the specified attribute.
 *
 * @param name attribute name
 * @return     attribute
 */
ValaAttribute* vala_code_node_get_attribute (ValaCodeNode* self, const char* name) {
	g_return_val_if_fail (VALA_IS_CODE_NODE (self), NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = self->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp1;
			ValaAttribute* a;
			_tmp1 = NULL;
			a = (_tmp1 = ((ValaAttribute*) a_it->data), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
			{
				/* FIXME: use hash table*/
				if (_vala_strcmp0 (vala_attribute_get_name (a), name) == 0) {
					return a;
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
	return NULL;
}


/**
 * Returns a string that represents this code node.
 *
 * @return a string representation
 */
static char* vala_code_node_real_to_string (ValaCodeNode* self) {
	GString* str;
	const char* _tmp1;
	char* _tmp2;
	g_return_val_if_fail (VALA_IS_CODE_NODE (self), NULL);
	str = g_string_new ("");
	g_string_append (g_string_append (str, "/* "), g_type_name (G_TYPE_FROM_INSTANCE (G_OBJECT (self))));
	if (vala_code_node_get_source_reference (self) != NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		g_string_append (g_string_append (str, "@"), (_tmp0 = vala_source_reference_to_string (vala_code_node_get_source_reference (self))));
		_tmp0 = (g_free (_tmp0), NULL);
	}
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = (_tmp1 = g_string_append (str, " */")->str, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL))), _tmp2);
}


char* vala_code_node_to_string (ValaCodeNode* self) {
	return VALA_CODE_NODE_GET_CLASS (self)->to_string (self);
}


ValaCodeNode* vala_code_node_get_parent_node (ValaCodeNode* self) {
	g_return_val_if_fail (VALA_IS_CODE_NODE (self), NULL);
	return self->priv->_parent_node;
}


void vala_code_node_set_parent_node (ValaCodeNode* self, ValaCodeNode* value) {
	g_return_if_fail (VALA_IS_CODE_NODE (self));
	self->priv->_parent_node = value;
}


ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self) {
	g_return_val_if_fail (VALA_IS_CODE_NODE (self), NULL);
	return self->priv->_source_reference;
}


void vala_code_node_set_source_reference (ValaCodeNode* self, ValaSourceReference* value) {
	ValaSourceReference* _tmp2;
	ValaSourceReference* _tmp1;
	g_return_if_fail (VALA_IS_CODE_NODE (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_source_reference = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_source_reference == NULL ? NULL : (self->priv->_source_reference = (g_object_unref (self->priv->_source_reference), NULL))), _tmp2);
}


ValaCCodeNode* vala_code_node_get_ccodenode (ValaCodeNode* self) {
	g_return_val_if_fail (VALA_IS_CODE_NODE (self), NULL);
	return self->priv->_ccodenode;
}


void vala_code_node_set_ccodenode (ValaCodeNode* self, ValaCCodeNode* value) {
	ValaCCodeNode* _tmp4;
	ValaCCodeNode* _tmp3;
	g_return_if_fail (VALA_IS_CODE_NODE (self));
	if (vala_code_node_get_source_reference (self) != NULL) {
		ValaCCodeLineDirective* _tmp2;
		char* _tmp1;
		_tmp2 = NULL;
		_tmp1 = NULL;
		vala_ccode_node_set_line (value, (_tmp2 = vala_ccode_line_directive_new ((_tmp1 = g_path_get_basename (vala_source_file_get_filename (vala_source_reference_get_file (vala_code_node_get_source_reference (self))))), vala_source_reference_get_first_line (vala_code_node_get_source_reference (self)))));
		(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
		_tmp1 = (g_free (_tmp1), NULL);
	}
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->priv->_ccodenode = (_tmp4 = (_tmp3 = value, (_tmp3 == NULL ? NULL : g_object_ref (_tmp3))), (self->priv->_ccodenode == NULL ? NULL : (self->priv->_ccodenode = (g_object_unref (self->priv->_ccodenode), NULL))), _tmp4);
}


ValaCodeBinding* vala_code_node_get_code_binding (ValaCodeNode* self) {
	g_return_val_if_fail (VALA_IS_CODE_NODE (self), NULL);
	return self->priv->_code_binding;
}


void vala_code_node_set_code_binding (ValaCodeNode* self, ValaCodeBinding* value) {
	ValaCodeBinding* _tmp2;
	ValaCodeBinding* _tmp1;
	g_return_if_fail (VALA_IS_CODE_NODE (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_code_binding = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_code_binding == NULL ? NULL : (self->priv->_code_binding = (g_object_unref (self->priv->_code_binding), NULL))), _tmp2);
}


gboolean vala_code_node_get_error (ValaCodeNode* self) {
	g_return_val_if_fail (VALA_IS_CODE_NODE (self), FALSE);
	return self->priv->_error;
}


void vala_code_node_set_error (ValaCodeNode* self, gboolean value) {
	g_return_if_fail (VALA_IS_CODE_NODE (self));
	self->priv->_error = value;
}


gboolean vala_code_node_get_tree_can_fail (ValaCodeNode* self) {
	g_return_val_if_fail (VALA_IS_CODE_NODE (self), FALSE);
	return self->priv->_tree_can_fail;
}


void vala_code_node_set_tree_can_fail (ValaCodeNode* self, gboolean value) {
	g_return_if_fail (VALA_IS_CODE_NODE (self));
	self->priv->_tree_can_fail = value;
}


static void vala_code_node_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaCodeNode * self;
	self = VALA_CODE_NODE (object);
	switch (property_id) {
		case VALA_CODE_NODE_PARENT_NODE:
		g_value_set_object (value, vala_code_node_get_parent_node (self));
		break;
		case VALA_CODE_NODE_SOURCE_REFERENCE:
		g_value_set_object (value, vala_code_node_get_source_reference (self));
		break;
		case VALA_CODE_NODE_CCODENODE:
		g_value_set_object (value, vala_code_node_get_ccodenode (self));
		break;
		case VALA_CODE_NODE_CODE_BINDING:
		g_value_set_object (value, vala_code_node_get_code_binding (self));
		break;
		case VALA_CODE_NODE_ERROR:
		g_value_set_boolean (value, vala_code_node_get_error (self));
		break;
		case VALA_CODE_NODE_TREE_CAN_FAIL:
		g_value_set_boolean (value, vala_code_node_get_tree_can_fail (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_code_node_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaCodeNode * self;
	self = VALA_CODE_NODE (object);
	switch (property_id) {
		case VALA_CODE_NODE_PARENT_NODE:
		vala_code_node_set_parent_node (self, g_value_get_object (value));
		break;
		case VALA_CODE_NODE_SOURCE_REFERENCE:
		vala_code_node_set_source_reference (self, g_value_get_object (value));
		break;
		case VALA_CODE_NODE_CCODENODE:
		vala_code_node_set_ccodenode (self, g_value_get_object (value));
		break;
		case VALA_CODE_NODE_CODE_BINDING:
		vala_code_node_set_code_binding (self, g_value_get_object (value));
		break;
		case VALA_CODE_NODE_ERROR:
		vala_code_node_set_error (self, g_value_get_boolean (value));
		break;
		case VALA_CODE_NODE_TREE_CAN_FAIL:
		vala_code_node_set_tree_can_fail (self, g_value_get_boolean (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_code_node_class_init (ValaCodeNodeClass * klass) {
	vala_code_node_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaCodeNodePrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_code_node_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_code_node_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_code_node_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_code_node_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_code_node_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_code_node_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_code_node_real_replace_expression;
	VALA_CODE_NODE_CLASS (klass)->to_string = vala_code_node_real_to_string;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_NODE_PARENT_NODE, g_param_spec_object ("parent-node", "parent-node", "parent-node", VALA_TYPE_CODE_NODE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_NODE_SOURCE_REFERENCE, g_param_spec_object ("source-reference", "source-reference", "source-reference", VALA_TYPE_SOURCE_REFERENCE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_NODE_CCODENODE, g_param_spec_object ("ccodenode", "ccodenode", "ccodenode", VALA_TYPE_CCODE_NODE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_NODE_CODE_BINDING, g_param_spec_object ("code-binding", "code-binding", "code-binding", VALA_TYPE_CODE_BINDING, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_NODE_ERROR, g_param_spec_boolean ("error", "error", "error", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CODE_NODE_TREE_CAN_FAIL, g_param_spec_boolean ("tree-can-fail", "tree-can-fail", "tree-can-fail", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_code_node_init (ValaCodeNode * self) {
	self->priv = VALA_CODE_NODE_GET_PRIVATE (self);
}


static void vala_code_node_dispose (GObject * obj) {
	ValaCodeNode * self;
	self = VALA_CODE_NODE (obj);
	(self->priv->_source_reference == NULL ? NULL : (self->priv->_source_reference = (g_object_unref (self->priv->_source_reference), NULL)));
	(self->attributes == NULL ? NULL : (self->attributes = (g_list_foreach (self->attributes, ((GFunc) g_object_unref), NULL), g_list_free (self->attributes), NULL)));
	(self->priv->_code_binding == NULL ? NULL : (self->priv->_code_binding = (g_object_unref (self->priv->_code_binding), NULL)));
	(self->priv->_ccodenode == NULL ? NULL : (self->priv->_ccodenode = (g_object_unref (self->priv->_ccodenode), NULL)));
	G_OBJECT_CLASS (vala_code_node_parent_class)->dispose (obj);
}


GType vala_code_node_get_type (void) {
	static GType vala_code_node_type_id = 0;
	if (G_UNLIKELY (vala_code_node_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCodeNodeClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_code_node_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCodeNode), 0, (GInstanceInitFunc) vala_code_node_init };
		vala_code_node_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaCodeNode", &g_define_type_info, G_TYPE_FLAG_ABSTRACT);
	}
	return vala_code_node_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




