/* valacompiler.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 * Copyright (C) 1996-2002, 2004, 2005, 2006 Free Software Foundation, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <compiler/valacompiler.h>
#include <stdlib.h>
#include <string.h>
#include <vala/valacodecontext.h>
#include <vala/valareport.h>
#include <stdio.h>
#include <vala/valasourcefile.h>
#include <glib/gstdio.h>
#include <vala/valasourcereference.h>
#include <gobject/valaccodegenerator.h>
#include <vala/valaparser.h>
#include <vala/valaattributeprocessor.h>
#include <vala/valasymbolresolver.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valacfgbuilder.h>
#include <vala/valanullchecker.h>
#include <vala/valamemorymanager.h>
#include <vala/valacodegenerator.h>
#include <vala/valainterfacewriter.h>
#include <gobject/valagidlwriter.h>
#include <gobject/valaccodecompiler.h>
#include <config.h>


#define VALA_TYPE_COMPILER (vala_compiler_get_type ())
#define VALA_COMPILER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_COMPILER, ValaCompiler))
#define VALA_COMPILER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_COMPILER, ValaCompilerClass))
#define VALA_IS_COMPILER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_COMPILER))
#define VALA_IS_COMPILER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_COMPILER))
#define VALA_COMPILER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_COMPILER, ValaCompilerClass))

typedef struct _ValaCompiler ValaCompiler;
typedef struct _ValaCompilerClass ValaCompilerClass;
typedef struct _ValaCompilerPrivate ValaCompilerPrivate;

struct _ValaCompiler {
	GObject parent_instance;
	ValaCompilerPrivate * priv;
};

struct _ValaCompilerClass {
	GObjectClass parent_class;
};



struct _ValaCompilerPrivate {
	ValaCodeContext* context;
};

#define VALA_COMPILER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_COMPILER, ValaCompilerPrivate))
enum  {
	VALA_COMPILER_DUMMY_PROPERTY
};
static char* vala_compiler_basedir;
static char* vala_compiler_directory;
static gboolean vala_compiler_version;
static char** vala_compiler_sources;
static char** vala_compiler_vapi_directories;
static char* vala_compiler_library;
static char** vala_compiler_packages;
static gboolean vala_compiler_ccode_only;
static gboolean vala_compiler_compile_only;
static char* vala_compiler_output;
static gboolean vala_compiler_debug;
static gboolean vala_compiler_thread;
static gboolean vala_compiler_disable_assert;
static gboolean vala_compiler_disable_checking;
static gboolean vala_compiler_disable_non_null;
static gboolean vala_compiler_non_null_experimental;
static gboolean vala_compiler_verbose;
static char* vala_compiler_cc_command;
static char** vala_compiler_cc_options;
static gboolean vala_compiler_save_temps;
static char** vala_compiler_defines;
static gboolean vala_compiler_quiet_mode;
static gint vala_compiler_quit (ValaCompiler* self);
static gboolean vala_compiler_add_package (ValaCompiler* self, ValaCodeContext* context, const char* pkg);
static gint vala_compiler_run (ValaCompiler* self);
static gboolean vala_compiler_ends_with_dir_separator (const char* s);
static char* vala_compiler_realpath (const char* name);
static gint vala_compiler_main (char** args, int args_length1);
static ValaCompiler* vala_compiler_new (void);
static gpointer vala_compiler_parent_class = NULL;
static void vala_compiler_dispose (GObject * obj);
static GType vala_compiler_get_type (void);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);

static const GOptionEntry VALA_COMPILER_options[] = {{"vapidir", ((gchar) 0), 0, G_OPTION_ARG_FILENAME_ARRAY, &vala_compiler_vapi_directories, "Look for package bindings in DIRECTORY", "DIRECTORY..."}, {"pkg", ((gchar) 0), 0, G_OPTION_ARG_STRING_ARRAY, &vala_compiler_packages, "Include binding for PACKAGE", "PACKAGE..."}, {"library", ((gchar) 0), 0, G_OPTION_ARG_STRING, &vala_compiler_library, "Library name", "NAME"}, {"basedir", 'b', 0, G_OPTION_ARG_FILENAME, &vala_compiler_basedir, "Base source directory", "DIRECTORY"}, {"directory", 'd', 0, G_OPTION_ARG_FILENAME, &vala_compiler_directory, "Output directory", "DIRECTORY"}, {"version", ((gchar) 0), 0, G_OPTION_ARG_NONE, &vala_compiler_version, "Display version number", NULL}, {"ccode", 'C', 0, G_OPTION_ARG_NONE, &vala_compiler_ccode_only, "Output C code", NULL}, {"compile", 'c', 0, G_OPTION_ARG_NONE, &vala_compiler_compile_only, "Compile but do not link", NULL}, {"output", 'o', 0, G_OPTION_ARG_FILENAME, &vala_compiler_output, "Place output in file FILE", "FILE"}, {"debug", 'g', 0, G_OPTION_ARG_NONE, &vala_compiler_debug, "Produce debug information", NULL}, {"thread", ((gchar) 0), 0, G_OPTION_ARG_NONE, &vala_compiler_thread, "Enable multithreading support", NULL}, {"define", 'D', 0, G_OPTION_ARG_STRING_ARRAY, &vala_compiler_defines, "Define SYMBOL", "SYMBOL..."}, {"disable-assert", ((gchar) 0), 0, G_OPTION_ARG_NONE, &vala_compiler_disable_assert, "Disable assertions", NULL}, {"disable-checking", ((gchar) 0), 0, G_OPTION_ARG_NONE, &vala_compiler_disable_checking, "Disable run-time checks", NULL}, {"disable-non-null", ((gchar) 0), 0, G_OPTION_ARG_NONE, &vala_compiler_disable_non_null, "Disable non-null types", NULL}, {"enable-non-null-experimental", ((gchar) 0), 0, G_OPTION_ARG_NONE, &vala_compiler_non_null_experimental, "Enable experimental enhancements for non-null types", NULL}, {"cc", ((gchar) 0), 0, G_OPTION_ARG_STRING, &vala_compiler_cc_command, "Use COMMAND as C compiler command", "COMMAND"}, {"Xcc", 'X', 0, G_OPTION_ARG_STRING_ARRAY, &vala_compiler_cc_options, "Pass OPTION to the C compiler", "OPTION..."}, {"save-temps", ((gchar) 0), 0, G_OPTION_ARG_NONE, &vala_compiler_save_temps, "Keep temporary files", NULL}, {"quiet", 'q', 0, G_OPTION_ARG_NONE, &vala_compiler_quiet_mode, "Do not print messages to the console", NULL}, {"verbose", 'v', 0, G_OPTION_ARG_NONE, &vala_compiler_verbose, "Include the source line text when reporting errors or warnings."}, {"", ((gchar) 0), 0, G_OPTION_ARG_FILENAME_ARRAY, &vala_compiler_sources, NULL, "FILE..."}, {NULL}};


static gint vala_compiler_quit (ValaCompiler* self) {
	g_return_val_if_fail (VALA_IS_COMPILER (self), 0);
	if (vala_report_get_errors () == 0 && vala_report_get_warnings () == 0) {
		return 0;
	}
	if (vala_report_get_errors () == 0) {
		if (!vala_compiler_quiet_mode) {
			fprintf (stdout, "Compilation succeeded - %d warning(s)\n", vala_report_get_warnings ());
		}
		return 0;
	} else {
		if (!vala_compiler_quiet_mode) {
			fprintf (stdout, "Compilation failed: %d error(s), %d warning(s)\n", vala_report_get_errors (), vala_report_get_warnings ());
		}
		return 1;
	}
}


static gboolean vala_compiler_add_package (ValaCompiler* self, ValaCodeContext* context, const char* pkg) {
	GError * inner_error;
	char* package_path;
	ValaSourceFile* _tmp2;
	char* _tmp4;
	char* _tmp3;
	char* _tmp5;
	char* deps_filename;
	gboolean _tmp11;
	g_return_val_if_fail (VALA_IS_COMPILER (self), FALSE);
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (context), FALSE);
	g_return_val_if_fail (pkg != NULL, FALSE);
	inner_error = NULL;
	if (vala_code_context_has_package (context, pkg)) {
		/* ignore multiple occurences of the same package*/
		return TRUE;
	}
	package_path = vala_code_context_get_package_path (context, pkg, -1, vala_compiler_vapi_directories);
	if (package_path == NULL) {
		gboolean _tmp1;
		return (_tmp1 = FALSE, (package_path = (g_free (package_path), NULL)), _tmp1);
	}
	vala_code_context_add_package (context, pkg);
	_tmp2 = NULL;
	vala_code_context_add_source_file (context, (_tmp2 = vala_source_file_new (context, package_path, TRUE)));
	(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
	_tmp4 = NULL;
	_tmp3 = NULL;
	_tmp5 = NULL;
	deps_filename = (_tmp5 = g_build_filename ((_tmp3 = g_path_get_dirname (package_path)), (_tmp4 = g_strdup_printf ("%s.deps", pkg)), NULL), (_tmp4 = (g_free (_tmp4), NULL)), (_tmp3 = (g_free (_tmp3), NULL)), _tmp5);
	if (g_file_test (deps_filename, G_FILE_TEST_EXISTS)) {
		{
			char* deps_content;
			gulong deps_len;
			gboolean _tmp7;
			char* _tmp6;
			deps_content = NULL;
			deps_len = 0UL;
			_tmp6 = NULL;
			_tmp7 = g_file_get_contents (deps_filename, &_tmp6, &deps_len, &inner_error);
			(deps_content = (g_free (deps_content), NULL));
			deps_content = _tmp6;
			_tmp7;
			if (inner_error != NULL) {
				if (inner_error->domain == G_FILE_ERROR) {
					goto __catch0_g_file_error;
				}
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
			}
			{
				char** dep_collection;
				char** dep_it;
				int dep_collection_length1;
				dep_collection = g_strsplit (deps_content, "\n", 0);
				dep_collection_length1 = -1;
				for (dep_it = dep_collection; *dep_it != NULL; dep_it = dep_it + 1) {
					const char* _tmp9;
					char* dep;
					_tmp9 = NULL;
					dep = (_tmp9 = *dep_it, (_tmp9 == NULL ? NULL : g_strdup (_tmp9)));
					{
						if (_vala_strcmp0 (dep, "") != 0) {
							if (!vala_compiler_add_package (self, context, dep)) {
								char* _tmp8;
								_tmp8 = NULL;
								vala_report_error (NULL, (_tmp8 = g_strdup_printf ("%s, dependency of %s, not found in specified Vala API directories", dep, pkg)));
								_tmp8 = (g_free (_tmp8), NULL);
							}
						}
						dep = (g_free (dep), NULL);
					}
				}
				dep_collection = (_vala_array_free (dep_collection, dep_collection_length1, ((GDestroyNotify) g_free)), NULL);
			}
			deps_content = (g_free (deps_content), NULL);
		}
		goto __finally0;
		__catch0_g_file_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				char* _tmp10;
				_tmp10 = NULL;
				vala_report_error (NULL, (_tmp10 = g_strdup_printf ("Unable to read dependency file: %s", e->message)));
				_tmp10 = (g_free (_tmp10), NULL);
			}
		}
		__finally0:
		;
	}
	return (_tmp11 = TRUE, (package_path = (g_free (package_path), NULL)), (deps_filename = (g_free (deps_filename), NULL)), _tmp11);
}


static gint vala_compiler_run (ValaCompiler* self) {
	ValaCodeContext* _tmp0;
	ValaCCodeGenerator* _tmp7;
	char** _tmp17;
	ValaParser* parser;
	ValaAttributeProcessor* attributeprocessor;
	ValaSymbolResolver* resolver;
	ValaSemanticAnalyzer* analyzer;
	ValaCFGBuilder* cfg_builder;
	ValaMemoryManager* memory_manager;
	gint _tmp31;
	g_return_val_if_fail (VALA_IS_COMPILER (self), 0);
	_tmp0 = NULL;
	self->priv->context = (_tmp0 = vala_code_context_new (), (self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL))), _tmp0);
	/* default to build executable*/
	if (!vala_compiler_ccode_only && !vala_compiler_compile_only && vala_compiler_output == NULL) {
		/* strip extension if there is one
		 else we use the default output file of the C compiler*/
		if (g_utf8_strrchr (vala_compiler_sources[0], ((glong) -1), ((gunichar) '.')) != NULL) {
			glong dot;
			char* _tmp3;
			char* _tmp2;
			char* _tmp1;
			dot = g_utf8_pointer_to_offset (vala_compiler_sources[0], g_utf8_strrchr (vala_compiler_sources[0], ((glong) -1), ((gunichar) '.')));
			_tmp3 = NULL;
			_tmp2 = NULL;
			_tmp1 = NULL;
			vala_compiler_output = (_tmp3 = g_path_get_basename ((_tmp2 = g_utf8_offset_to_pointer (vala_compiler_sources[0], ((glong) 0)), g_strndup (_tmp2, g_utf8_offset_to_pointer (_tmp2, dot) - _tmp2))), (vala_compiler_output = (g_free (vala_compiler_output), NULL)), _tmp3);
			_tmp1 = (g_free (_tmp1), NULL);
		}
	}
	vala_code_context_set_library (self->priv->context, vala_compiler_library);
	vala_code_context_set_assert (self->priv->context, !vala_compiler_disable_assert);
	vala_code_context_set_checking (self->priv->context, !vala_compiler_disable_checking);
	vala_code_context_set_non_null (self->priv->context, !vala_compiler_disable_non_null || vala_compiler_non_null_experimental);
	vala_code_context_set_non_null_experimental (self->priv->context, vala_compiler_non_null_experimental);
	vala_report_set_verbose_errors (vala_compiler_verbose);
	vala_code_context_set_ccode_only (self->priv->context, vala_compiler_ccode_only);
	vala_code_context_set_compile_only (self->priv->context, vala_compiler_compile_only);
	vala_code_context_set_output (self->priv->context, vala_compiler_output);
	if (vala_compiler_basedir != NULL) {
		char* _tmp4;
		_tmp4 = NULL;
		vala_code_context_set_basedir (self->priv->context, (_tmp4 = vala_compiler_realpath (vala_compiler_basedir)));
		_tmp4 = (g_free (_tmp4), NULL);
	}
	if (vala_compiler_directory != NULL) {
		char* _tmp5;
		_tmp5 = NULL;
		vala_code_context_set_directory (self->priv->context, (_tmp5 = vala_compiler_realpath (vala_compiler_directory)));
		_tmp5 = (g_free (_tmp5), NULL);
	} else {
		vala_code_context_set_directory (self->priv->context, vala_code_context_get_basedir (self->priv->context));
	}
	vala_code_context_set_debug (self->priv->context, vala_compiler_debug);
	vala_code_context_set_thread (self->priv->context, vala_compiler_thread);
	vala_code_context_set_save_temps (self->priv->context, vala_compiler_save_temps);
	if (vala_compiler_defines != NULL) {
		{
			char** define_collection;
			char** define_it;
			int define_collection_length1;
			define_collection = vala_compiler_defines;
			define_collection_length1 = -1;
			for (define_it = define_collection; *define_it != NULL; define_it = define_it + 1) {
				const char* _tmp6;
				char* define;
				_tmp6 = NULL;
				define = (_tmp6 = *define_it, (_tmp6 == NULL ? NULL : g_strdup (_tmp6)));
				{
					vala_code_context_add_define (self->priv->context, define);
					define = (g_free (define), NULL);
				}
			}
		}
	}
	_tmp7 = NULL;
	vala_code_context_set_codegen (self->priv->context, VALA_CODE_GENERATOR ((_tmp7 = vala_ccode_generator_new ())));
	(_tmp7 == NULL ? NULL : (_tmp7 = (g_object_unref (_tmp7), NULL)));
	/* default package */
	if (!vala_compiler_add_package (self, self->priv->context, "glib-2.0")) {
		vala_report_error (NULL, "glib-2.0 not found in specified Vala API directories");
	}
	if (vala_compiler_packages != NULL) {
		char** _tmp10;
		{
			char** package_collection;
			char** package_it;
			int package_collection_length1;
			package_collection = vala_compiler_packages;
			package_collection_length1 = -1;
			for (package_it = package_collection; *package_it != NULL; package_it = package_it + 1) {
				const char* _tmp9;
				char* package;
				_tmp9 = NULL;
				package = (_tmp9 = *package_it, (_tmp9 == NULL ? NULL : g_strdup (_tmp9)));
				{
					if (!vala_compiler_add_package (self, self->priv->context, package)) {
						char* _tmp8;
						_tmp8 = NULL;
						vala_report_error (NULL, (_tmp8 = g_strdup_printf ("%s not found in specified Vala API directories", package)));
						_tmp8 = (g_free (_tmp8), NULL);
					}
					package = (g_free (package), NULL);
				}
			}
		}
		_tmp10 = NULL;
		vala_compiler_packages = (_tmp10 = NULL, (vala_compiler_packages = (_vala_array_free (vala_compiler_packages, -1, ((GDestroyNotify) g_free)), NULL)), _tmp10);
	}
	if (vala_report_get_errors () > 0) {
		return vala_compiler_quit (self);
	}
	{
		char** source_collection;
		char** source_it;
		int source_collection_length1;
		source_collection = vala_compiler_sources;
		source_collection_length1 = -1;
		for (source_it = source_collection; *source_it != NULL; source_it = source_it + 1) {
			const char* _tmp16;
			char* source;
			_tmp16 = NULL;
			source = (_tmp16 = *source_it, (_tmp16 == NULL ? NULL : g_strdup (_tmp16)));
			{
				if (g_file_test (source, G_FILE_TEST_EXISTS)) {
					char* rpath;
					rpath = vala_compiler_realpath (source);
					if (g_str_has_suffix (source, ".vala")) {
						ValaSourceFile* _tmp12;
						_tmp12 = NULL;
						vala_code_context_add_source_file (self->priv->context, (_tmp12 = vala_source_file_new (self->priv->context, rpath, FALSE)));
						(_tmp12 == NULL ? NULL : (_tmp12 = (g_object_unref (_tmp12), NULL)));
					} else {
						if (g_str_has_suffix (source, ".vapi")) {
							ValaSourceFile* _tmp13;
							_tmp13 = NULL;
							vala_code_context_add_source_file (self->priv->context, (_tmp13 = vala_source_file_new (self->priv->context, rpath, TRUE)));
							(_tmp13 == NULL ? NULL : (_tmp13 = (g_object_unref (_tmp13), NULL)));
						} else {
							if (g_str_has_suffix (source, ".c")) {
								vala_code_context_add_c_source_file (self->priv->context, rpath);
							} else {
								char* _tmp14;
								_tmp14 = NULL;
								vala_report_error (NULL, (_tmp14 = g_strdup_printf ("%s is not a supported source file type. Only .vala, .vapi, and .c files are supported.", source)));
								_tmp14 = (g_free (_tmp14), NULL);
							}
						}
					}
					rpath = (g_free (rpath), NULL);
				} else {
					char* _tmp15;
					_tmp15 = NULL;
					vala_report_error (NULL, (_tmp15 = g_strdup_printf ("%s not found", source)));
					_tmp15 = (g_free (_tmp15), NULL);
				}
				source = (g_free (source), NULL);
			}
		}
	}
	_tmp17 = NULL;
	vala_compiler_sources = (_tmp17 = NULL, (vala_compiler_sources = (_vala_array_free (vala_compiler_sources, -1, ((GDestroyNotify) g_free)), NULL)), _tmp17);
	if (vala_report_get_errors () > 0) {
		return vala_compiler_quit (self);
	}
	parser = vala_parser_new ();
	vala_parser_parse (parser, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp19;
		return (_tmp19 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), _tmp19);
	}
	attributeprocessor = vala_attribute_processor_new ();
	vala_attribute_processor_process (attributeprocessor, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp20;
		return (_tmp20 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), _tmp20);
	}
	resolver = vala_symbol_resolver_new ();
	vala_symbol_resolver_resolve (resolver, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp21;
		return (_tmp21 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), _tmp21);
	}
	analyzer = vala_semantic_analyzer_new ();
	vala_semantic_analyzer_analyze (analyzer, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp22;
		return (_tmp22 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), (analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL))), _tmp22);
	}
	cfg_builder = vala_cfg_builder_new ();
	vala_cfg_builder_build_cfg (cfg_builder, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp23;
		return (_tmp23 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), (analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL))), (cfg_builder == NULL ? NULL : (cfg_builder = (g_object_unref (cfg_builder), NULL))), _tmp23);
	}
	if (vala_code_context_get_non_null_experimental (self->priv->context)) {
		ValaNullChecker* null_checker;
		null_checker = vala_null_checker_new ();
		vala_null_checker_check (null_checker, self->priv->context);
		if (vala_report_get_errors () > 0) {
			gint _tmp24;
			return (_tmp24 = vala_compiler_quit (self), (null_checker == NULL ? NULL : (null_checker = (g_object_unref (null_checker), NULL))), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), (analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL))), (cfg_builder == NULL ? NULL : (cfg_builder = (g_object_unref (cfg_builder), NULL))), _tmp24);
		}
		(null_checker == NULL ? NULL : (null_checker = (g_object_unref (null_checker), NULL)));
	}
	memory_manager = vala_memory_manager_new ();
	vala_memory_manager_analyze (memory_manager, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp25;
		return (_tmp25 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), (analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL))), (cfg_builder == NULL ? NULL : (cfg_builder = (g_object_unref (cfg_builder), NULL))), (memory_manager == NULL ? NULL : (memory_manager = (g_object_unref (memory_manager), NULL))), _tmp25);
	}
	vala_code_generator_emit (vala_code_context_get_codegen (self->priv->context), self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint _tmp26;
		return (_tmp26 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), (analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL))), (cfg_builder == NULL ? NULL : (cfg_builder = (g_object_unref (cfg_builder), NULL))), (memory_manager == NULL ? NULL : (memory_manager = (g_object_unref (memory_manager), NULL))), _tmp26);
	}
	if (vala_compiler_library != NULL) {
		ValaInterfaceWriter* interface_writer;
		char* vapi_filename;
		ValaGIdlWriter* gidl_writer;
		char* gidl_filename;
		char* _tmp29;
		interface_writer = vala_interface_writer_new ();
		vapi_filename = g_strdup_printf ("%s.vapi", vala_compiler_library);
		/* put .vapi file in current directory unless -d has been explicitly specified*/
		if (vala_compiler_directory != NULL) {
			char* _tmp27;
			_tmp27 = NULL;
			vapi_filename = (_tmp27 = g_strdup_printf ("%s/%s", vala_code_context_get_directory (self->priv->context), vapi_filename), (vapi_filename = (g_free (vapi_filename), NULL)), _tmp27);
		}
		vala_interface_writer_write_file (interface_writer, self->priv->context, vapi_filename);
		gidl_writer = vala_gidl_writer_new ();
		gidl_filename = g_strdup_printf ("%s.gidl", vala_compiler_library);
		/* put .gidl file in current directory unless -d has been explicitly specified*/
		if (vala_compiler_directory != NULL) {
			char* _tmp28;
			_tmp28 = NULL;
			gidl_filename = (_tmp28 = g_strdup_printf ("%s/%s", vala_code_context_get_directory (self->priv->context), gidl_filename), (gidl_filename = (g_free (gidl_filename), NULL)), _tmp28);
		}
		vala_gidl_writer_write_file (gidl_writer, self->priv->context, gidl_filename);
		_tmp29 = NULL;
		vala_compiler_library = (_tmp29 = NULL, (vala_compiler_library = (g_free (vala_compiler_library), NULL)), _tmp29);
		(interface_writer == NULL ? NULL : (interface_writer = (g_object_unref (interface_writer), NULL)));
		vapi_filename = (g_free (vapi_filename), NULL);
		(gidl_writer == NULL ? NULL : (gidl_writer = (g_object_unref (gidl_writer), NULL)));
		gidl_filename = (g_free (gidl_filename), NULL);
	}
	if (!vala_compiler_ccode_only) {
		ValaCCodeCompiler* ccompiler;
		ccompiler = vala_ccode_compiler_new ();
		if (vala_compiler_cc_options == NULL) {
			char** _tmp30;
			_tmp30 = NULL;
			vala_ccode_compiler_compile (ccompiler, self->priv->context, vala_compiler_cc_command, (_tmp30 = g_new0 (char*, 1 + 1), _tmp30[0] = NULL, _tmp30));
		} else {
			vala_ccode_compiler_compile (ccompiler, self->priv->context, vala_compiler_cc_command, vala_compiler_cc_options);
		}
		(ccompiler == NULL ? NULL : (ccompiler = (g_object_unref (ccompiler), NULL)));
	}
	return (_tmp31 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), (analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL))), (cfg_builder == NULL ? NULL : (cfg_builder = (g_object_unref (cfg_builder), NULL))), (memory_manager == NULL ? NULL : (memory_manager = (g_object_unref (memory_manager), NULL))), _tmp31);
}


static gboolean vala_compiler_ends_with_dir_separator (const char* s) {
	g_return_val_if_fail (s != NULL, FALSE);
	return G_IS_DIR_SEPARATOR (g_utf8_get_char (g_utf8_offset_to_pointer (s, g_utf8_strlen (s, -1) - 1)));
}


/* ported from glibc */
static char* vala_compiler_realpath (const char* name) {
	char* rpath;
	const char* start;
	const char* end;
	glong root_len;
	g_return_val_if_fail (name != NULL, NULL);
	rpath = NULL;
	/* start of path component*/
	start = NULL;
	/* end of path component*/
	end = NULL;
	if (!g_path_is_absolute (name)) {
		char* _tmp0;
		/* relative path*/
		_tmp0 = NULL;
		rpath = (_tmp0 = g_get_current_dir (), (rpath = (g_free (rpath), NULL)), _tmp0);
		start = end = name;
	} else {
		char* _tmp2;
		char* _tmp1;
		/* set start after root*/
		start = end = g_path_skip_root (name);
		/* extract root*/
		_tmp2 = NULL;
		_tmp1 = NULL;
		rpath = (_tmp2 = (_tmp1 = g_utf8_offset_to_pointer (name, ((glong) 0)), g_strndup (_tmp1, g_utf8_offset_to_pointer (_tmp1, g_utf8_pointer_to_offset (name, start)) - _tmp1)), (rpath = (g_free (rpath), NULL)), _tmp2);
	}
	root_len = g_utf8_pointer_to_offset (rpath, g_path_skip_root (rpath));
	for (; g_utf8_get_char (start) != 0; start = end) {
		glong len;
		while (G_IS_DIR_SEPARATOR (g_utf8_get_char (start))) {
			start = g_utf8_next_char (start);
		}
		/* find end of path component*/
		len = ((glong) 0);
		for (end = start; g_utf8_get_char (end) != 0 && !G_IS_DIR_SEPARATOR (g_utf8_get_char (end)); end = g_utf8_next_char (end)) {
			len++;
		}
		if (len == 0) {
			break;
		} else {
			if (len == 1 && g_utf8_get_char (start) == '.') {
			} else {
				/* do nothing*/
				if (len == 2 && g_str_has_prefix (start, "..")) {
					/* back up to previous component, ignore if at root already*/
					if (g_utf8_strlen (rpath, -1) > root_len) {
						do {
							char* _tmp4;
							char* _tmp3;
							_tmp4 = NULL;
							_tmp3 = NULL;
							rpath = (_tmp4 = (_tmp3 = g_utf8_offset_to_pointer (rpath, ((glong) 0)), g_strndup (_tmp3, g_utf8_offset_to_pointer (_tmp3, g_utf8_strlen (rpath, -1) - 1) - _tmp3)), (rpath = (g_free (rpath), NULL)), _tmp4);
						} while (!vala_compiler_ends_with_dir_separator (rpath));
					}
				} else {
					char* _tmp8;
					char* _tmp7;
					char* _tmp6;
					if (!vala_compiler_ends_with_dir_separator (rpath)) {
						char* _tmp5;
						_tmp5 = NULL;
						rpath = (_tmp5 = g_strconcat (rpath, (G_DIR_SEPARATOR_S), NULL), (rpath = (g_free (rpath), NULL)), _tmp5);
					}
					_tmp8 = NULL;
					_tmp7 = NULL;
					_tmp6 = NULL;
					rpath = (_tmp8 = g_strconcat (rpath, (_tmp7 = ((_tmp6 = g_utf8_offset_to_pointer (start, ((glong) 0)), g_strndup (_tmp6, g_utf8_offset_to_pointer (_tmp6, len) - _tmp6)))), NULL), (rpath = (g_free (rpath), NULL)), _tmp8);
					_tmp7 = (g_free (_tmp7), NULL);
				}
			}
		}
	}
	if (g_utf8_strlen (rpath, -1) > root_len && vala_compiler_ends_with_dir_separator (rpath)) {
		char* _tmp10;
		char* _tmp9;
		_tmp10 = NULL;
		_tmp9 = NULL;
		rpath = (_tmp10 = (_tmp9 = g_utf8_offset_to_pointer (rpath, ((glong) 0)), g_strndup (_tmp9, g_utf8_offset_to_pointer (_tmp9, g_utf8_strlen (rpath, -1) - 1) - _tmp9)), (rpath = (g_free (rpath), NULL)), _tmp10);
	}
	if (G_DIR_SEPARATOR != '/') {
		char** _tmp11;
		gint components_length1;
		char** components;
		char* _tmp12;
		/* don't use backslashes internally,
		 to avoid problems in #include directives*/
		_tmp11 = NULL;
		components = (_tmp11 = g_strsplit (rpath, "\\", 0), components_length1 = -1, _tmp11);
		_tmp12 = NULL;
		rpath = (_tmp12 = g_strjoinv ("/", components), (rpath = (g_free (rpath), NULL)), _tmp12);
		components = (_vala_array_free (components, components_length1, ((GDestroyNotify) g_free)), NULL);
	}
	return rpath;
}


static gint vala_compiler_main (char** args, int args_length1) {
	GError * inner_error;
	ValaCompiler* compiler;
	gint _tmp3;
	inner_error = NULL;
	{
		GOptionContext* opt_context;
		opt_context = g_option_context_new ("- Vala Compiler");
		g_option_context_set_help_enabled (opt_context, TRUE);
		g_option_context_add_main_entries (opt_context, VALA_COMPILER_options, NULL);
		g_option_context_parse (opt_context, &args_length1, &args, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == G_OPTION_ERROR) {
				goto __catch1_g_option_error;
			}
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
		}
		(opt_context == NULL ? NULL : (opt_context = (g_option_context_free (opt_context), NULL)));
	}
	goto __finally1;
	__catch1_g_option_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			fprintf (stdout, "%s\n", e->message);
			fprintf (stdout, "Run '%s --help' to see a full list of available command line options.\n", args[0]);
			return 1;
		}
	}
	__finally1:
	;
	if (vala_compiler_version) {
		fprintf (stdout, "Vala %s\n", PACKAGE_VERSION);
		return 0;
	}
	if (vala_compiler_sources == NULL) {
		fprintf (stderr, "No source file specified.\n");
		return 1;
	}
	compiler = vala_compiler_new ();
	return (_tmp3 = vala_compiler_run (compiler), (compiler == NULL ? NULL : (compiler = (g_object_unref (compiler), NULL))), _tmp3);
}


int main (int argc, char ** argv) {
	g_type_init ();
	return vala_compiler_main (argv, argc);
}


static ValaCompiler* vala_compiler_new (void) {
	ValaCompiler * self;
	self = g_object_newv (VALA_TYPE_COMPILER, 0, NULL);
	return self;
}


static void vala_compiler_class_init (ValaCompilerClass * klass) {
	vala_compiler_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaCompilerPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_compiler_dispose;
}


static void vala_compiler_init (ValaCompiler * self) {
	self->priv = VALA_COMPILER_GET_PRIVATE (self);
}


static void vala_compiler_dispose (GObject * obj) {
	ValaCompiler * self;
	self = VALA_COMPILER (obj);
	(self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL)));
	G_OBJECT_CLASS (vala_compiler_parent_class)->dispose (obj);
}


static GType vala_compiler_get_type (void) {
	static GType vala_compiler_type_id = 0;
	if (G_UNLIKELY (vala_compiler_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCompilerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_compiler_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCompiler), 0, (GInstanceInitFunc) vala_compiler_init };
		vala_compiler_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaCompiler", &g_define_type_info, 0);
	}
	return vala_compiler_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




