/* valatypecheck.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valatypecheck.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>




struct _ValaTypeCheckPrivate {
	ValaExpression* _expression;
	ValaDataType* _data_type;
};

#define VALA_TYPECHECK_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_TYPECHECK, ValaTypeCheckPrivate))
enum  {
	VALA_TYPECHECK_DUMMY_PROPERTY,
	VALA_TYPECHECK_EXPRESSION,
	VALA_TYPECHECK_TYPE_REFERENCE
};
static void vala_typecheck_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_typecheck_real_is_pure (ValaExpression* base);
static void vala_typecheck_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gpointer vala_typecheck_parent_class = NULL;
static void vala_typecheck_dispose (GObject * obj);



/**
 * Creates a new type check expression.
 *
 * @param expr   an expression
 * @param type   a data type
 * @param source reference to source code
 * @return       newly created type check expression
 */
ValaTypeCheck* vala_typecheck_new (ValaExpression* expr, ValaDataType* type, ValaSourceReference* source) {
	GParameter * __params;
	GParameter * __params_it;
	ValaTypeCheck * self;
	g_return_val_if_fail (VALA_IS_EXPRESSION (expr), NULL);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (type), NULL);
	g_return_val_if_fail (VALA_IS_SOURCE_REFERENCE (source), NULL);
	__params = g_new0 (GParameter, 1);
	__params_it = __params;
	__params_it->name = "expression";
	g_value_init (&__params_it->value, VALA_TYPE_EXPRESSION);
	g_value_set_object (&__params_it->value, expr);
	__params_it++;
	self = g_object_newv (VALA_TYPE_TYPECHECK, __params_it - __params, __params);
	vala_typecheck_set_type_reference (self, type);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


static void vala_typecheck_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaTypeCheck * self;
	self = VALA_TYPECHECK (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (self->priv->_expression), visitor);
	vala_code_node_accept (VALA_CODE_NODE (vala_typecheck_get_type_reference (self)), visitor);
	vala_code_visitor_visit_type_check (visitor, self);
}


static gboolean vala_typecheck_real_is_pure (ValaExpression* base) {
	ValaTypeCheck * self;
	self = VALA_TYPECHECK (base);
	return vala_expression_is_pure (self->priv->_expression);
}


static void vala_typecheck_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaTypeCheck * self;
	self = VALA_TYPECHECK (base);
	g_return_if_fail (VALA_IS_DATA_TYPE (old_type));
	g_return_if_fail (VALA_IS_DATA_TYPE (new_type));
	if (vala_typecheck_get_type_reference (self) == old_type) {
		vala_typecheck_set_type_reference (self, new_type);
	}
}


ValaExpression* vala_typecheck_get_expression (ValaTypeCheck* self) {
	g_return_val_if_fail (VALA_IS_TYPECHECK (self), NULL);
	return self->priv->_expression;
}


void vala_typecheck_set_expression (ValaTypeCheck* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_TYPECHECK (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_expression = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_expression == NULL ? NULL : (self->priv->_expression = (g_object_unref (self->priv->_expression), NULL))), _tmp2);
}


ValaDataType* vala_typecheck_get_type_reference (ValaTypeCheck* self) {
	g_return_val_if_fail (VALA_IS_TYPECHECK (self), NULL);
	return self->priv->_data_type;
}


void vala_typecheck_set_type_reference (ValaTypeCheck* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (VALA_IS_TYPECHECK (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_data_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (g_object_unref (self->priv->_data_type), NULL))), _tmp2);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_data_type), VALA_CODE_NODE (self));
}


static void vala_typecheck_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaTypeCheck * self;
	self = VALA_TYPECHECK (object);
	switch (property_id) {
		case VALA_TYPECHECK_EXPRESSION:
		g_value_set_object (value, vala_typecheck_get_expression (self));
		break;
		case VALA_TYPECHECK_TYPE_REFERENCE:
		g_value_set_object (value, vala_typecheck_get_type_reference (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_typecheck_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaTypeCheck * self;
	self = VALA_TYPECHECK (object);
	switch (property_id) {
		case VALA_TYPECHECK_EXPRESSION:
		vala_typecheck_set_expression (self, g_value_get_object (value));
		break;
		case VALA_TYPECHECK_TYPE_REFERENCE:
		vala_typecheck_set_type_reference (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_typecheck_class_init (ValaTypeCheckClass * klass) {
	vala_typecheck_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaTypeCheckPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_typecheck_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_typecheck_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_typecheck_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_typecheck_real_accept;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_typecheck_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_typecheck_real_replace_type;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_TYPECHECK_EXPRESSION, g_param_spec_object ("expression", "expression", "expression", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_TYPECHECK_TYPE_REFERENCE, g_param_spec_object ("type-reference", "type-reference", "type-reference", VALA_TYPE_DATA_TYPE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_typecheck_init (ValaTypeCheck * self) {
	self->priv = VALA_TYPECHECK_GET_PRIVATE (self);
}


static void vala_typecheck_dispose (GObject * obj) {
	ValaTypeCheck * self;
	self = VALA_TYPECHECK (obj);
	(self->priv->_expression == NULL ? NULL : (self->priv->_expression = (g_object_unref (self->priv->_expression), NULL)));
	(self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (g_object_unref (self->priv->_data_type), NULL)));
	G_OBJECT_CLASS (vala_typecheck_parent_class)->dispose (obj);
}


GType vala_typecheck_get_type (void) {
	static GType vala_typecheck_type_id = 0;
	if (G_UNLIKELY (vala_typecheck_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaTypeCheckClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_typecheck_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaTypeCheck), 0, (GInstanceInitFunc) vala_typecheck_init };
		vala_typecheck_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaTypeCheck", &g_define_type_info, 0);
	}
	return vala_typecheck_type_id;
}




