/* valastringliteral.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valastringliteral.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valaexpression.h>




struct _ValaStringLiteralPrivate {
	char* _value;
};

#define VALA_STRING_LITERAL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_STRING_LITERAL, ValaStringLiteralPrivate))
enum  {
	VALA_STRING_LITERAL_DUMMY_PROPERTY,
	VALA_STRING_LITERAL_VALUE
};
static void vala_string_literal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_string_literal_real_is_pure (ValaExpression* base);
static char* vala_string_literal_real_to_string (ValaCodeNode* base);
static gpointer vala_string_literal_parent_class = NULL;
static void vala_string_literal_dispose (GObject * obj);



/**
 * Creates a new string literal.
 *
 * @param s      the literal value
 * @param source reference to source code
 * @return       newly created string literal
 */
ValaStringLiteral* vala_string_literal_new (const char* s, ValaSourceReference* source) {
	ValaStringLiteral * self;
	g_return_val_if_fail (s != NULL, NULL);
	g_return_val_if_fail (VALA_IS_SOURCE_REFERENCE (source), NULL);
	self = g_object_newv (VALA_TYPE_STRING_LITERAL, 0, NULL);
	vala_string_literal_set_value (self, s);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source);
	return self;
}


/**
 * Evaluates the literal string value.
 *
 * @return the unescaped string
 */
char* vala_string_literal_eval (ValaStringLiteral* self) {
	char* noquotes;
	char* _tmp1;
	g_return_val_if_fail (VALA_IS_STRING_LITERAL (self), NULL);
	if (self->priv->_value == NULL) {
		return NULL;
	}
	/* remove quotes */
	noquotes = g_strndup (g_utf8_offset_to_pointer (self->priv->_value, ((glong) (1))), ((gulong) (((guint) ((g_utf8_strlen (self->priv->_value, -1) - 2))))));
	/* unescape string */
	_tmp1 = NULL;
	return (_tmp1 = g_strcompress (noquotes), (noquotes = (g_free (noquotes), NULL)), _tmp1);
}


static void vala_string_literal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaStringLiteral * self;
	self = VALA_STRING_LITERAL (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_string_literal (visitor, self);
	vala_code_visitor_visit_expression (visitor, VALA_EXPRESSION (self));
}


static gboolean vala_string_literal_real_is_pure (ValaExpression* base) {
	ValaStringLiteral * self;
	self = VALA_STRING_LITERAL (base);
	return TRUE;
}


static char* vala_string_literal_real_to_string (ValaCodeNode* base) {
	ValaStringLiteral * self;
	const char* _tmp0;
	self = VALA_STRING_LITERAL (base);
	_tmp0 = NULL;
	return (_tmp0 = self->priv->_value, (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
}


const char* vala_string_literal_get_value (ValaStringLiteral* self) {
	g_return_val_if_fail (VALA_IS_STRING_LITERAL (self), NULL);
	return self->priv->_value;
}


void vala_string_literal_set_value (ValaStringLiteral* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (VALA_IS_STRING_LITERAL (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_value = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_value = (g_free (self->priv->_value), NULL)), _tmp2);
}


static void vala_string_literal_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaStringLiteral * self;
	self = VALA_STRING_LITERAL (object);
	switch (property_id) {
		case VALA_STRING_LITERAL_VALUE:
		g_value_set_string (value, vala_string_literal_get_value (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_string_literal_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaStringLiteral * self;
	self = VALA_STRING_LITERAL (object);
	switch (property_id) {
		case VALA_STRING_LITERAL_VALUE:
		vala_string_literal_set_value (self, g_value_get_string (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_string_literal_class_init (ValaStringLiteralClass * klass) {
	vala_string_literal_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaStringLiteralPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_string_literal_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_string_literal_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_string_literal_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_string_literal_real_accept;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_string_literal_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->to_string = vala_string_literal_real_to_string;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_STRING_LITERAL_VALUE, g_param_spec_string ("value", "value", "value", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_string_literal_instance_init (ValaStringLiteral * self) {
	self->priv = VALA_STRING_LITERAL_GET_PRIVATE (self);
}


static void vala_string_literal_dispose (GObject * obj) {
	ValaStringLiteral * self;
	self = VALA_STRING_LITERAL (obj);
	self->priv->_value = (g_free (self->priv->_value), NULL);
	G_OBJECT_CLASS (vala_string_literal_parent_class)->dispose (obj);
}


GType vala_string_literal_get_type (void) {
	static GType vala_string_literal_type_id = 0;
	if (G_UNLIKELY (vala_string_literal_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaStringLiteralClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_string_literal_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaStringLiteral), 0, (GInstanceInitFunc) vala_string_literal_instance_init };
		vala_string_literal_type_id = g_type_register_static (VALA_TYPE_LITERAL, "ValaStringLiteral", &g_define_type_info, 0);
	}
	return vala_string_literal_type_id;
}




