/* valaconstant.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaconstant.h>
#include <vala/valadatatype.h>
#include <vala/valaexpression.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>




struct _ValaConstantPrivate {
	char* cname;
	gboolean lock_used;
	ValaDataType* _data_type;
	ValaExpression* _initializer;
};

#define VALA_CONSTANT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CONSTANT, ValaConstantPrivate))
enum  {
	VALA_CONSTANT_DUMMY_PROPERTY
};
static void vala_constant_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_constant_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_constant_real_get_lock_used (ValaLockable* base);
static void vala_constant_real_set_lock_used (ValaLockable* base, gboolean used);
static void vala_constant_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static void vala_constant_process_ccode_attribute (ValaConstant* self, ValaAttribute* a);
static gpointer vala_constant_parent_class = NULL;
static ValaLockableIface* vala_constant_vala_lockable_parent_iface = NULL;
static void vala_constant_finalize (ValaCodeNode* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new constant.
 *
 * @param name             constant name
 * @param type_reference   constant type
 * @param initializer      constant value
 * @param source_reference reference to source code
 * @return                 newly created constant
 */
ValaConstant* vala_constant_construct (GType object_type, const char* name, ValaDataType* type_reference, ValaExpression* initializer, ValaSourceReference* source_reference) {
	ValaConstant* self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (type_reference), NULL);
	g_return_val_if_fail (initializer == NULL || VALA_IS_EXPRESSION (initializer), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = ((ValaConstant*) (g_type_create_instance (object_type)));
	vala_symbol_init (VALA_SYMBOL (VALA_MEMBER (self)));
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_constant_set_type_reference (self, type_reference);
	vala_constant_set_initializer (self, initializer);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


ValaConstant* vala_constant_new (const char* name, ValaDataType* type_reference, ValaExpression* initializer, ValaSourceReference* source_reference) {
	return vala_constant_construct (VALA_TYPE_CONSTANT, name, type_reference, initializer, source_reference);
}


static void vala_constant_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaConstant * self;
	self = VALA_CONSTANT (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_member (visitor, VALA_MEMBER (self));
	vala_code_visitor_visit_constant (visitor, self);
}


static void vala_constant_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaConstant * self;
	self = VALA_CONSTANT (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (vala_constant_get_type_reference (self)), visitor);
	if (vala_constant_get_initializer (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_constant_get_initializer (self)), visitor);
	}
}


/**
 * Returns the name of this constant as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_constant_get_cname (ValaConstant* self) {
	const char* _tmp1;
	g_return_val_if_fail (VALA_IS_CONSTANT (self), NULL);
	if (self->priv->cname == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->cname = (_tmp0 = vala_constant_get_default_cname (self), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->cname, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
}


/**
 * Returns the default name of this constant as it is used in C
 * code.
 *
 * @return the name to be used in C code by default
 */
char* vala_constant_get_default_cname (ValaConstant* self) {
	g_return_val_if_fail (VALA_IS_CONSTANT (self), NULL);
	if (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)) == NULL) {
		const char* _tmp0;
		/* global constant*/
		_tmp0 = NULL;
		return (_tmp0 = vala_symbol_get_name (VALA_SYMBOL (self)), (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
	} else {
		char* _tmp3;
		char* _tmp2;
		char* _tmp4;
		_tmp3 = NULL;
		_tmp2 = NULL;
		_tmp4 = NULL;
		return (_tmp4 = g_strdup_printf ("%s%s", (_tmp3 = g_utf8_strup ((_tmp2 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))), -1)), vala_symbol_get_name (VALA_SYMBOL (self))), (_tmp3 = (g_free (_tmp3), NULL)), (_tmp2 = (g_free (_tmp2), NULL)), _tmp4);
	}
}


static gboolean vala_constant_real_get_lock_used (ValaLockable* base) {
	ValaConstant * self;
	self = VALA_CONSTANT (base);
	return self->priv->lock_used;
}


static void vala_constant_real_set_lock_used (ValaLockable* base, gboolean used) {
	ValaConstant * self;
	self = VALA_CONSTANT (base);
	self->priv->lock_used = used;
}


static void vala_constant_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaConstant * self;
	self = VALA_CONSTANT (base);
	g_return_if_fail (VALA_IS_DATA_TYPE (old_type));
	g_return_if_fail (VALA_IS_DATA_TYPE (new_type));
	if (vala_constant_get_type_reference (self) == old_type) {
		vala_constant_set_type_reference (self, new_type);
	}
}


static void vala_constant_process_ccode_attribute (ValaConstant* self, ValaAttribute* a) {
	g_return_if_fail (VALA_IS_CONSTANT (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cname")) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->cname = (_tmp0 = vala_attribute_get_string (a, "cname"), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp0);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			int filename_collection_length1;
			char** filename_it;
			filename_collection = g_strsplit (val, ",", 0);
			filename_collection_length1 = -1;
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* _tmp1;
				char* filename;
				_tmp1 = NULL;
				filename = (_tmp1 = *filename_it, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
				{
					vala_member_add_cheader_filename (VALA_MEMBER (self), filename);
					filename = (g_free (filename), NULL);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, ((GDestroyNotify) (g_free))), NULL);
		}
		val = (g_free (val), NULL);
	}
}


/**
 * Process all associated attributes.
 */
void vala_constant_process_attributes (ValaConstant* self) {
	g_return_if_fail (VALA_IS_CONSTANT (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = VALA_CODE_NODE (self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = ((ValaAttribute*) (a_it->data)), (_tmp0 == NULL ? NULL : vala_code_node_ref (_tmp0)));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_constant_process_ccode_attribute (self, a);
				}
				(a == NULL ? NULL : (a = (vala_code_node_unref (a), NULL)));
			}
		}
	}
}


ValaDataType* vala_constant_get_type_reference (ValaConstant* self) {
	g_return_val_if_fail (VALA_IS_CONSTANT (self), NULL);
	return self->priv->_data_type;
}


void vala_constant_set_type_reference (ValaConstant* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (VALA_IS_CONSTANT (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_data_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL))), _tmp2);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_data_type), VALA_CODE_NODE (self));
}


ValaExpression* vala_constant_get_initializer (ValaConstant* self) {
	g_return_val_if_fail (VALA_IS_CONSTANT (self), NULL);
	return self->priv->_initializer;
}


void vala_constant_set_initializer (ValaConstant* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_CONSTANT (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_initializer = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_initializer == NULL ? NULL : (self->priv->_initializer = (vala_code_node_unref (self->priv->_initializer), NULL))), _tmp2);
	if (self->priv->_initializer != NULL) {
		vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_initializer), VALA_CODE_NODE (self));
	}
}


static void vala_constant_class_init (ValaConstantClass * klass) {
	vala_constant_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_constant_finalize;
	g_type_class_add_private (klass, sizeof (ValaConstantPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_constant_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_constant_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_constant_real_replace_type;
}


static void vala_constant_vala_lockable_interface_init (ValaLockableIface * iface) {
	vala_constant_vala_lockable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_lock_used = vala_constant_real_get_lock_used;
	iface->set_lock_used = vala_constant_real_set_lock_used;
}


static void vala_constant_instance_init (ValaConstant * self) {
	self->priv = VALA_CONSTANT_GET_PRIVATE (self);
	self->priv->lock_used = FALSE;
}


static void vala_constant_finalize (ValaCodeNode* obj) {
	ValaConstant * self;
	self = VALA_CONSTANT (obj);
	self->priv->cname = (g_free (self->priv->cname), NULL);
	(self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL)));
	(self->priv->_initializer == NULL ? NULL : (self->priv->_initializer = (vala_code_node_unref (self->priv->_initializer), NULL)));
	VALA_CODE_NODE_CLASS (vala_constant_parent_class)->finalize (obj);
}


GType vala_constant_get_type (void) {
	static GType vala_constant_type_id = 0;
	if (vala_constant_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaConstantClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_constant_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaConstant), 0, (GInstanceInitFunc) vala_constant_instance_init, NULL };
		static const GInterfaceInfo vala_lockable_info = { (GInterfaceInitFunc) vala_constant_vala_lockable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_constant_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaConstant", &g_define_type_info, 0);
		g_type_add_interface_static (vala_constant_type_id, VALA_TYPE_LOCKABLE, &vala_lockable_info);
	}
	return vala_constant_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) (array))[i] != NULL)
			destroy_func (((gpointer*) (array))[i]);
		}
		else
		for (i = 0; ((gpointer*) (array))[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) (array))[i]);
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




