/* valaexpression.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaexpression.h>
#include <vala/valadatatype.h>
#include <vala/valasymbol.h>
#include <vala/valalocalvariable.h>




struct _ValaExpressionPrivate {
	ValaDataType* _value_type;
	ValaDataType* _target_type;
	ValaSymbol* _symbol_reference;
	gboolean _lvalue;
};

#define VALA_EXPRESSION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_EXPRESSION, ValaExpressionPrivate))
enum  {
	VALA_EXPRESSION_DUMMY_PROPERTY
};
static gboolean vala_expression_real_is_constant (ValaExpression* self);
static gboolean vala_expression_real_is_pure (ValaExpression* self);
static gboolean vala_expression_real_is_non_null (ValaExpression* self);
static gpointer vala_expression_parent_class = NULL;
static void vala_expression_finalize (ValaCodeNode* obj);



/**
 * Returns whether this expression is constant, i.e. whether this
 * expression only consists of literals and other constants.
 */
static gboolean vala_expression_real_is_constant (ValaExpression* self) {
	g_return_val_if_fail (VALA_IS_EXPRESSION (self), FALSE);
	return FALSE;
}


gboolean vala_expression_is_constant (ValaExpression* self) {
	return VALA_EXPRESSION_GET_CLASS (self)->is_constant (self);
}


static gboolean vala_expression_real_is_pure (ValaExpression* self) {
	g_return_val_if_fail (VALA_IS_EXPRESSION (self), FALSE);
	g_critical ("Type `%s' does not implement abstract method `vala_expression_is_pure'", g_type_name (G_TYPE_FROM_INSTANCE (self)));
	return FALSE;
}


/**
 * Returns whether this expression is pure, i.e. whether this expression
 * is free of side-effects.
 */
gboolean vala_expression_is_pure (ValaExpression* self) {
	return VALA_EXPRESSION_GET_CLASS (self)->is_pure (self);
}


/**
 * Returns whether this expression is guaranteed to be non-null.
 */
static gboolean vala_expression_real_is_non_null (ValaExpression* self) {
	g_return_val_if_fail (VALA_IS_EXPRESSION (self), FALSE);
	return FALSE;
}


gboolean vala_expression_is_non_null (ValaExpression* self) {
	return VALA_EXPRESSION_GET_CLASS (self)->is_non_null (self);
}


ValaDataType* vala_expression_get_value_type (ValaExpression* self) {
	g_return_val_if_fail (VALA_IS_EXPRESSION (self), NULL);
	return self->priv->_value_type;
}


void vala_expression_set_value_type (ValaExpression* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (VALA_IS_EXPRESSION (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_value_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_value_type == NULL ? NULL : (self->priv->_value_type = (vala_code_node_unref (self->priv->_value_type), NULL))), _tmp2);
}


ValaDataType* vala_expression_get_target_type (ValaExpression* self) {
	g_return_val_if_fail (VALA_IS_EXPRESSION (self), NULL);
	return self->priv->_target_type;
}


void vala_expression_set_target_type (ValaExpression* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (VALA_IS_EXPRESSION (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_target_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_target_type == NULL ? NULL : (self->priv->_target_type = (vala_code_node_unref (self->priv->_target_type), NULL))), _tmp2);
}


ValaSymbol* vala_expression_get_symbol_reference (ValaExpression* self) {
	g_return_val_if_fail (VALA_IS_EXPRESSION (self), NULL);
	return self->priv->_symbol_reference;
}


void vala_expression_set_symbol_reference (ValaExpression* self, ValaSymbol* value) {
	g_return_if_fail (VALA_IS_EXPRESSION (self));
	self->priv->_symbol_reference = value;
}


gboolean vala_expression_get_lvalue (ValaExpression* self) {
	g_return_val_if_fail (VALA_IS_EXPRESSION (self), FALSE);
	return self->priv->_lvalue;
}


void vala_expression_set_lvalue (ValaExpression* self, gboolean value) {
	g_return_if_fail (VALA_IS_EXPRESSION (self));
	self->priv->_lvalue = value;
}


static void vala_expression_class_init (ValaExpressionClass * klass) {
	vala_expression_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_expression_finalize;
	g_type_class_add_private (klass, sizeof (ValaExpressionPrivate));
	VALA_EXPRESSION_CLASS (klass)->is_constant = vala_expression_real_is_constant;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_expression_real_is_pure;
	VALA_EXPRESSION_CLASS (klass)->is_non_null = vala_expression_real_is_non_null;
}


static void vala_expression_instance_init (ValaExpression * self) {
	self->priv = VALA_EXPRESSION_GET_PRIVATE (self);
	self->temp_vars = gee_array_list_new (VALA_TYPE_LOCAL_VARIABLE, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal);
}


static void vala_expression_finalize (ValaCodeNode* obj) {
	ValaExpression * self;
	self = VALA_EXPRESSION (obj);
	(self->priv->_value_type == NULL ? NULL : (self->priv->_value_type = (vala_code_node_unref (self->priv->_value_type), NULL)));
	(self->priv->_target_type == NULL ? NULL : (self->priv->_target_type = (vala_code_node_unref (self->priv->_target_type), NULL)));
	(self->temp_vars == NULL ? NULL : (self->temp_vars = (gee_collection_object_unref (self->temp_vars), NULL)));
	VALA_CODE_NODE_CLASS (vala_expression_parent_class)->finalize (obj);
}


GType vala_expression_get_type (void) {
	static GType vala_expression_type_id = 0;
	if (vala_expression_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaExpressionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_expression_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaExpression), 0, (GInstanceInitFunc) vala_expression_instance_init, NULL };
		vala_expression_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaExpression", &g_define_type_info, G_TYPE_FLAG_ABSTRACT);
	}
	return vala_expression_type_id;
}




