/* valaformalparameter.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valaformalparameter.h>
#include <vala/valaexpression.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>




struct _ValaFormalParameterPrivate {
	ValaParameterDirection _direction;
	gboolean _ellipsis;
	ValaExpression* _default_expression;
	gboolean _no_array_length;
	gboolean _construct_parameter;
	double _cparameter_position;
	double _carray_length_parameter_position;
	double _cdelegate_target_parameter_position;
	char* _ctype;
	ValaDataType* _data_type;
};

#define VALA_FORMAL_PARAMETER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterPrivate))
enum  {
	VALA_FORMAL_PARAMETER_DUMMY_PROPERTY
};
static void vala_formal_parameter_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_formal_parameter_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_formal_parameter_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static void vala_formal_parameter_process_ccode_attribute (ValaFormalParameter* self, ValaAttribute* a);
static gpointer vala_formal_parameter_parent_class = NULL;
static void vala_formal_parameter_finalize (ValaCodeNode* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new formal parameter.
 *
 * @param name   parameter name
 * @param type   parameter type
 * @param source reference to source code
 * @return       newly created formal parameter
 */
ValaFormalParameter* vala_formal_parameter_construct (GType object_type, const char* name, ValaDataType* parameter_type, ValaSourceReference* source_reference) {
	ValaFormalParameter* self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (parameter_type), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = ((ValaFormalParameter*) (g_type_create_instance (object_type)));
	vala_symbol_init (VALA_SYMBOL (self));
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_formal_parameter_set_parameter_type (self, parameter_type);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	vala_symbol_set_access (VALA_SYMBOL (self), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	return self;
}


ValaFormalParameter* vala_formal_parameter_new (const char* name, ValaDataType* parameter_type, ValaSourceReference* source_reference) {
	return vala_formal_parameter_construct (VALA_TYPE_FORMAL_PARAMETER, name, parameter_type, source_reference);
}


/**
 * Creates a new ellipsis parameter representing an indefinite number of
 * parameters.
 */
ValaFormalParameter* vala_formal_parameter_construct_with_ellipsis (GType object_type, ValaSourceReference* source) {
	ValaFormalParameter* self;
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	self = ((ValaFormalParameter*) (g_type_create_instance (object_type)));
	vala_symbol_init (VALA_SYMBOL (self));
	vala_formal_parameter_set_ellipsis (self, TRUE);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source);
	vala_symbol_set_access (VALA_SYMBOL (self), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	return self;
}


ValaFormalParameter* vala_formal_parameter_new_with_ellipsis (ValaSourceReference* source) {
	return vala_formal_parameter_construct_with_ellipsis (VALA_TYPE_FORMAL_PARAMETER, source);
}


static void vala_formal_parameter_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaFormalParameter * self;
	self = VALA_FORMAL_PARAMETER (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_formal_parameter (visitor, self);
}


static void vala_formal_parameter_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaFormalParameter * self;
	self = VALA_FORMAL_PARAMETER (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	if (!self->priv->_ellipsis) {
		vala_code_node_accept (VALA_CODE_NODE (vala_formal_parameter_get_parameter_type (self)), visitor);
		if (self->priv->_default_expression != NULL) {
			vala_code_node_accept (VALA_CODE_NODE (self->priv->_default_expression), visitor);
		}
	}
}


static void vala_formal_parameter_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaFormalParameter * self;
	self = VALA_FORMAL_PARAMETER (base);
	g_return_if_fail (VALA_IS_DATA_TYPE (old_type));
	g_return_if_fail (VALA_IS_DATA_TYPE (new_type));
	if (vala_formal_parameter_get_parameter_type (self) == old_type) {
		vala_formal_parameter_set_parameter_type (self, new_type);
	}
}


static void vala_formal_parameter_process_ccode_attribute (ValaFormalParameter* self, ValaAttribute* a) {
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "type")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_formal_parameter_set_ctype (self, (_tmp0 = vala_attribute_get_string (a, "type")));
		_tmp0 = (g_free (_tmp0), NULL);
	}
	if (vala_attribute_has_argument (a, "array_length_pos")) {
		vala_formal_parameter_set_carray_length_parameter_position (self, vala_attribute_get_double (a, "array_length_pos"));
	}
	if (vala_attribute_has_argument (a, "delegate_target_pos")) {
		vala_formal_parameter_set_cdelegate_target_parameter_position (self, vala_attribute_get_double (a, "delegate_target_pos"));
	}
}


/**
 * Process all associated attributes.
 */
void vala_formal_parameter_process_attributes (ValaFormalParameter* self) {
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = VALA_CODE_NODE (self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = ((ValaAttribute*) (a_it->data)), (_tmp0 == NULL ? NULL : vala_code_node_ref (_tmp0)));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_formal_parameter_process_ccode_attribute (self, a);
				}
				(a == NULL ? NULL : (a = (vala_code_node_unref (a), NULL)));
			}
		}
	}
}


ValaFormalParameter* vala_formal_parameter_copy (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), NULL);
	if (!self->priv->_ellipsis) {
		ValaFormalParameter* result;
		result = vala_formal_parameter_new (vala_symbol_get_name (VALA_SYMBOL (self)), vala_formal_parameter_get_parameter_type (self), vala_code_node_get_source_reference (VALA_CODE_NODE (self)));
		return result;
	} else {
		return vala_formal_parameter_new_with_ellipsis (NULL);
	}
}


ValaDataType* vala_formal_parameter_get_parameter_type (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), NULL);
	return self->priv->_data_type;
}


void vala_formal_parameter_set_parameter_type (ValaFormalParameter* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_data_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL))), _tmp2);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_data_type), VALA_CODE_NODE (self));
}


ValaParameterDirection vala_formal_parameter_get_direction (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), 0);
	return self->priv->_direction;
}


void vala_formal_parameter_set_direction (ValaFormalParameter* self, ValaParameterDirection value) {
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	self->priv->_direction = value;
}


gboolean vala_formal_parameter_get_ellipsis (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), FALSE);
	return self->priv->_ellipsis;
}


void vala_formal_parameter_set_ellipsis (ValaFormalParameter* self, gboolean value) {
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	self->priv->_ellipsis = value;
}


ValaExpression* vala_formal_parameter_get_default_expression (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), NULL);
	return self->priv->_default_expression;
}


void vala_formal_parameter_set_default_expression (ValaFormalParameter* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_default_expression = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_default_expression == NULL ? NULL : (self->priv->_default_expression = (vala_code_node_unref (self->priv->_default_expression), NULL))), _tmp2);
}


gboolean vala_formal_parameter_get_no_array_length (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), FALSE);
	return self->priv->_no_array_length;
}


void vala_formal_parameter_set_no_array_length (ValaFormalParameter* self, gboolean value) {
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	self->priv->_no_array_length = value;
}


gboolean vala_formal_parameter_get_construct_parameter (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), FALSE);
	return self->priv->_construct_parameter;
}


void vala_formal_parameter_set_construct_parameter (ValaFormalParameter* self, gboolean value) {
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	self->priv->_construct_parameter = value;
}


double vala_formal_parameter_get_cparameter_position (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), 0.0);
	return self->priv->_cparameter_position;
}


void vala_formal_parameter_set_cparameter_position (ValaFormalParameter* self, double value) {
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	self->priv->_cparameter_position = value;
}


double vala_formal_parameter_get_carray_length_parameter_position (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), 0.0);
	return self->priv->_carray_length_parameter_position;
}


void vala_formal_parameter_set_carray_length_parameter_position (ValaFormalParameter* self, double value) {
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	self->priv->_carray_length_parameter_position = value;
}


double vala_formal_parameter_get_cdelegate_target_parameter_position (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), 0.0);
	return self->priv->_cdelegate_target_parameter_position;
}


void vala_formal_parameter_set_cdelegate_target_parameter_position (ValaFormalParameter* self, double value) {
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	self->priv->_cdelegate_target_parameter_position = value;
}


const char* vala_formal_parameter_get_ctype (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), NULL);
	return self->priv->_ctype;
}


void vala_formal_parameter_set_ctype (ValaFormalParameter* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_ctype = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_ctype = (g_free (self->priv->_ctype), NULL)), _tmp2);
}


static void vala_formal_parameter_class_init (ValaFormalParameterClass * klass) {
	vala_formal_parameter_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_formal_parameter_finalize;
	g_type_class_add_private (klass, sizeof (ValaFormalParameterPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_formal_parameter_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_formal_parameter_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_formal_parameter_real_replace_type;
}


static void vala_formal_parameter_instance_init (ValaFormalParameter * self) {
	self->priv = VALA_FORMAL_PARAMETER_GET_PRIVATE (self);
	self->priv->_direction = VALA_PARAMETER_DIRECTION_IN;
}


static void vala_formal_parameter_finalize (ValaCodeNode* obj) {
	ValaFormalParameter * self;
	self = VALA_FORMAL_PARAMETER (obj);
	(self->priv->_default_expression == NULL ? NULL : (self->priv->_default_expression = (vala_code_node_unref (self->priv->_default_expression), NULL)));
	self->priv->_ctype = (g_free (self->priv->_ctype), NULL);
	(self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL)));
	VALA_CODE_NODE_CLASS (vala_formal_parameter_parent_class)->finalize (obj);
}


GType vala_formal_parameter_get_type (void) {
	static GType vala_formal_parameter_type_id = 0;
	if (vala_formal_parameter_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaFormalParameterClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_formal_parameter_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaFormalParameter), 0, (GInstanceInitFunc) vala_formal_parameter_instance_init, NULL };
		vala_formal_parameter_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaFormalParameter", &g_define_type_info, 0);
	}
	return vala_formal_parameter_type_id;
}



GType vala_parameter_direction_get_type (void) {
	static GType vala_parameter_direction_type_id = 0;
	if (G_UNLIKELY (vala_parameter_direction_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_PARAMETER_DIRECTION_IN, "VALA_PARAMETER_DIRECTION_IN", "in"}, {VALA_PARAMETER_DIRECTION_OUT, "VALA_PARAMETER_DIRECTION_OUT", "out"}, {VALA_PARAMETER_DIRECTION_REF, "VALA_PARAMETER_DIRECTION_REF", "ref"}, {0, NULL, NULL}};
		vala_parameter_direction_type_id = g_enum_register_static ("ValaParameterDirection", values);
	}
	return vala_parameter_direction_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




