/* valamethod.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valamethod.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valabasicblock.h>
#include <vala/valaformalparameter.h>
#include <vala/valalocalvariable.h>
#include <vala/valaexpression.h>
#include <vala/valadatatype.h>
#include <vala/valablock.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>
#include <vala/valacodegenerator.h>
#include <vala/valaclass.h>
#include <vala/valacreationmethod.h>
#include <vala/valainterface.h>
#include <vala/valareport.h>
#include <vala/valasignal.h>




struct _ValaMethodPrivate {
	ValaBasicBlock* _entry_block;
	ValaBasicBlock* _exit_block;
	MemberBinding _binding;
	gboolean _is_abstract;
	gboolean _is_virtual;
	gboolean _overrides;
	gboolean _is_inline;
	gboolean _returns_modified_pointer;
	ValaFormalParameter* _this_parameter;
	ValaLocalVariable* _result_var;
	double _cinstance_parameter_position;
	double _carray_length_parameter_position;
	double _cdelegate_target_parameter_position;
	gboolean _printf_format;
	gboolean _has_construct_function;
	GeeList* parameters;
	char* cname;
	char* _vfunc_name;
	char* _sentinel;
	gboolean _no_array_length;
	GeeList* preconditions;
	GeeList* postconditions;
	ValaDataType* _return_type;
	ValaBlock* _body;
	ValaMethod* _base_method;
	ValaMethod* _base_interface_method;
	gboolean base_methods_valid;
};

#define VALA_METHOD_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_METHOD, ValaMethodPrivate))
enum  {
	VALA_METHOD_DUMMY_PROPERTY
};
static void vala_method_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_method_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_method_real_get_default_cname (ValaMethod* self);
static char* vala_method_real_get_real_cname (ValaMethod* self);
static void vala_method_process_ccode_attribute (ValaMethod* self, ValaAttribute* a);
static void vala_method_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static ValaCodeBinding* vala_method_real_create_code_binding (ValaCodeNode* base, ValaCodeGenerator* codegen);
static void vala_method_find_base_methods (ValaMethod* self);
static void vala_method_find_base_class_method (ValaMethod* self, ValaClass* cl);
static void vala_method_find_base_interface_method (ValaMethod* self, ValaClass* cl);
static gpointer vala_method_parent_class = NULL;
static void vala_method_finalize (ValaCodeNode* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new method.
 *
 * @param name        method name
 * @param return_type method return type
 * @param source      reference to source code
 * @return            newly created method
 */
ValaMethod* vala_method_construct (GType object_type, const char* name, ValaDataType* return_type, ValaSourceReference* source_reference) {
	ValaMethod* self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (return_type), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = ((ValaMethod*) (g_type_create_instance (object_type)));
	vala_symbol_init (VALA_SYMBOL (VALA_MEMBER (self)));
	vala_method_set_return_type (self, return_type);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_method_set_carray_length_parameter_position (self, ((double) (-3)));
	vala_method_set_cdelegate_target_parameter_position (self, ((double) (-3)));
	return self;
}


ValaMethod* vala_method_new (const char* name, ValaDataType* return_type, ValaSourceReference* source_reference) {
	return vala_method_construct (VALA_TYPE_METHOD, name, return_type, source_reference);
}


/**
 * Appends parameter to this method.
 *
 * @param param a formal parameter
 */
void vala_method_add_parameter (ValaMethod* self, ValaFormalParameter* param) {
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (param));
	if (vala_method_get_no_array_length (self)) {
		vala_formal_parameter_set_no_array_length (param, TRUE);
	}
	/* default C parameter position*/
	vala_formal_parameter_set_cparameter_position (param, ((double) (gee_collection_get_size (GEE_COLLECTION (self->priv->parameters)) + 1)));
	vala_formal_parameter_set_carray_length_parameter_position (param, vala_formal_parameter_get_cparameter_position (param) + 0.1);
	vala_formal_parameter_set_cdelegate_target_parameter_position (param, vala_formal_parameter_get_cparameter_position (param) + 0.1);
	gee_collection_add (GEE_COLLECTION (self->priv->parameters), param);
	if (!vala_formal_parameter_get_ellipsis (param)) {
		vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (param)), VALA_SYMBOL (param));
	}
}


GeeList* vala_method_get_parameters (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_FORMAL_PARAMETER, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->parameters));
}


/**
 * Remove all parameters from this method.
 */
void vala_method_clear_parameters (ValaMethod* self) {
	g_return_if_fail (VALA_IS_METHOD (self));
	{
		GeeList* param_collection;
		int param_it;
		param_collection = self->priv->parameters;
		for (param_it = 0; param_it < gee_collection_get_size (GEE_COLLECTION (param_collection)); param_it = param_it + 1) {
			ValaFormalParameter* param;
			param = ((ValaFormalParameter*) (gee_list_get (GEE_LIST (param_collection), param_it)));
			{
				if (!vala_formal_parameter_get_ellipsis (param)) {
					vala_scope_remove (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (param)));
				}
				(param == NULL ? NULL : (param = (vala_code_node_unref (param), NULL)));
			}
		}
	}
	gee_collection_clear (GEE_COLLECTION (self->priv->parameters));
}


static void vala_method_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaMethod * self;
	self = VALA_METHOD (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_method (visitor, self);
}


static void vala_method_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaMethod * self;
	self = VALA_METHOD (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	if (vala_method_get_return_type (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_method_get_return_type (self)), visitor);
	}
	{
		GeeList* param_collection;
		int param_it;
		param_collection = self->priv->parameters;
		for (param_it = 0; param_it < gee_collection_get_size (GEE_COLLECTION (param_collection)); param_it = param_it + 1) {
			ValaFormalParameter* param;
			param = ((ValaFormalParameter*) (gee_list_get (GEE_LIST (param_collection), param_it)));
			{
				vala_code_node_accept (VALA_CODE_NODE (param), visitor);
				(param == NULL ? NULL : (param = (vala_code_node_unref (param), NULL)));
			}
		}
	}
	{
		GeeList* error_type_collection;
		int error_type_it;
		error_type_collection = vala_code_node_get_error_types (VALA_CODE_NODE (self));
		for (error_type_it = 0; error_type_it < gee_collection_get_size (GEE_COLLECTION (error_type_collection)); error_type_it = error_type_it + 1) {
			ValaDataType* error_type;
			error_type = ((ValaDataType*) (gee_list_get (GEE_LIST (error_type_collection), error_type_it)));
			{
				vala_code_node_accept (VALA_CODE_NODE (error_type), visitor);
				(error_type == NULL ? NULL : (error_type = (vala_code_node_unref (error_type), NULL)));
			}
		}
		(error_type_collection == NULL ? NULL : (error_type_collection = (gee_collection_object_unref (error_type_collection), NULL)));
	}
	if (self->priv->_result_var != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_local_variable_get_variable_type (self->priv->_result_var)), visitor);
	}
	{
		GeeList* precondition_collection;
		int precondition_it;
		precondition_collection = self->priv->preconditions;
		for (precondition_it = 0; precondition_it < gee_collection_get_size (GEE_COLLECTION (precondition_collection)); precondition_it = precondition_it + 1) {
			ValaExpression* precondition;
			precondition = ((ValaExpression*) (gee_list_get (GEE_LIST (precondition_collection), precondition_it)));
			{
				vala_code_node_accept (VALA_CODE_NODE (precondition), visitor);
				(precondition == NULL ? NULL : (precondition = (vala_code_node_unref (precondition), NULL)));
			}
		}
	}
	{
		GeeList* postcondition_collection;
		int postcondition_it;
		postcondition_collection = self->priv->postconditions;
		for (postcondition_it = 0; postcondition_it < gee_collection_get_size (GEE_COLLECTION (postcondition_collection)); postcondition_it = postcondition_it + 1) {
			ValaExpression* postcondition;
			postcondition = ((ValaExpression*) (gee_list_get (GEE_LIST (postcondition_collection), postcondition_it)));
			{
				vala_code_node_accept (VALA_CODE_NODE (postcondition), visitor);
				(postcondition == NULL ? NULL : (postcondition = (vala_code_node_unref (postcondition), NULL)));
			}
		}
	}
	if (vala_method_get_body (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_method_get_body (self)), visitor);
	}
}


/**
 * Returns the interface name of this method as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_method_get_cname (ValaMethod* self) {
	const char* _tmp1;
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (self->priv->cname == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->cname = (_tmp0 = vala_method_get_default_cname (self), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->cname, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
}


/**
 * Returns the default interface name of this method as it is used in C
 * code.
 *
 * @return the name to be used in C code by default
 */
static char* vala_method_real_get_default_cname (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (_vala_strcmp0 (vala_symbol_get_name (VALA_SYMBOL (self)), "main") == 0 && vala_symbol_get_name (vala_symbol_get_parent_symbol (VALA_SYMBOL (self))) == NULL) {
		/* avoid conflict with generated main function*/
		return g_strdup ("_main");
	} else {
		if (g_str_has_prefix (vala_symbol_get_name (VALA_SYMBOL (self)), "_")) {
			char* _tmp1;
			char* _tmp2;
			_tmp1 = NULL;
			_tmp2 = NULL;
			return (_tmp2 = g_strdup_printf ("_%s%s", (_tmp1 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))), g_utf8_offset_to_pointer (vala_symbol_get_name (VALA_SYMBOL (self)), ((glong) (1)))), (_tmp1 = (g_free (_tmp1), NULL)), _tmp2);
		} else {
			char* _tmp4;
			char* _tmp5;
			_tmp4 = NULL;
			_tmp5 = NULL;
			return (_tmp5 = g_strdup_printf ("%s%s", (_tmp4 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))), vala_symbol_get_name (VALA_SYMBOL (self))), (_tmp4 = (g_free (_tmp4), NULL)), _tmp5);
		}
	}
}


char* vala_method_get_default_cname (ValaMethod* self) {
	return VALA_METHOD_GET_CLASS (self)->get_default_cname (self);
}


/**
 * Returns the implementation name of this data type as it is used in C
 * code.
 *
 * @return the name to be used in C code
 */
static char* vala_method_real_get_real_cname (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (vala_method_get_base_method (self) != NULL || vala_method_get_base_interface_method (self) != NULL) {
		char* _tmp0;
		char* _tmp1;
		_tmp0 = NULL;
		_tmp1 = NULL;
		return (_tmp1 = g_strdup_printf ("%s_real_%s", (_tmp0 = vala_symbol_get_lower_case_cname (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)), NULL)), vala_symbol_get_name (VALA_SYMBOL (self))), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
	} else {
		return vala_method_get_cname (self);
	}
}


char* vala_method_get_real_cname (ValaMethod* self) {
	return VALA_METHOD_GET_CLASS (self)->get_real_cname (self);
}


/**
 * Sets the name of this method as it is used in C code.
 *
 * @param cname the name to be used in C code
 */
void vala_method_set_cname (ValaMethod* self, const char* cname) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (cname != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cname = (_tmp1 = (_tmp0 = cname, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp1);
}


static void vala_method_process_ccode_attribute (ValaMethod* self, ValaAttribute* a) {
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cname")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_method_set_cname (self, (_tmp0 = vala_attribute_get_string (a, "cname")));
		_tmp0 = (g_free (_tmp0), NULL);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			int filename_collection_length1;
			char** filename_it;
			filename_collection = g_strsplit (val, ",", 0);
			filename_collection_length1 = -1;
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* _tmp1;
				char* filename;
				_tmp1 = NULL;
				filename = (_tmp1 = *filename_it, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
				{
					vala_member_add_cheader_filename (VALA_MEMBER (self), filename);
					filename = (g_free (filename), NULL);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, ((GDestroyNotify) (g_free))), NULL);
		}
		val = (g_free (val), NULL);
	}
	if (vala_attribute_has_argument (a, "vfunc_name")) {
		char* _tmp2;
		_tmp2 = NULL;
		vala_method_set_vfunc_name (self, (_tmp2 = vala_attribute_get_string (a, "vfunc_name")));
		_tmp2 = (g_free (_tmp2), NULL);
	}
	if (vala_attribute_has_argument (a, "sentinel")) {
		char* _tmp3;
		_tmp3 = NULL;
		vala_method_set_sentinel (self, (_tmp3 = vala_attribute_get_string (a, "sentinel")));
		_tmp3 = (g_free (_tmp3), NULL);
	}
	if (vala_attribute_has_argument (a, "instance_pos")) {
		vala_method_set_cinstance_parameter_position (self, vala_attribute_get_double (a, "instance_pos"));
	}
	if (vala_attribute_has_argument (a, "array_length_pos")) {
		vala_method_set_carray_length_parameter_position (self, vala_attribute_get_double (a, "array_length_pos"));
	}
	if (vala_attribute_has_argument (a, "delegate_target_pos")) {
		vala_method_set_cdelegate_target_parameter_position (self, vala_attribute_get_double (a, "delegate_target_pos"));
	}
	if (vala_attribute_has_argument (a, "has_construct_function")) {
		vala_method_set_has_construct_function (self, vala_attribute_get_bool (a, "has_construct_function"));
	}
}


/**
 * Process all associated attributes.
 */
void vala_method_process_attributes (ValaMethod* self) {
	g_return_if_fail (VALA_IS_METHOD (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = VALA_CODE_NODE (self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = ((ValaAttribute*) (a_it->data)), (_tmp0 == NULL ? NULL : vala_code_node_ref (_tmp0)));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_method_process_ccode_attribute (self, a);
				} else {
					if (_vala_strcmp0 (vala_attribute_get_name (a), "ReturnsModifiedPointer") == 0) {
						vala_method_set_returns_modified_pointer (self, TRUE);
					} else {
						if (_vala_strcmp0 (vala_attribute_get_name (a), "FloatingReference") == 0) {
							vala_data_type_set_floating_reference (vala_method_get_return_type (self), TRUE);
						} else {
							if (_vala_strcmp0 (vala_attribute_get_name (a), "NoArrayLength") == 0) {
								vala_method_set_no_array_length (self, TRUE);
							} else {
								if (_vala_strcmp0 (vala_attribute_get_name (a), "PrintfFormat") == 0) {
									vala_method_set_printf_format (self, TRUE);
								}
							}
						}
					}
				}
				(a == NULL ? NULL : (a = (vala_code_node_unref (a), NULL)));
			}
		}
	}
}


/**
 * Checks whether the parameters and return type of this method are
 * compatible with the specified method
 *
 * @param base_method a method
 * @param invalid_match error string about which check failed
 * @return true if the specified method is compatible to this method
 */
gboolean vala_method_compatible (ValaMethod* self, ValaMethod* base_method, char** invalid_match) {
	GeeIterator* method_params_it;
	gint param_index;
	gboolean _tmp13;
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	g_return_val_if_fail (VALA_IS_METHOD (base_method), FALSE);
	(*invalid_match) = NULL;
	if (!vala_data_type_equals (vala_method_get_return_type (self), vala_method_get_return_type (base_method))) {
		char* _tmp0;
		_tmp0 = NULL;
		(*invalid_match) = (_tmp0 = g_strdup ("incompatible return type"), ((*invalid_match) = (g_free ((*invalid_match)), NULL)), _tmp0);
		return FALSE;
	}
	method_params_it = gee_iterable_iterator (GEE_ITERABLE (self->priv->parameters));
	param_index = 1;
	{
		GeeList* base_param_collection;
		int base_param_it;
		base_param_collection = base_method->priv->parameters;
		for (base_param_it = 0; base_param_it < gee_collection_get_size (GEE_COLLECTION (base_param_collection)); base_param_it = base_param_it + 1) {
			ValaFormalParameter* base_param;
			base_param = ((ValaFormalParameter*) (gee_list_get (GEE_LIST (base_param_collection), base_param_it)));
			{
				ValaFormalParameter* _tmp4;
				gboolean _tmp5;
				/* this method may not expect less arguments */
				if (!gee_iterator_next (method_params_it)) {
					char* _tmp2;
					gboolean _tmp3;
					_tmp2 = NULL;
					(*invalid_match) = (_tmp2 = g_strdup ("too few parameters"), ((*invalid_match) = (g_free ((*invalid_match)), NULL)), _tmp2);
					return (_tmp3 = FALSE, (base_param == NULL ? NULL : (base_param = (vala_code_node_unref (base_param), NULL))), (method_params_it == NULL ? NULL : (method_params_it = (gee_collection_object_unref (method_params_it), NULL))), _tmp3);
				}
				_tmp4 = NULL;
				if ((_tmp5 = !vala_data_type_equals (vala_formal_parameter_get_parameter_type (base_param), vala_formal_parameter_get_parameter_type ((_tmp4 = ((ValaFormalParameter*) (gee_iterator_get (method_params_it)))))), (_tmp4 == NULL ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL))), _tmp5)) {
					char* _tmp6;
					gboolean _tmp7;
					_tmp6 = NULL;
					(*invalid_match) = (_tmp6 = g_strdup_printf ("incompatible type of parameter %d", param_index), ((*invalid_match) = (g_free ((*invalid_match)), NULL)), _tmp6);
					return (_tmp7 = FALSE, (base_param == NULL ? NULL : (base_param = (vala_code_node_unref (base_param), NULL))), (method_params_it == NULL ? NULL : (method_params_it = (gee_collection_object_unref (method_params_it), NULL))), _tmp7);
				}
				param_index++;
				(base_param == NULL ? NULL : (base_param = (vala_code_node_unref (base_param), NULL)));
			}
		}
	}
	/* this method may not expect more arguments */
	if (gee_iterator_next (method_params_it)) {
		char* _tmp8;
		gboolean _tmp9;
		_tmp8 = NULL;
		(*invalid_match) = (_tmp8 = g_strdup ("too many parameters"), ((*invalid_match) = (g_free ((*invalid_match)), NULL)), _tmp8);
		return (_tmp9 = FALSE, (method_params_it == NULL ? NULL : (method_params_it = (gee_collection_object_unref (method_params_it), NULL))), _tmp9);
	}
	/* this method may throw more but not less errors than the base method */
	{
		GeeList* method_error_type_collection;
		int method_error_type_it;
		method_error_type_collection = vala_code_node_get_error_types (VALA_CODE_NODE (self));
		for (method_error_type_it = 0; method_error_type_it < gee_collection_get_size (GEE_COLLECTION (method_error_type_collection)); method_error_type_it = method_error_type_it + 1) {
			ValaDataType* method_error_type;
			method_error_type = ((ValaDataType*) (gee_list_get (GEE_LIST (method_error_type_collection), method_error_type_it)));
			{
				gboolean match;
				match = FALSE;
				{
					GeeList* base_method_error_type_collection;
					int base_method_error_type_it;
					base_method_error_type_collection = vala_code_node_get_error_types (VALA_CODE_NODE (base_method));
					for (base_method_error_type_it = 0; base_method_error_type_it < gee_collection_get_size (GEE_COLLECTION (base_method_error_type_collection)); base_method_error_type_it = base_method_error_type_it + 1) {
						ValaDataType* base_method_error_type;
						base_method_error_type = ((ValaDataType*) (gee_list_get (GEE_LIST (base_method_error_type_collection), base_method_error_type_it)));
						{
							if (vala_data_type_compatible (method_error_type, base_method_error_type)) {
								match = TRUE;
								(base_method_error_type == NULL ? NULL : (base_method_error_type = (vala_code_node_unref (base_method_error_type), NULL)));
								break;
							}
							(base_method_error_type == NULL ? NULL : (base_method_error_type = (vala_code_node_unref (base_method_error_type), NULL)));
						}
					}
					(base_method_error_type_collection == NULL ? NULL : (base_method_error_type_collection = (gee_collection_object_unref (base_method_error_type_collection), NULL)));
				}
				if (!match) {
					char* _tmp11;
					char* _tmp10;
					gboolean _tmp12;
					_tmp11 = NULL;
					_tmp10 = NULL;
					(*invalid_match) = (_tmp11 = g_strdup_printf ("incompatible error type `%s'", (_tmp10 = vala_code_node_to_string (VALA_CODE_NODE (method_error_type)))), ((*invalid_match) = (g_free ((*invalid_match)), NULL)), _tmp11);
					_tmp10 = (g_free (_tmp10), NULL);
					return (_tmp12 = FALSE, (method_error_type == NULL ? NULL : (method_error_type = (vala_code_node_unref (method_error_type), NULL))), (method_error_type_collection == NULL ? NULL : (method_error_type_collection = (gee_collection_object_unref (method_error_type_collection), NULL))), (method_params_it == NULL ? NULL : (method_params_it = (gee_collection_object_unref (method_params_it), NULL))), _tmp12);
				}
				(method_error_type == NULL ? NULL : (method_error_type = (vala_code_node_unref (method_error_type), NULL)));
			}
		}
		(method_error_type_collection == NULL ? NULL : (method_error_type_collection = (gee_collection_object_unref (method_error_type_collection), NULL)));
	}
	return (_tmp13 = TRUE, (method_params_it == NULL ? NULL : (method_params_it = (gee_collection_object_unref (method_params_it), NULL))), _tmp13);
}


/**
 * Adds a precondition to this method.
 *
 * @param precondition a boolean precondition expression
 */
void vala_method_add_precondition (ValaMethod* self, ValaExpression* precondition) {
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (VALA_IS_EXPRESSION (precondition));
	gee_collection_add (GEE_COLLECTION (self->priv->preconditions), precondition);
	vala_code_node_set_parent_node (VALA_CODE_NODE (precondition), VALA_CODE_NODE (self));
}


/**
 * Returns a copy of the list of preconditions of this method.
 *
 * @return list of preconditions
 */
GeeList* vala_method_get_preconditions (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->preconditions));
}


/**
 * Adds a postcondition to this method.
 *
 * @param postcondition a boolean postcondition expression
 */
void vala_method_add_postcondition (ValaMethod* self, ValaExpression* postcondition) {
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (VALA_IS_EXPRESSION (postcondition));
	gee_collection_add (GEE_COLLECTION (self->priv->postconditions), postcondition);
	vala_code_node_set_parent_node (VALA_CODE_NODE (postcondition), VALA_CODE_NODE (self));
}


/**
 * Returns a copy of the list of postconditions of this method.
 *
 * @return list of postconditions
 */
GeeList* vala_method_get_postconditions (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->postconditions));
}


static void vala_method_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaMethod * self;
	GeeList* error_types;
	self = VALA_METHOD (base);
	g_return_if_fail (VALA_IS_DATA_TYPE (old_type));
	g_return_if_fail (VALA_IS_DATA_TYPE (new_type));
	if (vala_method_get_return_type (self) == old_type) {
		vala_method_set_return_type (self, new_type);
		return;
	}
	error_types = vala_code_node_get_error_types (VALA_CODE_NODE (self));
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size (GEE_COLLECTION (error_types)); i++) {
			ValaDataType* _tmp0;
			gboolean _tmp1;
			_tmp0 = NULL;
			if ((_tmp1 = (_tmp0 = ((ValaDataType*) (gee_list_get (((GeeList*) (error_types)), i)))) == old_type, (_tmp0 == NULL ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL))), _tmp1)) {
				gee_list_set (((GeeList*) (error_types)), i, new_type);
				(error_types == NULL ? NULL : (error_types = (gee_collection_object_unref (error_types), NULL)));
				return;
			}
		}
	}
	(error_types == NULL ? NULL : (error_types = (gee_collection_object_unref (error_types), NULL)));
}


static ValaCodeBinding* vala_method_real_create_code_binding (ValaCodeNode* base, ValaCodeGenerator* codegen) {
	ValaMethod * self;
	self = VALA_METHOD (base);
	g_return_val_if_fail (VALA_IS_CODE_GENERATOR (codegen), NULL);
	return vala_code_generator_create_method_binding (codegen, self);
}


static void vala_method_find_base_methods (ValaMethod* self) {
	g_return_if_fail (VALA_IS_METHOD (self));
	if (self->priv->base_methods_valid) {
		return;
	}
	if (VALA_IS_CLASS (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))) {
		/* VAPI classes don't specify overridden methods */
		if (!vala_symbol_get_external_package (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))) {
			if (!(VALA_IS_CREATION_METHOD (self))) {
				vala_method_find_base_interface_method (self, VALA_CLASS (vala_symbol_get_parent_symbol (VALA_SYMBOL (self))));
				if (self->priv->_is_virtual || self->priv->_is_abstract || self->priv->_overrides) {
					vala_method_find_base_class_method (self, VALA_CLASS (vala_symbol_get_parent_symbol (VALA_SYMBOL (self))));
				}
			}
		} else {
			if (self->priv->_is_virtual || self->priv->_is_abstract) {
				self->priv->_base_method = self;
			}
		}
	} else {
		if (VALA_IS_INTERFACE (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))) {
			if (self->priv->_is_virtual || self->priv->_is_abstract) {
				ValaMethod* _tmp1;
				ValaMethod* _tmp0;
				_tmp1 = NULL;
				_tmp0 = NULL;
				self->priv->_base_interface_method = (_tmp1 = (_tmp0 = self, (_tmp0 == NULL ? NULL : vala_code_node_ref (_tmp0))), (self->priv->_base_interface_method == NULL ? NULL : (self->priv->_base_interface_method = (vala_code_node_unref (self->priv->_base_interface_method), NULL))), _tmp1);
			}
		}
	}
	self->priv->base_methods_valid = TRUE;
}


static void vala_method_find_base_class_method (ValaMethod* self, ValaClass* cl) {
	ValaSymbol* sym;
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (VALA_IS_CLASS (cl));
	sym = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (cl)), vala_symbol_get_name (VALA_SYMBOL (self)));
	if (VALA_IS_METHOD (sym)) {
		ValaMethod* _tmp0;
		ValaMethod* base_method;
		_tmp0 = NULL;
		base_method = (_tmp0 = VALA_METHOD (sym), (_tmp0 == NULL ? NULL : vala_code_node_ref (_tmp0)));
		if (base_method->priv->_is_abstract || base_method->priv->_is_virtual) {
			char* invalid_match;
			char* _tmp3;
			gboolean _tmp2;
			char* _tmp1;
			invalid_match = NULL;
			_tmp3 = NULL;
			_tmp1 = NULL;
			if (!(_tmp2 = vala_method_compatible (self, base_method, &_tmp1), invalid_match = (_tmp3 = _tmp1, (invalid_match = (g_free (invalid_match), NULL)), _tmp3), _tmp2)) {
				char* _tmp6;
				char* _tmp5;
				char* _tmp4;
				vala_code_node_set_error (VALA_CODE_NODE (self), TRUE);
				_tmp6 = NULL;
				_tmp5 = NULL;
				_tmp4 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (self)), (_tmp6 = g_strdup_printf ("overriding method `%s' is incompatible with base method `%s': %s.", (_tmp4 = vala_symbol_get_full_name (VALA_SYMBOL (self))), (_tmp5 = vala_symbol_get_full_name (VALA_SYMBOL (base_method))), invalid_match)));
				_tmp6 = (g_free (_tmp6), NULL);
				_tmp5 = (g_free (_tmp5), NULL);
				_tmp4 = (g_free (_tmp4), NULL);
				invalid_match = (g_free (invalid_match), NULL);
				(base_method == NULL ? NULL : (base_method = (vala_code_node_unref (base_method), NULL)));
				(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
				return;
			}
			self->priv->_base_method = base_method;
			invalid_match = (g_free (invalid_match), NULL);
			(base_method == NULL ? NULL : (base_method = (vala_code_node_unref (base_method), NULL)));
			(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
			return;
		}
		(base_method == NULL ? NULL : (base_method = (vala_code_node_unref (base_method), NULL)));
	} else {
		if (VALA_IS_SIGNAL (sym)) {
			ValaSignal* _tmp7;
			ValaSignal* sig;
			_tmp7 = NULL;
			sig = (_tmp7 = VALA_SIGNAL (sym), (_tmp7 == NULL ? NULL : vala_code_node_ref (_tmp7)));
			if (vala_signal_get_is_virtual (sig)) {
				ValaMethod* base_method;
				char* invalid_match;
				char* _tmp10;
				gboolean _tmp9;
				char* _tmp8;
				base_method = vala_signal_get_method_handler (sig);
				invalid_match = NULL;
				_tmp10 = NULL;
				_tmp8 = NULL;
				if (!(_tmp9 = vala_method_compatible (self, base_method, &_tmp8), invalid_match = (_tmp10 = _tmp8, (invalid_match = (g_free (invalid_match), NULL)), _tmp10), _tmp9)) {
					char* _tmp13;
					char* _tmp12;
					char* _tmp11;
					vala_code_node_set_error (VALA_CODE_NODE (self), TRUE);
					_tmp13 = NULL;
					_tmp12 = NULL;
					_tmp11 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (self)), (_tmp13 = g_strdup_printf ("overriding method `%s' is incompatible with base method `%s': %s.", (_tmp11 = vala_symbol_get_full_name (VALA_SYMBOL (self))), (_tmp12 = vala_symbol_get_full_name (VALA_SYMBOL (base_method))), invalid_match)));
					_tmp13 = (g_free (_tmp13), NULL);
					_tmp12 = (g_free (_tmp12), NULL);
					_tmp11 = (g_free (_tmp11), NULL);
					(base_method == NULL ? NULL : (base_method = (vala_code_node_unref (base_method), NULL)));
					invalid_match = (g_free (invalid_match), NULL);
					(sig == NULL ? NULL : (sig = (vala_code_node_unref (sig), NULL)));
					(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
					return;
				}
				self->priv->_base_method = base_method;
				(base_method == NULL ? NULL : (base_method = (vala_code_node_unref (base_method), NULL)));
				invalid_match = (g_free (invalid_match), NULL);
				(sig == NULL ? NULL : (sig = (vala_code_node_unref (sig), NULL)));
				(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
				return;
			}
			(sig == NULL ? NULL : (sig = (vala_code_node_unref (sig), NULL)));
		}
	}
	if (vala_class_get_base_class (cl) != NULL) {
		vala_method_find_base_class_method (self, vala_class_get_base_class (cl));
	}
	(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
}


static void vala_method_find_base_interface_method (ValaMethod* self, ValaClass* cl) {
	g_return_if_fail (VALA_IS_METHOD (self));
	g_return_if_fail (VALA_IS_CLASS (cl));
	/* FIXME report error if multiple possible base methods are found*/
	{
		GeeList* type_collection;
		int type_it;
		type_collection = vala_class_get_base_types (cl);
		for (type_it = 0; type_it < gee_collection_get_size (GEE_COLLECTION (type_collection)); type_it = type_it + 1) {
			ValaDataType* type;
			type = ((ValaDataType*) (gee_list_get (GEE_LIST (type_collection), type_it)));
			{
				if (VALA_IS_INTERFACE (vala_data_type_get_data_type (type))) {
					ValaSymbol* sym;
					sym = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (vala_data_type_get_data_type (type))), vala_symbol_get_name (VALA_SYMBOL (self)));
					if (VALA_IS_METHOD (sym)) {
						ValaMethod* _tmp0;
						ValaMethod* base_method;
						_tmp0 = NULL;
						base_method = (_tmp0 = VALA_METHOD (sym), (_tmp0 == NULL ? NULL : vala_code_node_ref (_tmp0)));
						if (base_method->priv->_is_abstract || base_method->priv->_is_virtual) {
							char* invalid_match;
							char* _tmp3;
							gboolean _tmp2;
							char* _tmp1;
							ValaMethod* _tmp8;
							ValaMethod* _tmp7;
							invalid_match = NULL;
							_tmp3 = NULL;
							_tmp1 = NULL;
							if (!(_tmp2 = vala_method_compatible (self, base_method, &_tmp1), invalid_match = (_tmp3 = _tmp1, (invalid_match = (g_free (invalid_match), NULL)), _tmp3), _tmp2)) {
								char* _tmp6;
								char* _tmp5;
								char* _tmp4;
								vala_code_node_set_error (VALA_CODE_NODE (self), TRUE);
								_tmp6 = NULL;
								_tmp5 = NULL;
								_tmp4 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (self)), (_tmp6 = g_strdup_printf ("overriding method `%s' is incompatible with base method `%s': %s.", (_tmp4 = vala_symbol_get_full_name (VALA_SYMBOL (self))), (_tmp5 = vala_symbol_get_full_name (VALA_SYMBOL (base_method))), invalid_match)));
								_tmp6 = (g_free (_tmp6), NULL);
								_tmp5 = (g_free (_tmp5), NULL);
								_tmp4 = (g_free (_tmp4), NULL);
								invalid_match = (g_free (invalid_match), NULL);
								(base_method == NULL ? NULL : (base_method = (vala_code_node_unref (base_method), NULL)));
								(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
								(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
								(type_collection == NULL ? NULL : (type_collection = (gee_collection_object_unref (type_collection), NULL)));
								return;
							}
							_tmp8 = NULL;
							_tmp7 = NULL;
							self->priv->_base_interface_method = (_tmp8 = (_tmp7 = base_method, (_tmp7 == NULL ? NULL : vala_code_node_ref (_tmp7))), (self->priv->_base_interface_method == NULL ? NULL : (self->priv->_base_interface_method = (vala_code_node_unref (self->priv->_base_interface_method), NULL))), _tmp8);
							invalid_match = (g_free (invalid_match), NULL);
							(base_method == NULL ? NULL : (base_method = (vala_code_node_unref (base_method), NULL)));
							(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
							(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
							(type_collection == NULL ? NULL : (type_collection = (gee_collection_object_unref (type_collection), NULL)));
							return;
						}
						(base_method == NULL ? NULL : (base_method = (vala_code_node_unref (base_method), NULL)));
					}
					(sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL)));
				}
				(type == NULL ? NULL : (type = (vala_code_node_unref (type), NULL)));
			}
		}
		(type_collection == NULL ? NULL : (type_collection = (gee_collection_object_unref (type_collection), NULL)));
	}
}


ValaDataType* vala_method_get_return_type (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_return_type;
}


void vala_method_set_return_type (ValaMethod* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_return_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (vala_code_node_unref (self->priv->_return_type), NULL))), _tmp2);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_return_type), VALA_CODE_NODE (self));
}


ValaBlock* vala_method_get_body (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_body;
}


void vala_method_set_body (ValaMethod* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_body == NULL ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL))), _tmp2);
	if (self->priv->_body != NULL) {
		vala_symbol_set_owner (VALA_SYMBOL (self->priv->_body), vala_symbol_get_scope (VALA_SYMBOL (self)));
	}
}


ValaBasicBlock* vala_method_get_entry_block (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_entry_block;
}


void vala_method_set_entry_block (ValaMethod* self, ValaBasicBlock* value) {
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* _tmp1;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_entry_block = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_basic_block_ref (_tmp1))), (self->priv->_entry_block == NULL ? NULL : (self->priv->_entry_block = (vala_basic_block_unref (self->priv->_entry_block), NULL))), _tmp2);
}


ValaBasicBlock* vala_method_get_exit_block (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_exit_block;
}


void vala_method_set_exit_block (ValaMethod* self, ValaBasicBlock* value) {
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* _tmp1;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_exit_block = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_basic_block_ref (_tmp1))), (self->priv->_exit_block == NULL ? NULL : (self->priv->_exit_block = (vala_basic_block_unref (self->priv->_exit_block), NULL))), _tmp2);
}


MemberBinding vala_method_get_binding (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), 0);
	return self->priv->_binding;
}


void vala_method_set_binding (ValaMethod* self, MemberBinding value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_binding = value;
}


const char* vala_method_get_vfunc_name (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (self->priv->_vfunc_name == NULL) {
		char* _tmp1;
		const char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->_vfunc_name = (_tmp1 = (_tmp0 = vala_symbol_get_name (VALA_SYMBOL (self)), (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->_vfunc_name = (g_free (self->priv->_vfunc_name), NULL)), _tmp1);
	}
	return self->priv->_vfunc_name;
}


void vala_method_set_vfunc_name (ValaMethod* self, const char* value) {
	char* _tmp4;
	const char* _tmp3;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->priv->_vfunc_name = (_tmp4 = (_tmp3 = value, (_tmp3 == NULL ? NULL : g_strdup (_tmp3))), (self->priv->_vfunc_name = (g_free (self->priv->_vfunc_name), NULL)), _tmp4);
}


const char* vala_method_get_sentinel (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	if (self->priv->_sentinel == NULL) {
		return VALA_METHOD_DEFAULT_SENTINEL;
	}
	return self->priv->_sentinel;
}


void vala_method_set_sentinel (ValaMethod* self, const char* value) {
	char* _tmp3;
	const char* _tmp2;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->_sentinel = (_tmp3 = (_tmp2 = value, (_tmp2 == NULL ? NULL : g_strdup (_tmp2))), (self->priv->_sentinel = (g_free (self->priv->_sentinel), NULL)), _tmp3);
}


gboolean vala_method_get_is_abstract (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_is_abstract;
}


void vala_method_set_is_abstract (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_is_abstract = value;
}


gboolean vala_method_get_is_virtual (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_is_virtual;
}


void vala_method_set_is_virtual (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_is_virtual = value;
}


gboolean vala_method_get_overrides (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_overrides;
}


void vala_method_set_overrides (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_overrides = value;
}


gboolean vala_method_get_is_inline (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_is_inline;
}


void vala_method_set_is_inline (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_is_inline = value;
}


gboolean vala_method_get_returns_modified_pointer (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_returns_modified_pointer;
}


void vala_method_set_returns_modified_pointer (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_returns_modified_pointer = value;
}


ValaMethod* vala_method_get_base_method (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	vala_method_find_base_methods (self);
	return self->priv->_base_method;
}


ValaMethod* vala_method_get_base_interface_method (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	vala_method_find_base_methods (self);
	return self->priv->_base_interface_method;
}


ValaFormalParameter* vala_method_get_this_parameter (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_this_parameter;
}


void vala_method_set_this_parameter (ValaMethod* self, ValaFormalParameter* value) {
	ValaFormalParameter* _tmp2;
	ValaFormalParameter* _tmp1;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_this_parameter = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (vala_code_node_unref (self->priv->_this_parameter), NULL))), _tmp2);
}


ValaLocalVariable* vala_method_get_result_var (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), NULL);
	return self->priv->_result_var;
}


void vala_method_set_result_var (ValaMethod* self, ValaLocalVariable* value) {
	ValaLocalVariable* _tmp2;
	ValaLocalVariable* _tmp1;
	g_return_if_fail (VALA_IS_METHOD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_result_var = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_result_var == NULL ? NULL : (self->priv->_result_var = (vala_code_node_unref (self->priv->_result_var), NULL))), _tmp2);
}


double vala_method_get_cinstance_parameter_position (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), 0.0);
	return self->priv->_cinstance_parameter_position;
}


void vala_method_set_cinstance_parameter_position (ValaMethod* self, double value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_cinstance_parameter_position = value;
}


double vala_method_get_carray_length_parameter_position (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), 0.0);
	return self->priv->_carray_length_parameter_position;
}


void vala_method_set_carray_length_parameter_position (ValaMethod* self, double value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_carray_length_parameter_position = value;
}


double vala_method_get_cdelegate_target_parameter_position (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), 0.0);
	return self->priv->_cdelegate_target_parameter_position;
}


void vala_method_set_cdelegate_target_parameter_position (ValaMethod* self, double value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_cdelegate_target_parameter_position = value;
}


gboolean vala_method_get_no_array_length (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_no_array_length;
}


void vala_method_set_no_array_length (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_no_array_length = value;
	{
		GeeList* param_collection;
		int param_it;
		param_collection = self->priv->parameters;
		for (param_it = 0; param_it < gee_collection_get_size (GEE_COLLECTION (param_collection)); param_it = param_it + 1) {
			ValaFormalParameter* param;
			param = ((ValaFormalParameter*) (gee_list_get (GEE_LIST (param_collection), param_it)));
			{
				vala_formal_parameter_set_no_array_length (param, value);
				(param == NULL ? NULL : (param = (vala_code_node_unref (param), NULL)));
			}
		}
	}
}


gboolean vala_method_get_printf_format (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_printf_format;
}


void vala_method_set_printf_format (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_printf_format = value;
}


gboolean vala_method_get_has_construct_function (ValaMethod* self) {
	g_return_val_if_fail (VALA_IS_METHOD (self), FALSE);
	return self->priv->_has_construct_function;
}


void vala_method_set_has_construct_function (ValaMethod* self, gboolean value) {
	g_return_if_fail (VALA_IS_METHOD (self));
	self->priv->_has_construct_function = value;
}


static void vala_method_class_init (ValaMethodClass * klass) {
	vala_method_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_method_finalize;
	g_type_class_add_private (klass, sizeof (ValaMethodPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_method_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_method_real_accept_children;
	VALA_METHOD_CLASS (klass)->get_default_cname = vala_method_real_get_default_cname;
	VALA_METHOD_CLASS (klass)->get_real_cname = vala_method_real_get_real_cname;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_method_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->create_code_binding = vala_method_real_create_code_binding;
}


static void vala_method_instance_init (ValaMethod * self) {
	self->priv = VALA_METHOD_GET_PRIVATE (self);
	self->priv->_binding = MEMBER_BINDING_INSTANCE;
	self->priv->_has_construct_function = TRUE;
	self->priv->parameters = GEE_LIST (gee_array_list_new (VALA_TYPE_FORMAL_PARAMETER, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal));
	self->priv->preconditions = GEE_LIST (gee_array_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal));
	self->priv->postconditions = GEE_LIST (gee_array_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal));
}


static void vala_method_finalize (ValaCodeNode* obj) {
	ValaMethod * self;
	self = VALA_METHOD (obj);
	(self->priv->_entry_block == NULL ? NULL : (self->priv->_entry_block = (vala_basic_block_unref (self->priv->_entry_block), NULL)));
	(self->priv->_exit_block == NULL ? NULL : (self->priv->_exit_block = (vala_basic_block_unref (self->priv->_exit_block), NULL)));
	(self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (vala_code_node_unref (self->priv->_this_parameter), NULL)));
	(self->priv->_result_var == NULL ? NULL : (self->priv->_result_var = (vala_code_node_unref (self->priv->_result_var), NULL)));
	(self->priv->parameters == NULL ? NULL : (self->priv->parameters = (gee_collection_object_unref (self->priv->parameters), NULL)));
	self->priv->cname = (g_free (self->priv->cname), NULL);
	self->priv->_vfunc_name = (g_free (self->priv->_vfunc_name), NULL);
	self->priv->_sentinel = (g_free (self->priv->_sentinel), NULL);
	(self->priv->preconditions == NULL ? NULL : (self->priv->preconditions = (gee_collection_object_unref (self->priv->preconditions), NULL)));
	(self->priv->postconditions == NULL ? NULL : (self->priv->postconditions = (gee_collection_object_unref (self->priv->postconditions), NULL)));
	(self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (vala_code_node_unref (self->priv->_return_type), NULL)));
	(self->priv->_body == NULL ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL)));
	(self->priv->_base_interface_method == NULL ? NULL : (self->priv->_base_interface_method = (vala_code_node_unref (self->priv->_base_interface_method), NULL)));
	VALA_CODE_NODE_CLASS (vala_method_parent_class)->finalize (obj);
}


GType vala_method_get_type (void) {
	static GType vala_method_type_id = 0;
	if (vala_method_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaMethodClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_method_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaMethod), 0, (GInstanceInitFunc) vala_method_instance_init, NULL };
		vala_method_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaMethod", &g_define_type_info, 0);
	}
	return vala_method_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) (array))[i] != NULL)
			destroy_func (((gpointer*) (array))[i]);
		}
		else
		for (i = 0; ((gpointer*) (array))[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) (array))[i]);
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




