/* valascope.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valascope.h>
#include <gee/map.h>
#include <gee/list.h>
#include <gee/hashmap.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <vala/valasymbol.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <gobject/gvaluecollector.h>




struct _ValaScopePrivate {
	ValaSymbol* _owner;
	ValaScope* _parent_scope;
	GeeMap* symbol_table;
	GeeList* anonymous_members;
};

#define VALA_SCOPE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SCOPE, ValaScopePrivate))
enum  {
	VALA_SCOPE_DUMMY_PROPERTY
};
static gpointer vala_scope_parent_class = NULL;
static void vala_scope_finalize (ValaScope* obj);



/**
 * Creates a new scope.
 *
 * @return newly created scope
 */
ValaScope* vala_scope_construct (GType object_type, ValaSymbol* owner) {
	ValaScope* self;
	g_return_val_if_fail (owner == NULL || VALA_IS_SYMBOL (owner), NULL);
	self = ((ValaScope*) (g_type_create_instance (object_type)));
	vala_scope_set_owner (self, owner);
	return self;
}


ValaScope* vala_scope_new (ValaSymbol* owner) {
	return vala_scope_construct (VALA_TYPE_SCOPE, owner);
}


/**
 * Adds the specified symbol with the specified name to the symbol table
 * of this scope.
 *
 * @param name name for the specified symbol
 * @param sym  a symbol
 */
void vala_scope_add (ValaScope* self, const char* name, ValaSymbol* sym) {
	g_return_if_fail (VALA_IS_SCOPE (self));
	g_return_if_fail (VALA_IS_SYMBOL (sym));
	if (name != NULL) {
		if (self->priv->symbol_table == NULL) {
			GeeMap* _tmp0;
			_tmp0 = NULL;
			self->priv->symbol_table = (_tmp0 = GEE_MAP (gee_hash_map_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, VALA_TYPE_SYMBOL, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_str_hash, g_str_equal, g_direct_equal)), (self->priv->symbol_table == NULL ? NULL : (self->priv->symbol_table = (gee_collection_object_unref (self->priv->symbol_table), NULL))), _tmp0);
		} else {
			ValaSymbol* _tmp1;
			gboolean _tmp2;
			_tmp1 = NULL;
			if ((_tmp2 = (_tmp1 = vala_scope_lookup (self, name)) != NULL, (_tmp1 == NULL ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL))), _tmp2)) {
				char* _tmp4;
				char* _tmp3;
				vala_code_node_set_error (VALA_CODE_NODE (self->priv->_owner), TRUE);
				_tmp4 = NULL;
				_tmp3 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (sym)), (_tmp4 = g_strdup_printf ("`%s' already contains a definition for `%s'", (_tmp3 = vala_symbol_get_full_name (self->priv->_owner)), name)));
				_tmp4 = (g_free (_tmp4), NULL);
				_tmp3 = (g_free (_tmp3), NULL);
				return;
			}
		}
		gee_map_set (((GeeMap*) (self->priv->symbol_table)), ((const char*) (name)), sym);
	} else {
		if (self->priv->anonymous_members == NULL) {
			GeeList* _tmp5;
			_tmp5 = NULL;
			self->priv->anonymous_members = (_tmp5 = GEE_LIST (gee_array_list_new (VALA_TYPE_SYMBOL, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal)), (self->priv->anonymous_members == NULL ? NULL : (self->priv->anonymous_members = (gee_collection_object_unref (self->priv->anonymous_members), NULL))), _tmp5);
		}
		gee_collection_add (GEE_COLLECTION (self->priv->anonymous_members), sym);
	}
	vala_symbol_set_owner (sym, self);
}


void vala_scope_remove (ValaScope* self, const char* name) {
	g_return_if_fail (VALA_IS_SCOPE (self));
	g_return_if_fail (name != NULL);
	gee_map_remove (self->priv->symbol_table, name);
}


/**
 * Returns the symbol stored in the symbol table with the specified
 * name.
 *
 * @param name name of the symbol to be returned
 * @return     found symbol or null
 */
ValaSymbol* vala_scope_lookup (ValaScope* self, const char* name) {
	ValaSymbol* sym;
	g_return_val_if_fail (VALA_IS_SCOPE (self), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	if (self->priv->symbol_table == NULL) {
		return NULL;
	}
	sym = ((ValaSymbol*) (gee_map_get (((GeeMap*) (self->priv->symbol_table)), name)));
	if (sym != NULL && !vala_symbol_get_active (sym)) {
		ValaSymbol* _tmp1;
		_tmp1 = NULL;
		sym = (_tmp1 = NULL, (sym == NULL ? NULL : (sym = (vala_code_node_unref (sym), NULL))), _tmp1);
	}
	return sym;
}


/**
 * Returns whether the specified scope is an ancestor of this scope.
 *
 * @param scope a scope or null for the root scope
 * @return      true if this scope is a subscope of the specified
 *              scope, false otherwise
 */
gboolean vala_scope_is_subscope_of (ValaScope* self, ValaScope* scope) {
	g_return_val_if_fail (VALA_IS_SCOPE (self), FALSE);
	g_return_val_if_fail (scope == NULL || VALA_IS_SCOPE (scope), FALSE);
	if (scope == self) {
		return TRUE;
	}
	/* null scope is the root scope*/
	if (scope == NULL) {
		return TRUE;
	}
	if (self->priv->_parent_scope != NULL) {
		return vala_scope_is_subscope_of (self->priv->_parent_scope, scope);
	}
	return FALSE;
}


ValaSymbol* vala_scope_get_owner (ValaScope* self) {
	g_return_val_if_fail (VALA_IS_SCOPE (self), NULL);
	return self->priv->_owner;
}


void vala_scope_set_owner (ValaScope* self, ValaSymbol* value) {
	g_return_if_fail (VALA_IS_SCOPE (self));
	self->priv->_owner = value;
}


ValaScope* vala_scope_get_parent_scope (ValaScope* self) {
	g_return_val_if_fail (VALA_IS_SCOPE (self), NULL);
	return self->priv->_parent_scope;
}


void vala_scope_set_parent_scope (ValaScope* self, ValaScope* value) {
	g_return_if_fail (VALA_IS_SCOPE (self));
	self->priv->_parent_scope = value;
}


static void vala_value_scope_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void vala_value_scope_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		vala_scope_unref (value->data[0].v_pointer);
	}
}


static void vala_value_scope_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = vala_scope_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer vala_value_scope_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* vala_value_scope_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		ValaScope* object;
		object = value->data[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_OBJECT_TYPE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", G_OBJECT_TYPE (object), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* vala_value_scope_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	ValaScope** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = vala_scope_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* vala_param_spec_scope (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	ValaParamSpecScope* spec;
	g_return_val_if_fail (g_type_is_a (object_type, VALA_TYPE_SCOPE), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer vala_value_get_scope (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_SCOPE), NULL);
	return value->data[0].v_pointer;
}


void vala_value_set_scope (GValue* value, gpointer v_object) {
	ValaScope* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_SCOPE));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, VALA_TYPE_SCOPE));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		vala_scope_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		vala_scope_unref (old);
	}
}


static void vala_scope_class_init (ValaScopeClass * klass) {
	vala_scope_parent_class = g_type_class_peek_parent (klass);
	VALA_SCOPE_CLASS (klass)->finalize = vala_scope_finalize;
	g_type_class_add_private (klass, sizeof (ValaScopePrivate));
}


static void vala_scope_instance_init (ValaScope * self) {
	self->priv = VALA_SCOPE_GET_PRIVATE (self);
	self->ref_count = 1;
}


static void vala_scope_finalize (ValaScope* obj) {
	ValaScope * self;
	self = VALA_SCOPE (obj);
	(self->priv->symbol_table == NULL ? NULL : (self->priv->symbol_table = (gee_collection_object_unref (self->priv->symbol_table), NULL)));
	(self->priv->anonymous_members == NULL ? NULL : (self->priv->anonymous_members = (gee_collection_object_unref (self->priv->anonymous_members), NULL)));
}


GType vala_scope_get_type (void) {
	static GType vala_scope_type_id = 0;
	if (vala_scope_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { vala_value_scope_init, vala_value_scope_free_value, vala_value_scope_copy_value, vala_value_scope_peek_pointer, "p", vala_value_scope_collect_value, "p", vala_value_scope_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (ValaScopeClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_scope_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaScope), 0, (GInstanceInitFunc) vala_scope_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		vala_scope_type_id = g_type_register_fundamental (g_type_fundamental_next (), "ValaScope", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return vala_scope_type_id;
}


gpointer vala_scope_ref (gpointer instance) {
	ValaScope* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void vala_scope_unref (gpointer instance) {
	ValaScope* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		VALA_SCOPE_GET_CLASS (self)->finalize (self);
		g_type_free_instance (((GTypeInstance *) (self)));
	}
}




