/* valapropertyaccessor.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valapropertyaccessor.h>
#include <vala/valaproperty.h>
#include <vala/valablock.h>
#include <vala/valabasicblock.h>
#include <vala/valaformalparameter.h>
#include <vala/valatypesymbol.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>




struct _ValaPropertyAccessorPrivate {
	ValaProperty* _prop;
	gboolean _readable;
	gboolean _writable;
	gboolean _construction;
	ValaSymbolAccessibility _access;
	ValaBlock* _body;
	ValaBasicBlock* _entry_block;
	ValaBasicBlock* _exit_block;
	gboolean _automatic_body;
	ValaFormalParameter* _value_parameter;
	char* _cname;
};

#define VALA_PROPERTY_ACCESSOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_PROPERTY_ACCESSOR, ValaPropertyAccessorPrivate))
enum  {
	VALA_PROPERTY_ACCESSOR_DUMMY_PROPERTY
};
static void vala_property_accessor_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_property_accessor_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gpointer vala_property_accessor_parent_class = NULL;
static void vala_property_accessor_finalize (ValaCodeNode* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * The publicly accessible name of the function that performs the
 * access in C code.
 */
char* vala_property_accessor_get_cname (ValaPropertyAccessor* self) {
	ValaTypeSymbol* _tmp2;
	ValaTypeSymbol* t;
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), NULL);
	if (self->priv->_cname != NULL) {
		const char* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = self->priv->_cname, (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
	}
	_tmp2 = NULL;
	t = (_tmp2 = VALA_TYPESYMBOL (vala_symbol_get_parent_symbol (VALA_SYMBOL (self->priv->_prop))), (_tmp2 == NULL ? NULL : vala_code_node_ref (_tmp2)));
	if (self->priv->_readable) {
		char* _tmp3;
		char* _tmp4;
		char* _tmp5;
		_tmp3 = NULL;
		_tmp4 = NULL;
		_tmp5 = NULL;
		return (_tmp5 = (_tmp4 = g_strdup_printf ("%s_get_%s", (_tmp3 = vala_symbol_get_lower_case_cname (VALA_SYMBOL (t), NULL)), vala_symbol_get_name (VALA_SYMBOL (self->priv->_prop))), (_tmp3 = (g_free (_tmp3), NULL)), _tmp4), (t == NULL ? NULL : (t = (vala_code_node_unref (t), NULL))), _tmp5);
	} else {
		char* _tmp6;
		char* _tmp7;
		char* _tmp8;
		_tmp6 = NULL;
		_tmp7 = NULL;
		_tmp8 = NULL;
		return (_tmp8 = (_tmp7 = g_strdup_printf ("%s_set_%s", (_tmp6 = vala_symbol_get_lower_case_cname (VALA_SYMBOL (t), NULL)), vala_symbol_get_name (VALA_SYMBOL (self->priv->_prop))), (_tmp6 = (g_free (_tmp6), NULL)), _tmp7), (t == NULL ? NULL : (t = (vala_code_node_unref (t), NULL))), _tmp8);
	}
	(t == NULL ? NULL : (t = (vala_code_node_unref (t), NULL)));
}


/**
 * Creates a new property accessor.
 *
 * @param readable     true if get accessor, false otherwise
 * @param writable     true if set accessor, false otherwise
 * @param construction true if construct accessor, false otherwise
 * @param body         accessor body
 * @param source       reference to source code
 * @return             newly created property accessor
 */
ValaPropertyAccessor* vala_property_accessor_construct (GType object_type, gboolean readable, gboolean writable, gboolean construction, ValaBlock* body, ValaSourceReference* source_reference) {
	ValaPropertyAccessor* self;
	g_return_val_if_fail (body == NULL || VALA_IS_BLOCK (body), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = ((ValaPropertyAccessor*) (g_type_create_instance (object_type)));
	vala_property_accessor_set_readable (self, readable);
	vala_property_accessor_set_writable (self, writable);
	vala_property_accessor_set_construction (self, construction);
	vala_property_accessor_set_body (self, body);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


ValaPropertyAccessor* vala_property_accessor_new (gboolean readable, gboolean writable, gboolean construction, ValaBlock* body, ValaSourceReference* source_reference) {
	return vala_property_accessor_construct (VALA_TYPE_PROPERTY_ACCESSOR, readable, writable, construction, body, source_reference);
}


static void vala_property_accessor_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaPropertyAccessor * self;
	self = VALA_PROPERTY_ACCESSOR (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_property_accessor (visitor, self);
}


static void vala_property_accessor_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaPropertyAccessor * self;
	self = VALA_PROPERTY_ACCESSOR (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	if (self->priv->_body != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (self->priv->_body), visitor);
	}
}


/**
 * Process all associated attributes.
 */
void vala_property_accessor_process_attributes (ValaPropertyAccessor* self) {
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = VALA_CODE_NODE (self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp1;
			ValaAttribute* a;
			_tmp1 = NULL;
			a = (_tmp1 = ((ValaAttribute*) (a_it->data)), (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1)));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					if (vala_attribute_has_argument (a, "cname")) {
						char* _tmp0;
						_tmp0 = NULL;
						self->priv->_cname = (_tmp0 = vala_attribute_get_string (a, "cname"), (self->priv->_cname = (g_free (self->priv->_cname), NULL)), _tmp0);
					}
				}
				(a == NULL ? NULL : (a = (vala_code_node_unref (a), NULL)));
			}
		}
	}
}


ValaProperty* vala_property_accessor_get_prop (ValaPropertyAccessor* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), NULL);
	return self->priv->_prop;
}


void vala_property_accessor_set_prop (ValaPropertyAccessor* self, ValaProperty* value) {
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	self->priv->_prop = value;
}


gboolean vala_property_accessor_get_readable (ValaPropertyAccessor* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), FALSE);
	return self->priv->_readable;
}


void vala_property_accessor_set_readable (ValaPropertyAccessor* self, gboolean value) {
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	self->priv->_readable = value;
}


gboolean vala_property_accessor_get_writable (ValaPropertyAccessor* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), FALSE);
	return self->priv->_writable;
}


void vala_property_accessor_set_writable (ValaPropertyAccessor* self, gboolean value) {
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	self->priv->_writable = value;
}


gboolean vala_property_accessor_get_construction (ValaPropertyAccessor* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), FALSE);
	return self->priv->_construction;
}


void vala_property_accessor_set_construction (ValaPropertyAccessor* self, gboolean value) {
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	self->priv->_construction = value;
}


ValaSymbolAccessibility vala_property_accessor_get_access (ValaPropertyAccessor* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), 0);
	return self->priv->_access;
}


void vala_property_accessor_set_access (ValaPropertyAccessor* self, ValaSymbolAccessibility value) {
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	self->priv->_access = value;
}


ValaBlock* vala_property_accessor_get_body (ValaPropertyAccessor* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), NULL);
	return self->priv->_body;
}


void vala_property_accessor_set_body (ValaPropertyAccessor* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_body == NULL ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL))), _tmp2);
}


ValaBasicBlock* vala_property_accessor_get_entry_block (ValaPropertyAccessor* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), NULL);
	return self->priv->_entry_block;
}


void vala_property_accessor_set_entry_block (ValaPropertyAccessor* self, ValaBasicBlock* value) {
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* _tmp1;
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_entry_block = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_basic_block_ref (_tmp1))), (self->priv->_entry_block == NULL ? NULL : (self->priv->_entry_block = (vala_basic_block_unref (self->priv->_entry_block), NULL))), _tmp2);
}


ValaBasicBlock* vala_property_accessor_get_exit_block (ValaPropertyAccessor* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), NULL);
	return self->priv->_exit_block;
}


void vala_property_accessor_set_exit_block (ValaPropertyAccessor* self, ValaBasicBlock* value) {
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* _tmp1;
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_exit_block = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_basic_block_ref (_tmp1))), (self->priv->_exit_block == NULL ? NULL : (self->priv->_exit_block = (vala_basic_block_unref (self->priv->_exit_block), NULL))), _tmp2);
}


gboolean vala_property_accessor_get_automatic_body (ValaPropertyAccessor* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), FALSE);
	return self->priv->_automatic_body;
}


void vala_property_accessor_set_automatic_body (ValaPropertyAccessor* self, gboolean value) {
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	self->priv->_automatic_body = value;
}


ValaFormalParameter* vala_property_accessor_get_value_parameter (ValaPropertyAccessor* self) {
	g_return_val_if_fail (VALA_IS_PROPERTY_ACCESSOR (self), NULL);
	return self->priv->_value_parameter;
}


void vala_property_accessor_set_value_parameter (ValaPropertyAccessor* self, ValaFormalParameter* value) {
	ValaFormalParameter* _tmp2;
	ValaFormalParameter* _tmp1;
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_value_parameter = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_value_parameter == NULL ? NULL : (self->priv->_value_parameter = (vala_code_node_unref (self->priv->_value_parameter), NULL))), _tmp2);
}


static void vala_property_accessor_class_init (ValaPropertyAccessorClass * klass) {
	vala_property_accessor_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_property_accessor_finalize;
	g_type_class_add_private (klass, sizeof (ValaPropertyAccessorPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_property_accessor_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_property_accessor_real_accept_children;
}


static void vala_property_accessor_instance_init (ValaPropertyAccessor * self) {
	self->priv = VALA_PROPERTY_ACCESSOR_GET_PRIVATE (self);
}


static void vala_property_accessor_finalize (ValaCodeNode* obj) {
	ValaPropertyAccessor * self;
	self = VALA_PROPERTY_ACCESSOR (obj);
	(self->priv->_body == NULL ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL)));
	(self->priv->_entry_block == NULL ? NULL : (self->priv->_entry_block = (vala_basic_block_unref (self->priv->_entry_block), NULL)));
	(self->priv->_exit_block == NULL ? NULL : (self->priv->_exit_block = (vala_basic_block_unref (self->priv->_exit_block), NULL)));
	(self->priv->_value_parameter == NULL ? NULL : (self->priv->_value_parameter = (vala_code_node_unref (self->priv->_value_parameter), NULL)));
	self->priv->_cname = (g_free (self->priv->_cname), NULL);
	VALA_CODE_NODE_CLASS (vala_property_accessor_parent_class)->finalize (obj);
}


GType vala_property_accessor_get_type (void) {
	static GType vala_property_accessor_type_id = 0;
	if (vala_property_accessor_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaPropertyAccessorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_property_accessor_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaPropertyAccessor), 0, (GInstanceInitFunc) vala_property_accessor_instance_init, NULL };
		vala_property_accessor_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaPropertyAccessor", &g_define_type_info, 0);
	}
	return vala_property_accessor_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




