/* valabinaryexpression.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valabinaryexpression.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valablock.h>
#include <vala/valalocalvariable.h>
#include <vala/valavaluetype.h>
#include <vala/valadatatype.h>
#include <vala/valatypesymbol.h>
#include <vala/valadeclarationstatement.h>
#include <vala/valasymbol.h>
#include <vala/valaexpressionstatement.h>
#include <vala/valaassignment.h>
#include <vala/valamemberaccess.h>
#include <vala/valabooleanliteral.h>
#include <vala/valastatement.h>
#include <vala/valaifstatement.h>
#include <vala/valareport.h>
#include <vala/valapointertype.h>
#include <vala/valastruct.h>
#include <vala/valamethod.h>
#include <vala/valamethodcall.h>




struct _ValaBinaryExpressionPrivate {
	ValaBinaryOperator _operator;
	ValaExpression* _left;
	ValaExpression* _right;
};

#define VALA_BINARY_EXPRESSION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_BINARY_EXPRESSION, ValaBinaryExpressionPrivate))
enum  {
	VALA_BINARY_EXPRESSION_DUMMY_PROPERTY
};
static void vala_binary_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_binary_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_binary_expression_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static char* vala_binary_expression_get_operator_string (ValaBinaryExpression* self);
static char* vala_binary_expression_real_to_string (ValaCodeNode* base);
static gboolean vala_binary_expression_real_is_constant (ValaExpression* base);
static gboolean vala_binary_expression_real_is_pure (ValaExpression* base);
static gboolean vala_binary_expression_real_is_non_null (ValaExpression* base);
static gboolean vala_binary_expression_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static void vala_binary_expression_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection);
static void vala_binary_expression_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection);
static gpointer vala_binary_expression_parent_class = NULL;
static void vala_binary_expression_finalize (ValaCodeNode* obj);



/**
 * Creates a new binary expression.
 *
 * @param op     binary operator
 * @param left   left operand
 * @param right  right operand
 * @param source reference to source code
 * @return       newly created binary expression
 */
ValaBinaryExpression* vala_binary_expression_construct (GType object_type, ValaBinaryOperator op, ValaExpression* _left, ValaExpression* _right, ValaSourceReference* source) {
	ValaBinaryExpression* self;
	g_return_val_if_fail (_left != NULL, NULL);
	g_return_val_if_fail (_right != NULL, NULL);
	self = (ValaBinaryExpression*) g_type_create_instance (object_type);
	vala_binary_expression_set_operator (self, op);
	vala_binary_expression_set_left (self, _left);
	vala_binary_expression_set_right (self, _right);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source);
	return self;
}


ValaBinaryExpression* vala_binary_expression_new (ValaBinaryOperator op, ValaExpression* _left, ValaExpression* _right, ValaSourceReference* source) {
	return vala_binary_expression_construct (VALA_TYPE_BINARY_EXPRESSION, op, _left, _right, source);
}


static void vala_binary_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaBinaryExpression * self;
	self = (ValaBinaryExpression*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_binary_expression (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


static void vala_binary_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaBinaryExpression * self;
	self = (ValaBinaryExpression*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_binary_expression_get_left (self), visitor);
	vala_code_node_accept ((ValaCodeNode*) vala_binary_expression_get_right (self), visitor);
}


static void vala_binary_expression_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaBinaryExpression * self;
	self = (ValaBinaryExpression*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_binary_expression_get_left (self) == old_node) {
		vala_binary_expression_set_left (self, new_node);
	}
	if (vala_binary_expression_get_right (self) == old_node) {
		vala_binary_expression_set_right (self, new_node);
	}
}


static char* vala_binary_expression_get_operator_string (ValaBinaryExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	switch (self->priv->_operator) {
		case VALA_BINARY_OPERATOR_PLUS:
		{
			return g_strdup ("+");
		}
		case VALA_BINARY_OPERATOR_MINUS:
		{
			return g_strdup ("-");
		}
		case VALA_BINARY_OPERATOR_MUL:
		{
			return g_strdup ("*");
		}
		case VALA_BINARY_OPERATOR_DIV:
		{
			return g_strdup ("/");
		}
		case VALA_BINARY_OPERATOR_MOD:
		{
			return g_strdup ("%");
		}
		case VALA_BINARY_OPERATOR_SHIFT_LEFT:
		{
			return g_strdup ("<<");
		}
		case VALA_BINARY_OPERATOR_SHIFT_RIGHT:
		{
			return g_strdup (">>");
		}
		case VALA_BINARY_OPERATOR_LESS_THAN:
		{
			return g_strdup ("<");
		}
		case VALA_BINARY_OPERATOR_GREATER_THAN:
		{
			return g_strdup (">");
		}
		case VALA_BINARY_OPERATOR_LESS_THAN_OR_EQUAL:
		{
			return g_strdup ("<=");
		}
		case VALA_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL:
		{
			return g_strdup (">=");
		}
		case VALA_BINARY_OPERATOR_EQUALITY:
		{
			return g_strdup ("==");
		}
		case VALA_BINARY_OPERATOR_INEQUALITY:
		{
			return g_strdup ("!+");
		}
		case VALA_BINARY_OPERATOR_BITWISE_AND:
		{
			return g_strdup ("&");
		}
		case VALA_BINARY_OPERATOR_BITWISE_OR:
		{
			return g_strdup ("|");
		}
		case VALA_BINARY_OPERATOR_BITWISE_XOR:
		{
			return g_strdup ("^");
		}
		case VALA_BINARY_OPERATOR_AND:
		{
			return g_strdup ("&&");
		}
		case VALA_BINARY_OPERATOR_OR:
		{
			return g_strdup ("||");
		}
		case VALA_BINARY_OPERATOR_IN:
		{
			return g_strdup ("in");
		}
	}
	g_assert_not_reached ();
}


static char* vala_binary_expression_real_to_string (ValaCodeNode* base) {
	ValaBinaryExpression * self;
	char* _tmp3;
	char* _tmp2;
	char* _tmp1;
	char* _tmp0;
	char* _tmp4;
	self = (ValaBinaryExpression*) base;
	_tmp3 = NULL;
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp4 = NULL;
	return (_tmp4 = g_strconcat (_tmp2 = g_strconcat (_tmp0 = vala_code_node_to_string ((ValaCodeNode*) self->priv->_left), _tmp1 = vala_binary_expression_get_operator_string (self), NULL), _tmp3 = vala_code_node_to_string ((ValaCodeNode*) self->priv->_right), NULL), _tmp3 = (g_free (_tmp3), NULL), _tmp2 = (g_free (_tmp2), NULL), _tmp1 = (g_free (_tmp1), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp4);
}


static gboolean vala_binary_expression_real_is_constant (ValaExpression* base) {
	ValaBinaryExpression * self;
	gboolean _tmp0;
	self = (ValaBinaryExpression*) base;
	_tmp0 = FALSE;
	if (vala_expression_is_constant (vala_binary_expression_get_left (self))) {
		_tmp0 = vala_expression_is_constant (vala_binary_expression_get_right (self));
	} else {
		_tmp0 = FALSE;
	}
	return _tmp0;
}


static gboolean vala_binary_expression_real_is_pure (ValaExpression* base) {
	ValaBinaryExpression * self;
	gboolean _tmp0;
	self = (ValaBinaryExpression*) base;
	_tmp0 = FALSE;
	if (vala_expression_is_pure (vala_binary_expression_get_left (self))) {
		_tmp0 = vala_expression_is_pure (vala_binary_expression_get_right (self));
	} else {
		_tmp0 = FALSE;
	}
	return _tmp0;
}


static gboolean vala_binary_expression_real_is_non_null (ValaExpression* base) {
	ValaBinaryExpression * self;
	gboolean _tmp0;
	self = (ValaBinaryExpression*) base;
	_tmp0 = FALSE;
	if (vala_expression_is_non_null (vala_binary_expression_get_left (self))) {
		_tmp0 = vala_expression_is_non_null (vala_binary_expression_get_right (self));
	} else {
		_tmp0 = FALSE;
	}
	return _tmp0;
}


static gboolean vala_binary_expression_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaBinaryExpression * self;
	gboolean _tmp1;
	gboolean _tmp19;
	gboolean _tmp22;
	gboolean _tmp24;
	self = (ValaBinaryExpression*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	_tmp1 = FALSE;
	if (VALA_IS_BLOCK (vala_semantic_analyzer_get_current_symbol (analyzer))) {
		gboolean _tmp2;
		_tmp2 = FALSE;
		if (self->priv->_operator == VALA_BINARY_OPERATOR_AND) {
			_tmp2 = TRUE;
		} else {
			_tmp2 = self->priv->_operator == VALA_BINARY_OPERATOR_OR;
		}
		_tmp1 = (_tmp2);
	} else {
		_tmp1 = FALSE;
	}
	/* some expressions are not in a block,
	 for example, expressions in method contracts*/
	if (_tmp1) {
		ValaBlock* _tmp3;
		ValaBlock* old_insert_block;
		ValaBlock* _tmp4;
		char* _tmp6;
		ValaValueType* _tmp5;
		ValaLocalVariable* _tmp7;
		ValaLocalVariable* local;
		ValaDeclarationStatement* decl;
		ValaAssignment* _tmp9;
		ValaMemberAccess* _tmp8;
		ValaExpressionStatement* _tmp10;
		ValaExpressionStatement* right_stmt;
		ValaAssignment* _tmp13;
		ValaBooleanLiteral* _tmp12;
		ValaMemberAccess* _tmp11;
		ValaExpressionStatement* _tmp14;
		ValaExpressionStatement* stmt;
		ValaBlock* true_block;
		ValaBlock* false_block;
		ValaIfStatement* if_stmt;
		ValaBlock* _tmp17;
		ValaBlock* _tmp16;
		ValaMemberAccess* ma;
		gboolean _tmp18;
		_tmp3 = NULL;
		old_insert_block = (_tmp3 = analyzer->insert_block, (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3));
		_tmp4 = NULL;
		analyzer->insert_block = (_tmp4 = vala_expression_prepare_condition_split ((ValaExpression*) self, analyzer), (analyzer->insert_block == NULL) ? NULL : (analyzer->insert_block = (vala_code_node_unref (analyzer->insert_block), NULL)), _tmp4);
		/* convert conditional expression into if statement
		 required for flow analysis and exception handling*/
		_tmp6 = NULL;
		_tmp5 = NULL;
		_tmp7 = NULL;
		local = (_tmp7 = vala_local_variable_new ((ValaDataType*) (_tmp5 = vala_value_type_new (vala_data_type_get_data_type (analyzer->bool_type))), _tmp6 = vala_code_node_get_temp_name ((ValaCodeNode*) self), NULL, vala_code_node_get_source_reference ((ValaCodeNode*) self)), _tmp6 = (g_free (_tmp6), NULL), (_tmp5 == NULL) ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL)), _tmp7);
		decl = vala_declaration_statement_new ((ValaSymbol*) local, vala_code_node_get_source_reference ((ValaCodeNode*) self));
		vala_code_node_check ((ValaCodeNode*) decl, analyzer);
		_tmp9 = NULL;
		_tmp8 = NULL;
		_tmp10 = NULL;
		right_stmt = (_tmp10 = vala_expression_statement_new ((ValaExpression*) (_tmp9 = vala_assignment_new ((ValaExpression*) (_tmp8 = vala_member_access_new_simple (vala_symbol_get_name ((ValaSymbol*) local), vala_code_node_get_source_reference ((ValaCodeNode*) vala_binary_expression_get_right (self)))), vala_binary_expression_get_right (self), VALA_ASSIGNMENT_OPERATOR_SIMPLE, vala_code_node_get_source_reference ((ValaCodeNode*) vala_binary_expression_get_right (self)))), vala_code_node_get_source_reference ((ValaCodeNode*) vala_binary_expression_get_right (self))), (_tmp9 == NULL) ? NULL : (_tmp9 = (vala_code_node_unref (_tmp9), NULL)), (_tmp8 == NULL) ? NULL : (_tmp8 = (vala_code_node_unref (_tmp8), NULL)), _tmp10);
		_tmp13 = NULL;
		_tmp12 = NULL;
		_tmp11 = NULL;
		_tmp14 = NULL;
		stmt = (_tmp14 = vala_expression_statement_new ((ValaExpression*) (_tmp13 = vala_assignment_new ((ValaExpression*) (_tmp11 = vala_member_access_new_simple (vala_symbol_get_name ((ValaSymbol*) local), vala_code_node_get_source_reference ((ValaCodeNode*) vala_binary_expression_get_left (self)))), (ValaExpression*) (_tmp12 = vala_boolean_literal_new ((self->priv->_operator == VALA_BINARY_OPERATOR_OR), vala_code_node_get_source_reference ((ValaCodeNode*) vala_binary_expression_get_left (self)))), VALA_ASSIGNMENT_OPERATOR_SIMPLE, vala_code_node_get_source_reference ((ValaCodeNode*) vala_binary_expression_get_left (self)))), vala_code_node_get_source_reference ((ValaCodeNode*) vala_binary_expression_get_left (self))), (_tmp13 == NULL) ? NULL : (_tmp13 = (vala_code_node_unref (_tmp13), NULL)), (_tmp12 == NULL) ? NULL : (_tmp12 = (vala_code_node_unref (_tmp12), NULL)), (_tmp11 == NULL) ? NULL : (_tmp11 = (vala_code_node_unref (_tmp11), NULL)), _tmp14);
		true_block = vala_block_new (vala_code_node_get_source_reference ((ValaCodeNode*) self));
		false_block = vala_block_new (vala_code_node_get_source_reference ((ValaCodeNode*) self));
		if (self->priv->_operator == VALA_BINARY_OPERATOR_AND) {
			vala_block_add_statement (true_block, (ValaStatement*) right_stmt);
			vala_block_add_statement (false_block, (ValaStatement*) stmt);
		} else {
			vala_block_add_statement (true_block, (ValaStatement*) stmt);
			vala_block_add_statement (false_block, (ValaStatement*) right_stmt);
		}
		if_stmt = vala_if_statement_new (vala_binary_expression_get_left (self), true_block, false_block, vala_code_node_get_source_reference ((ValaCodeNode*) self));
		vala_expression_insert_statement ((ValaExpression*) self, analyzer->insert_block, (ValaStatement*) decl);
		vala_expression_insert_statement ((ValaExpression*) self, analyzer->insert_block, (ValaStatement*) if_stmt);
		if (!vala_code_node_check ((ValaCodeNode*) if_stmt, analyzer)) {
			gboolean _tmp15;
			return (_tmp15 = FALSE, (old_insert_block == NULL) ? NULL : (old_insert_block = (vala_code_node_unref (old_insert_block), NULL)), (local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL)), (decl == NULL) ? NULL : (decl = (vala_code_node_unref (decl), NULL)), (right_stmt == NULL) ? NULL : (right_stmt = (vala_code_node_unref (right_stmt), NULL)), (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), (true_block == NULL) ? NULL : (true_block = (vala_code_node_unref (true_block), NULL)), (false_block == NULL) ? NULL : (false_block = (vala_code_node_unref (false_block), NULL)), (if_stmt == NULL) ? NULL : (if_stmt = (vala_code_node_unref (if_stmt), NULL)), _tmp15);
		}
		_tmp17 = NULL;
		_tmp16 = NULL;
		analyzer->insert_block = (_tmp17 = (_tmp16 = old_insert_block, (_tmp16 == NULL) ? NULL : vala_code_node_ref (_tmp16)), (analyzer->insert_block == NULL) ? NULL : (analyzer->insert_block = (vala_code_node_unref (analyzer->insert_block), NULL)), _tmp17);
		ma = vala_member_access_new_simple (vala_symbol_get_name ((ValaSymbol*) local), vala_code_node_get_source_reference ((ValaCodeNode*) self));
		vala_expression_set_target_type ((ValaExpression*) ma, vala_expression_get_target_type ((ValaExpression*) self));
		vala_code_node_check ((ValaCodeNode*) ma, analyzer);
		vala_code_node_replace_expression (vala_code_node_get_parent_node ((ValaCodeNode*) self), (ValaExpression*) self, (ValaExpression*) ma);
		return (_tmp18 = TRUE, (old_insert_block == NULL) ? NULL : (old_insert_block = (vala_code_node_unref (old_insert_block), NULL)), (local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL)), (decl == NULL) ? NULL : (decl = (vala_code_node_unref (decl), NULL)), (right_stmt == NULL) ? NULL : (right_stmt = (vala_code_node_unref (right_stmt), NULL)), (stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)), (true_block == NULL) ? NULL : (true_block = (vala_code_node_unref (true_block), NULL)), (false_block == NULL) ? NULL : (false_block = (vala_code_node_unref (false_block), NULL)), (if_stmt == NULL) ? NULL : (if_stmt = (vala_code_node_unref (if_stmt), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp18);
	}
	_tmp19 = FALSE;
	if (!vala_code_node_check ((ValaCodeNode*) vala_binary_expression_get_left (self), analyzer)) {
		_tmp19 = TRUE;
	} else {
		_tmp19 = !vala_code_node_check ((ValaCodeNode*) vala_binary_expression_get_right (self), analyzer);
	}
	if (_tmp19) {
		/* if there were any errors in inner expressions, skip type check */
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	if (vala_expression_get_value_type (vala_binary_expression_get_left (self)) == NULL) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_binary_expression_get_left (self)), "invalid left operand");
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	_tmp22 = FALSE;
	if (self->priv->_operator != VALA_BINARY_OPERATOR_IN) {
		_tmp22 = vala_expression_get_value_type (vala_binary_expression_get_right (self)) == NULL;
	} else {
		_tmp22 = FALSE;
	}
	if (_tmp22) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_binary_expression_get_right (self)), "invalid right operand");
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	_tmp24 = FALSE;
	if (vala_data_type_get_data_type (vala_expression_get_value_type (vala_binary_expression_get_left (self))) == vala_data_type_get_data_type (analyzer->string_type)) {
		_tmp24 = self->priv->_operator == VALA_BINARY_OPERATOR_PLUS;
	} else {
		_tmp24 = FALSE;
	}
	if (_tmp24) {
		gboolean _tmp25;
		ValaDataType* _tmp27;
		gboolean _tmp28;
		_tmp25 = FALSE;
		if (vala_expression_get_value_type (vala_binary_expression_get_right (self)) == NULL) {
			_tmp25 = TRUE;
		} else {
			_tmp25 = vala_data_type_get_data_type (vala_expression_get_value_type (vala_binary_expression_get_right (self))) != vala_data_type_get_data_type (analyzer->string_type);
		}
		/* string concatenation*/
		if (_tmp25) {
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Operands must be strings");
			return FALSE;
		}
		_tmp27 = NULL;
		vala_expression_set_value_type ((ValaExpression*) self, _tmp27 = vala_data_type_copy (analyzer->string_type));
		(_tmp27 == NULL) ? NULL : (_tmp27 = (vala_code_node_unref (_tmp27), NULL));
		_tmp28 = FALSE;
		if (vala_expression_is_constant (vala_binary_expression_get_left (self))) {
			_tmp28 = vala_expression_is_constant (vala_binary_expression_get_right (self));
		} else {
			_tmp28 = FALSE;
		}
		if (_tmp28) {
			vala_data_type_set_value_owned (vala_expression_get_value_type ((ValaExpression*) self), FALSE);
		} else {
			vala_data_type_set_value_owned (vala_expression_get_value_type ((ValaExpression*) self), TRUE);
		}
	} else {
		gboolean _tmp29;
		gboolean _tmp30;
		gboolean _tmp31;
		_tmp29 = FALSE;
		_tmp30 = FALSE;
		_tmp31 = FALSE;
		if (self->priv->_operator == VALA_BINARY_OPERATOR_PLUS) {
			_tmp31 = TRUE;
		} else {
			_tmp31 = self->priv->_operator == VALA_BINARY_OPERATOR_MINUS;
		}
		if (_tmp31) {
			_tmp30 = TRUE;
		} else {
			_tmp30 = self->priv->_operator == VALA_BINARY_OPERATOR_MUL;
		}
		if (_tmp30) {
			_tmp29 = TRUE;
		} else {
			_tmp29 = self->priv->_operator == VALA_BINARY_OPERATOR_DIV;
		}
		if (_tmp29) {
			/* check for pointer arithmetic*/
			if (VALA_IS_POINTER_TYPE (vala_expression_get_value_type (vala_binary_expression_get_left (self)))) {
				ValaStruct* _tmp33;
				ValaTypeSymbol* _tmp32;
				ValaStruct* offset_type;
				gboolean _tmp34;
				_tmp33 = NULL;
				_tmp32 = NULL;
				offset_type = (_tmp33 = (_tmp32 = vala_data_type_get_data_type (vala_expression_get_value_type (vala_binary_expression_get_right (self))), VALA_IS_STRUCT (_tmp32) ? ((ValaStruct*) _tmp32) : NULL), (_tmp33 == NULL) ? NULL : vala_code_node_ref (_tmp33));
				_tmp34 = FALSE;
				if (offset_type != NULL) {
					_tmp34 = vala_struct_is_integer_type (offset_type);
				} else {
					_tmp34 = FALSE;
				}
				if (_tmp34) {
					gboolean _tmp35;
					_tmp35 = FALSE;
					if (self->priv->_operator == VALA_BINARY_OPERATOR_PLUS) {
						_tmp35 = TRUE;
					} else {
						_tmp35 = self->priv->_operator == VALA_BINARY_OPERATOR_MINUS;
					}
					if (_tmp35) {
						ValaDataType* _tmp36;
						/* pointer arithmetic: pointer +/- offset*/
						_tmp36 = NULL;
						vala_expression_set_value_type ((ValaExpression*) self, _tmp36 = vala_data_type_copy (vala_expression_get_value_type (vala_binary_expression_get_left (self))));
						(_tmp36 == NULL) ? NULL : (_tmp36 = (vala_code_node_unref (_tmp36), NULL));
					}
				} else {
					if (VALA_IS_POINTER_TYPE (vala_expression_get_value_type (vala_binary_expression_get_right (self)))) {
						/* pointer arithmetic: pointer - pointer*/
						vala_expression_set_value_type ((ValaExpression*) self, analyzer->size_t_type);
					}
				}
				(offset_type == NULL) ? NULL : (offset_type = (vala_code_node_unref (offset_type), NULL));
			}
			if (vala_expression_get_value_type ((ValaExpression*) self) == NULL) {
				ValaDataType* _tmp37;
				_tmp37 = NULL;
				vala_expression_set_value_type ((ValaExpression*) self, _tmp37 = vala_semantic_analyzer_get_arithmetic_result_type (analyzer, vala_expression_get_value_type (vala_binary_expression_get_left (self)), vala_expression_get_value_type (vala_binary_expression_get_right (self))));
				(_tmp37 == NULL) ? NULL : (_tmp37 = (vala_code_node_unref (_tmp37), NULL));
			}
			if (vala_expression_get_value_type ((ValaExpression*) self) == NULL) {
				char* _tmp40;
				char* _tmp39;
				char* _tmp38;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				_tmp40 = NULL;
				_tmp39 = NULL;
				_tmp38 = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp40 = g_strdup_printf ("Arithmetic operation not supported for types `%s' and `%s'", _tmp38 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_binary_expression_get_left (self))), _tmp39 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_binary_expression_get_right (self)))));
				_tmp40 = (g_free (_tmp40), NULL);
				_tmp39 = (g_free (_tmp39), NULL);
				_tmp38 = (g_free (_tmp38), NULL);
				return FALSE;
			}
		} else {
			gboolean _tmp42;
			gboolean _tmp43;
			gboolean _tmp44;
			_tmp42 = FALSE;
			_tmp43 = FALSE;
			_tmp44 = FALSE;
			if (self->priv->_operator == VALA_BINARY_OPERATOR_MOD) {
				_tmp44 = TRUE;
			} else {
				_tmp44 = self->priv->_operator == VALA_BINARY_OPERATOR_SHIFT_LEFT;
			}
			if (_tmp44) {
				_tmp43 = TRUE;
			} else {
				_tmp43 = self->priv->_operator == VALA_BINARY_OPERATOR_SHIFT_RIGHT;
			}
			if (_tmp43) {
				_tmp42 = TRUE;
			} else {
				_tmp42 = self->priv->_operator == VALA_BINARY_OPERATOR_BITWISE_XOR;
			}
			if (_tmp42) {
				ValaDataType* _tmp45;
				_tmp45 = NULL;
				vala_expression_set_value_type ((ValaExpression*) self, _tmp45 = vala_semantic_analyzer_get_arithmetic_result_type (analyzer, vala_expression_get_value_type (vala_binary_expression_get_left (self)), vala_expression_get_value_type (vala_binary_expression_get_right (self))));
				(_tmp45 == NULL) ? NULL : (_tmp45 = (vala_code_node_unref (_tmp45), NULL));
				if (vala_expression_get_value_type ((ValaExpression*) self) == NULL) {
					char* _tmp48;
					char* _tmp47;
					char* _tmp46;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp48 = NULL;
					_tmp47 = NULL;
					_tmp46 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp48 = g_strdup_printf ("Arithmetic operation not supported for types `%s' and `%s'", _tmp46 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_binary_expression_get_left (self))), _tmp47 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_binary_expression_get_right (self)))));
					_tmp48 = (g_free (_tmp48), NULL);
					_tmp47 = (g_free (_tmp47), NULL);
					_tmp46 = (g_free (_tmp46), NULL);
					return FALSE;
				}
			} else {
				gboolean _tmp50;
				gboolean _tmp51;
				gboolean _tmp52;
				_tmp50 = FALSE;
				_tmp51 = FALSE;
				_tmp52 = FALSE;
				if (self->priv->_operator == VALA_BINARY_OPERATOR_LESS_THAN) {
					_tmp52 = TRUE;
				} else {
					_tmp52 = self->priv->_operator == VALA_BINARY_OPERATOR_GREATER_THAN;
				}
				if (_tmp52) {
					_tmp51 = TRUE;
				} else {
					_tmp51 = self->priv->_operator == VALA_BINARY_OPERATOR_LESS_THAN_OR_EQUAL;
				}
				if (_tmp51) {
					_tmp50 = TRUE;
				} else {
					_tmp50 = self->priv->_operator == VALA_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL;
				}
				if (_tmp50) {
					gboolean _tmp53;
					_tmp53 = FALSE;
					if (vala_data_type_compatible (vala_expression_get_value_type (vala_binary_expression_get_left (self)), analyzer->string_type)) {
						_tmp53 = vala_data_type_compatible (vala_expression_get_value_type (vala_binary_expression_get_right (self)), analyzer->string_type);
					} else {
						_tmp53 = FALSE;
					}
					if (_tmp53) {
					} else {
						gboolean _tmp54;
						_tmp54 = FALSE;
						if (VALA_IS_POINTER_TYPE (vala_expression_get_value_type (vala_binary_expression_get_left (self)))) {
							_tmp54 = VALA_IS_POINTER_TYPE (vala_expression_get_value_type (vala_binary_expression_get_right (self)));
						} else {
							_tmp54 = FALSE;
						}
						if (_tmp54) {
						} else {
							ValaDataType* resulting_type;
							/* pointer arithmetic*/
							resulting_type = vala_semantic_analyzer_get_arithmetic_result_type (analyzer, vala_expression_get_value_type (vala_binary_expression_get_left (self)), vala_expression_get_value_type (vala_binary_expression_get_right (self)));
							if (resulting_type == NULL) {
								char* _tmp57;
								char* _tmp56;
								char* _tmp55;
								gboolean _tmp58;
								vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
								_tmp57 = NULL;
								_tmp56 = NULL;
								_tmp55 = NULL;
								vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp57 = g_strdup_printf ("Relational operation not supported for types `%s' and `%s'", _tmp55 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_binary_expression_get_left (self))), _tmp56 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_binary_expression_get_right (self)))));
								_tmp57 = (g_free (_tmp57), NULL);
								_tmp56 = (g_free (_tmp56), NULL);
								_tmp55 = (g_free (_tmp55), NULL);
								return (_tmp58 = FALSE, (resulting_type == NULL) ? NULL : (resulting_type = (vala_code_node_unref (resulting_type), NULL)), _tmp58);
							}
							(resulting_type == NULL) ? NULL : (resulting_type = (vala_code_node_unref (resulting_type), NULL));
						}
					}
					vala_expression_set_value_type ((ValaExpression*) self, analyzer->bool_type);
				} else {
					gboolean _tmp59;
					_tmp59 = FALSE;
					if (self->priv->_operator == VALA_BINARY_OPERATOR_EQUALITY) {
						_tmp59 = TRUE;
					} else {
						_tmp59 = self->priv->_operator == VALA_BINARY_OPERATOR_INEQUALITY;
					}
					if (_tmp59) {
						gboolean _tmp60;
						gboolean _tmp65;
						_tmp60 = FALSE;
						if (!vala_data_type_compatible (vala_expression_get_value_type (vala_binary_expression_get_right (self)), vala_expression_get_value_type (vala_binary_expression_get_left (self)))) {
							_tmp60 = !vala_data_type_compatible (vala_expression_get_value_type (vala_binary_expression_get_left (self)), vala_expression_get_value_type (vala_binary_expression_get_right (self)));
						} else {
							_tmp60 = FALSE;
						}
						/* relational operation */
						if (_tmp60) {
							char* _tmp63;
							char* _tmp62;
							char* _tmp61;
							_tmp63 = NULL;
							_tmp62 = NULL;
							_tmp61 = NULL;
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp63 = g_strdup_printf ("Equality operation: `%s' and `%s' are incompatible", _tmp61 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_binary_expression_get_right (self))), _tmp62 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_binary_expression_get_left (self)))));
							_tmp63 = (g_free (_tmp63), NULL);
							_tmp62 = (g_free (_tmp62), NULL);
							_tmp61 = (g_free (_tmp61), NULL);
							vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
							return FALSE;
						}
						_tmp65 = FALSE;
						if (vala_data_type_compatible (vala_expression_get_value_type (vala_binary_expression_get_left (self)), analyzer->string_type)) {
							_tmp65 = vala_data_type_compatible (vala_expression_get_value_type (vala_binary_expression_get_right (self)), analyzer->string_type);
						} else {
							_tmp65 = FALSE;
						}
						if (_tmp65) {
						}
						/* string comparison*/
						vala_expression_set_value_type ((ValaExpression*) self, analyzer->bool_type);
					} else {
						gboolean _tmp66;
						_tmp66 = FALSE;
						if (self->priv->_operator == VALA_BINARY_OPERATOR_BITWISE_AND) {
							_tmp66 = TRUE;
						} else {
							_tmp66 = self->priv->_operator == VALA_BINARY_OPERATOR_BITWISE_OR;
						}
						if (_tmp66) {
							/* integer type or flags type*/
							vala_expression_set_value_type ((ValaExpression*) self, vala_expression_get_value_type (vala_binary_expression_get_left (self)));
						} else {
							gboolean _tmp67;
							_tmp67 = FALSE;
							if (self->priv->_operator == VALA_BINARY_OPERATOR_AND) {
								_tmp67 = TRUE;
							} else {
								_tmp67 = self->priv->_operator == VALA_BINARY_OPERATOR_OR;
							}
							if (_tmp67) {
								gboolean _tmp68;
								_tmp68 = FALSE;
								if (!vala_data_type_compatible (vala_expression_get_value_type (vala_binary_expression_get_left (self)), analyzer->bool_type)) {
									_tmp68 = TRUE;
								} else {
									_tmp68 = !vala_data_type_compatible (vala_expression_get_value_type (vala_binary_expression_get_right (self)), analyzer->bool_type);
								}
								if (_tmp68) {
									vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
									vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Operands must be boolean");
								}
								vala_expression_set_value_type ((ValaExpression*) self, analyzer->bool_type);
							} else {
								if (self->priv->_operator == VALA_BINARY_OPERATOR_IN) {
									gboolean _tmp69;
									_tmp69 = FALSE;
									if (vala_data_type_compatible (vala_expression_get_value_type (vala_binary_expression_get_left (self)), analyzer->int_type)) {
										_tmp69 = vala_data_type_compatible (vala_expression_get_value_type (vala_binary_expression_get_right (self)), analyzer->int_type);
									} else {
										_tmp69 = FALSE;
									}
									if (_tmp69) {
									} else {
										ValaSymbol* _tmp70;
										ValaMethod* contains_method;
										GeeList* _tmp74;
										gboolean _tmp75;
										ValaMemberAccess* _tmp82;
										ValaMethodCall* _tmp83;
										ValaMethodCall* contains_call;
										gboolean _tmp84;
										/* integers or enums
										 otherwise require a bool contains () method*/
										_tmp70 = NULL;
										contains_method = (_tmp70 = vala_data_type_get_member (vala_expression_get_value_type (vala_binary_expression_get_right (self)), "contains"), VALA_IS_METHOD (_tmp70) ? ((ValaMethod*) _tmp70) : NULL);
										if (contains_method == NULL) {
											char* _tmp72;
											char* _tmp71;
											gboolean _tmp73;
											_tmp72 = NULL;
											_tmp71 = NULL;
											vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp72 = g_strdup_printf ("`%s' does not have a `contains' method", _tmp71 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_binary_expression_get_right (self)))));
											_tmp72 = (g_free (_tmp72), NULL);
											_tmp71 = (g_free (_tmp71), NULL);
											vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
											return (_tmp73 = FALSE, (contains_method == NULL) ? NULL : (contains_method = (vala_code_node_unref (contains_method), NULL)), _tmp73);
										}
										_tmp74 = NULL;
										if ((_tmp75 = gee_collection_get_size ((GeeCollection*) (_tmp74 = vala_method_get_parameters (contains_method))) != 1, (_tmp74 == NULL) ? NULL : (_tmp74 = (gee_collection_object_unref (_tmp74), NULL)), _tmp75)) {
											char* _tmp77;
											char* _tmp76;
											gboolean _tmp78;
											_tmp77 = NULL;
											_tmp76 = NULL;
											vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp77 = g_strdup_printf ("`%s' must have one parameter", _tmp76 = vala_symbol_get_full_name ((ValaSymbol*) contains_method)));
											_tmp77 = (g_free (_tmp77), NULL);
											_tmp76 = (g_free (_tmp76), NULL);
											vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
											return (_tmp78 = FALSE, (contains_method == NULL) ? NULL : (contains_method = (vala_code_node_unref (contains_method), NULL)), _tmp78);
										}
										if (!vala_data_type_compatible (vala_method_get_return_type (contains_method), analyzer->bool_type)) {
											char* _tmp80;
											char* _tmp79;
											gboolean _tmp81;
											_tmp80 = NULL;
											_tmp79 = NULL;
											vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp80 = g_strdup_printf ("`%s' must return a boolean value", _tmp79 = vala_symbol_get_full_name ((ValaSymbol*) contains_method)));
											_tmp80 = (g_free (_tmp80), NULL);
											_tmp79 = (g_free (_tmp79), NULL);
											vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
											return (_tmp81 = FALSE, (contains_method == NULL) ? NULL : (contains_method = (vala_code_node_unref (contains_method), NULL)), _tmp81);
										}
										_tmp82 = NULL;
										_tmp83 = NULL;
										contains_call = (_tmp83 = vala_method_call_new ((ValaExpression*) (_tmp82 = vala_member_access_new (vala_binary_expression_get_right (self), "contains", NULL)), NULL), (_tmp82 == NULL) ? NULL : (_tmp82 = (vala_code_node_unref (_tmp82), NULL)), _tmp83);
										vala_method_call_add_argument (contains_call, vala_binary_expression_get_left (self));
										vala_code_node_replace_expression (vala_code_node_get_parent_node ((ValaCodeNode*) self), (ValaExpression*) self, (ValaExpression*) contains_call);
										return (_tmp84 = vala_code_node_check ((ValaCodeNode*) contains_call, analyzer), (contains_method == NULL) ? NULL : (contains_method = (vala_code_node_unref (contains_method), NULL)), (contains_call == NULL) ? NULL : (contains_call = (vala_code_node_unref (contains_call), NULL)), _tmp84);
									}
									vala_expression_set_value_type ((ValaExpression*) self, analyzer->bool_type);
								} else {
									g_assert_not_reached ();
								}
							}
						}
					}
				}
			}
		}
	}
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


static void vala_binary_expression_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaBinaryExpression * self;
	self = (ValaBinaryExpression*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_defined_variables ((ValaCodeNode*) vala_binary_expression_get_left (self), collection);
	vala_code_node_get_defined_variables ((ValaCodeNode*) vala_binary_expression_get_right (self), collection);
}


static void vala_binary_expression_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaBinaryExpression * self;
	self = (ValaBinaryExpression*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_used_variables ((ValaCodeNode*) vala_binary_expression_get_left (self), collection);
	vala_code_node_get_used_variables ((ValaCodeNode*) vala_binary_expression_get_right (self), collection);
}


ValaBinaryOperator vala_binary_expression_get_operator (ValaBinaryExpression* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_operator;
}


void vala_binary_expression_set_operator (ValaBinaryExpression* self, ValaBinaryOperator value) {
	g_return_if_fail (self != NULL);
	self->priv->_operator = value;
}


ValaExpression* vala_binary_expression_get_left (ValaBinaryExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_left;
}


void vala_binary_expression_set_left (ValaBinaryExpression* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_left = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_left == NULL) ? NULL : (self->priv->_left = (vala_code_node_unref (self->priv->_left), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_left, (ValaCodeNode*) self);
}


ValaExpression* vala_binary_expression_get_right (ValaBinaryExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_right;
}


void vala_binary_expression_set_right (ValaBinaryExpression* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_right = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_right == NULL) ? NULL : (self->priv->_right = (vala_code_node_unref (self->priv->_right), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_right, (ValaCodeNode*) self);
}


static void vala_binary_expression_class_init (ValaBinaryExpressionClass * klass) {
	vala_binary_expression_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_binary_expression_finalize;
	g_type_class_add_private (klass, sizeof (ValaBinaryExpressionPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_binary_expression_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_binary_expression_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_binary_expression_real_replace_expression;
	VALA_CODE_NODE_CLASS (klass)->to_string = vala_binary_expression_real_to_string;
	VALA_EXPRESSION_CLASS (klass)->is_constant = vala_binary_expression_real_is_constant;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_binary_expression_real_is_pure;
	VALA_EXPRESSION_CLASS (klass)->is_non_null = vala_binary_expression_real_is_non_null;
	VALA_CODE_NODE_CLASS (klass)->check = vala_binary_expression_real_check;
	VALA_CODE_NODE_CLASS (klass)->get_defined_variables = vala_binary_expression_real_get_defined_variables;
	VALA_CODE_NODE_CLASS (klass)->get_used_variables = vala_binary_expression_real_get_used_variables;
}


static void vala_binary_expression_instance_init (ValaBinaryExpression * self) {
	self->priv = VALA_BINARY_EXPRESSION_GET_PRIVATE (self);
}


static void vala_binary_expression_finalize (ValaCodeNode* obj) {
	ValaBinaryExpression * self;
	self = VALA_BINARY_EXPRESSION (obj);
	(self->priv->_left == NULL) ? NULL : (self->priv->_left = (vala_code_node_unref (self->priv->_left), NULL));
	(self->priv->_right == NULL) ? NULL : (self->priv->_right = (vala_code_node_unref (self->priv->_right), NULL));
	VALA_CODE_NODE_CLASS (vala_binary_expression_parent_class)->finalize (obj);
}


GType vala_binary_expression_get_type (void) {
	static GType vala_binary_expression_type_id = 0;
	if (vala_binary_expression_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaBinaryExpressionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_binary_expression_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaBinaryExpression), 0, (GInstanceInitFunc) vala_binary_expression_instance_init, NULL };
		vala_binary_expression_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaBinaryExpression", &g_define_type_info, 0);
	}
	return vala_binary_expression_type_id;
}



GType vala_binary_operator_get_type (void) {
	static GType vala_binary_operator_type_id = 0;
	if (G_UNLIKELY (vala_binary_operator_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_BINARY_OPERATOR_NONE, "VALA_BINARY_OPERATOR_NONE", "none"}, {VALA_BINARY_OPERATOR_PLUS, "VALA_BINARY_OPERATOR_PLUS", "plus"}, {VALA_BINARY_OPERATOR_MINUS, "VALA_BINARY_OPERATOR_MINUS", "minus"}, {VALA_BINARY_OPERATOR_MUL, "VALA_BINARY_OPERATOR_MUL", "mul"}, {VALA_BINARY_OPERATOR_DIV, "VALA_BINARY_OPERATOR_DIV", "div"}, {VALA_BINARY_OPERATOR_MOD, "VALA_BINARY_OPERATOR_MOD", "mod"}, {VALA_BINARY_OPERATOR_SHIFT_LEFT, "VALA_BINARY_OPERATOR_SHIFT_LEFT", "shift-left"}, {VALA_BINARY_OPERATOR_SHIFT_RIGHT, "VALA_BINARY_OPERATOR_SHIFT_RIGHT", "shift-right"}, {VALA_BINARY_OPERATOR_LESS_THAN, "VALA_BINARY_OPERATOR_LESS_THAN", "less-than"}, {VALA_BINARY_OPERATOR_GREATER_THAN, "VALA_BINARY_OPERATOR_GREATER_THAN", "greater-than"}, {VALA_BINARY_OPERATOR_LESS_THAN_OR_EQUAL, "VALA_BINARY_OPERATOR_LESS_THAN_OR_EQUAL", "less-than-or-equal"}, {VALA_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL, "VALA_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL", "greater-than-or-equal"}, {VALA_BINARY_OPERATOR_EQUALITY, "VALA_BINARY_OPERATOR_EQUALITY", "equality"}, {VALA_BINARY_OPERATOR_INEQUALITY, "VALA_BINARY_OPERATOR_INEQUALITY", "inequality"}, {VALA_BINARY_OPERATOR_BITWISE_AND, "VALA_BINARY_OPERATOR_BITWISE_AND", "bitwise-and"}, {VALA_BINARY_OPERATOR_BITWISE_OR, "VALA_BINARY_OPERATOR_BITWISE_OR", "bitwise-or"}, {VALA_BINARY_OPERATOR_BITWISE_XOR, "VALA_BINARY_OPERATOR_BITWISE_XOR", "bitwise-xor"}, {VALA_BINARY_OPERATOR_AND, "VALA_BINARY_OPERATOR_AND", "and"}, {VALA_BINARY_OPERATOR_OR, "VALA_BINARY_OPERATOR_OR", "or"}, {VALA_BINARY_OPERATOR_IN, "VALA_BINARY_OPERATOR_IN", "in"}, {0, NULL, NULL}};
		vala_binary_operator_type_id = g_enum_register_static ("ValaBinaryOperator", values);
	}
	return vala_binary_operator_type_id;
}




