/* valalambdaexpression.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valalambdaexpression.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valablock.h>
#include <vala/valamethod.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valadelegatetype.h>
#include <vala/valareport.h>
#include <vala/valamember.h>
#include <vala/valadelegate.h>
#include <vala/valadatatype.h>
#include <vala/valasymbol.h>
#include <vala/valaformalparameter.h>
#include <vala/valascope.h>
#include <vala/valareturnstatement.h>
#include <vala/valastatement.h>
#include <vala/valaexpressionstatement.h>
#include <vala/valamethodtype.h>




struct _ValaLambdaExpressionPrivate {
	ValaExpression* _expression_body;
	ValaBlock* _statement_body;
	ValaMethod* _method;
	GeeList* parameters;
};

#define VALA_LAMBDA_EXPRESSION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_LAMBDA_EXPRESSION, ValaLambdaExpressionPrivate))
enum  {
	VALA_LAMBDA_EXPRESSION_DUMMY_PROPERTY
};
static void vala_lambda_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_lambda_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_lambda_expression_real_is_pure (ValaExpression* base);
static char* vala_lambda_expression_get_lambda_name (ValaLambdaExpression* self, ValaSemanticAnalyzer* analyzer);
static gboolean vala_lambda_expression_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_lambda_expression_parent_class = NULL;
static void vala_lambda_expression_finalize (ValaCodeNode* obj);



/**
 * Creates a new lambda expression.
 *
 * @param expression_body  expression body
 * @param source_reference reference to source code
 * @return                 newly created lambda expression
 */
ValaLambdaExpression* vala_lambda_expression_construct (GType object_type, ValaExpression* expression_body, ValaSourceReference* source_reference) {
	ValaLambdaExpression* self;
	g_return_val_if_fail (expression_body != NULL, NULL);
	g_return_val_if_fail (source_reference != NULL, NULL);
	self = (ValaLambdaExpression*) g_type_create_instance (object_type);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	vala_lambda_expression_set_expression_body (self, expression_body);
	return self;
}


ValaLambdaExpression* vala_lambda_expression_new (ValaExpression* expression_body, ValaSourceReference* source_reference) {
	return vala_lambda_expression_construct (VALA_TYPE_LAMBDA_EXPRESSION, expression_body, source_reference);
}


/**
 * Creates a new lambda expression with statement body.
 *
 * @param statement_body   statement body
 * @param source_reference reference to source code
 * @return                 newly created lambda expression
 */
ValaLambdaExpression* vala_lambda_expression_construct_with_statement_body (GType object_type, ValaBlock* statement_body, ValaSourceReference* source_reference) {
	ValaLambdaExpression* self;
	g_return_val_if_fail (statement_body != NULL, NULL);
	g_return_val_if_fail (source_reference != NULL, NULL);
	self = (ValaLambdaExpression*) g_type_create_instance (object_type);
	vala_lambda_expression_set_statement_body (self, statement_body);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	return self;
}


ValaLambdaExpression* vala_lambda_expression_new_with_statement_body (ValaBlock* statement_body, ValaSourceReference* source_reference) {
	return vala_lambda_expression_construct_with_statement_body (VALA_TYPE_LAMBDA_EXPRESSION, statement_body, source_reference);
}


/**
 * Appends implicitly typed parameter.
 *
 * @param param parameter name
 */
void vala_lambda_expression_add_parameter (ValaLambdaExpression* self, const char* param) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (param != NULL);
	gee_collection_add ((GeeCollection*) self->priv->parameters, param);
}


/**
 * Returns copy of parameter list.
 *
 * @return parameter list
 */
GeeList* vala_lambda_expression_get_parameters (ValaLambdaExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, self->priv->parameters);
}


static void vala_lambda_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaLambdaExpression * self;
	self = (ValaLambdaExpression*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_lambda_expression (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


static void vala_lambda_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaLambdaExpression * self;
	self = (ValaLambdaExpression*) base;
	g_return_if_fail (visitor != NULL);
	if (self->priv->_method == NULL) {
		if (self->priv->_expression_body != NULL) {
			vala_code_node_accept ((ValaCodeNode*) self->priv->_expression_body, visitor);
			vala_code_visitor_visit_end_full_expression (visitor, self->priv->_expression_body);
		} else {
			if (self->priv->_statement_body != NULL) {
				vala_code_node_accept ((ValaCodeNode*) self->priv->_statement_body, visitor);
			}
		}
	} else {
		vala_code_node_accept ((ValaCodeNode*) self->priv->_method, visitor);
	}
}


static gboolean vala_lambda_expression_real_is_pure (ValaExpression* base) {
	ValaLambdaExpression * self;
	self = (ValaLambdaExpression*) base;
	return FALSE;
}


static char* vala_lambda_expression_get_lambda_name (ValaLambdaExpression* self, ValaSemanticAnalyzer* analyzer) {
	char* result;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (analyzer != NULL, NULL);
	result = g_strdup_printf ("__lambda%d", analyzer->next_lambda_id);
	analyzer->next_lambda_id++;
	return result;
}


static gboolean vala_lambda_expression_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaLambdaExpression * self;
	gboolean in_instance_method;
	ValaMethod* current_method;
	ValaDelegate* _tmp2;
	ValaDelegate* cb;
	ValaMethod* _tmp4;
	char* _tmp3;
	gboolean _tmp5;
	GeeList* lambda_params;
	GeeIterator* lambda_param_it;
	ValaMethodType* _tmp11;
	gboolean _tmp12;
	self = (ValaLambdaExpression*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	if (!(VALA_IS_DELEGATE_TYPE (vala_expression_get_target_type ((ValaExpression*) self)))) {
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "lambda expression not allowed in this context");
		return FALSE;
	}
	in_instance_method = FALSE;
	current_method = vala_semantic_analyzer_find_current_method (analyzer);
	if (current_method != NULL) {
		in_instance_method = (vala_method_get_binding (current_method) == MEMBER_BINDING_INSTANCE);
	} else {
		in_instance_method = vala_semantic_analyzer_is_in_constructor (analyzer);
	}
	_tmp2 = NULL;
	cb = (_tmp2 = VALA_DELEGATE (vala_delegate_type_get_delegate_symbol ((VALA_DELEGATE_TYPE (vala_expression_get_target_type ((ValaExpression*) self))))), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
	_tmp4 = NULL;
	_tmp3 = NULL;
	vala_lambda_expression_set_method (self, _tmp4 = vala_method_new (_tmp3 = vala_lambda_expression_get_lambda_name (self, analyzer), vala_delegate_get_return_type (cb), NULL));
	(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL));
	_tmp3 = (g_free (_tmp3), NULL);
	_tmp5 = FALSE;
	if (!vala_delegate_get_has_target (cb)) {
		_tmp5 = TRUE;
	} else {
		_tmp5 = !in_instance_method;
	}
	if (_tmp5) {
		vala_method_set_binding (self->priv->_method, MEMBER_BINDING_STATIC);
	}
	vala_symbol_set_owner ((ValaSymbol*) self->priv->_method, vala_symbol_get_scope (vala_semantic_analyzer_get_current_symbol (analyzer)));
	lambda_params = vala_lambda_expression_get_parameters (self);
	lambda_param_it = gee_iterable_iterator ((GeeIterable*) lambda_params);
	{
		GeeList* _tmp6;
		GeeIterator* _tmp7;
		GeeIterator* cb_param_it;
		_tmp6 = NULL;
		_tmp7 = NULL;
		cb_param_it = (_tmp7 = gee_iterable_iterator ((GeeIterable*) (_tmp6 = vala_delegate_get_parameters (cb))), (_tmp6 == NULL) ? NULL : (_tmp6 = (gee_collection_object_unref (_tmp6), NULL)), _tmp7);
		while (gee_iterator_next (cb_param_it)) {
			ValaFormalParameter* cb_param;
			char* lambda_param;
			ValaFormalParameter* param;
			cb_param = (ValaFormalParameter*) gee_iterator_get (cb_param_it);
			if (!gee_iterator_next (lambda_param_it)) {
				/* lambda expressions are allowed to have less parameters */
				(cb_param == NULL) ? NULL : (cb_param = (vala_code_node_unref (cb_param), NULL));
				break;
			}
			lambda_param = (char*) gee_iterator_get (lambda_param_it);
			param = vala_formal_parameter_new (lambda_param, vala_formal_parameter_get_parameter_type (cb_param), NULL);
			vala_method_add_parameter (self->priv->_method, param);
			(cb_param == NULL) ? NULL : (cb_param = (vala_code_node_unref (cb_param), NULL));
			lambda_param = (g_free (lambda_param), NULL);
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		(cb_param_it == NULL) ? NULL : (cb_param_it = (gee_collection_object_unref (cb_param_it), NULL));
	}
	if (gee_iterator_next (lambda_param_it)) {
		gboolean _tmp8;
		/* lambda expressions may not expect more parameters */
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "lambda expression: too many parameters");
		return (_tmp8 = FALSE, (current_method == NULL) ? NULL : (current_method = (vala_code_node_unref (current_method), NULL)), (cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL)), (lambda_params == NULL) ? NULL : (lambda_params = (gee_collection_object_unref (lambda_params), NULL)), (lambda_param_it == NULL) ? NULL : (lambda_param_it = (gee_collection_object_unref (lambda_param_it), NULL)), _tmp8);
	}
	if (self->priv->_expression_body != NULL) {
		ValaBlock* block;
		block = vala_block_new (vala_code_node_get_source_reference ((ValaCodeNode*) self));
		vala_scope_set_parent_scope (vala_symbol_get_scope ((ValaSymbol*) block), vala_symbol_get_scope ((ValaSymbol*) self->priv->_method));
		if (vala_data_type_get_data_type (vala_method_get_return_type (self->priv->_method)) != NULL) {
			ValaReturnStatement* _tmp9;
			_tmp9 = NULL;
			vala_block_add_statement (block, (ValaStatement*) (_tmp9 = vala_return_statement_new (self->priv->_expression_body, vala_code_node_get_source_reference ((ValaCodeNode*) self))));
			(_tmp9 == NULL) ? NULL : (_tmp9 = (vala_code_node_unref (_tmp9), NULL));
		} else {
			ValaExpressionStatement* _tmp10;
			_tmp10 = NULL;
			vala_block_add_statement (block, (ValaStatement*) (_tmp10 = vala_expression_statement_new (self->priv->_expression_body, vala_code_node_get_source_reference ((ValaCodeNode*) self))));
			(_tmp10 == NULL) ? NULL : (_tmp10 = (vala_code_node_unref (_tmp10), NULL));
		}
		vala_method_set_body (self->priv->_method, block);
		(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
	} else {
		vala_method_set_body (self->priv->_method, self->priv->_statement_body);
	}
	vala_symbol_set_owner ((ValaSymbol*) vala_method_get_body (self->priv->_method), vala_symbol_get_scope ((ValaSymbol*) self->priv->_method));
	/* lambda expressions should be usable like MemberAccess of a method */
	vala_expression_set_symbol_reference ((ValaExpression*) self, (ValaSymbol*) self->priv->_method);
	if (self->priv->_method == NULL) {
		if (self->priv->_expression_body != NULL) {
			vala_code_node_check ((ValaCodeNode*) self->priv->_expression_body, analyzer);
		} else {
			if (self->priv->_statement_body != NULL) {
				vala_code_node_check ((ValaCodeNode*) self->priv->_statement_body, analyzer);
			}
		}
	} else {
		vala_code_node_check ((ValaCodeNode*) self->priv->_method, analyzer);
	}
	_tmp11 = NULL;
	vala_expression_set_value_type ((ValaExpression*) self, (ValaDataType*) (_tmp11 = vala_method_type_new (self->priv->_method)));
	(_tmp11 == NULL) ? NULL : (_tmp11 = (vala_code_node_unref (_tmp11), NULL));
	return (_tmp12 = !vala_code_node_get_error ((ValaCodeNode*) self), (current_method == NULL) ? NULL : (current_method = (vala_code_node_unref (current_method), NULL)), (cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL)), (lambda_params == NULL) ? NULL : (lambda_params = (gee_collection_object_unref (lambda_params), NULL)), (lambda_param_it == NULL) ? NULL : (lambda_param_it = (gee_collection_object_unref (lambda_param_it), NULL)), _tmp12);
}


ValaExpression* vala_lambda_expression_get_expression_body (ValaLambdaExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_expression_body;
}


void vala_lambda_expression_set_expression_body (ValaLambdaExpression* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_expression_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_expression_body == NULL) ? NULL : (self->priv->_expression_body = (vala_code_node_unref (self->priv->_expression_body), NULL)), _tmp2);
}


ValaBlock* vala_lambda_expression_get_statement_body (ValaLambdaExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_statement_body;
}


void vala_lambda_expression_set_statement_body (ValaLambdaExpression* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_statement_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_statement_body == NULL) ? NULL : (self->priv->_statement_body = (vala_code_node_unref (self->priv->_statement_body), NULL)), _tmp2);
}


ValaMethod* vala_lambda_expression_get_method (ValaLambdaExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_method;
}


void vala_lambda_expression_set_method (ValaLambdaExpression* self, ValaMethod* value) {
	ValaMethod* _tmp2;
	ValaMethod* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_method = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_method == NULL) ? NULL : (self->priv->_method = (vala_code_node_unref (self->priv->_method), NULL)), _tmp2);
}


static void vala_lambda_expression_class_init (ValaLambdaExpressionClass * klass) {
	vala_lambda_expression_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_lambda_expression_finalize;
	g_type_class_add_private (klass, sizeof (ValaLambdaExpressionPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_lambda_expression_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_lambda_expression_real_accept_children;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_lambda_expression_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->check = vala_lambda_expression_real_check;
}


static void vala_lambda_expression_instance_init (ValaLambdaExpression * self) {
	self->priv = VALA_LAMBDA_EXPRESSION_GET_PRIVATE (self);
	self->priv->parameters = (GeeList*) gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
}


static void vala_lambda_expression_finalize (ValaCodeNode* obj) {
	ValaLambdaExpression * self;
	self = VALA_LAMBDA_EXPRESSION (obj);
	(self->priv->_expression_body == NULL) ? NULL : (self->priv->_expression_body = (vala_code_node_unref (self->priv->_expression_body), NULL));
	(self->priv->_statement_body == NULL) ? NULL : (self->priv->_statement_body = (vala_code_node_unref (self->priv->_statement_body), NULL));
	(self->priv->_method == NULL) ? NULL : (self->priv->_method = (vala_code_node_unref (self->priv->_method), NULL));
	(self->priv->parameters == NULL) ? NULL : (self->priv->parameters = (gee_collection_object_unref (self->priv->parameters), NULL));
	VALA_CODE_NODE_CLASS (vala_lambda_expression_parent_class)->finalize (obj);
}


GType vala_lambda_expression_get_type (void) {
	static GType vala_lambda_expression_type_id = 0;
	if (vala_lambda_expression_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaLambdaExpressionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_lambda_expression_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaLambdaExpression), 0, (GInstanceInitFunc) vala_lambda_expression_instance_init, NULL };
		vala_lambda_expression_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaLambdaExpression", &g_define_type_info, 0);
	}
	return vala_lambda_expression_type_id;
}




