/* valaenum.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaenum.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <gee/readonlylist.h>
#include <vala/valasymbol.h>
#include <vala/valaenumvalue.h>
#include <vala/valamethod.h>
#include <vala/valasourcereference.h>
#include <vala/valascope.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valaattribute.h>
#include <vala/valasourcefile.h>
#include <vala/valacreationmethod.h>
#include <vala/valareport.h>
#include <vala/valamember.h>
#include <vala/valaformalparameter.h>
#include <vala/valaenumvaluetype.h>
#include <vala/valadatatype.h>




struct _ValaEnumPrivate {
	gboolean _is_flags;
	gboolean _has_type_id;
	GeeList* values;
	GeeList* methods;
	char* cname;
	char* cprefix;
	char* lower_case_cprefix;
	char* lower_case_csuffix;
	char* type_id;
};

#define VALA_ENUM_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_ENUM, ValaEnumPrivate))
enum  {
	VALA_ENUM_DUMMY_PROPERTY
};
static void vala_enum_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_enum_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_enum_real_get_cname (ValaTypeSymbol* base, gboolean const_type);
static char* vala_enum_real_get_lower_case_cprefix (ValaSymbol* base);
static char* vala_enum_get_lower_case_csuffix (ValaEnum* self);
static char* vala_enum_real_get_lower_case_cname (ValaSymbol* base, const char* infix);
static char* vala_enum_real_get_upper_case_cname (ValaTypeSymbol* base, const char* infix);
static gboolean vala_enum_real_is_reference_type (ValaTypeSymbol* base);
static void vala_enum_process_ccode_attribute (ValaEnum* self, ValaAttribute* a);
static char* vala_enum_real_get_type_id (ValaTypeSymbol* base);
static char* vala_enum_real_get_marshaller_type_name (ValaTypeSymbol* base);
static char* vala_enum_real_get_get_value_function (ValaTypeSymbol* base);
static char* vala_enum_real_get_set_value_function (ValaTypeSymbol* base);
static char* vala_enum_real_get_default_value (ValaTypeSymbol* base);
static char* vala_enum_real_get_type_signature (ValaTypeSymbol* base);
static gboolean vala_enum_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_enum_parent_class = NULL;
static void vala_enum_finalize (ValaCodeNode* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new enum.
 *
 * @param name             type name
 * @param source_reference reference to source code
 * @return                 newly created enum
 */
ValaEnum* vala_enum_construct (GType object_type, const char* name, ValaSourceReference* source_reference) {
	ValaEnum* self;
	g_return_val_if_fail (name != NULL, NULL);
	self = (ValaEnum*) vala_typesymbol_construct (object_type, name, source_reference);
	return self;
}


ValaEnum* vala_enum_new (const char* name, ValaSourceReference* source_reference) {
	return vala_enum_construct (VALA_TYPE_ENUM, name, source_reference);
}


/**
 * Appends the specified enum value to the list of values.
 *
 * @param value an enum value
 */
void vala_enum_add_value (ValaEnum* self, ValaEnumValue* value) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (value != NULL);
	gee_collection_add ((GeeCollection*) self->priv->values, value);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) value), (ValaSymbol*) value);
}


/**
 * Adds the specified method as a member to this enum.
 *
 * @param m a method
 */
void vala_enum_add_method (ValaEnum* self, ValaMethod* m) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	if (VALA_IS_CREATION_METHOD (m)) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) m), "construction methods may only be declared within classes and structs");
		vala_code_node_set_error ((ValaCodeNode*) m, TRUE);
		return;
	}
	if (vala_method_get_binding (m) == MEMBER_BINDING_INSTANCE) {
		ValaFormalParameter* _tmp1;
		ValaEnumValueType* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		vala_method_set_this_parameter (m, _tmp1 = vala_formal_parameter_new ("this", (ValaDataType*) (_tmp0 = vala_enum_value_type_new (self)), NULL));
		(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
		(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL));
		vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) vala_method_get_this_parameter (m)), (ValaSymbol*) vala_method_get_this_parameter (m));
	}
	gee_collection_add ((GeeCollection*) self->priv->methods, m);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) m), (ValaSymbol*) m);
}


/**
 * Returns a copy of the list of enum values.
 *
 * @return list of enum values
 */
GeeList* vala_enum_get_values (ValaEnum* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_ENUM_VALUE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->values);
}


/**
 * Returns a copy of the list of methods.
 *
 * @return list of methods
 */
GeeList* vala_enum_get_methods (ValaEnum* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->methods);
}


static void vala_enum_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaEnum * self;
	self = (ValaEnum*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_enum (visitor, self);
}


static void vala_enum_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaEnum * self;
	self = (ValaEnum*) base;
	g_return_if_fail (visitor != NULL);
	{
		GeeIterator* _value_it;
		_value_it = gee_iterable_iterator ((GeeIterable*) self->priv->values);
		while (gee_iterator_next (_value_it)) {
			ValaEnumValue* value;
			value = (ValaEnumValue*) gee_iterator_get (_value_it);
			vala_code_node_accept ((ValaCodeNode*) value, visitor);
			(value == NULL) ? NULL : (value = (vala_code_node_unref (value), NULL));
		}
		(_value_it == NULL) ? NULL : (_value_it = (gee_collection_object_unref (_value_it), NULL));
	}
	{
		GeeIterator* _m_it;
		_m_it = gee_iterable_iterator ((GeeIterable*) self->priv->methods);
		while (gee_iterator_next (_m_it)) {
			ValaMethod* m;
			m = (ValaMethod*) gee_iterator_get (_m_it);
			vala_code_node_accept ((ValaCodeNode*) m, visitor);
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
}


static char* vala_enum_real_get_cname (ValaTypeSymbol* base, gboolean const_type) {
	ValaEnum * self;
	const char* _tmp3;
	self = (ValaEnum*) base;
	if (self->priv->cname == NULL) {
		ValaAttribute* attr;
		attr = vala_code_node_get_attribute ((ValaCodeNode*) self, "CCode");
		if (attr != NULL) {
			char* _tmp0;
			_tmp0 = NULL;
			self->priv->cname = (_tmp0 = vala_attribute_get_string (attr, "cname"), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp0);
		}
		if (self->priv->cname == NULL) {
			char* _tmp2;
			char* _tmp1;
			_tmp2 = NULL;
			_tmp1 = NULL;
			self->priv->cname = (_tmp2 = g_strdup_printf ("%s%s", _tmp1 = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), vala_symbol_get_name ((ValaSymbol*) self)), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp2);
			_tmp1 = (g_free (_tmp1), NULL);
		}
		(attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL));
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->cname, (_tmp3 == NULL) ? NULL : g_strdup (_tmp3));
}


static char* vala_enum_real_get_lower_case_cprefix (ValaSymbol* base) {
	ValaEnum * self;
	const char* _tmp2;
	self = (ValaEnum*) base;
	if (self->priv->lower_case_cprefix == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->lower_case_cprefix = (_tmp1 = g_strdup_printf ("%s_", _tmp0 = vala_symbol_get_lower_case_cname ((ValaSymbol*) self, NULL)), self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL), _tmp1);
		_tmp0 = (g_free (_tmp0), NULL);
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->lower_case_cprefix, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2));
}


static char* vala_enum_get_lower_case_csuffix (ValaEnum* self) {
	const char* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->lower_case_csuffix == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->lower_case_csuffix = (_tmp0 = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name ((ValaSymbol*) self)), self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->lower_case_csuffix, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1));
}


static char* vala_enum_real_get_lower_case_cname (ValaSymbol* base, const char* infix) {
	ValaEnum * self;
	char* _tmp1;
	char* _tmp0;
	char* _tmp2;
	self = (ValaEnum*) base;
	if (infix == NULL) {
		infix = "";
	}
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = g_strdup_printf ("%s%s%s", _tmp0 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), infix, _tmp1 = vala_enum_get_lower_case_csuffix (self)), _tmp1 = (g_free (_tmp1), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp2);
}


static char* vala_enum_real_get_upper_case_cname (ValaTypeSymbol* base, const char* infix) {
	ValaEnum * self;
	char* _tmp0;
	char* _tmp1;
	self = (ValaEnum*) base;
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_utf8_strup (_tmp0 = vala_symbol_get_lower_case_cname ((ValaSymbol*) self, infix), -1), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
}


static gboolean vala_enum_real_is_reference_type (ValaTypeSymbol* base) {
	ValaEnum * self;
	self = (ValaEnum*) base;
	return FALSE;
}


/**
 * Returns the string to be prepended to the name of members of this
 * enum when used in C code.
 *
 * @return the prefix to be used in C code
 */
char* vala_enum_get_cprefix (ValaEnum* self) {
	const char* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->cprefix == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->cprefix = (_tmp1 = g_strdup_printf ("%s_", _tmp0 = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) self, NULL)), self->priv->cprefix = (g_free (self->priv->cprefix), NULL), _tmp1);
		_tmp0 = (g_free (_tmp0), NULL);
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->cprefix, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2));
}


/**
 * Sets the string to be prepended to the name of members of this enum
 * when used in C code.
 *
 * @param cprefix the prefix to be used in C code
 */
void vala_enum_set_cprefix (ValaEnum* self, const char* cprefix) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cprefix = (_tmp1 = (_tmp0 = cprefix, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->cprefix = (g_free (self->priv->cprefix), NULL), _tmp1);
}


static void vala_enum_process_ccode_attribute (ValaEnum* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	if (vala_attribute_has_argument (a, "cprefix")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_enum_set_cprefix (self, _tmp0 = vala_attribute_get_string (a, "cprefix"));
		_tmp0 = (g_free (_tmp0), NULL);
	}
	if (vala_attribute_has_argument (a, "lower_case_csuffix")) {
		char* _tmp1;
		_tmp1 = NULL;
		self->priv->lower_case_csuffix = (_tmp1 = vala_attribute_get_string (a, "lower_case_csuffix"), self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL), _tmp1);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** _tmp2;
			char** filename_collection;
			int filename_collection_length1;
			int filename_it;
			_tmp2 = NULL;
			filename_collection = _tmp2 = g_strsplit (val, ",", 0);
			filename_collection_length1 = g_strv_length (_tmp2);
			for (filename_it = 0; filename_it < g_strv_length (_tmp2); filename_it = filename_it + 1) {
				const char* _tmp3;
				char* filename;
				_tmp3 = NULL;
				filename = (_tmp3 = filename_collection[filename_it], (_tmp3 == NULL) ? NULL : g_strdup (_tmp3));
				{
					vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) self, filename);
					filename = (g_free (filename), NULL);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, (GDestroyNotify) g_free), NULL);
		}
		val = (g_free (val), NULL);
	}
	if (vala_attribute_has_argument (a, "has_type_id")) {
		vala_enum_set_has_type_id (self, vala_attribute_get_bool (a, "has_type_id"));
	}
	if (vala_attribute_has_argument (a, "type_id")) {
		char* _tmp4;
		_tmp4 = NULL;
		self->priv->type_id = (_tmp4 = vala_attribute_get_string (a, "type_id"), self->priv->type_id = (g_free (self->priv->type_id), NULL), _tmp4);
	}
}


/**
 * Process all associated attributes.
 */
void vala_enum_process_attributes (ValaEnum* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = (ValaAttribute*) a_it->data, (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_enum_process_ccode_attribute (self, a);
				} else {
					if (_vala_strcmp0 (vala_attribute_get_name (a), "Flags") == 0) {
						vala_enum_set_is_flags (self, TRUE);
					}
				}
				(a == NULL) ? NULL : (a = (vala_code_node_unref (a), NULL));
			}
		}
	}
}


void vala_enum_set_type_id (ValaEnum* self, const char* type_id) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->type_id = (_tmp1 = (_tmp0 = type_id, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->type_id = (g_free (self->priv->type_id), NULL), _tmp1);
}


static char* vala_enum_real_get_type_id (ValaTypeSymbol* base) {
	ValaEnum * self;
	const char* _tmp4;
	self = (ValaEnum*) base;
	if (self->priv->type_id == NULL) {
		if (self->priv->_has_type_id) {
			char* _tmp0;
			_tmp0 = NULL;
			self->priv->type_id = (_tmp0 = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) self, "TYPE_"), self->priv->type_id = (g_free (self->priv->type_id), NULL), _tmp0);
		} else {
			const char* _tmp1;
			char* _tmp3;
			const char* _tmp2;
			_tmp1 = NULL;
			if (self->priv->_is_flags) {
				_tmp1 = "G_TYPE_UINT";
			} else {
				_tmp1 = "G_TYPE_INT";
			}
			_tmp3 = NULL;
			_tmp2 = NULL;
			self->priv->type_id = (_tmp3 = (_tmp2 = _tmp1, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2)), self->priv->type_id = (g_free (self->priv->type_id), NULL), _tmp3);
		}
	}
	_tmp4 = NULL;
	return (_tmp4 = self->priv->type_id, (_tmp4 == NULL) ? NULL : g_strdup (_tmp4));
}


static char* vala_enum_real_get_marshaller_type_name (ValaTypeSymbol* base) {
	ValaEnum * self;
	self = (ValaEnum*) base;
	if (self->priv->_has_type_id) {
		if (self->priv->_is_flags) {
			return g_strdup ("FLAGS");
		} else {
			return g_strdup ("ENUM");
		}
	} else {
		if (self->priv->_is_flags) {
			return g_strdup ("UINT");
		} else {
			return g_strdup ("INT");
		}
	}
}


static char* vala_enum_real_get_get_value_function (ValaTypeSymbol* base) {
	ValaEnum * self;
	self = (ValaEnum*) base;
	if (self->priv->_has_type_id) {
		if (self->priv->_is_flags) {
			return g_strdup ("g_value_get_flags");
		} else {
			return g_strdup ("g_value_get_enum");
		}
	} else {
		if (self->priv->_is_flags) {
			return g_strdup ("g_value_get_uint");
		} else {
			return g_strdup ("g_value_get_int");
		}
	}
}


static char* vala_enum_real_get_set_value_function (ValaTypeSymbol* base) {
	ValaEnum * self;
	self = (ValaEnum*) base;
	if (self->priv->_has_type_id) {
		if (self->priv->_is_flags) {
			return g_strdup ("g_value_set_flags");
		} else {
			return g_strdup ("g_value_set_enum");
		}
	} else {
		if (self->priv->_is_flags) {
			return g_strdup ("g_value_set_uint");
		} else {
			return g_strdup ("g_value_set_int");
		}
	}
}


static char* vala_enum_real_get_default_value (ValaTypeSymbol* base) {
	ValaEnum * self;
	self = (ValaEnum*) base;
	return g_strdup ("0");
}


static char* vala_enum_real_get_type_signature (ValaTypeSymbol* base) {
	ValaEnum * self;
	self = (ValaEnum*) base;
	if (self->priv->_is_flags) {
		return g_strdup ("u");
	} else {
		return g_strdup ("i");
	}
}


static gboolean vala_enum_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaEnum * self;
	ValaSourceFile* _tmp1;
	ValaSourceFile* old_source_file;
	gboolean _tmp2;
	self = (ValaEnum*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_enum_process_attributes (self);
	_tmp1 = NULL;
	old_source_file = (_tmp1 = vala_semantic_analyzer_get_current_source_file (analyzer), (_tmp1 == NULL) ? NULL : vala_source_file_ref (_tmp1));
	if (vala_code_node_get_source_reference ((ValaCodeNode*) self) != NULL) {
		vala_semantic_analyzer_set_current_source_file (analyzer, vala_source_reference_get_file (vala_code_node_get_source_reference ((ValaCodeNode*) self)));
	}
	{
		GeeIterator* _value_it;
		_value_it = gee_iterable_iterator ((GeeIterable*) self->priv->values);
		while (gee_iterator_next (_value_it)) {
			ValaEnumValue* value;
			value = (ValaEnumValue*) gee_iterator_get (_value_it);
			vala_code_node_check ((ValaCodeNode*) value, analyzer);
			(value == NULL) ? NULL : (value = (vala_code_node_unref (value), NULL));
		}
		(_value_it == NULL) ? NULL : (_value_it = (gee_collection_object_unref (_value_it), NULL));
	}
	{
		GeeIterator* _m_it;
		_m_it = gee_iterable_iterator ((GeeIterable*) self->priv->methods);
		while (gee_iterator_next (_m_it)) {
			ValaMethod* m;
			m = (ValaMethod*) gee_iterator_get (_m_it);
			vala_code_node_check ((ValaCodeNode*) m, analyzer);
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
	vala_semantic_analyzer_set_current_source_file (analyzer, old_source_file);
	return (_tmp2 = !vala_code_node_get_error ((ValaCodeNode*) self), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), _tmp2);
}


gboolean vala_enum_get_is_flags (ValaEnum* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_flags;
}


void vala_enum_set_is_flags (ValaEnum* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_flags = value;
}


gboolean vala_enum_get_has_type_id (ValaEnum* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_has_type_id;
}


void vala_enum_set_has_type_id (ValaEnum* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_has_type_id = value;
}


static void vala_enum_class_init (ValaEnumClass * klass) {
	vala_enum_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_enum_finalize;
	g_type_class_add_private (klass, sizeof (ValaEnumPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_enum_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_enum_real_accept_children;
	VALA_TYPESYMBOL_CLASS (klass)->get_cname = vala_enum_real_get_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_enum_real_get_lower_case_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cname = vala_enum_real_get_lower_case_cname;
	VALA_TYPESYMBOL_CLASS (klass)->get_upper_case_cname = vala_enum_real_get_upper_case_cname;
	VALA_TYPESYMBOL_CLASS (klass)->is_reference_type = vala_enum_real_is_reference_type;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_id = vala_enum_real_get_type_id;
	VALA_TYPESYMBOL_CLASS (klass)->get_marshaller_type_name = vala_enum_real_get_marshaller_type_name;
	VALA_TYPESYMBOL_CLASS (klass)->get_get_value_function = vala_enum_real_get_get_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_set_value_function = vala_enum_real_get_set_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_default_value = vala_enum_real_get_default_value;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_signature = vala_enum_real_get_type_signature;
	VALA_CODE_NODE_CLASS (klass)->check = vala_enum_real_check;
}


static void vala_enum_instance_init (ValaEnum * self) {
	self->priv = VALA_ENUM_GET_PRIVATE (self);
	self->priv->_has_type_id = TRUE;
	self->priv->values = (GeeList*) gee_array_list_new (VALA_TYPE_ENUM_VALUE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->methods = (GeeList*) gee_array_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
}


static void vala_enum_finalize (ValaCodeNode* obj) {
	ValaEnum * self;
	self = VALA_ENUM (obj);
	(self->priv->values == NULL) ? NULL : (self->priv->values = (gee_collection_object_unref (self->priv->values), NULL));
	(self->priv->methods == NULL) ? NULL : (self->priv->methods = (gee_collection_object_unref (self->priv->methods), NULL));
	self->priv->cname = (g_free (self->priv->cname), NULL);
	self->priv->cprefix = (g_free (self->priv->cprefix), NULL);
	self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL);
	self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL);
	self->priv->type_id = (g_free (self->priv->type_id), NULL);
	VALA_CODE_NODE_CLASS (vala_enum_parent_class)->finalize (obj);
}


GType vala_enum_get_type (void) {
	static GType vala_enum_type_id = 0;
	if (vala_enum_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaEnumClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_enum_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaEnum), 0, (GInstanceInitFunc) vala_enum_instance_init, NULL };
		vala_enum_type_id = g_type_register_static (VALA_TYPE_TYPESYMBOL, "ValaEnum", &g_define_type_info, 0);
	}
	return vala_enum_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




