/* map.vala
 *
 * Copyright (C) 2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <gee/map.h>







/**
 * Returns the keys of this map as a read-only set.
 *
 * @return the keys of the map
 */
GeeSet* gee_map_get_keys (GeeMap* self) {
	return GEE_MAP_GET_INTERFACE (self)->get_keys (self);
}


/**
 * Returns the values of this map as a read-only collection.
 *
 * @return the values of the map
 */
GeeCollection* gee_map_get_values (GeeMap* self) {
	return GEE_MAP_GET_INTERFACE (self)->get_values (self);
}


/**
 * Determines whether this map contains the specified key.
 *
 * @param key the key to locate in the map
 *
 * @return    true if key is found, false otherwise
 */
gboolean gee_map_contains (GeeMap* self, gconstpointer key) {
	return GEE_MAP_GET_INTERFACE (self)->contains (self, key);
}


/**
 * Returns the value of the specified key in this map.
 *
 * @param key the key whose value is to be retrieved
 *
 * @return    the value associated with the key, or null if the key
 *            couldn't be found
 */
gpointer gee_map_get (GeeMap* self, gconstpointer key) {
	return GEE_MAP_GET_INTERFACE (self)->get (self, key);
}


/**
 * Inserts a new key and value into this map.
 *
 * @param key   the key to insert
 * @param value the value to associate with the key
 */
void gee_map_set (GeeMap* self, gconstpointer key, gconstpointer value) {
	GEE_MAP_GET_INTERFACE (self)->set (self, key, value);
}


/**
 * Removes the specified key from this map.
 *
 * @param key the key to remove from the map
 *
 * @return    true if the map has been changed, false otherwise
 */
gboolean gee_map_remove (GeeMap* self, gconstpointer key) {
	return GEE_MAP_GET_INTERFACE (self)->remove (self, key);
}


/**
 * Removes all items from this collection. Must not be called on
 * read-only collections.
 */
void gee_map_clear (GeeMap* self) {
	GEE_MAP_GET_INTERFACE (self)->clear (self);
}


gint gee_map_get_size (GeeMap* self) {
	return GEE_MAP_GET_INTERFACE (self)->get_size (self);
}


static void gee_map_base_init (GeeMapIface * iface) {
	static gboolean initialized = FALSE;
	if (!initialized) {
		initialized = TRUE;
	}
}


GType gee_map_get_type (void) {
	static GType gee_map_type_id = 0;
	if (gee_map_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeMapIface), (GBaseInitFunc) gee_map_base_init, (GBaseFinalizeFunc) NULL, (GClassInitFunc) NULL, (GClassFinalizeFunc) NULL, NULL, 0, 0, (GInstanceInitFunc) NULL, NULL };
		gee_map_type_id = g_type_register_static (G_TYPE_INTERFACE, "GeeMap", &g_define_type_info, 0);
		g_type_interface_add_prerequisite (gee_map_type_id, GEE_TYPE_COLLECTION_OBJECT);
	}
	return gee_map_type_id;
}




