/* valaclass.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaclass.h>
#include <gee/readonlylist.h>
#include <gee/arraylist.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <gee/collection.h>
#include <vala/valasymbol.h>
#include <vala/valadatatype.h>
#include <vala/valamethod.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valaattribute.h>
#include <vala/valatypesymbol.h>
#include <vala/valasourcefile.h>
#include <vala/valasourcereference.h>
#include <vala/valaobjecttype.h>
#include <vala/valareport.h>
#include <vala/valatypeparameter.h>
#include <vala/valaenum.h>
#include <vala/valafield.h>
#include <vala/valaconstant.h>
#include <vala/valaproperty.h>
#include <vala/valasignal.h>
#include <vala/valaconstructor.h>
#include <vala/valadestructor.h>
#include <vala/valascope.h>
#include <vala/valaformalparameter.h>
#include <vala/valastruct.h>
#include <vala/valadelegate.h>
#include <vala/valainterface.h>
#include <vala/valamember.h>
#include <vala/valacreationmethod.h>
#include <vala/valavoidtype.h>
#include <vala/valalocalvariable.h>
#include <vala/valaexpression.h>




static char* string_substring (const char* self, glong offset, glong len);
struct _ValaClassPrivate {
	ValaClass* _base_class;
	gboolean _is_abstract;
	gboolean _ref_function_void;
	char* _type_check_function;
	gboolean _has_private_fields;
	gboolean _has_class_private_fields;
	char* cname;
	char* const_cname;
	char* lower_case_cprefix;
	char* lower_case_csuffix;
	char* type_id;
	char* ref_function;
	char* unref_function;
	char* param_spec_function;
	char* copy_function;
	char* free_function;
	char* marshaller_type_name;
	char* get_value_function;
	char* set_value_function;
	char* type_signature;
	gboolean _is_compact;
	gboolean _is_immutable;
	GeeList* base_types;
	GeeList* constants;
	GeeList* fields;
	GeeList* methods;
	GeeList* properties;
	GeeList* signals;
	GeeList* classes;
	GeeList* structs;
	GeeList* enums;
	GeeList* delegates;
	ValaMethod* _default_construction_method;
	ValaConstructor* _constructor;
	ValaConstructor* _class_constructor;
	ValaConstructor* _static_constructor;
	ValaDestructor* _static_destructor;
	ValaDestructor* _class_destructor;
	ValaDestructor* _destructor;
};

#define VALA_CLASS_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CLASS, ValaClassPrivate))
enum  {
	VALA_CLASS_DUMMY_PROPERTY
};
static GeeList* vala_class_real_get_methods (ValaObjectTypeSymbol* base);
static GeeList* vala_class_real_get_properties (ValaObjectTypeSymbol* base);
static GeeList* vala_class_real_get_signals (ValaObjectTypeSymbol* base);
static void vala_class_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_class_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_class_real_get_cprefix (ValaSymbol* base);
static char* vala_class_real_get_cname (ValaTypeSymbol* base, gboolean const_type);
static char* vala_class_get_lower_case_csuffix (ValaClass* self);
static char* vala_class_real_get_lower_case_cname (ValaSymbol* base, const char* infix);
static char* vala_class_real_get_lower_case_cprefix (ValaSymbol* base);
static char* vala_class_real_get_upper_case_cname (ValaTypeSymbol* base, const char* infix);
static char* vala_class_real_get_type_signature (ValaTypeSymbol* base);
static gboolean vala_class_real_is_reference_type (ValaTypeSymbol* base);
static void vala_class_process_ccode_attribute (ValaClass* self, ValaAttribute* a);
static char* vala_class_real_get_type_id (ValaTypeSymbol* base);
static char* vala_class_real_get_marshaller_type_name (ValaTypeSymbol* base);
static char* vala_class_real_get_param_spec_function (ValaTypeSymbol* base);
static char* vala_class_real_get_get_value_function (ValaTypeSymbol* base);
static char* vala_class_real_get_set_value_function (ValaTypeSymbol* base);
static gboolean vala_class_real_is_reference_counting (ValaTypeSymbol* base);
static char* vala_class_real_get_ref_function (ValaTypeSymbol* base);
static char* vala_class_real_get_unref_function (ValaTypeSymbol* base);
static char* vala_class_real_get_dup_function (ValaTypeSymbol* base);
static char* vala_class_real_get_free_function (ValaTypeSymbol* base);
static gboolean vala_class_real_is_subtype_of (ValaTypeSymbol* base, ValaTypeSymbol* t);
static void vala_class_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static void vala_class_get_all_prerequisites (ValaClass* self, ValaInterface* iface, GeeList* list);
static gboolean vala_class_class_is_a (ValaClass* self, ValaClass* cl, ValaTypeSymbol* t);
static gboolean vala_class_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static void vala_class_set_has_private_fields (ValaClass* self, gboolean value);
static void vala_class_set_has_class_private_fields (ValaClass* self, gboolean value);
static gpointer vala_class_parent_class = NULL;
static void vala_class_finalize (ValaCodeNode* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);



static char* string_substring (const char* self, glong offset, glong len) {
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_return_val_if_fail (offset >= 0, NULL);
	} else {
		g_return_val_if_fail (offset <= string_length, NULL);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_return_val_if_fail ((offset + len) <= string_length, NULL);
	start = g_utf8_offset_to_pointer (self, offset);
	return g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
}


/**
 * Returns a copy of the list of classes.
 *
 * @return list of classes
 */
GeeList* vala_class_get_classes (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_CLASS, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->classes);
}


/**
 * Returns a copy of the list of structs.
 *
 * @return list of structs
 */
GeeList* vala_class_get_structs (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_STRUCT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->structs);
}


/**
 * Returns a copy of the list of enums.
 *
 * @return list of enums
 */
GeeList* vala_class_get_enums (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_ENUM, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->enums);
}


/**
 * Returns a copy of the list of delegates.
 *
 * @return list of delegates
 */
GeeList* vala_class_get_delegates (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_DELEGATE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->delegates);
}


/**
 * Creates a new class.
 *
 * @param name   type name
 * @param source reference to source code
 * @return       newly created class
 */
ValaClass* vala_class_construct (GType object_type, const char* name, ValaSourceReference* source_reference) {
	ValaClass* self;
	g_return_val_if_fail (name != NULL, NULL);
	self = (ValaClass*) vala_object_type_symbol_construct (object_type, name, source_reference);
	return self;
}


ValaClass* vala_class_new (const char* name, ValaSourceReference* source_reference) {
	return vala_class_construct (VALA_TYPE_CLASS, name, source_reference);
}


/**
 * Adds the specified class or interface to the list of base types of
 * this class.
 *
 * @param type a class or interface reference
 */
void vala_class_add_base_type (ValaClass* self, ValaDataType* type) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (type != NULL);
	gee_collection_add ((GeeCollection*) self->priv->base_types, type);
	vala_code_node_set_parent_node ((ValaCodeNode*) type, (ValaCodeNode*) self);
}


/**
 * Returns a copy of the base type list.
 *
 * @return list of base types
 */
GeeList* vala_class_get_base_types (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_DATA_TYPE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->base_types);
}


/**
 * Adds the specified constant as a member to this class.
 *
 * @param c a constant
 */
void vala_class_add_constant (ValaClass* self, ValaConstant* c) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (c != NULL);
	gee_collection_add ((GeeCollection*) self->priv->constants, c);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) c), (ValaSymbol*) c);
}


/**
 * Adds the specified field as a member to this class.
 *
 * @param f a field
 */
void vala_class_add_field (ValaClass* self, ValaField* f) {
	gboolean _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (f != NULL);
	gee_collection_add ((GeeCollection*) self->priv->fields, f);
	_tmp0 = FALSE;
	if (vala_symbol_get_access ((ValaSymbol*) f) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		_tmp0 = vala_field_get_binding (f) == MEMBER_BINDING_INSTANCE;
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		vala_class_set_has_private_fields (self, TRUE);
	} else {
		gboolean _tmp1;
		_tmp1 = FALSE;
		if (vala_symbol_get_access ((ValaSymbol*) f) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
			_tmp1 = vala_field_get_binding (f) == MEMBER_BINDING_CLASS;
		} else {
			_tmp1 = FALSE;
		}
		if (_tmp1) {
			vala_class_set_has_class_private_fields (self, TRUE);
		}
	}
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) f), (ValaSymbol*) f);
}


/**
 * Returns a copy of the list of fields.
 *
 * @return list of fields
 */
GeeList* vala_class_get_fields (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_FIELD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->fields);
}


/**
 * Returns a copy of the list of constants.
 *
 * @return list of constants
 */
GeeList* vala_class_get_constants (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_CONSTANT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->constants);
}


/**
 * Adds the specified method as a member to this class.
 *
 * @param m a method
 */
void vala_class_add_method (ValaClass* self, ValaMethod* m) {
	gboolean _tmp0;
	gboolean _tmp3;
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	_tmp0 = FALSE;
	if (vala_method_get_binding (m) == MEMBER_BINDING_INSTANCE) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = VALA_IS_CREATION_METHOD (m);
	}
	if (_tmp0) {
		ValaFormalParameter* _tmp2;
		ValaObjectType* _tmp1;
		if (vala_method_get_this_parameter (m) != NULL) {
			vala_scope_remove (vala_symbol_get_scope ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) vala_method_get_this_parameter (m)));
		}
		_tmp2 = NULL;
		_tmp1 = NULL;
		vala_method_set_this_parameter (m, _tmp2 = vala_formal_parameter_new ("this", (ValaDataType*) (_tmp1 = vala_object_type_new ((ValaObjectTypeSymbol*) self)), NULL));
		(_tmp2 == NULL) ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL));
		(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
		vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) vala_method_get_this_parameter (m)), (ValaSymbol*) vala_method_get_this_parameter (m));
	}
	_tmp3 = FALSE;
	if (!(VALA_IS_VOID_TYPE (vala_method_get_return_type (m)))) {
		GeeList* _tmp4;
		_tmp4 = NULL;
		_tmp3 = gee_collection_get_size ((GeeCollection*) (_tmp4 = vala_method_get_postconditions (m))) > 0;
		(_tmp4 == NULL) ? NULL : (_tmp4 = (gee_collection_object_unref (_tmp4), NULL));
	} else {
		_tmp3 = FALSE;
	}
	if (_tmp3) {
		ValaLocalVariable* _tmp6;
		ValaDataType* _tmp5;
		if (vala_method_get_result_var (m) != NULL) {
			vala_scope_remove (vala_symbol_get_scope ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) vala_method_get_result_var (m)));
		}
		_tmp6 = NULL;
		_tmp5 = NULL;
		vala_method_set_result_var (m, _tmp6 = vala_local_variable_new (_tmp5 = vala_data_type_copy (vala_method_get_return_type (m)), "result", NULL, NULL));
		(_tmp6 == NULL) ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL));
		(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL));
	}
	if (VALA_IS_CREATION_METHOD (m)) {
		ValaCreationMethod* _tmp7;
		ValaCreationMethod* cm;
		gboolean _tmp8;
		if (vala_symbol_get_name ((ValaSymbol*) m) == NULL) {
			vala_class_set_default_construction_method (self, m);
			vala_symbol_set_name ((ValaSymbol*) m, "new");
		}
		_tmp7 = NULL;
		cm = (_tmp7 = VALA_CREATION_METHOD (m), (_tmp7 == NULL) ? NULL : vala_code_node_ref (_tmp7));
		_tmp8 = FALSE;
		if (vala_creation_method_get_type_name (cm) != NULL) {
			_tmp8 = _vala_strcmp0 (vala_creation_method_get_type_name (cm), vala_symbol_get_name ((ValaSymbol*) self)) != 0;
		} else {
			_tmp8 = FALSE;
		}
		if (_tmp8) {
			char* _tmp10;
			char* _tmp9;
			/* type_name is null for constructors generated by GIdlParser*/
			_tmp10 = NULL;
			_tmp9 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) m), _tmp10 = g_strdup_printf ("missing return type in method `%s.%s´", _tmp9 = vala_symbol_get_full_name ((ValaSymbol*) self), vala_creation_method_get_type_name (cm)));
			_tmp10 = (g_free (_tmp10), NULL);
			_tmp9 = (g_free (_tmp9), NULL);
			vala_code_node_set_error ((ValaCodeNode*) m, TRUE);
			(cm == NULL) ? NULL : (cm = (vala_code_node_unref (cm), NULL));
			return;
		}
		(cm == NULL) ? NULL : (cm = (vala_code_node_unref (cm), NULL));
	}
	gee_collection_add ((GeeCollection*) self->priv->methods, m);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) m), (ValaSymbol*) m);
}


/**
 * Returns a copy of the list of methods.
 *
 * @return list of methods
 */
static GeeList* vala_class_real_get_methods (ValaObjectTypeSymbol* base) {
	ValaClass * self;
	self = (ValaClass*) base;
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->methods);
}


/**
 * Adds the specified property as a member to this class.
 *
 * @param prop a property
 */
void vala_class_add_property (ValaClass* self, ValaProperty* prop) {
	ValaFormalParameter* _tmp1;
	ValaObjectType* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (prop != NULL);
	gee_collection_add ((GeeCollection*) self->priv->properties, prop);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) prop), (ValaSymbol*) prop);
	_tmp1 = NULL;
	_tmp0 = NULL;
	vala_property_set_this_parameter (prop, _tmp1 = vala_formal_parameter_new ("this", (ValaDataType*) (_tmp0 = vala_object_type_new ((ValaObjectTypeSymbol*) self)), NULL));
	(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
	(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL));
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) prop), vala_symbol_get_name ((ValaSymbol*) vala_property_get_this_parameter (prop)), (ValaSymbol*) vala_property_get_this_parameter (prop));
	if (vala_property_get_field (prop) != NULL) {
		vala_class_add_field (self, vala_property_get_field (prop));
	}
}


/**
 * Returns a copy of the list of properties.
 *
 * @return list of properties
 */
static GeeList* vala_class_real_get_properties (ValaObjectTypeSymbol* base) {
	ValaClass * self;
	self = (ValaClass*) base;
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_PROPERTY, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->properties);
}


/**
 * Adds the specified signal as a member to this class.
 *
 * @param sig a signal
 */
void vala_class_add_signal (ValaClass* self, ValaSignal* sig) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (sig != NULL);
	gee_collection_add ((GeeCollection*) self->priv->signals, sig);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) sig), (ValaSymbol*) sig);
}


/**
 * Returns a copy of the list of signals.
 *
 * @return list of signals
 */
static GeeList* vala_class_real_get_signals (ValaObjectTypeSymbol* base) {
	ValaClass * self;
	self = (ValaClass*) base;
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_SIGNAL, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->signals);
}


/**
 * Adds the specified class as an inner class.
 *
 * @param cl a class
 */
void vala_class_add_class (ValaClass* self, ValaClass* cl) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (cl != NULL);
	gee_collection_add ((GeeCollection*) self->priv->classes, cl);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) cl), (ValaSymbol*) cl);
}


/**
 * Adds the specified struct as an inner struct.
 *
 * @param st a struct
 */
void vala_class_add_struct (ValaClass* self, ValaStruct* st) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (st != NULL);
	gee_collection_add ((GeeCollection*) self->priv->structs, st);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) st), (ValaSymbol*) st);
}


/**
 * Adds the specified enum as an inner enum.
 *
 * @param en an enum
 */
void vala_class_add_enum (ValaClass* self, ValaEnum* en) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (en != NULL);
	gee_collection_add ((GeeCollection*) self->priv->enums, en);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) en), (ValaSymbol*) en);
}


/**
 * Adds the specified delegate as an inner delegate.
 *
 * @param d a delegate
 */
void vala_class_add_delegate (ValaClass* self, ValaDelegate* d) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (d != NULL);
	gee_collection_add ((GeeCollection*) self->priv->delegates, d);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) d), (ValaSymbol*) d);
}


static void vala_class_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaClass * self;
	self = (ValaClass*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_class (visitor, self);
}


static void vala_class_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaClass * self;
	self = (ValaClass*) base;
	g_return_if_fail (visitor != NULL);
	{
		GeeIterator* _type_it;
		_type_it = gee_iterable_iterator ((GeeIterable*) self->priv->base_types);
		while (gee_iterator_next (_type_it)) {
			ValaDataType* type;
			type = (ValaDataType*) gee_iterator_get (_type_it);
			vala_code_node_accept ((ValaCodeNode*) type, visitor);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
		}
		(_type_it == NULL) ? NULL : (_type_it = (gee_collection_object_unref (_type_it), NULL));
	}
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* _p_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		_p_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_object_type_symbol_get_type_parameters ((ValaObjectTypeSymbol*) self))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (_p_it)) {
			ValaTypeParameter* p;
			p = (ValaTypeParameter*) gee_iterator_get (_p_it);
			vala_code_node_accept ((ValaCodeNode*) p, visitor);
			(p == NULL) ? NULL : (p = (vala_code_node_unref (p), NULL));
		}
		(_p_it == NULL) ? NULL : (_p_it = (gee_collection_object_unref (_p_it), NULL));
	}
	/* process enums first to avoid order problems in C code */
	{
		GeeIterator* _en_it;
		/* process enums first to avoid order problems in C code */
		_en_it = gee_iterable_iterator ((GeeIterable*) self->priv->enums);
		/* process enums first to avoid order problems in C code */
		while (gee_iterator_next (_en_it)) {
			ValaEnum* en;
			/* process enums first to avoid order problems in C code */
			en = (ValaEnum*) gee_iterator_get (_en_it);
			vala_code_node_accept ((ValaCodeNode*) en, visitor);
			(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
		}
		(_en_it == NULL) ? NULL : (_en_it = (gee_collection_object_unref (_en_it), NULL));
	}
	{
		GeeIterator* _f_it;
		_f_it = gee_iterable_iterator ((GeeIterable*) self->priv->fields);
		while (gee_iterator_next (_f_it)) {
			ValaField* f;
			f = (ValaField*) gee_iterator_get (_f_it);
			vala_code_node_accept ((ValaCodeNode*) f, visitor);
			(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
		}
		(_f_it == NULL) ? NULL : (_f_it = (gee_collection_object_unref (_f_it), NULL));
	}
	{
		GeeIterator* _c_it;
		_c_it = gee_iterable_iterator ((GeeIterable*) self->priv->constants);
		while (gee_iterator_next (_c_it)) {
			ValaConstant* c;
			c = (ValaConstant*) gee_iterator_get (_c_it);
			vala_code_node_accept ((ValaCodeNode*) c, visitor);
			(c == NULL) ? NULL : (c = (vala_code_node_unref (c), NULL));
		}
		(_c_it == NULL) ? NULL : (_c_it = (gee_collection_object_unref (_c_it), NULL));
	}
	{
		GeeIterator* _m_it;
		_m_it = gee_iterable_iterator ((GeeIterable*) self->priv->methods);
		while (gee_iterator_next (_m_it)) {
			ValaMethod* m;
			m = (ValaMethod*) gee_iterator_get (_m_it);
			vala_code_node_accept ((ValaCodeNode*) m, visitor);
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
	{
		GeeIterator* _prop_it;
		_prop_it = gee_iterable_iterator ((GeeIterable*) self->priv->properties);
		while (gee_iterator_next (_prop_it)) {
			ValaProperty* prop;
			prop = (ValaProperty*) gee_iterator_get (_prop_it);
			vala_code_node_accept ((ValaCodeNode*) prop, visitor);
			(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
		}
		(_prop_it == NULL) ? NULL : (_prop_it = (gee_collection_object_unref (_prop_it), NULL));
	}
	{
		GeeIterator* _sig_it;
		_sig_it = gee_iterable_iterator ((GeeIterable*) self->priv->signals);
		while (gee_iterator_next (_sig_it)) {
			ValaSignal* sig;
			sig = (ValaSignal*) gee_iterator_get (_sig_it);
			vala_code_node_accept ((ValaCodeNode*) sig, visitor);
			(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
		}
		(_sig_it == NULL) ? NULL : (_sig_it = (gee_collection_object_unref (_sig_it), NULL));
	}
	if (self->priv->_constructor != NULL) {
		vala_code_node_accept ((ValaCodeNode*) self->priv->_constructor, visitor);
	}
	if (self->priv->_class_constructor != NULL) {
		vala_code_node_accept ((ValaCodeNode*) self->priv->_class_constructor, visitor);
	}
	if (self->priv->_static_constructor != NULL) {
		vala_code_node_accept ((ValaCodeNode*) self->priv->_static_constructor, visitor);
	}
	if (vala_class_get_destructor (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_class_get_destructor (self), visitor);
	}
	if (self->priv->_static_destructor != NULL) {
		vala_code_node_accept ((ValaCodeNode*) self->priv->_static_destructor, visitor);
	}
	if (self->priv->_class_destructor != NULL) {
		vala_code_node_accept ((ValaCodeNode*) self->priv->_class_destructor, visitor);
	}
	{
		GeeIterator* _cl_it;
		_cl_it = gee_iterable_iterator ((GeeIterable*) self->priv->classes);
		while (gee_iterator_next (_cl_it)) {
			ValaClass* cl;
			cl = (ValaClass*) gee_iterator_get (_cl_it);
			vala_code_node_accept ((ValaCodeNode*) cl, visitor);
			(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
		}
		(_cl_it == NULL) ? NULL : (_cl_it = (gee_collection_object_unref (_cl_it), NULL));
	}
	{
		GeeIterator* _st_it;
		_st_it = gee_iterable_iterator ((GeeIterable*) self->priv->structs);
		while (gee_iterator_next (_st_it)) {
			ValaStruct* st;
			st = (ValaStruct*) gee_iterator_get (_st_it);
			vala_code_node_accept ((ValaCodeNode*) st, visitor);
			(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
		}
		(_st_it == NULL) ? NULL : (_st_it = (gee_collection_object_unref (_st_it), NULL));
	}
	{
		GeeIterator* _d_it;
		_d_it = gee_iterable_iterator ((GeeIterable*) self->priv->delegates);
		while (gee_iterator_next (_d_it)) {
			ValaDelegate* d;
			d = (ValaDelegate*) gee_iterator_get (_d_it);
			vala_code_node_accept ((ValaCodeNode*) d, visitor);
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
		}
		(_d_it == NULL) ? NULL : (_d_it = (gee_collection_object_unref (_d_it), NULL));
	}
}


static char* vala_class_real_get_cprefix (ValaSymbol* base) {
	ValaClass * self;
	self = (ValaClass*) base;
	return vala_typesymbol_get_cname ((ValaTypeSymbol*) self, FALSE);
}


static char* vala_class_real_get_cname (ValaTypeSymbol* base, gboolean const_type) {
	ValaClass * self;
	const char* _tmp7;
	self = (ValaClass*) base;
	if (const_type) {
		if (self->priv->const_cname != NULL) {
			const char* _tmp0;
			_tmp0 = NULL;
			return (_tmp0 = self->priv->const_cname, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
		} else {
			if (vala_class_get_is_immutable (self)) {
				char* _tmp2;
				char* _tmp3;
				_tmp2 = NULL;
				_tmp3 = NULL;
				return (_tmp3 = g_strconcat ("const ", _tmp2 = vala_typesymbol_get_cname ((ValaTypeSymbol*) self, FALSE), NULL), _tmp2 = (g_free (_tmp2), NULL), _tmp3);
			}
		}
	}
	if (self->priv->cname == NULL) {
		ValaAttribute* attr;
		attr = vala_code_node_get_attribute ((ValaCodeNode*) self, "CCode");
		if (attr != NULL) {
			char* _tmp5;
			_tmp5 = NULL;
			self->priv->cname = (_tmp5 = vala_attribute_get_string (attr, "cname"), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp5);
		}
		if (self->priv->cname == NULL) {
			char* _tmp6;
			_tmp6 = NULL;
			self->priv->cname = (_tmp6 = vala_class_get_default_cname (self), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp6);
		}
		(attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL));
	}
	_tmp7 = NULL;
	return (_tmp7 = self->priv->cname, (_tmp7 == NULL) ? NULL : g_strdup (_tmp7));
}


/**
 * Returns the default name of this class as it is used in C code.
 *
 * @return the name to be used in C code by default
 */
char* vala_class_get_default_cname (ValaClass* self) {
	char* _tmp0;
	char* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_strdup_printf ("%s%s", _tmp0 = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), vala_symbol_get_name ((ValaSymbol*) self)), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
}


/**
 * Sets the name of this class as it is used in C code.
 *
 * @param cname the name to be used in C code
 */
void vala_class_set_cname (ValaClass* self, const char* cname) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cname != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cname = (_tmp1 = (_tmp0 = cname, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp1);
}


static char* vala_class_get_lower_case_csuffix (ValaClass* self) {
	const char* _tmp5;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->lower_case_csuffix == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->lower_case_csuffix = (_tmp0 = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name ((ValaSymbol*) self)), self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL), _tmp0);
		/* remove underscores in some cases to avoid conflicts of type macros*/
		if (g_str_has_prefix (self->priv->lower_case_csuffix, "type_")) {
			char* _tmp1;
			_tmp1 = NULL;
			self->priv->lower_case_csuffix = (_tmp1 = g_strconcat ("type", g_utf8_offset_to_pointer (self->priv->lower_case_csuffix, g_utf8_strlen ("type_", -1)), NULL), self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL), _tmp1);
		} else {
			if (g_str_has_prefix (self->priv->lower_case_csuffix, "is_")) {
				char* _tmp2;
				_tmp2 = NULL;
				self->priv->lower_case_csuffix = (_tmp2 = g_strconcat ("is", g_utf8_offset_to_pointer (self->priv->lower_case_csuffix, g_utf8_strlen ("is_", -1)), NULL), self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL), _tmp2);
			}
		}
		if (g_str_has_suffix (self->priv->lower_case_csuffix, "_class")) {
			char* _tmp4;
			char* _tmp3;
			_tmp4 = NULL;
			_tmp3 = NULL;
			self->priv->lower_case_csuffix = (_tmp4 = g_strconcat (_tmp3 = string_substring (self->priv->lower_case_csuffix, (glong) 0, g_utf8_strlen (self->priv->lower_case_csuffix, -1) - g_utf8_strlen ("_class", -1)), "class", NULL), self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL), _tmp4);
			_tmp3 = (g_free (_tmp3), NULL);
		}
	}
	_tmp5 = NULL;
	return (_tmp5 = self->priv->lower_case_csuffix, (_tmp5 == NULL) ? NULL : g_strdup (_tmp5));
}


static char* vala_class_real_get_lower_case_cname (ValaSymbol* base, const char* infix) {
	ValaClass * self;
	char* _tmp1;
	char* _tmp0;
	char* _tmp2;
	self = (ValaClass*) base;
	if (infix == NULL) {
		infix = "";
	}
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = g_strdup_printf ("%s%s%s", _tmp0 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), infix, _tmp1 = vala_class_get_lower_case_csuffix (self)), _tmp1 = (g_free (_tmp1), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp2);
}


static char* vala_class_real_get_lower_case_cprefix (ValaSymbol* base) {
	ValaClass * self;
	const char* _tmp2;
	self = (ValaClass*) base;
	if (self->priv->lower_case_cprefix == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->lower_case_cprefix = (_tmp1 = g_strdup_printf ("%s_", _tmp0 = vala_symbol_get_lower_case_cname ((ValaSymbol*) self, NULL)), self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL), _tmp1);
		_tmp0 = (g_free (_tmp0), NULL);
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->lower_case_cprefix, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2));
}


static char* vala_class_real_get_upper_case_cname (ValaTypeSymbol* base, const char* infix) {
	ValaClass * self;
	char* _tmp0;
	char* _tmp1;
	self = (ValaClass*) base;
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_utf8_strup (_tmp0 = vala_symbol_get_lower_case_cname ((ValaSymbol*) self, infix), -1), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
}


static char* vala_class_real_get_type_signature (ValaTypeSymbol* base) {
	ValaClass * self;
	const char* _tmp0;
	self = (ValaClass*) base;
	_tmp0 = NULL;
	return (_tmp0 = self->priv->type_signature, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
}


static gboolean vala_class_real_is_reference_type (ValaTypeSymbol* base) {
	ValaClass * self;
	self = (ValaClass*) base;
	return TRUE;
}


static void vala_class_process_ccode_attribute (ValaClass* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	if (vala_attribute_has_argument (a, "ref_function")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_class_set_ref_function (self, _tmp0 = vala_attribute_get_string (a, "ref_function"));
		_tmp0 = (g_free (_tmp0), NULL);
	}
	if (vala_attribute_has_argument (a, "ref_function_void")) {
		vala_class_set_ref_function_void (self, vala_attribute_get_bool (a, "ref_function_void"));
	}
	if (vala_attribute_has_argument (a, "unref_function")) {
		char* _tmp1;
		_tmp1 = NULL;
		vala_class_set_unref_function (self, _tmp1 = vala_attribute_get_string (a, "unref_function"));
		_tmp1 = (g_free (_tmp1), NULL);
	}
	if (vala_attribute_has_argument (a, "copy_function")) {
		char* _tmp2;
		_tmp2 = NULL;
		vala_class_set_dup_function (self, _tmp2 = vala_attribute_get_string (a, "copy_function"));
		_tmp2 = (g_free (_tmp2), NULL);
	}
	if (vala_attribute_has_argument (a, "free_function")) {
		char* _tmp3;
		_tmp3 = NULL;
		vala_class_set_free_function (self, _tmp3 = vala_attribute_get_string (a, "free_function"));
		_tmp3 = (g_free (_tmp3), NULL);
	}
	if (vala_attribute_has_argument (a, "type_id")) {
		char* _tmp4;
		_tmp4 = NULL;
		self->priv->type_id = (_tmp4 = vala_attribute_get_string (a, "type_id"), self->priv->type_id = (g_free (self->priv->type_id), NULL), _tmp4);
	}
	if (vala_attribute_has_argument (a, "marshaller_type_name")) {
		char* _tmp5;
		_tmp5 = NULL;
		self->priv->marshaller_type_name = (_tmp5 = vala_attribute_get_string (a, "marshaller_type_name"), self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL), _tmp5);
	}
	if (vala_attribute_has_argument (a, "get_value_function")) {
		char* _tmp6;
		_tmp6 = NULL;
		self->priv->get_value_function = (_tmp6 = vala_attribute_get_string (a, "get_value_function"), self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL), _tmp6);
	}
	if (vala_attribute_has_argument (a, "set_value_function")) {
		char* _tmp7;
		_tmp7 = NULL;
		self->priv->set_value_function = (_tmp7 = vala_attribute_get_string (a, "set_value_function"), self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL), _tmp7);
	}
	if (vala_attribute_has_argument (a, "const_cname")) {
		char* _tmp8;
		_tmp8 = NULL;
		self->priv->const_cname = (_tmp8 = vala_attribute_get_string (a, "const_cname"), self->priv->const_cname = (g_free (self->priv->const_cname), NULL), _tmp8);
	}
	if (vala_attribute_has_argument (a, "cprefix")) {
		char* _tmp9;
		_tmp9 = NULL;
		self->priv->lower_case_cprefix = (_tmp9 = vala_attribute_get_string (a, "cprefix"), self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL), _tmp9);
	}
	if (vala_attribute_has_argument (a, "lower_case_csuffix")) {
		char* _tmp10;
		_tmp10 = NULL;
		self->priv->lower_case_csuffix = (_tmp10 = vala_attribute_get_string (a, "lower_case_csuffix"), self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL), _tmp10);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** _tmp11;
			char** filename_collection;
			int filename_collection_length1;
			int filename_it;
			_tmp11 = NULL;
			filename_collection = _tmp11 = g_strsplit (val, ",", 0);
			filename_collection_length1 = _vala_array_length (_tmp11);
			for (filename_it = 0; filename_it < _vala_array_length (_tmp11); filename_it = filename_it + 1) {
				const char* _tmp12;
				char* filename;
				_tmp12 = NULL;
				filename = (_tmp12 = filename_collection[filename_it], (_tmp12 == NULL) ? NULL : g_strdup (_tmp12));
				{
					vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) self, filename);
					filename = (g_free (filename), NULL);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, (GDestroyNotify) g_free), NULL);
		}
		val = (g_free (val), NULL);
	}
	if (vala_attribute_has_argument (a, "type_signature")) {
		char* _tmp13;
		_tmp13 = NULL;
		self->priv->type_signature = (_tmp13 = vala_attribute_get_string (a, "type_signature"), self->priv->type_signature = (g_free (self->priv->type_signature), NULL), _tmp13);
	}
	if (vala_attribute_has_argument (a, "type_check_function")) {
		char* _tmp14;
		_tmp14 = NULL;
		vala_class_set_type_check_function (self, _tmp14 = vala_attribute_get_string (a, "type_check_function"));
		_tmp14 = (g_free (_tmp14), NULL);
	}
	if (vala_attribute_has_argument (a, "param_spec_function")) {
		char* _tmp15;
		_tmp15 = NULL;
		self->priv->param_spec_function = (_tmp15 = vala_attribute_get_string (a, "param_spec_function"), self->priv->param_spec_function = (g_free (self->priv->param_spec_function), NULL), _tmp15);
	}
}


/**
 * Process all associated attributes.
 */
void vala_class_process_attributes (ValaClass* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = (ValaAttribute*) a_it->data, (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_class_process_ccode_attribute (self, a);
				} else {
					if (_vala_strcmp0 (vala_attribute_get_name (a), "Compact") == 0) {
						vala_class_set_is_compact (self, TRUE);
					} else {
						if (_vala_strcmp0 (vala_attribute_get_name (a), "Immutable") == 0) {
							vala_class_set_is_immutable (self, TRUE);
						}
					}
				}
				(a == NULL) ? NULL : (a = (vala_code_node_unref (a), NULL));
			}
		}
	}
}


static char* vala_class_real_get_type_id (ValaTypeSymbol* base) {
	ValaClass * self;
	const char* _tmp2;
	self = (ValaClass*) base;
	if (self->priv->type_id == NULL) {
		if (!vala_class_get_is_compact (self)) {
			char* _tmp0;
			_tmp0 = NULL;
			self->priv->type_id = (_tmp0 = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) self, "TYPE_"), self->priv->type_id = (g_free (self->priv->type_id), NULL), _tmp0);
		} else {
			char* _tmp1;
			_tmp1 = NULL;
			self->priv->type_id = (_tmp1 = g_strdup ("G_TYPE_POINTER"), self->priv->type_id = (g_free (self->priv->type_id), NULL), _tmp1);
		}
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->type_id, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2));
}


void vala_class_set_type_id (ValaClass* self, const char* type_id) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (type_id != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->type_id = (_tmp1 = (_tmp0 = type_id, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->type_id = (g_free (self->priv->type_id), NULL), _tmp1);
}


static char* vala_class_real_get_marshaller_type_name (ValaTypeSymbol* base) {
	ValaClass * self;
	const char* _tmp2;
	self = (ValaClass*) base;
	if (self->priv->marshaller_type_name == NULL) {
		if (self->priv->_base_class != NULL) {
			char* _tmp0;
			_tmp0 = NULL;
			self->priv->marshaller_type_name = (_tmp0 = vala_typesymbol_get_marshaller_type_name ((ValaTypeSymbol*) self->priv->_base_class), self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL), _tmp0);
		} else {
			char* _tmp1;
			_tmp1 = NULL;
			self->priv->marshaller_type_name = (_tmp1 = g_strdup ("POINTER"), self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL), _tmp1);
		}
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->marshaller_type_name, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2));
}


static char* vala_class_real_get_param_spec_function (ValaTypeSymbol* base) {
	ValaClass * self;
	const char* _tmp2;
	self = (ValaClass*) base;
	if (self->priv->param_spec_function == NULL) {
		gboolean _tmp0;
		_tmp0 = FALSE;
		if (vala_class_get_is_compact (self)) {
			_tmp0 = TRUE;
		} else {
			_tmp0 = self->priv->_base_class == NULL;
		}
		if (!(_tmp0)) {
			char* _tmp1;
			_tmp1 = NULL;
			self->priv->param_spec_function = (_tmp1 = vala_typesymbol_get_param_spec_function ((ValaTypeSymbol*) self->priv->_base_class), self->priv->param_spec_function = (g_free (self->priv->param_spec_function), NULL), _tmp1);
		}
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->param_spec_function, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2));
}


void vala_class_set_param_spec_function (ValaClass* self, const char* name) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (name != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->param_spec_function = (_tmp1 = (_tmp0 = name, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->param_spec_function = (g_free (self->priv->param_spec_function), NULL), _tmp1);
}


static char* vala_class_real_get_get_value_function (ValaTypeSymbol* base) {
	ValaClass * self;
	const char* _tmp3;
	self = (ValaClass*) base;
	if (self->priv->get_value_function == NULL) {
		if (vala_class_is_fundamental (self)) {
			char* _tmp0;
			_tmp0 = NULL;
			self->priv->get_value_function = (_tmp0 = vala_symbol_get_lower_case_cname ((ValaSymbol*) self, "value_get_"), self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL), _tmp0);
		} else {
			if (self->priv->_base_class != NULL) {
				char* _tmp1;
				_tmp1 = NULL;
				self->priv->get_value_function = (_tmp1 = vala_typesymbol_get_get_value_function ((ValaTypeSymbol*) self->priv->_base_class), self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL), _tmp1);
			} else {
				char* _tmp2;
				_tmp2 = NULL;
				self->priv->get_value_function = (_tmp2 = g_strdup ("g_value_get_pointer"), self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL), _tmp2);
			}
		}
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->get_value_function, (_tmp3 == NULL) ? NULL : g_strdup (_tmp3));
}


static char* vala_class_real_get_set_value_function (ValaTypeSymbol* base) {
	ValaClass * self;
	const char* _tmp3;
	self = (ValaClass*) base;
	if (self->priv->set_value_function == NULL) {
		if (vala_class_is_fundamental (self)) {
			char* _tmp0;
			_tmp0 = NULL;
			self->priv->set_value_function = (_tmp0 = vala_symbol_get_lower_case_cname ((ValaSymbol*) self, "value_set_"), self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL), _tmp0);
		} else {
			if (self->priv->_base_class != NULL) {
				char* _tmp1;
				_tmp1 = NULL;
				self->priv->set_value_function = (_tmp1 = vala_typesymbol_get_set_value_function ((ValaTypeSymbol*) self->priv->_base_class), self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL), _tmp1);
			} else {
				char* _tmp2;
				_tmp2 = NULL;
				self->priv->set_value_function = (_tmp2 = g_strdup ("g_value_set_pointer"), self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL), _tmp2);
			}
		}
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->set_value_function, (_tmp3 == NULL) ? NULL : g_strdup (_tmp3));
}


static gboolean vala_class_real_is_reference_counting (ValaTypeSymbol* base) {
	ValaClass * self;
	char* _tmp0;
	gboolean _tmp1;
	self = (ValaClass*) base;
	_tmp0 = NULL;
	return (_tmp1 = (_tmp0 = vala_typesymbol_get_ref_function ((ValaTypeSymbol*) self)) != NULL, _tmp0 = (g_free (_tmp0), NULL), _tmp1);
}


gboolean vala_class_is_fundamental (ValaClass* self) {
	gboolean _tmp0;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp0 = FALSE;
	if (!vala_class_get_is_compact (self)) {
		_tmp0 = self->priv->_base_class == NULL;
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		return TRUE;
	}
	return FALSE;
}


static char* vala_class_real_get_ref_function (ValaTypeSymbol* base) {
	ValaClass * self;
	gboolean _tmp0;
	gboolean _tmp3;
	self = (ValaClass*) base;
	_tmp0 = FALSE;
	if (self->priv->ref_function == NULL) {
		_tmp0 = vala_class_is_fundamental (self);
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		char* _tmp2;
		char* _tmp1;
		_tmp2 = NULL;
		_tmp1 = NULL;
		self->priv->ref_function = (_tmp2 = g_strconcat (_tmp1 = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) self), "ref", NULL), self->priv->ref_function = (g_free (self->priv->ref_function), NULL), _tmp2);
		_tmp1 = (g_free (_tmp1), NULL);
	}
	_tmp3 = FALSE;
	if (self->priv->ref_function == NULL) {
		_tmp3 = self->priv->_base_class != NULL;
	} else {
		_tmp3 = FALSE;
	}
	if (_tmp3) {
		return vala_typesymbol_get_ref_function ((ValaTypeSymbol*) self->priv->_base_class);
	} else {
		const char* _tmp5;
		_tmp5 = NULL;
		return (_tmp5 = self->priv->ref_function, (_tmp5 == NULL) ? NULL : g_strdup (_tmp5));
	}
}


void vala_class_set_ref_function (ValaClass* self, const char* name) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->ref_function = (_tmp1 = (_tmp0 = name, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->ref_function = (g_free (self->priv->ref_function), NULL), _tmp1);
}


static char* vala_class_real_get_unref_function (ValaTypeSymbol* base) {
	ValaClass * self;
	gboolean _tmp0;
	gboolean _tmp3;
	self = (ValaClass*) base;
	_tmp0 = FALSE;
	if (self->priv->unref_function == NULL) {
		_tmp0 = vala_class_is_fundamental (self);
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		char* _tmp2;
		char* _tmp1;
		_tmp2 = NULL;
		_tmp1 = NULL;
		self->priv->unref_function = (_tmp2 = g_strconcat (_tmp1 = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) self), "unref", NULL), self->priv->unref_function = (g_free (self->priv->unref_function), NULL), _tmp2);
		_tmp1 = (g_free (_tmp1), NULL);
	}
	_tmp3 = FALSE;
	if (self->priv->unref_function == NULL) {
		_tmp3 = self->priv->_base_class != NULL;
	} else {
		_tmp3 = FALSE;
	}
	if (_tmp3) {
		return vala_typesymbol_get_unref_function ((ValaTypeSymbol*) self->priv->_base_class);
	} else {
		const char* _tmp5;
		_tmp5 = NULL;
		return (_tmp5 = self->priv->unref_function, (_tmp5 == NULL) ? NULL : g_strdup (_tmp5));
	}
}


void vala_class_set_unref_function (ValaClass* self, const char* name) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->unref_function = (_tmp1 = (_tmp0 = name, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->unref_function = (g_free (self->priv->unref_function), NULL), _tmp1);
}


static char* vala_class_real_get_dup_function (ValaTypeSymbol* base) {
	ValaClass * self;
	const char* _tmp0;
	self = (ValaClass*) base;
	_tmp0 = NULL;
	return (_tmp0 = self->priv->copy_function, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
}


void vala_class_set_dup_function (ValaClass* self, const char* name) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->copy_function = (_tmp1 = (_tmp0 = name, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->copy_function = (g_free (self->priv->copy_function), NULL), _tmp1);
}


char* vala_class_get_default_free_function (ValaClass* self) {
	char* _tmp0;
	char* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_strconcat (_tmp0 = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) self), "free", NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
}


static char* vala_class_real_get_free_function (ValaTypeSymbol* base) {
	ValaClass * self;
	const char* _tmp1;
	self = (ValaClass*) base;
	if (self->priv->free_function == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->free_function = (_tmp0 = vala_class_get_default_free_function (self), self->priv->free_function = (g_free (self->priv->free_function), NULL), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->free_function, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1));
}


void vala_class_set_free_function (ValaClass* self, const char* name) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (name != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->free_function = (_tmp1 = (_tmp0 = name, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->free_function = (g_free (self->priv->free_function), NULL), _tmp1);
}


static gboolean vala_class_real_is_subtype_of (ValaTypeSymbol* base, ValaTypeSymbol* t) {
	ValaClass * self;
	self = (ValaClass*) base;
	g_return_val_if_fail (t != NULL, FALSE);
	if (VALA_TYPESYMBOL (self) == t) {
		return TRUE;
	}
	{
		GeeIterator* _base_type_it;
		_base_type_it = gee_iterable_iterator ((GeeIterable*) self->priv->base_types);
		while (gee_iterator_next (_base_type_it)) {
			ValaDataType* base_type;
			gboolean _tmp1;
			base_type = (ValaDataType*) gee_iterator_get (_base_type_it);
			_tmp1 = FALSE;
			if (vala_data_type_get_data_type (base_type) != NULL) {
				_tmp1 = vala_typesymbol_is_subtype_of (vala_data_type_get_data_type (base_type), t);
			} else {
				_tmp1 = FALSE;
			}
			if (_tmp1) {
				gboolean _tmp2;
				return (_tmp2 = TRUE, (base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL)), (_base_type_it == NULL) ? NULL : (_base_type_it = (gee_collection_object_unref (_base_type_it), NULL)), _tmp2);
			}
			(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
		}
		(_base_type_it == NULL) ? NULL : (_base_type_it = (gee_collection_object_unref (_base_type_it), NULL));
	}
	return FALSE;
}


static void vala_class_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaClass * self;
	self = (ValaClass*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size ((GeeCollection*) self->priv->base_types); i++) {
			ValaDataType* _tmp0;
			gboolean _tmp1;
			_tmp0 = NULL;
			if ((_tmp1 = (_tmp0 = (ValaDataType*) gee_list_get ((GeeList*) self->priv->base_types, i)) == old_type, (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)), _tmp1)) {
				gee_list_set ((GeeList*) self->priv->base_types, i, new_type);
				return;
			}
		}
	}
}


static void vala_class_get_all_prerequisites (ValaClass* self, ValaInterface* iface, GeeList* list) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (iface != NULL);
	g_return_if_fail (list != NULL);
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* _prereq_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		_prereq_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_interface_get_prerequisites (iface))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (_prereq_it)) {
			ValaDataType* prereq;
			ValaTypeSymbol* _tmp2;
			ValaTypeSymbol* type;
			prereq = (ValaDataType*) gee_iterator_get (_prereq_it);
			_tmp2 = NULL;
			type = (_tmp2 = vala_data_type_get_data_type (prereq), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
			/* skip on previous errors */
			if (type == NULL) {
				(prereq == NULL) ? NULL : (prereq = (vala_code_node_unref (prereq), NULL));
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				continue;
			}
			gee_collection_add ((GeeCollection*) list, type);
			if (VALA_IS_INTERFACE (type)) {
				vala_class_get_all_prerequisites (self, VALA_INTERFACE (type), list);
			}
			(prereq == NULL) ? NULL : (prereq = (vala_code_node_unref (prereq), NULL));
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
		}
		(_prereq_it == NULL) ? NULL : (_prereq_it = (gee_collection_object_unref (_prereq_it), NULL));
	}
}


static gboolean vala_class_class_is_a (ValaClass* self, ValaClass* cl, ValaTypeSymbol* t) {
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (cl != NULL, FALSE);
	g_return_val_if_fail (t != NULL, FALSE);
	if (VALA_TYPESYMBOL (cl) == t) {
		return TRUE;
	}
	{
		GeeList* _tmp1;
		GeeIterator* _tmp2;
		GeeIterator* _base_type_it;
		_tmp1 = NULL;
		_tmp2 = NULL;
		_base_type_it = (_tmp2 = gee_iterable_iterator ((GeeIterable*) (_tmp1 = vala_class_get_base_types (cl))), (_tmp1 == NULL) ? NULL : (_tmp1 = (gee_collection_object_unref (_tmp1), NULL)), _tmp2);
		while (gee_iterator_next (_base_type_it)) {
			ValaDataType* base_type;
			base_type = (ValaDataType*) gee_iterator_get (_base_type_it);
			if (VALA_IS_CLASS (vala_data_type_get_data_type (base_type))) {
				if (vala_class_class_is_a (self, VALA_CLASS (vala_data_type_get_data_type (base_type)), t)) {
					gboolean _tmp3;
					return (_tmp3 = TRUE, (base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL)), (_base_type_it == NULL) ? NULL : (_base_type_it = (gee_collection_object_unref (_base_type_it), NULL)), _tmp3);
				}
			} else {
				if (vala_data_type_get_data_type (base_type) == t) {
					gboolean _tmp4;
					return (_tmp4 = TRUE, (base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL)), (_base_type_it == NULL) ? NULL : (_base_type_it = (gee_collection_object_unref (_base_type_it), NULL)), _tmp4);
				}
			}
			(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
		}
		(_base_type_it == NULL) ? NULL : (_base_type_it = (gee_collection_object_unref (_base_type_it), NULL));
	}
	return FALSE;
}


static gboolean vala_class_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaClass * self;
	ValaSourceFile* _tmp1;
	ValaSourceFile* old_source_file;
	ValaSymbol* _tmp2;
	ValaSymbol* old_symbol;
	ValaClass* _tmp3;
	ValaClass* old_class;
	ValaClass* _tmp5;
	ValaClass* _tmp4;
	GeeList* prerequisites;
	GeeList* missing_prereqs;
	ValaClass* _tmp74;
	ValaClass* _tmp73;
	gboolean _tmp75;
	self = (ValaClass*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_class_process_attributes (self);
	_tmp1 = NULL;
	old_source_file = (_tmp1 = vala_semantic_analyzer_get_current_source_file (analyzer), (_tmp1 == NULL) ? NULL : vala_source_file_ref (_tmp1));
	_tmp2 = NULL;
	old_symbol = (_tmp2 = vala_semantic_analyzer_get_current_symbol (analyzer), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
	_tmp3 = NULL;
	old_class = (_tmp3 = analyzer->current_class, (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3));
	if (vala_code_node_get_source_reference ((ValaCodeNode*) self) != NULL) {
		vala_semantic_analyzer_set_current_source_file (analyzer, vala_source_reference_get_file (vala_code_node_get_source_reference ((ValaCodeNode*) self)));
	}
	vala_semantic_analyzer_set_current_symbol (analyzer, (ValaSymbol*) self);
	_tmp5 = NULL;
	_tmp4 = NULL;
	analyzer->current_class = (_tmp5 = (_tmp4 = self, (_tmp4 == NULL) ? NULL : vala_code_node_ref (_tmp4)), (analyzer->current_class == NULL) ? NULL : (analyzer->current_class = (vala_code_node_unref (analyzer->current_class), NULL)), _tmp5);
	{
		GeeList* _tmp6;
		GeeIterator* _tmp7;
		GeeIterator* _base_type_reference_it;
		_tmp6 = NULL;
		_tmp7 = NULL;
		_base_type_reference_it = (_tmp7 = gee_iterable_iterator ((GeeIterable*) (_tmp6 = vala_class_get_base_types (self))), (_tmp6 == NULL) ? NULL : (_tmp6 = (gee_collection_object_unref (_tmp6), NULL)), _tmp7);
		while (gee_iterator_next (_base_type_reference_it)) {
			ValaDataType* base_type_reference;
			base_type_reference = (ValaDataType*) gee_iterator_get (_base_type_reference_it);
			if (!vala_code_node_check ((ValaCodeNode*) base_type_reference, analyzer)) {
				gboolean _tmp8;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				return (_tmp8 = FALSE, (base_type_reference == NULL) ? NULL : (base_type_reference = (vala_code_node_unref (base_type_reference), NULL)), (_base_type_reference_it == NULL) ? NULL : (_base_type_reference_it = (gee_collection_object_unref (_base_type_reference_it), NULL)), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), (old_class == NULL) ? NULL : (old_class = (vala_code_node_unref (old_class), NULL)), _tmp8);
			}
			if (!(VALA_IS_OBJECT_TYPE (base_type_reference))) {
				char* _tmp11;
				char* _tmp10;
				char* _tmp9;
				gboolean _tmp12;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				_tmp11 = NULL;
				_tmp10 = NULL;
				_tmp9 = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp11 = g_strdup_printf ("base type `%s` of class `%s` is not an object type", _tmp9 = vala_code_node_to_string ((ValaCodeNode*) base_type_reference), _tmp10 = vala_symbol_get_full_name ((ValaSymbol*) self)));
				_tmp11 = (g_free (_tmp11), NULL);
				_tmp10 = (g_free (_tmp10), NULL);
				_tmp9 = (g_free (_tmp9), NULL);
				return (_tmp12 = FALSE, (base_type_reference == NULL) ? NULL : (base_type_reference = (vala_code_node_unref (base_type_reference), NULL)), (_base_type_reference_it == NULL) ? NULL : (_base_type_reference_it = (gee_collection_object_unref (_base_type_reference_it), NULL)), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), (old_class == NULL) ? NULL : (old_class = (vala_code_node_unref (old_class), NULL)), _tmp12);
			}
			/* check whether base type is at least as accessible as the class*/
			if (!vala_semantic_analyzer_is_type_accessible (analyzer, (ValaSymbol*) self, base_type_reference)) {
				char* _tmp15;
				char* _tmp14;
				char* _tmp13;
				gboolean _tmp16;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				_tmp15 = NULL;
				_tmp14 = NULL;
				_tmp13 = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp15 = g_strdup_printf ("base type `%s` is less accessible than class `%s`", _tmp13 = vala_code_node_to_string ((ValaCodeNode*) base_type_reference), _tmp14 = vala_symbol_get_full_name ((ValaSymbol*) self)));
				_tmp15 = (g_free (_tmp15), NULL);
				_tmp14 = (g_free (_tmp14), NULL);
				_tmp13 = (g_free (_tmp13), NULL);
				return (_tmp16 = FALSE, (base_type_reference == NULL) ? NULL : (base_type_reference = (vala_code_node_unref (base_type_reference), NULL)), (_base_type_reference_it == NULL) ? NULL : (_base_type_reference_it = (gee_collection_object_unref (_base_type_reference_it), NULL)), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), (old_class == NULL) ? NULL : (old_class = (vala_code_node_unref (old_class), NULL)), _tmp16);
			}
			vala_source_file_add_type_dependency (vala_semantic_analyzer_get_current_source_file (analyzer), base_type_reference, VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
			(base_type_reference == NULL) ? NULL : (base_type_reference = (vala_code_node_unref (base_type_reference), NULL));
		}
		(_base_type_reference_it == NULL) ? NULL : (_base_type_reference_it = (gee_collection_object_unref (_base_type_reference_it), NULL));
	}
	{
		GeeIterator* _type_it;
		_type_it = gee_iterable_iterator ((GeeIterable*) self->priv->base_types);
		while (gee_iterator_next (_type_it)) {
			ValaDataType* type;
			type = (ValaDataType*) gee_iterator_get (_type_it);
			vala_code_node_check ((ValaCodeNode*) type, analyzer);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
		}
		(_type_it == NULL) ? NULL : (_type_it = (gee_collection_object_unref (_type_it), NULL));
	}
	{
		GeeList* _tmp17;
		GeeIterator* _tmp18;
		GeeIterator* _p_it;
		_tmp17 = NULL;
		_tmp18 = NULL;
		_p_it = (_tmp18 = gee_iterable_iterator ((GeeIterable*) (_tmp17 = vala_object_type_symbol_get_type_parameters ((ValaObjectTypeSymbol*) self))), (_tmp17 == NULL) ? NULL : (_tmp17 = (gee_collection_object_unref (_tmp17), NULL)), _tmp18);
		while (gee_iterator_next (_p_it)) {
			ValaTypeParameter* p;
			p = (ValaTypeParameter*) gee_iterator_get (_p_it);
			vala_code_node_check ((ValaCodeNode*) p, analyzer);
			(p == NULL) ? NULL : (p = (vala_code_node_unref (p), NULL));
		}
		(_p_it == NULL) ? NULL : (_p_it = (gee_collection_object_unref (_p_it), NULL));
	}
	/* process enums first to avoid order problems in C code */
	{
		GeeIterator* _en_it;
		/* process enums first to avoid order problems in C code */
		_en_it = gee_iterable_iterator ((GeeIterable*) self->priv->enums);
		/* process enums first to avoid order problems in C code */
		while (gee_iterator_next (_en_it)) {
			ValaEnum* en;
			/* process enums first to avoid order problems in C code */
			en = (ValaEnum*) gee_iterator_get (_en_it);
			vala_code_node_check ((ValaCodeNode*) en, analyzer);
			(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
		}
		(_en_it == NULL) ? NULL : (_en_it = (gee_collection_object_unref (_en_it), NULL));
	}
	{
		GeeIterator* _f_it;
		_f_it = gee_iterable_iterator ((GeeIterable*) self->priv->fields);
		while (gee_iterator_next (_f_it)) {
			ValaField* f;
			f = (ValaField*) gee_iterator_get (_f_it);
			vala_code_node_check ((ValaCodeNode*) f, analyzer);
			(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
		}
		(_f_it == NULL) ? NULL : (_f_it = (gee_collection_object_unref (_f_it), NULL));
	}
	{
		GeeIterator* _c_it;
		_c_it = gee_iterable_iterator ((GeeIterable*) self->priv->constants);
		while (gee_iterator_next (_c_it)) {
			ValaConstant* c;
			c = (ValaConstant*) gee_iterator_get (_c_it);
			vala_code_node_check ((ValaCodeNode*) c, analyzer);
			(c == NULL) ? NULL : (c = (vala_code_node_unref (c), NULL));
		}
		(_c_it == NULL) ? NULL : (_c_it = (gee_collection_object_unref (_c_it), NULL));
	}
	{
		GeeIterator* _m_it;
		_m_it = gee_iterable_iterator ((GeeIterable*) self->priv->methods);
		while (gee_iterator_next (_m_it)) {
			ValaMethod* m;
			m = (ValaMethod*) gee_iterator_get (_m_it);
			vala_code_node_check ((ValaCodeNode*) m, analyzer);
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
	{
		GeeIterator* _prop_it;
		_prop_it = gee_iterable_iterator ((GeeIterable*) self->priv->properties);
		while (gee_iterator_next (_prop_it)) {
			ValaProperty* prop;
			prop = (ValaProperty*) gee_iterator_get (_prop_it);
			vala_code_node_check ((ValaCodeNode*) prop, analyzer);
			(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
		}
		(_prop_it == NULL) ? NULL : (_prop_it = (gee_collection_object_unref (_prop_it), NULL));
	}
	{
		GeeIterator* _sig_it;
		_sig_it = gee_iterable_iterator ((GeeIterable*) self->priv->signals);
		while (gee_iterator_next (_sig_it)) {
			ValaSignal* sig;
			sig = (ValaSignal*) gee_iterator_get (_sig_it);
			vala_code_node_check ((ValaCodeNode*) sig, analyzer);
			(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
		}
		(_sig_it == NULL) ? NULL : (_sig_it = (gee_collection_object_unref (_sig_it), NULL));
	}
	if (self->priv->_constructor != NULL) {
		vala_code_node_check ((ValaCodeNode*) self->priv->_constructor, analyzer);
	}
	if (self->priv->_class_constructor != NULL) {
		vala_code_node_check ((ValaCodeNode*) self->priv->_class_constructor, analyzer);
	}
	if (self->priv->_static_constructor != NULL) {
		vala_code_node_check ((ValaCodeNode*) self->priv->_static_constructor, analyzer);
	}
	if (vala_class_get_destructor (self) != NULL) {
		vala_code_node_check ((ValaCodeNode*) vala_class_get_destructor (self), analyzer);
	}
	if (self->priv->_static_destructor != NULL) {
		vala_code_node_check ((ValaCodeNode*) self->priv->_static_destructor, analyzer);
	}
	if (self->priv->_class_destructor != NULL) {
		vala_code_node_check ((ValaCodeNode*) self->priv->_class_destructor, analyzer);
	}
	{
		GeeIterator* _cl_it;
		_cl_it = gee_iterable_iterator ((GeeIterable*) self->priv->classes);
		while (gee_iterator_next (_cl_it)) {
			ValaClass* cl;
			cl = (ValaClass*) gee_iterator_get (_cl_it);
			vala_code_node_check ((ValaCodeNode*) cl, analyzer);
			(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
		}
		(_cl_it == NULL) ? NULL : (_cl_it = (gee_collection_object_unref (_cl_it), NULL));
	}
	{
		GeeIterator* _st_it;
		_st_it = gee_iterable_iterator ((GeeIterable*) self->priv->structs);
		while (gee_iterator_next (_st_it)) {
			ValaStruct* st;
			st = (ValaStruct*) gee_iterator_get (_st_it);
			vala_code_node_check ((ValaCodeNode*) st, analyzer);
			(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
		}
		(_st_it == NULL) ? NULL : (_st_it = (gee_collection_object_unref (_st_it), NULL));
	}
	{
		GeeIterator* _d_it;
		_d_it = gee_iterable_iterator ((GeeIterable*) self->priv->delegates);
		while (gee_iterator_next (_d_it)) {
			ValaDelegate* d;
			d = (ValaDelegate*) gee_iterator_get (_d_it);
			vala_code_node_check ((ValaCodeNode*) d, analyzer);
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
		}
		(_d_it == NULL) ? NULL : (_d_it = (gee_collection_object_unref (_d_it), NULL));
	}
	/* compact classes cannot implement interfaces */
	if (vala_class_get_is_compact (self)) {
		{
			GeeList* _tmp19;
			GeeIterator* _tmp20;
			GeeIterator* _base_type_it;
			_tmp19 = NULL;
			_tmp20 = NULL;
			_base_type_it = (_tmp20 = gee_iterable_iterator ((GeeIterable*) (_tmp19 = vala_class_get_base_types (self))), (_tmp19 == NULL) ? NULL : (_tmp19 = (gee_collection_object_unref (_tmp19), NULL)), _tmp20);
			while (gee_iterator_next (_base_type_it)) {
				ValaDataType* base_type;
				base_type = (ValaDataType*) gee_iterator_get (_base_type_it);
				if (VALA_IS_INTERFACE (vala_data_type_get_data_type (base_type))) {
					char* _tmp22;
					char* _tmp21;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp22 = NULL;
					_tmp21 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp22 = g_strdup_printf ("compact classes `%s` may not implement interfaces", _tmp21 = vala_symbol_get_full_name ((ValaSymbol*) self)));
					_tmp22 = (g_free (_tmp22), NULL);
					_tmp21 = (g_free (_tmp21), NULL);
				}
				(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
			}
			(_base_type_it == NULL) ? NULL : (_base_type_it = (gee_collection_object_unref (_base_type_it), NULL));
		}
	}
	/* gather all prerequisites */
	prerequisites = (GeeList*) gee_array_list_new (VALA_TYPE_TYPESYMBOL, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	{
		GeeList* _tmp23;
		GeeIterator* _tmp24;
		GeeIterator* _base_type_it;
		_tmp23 = NULL;
		_tmp24 = NULL;
		_base_type_it = (_tmp24 = gee_iterable_iterator ((GeeIterable*) (_tmp23 = vala_class_get_base_types (self))), (_tmp23 == NULL) ? NULL : (_tmp23 = (gee_collection_object_unref (_tmp23), NULL)), _tmp24);
		while (gee_iterator_next (_base_type_it)) {
			ValaDataType* base_type;
			base_type = (ValaDataType*) gee_iterator_get (_base_type_it);
			if (VALA_IS_INTERFACE (vala_data_type_get_data_type (base_type))) {
				vala_class_get_all_prerequisites (self, VALA_INTERFACE (vala_data_type_get_data_type (base_type)), prerequisites);
			}
			(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
		}
		(_base_type_it == NULL) ? NULL : (_base_type_it = (gee_collection_object_unref (_base_type_it), NULL));
	}
	/* check whether all prerequisites are met */
	missing_prereqs = (GeeList*) gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
	{
		GeeIterator* _prereq_it;
		_prereq_it = gee_iterable_iterator ((GeeIterable*) prerequisites);
		while (gee_iterator_next (_prereq_it)) {
			ValaTypeSymbol* prereq;
			prereq = (ValaTypeSymbol*) gee_iterator_get (_prereq_it);
			if (!vala_class_class_is_a (self, self, prereq)) {
				char* _tmp25;
				_tmp25 = NULL;
				gee_list_insert (missing_prereqs, 0, _tmp25 = vala_symbol_get_full_name ((ValaSymbol*) prereq));
				_tmp25 = (g_free (_tmp25), NULL);
			}
			(prereq == NULL) ? NULL : (prereq = (vala_code_node_unref (prereq), NULL));
		}
		(_prereq_it == NULL) ? NULL : (_prereq_it = (gee_collection_object_unref (_prereq_it), NULL));
	}
	/* report any missing prerequisites */
	if (gee_collection_get_size ((GeeCollection*) missing_prereqs) > 0) {
		char* _tmp26;
		char* _tmp27;
		char* error_string;
		gboolean first;
		char* _tmp30;
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		_tmp26 = NULL;
		_tmp27 = NULL;
		error_string = (_tmp27 = g_strdup_printf ("%s: some prerequisites (", _tmp26 = vala_symbol_get_full_name ((ValaSymbol*) self)), _tmp26 = (g_free (_tmp26), NULL), _tmp27);
		first = TRUE;
		{
			GeeIterator* _s_it;
			_s_it = gee_iterable_iterator ((GeeIterable*) missing_prereqs);
			while (gee_iterator_next (_s_it)) {
				char* s;
				s = (char*) gee_iterator_get (_s_it);
				if (first) {
					char* _tmp28;
					_tmp28 = NULL;
					error_string = (_tmp28 = g_strdup_printf ("%s`%s'", error_string, s), error_string = (g_free (error_string), NULL), _tmp28);
					first = FALSE;
				} else {
					char* _tmp29;
					_tmp29 = NULL;
					error_string = (_tmp29 = g_strdup_printf ("%s, `%s'", error_string, s), error_string = (g_free (error_string), NULL), _tmp29);
				}
				s = (g_free (s), NULL);
			}
			(_s_it == NULL) ? NULL : (_s_it = (gee_collection_object_unref (_s_it), NULL));
		}
		_tmp30 = NULL;
		error_string = (_tmp30 = g_strconcat (error_string, (") are not met"), NULL), error_string = (g_free (error_string), NULL), _tmp30);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), error_string);
		error_string = (g_free (error_string), NULL);
	}
	/* VAPI classes don't have to specify overridden methods */
	if (!vala_symbol_get_external_package ((ValaSymbol*) self)) {
		/* all abstract symbols defined in base types have to be at least defined (or implemented) also in this type */
		{
			GeeList* _tmp31;
			GeeIterator* _tmp32;
			GeeIterator* _base_type_it;
			/* all abstract symbols defined in base types have to be at least defined (or implemented) also in this type */
			_tmp31 = NULL;
			_tmp32 = NULL;
			_base_type_it = (_tmp32 = gee_iterable_iterator ((GeeIterable*) (_tmp31 = vala_class_get_base_types (self))), (_tmp31 == NULL) ? NULL : (_tmp31 = (gee_collection_object_unref (_tmp31), NULL)), _tmp32);
			/* all abstract symbols defined in base types have to be at least defined (or implemented) also in this type */
			while (gee_iterator_next (_base_type_it)) {
				ValaDataType* base_type;
				/* all abstract symbols defined in base types have to be at least defined (or implemented) also in this type */
				base_type = (ValaDataType*) gee_iterator_get (_base_type_it);
				if (VALA_IS_INTERFACE (vala_data_type_get_data_type (base_type))) {
					ValaInterface* _tmp33;
					ValaInterface* iface;
					gboolean _tmp34;
					_tmp33 = NULL;
					iface = (_tmp33 = VALA_INTERFACE (vala_data_type_get_data_type (base_type)), (_tmp33 == NULL) ? NULL : vala_code_node_ref (_tmp33));
					_tmp34 = FALSE;
					if (self->priv->_base_class != NULL) {
						_tmp34 = vala_typesymbol_is_subtype_of ((ValaTypeSymbol*) self->priv->_base_class, (ValaTypeSymbol*) iface);
					} else {
						_tmp34 = FALSE;
					}
					if (_tmp34) {
						/* reimplementation of interface, class is not required to reimplement all methods*/
						(iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL));
						(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
						break;
					}
					/* We do not need to do expensive equality checking here since this is done
					 * already. We only need to guarantee the symbols are present.
					 
					 check methods */
					{
						GeeList* _tmp35;
						GeeIterator* _tmp36;
						GeeIterator* _m_it;
						/* We do not need to do expensive equality checking here since this is done
						 * already. We only need to guarantee the symbols are present.
						 
						 check methods */
						_tmp35 = NULL;
						_tmp36 = NULL;
						_m_it = (_tmp36 = gee_iterable_iterator ((GeeIterable*) (_tmp35 = vala_object_type_symbol_get_methods ((ValaObjectTypeSymbol*) iface))), (_tmp35 == NULL) ? NULL : (_tmp35 = (gee_collection_object_unref (_tmp35), NULL)), _tmp36);
						/* We do not need to do expensive equality checking here since this is done
						 * already. We only need to guarantee the symbols are present.
						 
						 check methods */
						while (gee_iterator_next (_m_it)) {
							ValaMethod* m;
							/* We do not need to do expensive equality checking here since this is done
							 * already. We only need to guarantee the symbols are present.
							 
							 check methods */
							m = (ValaMethod*) gee_iterator_get (_m_it);
							if (vala_method_get_is_abstract (m)) {
								ValaSymbol* sym;
								ValaClass* _tmp37;
								ValaClass* base_class;
								sym = NULL;
								_tmp37 = NULL;
								base_class = (_tmp37 = self, (_tmp37 == NULL) ? NULL : vala_code_node_ref (_tmp37));
								while (TRUE) {
									gboolean _tmp38;
									ValaSymbol* _tmp39;
									ValaClass* _tmp41;
									ValaClass* _tmp40;
									_tmp38 = FALSE;
									if (base_class != NULL) {
										_tmp38 = !(VALA_IS_METHOD (sym));
									} else {
										_tmp38 = FALSE;
									}
									if (!_tmp38) {
										break;
									}
									_tmp39 = NULL;
									sym = (_tmp39 = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) base_class), vala_symbol_get_name ((ValaSymbol*) m)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp39);
									_tmp41 = NULL;
									_tmp40 = NULL;
									base_class = (_tmp41 = (_tmp40 = base_class->priv->_base_class, (_tmp40 == NULL) ? NULL : vala_code_node_ref (_tmp40)), (base_class == NULL) ? NULL : (base_class = (vala_code_node_unref (base_class), NULL)), _tmp41);
								}
								if (!(VALA_IS_METHOD (sym))) {
									char* _tmp44;
									char* _tmp43;
									char* _tmp42;
									vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
									_tmp44 = NULL;
									_tmp43 = NULL;
									_tmp42 = NULL;
									vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp44 = g_strdup_printf ("`%s' does not implement interface method `%s'", _tmp42 = vala_symbol_get_full_name ((ValaSymbol*) self), _tmp43 = vala_symbol_get_full_name ((ValaSymbol*) m)));
									_tmp44 = (g_free (_tmp44), NULL);
									_tmp43 = (g_free (_tmp43), NULL);
									_tmp42 = (g_free (_tmp42), NULL);
								}
								(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
								(base_class == NULL) ? NULL : (base_class = (vala_code_node_unref (base_class), NULL));
							}
							(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
						}
						(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
					}
					/* check properties */
					{
						GeeList* _tmp45;
						GeeIterator* _tmp46;
						GeeIterator* _prop_it;
						/* check properties */
						_tmp45 = NULL;
						_tmp46 = NULL;
						_prop_it = (_tmp46 = gee_iterable_iterator ((GeeIterable*) (_tmp45 = vala_object_type_symbol_get_properties ((ValaObjectTypeSymbol*) iface))), (_tmp45 == NULL) ? NULL : (_tmp45 = (gee_collection_object_unref (_tmp45), NULL)), _tmp46);
						/* check properties */
						while (gee_iterator_next (_prop_it)) {
							ValaProperty* prop;
							/* check properties */
							prop = (ValaProperty*) gee_iterator_get (_prop_it);
							if (vala_property_get_is_abstract (prop)) {
								ValaSymbol* sym;
								ValaClass* _tmp47;
								ValaClass* base_class;
								sym = NULL;
								_tmp47 = NULL;
								base_class = (_tmp47 = self, (_tmp47 == NULL) ? NULL : vala_code_node_ref (_tmp47));
								while (TRUE) {
									gboolean _tmp48;
									ValaSymbol* _tmp49;
									ValaClass* _tmp51;
									ValaClass* _tmp50;
									_tmp48 = FALSE;
									if (base_class != NULL) {
										_tmp48 = !(VALA_IS_PROPERTY (sym));
									} else {
										_tmp48 = FALSE;
									}
									if (!_tmp48) {
										break;
									}
									_tmp49 = NULL;
									sym = (_tmp49 = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) base_class), vala_symbol_get_name ((ValaSymbol*) prop)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp49);
									_tmp51 = NULL;
									_tmp50 = NULL;
									base_class = (_tmp51 = (_tmp50 = base_class->priv->_base_class, (_tmp50 == NULL) ? NULL : vala_code_node_ref (_tmp50)), (base_class == NULL) ? NULL : (base_class = (vala_code_node_unref (base_class), NULL)), _tmp51);
								}
								if (!(VALA_IS_PROPERTY (sym))) {
									char* _tmp54;
									char* _tmp53;
									char* _tmp52;
									vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
									_tmp54 = NULL;
									_tmp53 = NULL;
									_tmp52 = NULL;
									vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp54 = g_strdup_printf ("`%s' does not implement interface property `%s'", _tmp52 = vala_symbol_get_full_name ((ValaSymbol*) self), _tmp53 = vala_symbol_get_full_name ((ValaSymbol*) prop)));
									_tmp54 = (g_free (_tmp54), NULL);
									_tmp53 = (g_free (_tmp53), NULL);
									_tmp52 = (g_free (_tmp52), NULL);
								}
								(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
								(base_class == NULL) ? NULL : (base_class = (vala_code_node_unref (base_class), NULL));
							}
							(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
						}
						(_prop_it == NULL) ? NULL : (_prop_it = (gee_collection_object_unref (_prop_it), NULL));
					}
					(iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL));
				}
				(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
			}
			(_base_type_it == NULL) ? NULL : (_base_type_it = (gee_collection_object_unref (_base_type_it), NULL));
		}
		/* all abstract symbols defined in base classes have to be implemented in non-abstract classes */
		if (!self->priv->_is_abstract) {
			ValaClass* _tmp55;
			ValaClass* base_class;
			_tmp55 = NULL;
			base_class = (_tmp55 = self->priv->_base_class, (_tmp55 == NULL) ? NULL : vala_code_node_ref (_tmp55));
			while (TRUE) {
				gboolean _tmp56;
				ValaClass* _tmp72;
				ValaClass* _tmp71;
				_tmp56 = FALSE;
				if (base_class != NULL) {
					_tmp56 = base_class->priv->_is_abstract;
				} else {
					_tmp56 = FALSE;
				}
				if (!_tmp56) {
					break;
				}
				{
					GeeList* _tmp57;
					GeeIterator* _tmp58;
					GeeIterator* _base_method_it;
					_tmp57 = NULL;
					_tmp58 = NULL;
					_base_method_it = (_tmp58 = gee_iterable_iterator ((GeeIterable*) (_tmp57 = vala_object_type_symbol_get_methods ((ValaObjectTypeSymbol*) base_class))), (_tmp57 == NULL) ? NULL : (_tmp57 = (gee_collection_object_unref (_tmp57), NULL)), _tmp58);
					while (gee_iterator_next (_base_method_it)) {
						ValaMethod* base_method;
						base_method = (ValaMethod*) gee_iterator_get (_base_method_it);
						if (vala_method_get_is_abstract (base_method)) {
							ValaSymbol* _tmp59;
							ValaMethod* override_method;
							gboolean _tmp60;
							_tmp59 = NULL;
							override_method = (_tmp59 = vala_semantic_analyzer_symbol_lookup_inherited ((ValaSymbol*) self, vala_symbol_get_name ((ValaSymbol*) base_method)), VALA_IS_METHOD (_tmp59) ? ((ValaMethod*) _tmp59) : NULL);
							_tmp60 = FALSE;
							if (override_method == NULL) {
								_tmp60 = TRUE;
							} else {
								_tmp60 = !vala_method_get_overrides (override_method);
							}
							if (_tmp60) {
								char* _tmp63;
								char* _tmp62;
								char* _tmp61;
								vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
								_tmp63 = NULL;
								_tmp62 = NULL;
								_tmp61 = NULL;
								vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp63 = g_strdup_printf ("`%s' does not implement abstract method `%s'", _tmp61 = vala_symbol_get_full_name ((ValaSymbol*) self), _tmp62 = vala_symbol_get_full_name ((ValaSymbol*) base_method)));
								_tmp63 = (g_free (_tmp63), NULL);
								_tmp62 = (g_free (_tmp62), NULL);
								_tmp61 = (g_free (_tmp61), NULL);
							}
							(override_method == NULL) ? NULL : (override_method = (vala_code_node_unref (override_method), NULL));
						}
						(base_method == NULL) ? NULL : (base_method = (vala_code_node_unref (base_method), NULL));
					}
					(_base_method_it == NULL) ? NULL : (_base_method_it = (gee_collection_object_unref (_base_method_it), NULL));
				}
				{
					GeeList* _tmp64;
					GeeIterator* _tmp65;
					GeeIterator* _base_property_it;
					_tmp64 = NULL;
					_tmp65 = NULL;
					_base_property_it = (_tmp65 = gee_iterable_iterator ((GeeIterable*) (_tmp64 = vala_object_type_symbol_get_properties ((ValaObjectTypeSymbol*) base_class))), (_tmp64 == NULL) ? NULL : (_tmp64 = (gee_collection_object_unref (_tmp64), NULL)), _tmp65);
					while (gee_iterator_next (_base_property_it)) {
						ValaProperty* base_property;
						base_property = (ValaProperty*) gee_iterator_get (_base_property_it);
						if (vala_property_get_is_abstract (base_property)) {
							ValaSymbol* _tmp66;
							ValaProperty* override_property;
							gboolean _tmp67;
							_tmp66 = NULL;
							override_property = (_tmp66 = vala_semantic_analyzer_symbol_lookup_inherited ((ValaSymbol*) self, vala_symbol_get_name ((ValaSymbol*) base_property)), VALA_IS_PROPERTY (_tmp66) ? ((ValaProperty*) _tmp66) : NULL);
							_tmp67 = FALSE;
							if (override_property == NULL) {
								_tmp67 = TRUE;
							} else {
								_tmp67 = !vala_property_get_overrides (override_property);
							}
							if (_tmp67) {
								char* _tmp70;
								char* _tmp69;
								char* _tmp68;
								vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
								_tmp70 = NULL;
								_tmp69 = NULL;
								_tmp68 = NULL;
								vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp70 = g_strdup_printf ("`%s' does not implement abstract property `%s'", _tmp68 = vala_symbol_get_full_name ((ValaSymbol*) self), _tmp69 = vala_symbol_get_full_name ((ValaSymbol*) base_property)));
								_tmp70 = (g_free (_tmp70), NULL);
								_tmp69 = (g_free (_tmp69), NULL);
								_tmp68 = (g_free (_tmp68), NULL);
							}
							(override_property == NULL) ? NULL : (override_property = (vala_code_node_unref (override_property), NULL));
						}
						(base_property == NULL) ? NULL : (base_property = (vala_code_node_unref (base_property), NULL));
					}
					(_base_property_it == NULL) ? NULL : (_base_property_it = (gee_collection_object_unref (_base_property_it), NULL));
				}
				_tmp72 = NULL;
				_tmp71 = NULL;
				base_class = (_tmp72 = (_tmp71 = base_class->priv->_base_class, (_tmp71 == NULL) ? NULL : vala_code_node_ref (_tmp71)), (base_class == NULL) ? NULL : (base_class = (vala_code_node_unref (base_class), NULL)), _tmp72);
			}
			(base_class == NULL) ? NULL : (base_class = (vala_code_node_unref (base_class), NULL));
		}
	}
	vala_semantic_analyzer_set_current_source_file (analyzer, old_source_file);
	vala_semantic_analyzer_set_current_symbol (analyzer, old_symbol);
	_tmp74 = NULL;
	_tmp73 = NULL;
	analyzer->current_class = (_tmp74 = (_tmp73 = old_class, (_tmp73 == NULL) ? NULL : vala_code_node_ref (_tmp73)), (analyzer->current_class == NULL) ? NULL : (analyzer->current_class = (vala_code_node_unref (analyzer->current_class), NULL)), _tmp74);
	return (_tmp75 = !vala_code_node_get_error ((ValaCodeNode*) self), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), (old_class == NULL) ? NULL : (old_class = (vala_code_node_unref (old_class), NULL)), (prerequisites == NULL) ? NULL : (prerequisites = (gee_collection_object_unref (prerequisites), NULL)), (missing_prereqs == NULL) ? NULL : (missing_prereqs = (gee_collection_object_unref (missing_prereqs), NULL)), _tmp75);
}


ValaClass* vala_class_get_base_class (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_base_class;
}


void vala_class_set_base_class (ValaClass* self, ValaClass* value) {
	ValaClass* _tmp2;
	ValaClass* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_base_class = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_base_class == NULL) ? NULL : (self->priv->_base_class = (vala_code_node_unref (self->priv->_base_class), NULL)), _tmp2);
}


gboolean vala_class_get_is_abstract (ValaClass* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_abstract;
}


void vala_class_set_is_abstract (ValaClass* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_abstract = value;
}


gboolean vala_class_get_is_compact (ValaClass* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	if (self->priv->_base_class != NULL) {
		return vala_class_get_is_compact (self->priv->_base_class);
	}
	return self->priv->_is_compact;
}


void vala_class_set_is_compact (ValaClass* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_compact = value;
}


gboolean vala_class_get_is_immutable (ValaClass* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	if (self->priv->_base_class != NULL) {
		return vala_class_get_is_immutable (self->priv->_base_class);
	}
	return self->priv->_is_immutable;
}


void vala_class_set_is_immutable (ValaClass* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_immutable = value;
}


gboolean vala_class_get_ref_function_void (ValaClass* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_ref_function_void;
}


void vala_class_set_ref_function_void (ValaClass* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_ref_function_void = value;
}


const char* vala_class_get_type_check_function (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_type_check_function;
}


void vala_class_set_type_check_function (ValaClass* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_type_check_function = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_type_check_function = (g_free (self->priv->_type_check_function), NULL), _tmp2);
}


gboolean vala_class_get_has_private_fields (ValaClass* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_has_private_fields;
}


static void vala_class_set_has_private_fields (ValaClass* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_has_private_fields = value;
}


gboolean vala_class_get_has_class_private_fields (ValaClass* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_has_class_private_fields;
}


static void vala_class_set_has_class_private_fields (ValaClass* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_has_class_private_fields = value;
}


ValaMethod* vala_class_get_default_construction_method (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_default_construction_method;
}


void vala_class_set_default_construction_method (ValaClass* self, ValaMethod* value) {
	ValaMethod* _tmp2;
	ValaMethod* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_default_construction_method = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_default_construction_method == NULL) ? NULL : (self->priv->_default_construction_method = (vala_code_node_unref (self->priv->_default_construction_method), NULL)), _tmp2);
}


ValaConstructor* vala_class_get_constructor (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_constructor;
}


void vala_class_set_constructor (ValaClass* self, ValaConstructor* value) {
	ValaConstructor* _tmp2;
	ValaConstructor* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_constructor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_constructor == NULL) ? NULL : (self->priv->_constructor = (vala_code_node_unref (self->priv->_constructor), NULL)), _tmp2);
}


ValaConstructor* vala_class_get_class_constructor (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_class_constructor;
}


void vala_class_set_class_constructor (ValaClass* self, ValaConstructor* value) {
	ValaConstructor* _tmp2;
	ValaConstructor* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_class_constructor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_class_constructor == NULL) ? NULL : (self->priv->_class_constructor = (vala_code_node_unref (self->priv->_class_constructor), NULL)), _tmp2);
}


ValaConstructor* vala_class_get_static_constructor (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_static_constructor;
}


void vala_class_set_static_constructor (ValaClass* self, ValaConstructor* value) {
	ValaConstructor* _tmp2;
	ValaConstructor* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_static_constructor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_static_constructor == NULL) ? NULL : (self->priv->_static_constructor = (vala_code_node_unref (self->priv->_static_constructor), NULL)), _tmp2);
}


ValaDestructor* vala_class_get_destructor (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_destructor;
}


void vala_class_set_destructor (ValaClass* self, ValaDestructor* value) {
	ValaDestructor* _tmp2;
	ValaDestructor* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_destructor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_destructor == NULL) ? NULL : (self->priv->_destructor = (vala_code_node_unref (self->priv->_destructor), NULL)), _tmp2);
	if (self->priv->_destructor != NULL) {
		ValaFormalParameter* _tmp4;
		ValaObjectType* _tmp3;
		if (vala_destructor_get_this_parameter (self->priv->_destructor) != NULL) {
			vala_scope_remove (vala_symbol_get_scope ((ValaSymbol*) self->priv->_destructor), vala_symbol_get_name ((ValaSymbol*) vala_destructor_get_this_parameter (self->priv->_destructor)));
		}
		_tmp4 = NULL;
		_tmp3 = NULL;
		vala_destructor_set_this_parameter (self->priv->_destructor, _tmp4 = vala_formal_parameter_new ("this", (ValaDataType*) (_tmp3 = vala_object_type_new ((ValaObjectTypeSymbol*) self)), NULL));
		(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL));
		(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL));
		vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self->priv->_destructor), vala_symbol_get_name ((ValaSymbol*) vala_destructor_get_this_parameter (self->priv->_destructor)), (ValaSymbol*) vala_destructor_get_this_parameter (self->priv->_destructor));
	}
}


ValaDestructor* vala_class_get_static_destructor (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_static_destructor;
}


void vala_class_set_static_destructor (ValaClass* self, ValaDestructor* value) {
	ValaDestructor* _tmp2;
	ValaDestructor* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_static_destructor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_static_destructor == NULL) ? NULL : (self->priv->_static_destructor = (vala_code_node_unref (self->priv->_static_destructor), NULL)), _tmp2);
}


ValaDestructor* vala_class_get_class_destructor (ValaClass* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_class_destructor;
}


void vala_class_set_class_destructor (ValaClass* self, ValaDestructor* value) {
	ValaDestructor* _tmp2;
	ValaDestructor* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_class_destructor = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_class_destructor == NULL) ? NULL : (self->priv->_class_destructor = (vala_code_node_unref (self->priv->_class_destructor), NULL)), _tmp2);
}


gboolean vala_class_get_is_error_base (ValaClass* self) {
	ValaAttribute* _tmp0;
	gboolean _tmp1;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp0 = NULL;
	return (_tmp1 = (_tmp0 = vala_code_node_get_attribute ((ValaCodeNode*) self, "ErrorBase")) != NULL, (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)), _tmp1);
}


static void vala_class_class_init (ValaClassClass * klass) {
	vala_class_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_class_finalize;
	g_type_class_add_private (klass, sizeof (ValaClassPrivate));
	VALA_OBJECT_TYPE_SYMBOL_CLASS (klass)->get_methods = vala_class_real_get_methods;
	VALA_OBJECT_TYPE_SYMBOL_CLASS (klass)->get_properties = vala_class_real_get_properties;
	VALA_OBJECT_TYPE_SYMBOL_CLASS (klass)->get_signals = vala_class_real_get_signals;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_class_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_class_real_accept_children;
	VALA_SYMBOL_CLASS (klass)->get_cprefix = vala_class_real_get_cprefix;
	VALA_TYPESYMBOL_CLASS (klass)->get_cname = vala_class_real_get_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cname = vala_class_real_get_lower_case_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_class_real_get_lower_case_cprefix;
	VALA_TYPESYMBOL_CLASS (klass)->get_upper_case_cname = vala_class_real_get_upper_case_cname;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_signature = vala_class_real_get_type_signature;
	VALA_TYPESYMBOL_CLASS (klass)->is_reference_type = vala_class_real_is_reference_type;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_id = vala_class_real_get_type_id;
	VALA_TYPESYMBOL_CLASS (klass)->get_marshaller_type_name = vala_class_real_get_marshaller_type_name;
	VALA_TYPESYMBOL_CLASS (klass)->get_param_spec_function = vala_class_real_get_param_spec_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_get_value_function = vala_class_real_get_get_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_set_value_function = vala_class_real_get_set_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->is_reference_counting = vala_class_real_is_reference_counting;
	VALA_TYPESYMBOL_CLASS (klass)->get_ref_function = vala_class_real_get_ref_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_unref_function = vala_class_real_get_unref_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_dup_function = vala_class_real_get_dup_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_free_function = vala_class_real_get_free_function;
	VALA_TYPESYMBOL_CLASS (klass)->is_subtype_of = vala_class_real_is_subtype_of;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_class_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->check = vala_class_real_check;
}


static void vala_class_instance_init (ValaClass * self) {
	self->priv = VALA_CLASS_GET_PRIVATE (self);
	self->priv->base_types = (GeeList*) gee_array_list_new (VALA_TYPE_DATA_TYPE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->constants = (GeeList*) gee_array_list_new (VALA_TYPE_CONSTANT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->fields = (GeeList*) gee_array_list_new (VALA_TYPE_FIELD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->methods = (GeeList*) gee_array_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->properties = (GeeList*) gee_array_list_new (VALA_TYPE_PROPERTY, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->signals = (GeeList*) gee_array_list_new (VALA_TYPE_SIGNAL, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->classes = (GeeList*) gee_array_list_new (VALA_TYPE_CLASS, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->structs = (GeeList*) gee_array_list_new (VALA_TYPE_STRUCT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->enums = (GeeList*) gee_array_list_new (VALA_TYPE_ENUM, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->delegates = (GeeList*) gee_array_list_new (VALA_TYPE_DELEGATE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
}


static void vala_class_finalize (ValaCodeNode* obj) {
	ValaClass * self;
	self = VALA_CLASS (obj);
	(self->priv->_base_class == NULL) ? NULL : (self->priv->_base_class = (vala_code_node_unref (self->priv->_base_class), NULL));
	self->priv->_type_check_function = (g_free (self->priv->_type_check_function), NULL);
	self->priv->cname = (g_free (self->priv->cname), NULL);
	self->priv->const_cname = (g_free (self->priv->const_cname), NULL);
	self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL);
	self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL);
	self->priv->type_id = (g_free (self->priv->type_id), NULL);
	self->priv->ref_function = (g_free (self->priv->ref_function), NULL);
	self->priv->unref_function = (g_free (self->priv->unref_function), NULL);
	self->priv->param_spec_function = (g_free (self->priv->param_spec_function), NULL);
	self->priv->copy_function = (g_free (self->priv->copy_function), NULL);
	self->priv->free_function = (g_free (self->priv->free_function), NULL);
	self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL);
	self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL);
	self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL);
	self->priv->type_signature = (g_free (self->priv->type_signature), NULL);
	(self->priv->base_types == NULL) ? NULL : (self->priv->base_types = (gee_collection_object_unref (self->priv->base_types), NULL));
	(self->priv->constants == NULL) ? NULL : (self->priv->constants = (gee_collection_object_unref (self->priv->constants), NULL));
	(self->priv->fields == NULL) ? NULL : (self->priv->fields = (gee_collection_object_unref (self->priv->fields), NULL));
	(self->priv->methods == NULL) ? NULL : (self->priv->methods = (gee_collection_object_unref (self->priv->methods), NULL));
	(self->priv->properties == NULL) ? NULL : (self->priv->properties = (gee_collection_object_unref (self->priv->properties), NULL));
	(self->priv->signals == NULL) ? NULL : (self->priv->signals = (gee_collection_object_unref (self->priv->signals), NULL));
	(self->priv->classes == NULL) ? NULL : (self->priv->classes = (gee_collection_object_unref (self->priv->classes), NULL));
	(self->priv->structs == NULL) ? NULL : (self->priv->structs = (gee_collection_object_unref (self->priv->structs), NULL));
	(self->priv->enums == NULL) ? NULL : (self->priv->enums = (gee_collection_object_unref (self->priv->enums), NULL));
	(self->priv->delegates == NULL) ? NULL : (self->priv->delegates = (gee_collection_object_unref (self->priv->delegates), NULL));
	(self->priv->_default_construction_method == NULL) ? NULL : (self->priv->_default_construction_method = (vala_code_node_unref (self->priv->_default_construction_method), NULL));
	(self->priv->_constructor == NULL) ? NULL : (self->priv->_constructor = (vala_code_node_unref (self->priv->_constructor), NULL));
	(self->priv->_class_constructor == NULL) ? NULL : (self->priv->_class_constructor = (vala_code_node_unref (self->priv->_class_constructor), NULL));
	(self->priv->_static_constructor == NULL) ? NULL : (self->priv->_static_constructor = (vala_code_node_unref (self->priv->_static_constructor), NULL));
	(self->priv->_static_destructor == NULL) ? NULL : (self->priv->_static_destructor = (vala_code_node_unref (self->priv->_static_destructor), NULL));
	(self->priv->_class_destructor == NULL) ? NULL : (self->priv->_class_destructor = (vala_code_node_unref (self->priv->_class_destructor), NULL));
	(self->priv->_destructor == NULL) ? NULL : (self->priv->_destructor = (vala_code_node_unref (self->priv->_destructor), NULL));
	VALA_CODE_NODE_CLASS (vala_class_parent_class)->finalize (obj);
}


GType vala_class_get_type (void) {
	static GType vala_class_type_id = 0;
	if (vala_class_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaClassClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_class_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaClass), 0, (GInstanceInitFunc) vala_class_instance_init, NULL };
		vala_class_type_id = g_type_register_static (VALA_TYPE_OBJECT_TYPE_SYMBOL, "ValaClass", &g_define_type_info, 0);
	}
	return vala_class_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




